Module Util;

Exports

Const   UVersion = 3;   { Increment at changes that require user programs
                          to be recompiled.
                          }

Procedure UInit (V: Integer);   { Initialize utility package }

Procedure UCheckVersion (V: Integer; Mes: String);   { Check version number }

{ *************************************************************************** }

{ ************************* STRING HACKING SECTION ************************** }

Type    UStr255 = String [255];
        UStr25 = String [25];

Function  UStrCmp (Var s1, s2: UStr255): Integer;

Procedure UCnvUpper (Var S: Ustr255);

Procedure UBeep;

{ *************************************************************************** }

{ *********************** ARITHMETIC HACKING SECTION ************************ }

Function IMin (a, b: Integer): Integer;

Function IMax (a, b: Integer): Integer;

{ *************************************************************************** }

{ ************************ MEMORY MANAGEMENT SECTION ************************ }

Const   UWordsPBlock = 256;

Var     UHeap: Integer;

Type    UPointer = Record
                Case Boolean of
                    False: (P: Pointer);
                    True:  (Offset: Integer;
                            Segment: Integer)
                End;

Procedure UNewSegment;

Procedure UNew (Var P: Pointer; Size, Align: Integer);

Function  UBlocks (Words: Long): Integer;

Function  UCreateSegment (InitSize, Inc, MaxSize: Integer): Integer;

Procedure UCreatePSegment (Var P: Pointer; Words: Long);

Procedure UIncSegSize (P: Pointer);

Procedure UDispSegment (Seg: Integer);

Procedure UDispPSegment (Var P: Pointer);

Procedure UAllocNameDesc (N: Integer; Var P: Pointer);

{ *************************************************************************** }

{ **************************** DIRECTORY SECTION **************************** }

Type    UPtrFile = ^UFile;
        UFile = Record
                name: UStr25;      { max 25 chars in a file name }
                fid: Integer;           { FileID of file }
                next: UPtrFile;
                End;

Procedure USplit (Var Original, Directory, FileName: UStr255);

Function  UDirectory (DirName, Pattern: UStr255): UPtrFile;

Procedure UDispDirectory (Var PF: UPtrFile);

Procedure UDirPopList (Var List: Pointer; Var DirName, Pattern: UStr255);

{ *************************************************************************** }

{ ************************** SYSTEM WINDOW SECTION ************************** }

Type    UWindNum = 0 .. 17;

Var     UCurrWindow: UWindNum;

Function  UNewWindNum: Integer;

Procedure UUseWindNum (W: UWindNum);

Procedure UFreeWindNum (W: UWindNum);

Procedure UChangeWindow (W: UWindNum);

Procedure UTitle (S: String);

{ *************************************************************************** }

{ ******************************* AREA SECTION ****************************** }

Type    UCoord = Record
                UX: Integer;
                UY: Integer;
                End;

        URect = Record
                Case Boolean of
                    False: (ULeft: Integer;
                            UTop: Integer;
                            UWidth: Integer;
                            UHeight: Integer);
                    True:  (UUpLeft: UCoord);
                    End;

        UBorderType = (UNoBorder, USolid, UInvSolid, UBand, UInvBand,
                        USysTWindow, USysNoTWindow);

        UWindow = Record
                UStyle: UBorderType;
                UThick: Integer;     { if syswindow, this is window number }
                UInside: URect;
                USaved: Pointer;
                End;

Const   UScreenW = 48;          { same as SScreenW }

Var     UScreenP: Pointer;      { same as SScreenP }
        UScrOutside, UScrInside: URect;
        UScrWindow: UWindow;

Function  UCdToRect (C1, C2: UCoord): URect;

Function  URctBottom (R: URect): Integer;

Function  URctRight (R: URect): Integer;

Procedure UWndBelow (Var Below: UWindow; Above: UWindow; Between: Integer);

Procedure UWndRight (Var RightW: UWindow; LeftW: UWindow; Between: Integer);

Function  URctExpand (R: URect; Amt: Integer): URect;

Function  UWndOutside (W: UWindow): URect;

Function  USysWindow (Outside: URect; HasTitle: Boolean): UWindow;

Function  UCdInRect (C: UCoord; R: URect): Boolean;

Function  UCdInWindow (C: UCoord; W: UWindow): Boolean;

Function  UCdInBorder (C: UCoord; W: UWindow): Boolean;

{ *************************************************************************** }

{ ****************************** PAINT SECTION ****************************** }

Type    UColor = (UComplement, UTransparent, UOnly, UDefault, USame,
                UBlack, UWhite, UInvert, UGrey, UBackGnd, UForeGnd);

        UPntColor = UBlack .. UForeGnd;
        UCrsColor = UComplement .. UInvert;
        UBckColor = UDefault .. UWhite;

Procedure UPaint (C: UPntColor; R: URect);

Procedure UDrwWindow (W: UWindow);

Procedure UInvBorder (W: UWindow);

Procedure UStrCenter (R: Urect; S: UStr255);

Procedure USetCrsFunction (NBackColor: UBckColor; NCursColor: UCrsColor);

{ *************************************************************************** }

{ ****************************** CURSOR SECTION ***************************** }

Type    UPCrsList = ^UCrsList;

        UCrsList = Record
                NumRects, MaxRects: Integer;
                Rects: Array [1..1] of URect;
                End;

        UCrsBtnColor = (UYelBtn, UWhtBtn, UBluBtn, UGrnBtn);

        UCrsBtnType = Set of UCrsBtnColor;

Var     UCursor: UCoord;
        UButtons: UCrsBtnType;

Procedure UNewCrsList (Var List: UPCrsList; Max: Integer);

Procedure UFreCrsList (Var List: UPCrsList);

Procedure UOnCursor (R: URect);

Procedure UOffCursor;

Procedure UPollCursor;

Procedure UPollButtons;

Procedure UOffButton;

Procedure UOnButton;

Procedure UClickButton;

Function  UCrsInRect (List: UPCrsList): Integer;

{ *************************************************************************** }

{ ********************** SCREEN SAVE & RESTORE SECTION ********************** }

Function  ULineLength (R: URect): Integer;

Function  URctWords (R: URect): Long;

Function  USaveRect (R: URect): Pointer;

Procedure URstorRect (R: URect; P: Pointer);

Procedure USaveWindow (Var W: UWindow);

Procedure URstorWindow (Var W: UWindow);

{ *************************************************************************** }

{ ****************************** "GET" SECTION ****************************** }

Const   UPopX = 384;
        UPopY = 350;
        UPopMaxY = 500;

Var     UInScrBPrompt, UOutScrBPrompt: URect;
        UAbortAction: (UExitProgram, URaiseException, UDisallow);

Procedure UPrompt (Prompt: String);

Procedure UErasePrompt;

Function  UGetBool (Prompt, TrueAns, FalseAns: String): Boolean;

Function  UGet5Choices (Prompt, C1, C2, C3, C4, C5: String): Integer;

Function  UGet10Choices (Prompt: String; C1, C2, C3, C4, C5,
                                C6, C7, C8, C9, C10: UStr25): Integer;

Function  UGetFile (Prompt, Directory, Pattern: String): String;

Function  UGetInt (Prompt: String; min, max, def: Integer): Integer;

Function  UGetPoint (Prompt: String; R: URect): UCoord;

Function  UGetRect (Prompt: String; R: URect): URect;

Procedure UPlaceRect (Prompt: String; Var Place: URect; R: URect);

Procedure UPlaceWindow (Prompt: String; Var Place: UWindow; R: URect);

Exception UAbort;

{ *************************************************************************** }

{ ******************************* } PRIVATE { ******************************* }

{ ************** Private imports and global, initialized data *************** }

{ STRING HACKING SECTION }
Imports IO_Unit from IO_Unit;

{ MEMORY MANAGEMENT SECTION }
Imports Memory from Memory;
Imports Dynamic from Dynamic;

{ DIRECTORY SECTION }
Imports FileUtils from FileUtils;
Imports Sail_String from Sail_String;
Imports PMatch from PMatch;
Imports PopUp from PopUp;

{ SYSTEM WINDOW SECTION }
Imports Screen from Screen;

Var     AllocdSysWinds: Set of UWindNum;

{ PAINT SECTION }
Imports System from System;

Const   LineLength = 4;         { must be multiple of 4 words }
        BlockHeight = 8;        { arbitrary power of 2 }
        BlockWidth = 16 * LineLength;

Type    MyBlock = Packed Array [1..BlockHeight, 1..BlockWidth] of Boolean;
        PtrBlock = ^MyBlock;

Var     Grey: PtrBlock;
        BackBlack: Boolean;
        CursColor: 0..3;

{ CURSOR SECTION }
Imports IO_Others from IO_Others;

Var     Restrict: URect;
        OldTabCd: UCoord;

{ "GET" SECTION }
Const   CharHeight = 13;
        CharWidth = 9;

Var     PromptWindow: UWindow;


{ *************************************************************************** }

{ ************************** INITIALIZE EVERYTHING ************************** }

Procedure UInit (V: Integer);   { Initialize utility package }

        Var     i: Integer;

        Begin
        UCheckVersion (V, 'This program needs recompiled!');
        { MEMORY MANAGEMENT SECTION }
        UHeap := 0;     { use default (i.e. system) heap segment }

        { DIRECTORY SECTION }
        InitPopUp;

        { SYSTEM WINDOW SECTION }
        AllocdSysWinds := [0];
        UCurrWindow := 0;

        { AREA SECTION }
        {$ifc UScreenW <> SScreenW then }
        {$message 'UScreenW is not the same as SScreenW'}
        End.
        {$endc}
        UScreenP := ReCast (SScreenP, Pointer);

        with UScrOutside do
            begin
            UTop := 0;
            ULeft := 0;
            UHeight := 1024;
            UWidth := 768;
            end;
        UScrWindow := USysWindow (UScrOutside, True);
        UScrWindow . UThick := 0;
        UScrInside := UScrWindow . UInside;

        { PAINT SECTION }
        UNew (Grey, WordSize (MyBlock), 4);         { for quad-word alignment }
        for i := 1 to BlockHeight do
            begin
            Grey ^ [i, 16] := i mod 2 = 0;
            Grey ^ [i, 15] := i mod 2 <> 0;        { initialize left two cols }
            end;
        i := 2;
        while i < BlockWidth do    { copy as many cols as already initialized }
            begin
            RasterOp (RRpl, i, BlockHeight,
                        i, 0, LineLength, Grey,
                        0, 0, LineLength, Grey);
            i := i * 2;
            end;
        CursColor := DefCursFunct div 2;
        BackBlack := (CursColor = 1) = (DefCursFunct mod 2 = 0);

        { CURSOR SECTION }
        Restrict := UScrOutside;

        { "GET" SECTION }
        with PromptWindow, UInside do
            begin
            UStyle := UInvBand;
            UThick := 8;
            UTop := 20 + UThick;
            end;
        UInScrBPrompt := UScrInside;
        with PromptWindow, UInside do
            i := UTop + UThick + 5 * (CharHeight + 2) + 30;
        with UInScrBPrompt do
            begin
            UTop := i;
            UHeight := UHeight - i;
            end;
        UOutScrBPrompt := UScrOutside;
        with UOutScrBPrompt do
            begin
            UTop := i;
            UHeight := UHeight - i;
            end;
        UAbortAction := UExitProgram;
        End;

Procedure UCheckVersion (V: Integer; Mes: String);   { Check version number }

        Begin
        if V <> UVersion then
            begin
            IOBeep;
            writeln ('Util has changed:  ', Mes);
            Raise ExitProgram;
            end;
        End;

{ *************************************************************************** }

{ ************************* STRING HACKING SECTION ************************** }

Function UStrCmp (Var s1, s2: UStr255): Integer;

        Const   cnv = 32;       { = LXOr (ord ('A'), ord ('a')) = 0100000 }
                al = 'a';
                zl = 'z';
                A = 'A';
                Z = 'Z';

        Var     c1, c2: Char;
                notdone: Boolean;
                i: Integer;

        Begin
        i := 1;
        notdone := true;        { becomes false when characters differ }
        while (i <= length (s1)) and (i <= length (s2)) and notdone do
            begin
            if s1 [i] <> s2 [i] then
                begin
                c1 := s1 [i];
                c2 := s2 [i];
                if (c1 < A) or ((c1 > Z) and (c1 < al)) or (c1 > zl) or
                            (c2 < A) or ((c2 > Z) and (c2 < al)) or (c2 > zl)
                    then notdone := false
                    else begin
                        c1 := chr (LOr (ord (c1), cnv));
                        c2 := chr (LOr (ord (c2), cnv));
                        notdone := c1 = c2;
                        end;
                end;
            i := i + 1;
            end;
        if notdone              { characters do not differ }
            then begin
                if length (s1) = length (s2)
                    then UStrCmp := 0
                    else if length (s1) < length (s2)
                        then UStrCmp := -1
                        else UStrCmp := 1;
                end
            else begin
                if c1 < c2
                    then UStrCmp := -1
                    else UStrCmp := 1;
                end;
        End;

Procedure UCnvUpper (Var S: Ustr255);

        Const   cnv = ord ('a') - ord ('A');

        Var     i: Integer;

        Begin
        for i := 1 to Length (S) do
            if (S [i] >= 'a') and ( S [i] <= 'z') then
                S [i] := chr (ord (S [i]) - cnv);
        End;

Procedure UBeep;

        Begin
        IOBeep;
        End;

{ *************************************************************************** }

{ *********************** ARITHMETIC HACKING SECTION ************************ }

Function IMin (a, b: Integer): Integer;

        Begin
        if a < b
            then IMin := a
            else IMin := b;
        End;

Function IMax (a, b: Integer): Integer;

        Begin
        if a > b
            then IMax := a
            else IMax := b;
        End;

{ *************************************************************************** }

{ ************************ MEMORY MANAGEMENT SECTION ************************ }

Procedure UNewSegment;

        Begin
        write ('Old segment was ');
        if UHeap = 0
            then write ('default heap: ', MMHeap:1)
            else write (UHeap:1);
        CreateSegment (UHeap, 4, 4, 256);
        writeln ('; new segment is ', UHeap:1);
        End;

Procedure UNew (Var P: Pointer; Size, Align: Integer);

        Label   1;

        Var     Once: Boolean;

        Handler FullSegment;

                Begin
                if Once
                    then Raise FullSegment
                    else begin
                        UNewSegment;
                        Once := True;
                        GoTo 1;
                        end;
                End;

        Var     Trick: Record
                        Case Boolean of
                            False: (P: Pointer);
                            True:  (M: MMPointer);
                        End;

        Begin
        if Align < 1 then Align := 1;
        Once := False;
1:      NewP (UHeap, Align, Trick . M, Size);
        P := Trick . P;
        End;

Function  UBlocks (Words: Long): Integer;

        Begin
        UBlocks := Shrink ((Words + UWordsPBlock - 1) div UWordsPBlock);
        End;

Function  UCreateSegment (InitSize, Inc, MaxSize: Integer): Integer;

        Var     Seg: SegmentNumber;

        Begin
        CreateSegment (Seg, InitSize, Inc, MaxSize);
        UCreateSegment := Seg;
        End;

Procedure UCreatePSegment (Var P: Pointer; Words: Long);

        Var     Seg: SegmentNumber;
                Size: Integer;

        Begin
        Size := Shrink ((Words + UWordsPBlock - 1) div UWordsPBlock);
        CreateSegment (Seg, Size, 1, Size);
        P := MakePtr (Seg, 0, Pointer);
        End;

Procedure UIncSegSize (P: Pointer);

        Var     up: UPointer;

        Begin
        up . p := p;
        {$R-}
        with up, SAT^ [Segment], SIT^ [Segment] do
            ChangeSize (Segment, Size + Increment + 2);
        {$R+}
        End;

Procedure UDispSegment (Seg: Integer);

        Begin
        DecRefCount (Seg);
        End;

Procedure UDispPSegment (Var P: Pointer);

        Var     Trick: UPointer;

        Begin
        Trick . P := P;
        DecRefCount (Trick . Segment);
        P := nil;
        End;

Procedure UAllocNameDesc (N: Integer; Var P: Pointer);

        Label   1;

        Var     Once: Boolean;

        Handler FullSegment;

                Begin
                if Once
                    then Raise FullSegment
                    else begin
                        UNewSegment;
                        Once := True;
                        GoTo 1;
                        end;
                End;

        Var     PND: PNameDesc;

        Begin
        Once := False;
1:      AllocNameDesc (N, UHeap, PND);
        P := Recast (PND, Pointer);
        End;

{ *************************************************************************** }

{ **************************** DIRECTORY SECTION **************************** }

Var     NumFiles: Integer;

Procedure USplit (Var Original, Directory, FileName: UStr255);

        Var     Pos: Integer;
                NoDirec: Boolean;
                FilePart: UStr25;

        Begin
        Pos := Length (Original);
        NoDirec := True;
        while (Pos > 0) and NoDirec do
            if Original [Pos] = '>'
                then NoDirec := False
                else Pos := Pos - 1;
        if NoDirec
            then begin
                FileName := Original;
                Directory := '';
                end
            else begin
                FilePart := SubStrFor (Original, Pos + 1, inf);
                Directory := SubStrTo (Original, 1, Pos);
                FileName := FilePart;
                end;
        end;

Function UDirectory (DirName, Pattern: UStr255): UPtrFile;

        Var     FileName: UStr25;
                FI: FileId;
                ScanPtr: ptrScanRecord;
                List, Current, previous, subsequent: UPtrFile;
                NotFound: Boolean;
                DirPart: UStr255;
                

        Begin
        USplit (Pattern, DirPart, Pattern);
        if Pattern = '' then Pattern := '*';
        if length (DirName) > 0 then
            if DirName [length (DirName)] <> '>' then
                AppendChar (DirName, '>');
        AppendString (DirName, DirPart);
        UNew (ScanPtr, WordSize (ScanRecord), 1);
        ScanPtr^ . InitialCall := True;
        ScanPtr^ . DirName := DirName;
        NumFiles := 0;
        List := nil;
        while FSScan (ScanPtr, FileName, FI) do
            begin
            if PattMatch (FileName, Pattern, True) then
                begin
                NumFiles := NumFiles + 1;
                UNew (Current, WordSize (UFile), 1);
                with Current^ do
                    begin
                    name := FileName;
                    fid := FI;
                    end;

                previous := nil;        { alphabetize }
                subsequent := List;
                NotFound := True;
                while (subsequent <> nil) and NotFound do
                    if UStrCmp (subsequent^ . name, FileName) < 0
                        then begin
                            previous := subsequent;
                            subsequent := subsequent^ . next;
                            end
                        else NotFound := false;
                Current^ . next := subsequent;
                if previous = nil
                    then List := Current
                    else previous^ . next := Current;
                end;
            end;
        Dispose (ScanPtr);
        UDirectory := List;
        End;

Procedure UDispDirectory (Var PF: UPtrFile);

        Var     ThisPF, NextPF: UPtrFile;

        Begin
        ThisPF := PF;
        while ThisPF <> nil do
            begin
            NextPF := ThisPF^ . next;
            Dispose (ThisPF);
            ThisPF := NextPF;
            end;
        End;

Procedure UDirPopList (Var List: Pointer; Var DirName, Pattern: UStr255);

        Var     PND: PNameDesc;
                Direc, Disp: UPtrFile;
                i: Integer;

        Begin
        Direc := UDirectory (DirName, Pattern);
        UAllocNameDesc (NumFiles, PND);
        for i := 1 to NumFiles do
            begin
            {$R-  Turn off array subscript range checking }
            PND^ . Commands [i] := Direc^ . Name;
            {$R+  Turn checking back on again }
            Disp := Direc;
            Direc := Direc^ . Next;
            Dispose (Disp);
            end;
        { PND^ . NumCommands := NumFiles; (should be redundant) }
        List := ReCast (PND, Pointer);
        End;

{ *************************************************************************** }

{ ************************** SYSTEM WINDOW SECTION ************************** }

Function UNewWindNum: Integer;

        Var     W: UWindNum;

        Begin
        for W := 0 to 17 do
            if not (W in AllocdSysWinds) then
                begin
                UNewWindNum := W;
                AllocdSysWinds := AllocdSysWinds + [W];
                Exit (UNewWindNum);
                end;
        UNewWindNum := -1;
        End;

Procedure UUseWindNum (W: UWindNum);

        Begin
        AllocdSysWinds := AllocdSysWinds + [W];
        End;

Procedure UFreeWindNum (W: UWindNum);

        Begin
        AllocdSysWinds := AllocdSysWinds - [W];
        End;

Procedure UChangeWindow (W: UWindNum);

        Begin
        if W in AllocdSysWinds then
            begin
            ChangeWindow (W);
            UCurrWindow := W;
            end;
        End;

Procedure UTitle (S: String);

        Begin
        ChangeTitle (s);
        End;

{ *************************************************************************** }

{ ******************************* AREA SECTION ****************************** }

Const   SysLeft = 5;
        SysRight = 6;
        SysWidth = SysLeft + SysRight;
        SysTTop = 20;
        SysNoTTop = 5;
        SysBot = 6;
        SysTHeight = SysTTop + SysBot;
        SysNoTHeight = SysNoTTop + SysBot;

Function UCdToRect (C1, C2: UCoord): URect;

        Var     R: URect;

        Begin
        with R do
            begin
            if C1 . UX > C2 . UX
                then begin
                    UTop := C2 . UX;
                    UHeight := C1 . UX - UTop;
                    end
                else begin
                    UTop := C1 . UX;
                    UHeight := C2 . UX - UTop;
                    end;
            if C1 . UY > C2 . UY
                then begin
                    ULeft := C2 . UY;
                    UWidth := C1 . UY - ULeft;
                    end
                else begin
                    ULeft := C1 . UY;
                    UWidth := C2 . UY - ULeft;
                    end;
            end;
        UCdToRect := R;
        End;

Function URctBottom (R: URect): Integer;

        Begin
        with R do
            URctBottom := UTop + UHeight;
        End;

Function URctRight (R: URect): Integer;

        Begin
        with R do
            URctRight := ULeft + UWidth;
        End;

Procedure UWndBelow (Var Below: UWindow; Above: UWindow; Between: Integer);

        Var     Top: Integer;

        Begin
        with Above, UInside do
            begin
            Top := UTop + UHeight;
            case UStyle of
                UNoBorder:      { do nothing };
                USysTWindow,
                USysNoTWindow:  Top := Top + SysBot;
                Otherwise:      Top := Top + UThick;
                end;
            end;
        Top := Top + Between;
        with Below, UInside do
            begin
            case UStyle of
                UNoBorder:      UTop := Top;
                USysTWindow:    UTop := Top + SysTTop;
                USysNoTWindow:  UTop := Top + SysNoTTop;
                Otherwise:      UTop := Top + UThick;
                end;
            end;
        End;

Procedure UWndRight (Var RightW: UWindow; LeftW: UWindow; Between: Integer);

        Var     Left: Integer;

        Begin
        with LeftW, UInside do
            begin
            Left := ULeft + UHeight;
            case UStyle of
                UNoBorder:      { do nothing };
                USysTWindow,
                USysNoTWindow:  Left := Left + SysRight;
                Otherwise:      Left := Left + UThick;
                end;
            end;
        Left := Left + Between;
        with RightW, UInside do
            begin
            case UStyle of
                UNoBorder:      ULeft := Left;
                USysTWindow,
                USysNoTWindow:  ULeft := Left + SysLeft;
                Otherwise:      ULeft := Left + UThick;
                end;
            end;
        End;

Function URctExpand (R: URect; Amt: Integer): URect;

        Begin
        with R do
            begin
            UTop := UTop - Amt;
            ULeft := ULeft - Amt;
            UHeight := UHeight + 2 * Amt;
            UWidth := UWidth + 2 * Amt;
            end;
        URctExpand := R;
        End;

Function UWndOutside (W: UWindow): URect;

        Var     outside: URect;

        Begin
        case W . UStyle of
            UNoBorder:      outside := W . UInside;
            USysTWindow:    with W . UInside do
                                begin
                                outside . UTop := UTop - SysTTop;
                                outside . ULeft := ULeft - SysLeft;
                                outside . UHeight := UHeight + SysTHeight;
                                outside . UWidth := UWidth + SysWidth;
                                end;
            USysNoTWindow:  with W . UInside do
                                begin
                                outside . UTop := UTop - SysNoTTop;
                                outside . ULeft := ULeft - SysLeft;
                                outside . UHeight := UHeight + SysNoTHeight;
                                outside . UWidth := UWidth + SysWidth;
                                end;
            Otherwise:      outside := URctExpand (W . UInside, W . UThick);
            end;
        UWndOutside := outside;
        End;

Function USysWindow (Outside: URect; HasTitle: Boolean): UWindow;

        Var     W: UWindow;

        Begin
        with W, UInside do
            begin
            if HasTitle
                then begin
                    UStyle := USysTWindow;
                    UTop := Outside . UTop + SysTTop;
                    UHeight := Outside . UHeight - SysTHeight;
                    end
                else begin
                    UStyle := USysNoTWindow;
                    UTop := Outside . UTop + SysNoTTop;
                    UHeight := Outside . UHeight - SysNoTHeight;
                    end;
            UThick := -1;
            ULeft := Outside . ULeft + SysLeft;
            UWidth := Outside . UWidth - SysWidth;
            end;
        USysWindow := W;
        End;

Function UCdInRect (C: UCoord; R: URect): Boolean;

        Begin
        with C, R do
            UCdInRect := (UX >= ULeft) and (UX < ULeft + UWidth)
                        and (UY >= UTop) and (UY < UTop + UHeight);
        End;

Function UCdInWindow (C: UCoord; W: UWindow): Boolean;

        Begin
        UCdInWindow := UCdInRect (C, UWndOutside (W));
        End;

Function UCdInBorder (C: UCoord; W: UWindow): Boolean;

        Begin
        UCdInBorder := UCdInWindow (C, W) and not UCdInRect (C, W . UInside);
        End;

{ *************************************************************************** }

{ ****************************** PAINT SECTION ****************************** }

Procedure UPaint (C: UPntColor; R: URect);

        Var     rastfnc: Integer;
                Solid: Boolean;
                Use: PtrBlock;
                curw, curh: Integer;  { amount to fill this time }
                alrw, alrh: Integer;  { amount filled already }

        Begin
        with R do
            begin
            if (Uwidth < 1) or (Uheight < 1)
                then Exit (UPaint);

            Solid := True;
            Case C of
                UInvert:    rastfnc := RNot;
                UBackGnd:   rastfnc := RXor;
                UForeGnd:   rastfnc := RXNor;
                UGrey:      begin
                                Solid := False;
                                Use := Grey;
                                if (Utop + Uleft) mod 2 = 0
                                    then rastfnc := RRpl
                                    else rastfnc := RNot;
                            end;
                UBlack:     if BackBlack
                                then rastfnc := RXNor
                                else rastfnc := RXor;
                UWhite:     if BackBlack
                                then rastfnc := RXor
                                else rastfnc := RXNor;
                End;

            if Solid then
                begin
                RasterOp (rastfnc, Uwidth, Uheight,
                                Uleft, Utop, SScreenW, SScreenP,
                                Uleft, Utop, SScreenW, SScreenP);
                Exit (UPaint);
                end;

            alrw := imin (Uwidth, BlockWidth);
            alrh := imin (Uheight, BlockHeight);
            RasterOp (rastfnc, alrw, alrh, Uleft, Utop, SScreenW, SScreenP,
                                    0, 0, LineLength, Use);
            while alrw < Uwidth do
                begin
                curw := imin (Uwidth - alrw, alrw);
                RasterOp (RRpl, curw, alrh,
                                    Uleft + alrw, Utop, SScreenW, SScreenP,
                                    Uleft, Utop, SScreenW, SScreenP);
                alrw := alrw + curw;
                end;
            while alrh < Uheight do
                begin
                curh := imin (Uheight - alrh, alrh);
                RasterOp (RRpl, Uwidth, curh,
                                    Uleft, Utop + alrh, SScreenW, SScreenP,
                                    Uleft, Utop, SScreenW, SScreenP);
                alrh := alrh + curh;
                end;
            end;
        End;

Procedure UDrwWindow (W: UWindow);

        Var     c: UPntColor;
                t: Integer;
                out: URect;
                s: String [1];

        Begin
        out := UWndOutside (W);
        with W do
            begin
            Case UStyle of
                USysTWindow,
                USysNoTWindow: begin
                        if not (UThick in AllocdSysWinds) then
                            UThick := UNewWindow;
                        if UStyle = USysNoTWindow
                            then s := ''
                            else s := ' ';
                        with out do
                            CreateWindow (UThick, ULeft, UTop,
                                        UWidth, UHeight, s);
                        UCurrWindow := UThick;
                        Exit (UDrwWindow);
                        end;
                UNoBorder: begin
                        UPaint (UBackGnd, UInside);
                        Exit (UDrwWindow);
                        end;
                USolid: begin
                        c := UForeGnd;
                        t := 0;
                        end;
                UInvSolid: begin
                        c := UBackGnd;
                        t := 0;
                        end;
                UBand: begin
                        c := UForeGnd;
                        t := UThick div 2;
                        end;
                UInvBand: begin
                        c := UBackGnd;
                        t := UThick div 2;
                        end;
                end;
            UPaint (c, out);
            if t > 0 then UPaint (UInvert, URctExpand (UInside, t));
            if (UStyle = USolid) or (UStyle = UInvBand)
                then UPaint (UInvert, UInside);
            end;
        End;

Procedure UInvBorder (W: UWindow);

        Var     inv, out: URect;

        Begin
        out := UWndOutside (W);
        inv := out;
        with W, inv do
            begin
            Uheight := UInside . UTop - out . Utop;
            UPaint (UInvert, inv);               { top }
            Utop := URctBottom (UInside);
            Uheight := URctBottom (out) - Utop;
            UPaint (UInvert, inv);               { bottom }
            Uheight := UInside . UHeight;
            Utop := UInside . UTop;
            Uwidth := UInside . ULeft - Uleft;
            UPaint (UInvert, inv);               { left side }
            Uleft := URctRight (UInside);
            Uwidth := URctRight (out) - Uleft;
            UPaint (UInvert, inv);               { right side }
            end;
        End;

Procedure UStrCenter (R: Urect; S: UStr255);

        Const   ChrHeight = 13;
                ChrWidth = 9;

        Var     oldx, oldy: Integer;
                StrWidth: Integer;

        Begin
        if UCurrWindow <> 0 then
            ChangeWindow (0);
        SReadCursor (oldx, oldy);
        StrWidth := ChrWidth * length (s);
        with r do
            SSetCursor (uleft + (uwidth - StrWidth) div 2,
                        utop + (uheight + ChrHeight) div 2);
        write (s);
        SSetCursor (oldx, oldy);
        if UCurrWindow <> 0 then
            ChangeWindow (UCurrWindow);
        End;

Procedure USetCrsFunction (NBackColor: UBckColor; NCursColor: UCrsColor);

        Var     CF: CursFunction;

        Begin
        case NBackColor of
            UDefault:  BackBlack := (DefCursFunct div 2 = 1) =
                                (DefCursFunct mod 2 = 0);
            USame:     { no change };
            UBlack:    BackBlack := True;
            UWhite:    BackBlack := False;
            end;

        case NCursColor of
            UDefault:      CursColor := DefCursFunct div 2;
            USame:         { no change };
            UBlack,
            UWhite,
            UComplement:   CursColor := 2;
            UInvert:       CursColor := 3;
            UTransparent:  CursColor := 1;
            UOnly:         CursColor := 0;
            end;

        CF := Recast (CursColor * 2, CursFunction);
        if BackBlack <> (CursColor = 1)
            then CF := succ (CF);
        IOSetFunction (CF);
        End;

{ *************************************************************************** }

{ ****************************** CURSOR SECTION ***************************** }

Procedure UNewCrsList (Var List: UPCrsList; Max: Integer);

        Var     Size: Integer;

        Begin
        Size := 2 + Max * WordSize (URect);
        UNew (List, Size, 1);
        with List^ do
            begin
            MaxRects := Max;
            NumRects := 0;
            end;
        End;

Procedure UFreCrsList (Var List: UPCrsList);

        Var     Size: Integer;

        Begin
        Size := 2 + List^ . MaxRects * WordSize (URect);
        DisposeP (ReCast (List, MMPointer), Size);
        End;

Procedure UOnCursor (R: URect);

        Begin
        Restrict := R;
        with Restrict, UCursor do
            begin
            if UX < ULeft
                then UX := ULeft
                else if UX >= ULeft + UWidth
                    then UX := ULeft + UWidth - 1;
            if UY < UTop
                then UY := UTop
                else if UY >= UTop + UHeight
                    then UY := UTop + UHeight - 1;
            end;
        IOSetModeTablet (TabAbsTablet);
        IOCursorMode (IndepCursor);
        with UCursor do
            IOSetCursorPos (UX, UY);
        with OldTabCd do
            IOReadTablet (UX, UY);
        End;

Procedure UOffCursor;

        Begin
        IOCursorMode (OffCursor);
        IOSetModeTablet (OffTablet);
        End;

Procedure UPollCursor;

        Const   TabJump = 2500;  { Maximum distance (squared) mouse can move }
                                 { between polls without assuming it jumped }

        Var     New: UCoord;
                dx, dy: Long;

        Begin
        with New do
            begin
            IOReadTablet (UX, UY);         { get coordinates of mouse }
            dx := UX - OldTabCd . UX;
            dy := UY - OldTabCd . UY;
            end;
        if (dx * dx + dy * dy) < TabJump
            then with UCursor do
                    begin
                    UX := UX + Shrink (dx);    { Move relative }
                    UY := UY - Shrink (dy);    { (Y is reversed) }

                    with Restrict do
                        begin                        { clip }
                        if UX < ULeft
                            then UX := ULeft
                            else if UX >= ULeft + UWidth
                                then UX := ULeft + UWidth - 1;
                        if UY < UTop
                            then UY := UTop
                            else if UY >= UTop + UHeight
                                then UY := UTop + UHeight - 1;
                        end;
                    IOSetCursorPos (UX, UY);   { update screen cursor }
                    end;
        OldTabCd := New;
        UPollButtons;
        End;

Procedure UPollButtons;

        Begin
        UButtons := [];
        if TabYellow then UButtons := UButtons + [UYelBtn];
        if TabWhite  then UButtons := UButtons + [UWhtBtn];
        if TabBlue   then UButtons := UButtons + [UBluBtn];
        if TabGreen  then UButtons := UButtons + [UGrnBtn];
        End;

Procedure UOffButton;

        Var     Prev: UCrsBtnType;

        Begin
        repeat
            Prev := UButtons;
            UPollCursor;
            until UButtons = [];
        UButtons := Prev;
        End;

Procedure UOnButton;

        Begin
        Repeat
            UPollCursor;
            Until UButtons <> [];
        End;

Procedure UClickButton;

        Begin
        UOnButton;
        UOffButton;
        End;

Function  UCrsInRect (List: UPCrsList): Integer;

        Var     i: Integer;

        Begin
        with List^ do
            for i := NumRects downto 1 do
                begin
                {$R-    Turn off array bound checking }
                if UCdInRect (UCursor, Rects [i]) then
                    begin
                    {$R+    Turn checking back on }
                    UCrsInRect := i;
                    Exit (UCrsInRect);
                    end;
                end;
        UCrsInRect := 0;
        End;

{ *************************************************************************** }

{ ********************** SCREEN SAVE & RESTORE SECTION ********************** }

Function  ULineLength (R: URect): Integer;

        Begin
        with R do
            ULineLength := ((UWidth + 63) div 64) * 4;
        End;

Function  URctWords (R: URect): Long;

        Begin
        with R do
            URctWords := ((UWidth + 63) div 64) * 4 * Stretch (UHeight);
        End;

Function  USaveRect (R: URect): Pointer;

        Var     LLen: Integer;
                Size: Long;
                P: Pointer;

        Begin
        with R do
            begin
            LLen := ((UWidth + 63) div 64) * 4;
            Size := LLen * Stretch (UHeight);
            if Size > 1024
                then UCreatePSegment (P, Size)
                else UNew (P, Shrink (Size), 4);
            RasterOp (RRpl, UWidth, UHeight,
                                0,     0,    LLen,     P,
                                ULeft, UTop, SScreenW, SScreenP);
            end;
        USaveRect := P;
        End;

Procedure URstorRect (R: URect; P: Pointer);

        Var     LLen: Integer;
                Size: Long;

        Begin
        with R do
            begin
            LLen := ((UWidth + 63) div 64) * 4;
            Size := LLen * Stretch (UHeight);
            RasterOp (RRpl, UWidth, UHeight,
                                ULeft, UTop, SScreenW, SScreenP,
                                0,     0,    LLen,     P);
            end;
        if Size > 1024
            then UDispPSegment (P)
            else DisposeP (ReCast (P, MMPointer), Shrink (Size));
        End;

Procedure USaveWindow (Var W: UWindow);

        Begin
        W . USaved := USaveRect (UWndOutside (W));
        End;

Procedure URstorWindow (Var W: UWindow);

        Begin
        URstorRect (UWndOutside (W), W . USaved);
        End;

{ *************************************************************************** }

{ ****************************** "GET" SECTION ****************************** }

Procedure UPrompt (Prompt: String);

        Begin
        with PromptWindow . UInside do
            begin
            UHeight := CharHeight + 30;
            UWidth := CharWidth * Length (Prompt) + 40;
            ULeft := (768 - UWidth) div 2;
            end;
        USaveWindow (PromptWindow);
        UDrwWindow (PromptWindow);
        UStrCenter (PromptWindow . UInside, Prompt);
        End;

Procedure UErasePrompt;

        Begin
        URstorWindow (PromptWindow);
        End;

Function  UGetBool (Prompt, TrueAns, FalseAns: String): Boolean;

        Begin
        if length (TrueAns) < 1 then TrueAns := ' ';
        if length (FalseAns) < 1 then FalseAns := ' ';
        UGetBool := UGet5Choices (Prompt, TrueAns, FalseAns, '', '', '') = 1;
        End;

Function  UGet5Choices (Prompt, C1, C2, C3, C4, C5: String): Integer;

        Var     Choices: Array [1..5] of String;
                n, i, len, LastOne, NextOne: Integer;
                Restrict, PrmptRect: URect;
                done, BeenPressed: Boolean;
                List: UPCrsList;

        Begin
        Choices [1] := C1;
        Choices [2] := C2;
        Choices [3] := C3;
        Choices [4] := C4;
        Choices [5] := C5;
        n := 1;
        done := false;
        repeat
            if Choices [n] <> ''
                then n := n + 1
                else done := true;
            until (n > 5) or done;
        n := n - 1;
        if n < 2 then
            begin
            UGet5Choices := 1;
            Exit (UGet5Choices);
            end;

        with PrmptRect do
            begin
            UTop := PromptWindow . UInside . UTop + 15;
            UHeight := (CharHeight + 2) * n;
            UWidth := CharWidth * Length (Prompt) + 20;
            end;
        Restrict := PrmptRect;
        len := Length (Choices [1]);
        for i := 2 to n do
            if Length (Choices [i]) > len
                then len := Length (Choices [i]);
        Restrict . UWidth := len * CharWidth + 20;
        with PromptWindow, UInside do
            begin
            UWidth := Restrict . UWidth + PrmptRect . UWidth + 10;
            UHeight := PrmptRect . UHeight + 30;
            ULeft := (768 - UWidth) div 2;
            PrmptRect . ULeft := ULeft;
            end;
        with PrmptRect do
            Restrict . ULeft := ULeft + UWidth;
        USaveWindow (PromptWindow);
        UDrwWindow (PromptWindow);
        UNewCrsList (List, n);
        with List^ do
            begin
            NumRects := n;
            {$R-}
            Rects [1] := Restrict;
            Rects [1] . UHeight := CharHeight + 2;
            for i := 2 to n do
                begin
                Rects [i] := Rects [i - 1];
                with Rects [i] do
                    UTop := UTop + UHeight;
                end;
            for i := 1 to n do
                UStrCenter (Rects [i], Choices [i]);
            {$R+}
            end;
        UStrCenter (PrmptRect, Prompt);

        UOnCursor (Restrict);
        LastOne := UCrsInRect (List);
        {$R-}
        if LastOne > 0 then UPaint (UInvert, List^ . Rects [LastOne]);
        {$R+}
        BeenPressed := False;
        repeat
            UPollCursor;
            NextOne := UCrsInRect (List);
            if NextOne <> LastOne then
                begin
                {$R-}
                if LastOne > 0 then UPaint (UInvert, List^ . Rects [LastOne]);
                if NextOne > 0 then UPaint (UInvert, List^ . Rects [NextOne]);
                {$R+}
                LastOne := NextOne;
                end;
            if (not BeenPressed) and (UButtons <> [])
                then BeenPressed := True;
            until BeenPressed and (UButtons = []);
        UOffCursor;
        UFreCrsList (List);
        URstorWindow (PromptWindow);
        UGet5Choices := NextOne;
        End;

Function  UGet10Choices (Prompt: String; C1, C2, C3, C4, C5,
                                C6, C7, C8, C9, C10: UStr25): Integer;

        Label   1, 2;

        Var     PND: PNameDesc;

        Handler Outside;

                Begin
                Case UAbortAction of
                    UExitProgram: begin
                        UErasePrompt;
                        Raise ExitProgram;
                        end;
                    URaiseException: begin
                        DestroyNameDesc (PND);
                        UErasePrompt;
                        Raise UAbort;
                        end;
                    UDisallow: begin
                        IOBeep;
                        GoTo 2;
                        end;
                    end;
                GoTo 1;     { if UAbort is continued }
                End;

        Var     N: Integer;
                done: Boolean;
                result: ResRes;

        Begin
1:      UAllocNameDesc (10, PND);
        with PND^ do
            begin
            Header := '';
            {$R-}
            Commands [1] := C1;
            Commands [2] := C2;
            Commands [3] := C3;
            Commands [4] := C4;
            Commands [5] := C5;
            Commands [6] := C6;
            Commands [7] := C7;
            Commands [8] := C8;
            Commands [9] := C9;
            Commands [10] := C10;
            n := 1;
            done := false;
            repeat
                if Commands [n] <> ''
                    then n := n + 1
                    else done := true;
                until (n > 10) or done;
            {$R+}
            n := n - 1;
            NumCommands := n;
            end;
        if n < 2 then
            begin
            DestroyNameDesc (PND);
            UGet10Choices := 1;
            Exit (UGet10Choices);
            end;
        UPrompt (Prompt);
2:      Menu (PND, False, 1, N, UPopX, UPopY, UPopMaxY, result);
        UErasePrompt;
        with result^ do
            if NumIndices > 0
                then UGet10Choices := indices [1]
                else UGet10Choices := 0;
        DestroyRes (result);
        DestroyNameDesc (PND);
        End;

Function  UGetFile (Prompt, Directory, Pattern: String): String;

        Label   1, 2;

        Var     List: PNameDesc;

        Handler Outside;

                Begin
                Case UAbortAction of
                    UExitProgram: begin
                        UErasePrompt;
                        Raise ExitProgram;
                        end;
                    URaiseException: begin
                        DestroyNameDesc (List);
                        UErasePrompt;
                        Raise UAbort;
                        end;
                    UDisallow: begin
                        IOBeep;
                        GoTo 2;
                        end;
                    end;
                GoTo 1;     { if UAbort is continued }
                End;

        Var     result: ResRes;
                CompilerBug: Integer;

        Begin
1:      UDirPopList (List, Directory, Pattern);
        if List^ . NumCommands < 2 then
            begin
            if List^ . NumCommands = 1
                then UGetFile := List^ . Commands [1]
                else UGetFile := '';
            DestroyNameDesc (List);
            Exit (UGetFile);
            end;
        List^ . Header := '';
        UPrompt (Prompt);
2:      Menu (List, False, 1, List^ . NumCommands,
                                UPopX, UPopY, UPopMaxY, result);
        UErasePrompt;
        with result^ do
            if NumIndices > 0
                then begin
                    CompilerBug := indices [1];
                    {$R-}
                    UGetFile := List^ . Commands [CompilerBug];
                    {$R+}
                    end
                else UGetFile := '';
        DestroyRes (result);
        DestroyNameDesc (List);
        End;

Function  UGetInt (Prompt: String; min, max, def: Integer): Integer;

        Type DelRange = -511 .. 511;

        Var     Restrict: URect;
                Answer, delta, oldx, oldy, x, y: Integer;
                acc: Integer;

        Function Del (y: DelRange): Integer;

                Var     n: Integer;

                Begin
                n := y div 32;
                acc := acc + n * n * n;
                del := acc div 128;
                acc := acc mod 128;
                End;

        Begin
        Answer := Def;
        with PromptWindow, UInside do
            begin
            UHeight := CharHeight + 30;
            UWidth := CharWidth * (Length (Prompt) + 7) + 40;
            ULeft := (768 - UWidth) div 2;
            end;
        USaveWindow (PromptWindow);
        UDrwWindow (PromptWindow);
        with Restrict do
            begin
            UTop := 1;
            UHeight := 1023;            {  ranges from 1 to 1023 }
            UWidth := 1;
            ULeft := URctRight (PromptWindow . UInside);
            end;
        if UCurrWindow <> 0 then
            ChangeWindow (0);
        SReadCursor (oldx, oldy);
        with PromptWindow . UInside do
            SSetCursor (ULeft + 20, UTop + (UHeight + CharHeight) div 2);
        write (prompt);
        SReadCursor (x, y);

        write (Answer: 7);
        UOnCursor (Restrict);
        UCursor . UY := 512;
        acc := 0;
        repeat
            UPollCursor;
            delta := del (512 - UCursor . UY);          { del (-511 .. 511) }
            if delta > 0
                then begin
                    if Max - delta < Answer
                        then delta := Max - Answer;
                    end
                else if Min - delta > Answer
                    then delta := Min - Answer;
            Answer := Answer + delta;
            SSetCursor (x, y);
            write (Answer: 7);
            until UButtons <> [];
        UOffButtons;

        UGetInt := Answer;
        URstorWindow (PromptWindow);
        SSetCursor (oldx, oldy);
        if UCurrWindow <> 0 then
            ChangeWindow (UCurrWindow);
        End;

Function  UGetPoint (Prompt: String; R: URect): UCoord;

        Begin
        UPrompt (Prompt);
        UOnCursor (R);
        UClickButton;
        UGetPoint := UCursor;
        UOffCursor;
        UErasePrompt;
        End;

Function  UGetRect (Prompt: String; R: URect): URect;

        Var     inv: URect;   { inv is only external variable used in OneDim }

        Procedure OneDim (origin, old, new: Integer;
                        Var istart, ilength, rstart, rlength: Integer);

                Begin
                if old < new
                    then begin
                        if origin <= old           { origin <= old < new }
                            then begin
                                istart := old + 1;
                                ilength := new - old;
                                UPaint (UInvert, inv);
                                rlength := rlength + ilength;
                                end
                            else if new <= origin  { old < new <= origin }
                                then begin
                                    { istart := old; }
                                    ilength := new - old;
                                    UPaint (UInvert, inv);
                                    rstart := new;
                                    rlength := rlength - ilength;
                                    end
                                else begin         { old < origin < new }
                                    { istart := old; }
                                    ilength := origin - old;
                                    UPaint (UInvert, inv);
                                    istart := origin + 1;
                                    ilength := new - origin;
                                    UPaint (UInvert, inv);
                                    rstart := origin;
                                    rlength := ilength + 1;
                                    end;
                        end
                    else if new < old then
                        begin
                        if origin <= new           { origin <= new < old }
                            then begin
                                istart := new + 1;
                                ilength := old - new;
                                UPaint (UInvert, inv);
                                rlength := rlength - ilength;
                                end
                            else if old <= origin  { new < old <= origin }
                                then begin
                                    istart := new;
                                    ilength := old - new;
                                    UPaint (UInvert, inv);
                                    rstart := new;
                                    rlength := rlength + ilength;
                                    end
                                else begin         { new < origin < old }
                                    istart := origin + 1;
                                    ilength := old - origin;
                                    UPaint (UInvert, inv);
                                    istart := new;
                                    ilength := origin - new;
                                    UPaint (UInvert, inv);
                                    rstart := new;
                                    rlength := ilength + 1;
                                    end;
                        end;
                End;

        Var     origin, old: UCoord;
                rect: URect;

        Begin
        UPrompt (Prompt);
        UOnCursor (R);
        UOnButton;
        origin := UCursor;
        old := origin;
        with rect do
           begin
           UUpLeft := origin;
           UHeight := 1;
           UWidth := 1;
           end;
        UPaint (UInvert, rect);
        repeat
            UPollCursor;
            inv := rect;
            OneDim (origin . UX, old . UX, UCursor . UX,
                                    inv . ULeft, inv . UWidth,
                                    rect . ULeft, rect . UWidth);
            inv := rect;
            OneDim (origin . UY, old . UY, UCursor . UY,
                                    inv . UTop, inv . UHeight,
                                    rect . UTop, rect . UHeight);
            old := UCursor;
            until (UButtons = []);
        UPaint (UInvert, rect);
        UErasePrompt;
        UGetRect := rect;
        End;

Procedure UPlaceRect (Prompt: String; Var Place: URect; R: URect);

        Var     inv: URect;

        Procedure OneDim (NewStart: Integer; Var Start: Integer;
                            Length: Integer; Var InvStart, InvLength: Integer);

                Begin
                if NewStart <> Start then
                    begin
                    inv := Place;
                    if NewStart > Start
                        then begin
                            InvLength := NewStart - Start;
                            UPaint (UInvert, inv);
                            InvStart := InvStart + Length;
                            UPaint (UInvert, inv);
                            end
                        else begin
                            InvLength := Start - NewStart;
                            InvStart := InvStart - InvLength;
                            UPaint (UInvert, inv);
                            InvStart := InvStart + Length;
                            UPaint (UInvert, inv);
                            end;
                    Start := NewStart;
                    end;
                End;

        Begin
        UPrompt (Prompt);
        with R do
            begin
            UHeight := UHeight - Place . UHeight;
            UWidth := UWidth - Place . UWidth;
            end;
        UOnCursor (R);
        with Place do
            begin
            UUpLeft := UCursor;
            end;
        UPaint (UInvert, Place);
        repeat
            with Place do
                begin
                OneDim (UCursor . UX, ULeft, UWidth,
                                        inv . ULeft, inv . UWidth);
                OneDim (UCursor . UY, UTop, UHeight,
                                        inv . UTop, inv . UHeight);
                end;
            UPollCursor;
            until UButtons <> [];
        UOffButtons;
        UOffCursor;
        UPaint (UInvert, Place);
        UErasePrompt;
        End;

Procedure UPlaceWindow (Prompt: String; Var Place: UWindow; R: URect);

        Var     rec: URect;
                w: UWindow;

        Begin
        rec := UWndOutside (Place);
        UPlaceRect (Prompt, rec, R);
        with Place do
            if (UStyle = USysTWindow) or (UStyle = USysNoTWindow)
                then begin
                    W := USysWindow (rec, UStyle = USysTWindow);
                    UInside := W . UInside;
                    end
                else UInside := URctExpand (rec, - UThick);
        End.
