Module Image;

Exports

imports Util from Util;

Const   ImgDirectory = 'Sys:Part1>Images>';
        ImgExtension = '.PkdBin';
        ImgPattern = '*.PkdBin';
        ImgMaxSize = 256;       { MUST be a multiple of 16 }
        ImgVersion = 1;     { increment at every incompatible change }

Type    ImgHeader = Record
                ImgVers: Integer;       { version of image package }
                ImgName: String [25];   { filename of image without extension }
                Rows, Cols: Integer;    { # rows, columns in image }
                ImgData: Pointer;       { pointer to data of image }
                ImgUpLeft: UCoord;      { up-left coord of display of image }
                End;

        Image = ^ImgHeader;

        ImgRow = Array [1..ImgMaxSize] of Integer;

Procedure ImgCreate (Var im: Image);
Function  ImgRead (Name: String): Image;
Procedure ImgWrite (im: Image);
Procedure ImgDispose (Var im: Image);

Procedure ImgGetRow (im: Image; rownum: Integer; Var rowarray: ImgRow);
Procedure ImgPutRow (im: Image; rownum: Integer; Var rowarray: ImgRow);

Function  ImgPixel (im: Image; r, c: Integer): Integer;
Procedure ImgSetPixel (im: Image; r, c, value: Integer);

Function  ImgRect (im: Image): URect;
Procedure ImgDisplay (im: Image);
Procedure Img2Display (im: Image);

Procedure TickPixel (im: Image; prow, pcol: Integer);
Procedure TickVar (im: Image; prow, pcol, width: Integer);
Procedure TickRow (im: Image; prow: Integer);

Exception ImgExcept (reason: String);

Private

Imports Sail_String from Sail_String;
Imports Util from Util;
Imports PMatch from PMatch;
Imports FileSystem from FileSystem;
Imports MultiRead from MultiRead;
Imports Screen from Screen;

Type    Data = Packed Array [1..ImgMaxSize, 1..ImgMaxSize] of 0..1;
        PtrData = ^Data;

Const   ImgDatWords = WordSize (Data);          { words in data part of image }
        ImgLinWords = ImgMaxSize div 16;        { words in a row }
        ImgHedWords = WordSize (Image);         { words in header }
        ImgDatBlks = (ImgDatWords + 255) div 256;  { blocks in data }
        ImgHedBlks = (ImgHedWords + 255) div 256;  { blocks in header }
        ImgBlks = ImgDatBlks + ImgHedBlks;      { blocks in image file }
        ImgBits = 4096;                         { bits in last block }

{$ifc UWordsPBlock <> 256 then }
{$message 'Words per block has changed' }
End.
{$endc}
{$ifc (ImgMaxSize mod 16) <> 0 then }
{$message 'Bad value for ImgMaxSize; must be a multiple of 16' }
End.
{$endc}
{$ifc ImgHedBlks > 1 then }
{$message 'Aaargh! The header must not be larger than one block' }
End.
{$endc}

Type    Ptrxx = Record
            Case Integer of
                0: (I: Image);
                1: (D: PDirBlk);
                2: (P: Pointer);
                4: (U: UPointer);
                5: (ID: PtrData);
                End;

Const   InitVal = #35652;

Var     Init: Integer;

        Table: Array [0 .. 15] of Integer;

Procedure InitTab;

        Var     i, val: Integer;

        Begin
        val := 15;
        for i := 1 to 15 do
            begin
            Table [i] := val;
            val := val - 2;
            end;
        Table [0] := val;
        Init := InitVal;
        End;

Procedure ImgCreate (Var im: Image);

        Var     pnt: Ptrxx;

        Begin
        UCheckVersion (UVersion, 'Recompile the Image module.');
        UCreatePSegment (im, ImgBlks * 256);
        pnt . I := im;
        pnt . U . Offset := ImgHedBlks * 256;
        with im^ do
            begin
            if Rows < 1 then Rows := 1;
            if Rows > ImgMaxSize then Rows := ImgMaxSize;
            if Cols < 1 then Cols := 1;
            if Cols > ImgMaxSize then Cols := ImgMaxSize;
            ImgData := pnt . P;
            rasterop (RXor, ImgMaxSize, ImgMaxSize,
                                        0, 0, ImgLinWords, ImgData,
                                        0, 0, ImgLinWords, ImgData);
            ImgName := '';
            ImgVers := ImgVersion;
            end;
        End;

Function ImgRead (Name: String): Image;

        Var     FileName: String;
                fid: FileID;
                im: Image;
                Blks, Bits: Integer;
                pnt: Ptrxx;
                IPat: String;

        Begin
        FileName := ImgDirectory;
        IPat := ImgPattern;
        if PattMatch (Name, IPat, True) then
            with im^ do
                Adjust (Name, Length (Name) - Length (ImgExtension));
                                                { cut off extension }
        AppendString (FileName, Name);
        AppendString (FileName, ImgExtension);
        fid := FSInternalLookUp (FileName, Blks, Bits);
        if fid = 0 then
            Raise ImgExcept ('ImgRead: file does not exist');
        if Blks <> ImgBlks then
            Raise ImgExcept ('ImgRead: wrong # of blocks in file');
        if Bits <> ImgBits then
            Raise ImgExcept ('ImgRead: wrong # of bits in last block');
        ImgCreate (im);
        pnt . I := im;
        MultiRead (fid, pnt . D, 0, Blks);
        pnt . U . Offset := ImgHedBlks * 256;
        im^ . ImgData := pnt . P;
        if im^ . ImgName <> Name then
            Raise ImgExcept ('ImgRead: filename and image name are different');
        if im^ . ImgVers <> ImgVersion then
            writeln ('*** Incompatible versions of image and software ***');
        im^ . ImgVers := ImgVersion;
        ImgRead := im;
        End;

Procedure ImgWrite (im: Image);

        Var     FileName: String;
                fid: FileID;
                pnt: Ptrxx;
                IPat: String;
                i: Integer;

        Begin
        with im^ do
            begin
            IPat := ImgPattern;
            if PattMatch (ImgName, IPat, True) then
                Adjust (ImgName, Length (ImgName) - Length (ImgExtension));
                                                { cut off extension }
            FileName := ImgDirectory;
            AppendString (FileName, ImgName);
            AppendString (FileName, ImgExtension);     { Packed Binary image }
            end;
        fid := FSEnter (FileName);
        if fid = 0 then
            Raise ImgExcept ('ImgWrite: unable to "Enter" image file');
        pnt . I := im;
        for i := 0 to ImgBlks - 1 do
            begin
            pnt . U . Offset := i * 256;
            FSBlkWrite (fid, i, pnt . D);
            end;
        FSClose (fid, ImgBlks, ImgBits);
        End;

Procedure ImgDispose (Var im: Image);

        Begin
        UDispPSegment (im);
        End;

Procedure ImgGetRow (im: Image; rownum: Integer; Var rowarray: ImgRow);

        Var     pnt: Ptrxx;
                i: Integer;

        Begin
        if not (rownum in [1 .. im^ . Rows]) then
            Raise ImgExcept ('ImgGetRow: Row number out of range');
        if Init <> InitVal then InitTab;
        pnt . P := im^ . ImgData;
        for i := 1 to im^ . Cols do
            rowarray [i + Table [i mod 16]] := pnt . ID ^ [rownum, i];
        End;

Procedure ImgPutRow (im: Image; rownum: Integer; Var rowarray: ImgRow);

        Var     pnt: Ptrxx;
                i: Integer;

        Begin
        if not (rownum in [1 .. im^ . Rows]) then
            Raise ImgExcept ('ImgPutRow: Row number out of range');
        if Init <> InitVal then InitTab;
        pnt . P := im^ . ImgData;
        for i := 1 to im^ . Cols do
            pnt . ID ^ [rownum, i + Table [i mod 16]] :=
                                                LAnd (rowarray [i], 1);
        End;

Function ImgPixel (im: Image; r, c: Integer): Integer;

        Var     pnt: Ptrxx;

        Begin
        if not (r in [1 .. im^ . Rows]) then
            Raise ImgExcept ('ImgPixel: Row number out of range');
        if not (c in [1 .. im^ . Cols]) then
            Raise ImgExcept ('ImgPixel: Column number out of range');
        if Init <> InitVal then InitTab;
        pnt . P := im^ . ImgData;
        ImgPixel := pnt . ID ^ [r, c + Table [c mod 16]];
        End;

Procedure ImgSetPixel (im: Image; r, c, value: Integer);

        Var     pnt: Ptrxx;

        Begin
        if not (r in [1 .. ImgMaxSize]) then
            Raise ImgExcept ('ImgSetPixel: Row number out of range');
        if not (c in [1 .. ImgMaxSize]) then
            Raise ImgExcept ('ImgSetPixel: Column number out of range');
        if Init <> InitVal then InitTab;
        pnt . P := im^ . ImgData;
        pnt . ID ^ [r, c + Table [c mod 16]] := LAnd (value, 1);
        End;

Function  ImgRect (im: Image): URect;

        Var     rec: URect;

        Begin
        with im^, rec do
            begin
            UUpLeft := ImgUpLeft;
            UHeight := Rows;
            UWidth := Cols;
            end;
        ImgRect := rec;
        End;

Procedure ImgDisplay (im: Image);

        Begin
        with im^, ImgUpLeft do
            rasterop (RRpl, cols, rows, UX, UY, SScreenW, SScreenP,
                                        0, 0, ImgLinWords, ImgData);
        End;

Const   FooInitVal = #35265;

        Type    LineT = Array [0 .. 2 * ImgLinWords - 1] of Integer;
                PLine = ^LineT;

Var     FooInit: Integer;
        Lin: PLine;
        cnv: array [0 .. 255] of Integer;

Procedure Img2Display (im: Image);

        Type    FooImg = Packed Array [1 .. ImgMaxSize,
                                       1 .. 2 * ImgLinWords] of 0 .. 255;
                PFoo = ^FooImg;

{$ifc WordSize (FooImg) <> WordSize (Data) then }
{$message 'in Img2Display: declaration of FooImg is wrong.' }
End.
{$endc}

        Var     pf: PFoo;
                i, j, nc: Integer;

        Procedure SetUpCnv;

                Var     i, j, out, n, mask: Integer;

                Begin
                for i := 0 to 255 do
                    begin
                    out := LAnd (i, 1);
                    mask := 6;
                    n := Shift (i, 1);
                    for j := 1 to 8 do
                        begin
                        out := LOr (out, LAnd (n, mask));
                        n := Shift (n, 1);
                        mask := Shift (mask, 2);
                        end;
                    cnv [i] := out;
                    end;
                End;

        Begin
        if FooInit <> FooInitVal then
            begin
            UNew (Lin, WordSize (LineT), 4);
            SetUpCnv;
            FooInit := FooInitVal;
            end;
        with im^, ImgUpLeft do
            begin
            pf := ReCast (ImgData, PFoo);
            nc := (Cols + 7) div 8;
            for j := 1 to Rows do
                begin
                for i := 1 to nc do
                    Lin ^ [LXOr (i - 1, 1)] := cnv [pf^ [j, i]];
                RasterOp (RRpl, Cols * 2, 1, UX, UY + (j - 1) * 2, SScreenW,
                                SScreenP, 0, 0, WordSize (LineT), Lin);
                RasterOp (RRpl, Cols * 2, 1, UX, UY + j * 2 - 1, SScreenW,
                                SScreenP, 0, 0, WordSize (LineT), Lin);
                end;
            end;
        End;

Procedure TickPixel (im: Image; prow, pcol: Integer);

        Var     pd: PtrData;

        Begin
        with im^, ImgUpLeft do
            RasterOp (RRpl, 1, 1,
                        UX + pcol - 1, UY + prow - 1, SScreenW, SScreenP,
                        pcol - 1, prow - 1, ImgLinWords, ImgData);
        End;

Procedure TickVar (im: Image; prow, pcol, width: Integer);

        Begin
        with im^, ImgUpLeft do
            RasterOp (RRpl, width, 1,
                        UX + pcol - 1, UY + prow - 1, SScreenW, SScreenP,
                        pcol - 1, prow - 1, ImgLinWords, ImgData);
        End;

Procedure TickRow (im: Image; prow: Integer);

        Begin
        with im^, ImgUpLeft do
            RasterOp (RRpl, Cols, 1,
                        UX, UY + prow - 1, SScreenW, SScreenP,
                        0, prow - 1, ImgLinWords, ImgData);
        End.
