program PERQMan;
{------------------------------------------------------------------------
Written by Brad A. Myers Sept. 9, 1981

Copyright (C) 1981 - Brad A. Myers
------------------------------------------------------------------------}

{------------------------------------------------------------------------
   Change Log:
       5 Oct 81  V2.4  Brad A. Myers   Ignore a control that would send in
                                        current direction (allow chords).
                                       Turn off tablet while AutoPlay
       5 Oct 81  V2.3  Brad A. Myers   New key control by allowing type ahead
       2 Oct 81  V2.2  Brad A. Myers   New blink for players
      23 Sep 81  V2.1  Brad A. Myers   4 players
                                       All high scores in same file w/ names
                                       No erase scores w/ credits (window 1)
      17 Sep 81  V2.0  Brad A. Myers   Animation credit between games.
                                       Skill levels
                                       Ghosts catch up with circle
       9 Sep 81  V1.0  Brad A. Myers   Started
------------------------------------------------------------------------}

{------------------------------------------------------------------------
  DESIGN:
    In the code, x and y positions that refer to maze coords are preceeded by
      M, N or T (e.g. MX);
    Screen coords either have scr in front or no prefix.
    
  REQUIREMENTS FOR MAZE:
      The maze size is fixed at 26 by 28.
      The maze is not allowed to have any dead ends.
      All boxes in the maze are defined by paths.
      The upper left corner of each box is signified by a small s.
      Use the letters r,d,l,u to signify the direction to travel around boxes.
      Corners must be signified by changing the letter.
      The tunnel start must be signified by a capital S.
      The exit box must have the opening at the top.
      The ghost number 0 must be started directly above this exit.
      The exit should be signified by capital R's.
      All ghosts except for one must start in the box.
      The amount of time spent in the box is specified by the number which
        shows where the ghost goes.
      Small dots in the maze are signified by periods '.';
      Large dots in the maze are signified by asterisks '*';
      Tunnels may be on left and right.
      All tunnels must have exits at both sides.
      There must be a row around the entire maze defining the border.
      All non-tunnel parts of the border must be 'x's
      The area in the tunnel in which the ghosts go slowly is defined as the
         area between the exit and the first non-space character.  Thus a '.'
         causes the ghosts to speed up.
------------------------------------------------------------------------}

imports FastRandom from FastRandom;
imports Screen from Screen;
imports IO_Others from IO_Others;
imports IO_Unit from IO_Unit;
imports IOErrors from IOErrors;
imports FileSystem from FileSystem;
imports Memory from Memory;
imports PERQ_String from PERQ_String;

{$R-}

Const TitleString = '                       PERQMan version V2.4.     By Brad A. Myers';

      yBase = 170;                  {y Start of maze}
      xBase = 45;                   {x Start of maze}
      square = 26;                  {size of each square of the maze}
      ySize = 28;                   {number of squares in y}
      xSize = 26;                   {number of squares in x}
      gOffset = (square div 2) - 3; {offset of ghosts and circle when RasterOp}
      FruitMX = 13;                 {x maze position of fruit}
      FruitMY = 16;                 {y maze position of fruit}
      CatchUp = 3;                  {amount circle catches up with blueGhost}
      DEL = Chr(127);
      FF = Chr(12);
      HELP = Chr(7);
      CR = chr(13);
      MaxNumPlayers = 4;            {number of players allowed}

      DebugConst = True;
      ScoreFileName = 'PERQMan.HighScores';
      
Type CornDir = (ur, rd, dl, lu);
     DotSize = (noDot, bigDot, smallDot);
     Direction = (right, up, left, down);
     GhostNames = (Pokey, Bashful, Speedy, Shadow);
     
     ScoreRec = record
                   score: integer;
                   who: String[21];
                end;
    ScoreArray = Array[0..10] of ScoreRec;

Var Maze: Packed Array[0..xSize+1, 0..ySize+1] of char;
    Corner: Array [CornDir] of CurPatPtr;
    circles: Array[direction] of Array[0..3] of CurPatPtr;
    circleExplode: Array[0..8] of CurPatPtr;
    ghostPics: Array[ghostNames] of Array[0..1] of CurPatPtr;
    blueGhost: array[0..1] of CurPatPtr;
    bluePics: Array[0..1] of CurPatPtr;
    eyes: Array[direction] of CurPatPtr;
    fruitPics: Array[1..8] of CurPatPtr;
    arrow: CurPatPtr;
    blueBlink, blueIncBlink, boxX, boxY, fruitTime,
       fruitX, fruitY, numPlayers, skillLevel, cirCatchUp, pauseTime, 
       highPlayer, autoScore: integer;
    randomT: pRanTable;
    startPos: Array[ghostNames] of record
                                     x,y,time: integer;
                                   end;
    ghosts: Array[ghostNames] of record
                                    dir: Direction;
                                    delay: integer;
                                    scrx, scry: Integer;
                                    mx, my: integer;
                                    blueTime: Integer; {if zero then not blue}
                                    eyesOnly: Boolean; {going to box}
                                    enteringBox: boolean; {going down into box}
                                    inBox: boolean;    {inside or leaving}
                                    boxTime: Integer;  {countdown until leave}
                                    count: Integer;    {incremented every tic}
                                    pic: Integer;      {0 or 1}
                                    inTunnel: boolean;
                                    end; 
                                    
                                    
                                   
    picBigDot, picSmallDot: CurPatPtr;
    curBlueTime, score, numCir, fruitMaze, numDots,
          fruitChances: array[1..MaxNumPlayers] of integer;
    fruitsGotten: Array[1..MaxNumPlayers] of Array[1..8] of integer;
    highScore, player, cirMx, cirMy, ghostsSinceDot: Integer;
    s: String;
    walls: Array[-2..xSize+3, 0..ySize+1] of boolean;
    dots: Array[1..MaxNumPlayers] of Array[-1..xSize+2, 0..ySize+1] of DotSize;
    ans: string[1];
    fruitOn, autoPlay: boolean;
    tunnel: Array[-1..xSize+2, 0..ySize+1] of boolean; 

    allHighScores: ScoreArray;

    saveChar: Char;
    
{$ifc debugConst then}
   Const debug = false;
         debug1 = false;
         invincible = false;
{$elsec}
var debug, debug1, invincible: boolean;
{$endc}

Label 99;

Exception DelHit;
{--------------------------------------------------------------------------
 Abstract: Raised when Delete key Hit so game can be restarted.
--------------------------------------------------------------------------}

Procedure Beep;
{--------------------------------------------------------------------------
 Abstract: Goes beep if not autoPlay;
--------------------------------------------------------------------------}
  begin
  If not autoPlay then IOBeep;
  end;

Function ReverseDir(dir: Direction): Direction;
{--------------------------------------------------------------------------
 Abstract: returns the reverse of the parameter (left goes to right, etc.);
--------------------------------------------------------------------------}
   begin
   ReverseDir := RECAST((ord(dir)+2) mod 4, Direction);
   end;

Procedure Pause(len: integer);
{--------------------------------------------------------------------------
 Abstract: Wait for len tics;
--------------------------------------------------------------------------}
   var i,j: integer;
   begin
   for i := 1 to len do;
   end;

Procedure KeyInput;
{--------------------------------------------------------------------------
 Abstract: Wait for a while and check keyboard for commands
--------------------------------------------------------------------------}
   var c: char;
       x, y, x4, y3: integer;
   begin
   if IOCRead(TransKey, c) = IOEIOC then
   begin
      if c = DEL then Raise DelHit
      else if c = 'j' then saveChar := 'a'
      else if c = 'l' then saveChar := 'f' 
      else if c = 'i' then saveChar := 's'
      else if c = ',' then saveChar := 'd'
{$ifc not debugConst then}
         else if c = 'D' then debug1 := not debug1
         else if c = 'I' then invincible := not invincible;
{$endc}
      end;

      if not autoPlay then
      begin
{       IOReadTablet(x,y);
        x4:=x*4; y3:=y*3;
        if x4>y3 then
            if x4>3072-y3
            then saveChar:='f'
            else saveChar:='s'
          else
            if x4>3072-y3
            then saveChar:='d'
            else saveChar:='a';
} 
        if TabGreen  then saveChar := 'f';  { right }
        if TabBlue   then saveChar := 'd';  { down }
        if TabWhite  then saveChar := 'a';  { left }
        if TabYellow then saveChar := 's';  { up }
      end;
      
   end;

Procedure LongPause;
{--------------------------------------------------------------------------
 Abstract: Wait for a while and check keyboard for commands
--------------------------------------------------------------------------}
   var i: integer;
   begin
   for i := 1 to 5000 do
         KeyInput;
   end;

Procedure TransPt(mx,my: integer; var scrX, scrY: integer);
{--------------------------------------------------------------------------
 Abstract: Convert from maze coords to screen coords.
--------------------------------------------------------------------------}
   begin
   scrX := (mx-1)*Square+xBase;
   scrY := (my-1)*Square+yBase;
   end;

Procedure UnTransPt(scrX, scrY: integer; var mx,my: integer);
{--------------------------------------------------------------------------
 Abstract: Convert from screen coords to maze coords.
--------------------------------------------------------------------------}
   begin
   mx := ((scrX-xBase) div Square) + 1;
   my := ((scrY-yBase) div Square) + 1;
   end;

Function GCentered(scrX, scrY: integer): boolean;
{--------------------------------------------------------------------------
 Abstract: Returns true if the screen position is in the center of a square.
           If true, then scrX and scrY would be valid values returned by
           TransPt.
--------------------------------------------------------------------------}
   begin
   GCentered := ((scrX-xBase) mod Square=0) and ((scrY-yBase) mod Square=0)
   end;

Procedure DrawCorner(mx,my: integer; dir: CornDir);
{--------------------------------------------------------------------------
 Abstract: Draw a corner at MAZE position mx,my turning the the direction dir.
--------------------------------------------------------------------------}
   var x,y: integer;
   begin
   TransPt(mx,my,x,y);
   RasterOp(RRpl, Square, Square, x, y, SScreenW, SScreenP,
                                  0, 0, 4, Corner[dir]);
   end;

Procedure DDrawLine(mx,my: Integer; dir: Char);
{--------------------------------------------------------------------------
 Abstract: Draw a maze line from mx,my in the direction dir
 Parameters: dir should be d,u,l,r, (or R for a thin line).
--------------------------------------------------------------------------}
   var x,y: integer;
   begin
   TransPt(mx,my,x,y);
   case dir of
      'd','u' : RasterOp(RXNor, 2, Square, x+Square div 2, y,
                                     SScreenW, SScreenP,
                                       x+Square div 2, y, SScreenW, SScreenP);
      'l','r' : RasterOp(RXNor, Square, 2, x, y+Square div 2,
                                     SScreenW, SScreenP,
                                       x, y+Square div 2, SScreenW, SScreenP);
      'R'     : RasterOp(RXNor, Square, 1, x, y+Square div 2,
                                     SScreenW, SScreenP,
                                       x, y+Square div 2, SScreenW, SScreenP);
      otherwise: WriteLn('** DDrawLine value error');
      end;
   end;

Procedure DrawBox(mx,my: Integer);
{--------------------------------------------------------------------------
 Abstract: Draws a box starting at maze position mx, my
 Design: mx, my should be an s or S position in maze.
         Travels around path until reach x or s.
--------------------------------------------------------------------------}
   var last: Char;
   begin
   Pause(pauseTime*10);
   Walls[mx,my] := true;
   if maze[mx,my] = 's' then DrawCorner(mx,my,ur)
   else DDrawLine(mx,my,'r');
   mx := mx+1;
   last := 'r';
   repeat
     Pause(pauseTime);
     Walls[mx,my] := true;
     case maze[mx,my] of
           's','S' : exit(DrawBox);
           'd' : begin
                 if last = 'r' then DrawCorner(mx,my,rd)
                 else if last = 'l' then DrawCorner(mx,my,ur)
                 else DDrawLine(mx,my,'d');
                 last := 'd';
                 my := my+1;
                 end;
           'l' : begin
                 if last = 'd' then DrawCorner(mx,my,dl)
                 else if last = 'u' then DrawCorner(mx,my,rd)
                 else DDrawLine(mx,my,'l');
                 last := 'l';
                 mx := mx-1;
                 end;
           'r','R' : begin
                 if last = 'u' then DrawCorner(mx,my,ur)
                 else if last = 'd' then DrawCorner(mx,my,lu)
                 else DDrawLine(mx,my,Maze[mx,my]);
                 last := 'r';
                 mx := mx+1;
                 end;
           'u' : begin
                 if Last = 'l' then DrawCorner(mx,my,lu)
                 else if last = 'r' then DrawCorner(mx,my,dl)
                 else DDrawLine(mx,my,'u');
                 last := 'u';
                 my := my-1;
                 end;
           'x' : begin
                 DDrawLine(mx,my,last);
                 exit(DrawBox);
                 end;
           otherwise: begin
                      WriteLn('** Maze value error: "',maze[mx,my],
                                    '" at ',mx:1,',',my:1);
                      exit(DrawBox);
                      end;
           end;
   until false;
   end;

Procedure FixExits;
{--------------------------------------------------------------------------
 Abstract: Look for tunnels on the borders.  For each, show the area as black
           on the screen and set the Walls and tunnel global variables to
           reflect the presence of the tunnel.
--------------------------------------------------------------------------}
   var x,y,t: integer;
   begin
   Pause(pauseTime*10);
   For y := 1 to ySize do
      if maze[0,y] = ' ' then
         begin
         Walls[-1,y] := false;
         Walls[-2,y] := false;
         x := -1;
         repeat
           tunnel[x,y] := true;
           x := x+1;
         until maze[x,y] <> ' ';
         TransPt(0,y,x,t);
         RasterOp(RNot, xBase-(Square div 2)-5, Square*2+6,
                          3, t-2-Square div 2, SScreenW, SScreenP,
                          3, t-2-Square div 2, SScreenW, SScreenP);
         end;

   Pause(pauseTime*10);
   For y := 1 to ySize do
      if maze[xSize+1,y] = ' ' then
         begin
         Walls[xSize+2,y] := false;
         Walls[xSize+3,y] := false;
         x := xSize+2;
         repeat
           tunnel[x,y] := true;
           x := x-1;
         until maze[x,y] <> ' ';
         TransPt(0,y,x,t);
         RasterOp(RNot, xBase-(Square div 2)-5, Square*2+6,
                xBase-(Square div 2) + Square*(xSize+1)+4, t-2-Square div 2,
                               SScreenW, SScreenP,
                xBase-(Square div 2) + Square*(xSize+1)+4, t-2-Square div 2,
                               SScreenW, SScreenP);
         end;
  end;

Procedure DrawDot(mx,my: integer; size: DotSize);
{--------------------------------------------------------------------------
 Abstract: Xors a dot at the maze position mx,my.  Size tells whether a big or
            small dot
--------------------------------------------------------------------------}
   begin
   TransPt(mx,my,mx,my);
   if size = bigDot then RasterOp(RXor, 24, 24, mx,my,SScreenW, SScreenP,
                                                0,0, 4, picBigDot)
   else if size = smallDot then RasterOp(RXor,7,7,mx+9,my+9,SScreenW,SScreenP,
                                                9,9, 4, picSmallDot);
   end;

Procedure ReadAllHighScores;
{--------------------------------------------------------------------------
 Abstract: Reads all high scores and names into the global table allHighScores.
           If file not found, then sets all high scores to zero.
--------------------------------------------------------------------------}
   var fid: FileID;
       blks, bits, level: integer;
       b: File Of ScoreArray;
   begin
   fid := FSLookUp(ScoreFileName, blks,bits);
   if fid = 0 then 
       for level := 0 to 10 do
          begin
          allHighScores[level].score := 0;
          allHighScores[level].who := '';
          end
   else begin
        Reset(b,ScoreFileName);
        allHighScores := b^;
        Close(b);
        end;
   end;

Procedure WriteHighScore;
{--------------------------------------------------------------------------
 Abstract: if highScore is better than old high score for this skill level then
           asks for player's name and enters name and score into table and
           writes file.
--------------------------------------------------------------------------}
     var b: File Of ScoreArray;
     begin
     if (highScore >= allHighScores[skillLevel].score) and
        (highPlayer <> -1) then
        begin
        IOBeep;
        ChangeWindow(1);
        RefreshWindow(1);
        Write(FF);
        SSetCursor(348, 340);
        WriteLn('Player ',highPlayer:1);
        SSetCursor(253, 370);
        WriteLn('has beaten the high score for');
        SSetCursor(321, 400);
        WriteLn('skill level ',skillLevel:1,'.');
        if allHighScores[skillLevel].score <> 0 then
            begin
            SSetCursor(276, 430);
            WriteLn('The old record was ',allHighScores[skillLevel].score:1,
                      '0');
            SSetCursor(312, 460);
            WriteLn('held by: ',allHighScores[skillLevel].who,'.');
            SSetCursor(200, 520);
            end
        else SSetCursor(200, 460);
        Write('Type player ',highPlayer:1,'''s name or initials: ');
        IOBeep;
        ReadLn(allHighScores[skillLevel].who);
        allHighScores[skillLevel].score := highScore;
        ReWrite(b, ScoreFileName);
        b^ := allHighScores;
        put(b);
        Close(b);
        end;
   end;

Procedure Initialize;
   var c: cornDir;
       x,y: integer;
       d: Direction;
       g: GhostNames;
       seg, indx: integer;
       mf: Text;
       fid: FileID;
       blks, bits: integer;
       doub: double;
       m: Char;       
   begin

   ChangeTitle(TitleString);
   
   new(randomT);
   RandomTable(randomT);
   InitRandom(randomT);
   IOGetTime(doub);
   for x := 1 to doub[0] mod 10 do  {Randomize start}
      y := Random;
   Write(' Random');

   fid := FSLookUp('PERQMan.animate', blks, bits);
   if fid = 0 then begin
                   WriteLn('** PERQMan.Animate not found');
                   exit(PERQMan);
                   end;
   
   if blks <> 55 then begin
                      WriteLn('** PERQMan.Animate has wrong size');
                      exit(PERQMan);
                      end;
   
   CreateSegment(seg, blks, 1, blks);
   Write(' Memory');
   
   for indx := 0 to blks-1 do
      FSBlkRead(fid, indx, MakePtr(seg, indx*256, pDirBlk));
   Write(' File');
   
   indx := 0;
   
   for c := ur to lu do
     begin
     corner[c] := MakePtr(seg, indx*256, CurPatPtr);
     indx := indx+1;
     end;
   
   picBigDot := MakePtr(seg, indx*256, CurPatPtr);
   indx := indx+1;
   picSmallDot := MakePtr(seg, indx*256, CurPatPtr);
   indx := indx+1;

   for d := right to down do
     for x := 0 to 3 do
       begin
       circles[d][x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;

   for g := Pokey to Shadow do
     for x := 0 to 1 do
       begin
       ghostPics[g][x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;

  for x := 0 to 1 do
       begin
       BlueGhost[x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;
  for x := 0 to 1 do
       begin
       bluePics[x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;

  for d := right to down do
       begin
       eyes[d] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;

  for x := 0 to 8 do
       begin
       circleExplode[x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;
   
  for x := 1 to 8 do
       begin
       fruitPics[x] := MakePtr(seg, indx*256, CurPatPtr);
       indx := indx+1;
       end;
   
   reset(mf, 'PERQMan.Maze');
   for y := 0 to ySize+1 do
     begin
     for x := 0 to xSize+1 do
       begin
       Read(mf, m);
       maze[x,y] := m;
       end;
     readln(mf);
     end;
   Write(' Maze');
   Close(mf);
   
   TransPt(FruitMx, FruitMy, FruitX, FruitY);
   
   ReadAllHighScores;
   highScore := allHighScores[skillLevel].score;
   Write(' HighScore');
  
  g := Pokey;
  for y := 1 to ySize do
        for x := 1 to xSize do
          if maze[x,y] in ['0'..'9'] then
                  begin
                  startPos[g].x := x;
                  startPos[g].y := y;
                  startPos[g].time := ord(maze[x,y])-ord('0');
                  if g < Shadow then g := succ(g);
                  if maze[x,y] = '0' then
                     begin
                     BoxX := x;
                     BoxY := y;
                     end;
                  end;

  if g <> Shadow then 
      begin
      WriteLn('****ONLY ',ord(g):1,' ghosts found!****');
      exit(PERQMan);
      end;
  
  Write(' Starting');

  for x := -1 to xSize+2 do
    for y := 0 to ySize+1 do
       tunnel[x,y] := false; 
  
  Write(FF);
  CreateWindow(1, 0, yBase-(Square div 2)-3, 768,
                  1024-(yBase-(Square div 2)-3), '');
  end;
 
Procedure SetDots(player: Integer);
  var x,y : integer;
  begin
  for y := 1 to ySize do
        begin
        Dots[player][-1, y] := noDot;
        Dots[player][0, y] := noDot;
        Dots[player][xSize+1, y] := noDot;
        Dots[player][xSize+2, y] := noDot;
        for x := 1 to xSize do
          if maze[x,y] = '.' then begin
                                  Dots[player][x,y] := smallDot;
                                  numDots[player] := numDots[player]+1;
                                  end
          else if maze[x,y] = '*' then begin
                                       Dots[player][x,y] := bigDot;
                                       numDots[player] := numDots[player]+1;
                                       end
          else Dots[player][x,y] := noDot;
       end;
  end;

Procedure InitGame;
  var i,j : integer;
  begin
  
  if not autoPlay then IOSetModeTablet(relTablet)
  else autoScore := 0;

  pauseTime := -skillLevel*20 + 200;
  cirCatchUp := -skillLevel*4 + 46;
  highPlayer := -1;
  for j := 1 to MaxNumPlayers do
     begin
     numDots[j] := 0;
     numCir[j] := 3;
     fruitChances[j] := 0;
     SetDots(j);
     curBlueTime[j] := 1+(-skillLevel*60 + 900);
     if not autoPlay then
       begin
       score[j] := 0;
       fruitMaze[j] := 1;
       for i := 1 to 8 do
           fruitsGotten[j][i] := 0;
       end;
     end;
  end;

Procedure UpDateScore(amt: Integer);
  var i, temp,x,y: integer;
  begin
  if autoPlay then 
     begin
     SSetCursor(489,80);
     autoScore := autoScore + amt;
     Write(autoScore:1,'0');
     exit(UpDateScore);
     end;
  temp := score[player] + amt;
  if temp >= 1000 then
     if score[player] < 1000 then
         begin
         Beep;
         for i := 1 to 6 do
           begin
           IOSetFunction(Recast((i mod 2)+6, CursFunction));
           Pause(10000);
           end;
         numCir[player] := numCir[player]+1;
         RasterOp(RXor, 50, 50, xBase+(numCir[player]-1)*60, 1024-50,
                     SScreenW, SScreenP, 0, 0, 4, circles[right][2]);
         end;
  score[player] := temp;
  if player mod 2 = 0 then x := 627
  else x := 127;
  if player < 3 then y := 55
  else y := 95;
  SSetCursor(x,y);
  Write(Score[player]:1,'0');
  if score[player] > highScore then
     begin
     highPlayer := player;
     highScore := score[player];
     SSetCursor(348,50);
     Write(highScore:1,'0');
     end;
  end;

Procedure DrawBotFruit;
   var i,j,cnt, xStart: integer;
   begin
   cnt := 0;
   for i := 8 downto 1 do
     cnt := cnt+FruitsGotten[player][i];
   if cnt > 8 then cnt := 8;
   xStart := 768-5-cnt*60;
   for i := 8 downto 1 do
      for j := 1 to FruitsGotten[player][i] do
         begin
         if xStart > 768-60 then exit(DrawBotFruit);
         RasterOp(RRpl, 60, 45, xStart, 1024-50, SScreenW, SScreenP,
                                0, 0, 4, FruitPics[i]);
         xStart := xStart+60;
         end;
   end;

Procedure ShowPlayerScore(player: integer);
  var x,y: integer;
  begin
  SChrFunc(RXor);
  if player mod 2 = 0 then x := 600
  else x := 100;
  if player < 3 then y := 40
  else y := 80;
  SSetCursor(x,y);
  Write('  Player ',player:1,' ');
  SSetCursor(x+27,y+15);
  if score[player] = 0 then Write('0')
  else Write(score[player]:1,'0');
  SChrFunc(RRpl);
  end;
  
Procedure BlinkPause;
{--------------------------------------------------------------------------
 Abstract: Wait for a while and check keyboard for commands while blinking
           current players score
--------------------------------------------------------------------------}
   var i,j: integer;
       on: boolean;

   Handler DelHit;
     begin
     if not on then ShowPlayerScore(player);
     Raise DelHit;
     end;
   
   begin
   on := true;
   if autoPlay then LongPause
   else for i := 1 to 16 do
           begin
           ShowPlayerScore(player);
           on := not on;
           for j := 1 to 400 do
              KeyInput;
           end;
   end;

Procedure DrawMaze;
{--------------------------------------------------------------------------
 Abstract: Draw the Maze, the dots, the scores, etc on the screen.
--------------------------------------------------------------------------}
   var x,y: integer;
       p: CurPatPtr;
   begin

   ChangeWindow(0);
   RefreshWindow(0);
   ChangeTitle(TitleString);
   Write(FF);

   for y := 0 to ySize+1 do
     begin
     Walls[-1,y] := true;
     Walls[-2,y] := true;
     Walls[xSize+2,y] := true;
     Walls[xSize+3,y] := true;
     for x := 0 to xSize+1 do
        Walls[x,y] := false;
     end;

   for y := 0 to ySize+1 do
     for x := 0 to xSize+1 do
       if (maze[x,y] = 's') or (maze[x,y] = 'S') then 
           begin
           DrawBox(x,y);
           KeyInput;  {check for abort}
           end
       else if maze[x,y] = 'x' then Walls[x,y] := true; {borders}

   Pause(pauseTime*10);

   RasterOp(RNot, Square*(xSize+1)+6, Square*(ySize+1)+6,
         xBase-(Square div 2)-2, yBase-(Square div 2)-2, SScreenW, SScreenP,
         xBase-(Square div 2)-2, yBase-(Square div 2)-2, SScreenW, SScreenP);

   FixExits;

   for y := 1 to ySize do
     for x := 1 to xSize do
       if Dots[player][x,y] = smallDot then begin
                                            Pause(pauseTime);
                                            DrawDot(x,y, smallDot);
                                            end
       else if Dots[player][x,y] = bigDot then begin
                                               Pause(pauseTime);
                                               DrawDot(x,y,bigDot);
                                               end
       else KeyInput; {check for abort}
  
  for x := 1 to 4 do
     begin
     ShowPlayerScore(x);
     end;

  SSetCursor(321,35);
  Write('High Score (',skillLevel:1,')');
  SSetCursor(348,50);
  if highScore = 0 then Write('0')
  else Write(highScore:1,'0');

  SSetCursor(325,80);
  Write('   PERQMan   ');
  SSetCursor(329,100);
  Write('     by     ');
  SSetCursor(325,120);
  Write('Brad A. Myers');
  
  SSetCursor(5,100);
  WriteLn('Type DEL');
  WriteLn('for Restart');
  Write  ('and Help.');
  
  for x := 1 to numCir[player] do
     begin
     Pause(pauseTime*10);
     RasterOp(RXor, 50, 50, xBase+(x-1)*60, 1024-50, SScreenW, SScreenP,
                            0, 0, 4, circles[right][2]);
     end;
  
  DrawBotFruit;

  p := FruitPics[FruitMaze[player]];  
  RasterOp(RRpl, 45, 45, 715, 95, SScreenW, SScreenP, 0, 0, 4, p);
  RasterOp(RNot, 55, 55, 710, 90, SScreenW, SScreenP,
                         710, 90, SScreenW, SScreenP);

  if autoPlay then begin
                   SChrFunc(RXor);
                   SSetCursor(339,yBase+Square*16);
                   Write('GAME OVER!');
                   SChrFunc(RRpl);
                   SSetCursor(5, 950);
                   Write('** Type DEL to begin **');
                   SSetCursor(480,65);
                   Write('Auto Score');
                   SSetCursor(489,80);
                   Write('0');
                   end;
  end;

Procedure RemoveCircle;
   begin
   RasterOp(RXor, 50, 50, xBase+(numCir[player]-1)*60, 1024-50,
                  SScreenW, SScreenP, 0, 0, 4, circles[right][2]);
   end;

Procedure DrawGhost(g: GhostNames);
   var p: CurPatPtr;
       inc, wInc: integer;
   begin
   with ghosts[g] do
      begin
      inc := scrX-gOffset;
      wInc := 0;
      if (inc < 0) then if inc <= -45 then exit(DrawGhost) {fully off}
                        else wInc := inc
      else if (inc > 768-45) then if (inc > 767) then exit(DrawGhost)
                                  else begin
                                       wInc := 768-45-inc;  {will be negative}
                                       inc := 0;
                                       end
      else inc := 0;
      
      if eyesOnly then
           begin
           p := eyes[dir];
           RasterOp(RXor, 45+wInc, 21, scrX-gOffset-inc,
                                       scrY-gOffset, SScreenW,SScreenP,
                                       -inc, 0, 4, p);
        {fake rasterOp so speed same}
           p := ghostPics[g][pic];
           RasterOp(RRpl, 45+wInc, 45, 0, 0, 4, p, 0, 0, 4, p);
           end
      else if blueTime > 0 then
           begin
           if (blueTime < blueBlink) and
              (blueTime mod blueIncBlink > blueIncBlink div 2) then
                  p := bluePics[pic]
           else p := blueGhost[pic];
           RasterOp(RXor, 45+wInc, 45, scrX-gOffset-inc,
                                       scrY-gOffset, SScreenW,SScreenP,
                                       -inc, 0, 4, p);
        {fake rasterOp so speed same}
           p := eyes[dir];
           RasterOp(RRpl, 45+wInc, 21, 0, 0, 4, p, 0, 0, 4, p);
           end
      else begin
           p := ghostPics[g][pic];
           RasterOp(RXor, 45+wInc, 45, scrX-gOffset-inc,
                                       scrY-gOffset, SScreenW,SScreenP,
                                       -inc, 0, 4, p);
           p := eyes[dir];
           RasterOp(RXor, 45+wInc, 13, scrX-gOffset-inc,
                                       scrY-gOffset+8, SScreenW,SScreenP,
                                       -inc, 8, 4, p);
           end;
      end; {with}
   end;       

Procedure ChangeGhosts;
  var p: GhostNames;
  begin
  Beep;
  ghostsSinceDot := 0;
  for p := Pokey to Shadow do
     if not ghosts[p].eyesOnly then
       begin
       DrawGhost(p);
       ghosts[p].blueTime := curBlueTime[player];
       if (ghosts[p].boxTime = 0) and (not ghosts[p].inBox) then
             ghosts[p].dir := ReverseDir(ghosts[p].dir);
       DrawGhost(p); {will be blue now}
       end;
  end;

Function DoInc(dir: Direction; posx, posy, mx, my: integer;
              var x,y,nx,ny: integer): boolean;
     var tx, ty: integer;
     begin
     x := posx;
     y := posy;
     tx := mx;
     ty := my;
      
     case dir of
             up: begin
                 y := posy-2;
                 ty := my-1;
                 end;
             down: begin
                   y := posy+2;
                   ty := my+1;
                   end;
             left: begin
                   x := posx-2;
                   tx := mx-1;
                   end;
             right: begin
                    x := posx+2;
                    tx := mx+1;
                    end;
             end;
      UntransPt(x,y, nx, ny);
      if tx = -2 then tx := xSize+2
      else if tx = xSize+3 then tx := -1;
      DoInc := true;
      if debug then Write(' dir=',ord(dir):1,' in x,y,mx,my: ',
         posx,posy, mx, my,' * tx, ty=',tx,ty,' * out x,y,nx,ny: ',x,y,nx,ny);
      if (nx = -2) and (dir = left) then begin
                                         nx := xSize+2;
                                         TransPt(nx,ny,x,y);
                                         end
      else if (nx = xSize+3) and (dir = right) then
                             begin
                             nx := -1;
                             TransPt(nx,ny,x,y);
                             end
      else if not (walls[nx,ny] or (GCentered(posx, posy) and walls[tx,ty]))
               then
      else begin
           if debug then
              if walls[nx,ny] then Write(' Wall')
              else if GCentered(posX, posy) then 
                begin
                Write(' GCentered');
                if walls[tx,ty] then Write(' Twall ');
                end;
           DoInc := false;
           end;
      end;

Function CheckInc(dir: Direction; mx, my: integer): boolean;
   begin
   case dir of
      up    : CheckInc := not walls[mx, my-1];
      right : CheckInc := not walls[mx+1, my];
      down  : CheckInc := not walls[mx, my+1];
      left  : CheckInc := not walls[mx-1, my];
      end;
   end;

Procedure HeadTo(destX, destY: integer; scrx, scry, mx, my: integer;
                 var dir: Direction; var x,y,nx,ny: integer);
(**** only called when GCentered****)
  var dirAr: Array[1..4] of Direction;
      xInc, yInc, s, i: Integer;
      td, rev: Direction;
  label 1;
  begin
  rev := ReverseDir(dir);
  xInc := mx-destX;
  yInc := my-destY;
  if Abs(xInc) > Abs(yInc) then s := 2
  else s := 1;
  if xInc < 0 then begin
                   dirAr[3-s] := right;
                   dirAr[s+2] := left;
                   end
  else begin
       dirAr[3-s] := left;
       dirAr[s+2] := right;
       end;
  if yInc < 0 then begin
                   dirAr[s] := down;
                   dirAr[5-s] := up;
                   end
  else begin
       dirAr[s] := up;
       dirAr[5-s] := down;
       end;

  for i := 1 to 4 do  {adjust so reverse is last choice}
     if dirAr[i]=rev then
         begin
         for s := i to 3 do
            dirAr[s] := dirAr[s+1];
         dirAr[4] := rev;
         goto 1;
         end;
   
1: for s := 1 to 4 do
     begin
     if CheckInc(dirAr[s], mx, my) then
        begin
        dir := dirAr[s];
        if not DoInc(dir, scrx, scry, mx, my, x, y, nx, ny) then
           Write('**ERROR HEAD**');
        exit(HeadTo);
        end;
     end;
  Write('********* NOWHERE TO HEAD *****');
  end;

Function DoRandomDir(dir: Direction; scrx, scry, mx, my: integer;
                     var x,y,nx,ny: integer): Direction;
   var test, i: integer;
       newDir, rev: Direction;
       dum: boolean;
   begin
   test := RandomRange(0,2);
   rev := ReverseDir(dir);
   if (test = 1) or (not CheckInc(dir, mx, my)) then
        begin
        newDir := RECAST(RandomRange(0,3), Direction);
        for i := 0 to 3 do
           begin
           if (newDir <> rev) then
              if CheckInc(newDir, mx, my) then
                begin
                DoRandomDir := newDir;
                if not DoInc(newDir, scrx, scry, mx, my, x, y, nx, ny)
                      then write('**ERROR 1 RANDOM**');
                exit(DoRandomDir);
                end;
           newDir := RECAST((ord(newDir)+1) mod 4, Direction);
           end;
        Write('** Stuck **');
(*DB*)  if debug then begin
                      SSetCursor(5,80);
                      Write('newdir=',ord(newDir):1);
                      readln;
                      exit(Perqman);
                      end;
        end
    else begin
         if not DoInc(dir, scrx, scry, mx, my, x, y, nx, ny)
                 then write('**ERROR 2 RANDOM**');
         DoRandomDir := dir;
         end;
    end;

Procedure UpdateGhosts;  {move each ghost one bit in
                                      appropriate direction;
                                      change direction if appropriate }
  var g: GhostNames;
      x,y,nx,ny: Integer;
      newDir: Direction;
      ok, beMean: boolean;
(*DB*) cnt: integer;
  label 1;
  begin
  for g := Pokey to Shadow do
     with ghosts[g] do
(*     if (i mod ghosts[g].delay <> 0) then
*)
       begin
       count := count+1;
       if inBox or inTunnel then 
          begin
          if (count mod 2 = 0) then goto 1;  {slow in box}
          end;
       if blueTime > 0 then 
           begin
           if count mod CatchUp = 0 then goto 1;  {go slower if blue}
           DrawGhost(g); {erase old before change blueTime}
           blueTime := blueTime - 1;
           end
       else DrawGhost(g); {erase old}
       if count mod 7 = 0 then pic := (pic + 1) mod 2;
       if GCentered(scrx, scry) then
         begin
         inTunnel := Tunnel[mx,my];
         if (blueTime = 0) then
            if (skillLevel < 5) then
                   beMean := RandomRange(0,10-skillLevel-ord(g)) = 0
            else beMean := RandomRange(0,skillLevel-5+ord(g)) <> 0
         else beMean := false;
         if debug1 then
              begin
              SSetCursor(5,50);
              Write('** ghost ',ord(g):1,' is at mx,my: ',mx,my, ' tunnel=',inTunnel);
              readln;
              end;
         if inBox then if (mx=boxX) and (my=boxY) then inBox := false;
         if eyesOnly then
            begin
            if (not enteringBox) and (mx=boxX) and (my=boxY) then
               begin
               dir := down;
               enteringBox := true;
               ok := DoInc(dir,scrx, scry, mx, my, x, y, nx, ny); 
               end
            else if enteringBox then
                    if (my > boxY + 2) and
                       (not DoInc(dir,scrx,scry,mx,my,x, y, nx, ny)) then
                         begin
                         dir := up;
                         enteringBox := false;
                         inBox := true;
                         eyesOnly := false;
                         ok := DoInc(dir,scrx,scry,mx,my,x,y,nx, ny);
                         end
                    else {DoInc has set the values}
            else HeadTo(boxX, boxY, scrX,scrY,mx,my,dir,x,y,nx,ny)
            end {eyes only}
         else if boxTime <> 0 then {inBox should be true also}
               begin
               boxTime := boxTime - 1;
               if boxTime < 0 then {heading to exit}
                   begin
                   if (mx=boxX) then  {found exit}
                         begin
                         boxTime := 0;
                         dir := up;
                         ok := DoInc(dir,scrx,scry,mx,my,x,y, nx, ny);
                         end
                   else HeadTo(boxX, boxY, scrX,scrY,mx,my,dir,x,y,nx,ny);
                   end
               else if boxTime = 0 then {start heading to exit}
                    begin
                    boxTime := -1;
                    HeadTo(boxX, boxY, scrX,scrY,mx,my,dir,x,y,nx,ny);
                    end
               else if not DoInc(dir,scrx,scry,mx,my,x,y, nx, ny) then
                    begin
                    dir := ReverseDir(dir); {bounce up a down a while}
                    ok := DoInc(dir,scrx,scry,mx,my,x,y, nx, ny);
                    end;
               end {boxTime <> 0}
         else if inBox then {must be leaving the box; just keep going}
                   ok := DoInc(dir, scrx, scry, mx, my, x, y, nx, ny)
         else if beMean then {chase the circle}
                    HeadTo(cirMX, cirMY, scrX,scrY,mx,my,dir,x,y,nx,ny)
         else dir := DoRandomDir(dir,scrx, scry, mx, my, x, y, nx, ny);
         end {GCentered}
       else begin {not GCentered}
            ok := DoInc(dir,scrx, scry, mx, my, x, y, nx, ny);
            end;
       scrx := x;
       scry := y;
       mx := nx;
       my := ny;
       DrawGhost(g); {draw new}
    1: end; {with and loop}
  end;

Procedure NewGhosts(drawThem: boolean);
   var p: GhostNames;
   begin
   for p := Pokey to Shadow do
      begin
      ghosts[p].dir := up;
      ghosts[p].mx := StartPos[p].x;
      ghosts[p].my := StartPos[p].y;
      TransPt(ghosts[p].mx, ghosts[p].my, ghosts[p].scrX, ghosts[p].scrY);
      ghosts[p].blueTime := 0;
      ghosts[p].eyesOnly := false;
      ghosts[p].boxTime := ((-2*skillLevel+25) div 5) * startPos[p].time;
      if ghosts[p].boxTime = 0 then ghosts[p].inBox := false
      else ghosts[p].inBox := true;
      ghosts[p].enteringBox := false;
      ghosts[p].count := 0;
      ghosts[p].delay := 5;
      ghosts[p].pic := ord(p) mod 2;
      ghosts[p].inTunnel := false;
      if drawThem then DrawGhost(p);
      end;
   end;
   
Function CheckCollision(nx, ny: Integer; var g: GhostName): Boolean;
    var tg: GhostNames;
    begin
    for tg := Pokey to Shadow do
       with ghosts[tg] do 
         if (mx=nx) then if (my=ny) then if (not eyesOnly) then 
           begin
           g := tg;
           CheckCollision := true;
           exit(CheckCollision);
           end;
    CheckCollision := false;
    end;

Procedure EraseGhosts;
  var g: GhostNames;
  begin
  for g := Pokey to Shadow do
     DrawGhost(g);  {erase all ghosts}
  end;

Procedure ResetMaze;
  var i,j: integer;
  begin
  EraseGhosts;
  LongPause;
  for i := 1 to 20 do
    begin
    IOSetFunction(Recast((i mod 2)+6, CursFunction));
    Pause(10000);
    end;
  IOCursorMode(offCursor);
  LongPause;
  if fruitMaze[player] < 8 then fruitMaze[player] := fruitMaze[player]+1;
  fruitChances[player] := 0;
  SetDots(player);
  DrawMaze;
  if curBlueTime[player]>1 then curBlueTime[player] := curBlueTime[player]-60;
  end;

Procedure DestroyBlue(g: GhostNames);
   var i,inc,x: Integer;
   begin
   DrawGhost(g); {turn off}
   Ghosts[g].eyesOnly := true;
   Ghosts[g].blueTime := 0;
   inc := 20;
   for i := 1 to ghostsSinceDot do
      inc := inc*2;
   ghostsSinceDot := ghostsSinceDot+1;
   x := ghosts[g].scrX+10;
   if x > 740 then x := 740
   else if x < 5 then x := 5;
   SChrFunc(RXor);
   SSetCursor(x, ghosts[g].scry+10);
   Write(inc:1,'0');
   Beep;
   LongPause;
   SSetCursor(x, ghosts[g].scry+10);
   Write(inc:1,'0');
   SChrFunc(RRpl);
   DrawGhost(g); {turn on as eyesOnly}
   UpdateScores(inc);
   end;

Procedure UpdateFruit;
   var p: CurPatPtr;
   begin
   p := FruitPics[FruitMaze[player]];
   RasterOp(RXor, 45, 45, FruitX-gOffset, FruitY-gOffset, SScreenW, SScreenP,
                          0, 0, 4, p);
   if fruitOn then {turning fruit off}
      begin
      fruitChances[player] := fruitChances[player] + 1;
      if fruitChances[player] > 2 then fruitTime := -1 {already had 2 chances}
      else fruitTime := RandomRange(1000,2500);
      end
   else fruitTime := RandomRange(500,1000);  {turning fruit on}
   fruitOn := not fruitOn;
   end;

Procedure DestroyFruit;
   var inc: integer;
       p: CurPatPtr;
   begin
   fruitsGotten[player][FruitMaze[player]] :=
         fruitsGotten[player][FruitMaze[player]] + 1;
   UpdateFruit;    {turn fruit off}
   case FruitMaze[player] of
      1: inc := 10;
      2: inc := 30;
      3: inc := 50;
      4: inc := 70;
      5: inc := 100;
      6: inc := 200;
      7: inc := 300;
      8: inc := 500;
      otherwise: WriteLn('**ILLEGAL FRUIT**');
      end;
   UpdateScores(inc);
   SChrFunc(RXor);
   SSetCursor(fruitX+10, fruitY+10);
   Beep;
   Write(inc:1,'0');
   DrawBotFruit;
   LongPause;
   SSetCursor(fruitX+10, fruitY+10);
   Write(inc:1,'0');
   SChrFunc(RRpl);
   end;

Procedure DrawCir(p: CurPatPtr; x,y: integer);
      var wOff: integer;
      begin
      if (x < 0) then if 55 + x < gOffset then exit(DrawCir) {fully off}
                      else begin
                           wOff := gOffset-x;
                           x := 0;
                           end
      else if (x > 767) then if (x-gOffset >= 767) then exit(DrawCir) {off}
                             else begin
                                  wOff := gOffset-(x-767);
                                  x := 767;
                                  end
      else wOff := gOffset;
      IOLoadCursor(p, wOff, gOffset);
      IOSetCursorPos(x,y);
      end;

Procedure Play;
   var curDir, newDir: Direction;
       posX, posY, x, y, nx, ny, tx, ty, count, inc: Integer;
       c: Char;
       exploding: boolean;
       g, tg: GhostNames;
   label 1,2;
   Procedure ExplodeCircle;
      var i: integer;
      begin
      for i := 0 to 8 do
         begin
         DrawCir(circleExplode[i],posX, posY);
         Pause(5000);
         end;
      LongPause;
      end;

   Procedure DoUpdate;
      begin
      count := count+1;
      if count mod cirCatchUp = 0 then exit(DoUpdate);  {go slower than ghosts}
      DrawCir(circles[newDir][inc], x, y);
      if count mod 4 = 0 then inc := (inc + 1) mod 4;
      if fruitOn then if (nx = FruitMX) and (ny = FruitMY) then DestroyFruit;
      if dots[player][nx, ny] <> noDot then
         begin
         if dots[player][nx, ny] = smallDot then UpdateScores(1)
         else begin
              ChangeGhosts;
              UpdateScores(5);
              end;
         numDots[player] := numDots[player]-1;
         DrawDot(nx,ny,dots[player][nx,ny]);
         dots[player][nx,ny] := noDot;
         if numDots[player] = 0 then 
             begin
             ResetMaze;
             goto 1;
             end;
         end;
      curDir := newDir;
      posX := x;
      posY := y;
      cirMx := nx;
      cirMy := ny;
      end; {DoUpdate}

  Procedure ChangePlayers;
     var cnt: integer;
     begin {change players}
     if (numPlayers = 1) then 
        begin
        if fruitOn then UpDateFruit;
        goto 2;
        end;
     cnt := 0;
     repeat
       cnt := cnt+1;
       player := 1+(player mod numPlayers);
       if cnt > 5 then {game all over}
          begin
          autoPlay := true;
          goto 99;
          end;
     until (numCir[player] <> 0);
     write(FF);
     SSetCursor(348, 500);
     Write('Player ',player:1);
     LongPause;
     DrawMaze;
     BlinkPause;
     goto 2;
     end;

   Procedure HandleCollision(g: GhostNames);
         begin
         if ghosts[g].blueTime > 0 then
             begin
             DestroyBlue(g);
             end
         else if not invincible then
              begin
              Beep;
              DrawGhost(g);  {erase one that ate circle}
              ExplodeCircle;
              for tg := Pokey to Shadow do
                 if tg <> g then DrawGhost(tg);
              IOCursorMode(offCursor);
              if autoPlay then goto 99
              else if numCir[player] = 0 then
                  begin
                  SChrFunc(RXor);
                  SSetCursor(339,yBase+Square*16);
                  Write('GAME OVER!');
                  LongPause;
                  SSetCursor(339,yBase+Square*16);
                  Write('GAME OVER!');
                  SChrFunc(RRpl);
                  if (numPlayers=1) then 
                     begin
                     autoPlay := true;
                     goto 99; {all over}
                     end;
                  end
              else LongPause;
              ChangePlayers;
              end;
        end; {HandleCollision}
   begin {play}
   fruitOn := false;
   blueBlink := 200;
   blueIncBlink := 25;
   if numPlayers = 1 then begin
                          player := 1;
                          DrawMaze;
                          BlinkPause;
                          end
   else begin
        player := numPlayers;
        ChangePlayers;
        end;
2: RemoveCircle;    {jump here if have been eaten or starting new game}
   numCir[player] := numCir[player] - 1;
1: curDir := left;     {jump here if got all dots}
   fruitOn := false;
   saveChar := ' ';
   inc := 0;
   count := 1;
   posX := (768 div 2)-11;
   posY := yBase+Square*21;
   fruitTime := RandomRange(1000,2500);
   UnTransPt(posx, posy, cirMx, cirMy);
   DrawCir(circles[curDir][inc], posX, posY);
   IOSetFunction(CTCursCompl);
   IOCursorMode(IndepCursor);
   NewGhosts(true);
   if not autoPlay then
      begin
      SChrFunc(RXor);
      SSetCursor(357,yBase+Square*16);
      Write('READY!');
      BlinkPause;
      SChrFunc(RXor);
      SSetCursor(357,yBase+Square*16);
      Write('READY!');
      SChrFunc(RRpl);
      end;
   repeat
      UpdateGhosts;
      if CheckCollision(cirMx, cirMy, g) then HandleCollision(g);
      if fruitTime <> -1 then 
         begin
         fruitTime := fruitTime - 1;
         if fruitTime = 0 then UpDateFruit;
         end;
      newDir := curDir;
      if GCentered(posx, posy) then
         begin
         KeyInput;
         if autoPlay then
                newDir := DoRandomDir(curDir,posx,posy,cirMx,cirMy,x,y,nx,ny)
         else begin
              case curDir of
                 up: if saveChar = 'f'
                         then if (not walls[cirMx+1, cirMy])
                                  then newDir := right
                              else {skip}
                     else if saveChar = 'd'
                         then if (not walls[cirMx, cirMy+1])
                                  then newDir := down
                              else {skip}
                     else if saveChar = 'a'
                         then if (not walls[cirMx-1, cirMy])
                                  then newDir := left
                              else {skip};
               down: if saveChar = 's'
                         then if (not walls[cirMx, cirMy-1])
                                  then newDir := up
                              else {skip}
                     else if saveChar = 'f'
                         then if (not walls[cirMx+1, cirMy])
                                  then newDir := right
                              else {skip}
                     else if saveChar = 'a'
                         then if (not walls[cirMx-1, cirMy])
                                  then newDir := left
                              else {skip};
              right: if saveChar = 's'
                         then if (not walls[cirMx, cirMy-1])
                                  then newDir := up
                              else {skip}
                     else if saveChar = 'd'
                         then if (not walls[cirMx, cirMy+1])
                                  then newDir := down
                              else {skip}
                     else if saveChar = 'a'
                         then if (not walls[cirMx-1, cirMy])
                                  then newDir := left
                              else {skip};
               left: if saveChar = 's'
                         then if (not walls[cirMx, cirMy-1])
                                  then newDir := up
                              else {skip}
                     else if saveChar = 'f'
                         then if (not walls[cirMx+1, cirMy])
                                  then newDir := right
                              else {skip}
                     else if saveChar = 'd'
                         then if (not walls[cirMx, cirMy+1])
                                  then newDir := down
                              else {skip};
               end; {case}
              if newDir <> curDir then saveChar := ' ';
              end;
         end; {GCentered}
      
      if DoInc(newDir, posx, posy, cirMx, cirMy, x, y, nx, ny)
             then begin
(*                  if newDir <> curDir then
                     begin
                     DoUpDate;
                     if CheckCollision(cirMx,cirMy,g) then HandleCollision(g);
                     if not DoInc(curDir,posx,posy,cirMx,cirMy,x,y,nx,ny)
                         then WriteLn('**Error Inc**');
                     end;
*)
                   DoUpDate;
                   end
      else begin
           if not GCentered(posX, posY) then DoUpDate {until centered}
           else IOLoadCursor(circles[curDir][2], gOffset, gOffset);
           end;

      if CheckCollision(cirMx, cirMy, g) then HandleCollision(g);
   until false;
   end;

Procedure DoHelp;
   begin
   WriteLn;
   WriteLn(TitleString);
   WriteLn('Pictures of fruit by Terry Vavra.');
   WriteLn;
   WriteLn('Move the circle around the maze.');
   WriteLn('Eating a small dot = 10 points.');
   WriteLn('Eating a large dot = 50 points.');
   WriteLn('Eating fruit when visible = 100 up to 5000 points.');
   WriteLn('Eating all the dots makes fruit more valuable.');
   WriteLn('Eating large dot turns ghosts white.');
   WriteLn('While white or blinking, ghosts = 200, 400, 800 and 1600 points.');
   WriteLn('Avoid being eaten by ghosts when not white or blinking.');
   WriteLn;
   WriteLn('Three circles per game.');
   WriteLn('Extra circle awarded at 10000 points.');
   WriteLn;
   WriteLn('NOTE ****');
   WriteLn('  To avoid damage to the mouse buttons');
   WriteLn('  this version has been modified to be');
   WriteLn('  to be controled by mouse position.');
   WriteLn('  Move the mouse in the obvious way to control the PERQman..');
   WriteLn;
   WriteLn('Skill level 0 = easy; 10 = hard.  Current = ',skillLevel:1,
               '.  Standard = 5.');
   WriteLn;
   WriteLn('DEL aborts game at any time.');
   WriteLn;
   WriteLn('Commands are:');
   WriteLn('  ?, HELP, or h - print help message.');
   WriteLn('  s - set the skill level.');
   WriteLn('  a - go into auto play mode.');
   WriteLn('  1, 2, 3 or 4 - start a game with that number of players.');
   WriteLn('  0 - quit PERQMan.');
   WriteLn;
   end;

Procedure DoCredits;
   var g: GhostNames;
       x,y,cirx, cirY, inc, dotx, doty, i,j: integer;
   begin
   ChangeWindow(1);
   RefreshWindow(1);
   Write(FF);

   SSetCursor(5, 520);
   DoHelp;
   SSetCursor(5, 950);
   Write('** Type DEL to begin **');

   For g := Pokey to Shadow do
      begin
      if ord(g) > 1 then y := yBase+110 else y := yBase+30;
      if ord(g) mod 2 = 0 then x := 100 else x := 384;
      
      RasterOp(RRpl, 45, 45, x, y, SScreenW,SScreenP,
                                    0, 0, 4, ghostPics[g][0]);
      RasterOp(RXor, 45, 21, x, y, SScreenW,SScreenP,
                                    0, 0, 4, eyes[RECAST(g, Direction)]);
      LongPause;
      SSetCursor(x+100, y+25);
      case g of
        Pokey  : Write('- Pokey');
        Bashful: Write('- Bashful');
        Speedy : Write('- Speedy');
        Shadow : Write('- Shadow');
        end;
      LongPause;
      end;

   NewGhosts(false);
   UnTransPt(130, 450, dotx, doty);
   UnTransPt(860, 450, x, y);
   TransPt(x,y, i, cirY);
   RasterOp(RNot, 762, 100, 3, cirY-37, SScreenW, SScreenP,
                            3, cirY-37, SScreenW, SScreenP);

   for g := Pokey to Shadow do
      with ghosts[g] do
        begin
        mx := x+ord(g)*2;
        my := dotY;
        dir := RECAST(ord(g), DIRECTION);
        TransPt(mx, my, scrx, scry);
        DrawGhost(g);  {should be invisible}
        end;
   DrawDot(dotx, doty, bigDot);
   cirX := 720;
   inc := 0;
   IOCursorMode(indepCursor);
   for i := 1 to 662 do
      begin
      KeyInput;
      pause(80);
      if i mod 8 <> 0 then cirX := cirX-1;
      DrawCir(circles[left][inc], cirx, cirY);
      if i mod 4 = 0 then inc := (inc + 1) mod 4;
      for g := Pokey to Shadow do
        with ghosts[g] do
          begin
          DrawGhost(g); {erase old}
          scrx := scrx-1;
          if i mod 13 = 0 then pic := (pic + 1) mod 2;
          if i mod 18 = 0 then dir := RECAST((ord(dir)+1) mod 4, DIRECTION);
          DrawGhost(g); {draw new}
          end;
      end;
   DrawDot(dotx, doty, bigDot);
   for g := Pokey to Shadow do
     begin
     DrawGhost(g); {erase old}
     ghosts[g].blueTime := 32000;
     DrawGhost(g); {draw new as blue}
     end;
   j := 200;
   x := 1;
   SChrFunc(RXor);
   for i := 1 to 665 do
      begin
      KeyInput;
      pause(80*x);
      if i mod 26 <> 0 then cirX := cirX+1;
      DrawCir(circles[right][inc], cirx, cirY);
      if i mod 4 = 0 then inc := (inc + 1) mod 4;
      for g := Pokey to Shadow do
        with ghosts[g] do
          if not eyesOnly then
            begin
            DrawGhost(g); {erase old}
            if i mod 2 <> 0 then scrx := scrx+1;
            if i mod 13 = 0 then pic := (pic + 1) mod 2;
            if cirx >= scrx-20 then 
                begin
                eyesOnly := true;
                IOCursorMode(offCursor);
                SSetCursor(scrx, 450);
                Write(j:1);
                LongPause;
                SSetCursor(scrx, 450);
                Write(j:1);
                IOCursorMode(indepCursor);
                j := j*2;
                x := x+1;
                end
            else DrawGhost(g); {draw new}
            end;
      end;
   IOCursorMode(offCursor);
   SChrFunc(RRpl);
   LongPause;
   end; {DoCredits}

Procedure DoHighScores;
   var level: integer;
   begin
   ChangeWindow(1);
   RefreshWindow(1);
   Write(FF);
   SChrFunc(ROr);
   SSetCursor(334, 200);
   Write('High Scores');
   SSetCursor(335, 200);
   Write('High Scores');
   SSetCursor(334, 201);
   Write('___________');
   SSetCursor(200, 300);
   Write('Skill level         Score      Who');
   SSetCursor(200, 301);
   Write('___________         _____      ___');
   SChrFunc(RRpl);
   for level := 0 to 10 do
      begin
      SSetCursor(200, 330 + level*30);
      if level = skillLevel then Write('*')
      else write(' ');
      Write('   ',level:2,'         =   ');
      if allHighScores[level].score = 0 then write('     -')
      else write(allHighScores[level].score:5,'0');
      Write('      ', allHighScores[level].who);
      end;
   SSetCursor(5, 950);
   Write('** Type DEL to begin **');
   LongPause;
   LongPause;
   LongPause;
   LongPause;
   LongPause;
   end;

var c: Char;

Handler DelHit;
  begin
  SChrFunc(RRpl);
  autoPlay := false;
  goto 99;
  end;
  
begin

{$ifc not debugConst then}
debug := false;
debug1 := false;
invincible := false;
{$endc}

numPlayers := 1;
skillLevel := 5;

autoPlay := false;
InitGame;

Initialize;

IOSetModeTablet(offTablet);
autoPlay := true;


repeat
  InitGame;
  Play;
99: IOSetModeTablet(offTablet);
    if autoPlay then begin
                     c := '1';
                     numPlayers := 1;
                     WriteHighScore;
                     DoHighScores;
                     DoCredits;
                     end
  else begin
       IOCursorMode(offCursor);
       SSetCursor(5,950);
       repeat
        Write('Command: (? for help): [1] ');
          SCurOn;
          repeat
            repeat until IOCRead(transKey, c) = IOEIOC;
          until (c <> DEL) and (c <> ' ');
          if c = HELP then c := '?'
          else if c = CR then c := '1';
          write(c);
          SCurOff;
          case c of 
            '0': ;
            '1': numPlayers := 1;
            '2': numPlayers := 2;
            '3': numPlayers := 3;
            '4': numPlayers := 4;
            'a','A' : begin
                      c := '1';
                      numPlayers := 1;
                      autoPlay := true;
                      end;
            's','S': begin
                     WriteLn('kill level now is ',skillLevel:1,
                               '.  0=easy, 10=hard.');
                     Write('New skill level [5] : ');
                     if eoln then skillLevel := 5
                     else read(skillLevel);
                     readln;
                     if (skillLevel > 10) or (skillLevel < 0) then
                        begin
                        WriteLn('** Illegal skill level');
                        SkillLevel := 5;
                        end;
                     highScore := allHighScores[skillLevel].score;
                     end;
            '?','h' : begin
                      ChangeWindow(1);
                      RefreshWindow(1);
                      Write(FF);
                      DoHelp;
                      end;
            otherwise: WriteLn('  ** Illegal command; type ? for help.');
            end;
       until (c >= '0') and (c <= '4');
       end;
until c='0';

SReadCursor(boxX, boxY);
ChangeWindow(0);
SSetCursor(boxX, boxY);
writeLn;

end.
