program CursorDesign(input, output);
{----------------------------------------------------------------------
  Abstract: A system to aid in the design of PERQ cursors.
 
  Written by Brad A. Myers
  
  Copyright (C) 1980, 1981, 1982 Three Rivers Computer Corporation
----------------------------------------------------------------------}

{----------------------------------------------------------------------
  Change log:
  
  V6.1   5-Apr-83 Brad A. Myers  Fixed bug in landscape monitor. 
  V6.0  28-Feb-83 Brad A. Myers  Made to work for landscape monitor. 
  V5.5  14-Dec-82 Brad A. Myers  Added new Join which asks for function. 
  V5.4  18-Nov-82 Brad A. Myers  Modified type of Solar Cursors. 
  V5.3   2-Nov-82 Brad A. Myers  Added new type for Solar Cursors. 
                                 Join works in the current mode.
                                 Remember x,y offsets for cursors.
                                 Move does multiple at a time.
  V5.2   2-Oct-81 Brad A. Myers  Fixed small bugs 
  V5.1  22-sep-81 Brad A. Myers  Removed transcript 
  V5.0  27-Jul-81 Brad A. Myers  Added transcript 
  V4.2  15-Jul-81 Brad A. Myers  Fixed bugs 
  V4.1  11-Jul-81 Brad A. Myers  New module for line and rect 
  V4.0   7-Jul-81 Brad A. Myers  Window for commands 
                                 New Animate command
                                 New List command
                                 Series I/O
                                 Defaults for many commands
  V3.4   9-May-81 Brad A. Myers  Add mirror on diag and append to file; 
  V3.3  19-Mar-81 Brad A. Myers  Show every 5th line; type compatible; 
                                  CreateWindow on entry; Mirror command 
  V3.2  28-feb-81 Brad A. Myers  Fixed to work with new FileSystem and OS 
  V3.1  10-Dec-80 Brad A. Myers  Added Join with small picture 
  V3.0   5-Dec-80 Brad A. Myers  Added multiple small pictures 
  V2.0  16-Nov-80 Brad A. Myers  Added output to binary files 
----------------------------------------------------------------------}



CONST Trans = FALSE;
      SBitWidth = 768;          {This should be around elsewhere}

imports Screen from Screen;
imports Raster from Raster;
imports io_Unit from io_Unit;
imports IO_Others from IO_Others;
imports ioerrors from ioerrors;
imports PERQ_String from PERQ_String;
imports FileUtils from FileUtils;
imports System from System;
imports LineRect from LineRect;
imports CmdParse from CmdParse;
imports Stream from Stream;
{$Ifc Trans then}
imports Transcript from Transcript;
imports TransUtils from TransUtils;
{$endc}

{$R-}  {range checking off so faster}

Const Version = 'V6.1';
{$ifc trans then}
      CursDTranscript = 'CursDesign.Transcript';
{$endc}
      
type quad = integer;
     wb = (word, bit);
     direction = (Up, down, left, right);
     wordbit = record case wb of
                   word : (w: integer);
                   bit : (b: packed array [0..15] of boolean); {0=rightmost}
                   end;
     ExtString = String[15];
     

{$ifc not trans then}
type pressRec = record
                  yellow, white, green, blue: boolean;
                 end;
{$endc}

var cursor: CursorPattern;
    lastx, lasty: integer;
    justDown : boolean;
    mode: Funct;
    curLitX, curLitY: integer;
    curCursFunct: integer;
    buttons: PressRec;
    
const CR = Chr(#15);
      BeepChar = Chr(7);
      xBase = 22;
      yBase = 100;
      yTextStart = 755;      
      littleX = 508;
      littleY = 15;
      prompt2 = Chr(#212);
      prompt1 = Chr(#213);
      
      SeriesExt = '.Animate';
      TextExt = '.Text';
      BinaryExt = '.Cursor';
      SolarExt = '.SCursor';

      USeriesExt = '.ANIMATE';
      UTextExt = '.TEXT';
      UBinaryExt = '.CURSOR';
      USolarExt = '.SCURSOR';
      
      Null = Chr(0);
      
Type OffRec = record
               x,y: Integer;
              End;

var littleOffset: Array[0..119] of OffRec;
    curOffset: OffRec;
                               
    offsetsRemembered: Array[0..119] of
        record
          whichOne: Integer;
          off: OffRec;
        end;
    remOffCnt: Integer;
    workBuf: CurPatPtr;
    numLittleAcross: Integer;
    
{$ifc not trans then}
Function GetAction(var x,y: integer; wait: boolean; buttons: PressRec): boolean;
   begin
   IOReadTablet(x,y);
   GetAction := TabSwitch;
   buttons.blue := tabblue;
   buttons.yellow := tabyellow;
   buttons.green := tabgreen;
   buttons.white := tabwhite;
   end;

Function GetCharAction(var x,y: integer;wait:boolean; var buttons: pressrec;
                        var existsChar: boolean; var c: Char): boolean;
   begin
   IOReadTablet(x,y);
   existsChar := IOCRead(TransKey,c)=IOEIOC;
   if not existsChar then c := chr(0);
   GetCharAction := TabSwitch;
   buttons.blue := tabblue;
   buttons.yellow := tabyellow;
   buttons.green := tabgreen;
   buttons.white := tabwhite;
   end;

Function ReadString(var s: String; b: boolean): boolean;
  begin
  Readln(s);
  ReadString := false;
  end;

Function ReadNumber(var i: integer; def:integer; b: Boolean): boolean;
   begin
   if eoln then i := def
   else read(i);
   readln;
   ReadNumber := false;
   end;
{$endc}

Procedure ShowCurOffsets;
  begin
  ChangeWindow(2);
  SSetCursor(298 + 4, 666);
  Write('Cursor Origin');
  SSetCursor(298, 680); 
  Write('X = ',curOffset.x:2, '  Y = ',curOffset.y:2);
  ChangeWindow(1);
  end;

Procedure WaitNoPress;
   var press: boolean;
       x,y: integer;
   begin
   repeat
     press := GetAction(x,y,TRUE,buttons);
   until not press;
   end;

Procedure WaitPress(var x,y: Integer);
   var press: boolean;
   begin
   repeat
     press := GetAction(x,y,TRUE,buttons);
   until press;
   end;

procedure CursorCursor(default: boolean);
   begin
   if default then IOLoadCursor(DefaultCursor,0,0)
   else begin
        workBuf^ := cursor;
        IOLoadCursor(workBuf, curOffset.x, curOffset.y);
        end;
   end;

procedure SetBit(y,x: quad; val: boolean ) ;
   var hack: wordBit;
   begin
   hack.w := cursor[y, x div 16];
   hack.b[ (15 - (x mod 16))] := val;
   cursor[y,x div 16] := hack.w;
   end;

function GetBit(y,x: quad): boolean;
   var hack: wordBit;
   begin
   hack.w := cursor[y, x div 16];
   GetBit := hack.b[(15 - (x mod 16))];
   end;

procedure DisplayBoard;  {draws the lines separating the squares}
   var i,j: integer;
   begin
   for i := 0 to 56 do {do vertical lines}
      if i = 28 then
          RasterOp(7,1,543, xBase+i*8, yBase-15, SScreenW, SScreenP,
                            xBase+i*8, yBase-15, SScreenW, SScreenP)
      else if i mod 5 = 0 then
          for j := 0 to 527 div 2 do
            RasterOp(7,1,1, xBase+i*8, yBase-7+j*2, SScreenW, SScreenP,
                            xBase+i*8, yBase-7+j*2, SScreenW, SScreenP)
      else
          RasterOp(7,1,513, xBase+i*8, yBase, SScreenW, SScreenP,
                            xBase+i*8, yBase, SScreenW, SScreenP);

   for i := 0 to 64 do {do horizontal lines}
      if i = 32 then
          RasterOp(7,479,1, xBase-15, i*8+yBase, SScreenW, SScreenP,
                            xBase-15, i*8+yBase, SScreenW, SScreenP)
      else if i mod 5 = 0 then
          for j := 0 to 463 div 2 do
            RasterOp(7,1,1, xBase-7+j*2, i*8+yBase, SScreenW, SScreenP,
                            xBase-7+j*2, i*8+yBase, SScreenW, SScreenP)
      else
          RasterOp(7,449,1, xBase, i*8+yBase, SScreenW, SScreenP,
                            xBase, i*8+yBase, SScreenW, SScreenP);

   end;
   
procedure DisplayLittle;  {draws the lines separating the dest boxes}
   var i, w: integer;
   begin
   for i := 0 to numLittleAcross do {vertical lines}
      RasterOp(7,1,721, littleX+i*64, littleY, SScreenW, SScreenP,
                        littleX+i*64, littleY, SScreenW, SScreenP);
   w := numLittleAcross*64 + 1;
   for i := 0 to 10 do {horiz lines}
      RasterOp(7,w,1, littleX, littleY+i*72, SScreenW, SScreenP,
                      littleX, littleY+i*72, SScreenW, SScreenP);
   end;
   
procedure CursPt(y,x: integer; f: Funct);
   var rfunct, w: integer;
       val: boolean;
   begin
   val := GetBit(y,x);
   if ((f = black) and val) or ((f = white) and (not val)) then exit(CursPt);
   if (f = white) or ( (f = toggle) and val) then
                       begin
                       SetBit(y,x, false);
                       rfunct := RXor;
                       end
   else begin
        SetBit(y,x, true);
        rfunct := RXNor;
        end;
   RasterOp(rfunct,7,7,xBase+1+x*8,yBase+1+y*8,SScreenW, SScreenP, xBase+1+x*8,
         yBase+1+y*8,SScreenW, SScreenP);
   w := x div 16;  {word index}
   RasterOp(rfunct,1,1,curLitX+x, curLity+y,SScreenW,SScreenP,
                       curLitX+x, curLity+y,SScreenW,SScreenP);
   end;
   
Function GetLittleBit(litx,lity: Integer; y,x: quad): boolean;
   begin
   workBuf^[0,0] := 0;
   RasterOp(RRpl, 1, 1, 0,0,4, workBuf, litx+x,lity+y, SScreenW,SScreenP);
   GetLittleBit := workBuf^[0,0] <> 0;
   end;

Procedure XSquare(y,x: integer);
  begin
  Line(XorLine,xBase+1+x*8,yBase+1+y*8,xBase+7+x*8,yBase+7+y*8,SScreenP);
  Line(XorLine,xBase+1+x*8+7,yBase+1+y*8,xBase+1+x*8,yBase+7+y*8,SScreenP);
  end;

procedure FixScreen;
   var x,y : quad;
       w: integer;
   begin
  {erase}
   RasterOp (RXor,littleX-1,yTextStart-yBase,0,yBase,SScreenW,SScreenP,
                                             0,yBase,SScreenW,SScreenP);
   for y := 0 to 63 do
      for x := 0 to 55 do
         if GetBit(y,x) then RasterOp(7,7,7,xBase+1+x*8,yBase+1+y*8,SScreenW,
               SScreenP, xBase+1+x*8, yBase+1+y*8,SScreenW, SScreenP);
   workBuf^ := Cursor;
   RasterOp(RRpl, 56, 64, curLitX, curLitY, SScreenW,SScreenP,
                          0, 0, 4, workBuf);
   DisplayBoard;
   ChangeWindow(2);
   SSetCursor(50,680);
   Write('X = ',0:2,'  Y = ',0:2,' ');
   ShowCurOffsets;
   ChangeWindow(1);
   end;
   
Procedure Verify(litx,lity: integer);
  var x,y: quad;
  begin
  FixScreen;
  for y := 0 to 63 do
    for x := 0 to 55 do
       if GetLittleBit(litx,lity,y,x) <> GetBit(y,x) then XSquare(y,x);
  end;

Function WhichLittle(baseX, baseY: Integer): Integer;
  var i: Integer;
  begin
  i := (((baseY-4-littleY) div 72)*numLittleAcross) + ((baseX-littleX) div 64);
  WhichLittle := i;
  end;

procedure ReadLittle(baseX,baseY: integer);
   begin
   RasterOp(RRpl, 56, 64, 0,0,4,workBuf, baseX,baseY,SScreenW,SScreenP);
   cursor := workBuf^;
   curOffset := littleOffset[whichLittle(baseX, baseY)];
   FixScreen;
   end;

procedure SetOffset(x,y: Integer);
  begin
  curOffset.x := x;
  curOffset.y := y;
  littleOffset[WhichLittle(curLitX, curLitY)] := curOffset;
  ShowCurOffsets;
  end;

procedure JoinLittle(baseX,baseY: integer; m: Funct);
   var w, y, val: integer;
   begin
   {$R-} {no range checking}
   RasterOp(RRpl, 56, 64, 0,0,4,workBuf,baseX,baseY,SScreenW,SScreenP);
   for y := 0 to 63 do
      for w := 0 to 3 do 
          begin
          val := workBuf^[y,w];
          if w = 3 then val := LAnd(val, #177400);
          case m of
             toggle: cursor[y,w] := LXor(cursor[y,w], val);
             black : cursor[y,w] := LOr(cursor[y,w], val);
             white : cursor[y,w] := LAnd(cursor[y,w], val);
             end;
          end;
   {$R+}
   FixScreen;
   end;

procedure Init;
   var i, j : integer;
   begin
   New(0,4,workBuf);
   lastx := -1;
   lasty := -1;
   numLittleAcross := (SBitWidth-4-littleX) div 64;
   justDown := false;
   mode := toggle;
   CreateWindow(2, 0, 0, SBitWidth, 1024, '');
   curLitX := littleX+4;
   curLitY := littleY+4;
   DisplayLittle;
   for i := 0 to 63 do
      for j := 0 to 3 do
         cursor[i,j] := 0;  {all off}
   IOSetModeTablet(relTablet);
   IOSetFunction(CTCursCompl);
   curCursFunct := ord(CTCursCompl);
   ChangeWindow(2);
   SSetCursor(256,20);
   Write('CursDesign, Version ',Version);
   CreateWindow(1, 0, yTextStart, SBitWidth, 1024-yTextStart,
                'Command       Type ? for Help');
   for i := 0 to 119 do
     begin
     littleOffset[i].x := 0;
     littleOffset[i].y := 0;
     end;
   curOffset.x := 0;
   curOffset.y := 0;
   FixScreen;
   end;
   
Procedure DoQuit;
  var x,y: integer;
  begin
  WriteLn;
{$ifc trans then}
  WriteLn('  Writing transcript...');
  CloseTranscript;
{$endc}
  SReadCursor(x,y);
  ChangeWindow(0);
  SSetCursor(x,y);
  WriteLn;
  Exit(CursorDesign);
  end;

function EightTimes(i: integer): integer;  {uses adds so OK to overflow}
   begin
   i := i+i;  {2*i}
   i:= i+i;   {4*i}
   EightTimes := i+i;   {8*i}
   end;

procedure DumpCursor(var f: Text);
   var y: quad; w: integer;
   begin
   WriteLn(f, 'Var foo: ^ARRAY [0..63, 0..3] of integer; ');
   WriteLn(f, '   {foo is indexed as foo^[y,x] where x is in words}');
   for y := 0 to 63 do
      for w := 0 to 3 do
         if cursor[y,w] <> 0 then
             WriteLn(f, 'foo^[',y:2,', ',w:1,'] := #',cursor[y,w]:1:-8,';':1);
   end;

function GetNum(s: String; startIndex: integer): integer;
   var temp, i: integer;
   begin
   temp := 0;
   i := startIndex+1;
   while s[i] in ['0'..'7'] do {octal}
       begin
       temp := EightTimes(temp) + ord(s[i]) - ord('0');
       i := i+1;
       end;
   GetNum := temp;
   end;

function GetDecNum(s: String; startIndex: integer): integer;
   var temp, i: integer;
   begin
   temp := 0;
   i := startIndex;
   repeat
    i := i+1
   until s[i] <> ' ';
   if not (s[i] in ['0'..'9']) then begin
                                    GetDecNum := -1;
                                    exit(GetDecNum);
                                    end;
   while s[i] in ['0'..'9'] do
       begin
       temp := temp*10 + ord(s[i]) - ord('0');
       i := i+1;
       end;
   GetDecNum := temp;
   end;

procedure ReadDumpCursor(var f: Text);
   var y: quad; w,index, val: integer;
   s: String;
   label 1;
   begin
   for y := 0 to 63 do
      for w := 0 to 3 do
         cursor[y,w] := 0;   {all zero first}
   repeat
      if EOF(f) then begin  {file has no data}
                     FixScreen;
                     exit(ReadDumpCursor);
                     end;
      ReadLn(f, s)
   until POS(s, '#') <> 0;
   while true do
         begin
         index := POS(s, '[');
         if index = 0 then goto 1;
         y := GetDecNum(s, index);
         if y = -1 then goto 1;
         index := POS(s,',');
         if index = 0 then goto 1;
         w := GetDecNum(s, index);
         index := POS(s, '#');
         if index = 0 then goto 1;
         val := GetNum(s,index);
         if w = 3 then val := LAND (val, #177400); {remove last 8 bits}
         Cursor[y,w] := val;
         repeat
            IF EOF(f) then begin
                           FixScreen;
                           exit(ReadDumpCursor);
                           end;
            ReadLn(f, s);
         until s <> '';
         end;
   curOffset.x := 0;
   curOffset.y := 0;
   littleOffset[WhichLittle(curLitX, curLitY)] := curOffset;
   1 : Write(' *** Bad format at: "',s,'", index =',index:1,'.  Aborted***');
   end;

procedure ReadBinCursor(f: FileID; blk: integer);
   begin
   FSBlkRead(f, blk-1, recast(workBuf, pDirBlk));
   cursor := workBuf^;
   curOffset.x := 0;
   curOffset.y := 0;
   littleOffset[WhichLittle(curLitX, curLitY)] := curOffset;
   FixScreen;
   end;

procedure ReadCurCursor;
   var dum, y: integer;
      w: Quad;
   begin
   for y := 0 to 63 do
      for w := 0 to 3 do 
          workBuf^[y,w] := 0;
   IOReadCursPicture(workBuf, curOffset.x, curOffset.y);
   littleOffset[WhichLittle(curLitX, curLitY)] := curOffset;
   cursor := workBuf^;
   FixScreen;
   end;

procedure DumpBinCursor(f: FileID; blk: integer);
   begin
   workBuf^ := cursor;
   FSBlkWrite(f, blk-1, recast(workBuf, pDirBlk));
   FSClose(f, blk, 4096);
   end;

procedure DumpBinLittle(x,y: integer; f: FileID; blk: integer);
    {write block blk}
   var i,j, val: integer;
   begin
   RasterOp(RXor, 64, 64, 0, 0, 4, workBuf, 0, 0, 4, workBuf); {erase buffer}
   RasterOp(RRpl, 56, 64, 0, 0, 4, workBuf, x, y, SScreenW, SScreenP);
   IOLoadCursor(workBuf, 0,0);
   FSBlkWrite(f, blk, RECAST(workBuf, pDirBlk));
   end;

procedure ReadBinLittle(x,y: integer; f: FileID; blk:integer;isSolar: boolean);
    {Read block blk; x in words}
   var buf: record case boolean of
              true: (p: pDirBlk);
              false: (c: CurPatPtr);
              end;
       offx, offy: integer;
   begin
   NEW(0, 256, buf.p);
   if isSolar then 
     begin
     FSBlkRead(f, 0, buf.p);
     blk := blk+1;
     offx := buf.p^.buffer[blk*2];  {offsets start at word 2}
     offy := buf.p^.buffer[blk*2+1];
     end
   else begin
        offx := 0;
        offy := 0;
        end;
   FSBlkRead(f, blk, buf.p);
   if (x=curLitx) and (y=curLity) then
      begin
      cursor := buf.c^;
      curOffset.x := offx;
      curOffset.y := offy;
      FixScreen;
      end
   else RasterOp(RRpl, 56, 64, x, y, SScreenW, SScreenP,
                               0, 0, 4, buf.c);
   with littleOffset[WhichLittle(x,y)] do
     begin
     x := offx;
     y := offy;
     end;
   DISPOSE(buf.p);
   end;

Procedure RememberOffset(x,y,i: Integer);
  var temp: Integer;
  begin
  temp := WhichLittle(x,y);
  with offsetsRemembered[remOffCnt] do
    begin
    whichOne := i;
    off := littleOffset[temp];
    end;
  remOffCnt := remOffCnt+1;
  end;

Procedure CloseUpSolar(fid: FileID; BlkCnt: Integer);
{blkCnt is number of cursors + 1}
  var buf: pDirBlk;
      i: Integer;
  begin
  New(Buf);
  FSBlkRead(fid, 0, buf);
  buf^.buffer[0] := blkCnt-1;
  buf^.buffer[1] := 0;
  for i := 0 to remOffCnt-1 do
    with offsetsRemembered[i] do
     begin
     buf^.buffer[(whichOne+1)*2] := off.x;
     buf^.buffer[(whichOne+1)*2+1] := off.y;
     end;
  for i := 2*blkCnt to 255 do {clear out rest}
     buf^.buffer[i] := 0;
    
  FSBlkWrite(fid, 0, buf);
  FSClose(fid, BlkCnt, 4096);
  Dispose(Buf);
  end;

procedure All(val: Boolean);
  var x,y: quad
  begin
  for y := 0 to 63 do
     for x := 0 to 55 do
         if GetBit(y,x) <> val then
              CursPt(y,x, Toggle);
  end;

procedure InvertAll;
  var x,y: quad;
      val: boolean;
  begin
  for y := 0 to 63 do
     for x := 0 to 55 do
         CursPt(y,x, Toggle);
  end;

Procedure SetTBit(var t: CursorPattern; y, x: Quad; val: boolean);
   var hack: wordBit;
   begin
   hack.w := t[y, x div 16];
   hack.b[ (15 - (x mod 16))] := val;
   t[y,x div 16] := hack.w;
   end;

procedure Mirror(dir: char);  {r=rightleft, u=updown, d=diag}
  var x,y: quad;
      temp: CursorPattern;
  begin
  for y := 0 to 63 do
     for x := 0 to 3 do
        temp[y,x] := 0;
  if dir = 'r' then 
    for y := 0 to 63 do
       for x := 0 to 55 do
         SetTBit(temp, y, 55-x, GetBit(y,x))
  else if dir = 'u' then 
    for x := 0 to 55 do
       for y := 0 to 63 do
         SetTBit(temp, 63-y, x, GetBit(y,x))
  else
    for x := 0 to 55 do
       for y := 0 to 55 do {lose information}
         SetTBit(temp, y, x , GetBit(x, y));
  cursor := temp;
  fixScreen;
  end;
  
procedure MoveN(d: Direction; off: Integer);
   var x,y: quad;
       temp : Boolean;
   begin
   if off = 0 then exit(MoveN)
   else if (off < 0) or 
           (((d = down) or (d=up)) and (off > 63)) or
           (((d = left) or (d=right)) and (off > 55)) then
              begin
              Write(' ** Illegal move value');
              exit(MoveN);
              end;
   case d of 
     down : begin
            for y := 63-off downto 0 do
              for x := 0 to 55 do
                 begin
                 temp := GetBit(y,x);
                 if temp then CursPt(y+off,x, black)
                 else CursPt(y+off,x, white);
                 end;
            for y := 0 to off-1 do
              for x := 0 to 55 do 
                 CursPt(y,x, white);
            end;
     up: begin
         for y := off to 63 do
              for x := 0 to 55 do
                 begin
                 temp := GetBit(y,x);
                 if temp then CursPt(y-off,x, black)
                 else CursPt(y-off,x, white);
                 end;
         for y := 0 to off-1 do
           for x := 0 to 55 do 
                 CursPt(63-y, x, white);
         end;
     left: begin
           for x := off to 55 do
              for y := 0 to 63 do
                 begin
                 temp := GetBit(y,x);
                 if temp then CursPt(y,x-off, black)
                 else CursPt(y,x-off, white);
                 end;
           for y := 0 to 63 do 
             for x := 0 to off-1 do
                 CursPt(y, 55-x, white);
           end;
     right: begin
            for x := 55-off downto 0 do
              for y := 0 to 63 do
                 begin
                 temp := GetBit(y,x);
                 if temp then CursPt(y,x+off, black)
                 else CursPt(y,x+off, white);
                 end;
            for y := 0 to 63 do 
             for x := 0 to off-1 do
                 CursPt(y, x, white);
            end;
     end;
  end; {MoveN}

procedure ReadChar(var c: char; def, pressChar: char);
   var x,y: integer;
       press,existsChar: boolean;
   begin
   if def <> Null then write('[',def,'] ');
   SCurOn;
   repeat
     press := GetCharAction(x,y,TRUE,buttons,existsChar,c);
   until (existsChar) or (press and (pressChar <> Null));
   if (press and (pressChar <> Null)) then
      begin
      WaitNoPress;
      c := pressChar;
      end
   else if (c = CR) and (def <> Null) then c := def;
   if c <> CR then write(c);
   end; {ReadChar}

function Confirm(s: String; def: Char): boolean;
   var c: Char;
       status: integer;
   begin
   Write(s,' (Confirm) ');
   while true do
      begin
      ReadChar(c, def, Null);
      if c = 'y' then begin
                      Confirm:=true;
                      exit(Confirm);
                      end
      else if c = 'n' then begin
                           Confirm:=false;
                           exit(Confirm);
                           end
      else Write(' "y" or "n": ');
      end;
   end; {Confirm}

{Sets x,y to upper left corner of box to use}
Function FindLittle(Var x,y: integer): boolean;
  var press: boolean;
  begin
  WaitNoPress;
  WaitPress(x,y);
  if (x<=littleX+4) or (x >= SBitWidth-4) or (y < littleY-4) or
     (y > littleY+721) then FindLittle := false
  else begin
       FindLittle := true;
       x := ((x - littleX) div 64)*64 + littleX + 4;
       y := (((y - littleY) div 72)*72)+littleY+4;
       end;
  WaitNoPress;
  end;

Function IncLittle(var x,y: integer): boolean;
   begin
   IncLittle := true;
   if x >= SBitWidth - 64 then
      begin
      x := (littleX+4);
      if y > littleY+593 then IncLittle := false
      else y := y + 72;
      end
   else x := x + 64;
   end;

Procedure DoAnimate(x,y,x2,y2,delay: integer);
  var i, j, val: integer;
      curX, curY, dumx, dumy: integer;
      press: boolean;
  begin
  curX := x;
  curY := y;
  repeat
    press := GetAction(dumx, dumy,TRUE, buttons);
    RasterOp(RRpl,56,64, 0, 0, 4, workBuf, curX, curY, SScreenW, SScreenP);
    IOLoadCursor(workBuf, 0,0);
   
    for i := 1 to delay do;

    if (curX = x2) and (curY = y2) then begin
                                        curX := x;
                                        curY := y;
                                        end
    else if not IncLittle(curX, curY) then
          begin
          WriteLn('** IMPOSSIBLE in Animate');
          exit(DoAnimate);
          end;
  until press;
  WaitNoPress;
  IOLoadCursor(DefaultCursor,0,0)
  end;

Function GetPointPair(var x, y, x2, y2: integer): boolean;
   begin
   GetPointPair := false;
   Write('Point at first little');
   if FindLittle(x,y) then
        begin
        Write('.  Point at end little');
        if FindLittle(x2, y2) then 
             if (y2 > y) or ((y2=y) and (x2>=x)) then
                    GetPointPair := true
             else begin
                  Writeln(BeepChar);
                  write('** Second point before first');
                  end
        else write(' XXX');
        end
   else write(' XXX');
   end;  {FindPointPair}

Function AddExtension(var fileName: PathName; ext: ExtString): boolean;
   var upperN: PathName;
       upperE: ExtString;
   begin
   if fileName = '' then
     begin
     WriteLn('** Name empty; aborting!');
     AddExtension := false;
     end
   else begin
        AddExtension := true;
        upperN := fileName;
        upperE := ext;
        ConvUpper(upperN);
        ConvUpper(upperE);
        if Pos(upperN, upperE) = 0 then fileName := Concat(fileName, ext);
        end;
   end;

Function GetOutFile(var fid: FileID; var blk: Integer; ext: ExtString;
                    var fileName: PathName): boolean;
   var i: integer;
       c: Char;
       dummy: boolean;
   begin
   GetOutFile := false;
   dummy := ReadString(fileName, FALSE);
   if not AddExtension(fileName, ext) then exit(GetOutFile);
   fid := FSLocalLookUp(fileName, blk, i);
   if fid <> 0 then
      if blk = 0 then blk := 1
      else begin
           WriteLn;
           if ext = SolarExt then blk := blk-1;
           Write('** ',fileName,' exists and has ',blk:1,
                    ' blocks.  Overwrite? ');
           if confirm('','y') then
              begin
              WriteLn;
              if Confirm('Append to end? ','n')
                 then blk := blk+1
              else blk := 1;
              writeln;
              end
           else exit(GetOutFile); { don't overwrite }
           end
   else begin
        fid := FSEnter(fileName);
        if fid = 0 then begin
                        Write('** Sorry, Cannot create ',fileName);
                        exit(GetOutFile);
                        end;
        blk := 1;
        end;
   GetOutFile := true;
   end;

Procedure DoFunct;
  Handler NotNumber(f: PathName);
    begin
    StreamKeyboardReset(input);
    WriteLn(BeepChar);
    WriteLn('** Not valid! The cursor function must be an integer from 0 to 7');
    WriteLn('   of which only a few are useful.  The odd functions are the');
    WriteLn('   same as the even ones with the screen color inverted.');
    WriteLn('     0 is screen off but cursor visible.');
    WriteLn('     2 and 3 are useless.');
    WriteLn('     4 means 0''s in cursor allow background to show and 1''s are black');
    WriteLn('     6 means the cursor is XORed with the background.');
    WriteLn('   The current cursor function is ',curCursFunct:1);
    SCurOn;
    exit(DoFunct);
    end;
{$ifc trans then}
  Handler BadNumber;
    begin
    Raise NotNumber('');
    end;
{$endc}
  var i: integer;
      dummy: boolean;
  begin
  dummy := ReadNumber(i, 6, false);
  if (i > 7) or (i < 0) then Raise NotNumber('')
  else begin
       curCursFunct := i;
       IOSetFunction(RECAST(i, CursFunction));
       end;
  end;

Procedure DoHelp;
   var c: Char;
   begin
   WriteLn;
   WriteLn;
   WriteLn('    ? or HELP key = This message');
   WriteLn('    A = Animate (a series of cursors)');
   WriteLn('    a = all (black, white or invert)');
   WriteLn('    b = black (mode for press)');
   WriteLn('    c = cursor (load cursor with current picture or default)');
   WriteLn('    C = Circle (a circle can be drawn)');
   WriteLn('    d = define x,y offsets (when used as a cursor)');
   WriteLn('    e = edit (change to little box w/o changing its contents)');
   WriteLn('    f = function (for cursor)');
   WriteLn('    g = get (cursor from file, screen, or current cursor)');
   WriteLn('    G = Get series or Solar (read multiple cursors from file)');
   WriteLn('    i = initialize (destroy current picture)');
   WriteLn('    j = join (little picture with current in current mode)');
   WriteLn('    J = Join with function');
   WriteLn('    l = line (draws a horiz or vert line defined by pen in current mode)');
   WriteLn;
   Write('** Hit any key for more **');
   ReadChar(c, Null, Null);
   WriteLn;
   WriteLn('    L = List (all animate or cursor files in directory)');
   WriteLn('    m = move (entire picture up, down, left or right)');
   WriteLn('    M = Mirror image (rotate around x or y center)');
   WriteLn('    o = output (to text or binary file)');
   WriteLn('    O = Output series or solar (write multiple cursors onto file)');
   WriteLn('    q = quit');
   WriteLn('    r = rectangle (draw a box defined by pen in current mode)');
   WriteLn('    s = specify (point at little box to hold picture)');
   WriteLn('    t = toggle (change mode to toggle)');
   WriteLn('    T = Text (a line of text followed by CR can be typed');
   WriteLn('    v = verify (compare current with little)');
   WriteLn('    w = white (change mode to white)');
   WriteLn('    x = fixScreen (refresh display)');
   end;

Function DoPoint(var x,y: Integer; doDraw: boolean;var press: boolean): boolean;
  var tabx,taby: integer;
  begin
  press := GetAction(tabx, taby, TRUE, buttons);
  DoPoint := false;
  if (tabx < xBase) or (taby < yBase) or (tabx >= xBase+448) or
     (taby >= yBase+512) then if press then IOBeep
                              else
  else begin
       y := (taby-yBase) div 8;
       x := (tabx-xBase) div 8;
       DoPoint := true;
       if (lastx <> x) or (lasty <> y) or (press and not justDown) then
          begin {new point}
          if (lastx <> x) or (lasty <> y) then
            begin
            SCursorOn := false;
            ChangeWindow(2);
            SSetCursor(50,680);
            Write('X = ',x:2,'  Y = ',y:2,' ');
            ChangeWindow(1);
            SCursorOn := true;
            lastx := x;
            lasty := y;
            end;
          if press then 
             begin
             if doDraw then 
               if buttons.White then CursPt(lastY, lastX, white)
               else if buttons.Green then CursPt(lastY, lastX, black)
               else if buttons.Blue then CursPt(lastY, lastX, toggle)
               else CursPt(lastY, lastX, mode);
             justDown := true;
             end;
          end;
       end;
   if not press then JustDown := false;
   end;

Function WaitPoint(var x,y: Integer): Boolean;
  var press, ok: boolean;
  begin
  WaitPoint := false;
  Repeat
    ok := DoPoint(x,y,false, press);
  until press;  
  WaitPoint := ok;
  end;


Procedure DoList;
   var scanPtr: ptrScanRecord;
       s, s2: PathName;
       c: Char;
       fid: FileID;
       i: Integer;
       dummy: Boolean;
    Procedure DoOne;
      begin
      i := i+1;
      write(s:19,' ');
      if i mod 4 = 0 then writeLn;
      end;
   begin
   Write('ist animates, cursors, solarcursors, or everything? ');
   ReadChar(c, 'e','e');
   if (c <> 'a') and (c <> 'c') and (c <> 'e') and (c <> 's') then
      begin
      Write('  XXX');
      exit(DoList);
      end;
   writeLn;
   Write('In directory: [',FSDirPrefix,'] : ');
   dummy := ReadString(s, FALSE);
   if s <> '' then begin
                   if s[length(s)] <> '>' then AppendChar(s, '>');
                   fid := FSSearch(FSSysSearchList, s, i, i);
                   if fid = 0 then begin
                                   Write(chr(7),'** ',s,' not found.');
                                   exit(DoList);
                                   end;
                   end
   else s := FSDirPrefix;
   NEW(scanPtr);
   scanPtr^.initialCall := true;
   scanPtr^.dirName := s;
   i := 0;
   while FSScan(scanPtr, s, fid) do
     begin
     s2 := s;
     ConvUpper(s2);
     if ((c = 'a') or (c = 'e')) and (Pos(s2, USeriesExt) <> 0) then DoOne
     else if ((c = 's') or (c = 'e')) and (Pos(s2, USolarExt) <> 0) then DoOne
     else if ((c = 'c') or (c = 'e')) and 
             ((Pos(s2, UTextExt) <> 0) or (Pos(s2, UBinaryExt) <> 0)) then
        DoOne;
     end;
   WriteLn;
   Write(i:1,' file');
   if i <> 1 then Write('s');
   WriteLn(' found.');
   end;

procedure MainLoop;
   var c: char;
       fileName, s: STRING;
       f: Text;
       fid: FileId;
       status, tabx, taby, count, i, j, x, y, x2, y2, delay, blks,cnt: integer;
       dir: Direction;
       val, ok, leave, press, existsChar: boolean;
   label 1, 4, 99;
{$ifc trans then}
  Handler BadNumber;
      begin
      Write('** Bad Number');
      SCurOn;
      Goto 99;
      end;
{$elsec}
  Handler NotNumber(f: PathName);
      begin
      StreamKeyboardReset(input);
      Write('** Bad Number');
      SCurOn;
      Goto 99;
      end;
{$endc}
   Handler CtlC;
      begin
      WriteLn('^C');
      IOKeyClear;
      goto 99;
      end;
   Handler HelpKey(var s: Sys9s);
      begin
      s := '?';
      end;
   begin
   Write(prompt1, prompt2);
   while true do
      begin
      SCurOn;
      press := GetCharAction(x,y,TRUE,buttons,existsChar,c);
      if existsChar then
          begin
          Write(c);
          case c of {see what user wants to do}
             CR,' ' : ;
             'A' : begin
                   Write('nimate.  ');
                   If GetPointPair(x,y,x2,y2) then
                          begin
                          Write('.  Type delay [10000]:');
                          press := ReadNumber(delay, 10000, false);
                          WaitNoPress;
                          Write('  Press to GO...');
                          WaitPress(i,j); {ignore point}
                          WaitNoPress;
                          Write('   Press to stop...');
                          DoAnimate(x, y, x2, y2, delay);
                          end;
                   end;
             'a' : begin
                   Write('ll: black, white, or invert? ');
                   ReadChar(c,Null,Null);
                   case c of
                       'b' : if confirm('lack',Null) then All(true);
                       'w' : if confirm('hite',Null) then All(false);
                       'i' : InvertAll;
                       otherwise : Write( ' XXX');
                       end;
                   end;
             'b' : begin write('lack'); mode := black; end;
             'c' : begin
                   Write('ursor: current or default? ');
                   ReadChar(c,'d','d');
                   case c of
                       'c' : CursorCursor(false);
                       'd' : CursorCursor(true);
                       otherwise : Write(' XXX');
                       end;
                   end;
             'C' : begin
                   Write('ircle.  Press at center, then press for radius');
                   MakeCircle(yBase,xBase,447,511,mode,CursPt);
                   end;
             'd' : begin
                   Write('efine offset; press at origin of cursor');
                   if WaitPoint(x,y) then SetOffset(x,y)
                   else Write('  XXX');
                   end;
             'e' : begin
                   Write('dit.  Point at little box to edit');
                   if FindLittle(x,y) then
                     begin
                     curLitX := x;
                     curLitY := y;
                     ReadLittle(x,y);
                     end
                   else Write('  XXX');
                   end;
             'f' : begin
                   Write('unction for cursor (0..7): [6] ');
                   DoFunct;
                   end;
             'g' : begin
                   writeLn('et from text or binary file, cursor, or');
                   Write('picture? ');
                   ReadChar(c,'b','p');
                   case c of
                      't' : begin
                            write('ext file name: ');
                            press := ReadString(fileName, false);
                            if AddExtension(fileName, TextExt) then
                              begin
                              fid := FSLookup(fileName,i,i);
                              if fid = 0 then write(BeepChar,'** Sorry, ',filename,' not found')
                              else begin
                                   Reset(f,fileName);
                                   ReadDumpCursor(f);
                                   end;
                              end;
                            end;
                      'b' : begin
                            write('inary file name: ');
                            press := ReadString(fileName, false);
                            if AddExtension(fileName, BinaryExt) then
                              begin
                              fid := FSLookup(fileName,blks, i);
                              if fid = 0 then write('** Sorry, ',filename,' not found')
                              else begin
                                   if blks > 1 then
                                     begin
                                     Write(chr(7),'** ',fileName,' has ',blks:1,' cursors in it.  Which one do you want to use? [1] ');
                                     press := ReadNumber(i, 1, false);
                                     end
                                   else i := 1;
                                   if i > blks then write('** ',i:1,' too big')
                                   else ReadBinCursor(fid, i);
                                   end;
                              end;
                            end;
                      'c' : if Confirm ('ursor','y') then ReadCurCursor;
                      'p' : begin
                            WriteLn('icture.');
                            Write('   Point at little box to read');
                            if FindLittle(x,y) then ReadLittle(x,y)
                            else Write(' XXX');
                            end;
                      otherwise : Write(' XXX');
                      end;
                   end;
             'G' : begin
                   leave := false;
                   j := 0;
                   write('et from solar or animate file? ');
                   ReadChar(c,'a','a');
                   if (c <> 's') and (c <> 'a') then
                      begin
                      write('  XXX');
                      goto 4;
                      end;
                   Write('  Filename: ');
                   press := ReadString(fileName, false);
                   if c = 'a' then
                     if not AddExtension(fileName, SeriesExt) then goto 4
                     else
                   else if not AddExtension(fileName, SolarExt) then goto 4;
                   fid := FSLookup(fileName,blks, i);
                   if c = 'a' then cnt := blks
                   else cnt := blks-1;
                   if fid = 0 then write('** Sorry, ',fileName,' not found')
                   else begin
                        if cnt > 1 then
                           begin
                           Write(fileName,' has ',cnt:1,' cursors in it.  How many do you want to read? [',cnt:1,'] ');
                           press := ReadNumber(i, cnt, false);
                           Write('Starting at [1] ');
                           press := ReadNumber(j,1, false);
                           j := j-1;
                           if j+i > cnt then begin
                                             WriteLn(Chr(7),'** Can''t read past ',cnt:1);
                                             goto 4;
                                             end;
                           end
                        else i := 1;
                        Write('Point at box where start reading.');
                        blks := 0;
                        if FindLittle(x,y) then
                           repeat
                           ReadBinLittle(x,y,fid,j, c='s');
                           blks := blks+1;
                           j := j+1;
                           if blks >= i then leave := true
                           else if not IncLittle(x,y) then
                              begin
                              WriteLn(BeepChar);
                              WriteLn('** Could only fit ',blks:1,' cursors.');
                              leave := true;
                              end;
                           until leave
                        else Write('  XXX');
                        end;
                4: end;
             'i' : If confirm('nit','n') then
                          begin
                          Init;
                          CursorCursor(true);
                          end;
             'j' : begin
                   Write('oin.  Point at little box to join with current');
                   if FindLittle(x,y) then JoinLittle(x,y, Black)
                   else Write(' XXX');
                   end;
             'J' : begin
                   Write('oin.  Point at little box to join with current');
                   if FindLittle(x,y) then 
                      begin
                      WriteLn;
                      Write('Function: (Xor, Or, And) [O] ');
                      ReadChar(c, 'o', 'o');
                      case c of
                        'x', 'X' : JoinLittle(x,y,toggle);
                        'o', 'O' : JoinLittle(x,y,Black);
                        'a', 'A' : JoinLittle(x,y,White);
                        otherwise: Write(' XXX');
                        end;
                      end
                   else Write(' XXX');
                   end;
             'l' : begin
                   Write('ine.  Press at end points');
                   MakeLine(yBase,xBase,447,511,mode,CursPt);
                   end;
             'L' : DoList;
             'm' : Begin
                   Write('ove in which direction? ');
                   ReadChar(c,Null,Null);
                   if c IN ['u','d','l','r'] then
                       begin
                       case c of 
                          'u' : dir := Up;
                          'd' : dir := Down;
                          'l' : dir := left;
                          'r' : dir := right;
                          end;
                       Write('.  How many places? [1] ');
                       press := ReadNumber(count,1, false);
                       MoveN(dir, count);
                       end
                   else Write(' XXX');
                   end;
             'M' : begin
                   Write('irror; right-left, up-down, or diag? ');
                   ReadChar(c,Null,Null);
                   if (c = 'r') or (c='u') or (c='d') then Mirror(c)
                   else Write(' XXX');
                   end;
             'o' : begin
                   write('utput on Text or Binary file? ');
                   ReadChar(c,'b',Null);
                   case c of
                      't' : begin
                            write('ext file name: ');
                            press := ReadString(fileName, false);
                            if not AddExtension(fileName, TextExt) then goto 1;
                            if POS(filename, ':') = 0 then
                                begin
                                fid := FSEnter(fileName);
                                if fid=0 then
                                       begin
                                       Write('** Sorry, cannot create file');
                                       goto 1;
                                       end;
                                end;
                            ReWrite(f,fileName);
                            DumpCursor(f);
                            Close(f);
                            Write('  ',fileName,' written.');
                          1:end;
                      'b' : begin
                            write('inary file name: ');
                            if GetOutFile(fid, i, BinaryExt, fileName) then
                                 begin
                                 DumpBinCursor(fid, i);
                                 Write('  ',fileName,' written.');
                                 end;
                            end;
                      otherwise : Write(' XXX');
                      end;
                   end;
             'O' : begin
                   leave := false;
                   j := 0;
                   write('utput on solar or animate file? ');
                   remOffCnt := 0;
                   ReadChar(c,'a','a');
                   if (c = 's') or (c = 'a') then
                     begin
                     if c = 's' then s := SolarExt
                     else s := SeriesExt;
                     Write('  Filename: ');
                     if GetOutFile(fid, i, s, fileName) then
                       if GetPointPair(x,y,x2,y2) then
                         begin
                         repeat
                           if c = 'a' then
                                DumpBinLittle(x,y,fid,i-1) {write block i-1}
                           else begin 
                                RememberOffset(x,y,i-1);
                                DumpBinLittle(x,y,fid,i); {write block i}
                                end;
                           i := i+1;
                           j := j+1;
                           if (x=x2) and (y=y2) then leave := true
                           else if not IncLittle(x,y) then
                                 WriteLn('** IMPOSSIBLE from IncLittle');
                           until leave;
                         if c = 'a' then FSClose(fid,i-1,4096)
                         else CloseUpSolar(fid, i);
                         IOLoadCursor(defaultCursor,0,0);
                         WriteLn('.  ',j:1,' cursors written onto ',fileName);
                         end;
                     end {OK Char}
                   else write('  XXX');
                   end;
             'q' : IF confirm('uit',Null) then DoQuit;
             'r' : begin
                   Write('ectangle: Press at corners');
                   MakeRect(yBase,xBase,447,511,mode,CursPt);
                   end;
             's' : begin
                   Write('pecify.  Point at box for output');
                   if FindLittle(x,y) then
                     begin
                     curLitX := x;
                     curLitY := y;
                     littleOffset[whichLittle(x, y)] := curOffset;
                     workBuf^ := Cursor;
                     RasterOp(RRpl, 56, 64, curLitX,curLitY, SScreenW,SScreenP,
                                            0, 0, 4, workBuf);
                     end
                   else write(' XXX');
                   end;
             't' : begin write('oggle'); mode := toggle; end;
             'T' : begin
                   Write('ext: ');
                   press := ReadString(fileName, false);
                   SCurOn;
                   end;
             'w' : begin write('hite'); mode := white; end;
             'v', 'V' : begin
                   Write('erify.  Point at little to compare with current.');
                   If FindLittle(x,y) then Verify(x,y)
                   else Write(' XXX');
                   end;
             'x' : FixScreen;
             '?' : DoHelp;
             otherwise : Write(' <illegal>');
             end; {case}
      99:  WriteLn;
           Write(prompt1, prompt2);
           end; {when have char}
      {check for cursor pressed over a square}
       press := DoPoint(x, y, true, press)
       end; {loop}
end;
   
   
{$ifc trans then}
Procedure ReadCmdLine;
   var s: String;
       broke: String[1];
   begin
   RemDelimiters(UsrCmdLine, ' /', broke);
   GetSymbol(UsrCmdLine,s,' /',broke);  {remove "Cursdesign"}
   RemDelimiters(UsrCmdLine,' /', broke);
   GetSymbol(UsrCmdLine,s,' /',broke);
   CnvUpper(s);
   if s = 'REPLAY' then 
      begin
      WriteLn('Replaying last session.');
      InitTranscript(CursDTranscript, TRUE);
      end
   else if s <> '' then begin
                        WriteLn('** Unknown parameter ',s);
                        exit(CursorDesign);
                        end
   else InitTranscript(CursDTranscript, FALSE);
   end;
{$endc}

begin

{$ifc trans then}
ReadCmdLine;
{$elsec}
IOCursorMode(trackCursor);
{$endc}

Init;
MainLoop;
end.



(****
  If TabSwitch then {IO variable which tells when pressed}
          begin
          IOReadTablet(tabx, taby);
          if (tabx < xBase) or (taby < yBase) or (tabx >= xBase+448) or (taby >= yBase+512) then IOBeep
          else if (not justDown) or (lastx <> (tabx-xBase) div 8) or 
                            (lasty <> (taby-yBase) div 8 ) then {new point}
                    begin
                    lasty := (taby-yBase) div 8;
                    lastx := (tabx-xBase) div 8;
                    justDown := true;
                    CursPt(lasty, lastx, mode);
                    end;
          end {pt exists}
  else justDown := false;  {no point}
***)

