module AllocDisk;
{--------------------------------------------------------------------------
{AllocDisk: Module to handle allocating and freeing of pages off disk
{Written by CMU-people
{Copyright (C) 1980 Three Rivers Computer Corporation
{
{
{Abstract:
{    Allocdisk allocates and deallocates disk pages.
{
{ The partition has some number of contiguous pages on it. 
{ The number of pages in a partition is specified
{ when the partition is created (using the Partition program).  
{ Segments can be created within a partition, e.g. segments may not
{ span partitions.

{ The entire disk can be thought of as a partition (the Root Partition)
{
{ A DiskInformationBlock (DiskInfoBlock or DIB) contains all the fixed
{ information about a disk, including its partion names, locations and sizes.
{ It also contains a table used to locate boot segments
{ 
{ A disk can be 'mounted' which means that its root partition is known
{ to the system as an entry in the DiskTable.
{ 
{ A Partition Information Block (PartInfoBlock or PIB) contains all of the
{ fixed information about a partition,
{ 
{ A partition can also be 'mounted', and this is usually done as part of
{ mounting the disk itself. Partitions mounted are entries in the PartTable
{   
{ Within a partition, segments are allocated as doubly linked lists of pages
{ 
{ The Free List of a segment is a doubly linked list of free pages
{
{ This module maintains this list, as well as the DeviceTable and PartTable
{ It contains procedures for mounting and dismounting disks and partitions, 
{ as well as allocating and deallocating space within a partition.

{ When allocating pages, the module updates the PartInfoBlock every
{ MaxAllocs calls on AllocDisk
{
{ Since the system may crash some time between updates, the pointers
{ and free list size may not be accurate.
{
{ When a partition is mounted, the pointers are checked to see if they
{ point to free pages.  If not, the head of the pointer is found by
{ looking at the "filler" word of the block the free head does point to
{ (which presumably was allocated after the last update of PartInfoBlock).
{ The filler word has a short pointer to the next "free" page, and forms
{ a linked list to the real free list header.  Likewise, if the Free tail
{ does not have a next pointer of 0, a deallocate is presumed to have been done
{ since a PartInfoBlock update, and NextAdr pointers are chased to find the
{ real end of the Free List.
{
{--------------------------------------------------------------------------}

{--------------------------------------------------------------------------
 Versions:
    V2.8   3-Dec-81  Brad Myers   Changed definition of FlushFail
    V2.7  15-Jul-81  John Strait  Fixed bug in DeallocChain: increment
                                  PartNumOps by the length of the deallocated
                                  chain.  Change order of operations in
                                  AllocDisk routine to reduce probability
                                  of creating an inconsistent free list.
    V2.6  23-Jun-81  Brad Myers  Changed UpdatePartInfo to only write PartInfo
                                  if changed so can boot from write-protected
                                  floppies.
    V2.5   1-Jun-81  Brad Myers  Made UpdatePartInfo more fail-safe.
    V2.4  12-May-81  Brad Myers  New exceptions; no Write statements.
                                 Changed addresses to print in unsigned decimal
    V2.3  20-Apr-81  John Strait Use DiskSegment for allocation.
    V2.2  14-Apr-81  Brad Myers  Changed to report "device name" instead of
                                  "root partition" in display partitions
                                 ForgetAll at end DeviceDismount
    V2.1  27-Mar-81  Brad Myers  Fixed bug in FindPartition for missing device
                                  and added comments
    V2.0  19-Mar-81  Brad Myers  Combined previous changes
    V1.3   ?-Mar-81   ??         Fixed GetPartName
    V1.2  10-Mar-81  Brad Myers  Bit32 => FSBit32; fixed for new FS syntax
    V1.0  ??-???-??  CMU people  written
--------------------------------------------------------------------------}

{ 
{******************} exports {***************************}

imports Arith from Arith;
imports ReadDisk from ReadDisk;

const
  MAXDISKS      =  2;  {Floppy and HardDisk}
  MAXPARTITIONS = 10;  {Maximum number of mounted partitions}
  MAXPARTCHARS  =  8;  {how many characters in a partition name}

type

  PartString   = string[MAXPARTCHARS];

  DeviceRecord = record                 {entry in the DeviceTable}
                    InfoBlk: DiskAddr;  {where the DiskInfoBlock is}
                    InUse  : boolean;   {this DeviceTable entry is valid}
                    RootPartition: PartString {name of this disk}
                 end;
  
  PartRecord   = record                      {entry in the PartTable}
                    PartHeadFree : DiskAddr; {pointer to Head of Free List}
                    PartTailFree : DiskAddr; {pointer to tail of Free List}
                    PartInfoBlk  : DiskAddr; {pointer to PartInfoBlock}
                    PartRootDir  : DiskAddr; {pointer to Root Directory}
                    PartNumOps   : integer;  {how many operations done since
                                              last update of PartInfoBlock}
                    PartNumFree  : FSBit32;  {HINT of how many free pages}
                    PartInUse    : boolean;  {this entry in PartTable is valid}
                    PartMounted  : boolean;  {this partition is mounted}
                    PartDevice   : integer;  {which disk this partition is in}
                    PartStart    : DiskAddr; {Disk Address of 1st page}
                    PartEnd      : DiskAddr; {Disk Address of last page}
                    PartKind     : PartitionType; {Root or Leaf}
                    PartName     : PartString {name of this partition}
                  end;


var
  DiskTable   : array [0..MAXDISKS-1]  of DeviceRecord;
  PartTable   : array [1..MAXPARTITIONS] of PartRecord;
  

procedure InitAlloc;  {initialize the AllocDisk module, called during boot}

procedure DeviceMount(disk: integer);  {mount a disk}

procedure DeviceDismount(disk : integer);  {dismount a disk}

function  MountPartition(name : string) : integer;  {mount a partion,
                                                     return PartTable index}

procedure DismountPartition(name : string);  {dismount a partition}

function  FindPartition(name : string) : integer;  {given a partion name, look
                                             for it in PartTable, return index}

function  AllocDisk(partition : integer) : DiskAddr;  {allocate a free page
                                                       from a partition}

procedure DeallocDisk(addr : DiskAddr);  {return a page to the free list}

procedure DeallocChain(firstaddr,lastaddr : DiskAddr; numblks : integer);
                                {return a bunch of pages to free list}
                                
function  WhichPartition(addr : DiskAddr) : integer;  {given a Disk Address,
                                        figure out which partition it is in}

procedure DisplayPartitions;  {print the PartTable}

Exception NoFreePartitions;
{-----------------------------------------------------
 Abstract:  Raised when too many partitions are accessed at one time.  The
             limit is MAXPARTITIONS. 
-------------------------------------------------------}

Exception BadPart(msg, partName: String);
{-----------------------------------------------------
 Abstract:  Raised when there is something wrong with a partition.  This
            means that the Scavenger should be run.
 Parameters: msg is the problem and partName is the partition name.  Print
             error message as: WriteLn('** ',msg,' for ',partName); 
-------------------------------------------------------}

Exception PartFull(partName: String);
{-----------------------------------------------------
 Abstract:  Raised when there are no free blocks in a partition to be
             allocated.  This means that some files should be deleted and then
             the Scavenger should be run.
 Parameters: partName is the full partition 
-------------------------------------------------------}

{******************} private {***************************}
