module Browser;

(***************************************************
 *      EXPORTED VALUES
 ***************************************************)

exports

imports PascalInit from PascalInit;
imports Sapph from Sapphuser;

procedure InitBrowser;

function CreateBrowserWindow( parentWindow: Window;
                              fixedPosition: Boolean; x, y: Integer;
                              fixedSize: Boolean; w, h: Integer;
                              title: TitStr;
                              var vp: Viewport ): Window;

procedure HandleBrowser( vp: Viewport );
    
(***************************************************
 *      PRIVATE VALUES
 ***************************************************)

private

imports AccInt from AccentUser;
imports Except from Except;
imports AccCall from AccCall;
imports SaphEmrServer from SaphEmrServer;
imports SaphEmrExceptions from SaphEmrExceptions;
imports Spice_String from Spice_String;
imports Colors from Colors;
  
  (***************************************************
   *    Definitions for emergency messages
   ***************************************************)

const
  maxMsgSize = 2048;  (* Maximum number of characters per message *)
  numMsgs = 10;       (* Number of messages available in buffer *)

type
  Space = Array[0..maxMsgSize div 2 - 1] of Integer;
  pDummyMsg = ^dummyMsg;
  dummyMsg = record
               case boolean of
                 true: ( head: Msg;
                         retType: TypeType;
                         retCode: Integer;
                         body: Space );
                 false: ( nextFreeMsg: pDummyMsg );
             end;

var
  pInMsg: Array[1..numMsgs] of pDummyMsg;     (* Pointer to received message *)
  pRepMsg: pDummyMsg;                         (* Pointer to returned message *)
  msgInUse: Array[1..numMsgs] of Boolean;
  dum: Boolean;

  (***************************************************
   *    Definitions for browser
   ***************************************************)

const
  (* Size parameters for browser shape *)
  barWidth = 30;
  barBorder = 5;
  buttonSize = barWidth - 2 * barBorder;

  (* Regions in browser *)
  BrowseRegion = 1;
  BarRegion = 2;
  GetPopupRegion = 3;
  PageForwardRegion = 4;
  PageBackwardRegion = 5;
  ScrollRegion = 6;

  debug = true;

procedure InitBrowser;
  var
    i: Integer;
  begin
    if debug then
      writeln( 'Initializing browser' );

    (***************************************************
     *          Initialize emergency messages
     ***************************************************)
    dum := true;
    if SoftInterrupt( KernelPort, false, dum ) = success then begin
      for i := 1 to numMsgs do begin
        new( pInMsg[i] );
        msgInUse[i] := false;
      end;
      new( pRepMsg );
    end;

    InitColors;         (* Initialize the gray scales *)
  end;

procedure blankRegion( vp: Viewport; regionNum: Integer; 
                       x, y, w, h: Integer );
  begin
    ModifyRegion( vp, regionNum, x, y, w, h );
    ColorRectangle( vp, clrWhite, x, y, w, h );
    VPLine( vp, DrawLine, x, y, x + w, y );
    VPLine( vp, DrawLine, x, y, x, y + h );
    VPLine( vp, DrawLine, x + w, y, x + w, y + h );
    VPLine( vp, DrawLine, x, y + h, x + w, y + h );
  end;

procedure drawBrowser( vp: Viewport );
  var
    w, h: Integer;
    dontCareInt: Integer;
    dontCareBool: Boolean;
  begin
    if debug then
      writeln( 'Drawing browser window' );

    (* Get the viewport's dimensions *)
    ViewportState( vp, dontCareInt, dontCareInt, w, h, dontCareInt,
                   dontCareBool, dontCareBool, dontCareBool );
    
    (* Bar Region: The entire left side bar *)
    ModifyRegion( vp, BarRegion, 0, 0, barWidth + 1, h );
    ColorRectangle( vp, clrGray, 0, 0, barWidth + 1, h );
    VPLine( vp, DrawLine, 0, 0, 0, h );
    VPLine( vp, DrawLine, barWidth + 1, 0, barWidth + 1, h );
    VPLine( vp, DrawLine, 0, h, barWidth + 1, h );

    (* Popup Menu Button Region: The icon that gives the popup menu *)
    blankRegion( vp, GetPopupRegion,
                 barBorder, barBorder, 
                 buttonSize, buttonSize );

    (* Page Backward Button Region: The area that pages backwards if clicked *)
    blankRegion( vp, PageBackwardRegion,
                 barBorder, barWidth,
                 buttonSize, buttonSize );
    VPLine( vp, DrawLine, 
            barBorder + buttonSize div 2, barWidth,
            barBorder, barWidth + buttonSize );
    VPLine( vp, DrawLine, 
            barBorder + buttonSize div 2, barWidth,
            barBorder + buttonSize, barWidth + buttonSize );

    (* Page Forward Button Region: The area that pages forward if clicked *)
    blankRegion( vp, PageForwardRegion,
                 barBorder, h - barWidth + barBorder,
                 buttonSize, buttonSize );
    VPLine( vp, DrawLine, 
            barBorder, h - barWidth + barBorder,
            barBorder + buttonSize div 2, h-barWidth + barBorder + buttonSize );
    VPLine( vp, DrawLine, 
            barBorder + buttonSize, h - barWidth + barBorder,
            barBorder + buttonSize div 2, h-barWidth + barBorder + buttonSize );

    (* Scroll Bar Region: The area in which the scroll bar resides *)
    blankRegion( vp, ScrollRegion,
                 barBorder, barWidth + buttonSize + 2,
                 buttonSize, h - 2*barWidth + barBorder - buttonSize - 4 );

    (* Menu Region: The place for all the actual information *)
    ModifyRegion( vp, BrowseRegion, barWidth, 0, w - barWidth, h );
  end;

function CreateBrowserWindow( parentWindow: Window;
                              fixedPosition: Boolean; x, y: Integer;
                              fixedSize: Boolean; w, h: Integer;
                              title: TitStr;
                              var vp: Viewport ): Window;
  var
    browserWindow: Window;
    progName: progStr;

  begin
    if debug then
      writeln( 'Creating browser window' );

    (* Define the entire window *)
    progName := 'browse';
    browserWindow := CreateWindow( parentWindow,
                                   fixedPosition, x, y,
                                   fixedSize, w, h,
                                   true, true,
                                   title, progName, true, vp );

    (* Define the regions *)
    PushRegion( vp, BarRegion, 0, 0, 0, 0 );
    PushRegion( vp, BrowseRegion, 0, 0, 0, 0 );
    PushRegion( vp, GetPopupRegion, 0, 0, 0, 0 );
    PushRegion( vp, PageBackwardRegion, 0, 0, 0, 0 );
    PushRegion( vp, PageForwardRegion, 0, 0, 0, 0 );
    PushRegion( vp, ScrollRegion, 0, 0, 0, 0 );

    drawBrowser( vp );          (* Actually draw the regions mentioned *)

    EnableNotifyExceptions( vp, DataPort, true, true );

    CreateBrowserWindow := browserWindow;
  end;

procedure HandleBrowser( vp: Viewport );
  
  label 1;

  exception EmergMsgErr( message: String );
  handler EmergMsgErr( message: String );
    begin
      writeln( '** ERROR in message handling: ', message );
      goto 1;
    end;

  procedure handleAllEmergMsgs;
    var
      i: Integer;
    begin
      for i := 1 to numMsgs do
        if msgInUse[i] then begin
          if not SaphEmrServer( pInMsg[i], pRepMsg ) then
            raise EmergMsgErr( concat( 'Unknown emerg. msg.: ',
                                       CVLS(pInMsg[i]^.head.id,1,10,' ') ) );
          msgInUse[i] := false;
        end;
    end;

  handler EmergMsg;
    var
      gr: GeneralReturn;
      num, i: Integer;
    begin
      if debug then
        writeln( 'Emergency message received' );
      (* Find an empty message to use *)
      num := -1;
      for i := 1 to numMsgs do
        if not msgInUse[i] then num := i;
      if num = -1 then
        raise EmergMsgErr( 'All messages in use.' )
      else begin
        if debug then
          writeln( 'Using emergency message ', i );

        (* Reserve it *)
        pInMsg[num]^.head.msgSize := MaxMsgSize;
        gr := Receive( pInMsg[num]^.head, 0, AllPts, ReceiveIt );
        if gr <> success then
          raise EmergMsgErr( 'Lost the emerg. msg.' )
        else
          msgInUse[num] := true;
      end;

      dum := true;
      handleAllEmergMsgs;
      if SoftInterrupt( KernelPort, false, dum ) <> success then
        raise EmergMsgErr( 'Failed' );
    end;

  handler EViewPtChanged( vpt: Viewport; x, y, w, h, r: Integer );
    begin
      if debug then
        writeln( 'Viewport changed' );
      if r = 1 then
        drawBrowser( vpt );
    end;

  handler EViewPtExposed( vp1: Viewport; ra: pRectArray; numRects: Long );
    var
      i, j: Integer;
    begin
      if debug then
        writeln( 'Viewport exposed' );
      drawBrowser( vp1 );
      if InvalidateMemory( KernelPort, recast( ra, VirtualAddress ),
                           wordsize(ra) * 2 ) <> Success then
        exit(HandleBrowser);
    end;

  (***************************************************
   *    Deal with the browser interface
   ***************************************************)

  var
    i: Integer;
    keyEv: KeyEvent;

  begin
    if debug then
      writeln( 'Entering browser handler' );
    1:
    keyEv := GetEvent( vp, KeyWaitEvent );
    if debug then
      writeln( 'Exiting browser handler' );
  end.
