MODULE Tek4012 ;
{ provides simulation of Tektronics 4012 terminal which is 
  equivalently the Tektronics 4051 terminal in TERMINAL mode. }
{ COPYRIGHT (C) 1982 SIEMENS CRT }

{ HISTORY
{ 
{ When: 13 May 82
{ Who: Bruce Ladendorf
{ What: Created
{ }

EXPORTS

PROCEDURE Init4012 (trx, try, blx, bly : INTEGER; drawBorder : BOOLEAN);
{ inits, must be called before other routines in this module }

PROCEDURE Simulate4012 ;

PROCEDURE Finish4012 ;

PRIVATE
IMPORTS IO FROM IO;
IMPORTS IOErrors FROM IOErrors;
IMPORTS RS232Baud FROM RS232Baud;
IMPORTS Screen FROM Screen;
IMPORTS System FROM System;

CONST
    BLACKBORDERWIDTH = 1;
    WHITEBORDERWIDTH = 1;
    HALFBORDERWIDTH = BLACKBORDERWIDTH + WHITEBORDERWIDTH;
    TOTBORDERWIDTH = 2 * (BLACKBORDERWIDTH + WHITEBORDERWIDTH);

VAR
    BadInit : BOOLEAN;
    XTranslate : array [32..63, 64..95] of integer;
    YTranslate : array [32..63, 96..127] of integer;
    EraseX, EraseY, EraseWidth, EraseHeight : integer;


PROCEDURE Init4012 (trx, try, blx, bly : INTEGER; drawBorder : BOOLEAN);
VAR
    BadParameters : BOOLEAN;
    i, j, bits10, temp : INTEGER;
BEGIN
    { if bogus parameters then set badinit flag to TRUE and return }
    BadParameters := FALSE;
    if ((trx < 0) or (trx > 767)) then BadParameters := TRUE;
    if ((blx < 0) or (blx > 767)) then BadParameters := TRUE;
    if ((try < 0) or (try > 1023)) then BadParameters := TRUE;
    if ((bly < 0) or (bly > 1023)) then BadParameters := TRUE;
    
    if (((trx > blx) or (try > bly)) and (not drawborder)) then
        BadParameters := TRUE;
    if ((trx > blx - TOTBORDERWIDTH-1) or (try > bly - TOTBORDERWIDTH-1) and
      drawborder) then
        BadParameters := TRUE;

    if (BadParameters) then BEGIN
            BadInit := TRUE;
            exit (Init4012);
    END
    else BadInit := FALSE;
    
    { setup translation tables for tektronics to screen tranlation }
    for i := 32 to 63 do BEGIN
        for j := 0 to 31 do BEGIN
            bits10 := SHIFT (LAND (i, 31), 5) + j;
            temp := TRUNC ((bits10 / 780.0) * (bly-try-TOTBORDERWIDTH));
            if (temp >= (bly-try-TOTBORDERWIDTH)) then 
                temp := bly-try-TOTBORDERWIDTH-1;
            YTranslate[i, j+96] := bly - HALFBORDERWIDTH - temp;
            temp := TRUNC ((bits10 / 1023.0) * (blx-trx-TOTBORDERWIDTH));
            if (temp >= (blx-trx-TOTBORDERWIDTH)) then
                temp := blx-trx-TOTBORDERWIDTH-1;
            XTranslate[i, j+64] := temp + trx + HALFBORDERWIDTH;
        END;
    END;

    { blank out the display area }
    RASTEROP (RXor, bly-try+1, blx-trx+1,
        trx, try, SScreenW, SScreenP,
        trx, try, SScreenW, SScreenP);
    
    { set up erase variables to indicate where to erase during PAGE }
    EraseX := trx+HALFBORDERWIDTH;
    EraseY := try+HALFBORDERWIDTH;
    EraseWidth := blx-trx+1-TOTBORDERWIDTH;
    EraseHeight := bly-try+1-TOTBORDERWIDTH;

    { draw border if desired }
    if (DrawBorder) then BEGIN
        RASTEROP (RXNor, blx-trx+1, BLACKBORDERWIDTH,
            trx, try, SScreenW, SScreenP,
            trx, try, SScreenW, SScreenP);

        RASTEROP (RXNor, BLACKBORDERWIDTH, bly-try+1,
            trx, try, SScreenW, SScreenP,
            trx, try, SScreenW, SScreenP);

        RASTEROP (RXNor, blx-trx+1, BLACKBORDERWIDTH,
            trx, bly-BLACKBORDERWIDTH+1, SScreenW, SScreenP,
            trx, bly-BLACKBORDERWIDTH+1, SScreenW, SScreenP);

        RASTEROP (RXNor, BLACKBORDERWIDTH, bly-try+1,
            blx-BLACKBORDERWIDTH+1, try, SScreenW, SScreenP,
            blx-BLACKBORDERWIDTH+1, try, SScreenW, SScreenP);
    END;
END;


CONST
    MAXBACKUP = 1;
TYPE
    TekChar = 0..127;
VAR
    NumCharsBackedUp : integer;
    BackupArray : array [1..MAXBACKUP] of TekChar;

PROCEDURE InitRS ;
VAR
    stat : DevStatusBlock;
BEGIN
    NumCharsBackedUp := 0;
    
    { rs initialization, copied from li4.pas }
    with stat do BEGIN
        ByteCnt := 1;
        RSRcvEnable := TRUE;
    END;
    IOPutStatus (RS232In, Stat);
    SetBaud ('4800', TRUE);
END;

FUNCTION NextChar : TekChar ;
CONST
    VERBOSE = TRUE;
VAR
    answer : TekChar;
    ch : char;
    foundone : BOOLEAN;
BEGIN
    if (NumCharsBackedUp > 0) then BEGIN
        answer := BackupArray[NumCharsBackedUp];
        NumCharsBackedUp := NumCharsBackedUp - 1;
    END
    else BEGIN
        foundone := FALSE;
        while (not foundone) do BEGIN
            while (IOEIOC <> IOCRead (RS232In, ch)) do ;
            answer := LAND (ord (ch), 127);
            if (answer <> 22) then foundone := TRUE; { skip SYN chars }
        END; {while}
    END;
    NextChar := answer;
    {$IFC VERBOSE THEN}
        writeln ('Trace, NextChar, char was ', answer:3);
    {$ENDC}
END;

PROCEDURE PushBack (ch : TekChar);
CONST
    VERBOSE = TRUE;
BEGIN
    if (NumCharsBackedUp < MAXBACKUP) then BEGIN
        NumCharsBackedUp := NumCharsBackedUp + 1;
        BackupArray[NumCharsBackedUp] := ch;
    END
    else writeln ('Error, PushBack');
    {$IFC VERBOSE THEN}
        writeln ('Trace, PushBack, pushed ', ch:3);
    {$ENDC}
END;

TYPE
    stateType = (NEWGRAPH, VECTOR, INVISO, ERROR, STARTERROR, ESCAPE);
    charPos = record
        YHigh, YLow, XHigh, XLow : integer;
    end;


FUNCTION IsHigh (ch : TekChar) : BOOLEAN;
BEGIN
    if ((32 <= ch) and (ch <= 63)) then IsHigh := TRUE
    else IsHigh := FALSE;
END;

FUNCTION IsYLow (ch : TekChar) : BOOLEAN;
BEGIN
    if ((96 <= ch) and (ch <= 127)) then IsYLow := TRUE
    else IsYLow := FALSE;
END;

FUNCTION IsXLow (ch : TekChar) : BOOLEAN;
BEGIN
    if ((64 <= ch) and (ch <= 95)) then IsXLow := TRUE
    else IsXLow := FALSE;
END;


FUNCTION GetAddress (pos : charPos; VAR iserror : BOOLEAN) : charPos;
{ gets a new cursor position, for use in VECTOR and INVISO states,
  returns st = FALSE if an error is detected }
CONST
    VERBOSE = TRUE;
TYPE
    st = (START, YHIGH, YLOW, XHIGH, XLOW, ERROR);
VAR
    state : st;
    ch : TekChar;
    outpos : charPos;
BEGIN
    state := START;
    outpos := pos;
    while TRUE do BEGIN
        {$IFC VERBOSE THEN}
            write ('Trace, GetAddress, state=');
            case state of
              START: write ('START');
              YHIGH: write ('YHIGH');
              YLOW:  write ('YLOW');
              XHIGH: write ('XHIGH');
              XLOW:  write ('XLOW');
              ERROR: write ('ERROR');
            end; {case}
            writeln ;
        {$ENDC}
        case state of
          START:
            BEGIN
                ch := NextChar ;
                if (IsHigh (ch)) then begin
                    outpos.YHigh := ch;
                    state := YHIGH;
                end
                else if (IsYLow (ch)) then begin
                    outpos.YLow := ch;
                    state := YLOW;
                end
                else if (IsXLow (ch)) then begin
                    outpos.XLow := ch;
                    state := XLOW;
                end
                else begin
                    pushback (ch);
                    state := ERROR;
                end;
            END;
          YHIGH:
            BEGIN
                ch := NextChar ;
                if (IsHigh (ch)) then begin
                    outpos.XHigh := ch;
                    state := XHIGH;
                end
                else if (IsYLow (ch)) then begin
                    outpos.YLow := ch;
                    state := YLOW;
                end
                else if (IsXLow (ch)) then begin
                    outpos.XLow := ch;
                    state := XLOW;
                end
                else begin
                    pushback (ch);
                    state := ERROR;
                end;
            END;
          YLOW:
            BEGIN
                ch := NextChar ;
                if (IsHigh (ch)) then begin
                    outpos.XHigh := ch;
                    state := XHIGH;
                end
                else if (IsXLow (ch)) then begin
                    outpos.XLow := ch;
                    state := XLOW;
                end
                else begin
                    pushback (ch);
                    state := ERROR;
                end;
            END;
          XHIGH:
            BEGIN
                ch := NextChar ;
                if (IsXLow (ch)) then begin
                    outpos.XLow := ch;
                    state := XLOW;
                end
                else begin
                    pushback (ch);
                    state := ERROR;
                end;
            END;
          XLOW:
            BEGIN
                GetAddress := outpos;
                iserror := FALSE; { no error found }
                exit (GetAddress);
            END;
          ERROR:
            BEGIN
                GetAddress := pos;
                iserror := TRUE; { error found }
                exit (GetAddress);
            END;
          otherwise:
            writeln ('Error, GetAddress');
        end; {case}
    END;
END;

PROCEDURE Simulate4012 ;
LABEL
    1111 ;
CONST
    FF = 12;
    SYN = 22;
    ESC = 27;
    GS = 29;
    US = 31;

    VERBOSE = TRUE;
TYPE
    modeType = (ALPHA, GRAPH);
VAR
    mode : modeType;
    done, failed : BOOLEAN;
    ch : TekChar;
    state : stateType;
    curpos, newpos : charPos;
BEGIN
    { stop if bad init }
    if (BadInit) then BEGIN
        writeln ('Error, Simulate4012, Uninitialized');
        exit (Simulate4012);
    END;

    { inits }
    mode := ALPHA;
    InitRS ;

    done := FALSE;
    WHILE (not done) DO BEGIN
        {$IFC VERBOSE THEN}
            write ('Trace, Simulate4012, mode=');
            case mode of
              ALPHA: write ('ALPHA');
              GRAPH: write ('GRAPH');
            end;
            write (' state=');
            case state of
              NEWGRAPH: write ('NEWGRAPH');
              VECTOR: write ('VECTOR');
              INVISO: write ('INVISO');
              ERROR: write ('ERROR');
              STARTERROR: write ('STARTERROR');
              ESCAPE: write ('ESCAPE');
            end;
            writeln ;
        {$ENDC}
        if (mode = ALPHA) then BEGIN
            ch := NextChar ;
            if (ch = GS) then BEGIN
                mode := GRAPH;
                state := NEWGRAPH;
            END;
        END
        else if ((mode = GRAPH) and (state = NEWGRAPH)) then BEGIN
            ch := NextChar ;
            if (not IsHigh (ch)) then BEGIN
                PushBackChar (ch);
                state := STARTERROR;
                goto 1111;
            END;
            newpos.YHigh := ch;
            ch := NextChar ;
            if (not IsYLow (ch)) then BEGIN
                PushBackChar (ch);
                state := STARTERROR;
                goto 1111;
            END;
            newpos.YLow := ch;
            ch := NextChar ;
            if (not IsHigh (ch)) then BEGIN
                PushBackChar (ch);
                state := STARTERROR;
                goto 1111;
            END;
            newpos.XHigh := ch;
            ch := NextChar ;
            if (not IsXLow (ch)) then BEGIN
                PushBackChar (ch);
                state := STARTERROR;
                goto 1111;
            END;
            newpos.XLow := ch;
            curpos := newpos;
            STATE := VECTOR;
        END                         
        else if ((mode = GRAPH) and (state = VECTOR)) then BEGIN
            failed := TRUE;
            newpos := GetAddress (curpos, failed);
            if (not failed) then BEGIN
                { draw vector }
                {$IFC VERBOSE THEN}
                    writeln ('Drawing vector from ',
                        XTranslate[curpos.XHigh, curpos.XLow]:3, ' ',
                        YTranslate[curpos.YHigh, curpos.YLow]:3, ' to ',
                        XTranslate[newpos.XHigh, newpos.XLow]:3, ' ',
                        YTranslate[newpos.YHigh, newpos.YLow]:3);
                {$ENDC}
                Line (DrawLine, XTranslate[curpos.XHigh, curpos.XLow],
                    YTranslate[curpos.YHigh, curpos.YLow],
                    XTranslate[newpos.XHigh, newpos.XLow],
                    YTranslate[newpos.YHigh, newpos.YLow], SScreenP);
                { put in missing endpoints }
                     { NOT IMPLEMENTED }
                curpos := newpos;
            END
            else BEGIN
                ch := NextChar ;
                if (ch = ESC) then state := ESCAPE
                else if (ch = US) then mode := ALPHA
                else if (ch = GS) then state := INVISO
                else BEGIN
                    PushBack (ch);
                    state := ERROR;
                END;
            END;
        END
        else if ((mode = GRAPH) and (state = INVISO)) then BEGIN
            failed := TRUE;
            newpos := GetAddress (curpos, failed);
            if (not failed) then BEGIN
                state := VECTOR;
                curpos := newpos;
            END
            else BEGIN
                ch := NextChar ;
                if (ch = ESC) then state := ESCAPE
                else if (ch = US) then mode := ALPHA
                else BEGIN
                    PushBack (ch);
                    state := ERROR;
                END;
            END
        END
        else if ((mode = GRAPH) and (state = ERROR)) then BEGIN
            ch := NextChar ;
            if ((ch = GS) or (IsXLow (ch))) then state := INVISO;
            {$IFC TRUE THEN} write ('E'); {$ENDC}
        END
        else if ((mode = GRAPH) and (state = STARTERROR)) then BEGIN
            ch := NextChar ;
            if ((ch = GS) or (IsXLow (ch))) then state := NEWGRAPH;
            {$IFC TRUE THEN} write ('S'); {$ENDC}
        END
        else if ((mode = GRAPH) and (state = ESCAPE)) then BEGIN
            ch := NextChar ;
            if (ch = FF) then BEGIN
                { clear screen and return to alpha mode }
                RASTEROP (RXor, EraseWidth, EraseHeight,
                    EraseX, EraseY, SScreenW, SScreenP,
                    EraseX, EraseY, SScreenW, SScreenP);
                mode := ALPHA;
            END
            else BEGIN
                PushBack (ch);
                state := ERROR;
            END;
        END
        else writeln ('Error, Simulate4012, bad state');
      1111 : ;
    END; {while}

END;

PROCEDURE Finish4012 ;
{ currently a noop }
BEGIN
END.
{ }
