module PaneTraverse;

(***********************************************************************
 *
 *    AUTHOR: Lee Iverson
 *    ABSTRACT:
 *      This module implements certain routines to move along the borders
 *      of the current pane structure.
 *
 *    -----------
 *    CHANGE LOG:
 *      Jul 02 1984     Lee Iverson     ** Created **
 *
 ***********************************************************************)

exports

imports PaneGlobs from PaneGlobs;

type
  Border = (bordTop, bordBottom, bordLeft, bordRight);
  Direction = (dirUp, dirDown, dirLeft, dirRight);

function
  FindAbove(
    pn: pPane;
    x: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Find the pane directly above the given pane along the given
  *     X coordinate.
  *
  ***********************************************************************)

function
  FindBelow(
    pn: pPane;
    x: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Find the pane directly below the given pane along the given
  *     X coordinate.
  *
  ***********************************************************************)

function
  FindLeft(
    pn: pPane;
    y: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Find the pane directly to the left of the given pane along the
  *     given Y coordinate.
  *
  ***********************************************************************)

function
  FindRight(
    pn: pPane;
    y: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Find the pane directly to the right of the given pane along the
  *     given Y coordinate.
  *
  ***********************************************************************)

function
  FolBotTo(
    pn: pPane;
    x: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Follow the bottom border of the given pane until a pane is found
  *     which contains the given X coordinate.
  *
  ***********************************************************************)

function
  FolLeftTo(
    pn: pPane;
    y: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Follow the left border of the given pane until a pane is found
  *     which contains the given Y coordinate.
  *
  ***********************************************************************)

function
  FolRightTo(
    pn: pPane;
    y: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Follow the right border of the given pane until a pane is found
  *     which contains the given Y coordinate.
  *
  ***********************************************************************)

function
  FolTopTo(
    pn: pPane;
    x: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Follow the top border of the given pane until a pane is found
  *     which contains the given X coordinate.
  *
  ***********************************************************************)

function
  GetCorner(
    pn: pPane;
    bord: Border;
    dir: Direction;
    endVal: Integer
  ): Integer;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *       Find the coordinate (x or y) of the corner in the direction given
  *     along the border given, which allows one to move perpendicularly
  *     and reach the end value.  Returns -1 on an error.
  *       Thus, if we are given bordLeft and dirUp, we will move successively
  *     up the left border of the starting pane, until we find a horizon-
  *     tal border crossing this one that goes as far as the end value to
  *     whichever of the left or right is appropriate.
  *
  ***********************************************************************)

function
  GetEnd(
    pn: pPane;
    dir: Direction;
    bord: Border
  ): Integer;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Determine the x or y value of the end of the specifeid border of
  *     the specified pane when moving in the given direction.
  *
  ***********************************************************************)

function
  GetEndPane(
    pn: pPane;
    dir: Direction;
    bord: Border
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Determine the last pane along the specified border in the given
  *     direction.
  *
  ***********************************************************************)

function
  PaneWhichContains(
    x: Integer;
    y: Integer
  ): pPane;

 (***********************************************************************
  *
  *   AUTHOR: Lee Iverson
  *   ABSTRACT:
  *     Determines which pane contains the point defined by the coordinates
  *     (x,y).  It does a linear search through all panes.
  *
  ***********************************************************************)

private

{$include PaneTraverse.impl}.
