	PAGE	,132
	TITLE	MENUDIAG - Menu Driven Diagnostics

;**********************************************************************
;	   R E S T R I C T E D   R I G H T S   L  E G E N D 
;**********************************************************************

;**********************************************************************
;	Use, duplication or disclosure by the Government is
; subject to restrictions as set forth in paragraph (b) (3) (B)
; of the Rights in Technical Data and Computer Software clause
; in DAR 7-104.9(a). Contractor/manufacturer is Zenith Data
; Systems of Hilltop Road, St. Joseph, Michigan  49085.
;**********************************************************************



;**********************************************************************
;
;			-------------------------
;----------------------- MENU DRIVEN DIAGNOSTICS ----------------------
;			-------------------------
;
;		Copyright (C), Zenith Data Systems, Inc., 1983
;
;**********************************************************************

;**********************************************************************
;	The menu driven diagnostics have to be invoked by the user.
; It provides tests that can be repeated until it is stopped by the
; user. Hence time dependent errors have a greater probability of
; being caught. The menu allows the user to select tests that can
; test the keyboard, repeatedly read the disk, repeatedly perform
; the same tests done at power-up time and perform an extensive test
; on the memory. Since the usage of variables has to be avoided,
; the registers DX & BP are used as dedicated registers. DX is used
; as current cursor position & BP as the current test count. Also
; all DIAG_??? procedures expect DX to have the current cursor po-
; sition apart from other required parameters.
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC 

	INCLUDE	../ROM/ROM.LIT
	INCLUDE	../ROM/ROM.EXT
	INCLUDE	../ROM/INTR.LIT
	INCLUDE	../ROM/IO.LIT
	INCLUDE	../SYS/SYS.EXT
	INCLUDE	../VIDEO/VIDEO.LIT
	INCLUDE	../VIDEO/VIDEO.EXT
	INCLUDE	../KEYBOARD/KEYBOARD.LIT
	INCLUDE MENUDIAG.LIT


PAGE

	ASSUME	CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES:NOTHING

	EXTRN	DIAG_CONSOLE_OUTPUT:NEAR,DIAG_DISPLAY_CHAR:NEAR
	EXTRN	DIAG_DISP_STRING:NEAR, DIAG_CLEAR_SCREEN:NEAR, SET_SCROLL_MODE:NEAR
	EXTRN	DIAG_DISP_DEC_WORD:NEAR, DIAG_DISP_HEX_WORD:NEAR
	EXTRN	TEST_BREAK:NEAR, WAIT_BREAK:NEAR, ROM_START:NEAR
	EXTRN	SET_VIDEO_MODE:NEAR, GET_KEY_BUFF:NEAR, GET_KEY_STATUS:NEAR
	EXTRN	BEEP:NEAR, CLICK:NEAR, DISK_BOOT:NEAR, TIMER_INTERRUPT_TEST:NEAR
	EXTRN	BAD_TINTR_MSG:NEAR, BAD_PROCESSOR:NEAR, BAD_ROM:NEAR
	EXTRN	BAD_RAM:NEAR, DISPLAY_ERROR_MSG:NEAR
	EXTRN	INIT_INTERRUPTS:NEAR, POLL_EXTERNAL_ROMS:NEAR

	IF	NOT MORROW
	EXTRN	BAD_PARITY:NEAR
	ENDIF

	EXTRN	MAIN_MENU:BYTE, DATA_SEGMENT:WORD

PAGE

;**********************************************************************
; TEST_COMMAND:
;
;	This routine displays the main menu and waits for user
; input. It also checks the user input and branches to the re-
; levant routine. The register DX is going to be dedicated to
; holding the current cursor position.
;
;	NOTE: All DIAG_??? routines update the register DX as
;	if it is current cursor position. This is inherently
;	assumed in these routines. Also all the DIAG_??? rou-
;	tines assume the 80 x 25 Color mode. Hence these co-
;	lumn & row numbers are hard coded so that RAM varia-
;	bles are not used. BP is used as current test count.
;	ES is used as the current screen segment pointer.
;
; Input:
;	DS points to Monitor Data Segment
;
; Output:
;	Menu displayed and user input processed
;**********************************************************************
TEST_COMMAND PROC NEAR
	PUBLIC	TEST_COMMAND
	PUSHREG	<AX,BX,CX,DX,SI,BP,ES>
	MOV	CL,VIDEO_MODE		;Get the current video mode in CL
	MOV	CH,SCROLL_MODE		;...and the current scroll mode in CH
	MOV	AL,2			;Get mode for 80x25 display configuration
	CALL	SET_VIDEO_MODE		;Enter 80x25 display mode
	MOV	AL,0			;Now, set code to software-scrolling
	CALL	SET_SCROLL_MODE		;Set software scrolling for diagnostics
	MOV	BX,SCREEN_SEGMENT	;Point BX to the color card segment
	MOV	AL,BYTE PTR IO_CONFIG	;Get the current I/O configuration
	AND	AL,VIDEO_MODE_MASK	;Mask out the video mode select bits
	CMP	AL,VIDEO_MODE_MONO	;Is the monochrome card selected?
	JNE	MD_0			;No, have the correct video segment
	MOV	BX,MONO_SEGMENT		;Yes - point to the monochrome card
MD_0:	MOV	ES,BX			;Place the video segment in ES
MD_1:	CALL	DIAG_CLEAR_SCREEN	;Clear the screen
	MOV	DX,0			;Home cursor just in case
	MOV	BP,DX			;Zero test count
	MOV	SI,OFFSET CS:MAIN_MENU	;CS:SI points to main menu
	CALL	DIAG_DISP_STRING	;Display menu (DX updated)
MD_2:	CALL	GET_KEY_BUFF		;Get a key from the keyboard buffer
	CMP	AL,'1'			;Check lower bound
	JNC	MD_3			;If above bound carry on
	CALL	BEEP			;Signal error
	JMP	MD_2			;Get another key
MD_3:	CMP	AL,'6'			;Check upper bound of selection
	JC	MD_4			;If valid character carry on
	CALL	BEEP			;Signal error
	JMP	MD_2			;Get another key
MD_4:	CALL	DIAG_CONSOLE_OUTPUT	;Display character (AL,DX = Char,Pos)
	CMP	AL,'5'			;Check for user exit
	JZ	MD_5			;If so exit routine
	SUB	AL,'1'			;Convert to menu offset
	MOV	BH,0			;Zero MS Byte
	MOV	BL,AL			;BX has selection in menu
	SHL	BX,1			;Convert to word offset
	CALL	CS:MENU_PROCEDURES[BX]	;Jump to relevant routine
	JMP	MD_1			;Repeat entire process
MD_5:	MOV	AL,CL			;Get the old video mode
	CALL	SET_VIDEO_MODE		;Return to the old viddeo configuration
	MOV	AL,CH			;Get the old scroll mode
	CALL	SET_SCROLL_MODE		;Restore the old scroll mode
	POPREG	<ES,BP,SI,DX,CX,BX,AX>
	RET		
TEST_COMMAND ENDP


PAGE

;**********************************************************************
; KEYBOARD_TEST:
;
;	This routine performs the keyboard test. The screen is filled
; with any character that is typed. The cursor position DX is preserved
; because the screen is going to be cleared anyway.
;
; Input:
;	None
;
; Output:
;	None
;
; NOTE:
;	ES is used as a screen segment pointer
;**********************************************************************
KEYBOARD_TEST PROC NEAR
	PUBLIC	KEYBOARD_TEST
	PUSHREG	<AX,BX,CX,DX,SI>
	CALL	DIAG_CLEAR_SCREEN	;Clear the screen & home cur. (DX=0)
	MOV	SI,CS:CURRENT_TEST_MESSAGE[2];Get keyboard message offset
	CALL	DIAG_DISP_STRING	;Display message (DX updated)
	CALL	DISPLAY_EXIT_MESSAGE	;Display exit message
	MOV	BX,80*(25-2)		;Number of characters to be displayed
KT_1:	MOV	CX,BX			;Get count
	CALL	GET_KEY_BUFF		;Get key from buffer
	CMP	AX,BREAK_CODE		;Is it an exit test key?
	JZ	KT_3			;If so exit routine
	CMP	AX,ESC_CODE		;Is it an exit test key?
	JZ	KT_3			;If so exit routine
	MOV	DX,0			;Home the cursor
	MOV	SI,CS:CURRENT_TEST_MESSAGE[2];Get keyboard message offset
	CALL	DIAG_DISP_STRING	;Display the test name
	MOV	SI,OFFSET CS:CHAR_CODE_MESSAGE;Get pointer to message
	CALL	DIAG_DISP_STRING	;Display the character code message
	PUSH	AX			;Save character code
	CALL	DIAG_DISP_HEX_WORD	;Display character code
	MOV	AL,'H'			;Character to be displayed
	CALL	DIAG_DISPLAY_CHAR	;The number displayed is Hex
	POP	AX			;Retrieve character code
	MOV	DH,1			;Row 1
	MOV	DL,0			;Column 0
KT_2:	CALL	DIAG_DISPLAY_CHAR	;Display character (DX is updated)
	LOOP	KT_2			;Display till end of count
	CALL	DISPLAY_EXIT_MESSAGE	;Display exit message
	JMP	KT_1			;Repeat entire process	
KT_3:	POPREG	<SI,DX,CX,BX,AX>
KEYBOARD_TEST ENDP


PAGE

;**********************************************************************
; DISPLAY_EXIT_MESSAGE:
;
;	Displays the message informing user how to exit the test.
; The routine positions the cursor in the last row and displays
; the message.
;
; Input:
;	None
;
; Output:
;	Exit message displayed on last line
;
; NOTE:
;	ES is screen segment pointer
;**********************************************************************
DISPLAY_EXIT_MESSAGE PROC NEAR
	PUBLIC	DISPLAY_EXIT_MESSAGE
	PUSHREG	<DX,SI>
	MOV	DL,0			;Column number 0
	MOV	DH,24			;Last row
	MOV	SI,OFFSET CS:EXIT_MESSAGE;CS:SI points to message
	CALL	DIAG_DISP_STRING	;Display message (ES points to screen)
	POPREG	<SI,DX>
	RET
DISPLAY_EXIT_MESSAGE ENDP


PAGE

;**********************************************************************
; DISPLAY_TEST_MESSAGE:
;
;	This is the second screen that comes up for all tests. The
; display consists of the name of the test, the number of passes &
; the method of aborting the test.
;
; Input:
;	BX has index for name of test
;
; Output:
; 	Test screen displayed
;
; NOTE:
;	ES is a pointer to the screen segment
;**********************************************************************
DISPLAY_TEST_MESSAGE PROC NEAR
	PUBLIC	DISPLAY_TEST_MESSAGE
	PUSHREG	<BX,DX,SI>
	MOV	DL,32			;Column number 32 (assume 80 x 25)
	MOV	DH,10			;Row number 10
	MOV	SI,OFFSET CS:TEST_MESSAGE;CS:SI points to message
	CALL	DIAG_DISP_STRING	;Display test message
	MOV	DL,0			;Column 0
	MOV	DH,24			;Last row
	MOV	SI,OFFSET CS:ABORT_MESSAGE;Pointer to  message
	CALL	DIAG_DISP_STRING	;Display abort message
	MOV	DX,0			;Top of diplayed page
	SHL	BX,1			;Convert to word offset
	MOV	SI,CS:CURRENT_TEST_MESSAGE[BX];CS:SI points to message
	CALL	DIAG_DISP_STRING	;Display current test name
	POPREG	<SI,DX,BX>
	RET
DISPLAY_TEST_MESSAGE ENDP


PAGE

;**********************************************************************
; UPDATE_PASS_COUNT:
;
;	Increments test count and displays it. A hard coded cursor
; position is used because no variables can be used and 80 x 25
; Color display is assumed anyway.
;
; Input:
;	BP = Current count.
;
; Output:
;	Increments BP and displays it
;
; NOTE:
;	ES is a screen segment pointer
;**********************************************************************
UPDATE_PASS_COUNT PROC NEAR
	PUBLIC	UPDATE_PASS_COUNT
	PUSHREG	<AX,CX,DX>
	MOV	DL,45			;Column number
	MOV	DH,10			;Row number
	INC	BP			;Update counter
	MOV	AX,BP			;Get updated count
	CALL	DIAG_DISP_DEC_WORD	;Display count in decimal
	MOV	CX,5			;Count for display
	MOV	AL,' '			;Character to be displayed
UPC_1:	CALL	DIAG_DISPLAY_CHAR	;Display character to wipe ...
	LOOP	UPC_1			;out extra characters
	POPREG	<DX,CX,AX>
	RET
UPDATE_PASS_COUNT ENDP


PAGE

;**********************************************************************
; MEMORY_TEST:
;
;	The Moving Inversion test in performed on the system RAM
; as well as Video RAM. The philosophy behind these tests is to
; get the disk based diagnostics up in case a problem is seen.
; Hence only the first 64K bank and the last 64K bank of system
; RAM is tested. The NMI interrupts are disbled during this test
; so that a parity or a bus error does not cause the system to
; die in the process of testing.
;
;
; Input:
;	None
;
; Output:
;	Moving Inversion test performed until user aborts
;
;	NOTE: Test Count in BP is modified. ES is a screen segment ptr.
;**********************************************************************
MEMORY_TEST PROC NEAR
	PUBLIC	MEMORY_TEST
	PUSHREG	<AX,BX>

	IF	NOT MORROW
	MOV	AL,NOT NMI_ENABLE	;Disable Non Maskable ...
	OUT	NMI_CONTROL,AL		;Interrupts
	ENDIF

	MOV	BP,0			;Initialize test count
	CALL	DIAG_CLEAR_SCREEN	;Clear screen
	MOV	BX,2			;Index for name of test
	CALL	DISPLAY_TEST_MESSAGE	;Put up Test message
MT_1:	CALL	UPDATE_PASS_COUNT	;Update count on screen

	IF	NOT MORROW
	CALL	TEST_PARITY_TREE	;Test parity circuitry
	ENDIF

	CALL	TEST_SYSTEM_RAM		;Test the system RAM
	JC	MT_2			;If user abort get out
	CALL	TEST_VIDEO_RAM		;Test Video RAM
	JNC	MT_1			;In no user abort carry on

MT_2:	CALL	DISPLAY_EXIT_MESSAGE	;Display exit message
	PUSH	BX			;Save the test index
	PUSH	CX			;Save register CX
	MOV	BX,Z100_EXT_ROM_SEGMENT	;Point to the Z-100 expansion ROM area
	MOV	CX,Z100_EXT_ROMS	;Get number of ROMs to check
	CALL	POLL_EXTERNAL_ROMS	;Check for Z-100 mode expansion ROMs
	MOV	BX,Z150_EXT_ROM_SEGMENT	;Point to the Z150 external ROM segment
	MOV	CX,Z150_EXT_ROMS	;Set the number of ROM slots
	CALL	POLL_EXTERNAL_ROMS	;Check for extra interface ROMs
	IN	AL,INTERRUPT_MASK	;Read the interrupt mask register
	AND	AL,TIMER_INTR_MASK	;Make sure that the timer is on...
	OUT	INTERRUPT_MASK,AL	;...in case external ROMs disabled it
	POP	CX			;Restore register CX
	POP	BX			;Restore register BX
	CALL	WAIT_BREAK		;Wait for user to exit

	IF	NOT MORROW
	MOV	AL,NMI_ENABLE		;Enable Non Maskable ...
	OUT	NMI_CONTROL,AL		;Interrupts
	ENDIF

	POPREG	<BX,AX>
	RET
MEMORY_TEST ENDP


PAGE

;**********************************************************************
; TEST_SYSTEM_RAM:
;
;	This routine tests the Bottom 64K and Top 64K of System
; RAM. There will be an overlap of the testing area if the there
; greater than 64K RAM but less than 128K RAM. A variable out of
; the Data Segment is used to determine size of memory. Inter-
; rupts are disabled throughout the testing. This is actually
; necessary only when testing the Bottom 64K (because the inter-
; rupt pointers no longer exist).
;
; Input:
;	MEMORY_SIZE has size of System Memory in Kilobytes
;
; Output:
;	If no error: Carry cleared, Bottom and Top 64K banks tested
;	If user abort: Carry Set
;	If error: Exit to error routine (ie. No Return)
;
; NOTE:
;	ES should point to screen segment whenver DIAG_DISP_??? used.
;**********************************************************************
TEST_SYSTEM_RAM PROC NEAR
	PUBLIC	TEST_SYSTEM_RAM
	PUSHREG	<AX,BX,CX,DX,SI,DI,ES>
	CLD				;Set up for auto-increment
	MOV	DX,ES			;Save screen segment pointer
	MOV	BX,MEMORY_SIZE		;Get size of memory in Kilobytes
	SUB	BX,64			;Justify count from 0
	MOV	CL,6			;Set up for multiply by 1024/16
	SHL	BX,CL			;Convert last page to segment

	PUSHF				;Save interrupt status
	CLI				;Disable interrupts
	PUSH	DS			;Save data segment pointer

	PUSHF				;Save interrupt status
	CLI				;Disable interrupts
	MOV	AX,INTERRUPT_SEGMENT	;Source segment in ...
	MOV	DS,AX			;DS register
	MOV	AX,Z100_MODE_DATA	;Destination segment in ...
	MOV	ES,AX			;ES register
	MOV	DI,0			;ES:DI points to buffer
	MOV	DS,DATA_SEGMENT		;Compatible data segment
	MOV	SI,0			;DS:SI points to current data segment
	MOV	CX,SIZE_OF_MOVE1	;Size of d ata segment
	REP	MOVSB			;Save current data segment
	POPF				;Restore interrupt status

	POP	DS			;Restore data segment pointer
	MOV	SI,SEGMENT_BANK_0	;Segment address of bottom 64K
	MOV	ES,DX			;Restore screen segment pointer
TSR_0:	PUSH	ES			;Save screen segment pointer
	CALL	DISPLAY_BANK_MESSAGE	;Display bank being tested message
	MOV	ES,SI			;Init pointer to base of test RAM
	MOV	CX,SIZE_OF_TSTRAM-1	;Last address in bank
	MOV	DX,FIRST_PATTERN	;Pattern & complement (same parity)
	MOV	AH,PASS_COUNT_FOR_TSTRAM;Pass cnt = log2(size_of_test_RAM)
	CALL	MOVING_INVERSION_TEST	;Perform moving inversion test
	JC	TSR_1			;If error or abort bail out
	MOV	AH,PASS_COUNT_FOR_TSTRAM;Pass cnt = log2(size_of_test_RAM)
	MOV	DX,SECOND_PATTERN	;Two patterns (different parity)
	CALL	MOVING_INVERSION_TEST	;Perform moving inversion test
	JC	TSR_1			;If error or abort bail out
	CMP	SI,BX			;Is all RAM tested?
	JZ	TSR_1			;If so exit loop (Carry cleared)
	POP	ES			;Retrieve screen segment pointer
	ADD	SI,SEGMENT_INCREMENT	;Update current segment address
	CMP	SI,BX			;Is address past end of RAM
	JBE	TSR_0			;If not test new bank
	MOV	SI,BX			;Get last 64K (not on 64K boundary)
	JMP	TSR_0			;Test last 64K (part of it already tested)

TSR_1:	POP	CX			;Reset stack (pop ES)
	PUSH	DS			;Save data segment pointer
	PUSHF				;Save interrupt and error status
	CLI				;Disable interrupts
	PUSH	ES			;Save pointer to RAM ...
	PUSH	DI			;just in case
	CLD				;Set up for auto-increment
	MOV	BX,Z100_MODE_DATA	;Source segment in ...
	MOV	DS,BX			;DS register
	MOV	SI,0			;Buffer is source
	MOV	BX,INTERRUPT_SEGMENT	;Destination segment in ...
	MOV	ES,BX			;ES register
	MOV	ES,DATA_SEGMENT		;Cuurent data segment
	MOV	DI,0			;ES:DI points to current data segment
	MOV	CX,SIZE_OF_MOVE1	;Size of current data segment
	REP	MOVSB			;Restore current data segment
	POP	DI			;Restore pointer to RAM ...
	POP	ES			;just in case
	CALL	INIT_INTERRUPTS		;Reset all interrupt vectors
	POPF				;Restore interrupt and error status
	MOV	BH,AH			;Save error code
	LAHF				;Save error status
	POP	DS			;Restore data segment pointer

	POPF				;Restore interrupt status
	SAHF				;Restore error status
	MOV	AH,BH			;Restore error code

	JNC	TSR_2			;If no error exit routine
	CMP	AH,2			;See if user abort?
	JC	TSR_3			;If not process hard error
	STC				;Signal user abort
TSR_2:	POPREG	<ES,DI,SI,DX,CX,BX,AX>
	RET

TSR_3:	MOV	CX,ES			;Save pointer to bad RAM in CX
	POP	ES			;Get video segment in ES
	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	MOV	SI,ES			;Save the video segment in SI
	MOV	ES,CX			;Retrieve pointer to bad RAM

	IF	NOT MORROW
	OR	AH,AH			;Is it Bad RAM?
	JNZ	TSR_4			;If not assume Parity error
	ENDIF

	STC				;Show video segment already set in SI
	JMP	BAD_RAM			;AL,DL = Obtained, Expected data
					;ES:DI is pointer to bad RAM

	IF	NOT MORROW
TSR_4:	STC				;Show video segment in SI
	JMP	BAD_PARITY		;ES:DI ptr to seg where parity error
	ENDIF
	
TEST_SYSTEM_RAM ENDP


PAGE

;**********************************************************************
; TEST_VIDEO_RAM:
;
;	Video RAM exists in two parts on the Z-150. The displayable
; Video RAM resides at 0B800:0H and the non-displayable Video RAM
; resides at 0F000:0H. Each is 16K bytes long. The non-dosplayable
; Video RAM contains varibles which are necessary for all the func-
; tions provided by the firmware. Both these areas are tested by
; this routine. Interrupts are disabled during testing to be con-
; sistent. This also avoids executing Interrupt Service routines
; that might use any varibles residing in the non-displayable
; Video RAM while it is being tested.
;
; Input:
;	None
;
; Output:
;	If no errors: Carry cleared, Video RAM tested
;	If user abort: Carry set
;	If error: Exit to error routine (ie. No Return)
;
; NOTE:
;	ES is screen segment pointer whenver DIAG_DISP_??? used.
;**********************************************************************
TEST_VIDEO_RAM PROC NEAR
	PUBLIC	TEST_VIDEO_RAM
	PUSHREG	<AX,BX,CX,DX,SI,DI,ES>
	CLD				;Set up for auto-increment
	MOV	DX,ES			;Save screen segment pointer
	PUSHF				;Save interrupt status
	CLI				;Disable interrupts
	PUSH	DS			;Save data sgement pointer

	PUSHF				;Save interrupt status
	CLI				;Disable interrupts
	MOV	BX,CS			;Source of move in ...
	MOV	DS,BX			;DS register
	MOV	SI,0			;Move base of segment
	MOV	BX,SEGMENT_BANK_1	;Top of first 64K bank
	SUB	BX,SIZE_OF_MOVE/16	;Allocate required bytes
	MOV	ES,BX			;Destination of move
	MOV	SS,BX			;Stack has same base (SP is valid)
	MOV	DI,0			;Destination pointer
	MOV	CX,SIZE_OF_MOVE		;Number of bytes to move
	REP	MOVSB			;Move data segment + stack
	POPF				;Restore interrupt status

	POP	DS			;Restore data segment pointer
	MOV	ES,DX			;Retrieve screen segment pointer
	MOV	SI,CS			;Segment of Non-Displayable VRAM
	CALL	DISPLAY_BANK_MESSAGE	;Display bank of memory being tested
	PUSH	ES			;Save screen segment pointer
	MOV	ES,SI			;Init segment ptr to base of test RAM
	MOV	CX,SIZE_OF_NDVRAM-1	;Last address
	MOV	DX,FIRST_PATTERN	;Pattern & complement (same parity)
	MOV	AH,PASS_COUNT_FOR_NDVRAM;Pass cnt = log2(size_of_test_RAM)
	CALL	MOVING_INVERSION_TEST	;Perform test
	POP	CX			;Retrieve screen segment pointer
	JC	TVR_2			;If error process it

	MOV	ES,CX			;Init test RAM pointer (screen seg.)
	MOV	DX,CX			;Save screen segment ptr.
	MOV	CX,SIZE_OF_VRAM-1	;Last address (Assume color card)
	MOV	AH,PASS_COUNT_FOR_VRAM	;Pass cnt = log2(size_of_test_RAM)
	CMP	DX,SCREEN_SEGMENT	;Is color card the default?
	JZ	TVR_1			;Yes - perform test
	MOV	CX,SIZE_OF_MONOVRAM-1	;Last address (Monochrome card)
	MOV	AH,PASS_COUNT_FOR_MONOVRAM;Pass cnt = log2(size_of_test_RAM)
TVR_1:	MOV	DX,FIRST_PATTERN	;Pattern & complement (same parity)
	CALL	MOVING_INVERSION_TEST	;Perform test
	PUSHF				;Save error status
	CALL	DIAG_CLEAR_SCREEN	;Erase test data from the screen
	PUSH	BX			;Save address of RAM buffer on stack
	MOV	BX,2			;Display 'TEST_RAM' message
	CALL	DISPLAY_TEST_MESSAGE	;Display 'MEMORY TEST' message
	POP	BX			;Restore pointer to buffer
	DEC	BP			;Restore count ...
	CALL	UPDATE_PASS_COUNT	;on screen
	POPF				;Restore error status
;	JC	TVR_2			;If error process it

TVR_2:	PUSH	DS			;Save data segment pointer
	PUSHF				;Save interrupt and error status
	CLI				;Disable interrupts
	PUSH	ES			;Save pointer to RAM ...
	PUSH	DI			;just in case
	MOV	DS,BX			;Previous dest is source now
	MOV	SI,0			;Point to base of segment
	MOV	BX,CS			;Destination of move is ...
	MOV	ES,BX			;code segment
	MOV	SS,BX			;Stack has same base (SP is valid)
	MOV	DI,0			;Point to base of segment
	MOV	CX,SIZE_OF_MOVE		;Number of bytes to move
	REP	MOVSB			;Restore data segment + stack
	POP	DI			;Restore pointer to RAM ...
	POP	ES			;just in case
	POPF				;Restore interrupt & error status
	MOV	BH,AH			;Save error code
	LAHF				;Save error status
	POP	DS			;Restore data segment pointer

	POPF				;Restore interrupt status
	SAHF				;Restore error status
	MOV	AH,BH			;Restore error code

	JNC	TVR_3			;If no error exit routine
	CMP	AH,2			;Did user abort?
	JC	TVR_4			;Otherwise process hard error
	STC				;Assert user abort
TVR_3:	POPREG	<ES,DI,SI,DX,CX,BX,AX>
	RET

TVR_4:	MOV	CX,ES			;Save bad RAM segment in CX
	POP	ES			;Get the video segment
	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	MOV	SI,ES			;Place the video segment in SI
	MOV	ES,CX			;Retrieve bad RAM segment

	IF	NOT MORROW
	OR	AH,AH			;Check for bad RAM
	JNZ	TVR_5			;Should never be true!!!
	ENDIF

	STC				;Flag that video segment is in SI
	JMP	BAD_RAM			;Exit to error routine
					;AL,DL have Obtained, Expected data
					;ES:DI has pointer to Bad RAM

	IF	NOT MORROW
TVR_5:	STC				;Flag that video segment is in SI
	JMP	BAD_PARITY		;ES:DI ptr to RAM where error was
					;No parity in VRAM!!!
	ENDIF

TEST_VIDEO_RAM ENDP


PAGE


;**********************************************************************
; DISPLAY_BANK_MESSAGE:
;
;	Displays the current bank of memory being tested. This
; routine positions the cursor displays the message.
;
; Input:
;	SI has base segment of test RAM
;
; Output:
;	Bank message displayed
;
; NOTE:
;	ES is a screen segment pointer
;**********************************************************************
DISPLAY_BANK_MESSAGE PROC NEAR
	PUBLIC	DISPLAY_BANK_MESSAGE
	PUSHREG	<AX,DX,SI>
	PUSH	SI			;Save base segment address
	MOV	DH,0			;Row 0
	MOV	DL,51			;Column 51
	MOV	SI,OFFSET CS:BANK_MESSAGE ;Get pointer to message
	CALL	DIAG_DISP_STRING	;Display the bank being tested message
	POP	AX			;Retrieve segment address
	CALL	DIAG_DISP_HEX_WORD	;Display address of test bank
	MOV	AL,'H'			;Character to be displayed
	CALL	DIAG_DISPLAY_CHAR	;The address in in Hex
	POPREG	<SI,DX,AX>
	RET
DISPLAY_BANK_MESSAGE ENDP

PAGE


;**********************************************************************
; MOVING_INVERSION_TEST:(Segment_Base, Pattern, Max_Addr_Bit)
;
;	This routine performs the Byte Moving Inversion test
; on RAM of size 2**n. The number of passes depends on the
; size of the test RAM. It is logrithm to the base 2 of the
; size of test RAM. The test clicks the speaker whenever pos-
; sible to show it is still executing. 
;
; NOTE: The size of RAM should be 2**n (where n > 0). The 
; base of RAM should fall on a segment boundary, and this
; segment base is passed as a parameter. The test RAM sho-
; uld extend from segment:0 to segment:last_address.
;
; Input:
;	ES = Base of test RAM
;	CX = Last address of test RAM (eg. 0FFFFH for 64K)
;	DX = Patterns to be used
;	AH = Number of passes (log2(size_of_test_RAM) > 0)
;
; Output:
;	If error: Carry set, AH = error code
;		AH = 0 implies Bad RAM
;			AL,DL = Obtained, Expected data
;			ES:DI = Pointer to Bad RAM
;		AH = 1 implies Parity error
;			ES is segment of test RAM
;		AH = 2 implies User Abort
;	If no error: Carry cleared, AH undefined
;
;	NOTE: AX,DX,DI,ES are not preserved
;**********************************************************************
MOVING_INVERSION_TEST PROC NEAR
	PUBLIC	MOVING_INVERSION_TEST
	PUSHREG	<BX,CX,SI>
	MOV	DI,0			;Initialize pointer to base
	MOV	BX,1			;Initialize increment
	MOV	SI,CX			;Save end address of test RAM
	CALL	TEST_BREAK		;Did user abort?
	JC	MIT_6A			;If so exit routine

	CLD				;Auto increment
	MOV	AL,DL			;Pattern in AL
	REP	STOSB			;Initialize memory with pattern
	STOSB				;Take care of extra byte
	CALL	CLICK			;Click Speaker
	CALL	TEST_BREAK		;Did user abort?
	JC	MIT_6A			;If so exit routine

MIT_1:	MOV	CX,SI			;Last address + 1 is ...
	INC	CX			;equal to count for loop.
	MOV	DI,0			;Initialize pointer to base

MIT_2:	MOV	AL,ES:[DI]		;Read byte value in RAM
	CMP	AL,DL			;Verify value
	JNZ	MIT_6			;Process error
	MOV	ES:[DI],DH		;Write complement
	ADD	DI,BX			;Jump by increment
	ADC	DI,0			;If 64K do wrap around
	CMP	DI,SI			;Is it past last address (for < 64K)
	JBE	MIT_2A			;If not carry on
	SUB	DI,SI			;If not wrap around by doing mod
MIT_2A:	LOOP	MIT_2			;Repeat for entire test RAM

	CALL	CLICK			;Click Keyboard
	CALL	TEST_BREAK		;Did user abort?
	JC	MIT_6A			;If so exit routine
	MOV	DI,SI			;Initialize pointer to end of page
	MOV	CX,SI			;Last address + 1 is ...
	INC	CX			;equal to loop count
	XCHG	DH,DL			;Get alternate pattern

MIT_3:	MOV	AL,ES:[DI]		;Read byte value in RAM
	CMP	AL,DL			;Verify value
	JNZ	MIT_6			;Process error
	MOV	ES:[DI],DH		;Write complement
	SUB	DI,BX			;Step by decrement (Reverse pass)
	SBB	DI,0			;Wrap around
	CMP	DI,SI			;Is it past last address (for < 64K)
	JBE	MIT_3A			;If not carry on
	AND	DI,SI			;Force wrap around
MIT_3A:	LOOP	MIT_3			;Repeat for size of RAM

	CALL	CLICK			;Click Keyboard
	CALL	TEST_BREAK		;Did user abort?
	JC	MIT_6A			;If so exit routine
	XCHG	DH,DL			;Get original pattern
	SHL	BX,1			;Update increment (= 2**n)
	DEC	AH			;Update pass count
	JNZ	MIT_1			;Repeat log2(size_of_RAM) times

MIT_4:	MOV	DI,SI			;Initialize pointer to end of page
	MOV	CX,SI			;Count of bytes
	MOV	AL,DL			;Pattern to check
	STD				;Auto decrement
	REPZ	SCASB			;Scan for mismatch with fast access
	JNZ	MIT_5			;If mismatch, error
	SCASB				;Get last byte
	JZ	MIT_7			;If no error continue
	CALL	TEST_BREAK		;Did user abort?
	JC	MIT_6A			;If so exit routine

MIT_5:	INC	DI			;Pointer to error location
	MOV	AL,ES:[DI]		;Get actual data
;	JMP	SHORT MIT_6		;Process error
MIT_6:	MOV	AH,0			;Init MS Byte of value obtained
	MOV	DH,AH			;Init MS Byte of value expected
	STC				;Assert error
	JMP	SHORT MIT_8		;Exit routine

MIT_6A:	MOV	AH,2			;User abort error code
	STC				;Assert soft error
	JMP	SHORT MIT_8		;Exit routine

MIT_7:
	IF	NOT MORROW
	MOV	BX,AX			;Save data
	IN	AL,MEMORY_STATUS	;Read memory status port
	TEST	AL,PARITY_ERR OR MEMORY_ERR;Check for parity or Bus error
	MOV	AX,BX			;Restore data (flags not affected)
	ENDIF

	CLC				;Assume no error (Clears only carry)

	IF	NOT MORROW
	JZ	MIT_8			;If no error carry on
	MOV	AH,1			;Parity error code
	MOV	DI,0			;ES:DI points to base of segment
	STC				;Asset error
	ENDIF

MIT_8:	CLD				;Reset direction flag
	POPREG	<SI,CX,BX>
	RET
MOVING_INVERSION_TEST ENDP


PAGE

	IF	NOT MORROW

;**********************************************************************
; TEST_PARITY_TREE:
;
;	A set of constants is piped through a particular location
; in RAM to check the parity tree implemented in the Z-150. The
; RAM location used for this purpose is 0:0. This is valid as the
; location has already been tested. Since some of the constants
; generate a parity error, it is necessary to disable NMI.
;
; Input:
;	NMI disabled!!!
;
; Output:
;	If no error: Parity tree checked
;	If error: Exit to error routine (ie. No Return)
;**********************************************************************
TEST_PARITY_TREE PROC NEAR
	PUBLIC	TEST_PARITY_TREE
	PUSHREG	<AX,CX,DX,SI,ES>
	MOV	AX,INTERRUPT_SEGMENT	;Interrupt Segment has ...
	MOV	ES,AX			;location to be tested
	MOV	AH,BYTE PTR ES:0	;Save variable at location 0:0
	MOV	SI,OFFSET CS:PARITY_TEST_CONSTANTS;Pointer to table
	MOV	DX,MEMORY_CTRL		;Point to the memory control port

	PUSHF				;Save interrupt status
	CLI				;Disable interrupts
	MOV	AL,NOT PARITY_GEN_ENABLE;Disable parity generation ...
	OUT	DX,AL			;(ie. force 1 in parity RAM)
	MOV	CX,4			;Size of error constants
TPT_1:	IN	AL,SYS_CTRL_PORT	;Read value in port
	OR	AL,DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK
	OUT	SYS_CTRL_PORT,AL	;Clear any parity or bus errors
	NOP				;Delay for error bits to be cleared
	AND	AL,NOT (DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK)
	OUT	SYS_CTRL_PORT,AL	;Reenable checking
	MOV	AL,CS:[SI]		;Get constant
	INC	SI			;Update pointer
	MOV	BYTE PTR ES:0,AL	;Write value to 0:0
	MOV	AL,BYTE PTR ES:0	;Read just written value
	IN	AL,MEMORY_STATUS	;Read current status
	TEST	AL,PARITY_ERR OR MEMORY_ERR; We expect an error
	JZ	TPT_3			;If no error bad circuit
	LOOP	TPT_1			;Repeat for all error patterns

	IN	AL,SYS_CTRL_PORT	;Read value in port
	OR	AL,DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK
	OUT	SYS_CTRL_PORT,AL	;Clear any parity or bus errors
	NOP				;Wait for errors to be cleared
	AND	AL,NOT (DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK)
	OUT	SYS_CTRL_PORT,AL	;Reenable checking
	MOV	AL,PARITY_GEN_ENABLE OR PARITY_CHK_ENABLE ;Enable parity
	OUT	DX,AL			;(ie. correct parity in RAM)
	MOV	CX,5			;Size of no error constants
TPT_2:	MOV	AL,CS:[SI]		;Get constant
	INC	SI			;Update pointer
	MOV	BYTE PTR ES:0,AL	;Write value in 0:0
	MOV	AL,BYTE PTR ES:0	;Read just written value
	IN	AL,MEMORY_STATUS	;Read current status
	TEST	AL,PARITY_ERR OR MEMORY_ERR;No error expected
	JNZ	TPT_3			;If error process it
	LOOP	TPT_2			;Repeat for all no error constants
	MOV	BYTE PTR ES:0,AH	;Restore value at 0:0
	MOV	AL,PARITY_GEN_ENABLE OR PARITY_CHK_ENABLE ;Enable parity & chk
;;;;	MOV	AL,PARITY_GEN_ENABLE	;Enable parity generation
	OUT	DX,AL			;(ie. correct parity in RAM)
	POPF				;Restore interrupt status
	POPREG	<ES,SI,DX,CX,AX>
	RET

TPT_3:	POPF				;Restore interrupt status
	MOV	CX,ES			;Save segment where error occurred
	POP	ES			;Retrieve screen segment pointer
	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	MOV	ES,CX			;Retrieve segment where error occurred
	MOV	BYTE PTR ES:0,AH	;Restore value at 0:0
	MOV	DI,0			;Point to RAM causing error
	JMP	BAD_PARITY		;ES:DI points to 0:0

TEST_PARITY_TREE ENDP


	ENDIF



PAGE

;**********************************************************************
; DISK_READ_TEST:
;
;	If for some reason a particular disk cannot be read, it
; is helpful for the Field Service Personnel to execute a command
; that would repeatedly read the disk. This facilitates putting
; an oscilloscope and trouble shooting.
;
; Input:
;	None
;
; Output:
;	Default boot device (drive 0) repeatedly read
;
;	NOTE: Test count in BP modified. ES is screen segment pointer.
;**********************************************************************
DISK_READ_TEST PROC NEAR
	PUBLIC	DISK_READ_TEST
	PUSHREG	<AX,BX,SI>
	MOV	BP,0			;Initialize test count
	CALL	DIAG_CLEAR_SCREEN	;Clear the display
	MOV	BX,0			;Index for test name
	CALL	DISPLAY_TEST_MESSAGE	;Display test screen
DRT_1:	CALL	UPDATE_PASS_COUNT	;Increment count on screen
DRT_2:	CALL	DISK_BOOT		;Read Boot code
	JNC	DRT_1			;If no problems continue
	TEST	AH,AH			;Check for user abort
	JZ	DRT_3			;If so abort test
	MOV	DX,(1 SHL 8) OR 0	;Position cursor on line one
	MOV	SI,OFFSET CS:DISK_READ_ERROR; Pointer to message
	CALL	DIAG_DISP_STRING	;Display error message
	JMP	DRT_2			;Do not update count
DRT_3:	CALL	DISPLAY_EXIT_MESSAGE	;Display message to exit test
	CALL	WAIT_FOR_USER_EXIT	;Wait for user to exit routine
	POPREG	<SI,BX,AX>
	RET
DISK_READ_TEST ENDP


PAGE

;**********************************************************************
; WAIT_FOR_USER_EXIT:
;
;	Wait_For_User_Exit wait for the user to type CTRL_BREAK or
; an ESCAPE key. This routine assumes that interrupts are enabled.
; If interrupts are disbaled WAIT_BREAK should be called to poll
; for these keys.
;**********************************************************************
WAIT_FOR_USER_EXIT PROC NEAR
	PUBLIC	WAIT_FOR_USER_EXIT
	PUSHREG	<AX>
WFUE_1:	CALL	GET_KEY_BUFF		;Wait for user to type a key
	CMP	AX,BREAK_CODE		;Is it an exit key?
	JZ	WFUE_2			;If so exit routine
	CMP	AX,ESC_CODE		;Is it an exit key?
	JNZ	WFUE_1			;If not wait for right key
WFUE_2:	POPREG	<AX>
	RET
WAIT_FOR_USER_EXIT ENDP


PAGE

;**********************************************************************
; POWER_UP_TEST:
;
;	This routine just repeats all the tests performed at power-
; up time. This repetitive execution is necessary to replicate pos-
; sible time dependent errors.
;
; Input:
;	None
;
; Output:
;	If user abort: Resturn to main menu
;	If error: Exit to error routine (ie. No Return)
;
;	NOTE: Modifies the Test Count in BP & Cursor Position in DX.
;	      ES is screen segment pointer.
;**********************************************************************
POWER_UP_TEST PROC NEAR
	PUBLIC	POWER_UP_TEST
	PUSHREG	<AX,BX,SI>
	MOV	BP,0			;Initialize test count
	CALL	DIAG_CLEAR_SCREEN	;Clear the screen
	MOV	BX,3			;Index for test name
	CALL	DISPLAY_TEST_MESSAGE	;Display test screen
PUT_1:	CALL	UPDATE_PASS_COUNT	;Increment count on screen
	CALL	MENU_PROCESSOR_TEST	;Test the 8088 Processor
	CALL	MENU_ROM_TEST		;Test the RPM check sum
	CALL	TIMER_INTERRUPT_TEST	;Test out the 8253 timer and intr
	JC	PUT_5			;If error jump to error handler
	CALL	GET_KEY_STATUS		;Check key in buffer
	JZ	PUT_1			;If no key carry on
	CALL	GET_KEY_BUFF		;Since there is a key, get it
	CMP	AX,BREAK_CODE		;Is it an abort key?
	JZ	PUT_2			;If so abort test
	CMP	AX,ESC_CODE		;Is it an abort key?
	JNZ	PUT_1			;If not continue process
PUT_2:	CALL	DISPLAY_EXIT_MESSAGE	;Display way to exit test
	CALL	WAIT_FOR_USER_EXIT	;Wait for user to exit routine
	POPREG	<SI,BX,AX>
	RET

PUT_5:	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	MOV	SI,OFFSET BAD_TINTR_MSG	;ERROR - timer interrupt failed
	JMP	DISPLAY_ERROR_MSG	;Exit to monitor/debugger

POWER_UP_TEST ENDP


PAGE

;**********************************************************************
; MENU_PROCESSOR_TEST:
;
;	Menu_Processor_Test is called to check out the 8088 processor 
; chip. It performs the following tests:
;		- Fills all registers (except CS,IP,SS,SP) 
;		  with zeros and ones.
;		- Sets and clears flag bits
;		- Determines if conditional jumps work
;		- Tests several arithmetic/logic operations,
;		  to test out the ALU.
;
; NOTE: ES is screen segment pointer.
;**********************************************************************
MENU_PROCESSOR_TEST PROC NEAR
	PUBLIC	MENU_PROCESSOR_TEST
	PUSHREG	<AX,BX,CX,DX,SI,DI,BP,ES,DS>
	STC				;Set the carry flag
	JNC	MENU_BAD_PROCESSOR	;ERROR - carry not set
	CLC				;Now, clear the carry flag
	JC	MENU_BAD_PROCESSOR	;ERROR - carry not cleared
	XOR	AX,AX			;Test ALU, set AX to 0
	JNZ	MENU_BAD_PROCESSOR	;ERROR - invalid zero flag
MPT_1:	MOV	DS,AX			;Copy AX into DS
	MOV	BX,DS			;Copy DS into BX
	MOV	CX,BX			;Copy BX into CX
	MOV	ES,CX			;Copy CX into ES
	MOV	DX,ES			;Copy ES into DX
	MOV	BP,DX			;Copy DX into BP
	MOV	SI,BP			;Copy BP into SI
	MOV	DI,SI			;Copy SI into DI
	CMP	AX,DI			;Did the copy work correctly?
	JNZ	MENU_BAD_PROCESSOR	;ERROR - bad register(s)
	DEC	AX			;Yes - decrement AX (all 1's first)
	JC	MPT_1			;Loop for 1's test
	TEST	AX,AX			;Set sign/zero flags
	JNS	MENU_BAD_PROCESSOR	;ERROR - sign bit not set (AX is -2)
	JZ	MENU_BAD_PROCESSOR	;ERROR - zero bit incorrect
	NEG	AX			;Now, set AX to +2
	JS	MENU_BAD_PROCESSOR	;ERROR - sign bit not cleared
	POPREG	<DS,ES,BP,DI,SI,DX,CX,BX,AX>
	RET

MENU_BAD_PROCESSOR:
	POP	DS			;Restore data segment pointer
	POP	ES			;Restore screen segment pointer
	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	JMP	BAD_PROCESSOR		;Signal bad proceesor
MENU_PROCESSOR_TEST ENDP


PAGE

;**********************************************************************
; MENU_ROM_TEST:
;
;	Menu_ROM_Test adds every byte in the ROM together, and com-
; pares the result with 0 (a checksum placed at the last two bytes 
; of the ROM should force the generated checksum to be 0).
;
; NOTE: ES is screen segment pointer.
;**********************************************************************
MENU_ROM_TEST PROC NEAR
	PUBLIC	MENU_ROM_TEST
	PUSHREG	<AX,BX,CX>
	MOV	AX,0			;Clear checksum
	MOV	BX,OFFSET ROM_START	;Initialize pointer into ROM
	MOV	CX,ROM_LENGTH SHR 1	;Number of words in ROM to CX
MRT_1:	ADD	AX,WORD PTR CS: [BX]	;Add a word into the checksum
	ADC	AX,0			;Add carries back into checksum
	ADD	BX,2			;Point to the next word
	LOOP	MRT_1			;Repeat for every word of the ROM
	CMP	AX,1			;Did the ROM test pass?
	JNE	MENU_BAD_ROM		;ERROR - invalid ROM checksum
	POPREG	<CX,BX,AX>
	RET

MENU_BAD_ROM:
	CALL	DISPLAY_EXIT_MESSAGE	;Update user on how to exit
	JMP	BAD_ROM			;Signal bad ROM
MENU_ROM_TEST ENDP


PAGE

;**********************************************************************
;				MESSAGES
;**********************************************************************


MENU_PROCEDURES LABEL WORD
	DW	DISK_READ_TEST		;Repeated Disk Read Test
	DW	KEYBOARD_TEST		;Keyboard Test
	DW	MEMORY_TEST		;Repeated Moving Inversion Test
	DW	POWER_UP_TEST		;Repeated Power-up Test


EXIT_MESSAGE LABEL BYTE
	DB	'TYPE  <ESC>  TO EXIT',0


TEST_MESSAGE LABEL BYTE
	DB	'TEST COUNT = ',0


ABORT_MESSAGE LABEL BYTE
	DB	'TYPE <ESC>  TO ABORT',0

BANK_MESSAGE LABEL BYTE
	DB	'TESTING MEMORY BANK AT ',0


CURRENT_TEST_MESSAGE LABEL WORD
	DW	DISK_READ_TEST_MESSAGE
	DW	KEYBOARD_TEST_MESSAGE
	DW	MEMORY_TEST_MESSAGE
	DW	POWER_UP_TEST_MESSAGE


DISK_READ_TEST_MESSAGE LABEL BYTE
	DB	'DISK READ TEST',0


KEYBOARD_TEST_MESSAGE LABEL BYTE
	DB	'KEYBOARD TEST',0


CHAR_CODE_MESSAGE LABEL BYTE
	DB	TAB,TAB,TAB,TAB,TAB,TAB,'  CHARACTER CODE = ',0


MEMORY_TEST_MESSAGE LABEL BYTE
	DB	'SYSTEM AND VIDEO MEMORY TEST',0


POWER_UP_TEST_MESSAGE LABEL BYTE
	DB	'POWER-UP TEST',0


DISK_READ_ERROR LABEL BYTE
	DB	'DEVICE ERROR',0


	IF	NOT MORROW
PARITY_TEST_CONSTANTS LABEL BYTE
	DB	076H,0C4H,020H,0BFH,06DH,092H,099H,00BH,0C1H
	ENDIF


MONITOR_SEGMENT ENDS

	END

