;**********************************************************************
; SCROLL: (LINE_COUNT, WINDOW_START, WINDOW_END, ATTRIB)
;
;	Scroll is called to scroll the screen upwards or
; downwards (upwards means that the existing screen lines 
; are moved up).  Arbitrary windows may be scrolled;  if 
; the entire screen is scrolled, and one of the hardware 
; scrolling modes is enabled, then the CRT controller will 
; actually be updated with a new start address.
;	Note that Scroll actually contains two entry points:
; Scroll_Up and Scroll_Down.
;
; Input:
;	AL: Number of lines to scroll (usually 1).  If the
;	    scroll count is zero, then the entire window is
;	    cleared.
;	CX: Row and column of the upper-left hand window corner.
;	DX: Row/column of the lower-right hand window corner.
;	BH: Contains the attribute byte to be used on blank
;	    lines.
;**********************************************************************
SCROLL PROC NEAR
	PUBLIC	SCROLL_UP
	PUBLIC	SCROLL_DOWN
SCROLL_UP:
	PUSH	AX
	MOV	AH,FALSE		;Set upwards-scroll flag
	JMP	SHORT SC0
SCROLL_DOWN:
	PUSH	AX
	MOV	AH,TRUE			;Set downwards-scroll flag

SC0:	PUSHREG	<BX,CX,DX,SI,DI,ES>
;
; Determine page number in case CRT_START_ADDRESS or DISPLAYED_PAGE was poked
;
	PUSH	BX			;Save display attribute
	PUSH	AX			;Save the scroll count
	PUSH	DX			;Save lower window corner position
	MOV	AL,DISPLAYED_PAGE	;Get the new display page
	CMP	AL,OLD_DISPLAYED_PAGE	;Has the display page been poked?
	JNE	SC05			;Yes - use the new poked value
	MOV	AX,CRT_START_ADDRESS	;No, get the current CRT start address
	SHL	AX,1			;Make it a software address
	MOV	DX,0			;Convert to a double-word for divide
	DIV	PAGE_SIZE		;Determine which page is active
SC05:	MOV	BH,AL			;Place page number in BH
	CALL	SET_IO_PAGE		;Set this page to be scrolled
	POP	DX			;Restore the lower window position
	POP	AX			;Restore the scroll count
	POP	BX			;Restore the display attribute
	CMP	VIDEO_MODE,1		;In 40 character mode?
	JBE	SC3			;Yes - must do software scroll
	CMP	SCROLL_MODE,1		;No, is hardware scroll enabled?
	JB	SC3			;Software scrolling enabled
	CMP	AL,1			;Is the screen being scrolled 1 line?
	JNE	SC3			;No, use software scroll routine
	MOV	BL,BYTE PTR NUMBER_OF_COLUMNS	;Get the number of columns - 1
	DEC	BL
	CMP	BL,DL			;Scroll stop at last char on screen?
	JNE	SC3			;No, must software scroll
	TEST	CX,CX			;Yes - does scroll start at first line?
	JNE	SC3			;No, must software scroll
	CMP	DH,LAST_ROW		;Does scroll stop on last line?
	JNE	SC3			;No, software scroll
	CMP	VIDEO_MODE,7		;On the monochrome board?
	JE	SC1			;Yes - cannot smooth scroll!
	CMP	VIDEO_MODE,3		;In 80-column mode on the color card?
	JBE	SC1			;Yes - cannot smooth scroll
	CMP	SCROLL_MODE,1		;Use 'Jump' scroll?
	JNE	SC2			;No, do a 'Smooth' scroll
SC1:	CALL	JUMP_SCROLL		;Scroll page a line
	JMP	SHORT SC7		;Return
SC2:	CALL	SMOOTH_SCROLL		;Smooth scroll page a line
	JMP	SHORT SC7		;Return
SC3:	CALL	SET_SCROLL_ADDRESSES	;Set source and dest scroll addresses
	TEST	AL,AL			;Just clearing the screen?
	JE	SC4			;Yes - don't try to scroll!
	CALL	SOFTWARE_SCROLL		;Scroll the screen by moving text
SC4:	TEST	AL,AL			;Clearing the screen only?
	JNE	SC5			;No, continue
	MOV	AL,CH			;Yes - set the line count
SC5:	MOV	CH,0			;Get char count only in CX
SC6:	PUSH	AX			;Save line count
	MOV	AL,FALSE		;Set erase-character-line mode
	CALL	CLEAR_LINE		;Erase a line of text
	POP	AX			;Restore line counter
	MOV	DX,ROW_INCREMENT	;Get increment to start of next line
	CALL	INCREMENT_POSITION	;Point to the next line
	DEC	AL			;Done with clear yet?
	JNE	SC6			;No, loop until enough lines cleared
SC7:	POPREG	<ES,DI,SI,DX,CX,BX>	;Restore registers
	POP	AX
	RET
SCROLL ENDP



;**********************************************************************
; JUMP_SCROLL: (DIRECTION, SCROLL_COUNT, ATTRIBUTE)
;
;	Jump_Scroll is called to perform an enhanced screen
; scroll.  Since the scroll is performed in hardware, it is
; much faster than the normal 'software' scroll.  Both
; upwards and downwards scrolling may be performed.  Note
; that multiple-line scrolling is not supported - this may
; be emulated by multiple calls to the jump-scroll routine.
;
; Input:
;	AH: Scroll direction flag.  If set (non-zero), then
;	    downward scrolling will be performed.  Otherwise,
;	    upwards scrolling is used.
;	BH: Attribute to be used on blanked lines.
;
; Output:
;	Sets global variables PAGE_BASE and CRT_START_ADDRESS.
;**********************************************************************
JUMP_SCROLL PROC NEAR
	PUBLIC	JUMP_SCROLL
	PUSHREG	<AX,BX,CX,DX>
	MOV	CX,0			;Get position of first line to erase
	TEST	AH,AH			;Upwards scroll requested?
	JE	JS1			;Yes - have the right position
	MOV	CH,LAST_ROW		;No, get posn of last line
	MOV	CL,0
JS1:	CALL	SET_SCREEN_PTR		;Point to it
	MOV	AL,FALSE		;Set erase-character-line parameter
	MOV	CX,NUMBER_OF_COLUMNS	;Get the line length in CX
	CALL	CLEAR_LINE		;Erase a line
	MOV	BL,CURRENT_PAGE		;Get the page as a word in BX
	MOV	BH,0
	SHL	BX,1			;Make page into a word offset
	CMP	VIDEO_MODE,7		;Is this the monochrome board?
	JNE	JS2			;No, have correct value for page
	MOV	BX,8 * 2		;Yes - treat monochrome as page 8
JS2:	MOV	DX,PAGE_BASE[BX]	;Point to the start of this page
	TEST	AH,AH			;Scrolling upwards?
	JNE	JS3			;No, scrolling downwards
	ADD	DX,ROW_INCREMENT	;Yes - point to the next row
	CMP	DX,END_OF_PAGE		;Did we wrap around?
	JBE	JS4			;No, continue
	SUB	DX,PAGE_SIZE		;Yes - wrap around screen
	JMP	SHORT JS4
JS3:	SUB	DX,ROW_INCREMENT	;Point to the previous line
	CMP	DX,START_OF_PAGE	;Did we wrap around?
	JGE	JS4			;No, continue
	ADD	DX,PAGE_SIZE		;Yes - wrap around to start of page
JS4:	MOV	PAGE_BASE[BX],DX	;Store start-of-screen address
	SHR	DX,1			;Make it into a hardware address
	MOV	CRT_START_ADDRESS,DX	;Update hardware start-of-screen
	MOV	BX,DX			;Get the start address in BX
	MOV	DX,VIDEO_BASE		;Point to this card's I/O ports
	CMP	DX,MONO_CARD		;Talking to the monochrome card?
	JE	JS6			;Yes - no VSYNC to wait for!
	ADD	DX,CRT_STATUS_PORT	;Point to the status port
JS5:	IN	AL,DX			;Read the CRT status port
	AND	AL,VERTICAL_SYNC	;Is a VSYNC in progress?
	JNZ	JS5			;Yes - wait for it to go away
JS6:	MOV	AL,OLD_DISPLAYED_PAGE	;Get the 'real' displayed page number
	CMP	AL,CURRENT_PAGE		;Are we scrolling the current page?
	JNE	JS7			;No, don't update the hardware
	MOV	AL,CRT_START_PORT	;Now, get the CRT start-address port
	CALL	SEND_VIDEO_WORD		;Send new start-address to CRT chip
JS7:	POPREG	<DX,CX,BX,AX>
	RET
JUMP_SCROLL ENDP



;**********************************************************************
; SMOOTH_SCROLL: (DIRECTION, SCROLL_COUNT, ATTRIBUTE)
;
;	Smooth_Scroll performs a scan-line-by-scan-line
; scroll of a single character line.  Like 'Jump_Scroll'
; this is a hardware scroll.  In this case, however, speed
; is sacrificed for improved readability.  Also, note that
; this routine will only work on the color card in one of
; the graphics modes.
;
; Input:
;	AH: Scroll direction flag.  If set (non-zero), then
;	    downward scrolling will be performed.  Otherwise,
;	    upwards scrolling is used.
;	BH: Attribute to be used on blanked lines.
;
; Output:
;	Sets global variables PAGE_BASE and CRT_START_ADDRESS.
;**********************************************************************
SMOOTH_SCROLL PROC NEAR
	PUBLIC	SMOOTH_SCROLL
	PUSHREG	<AX,BX,CX,DX,SI,BP>
	MOV	BP,FONT_HEIGHT		;Set BP as a scan line counter
	MOV	CX,0			;Get position of first line to erase
	TEST	AH,AH			;Upwards scroll requested?
	JE	SS1			;Yes - have the right position
	MOV	CH,LAST_ROW		;No, get posn of last line
	MOV	CL,0
SS1:	CALL	SET_SCREEN_PTR		;Point to first line to erase
	TEST	AH,AH			;Upwards scrolling?
	JE	SS2			;Yes - have correct address
	ADD	DI,ROW_INCREMENT	;No, point DI to last scan line
	SUB	DI,SCAN_LINE_INCREMENT
	CMP	DI,OFFSET END_OF_PART	;Wrap around if necessary
	JBE	SS2
	SUB	DI,SCREEN_BYTES SHR 1
SS2:	MOV	BX,PAGE_BASE[0]		;Point to the start of this page
SS3:	MOV	AL,TRUE			;Set erase-scan-line parameter
	MOV	CX,NUMBER_OF_COLUMNS	;Get the length of a line
	CALL	CLEAR_LINE		;Erase a single scan line
	XOR	BX,SCREEN_PARTITION_OFS	;Point to scan line to display
	XOR	DI,SCREEN_PARTITION_OFS	;Point to scan line to erase
	TEST	BP,1			;Did we just point to an even line?
	JE	SS7			;No, pointing to an odd scan line
	TEST	AH,AH			;Scrolling upwards?
	JNE	SS5			;No
	ADD	BX,SCAN_LINE_INCREMENT	;Yes - point to the next scan line
	CMP	BX,OFFSET END_OF_PART	;Did we wrap around?
	JBE	SS4			;No, continue
	SUB	BX,SCREEN_BYTES	SHR 1	;Yes - wrap around partition
SS4:	ADD	DI,SCAN_LINE_INCREMENT	;Point to next scan line to erase
	CMP	DI,OFFSET END_OF_PART	;Wrap around if necessary
	JBE	SS7
	SUB	DI,SCREEN_BYTES SHR 1
	JMP	SHORT SS7		;Set new page address
SS5:	SUB	BX,SCAN_LINE_INCREMENT	;Point to the previous scan line
	JNB	SS6
	ADD	BX,SCREEN_BYTES	SHR 1	;Wrap-around if necessary
SS6:	SUB	DI,SCAN_LINE_INCREMENT	;Point to next line to erase
	JNB	SS7			;Wrap-around if necessary
	ADD	DI,SCREEN_BYTES SHR 1
SS7:	MOV	PAGE_BASE[0],BX		;Store start-of-screen address
	SHR	BX,1			;Modify to get hardware address
	MOV	CRT_START_ADDRESS,BX	;Save CRT start address, as well
	CALL	WAIT_VERTICAL_SYNC	;Wait for a second VSYNC to occur
	MOV	AL,CRT_START_PORT	;Get the CRT start-address port
	CALL	SEND_VIDEO_WORD		;Send new start-address to CRT chip
	PUSH	DS			;Save the current data segment
	MOV	DX,COLOR_CARD+CRT_BASE_REGISTER	;Point DX to the video ctrl port
	MOV	AL,CRT_V_TOTAL_ADJ	;Get the number of the vert. adjust reg
	OUT	DX,AL			;Select the vertical total register
	MOV	AL,VIDEO_PARMS_INTR	;Point to the video parameters interrupt
	CALL	GET_INTR_PTR		;Point to the video parameter block
	MOV	AL,DS:[SI].V_TOTAL_ADJ	;Get the vertical adjust value
	TEST	BP,1			;Did we just point to an even scan line?
	JNE	SS9			;Yes - vertical adjust is correct
	TEST	AH,AH			;Is this a forward scroll?
	JNZ	SS8			;No, reverse scroll
	DEC	AL			;Yes - decrement S.L. to move screen up
	JMP	SHORT SS9		;Update the hardware
SS8:	INC	AL			;Increment S.L. to move screen down
SS9:	INC	DX			;Point DX to the CRT data port
	OUT	DX,AL			;Set the new vertical adjust value
	POP	DS			;Restore the data segment
	DEC	BP			;Scrolled entire line yet?
	JZ	SS10			;Yes - return
	CALL	WAIT_VERTICAL_SYNC	;No, wait for a vertical sync pulse
	JMP	SS2			;Go on to the next scan line
SS10:	POPREG	<BP,SI,DX,CX,BX,AX>
	RET
SMOOTH_SCROLL ENDP



;**********************************************************************
; WAIT_VERTICAL_SYNC:
;
;	Wait_Vertical_Sync waits for a vertical sync pulse - note that
; the monochrome card does not have a VSYNC indicator, so this routine
; will immediately return if the monochrome card is being referenced.
; Note that the routine contains a timeout to prevent looping forever!
;**********************************************************************
WAIT_VERTICAL_SYNC PROC NEAR
	PUBLIC	WAIT_VERTICAL_SYNC
	PUSHREG	<AX,CX,DX>
	MOV	DX,VIDEO_BASE		;Point to the CRT status port
	CMP	DX,MONO_CARD		;Is this the monochrome card?
	JE	WVS3			;Yes - there isn't a VSYNC!
	ADD	DX,CRT_STATUS_PORT	;No, point to the video status port
	MOV	CX,10000		;Maximum time to wait for VSYNC
WVS1:	IN	AL,DX			;First, wait for NOT VSYNC
	AND	AL,VERTICAL_SYNC	;...this prevents multiple scrolling
	LOOPNZ	WVS1			;...during a single sync time
	JCXZ	WVS3			;Timeout - just return
WVS2:	IN	AL,DX			;Read the CRT status port
	AND	AL,VERTICAL_SYNC	;Gotten VSYNC yet?
	LOOPZ	WVS2			;No, wait until VSYNC received
WVS3:	POPREG	<DX,CX,AX>
	RET
WAIT_VERTICAL_SYNC ENDP



;**********************************************************************
; SET_SCROLL_ADDRESSES: (DIRECTION, COUNT, WINDOW_START, WINDOW_END)
;
;	Set_Scroll_Addresses is called when 'software scrolling' is to
; be performed.  It translates a pair of window coordinates and a
; scroll count into source and destination addresses for text moves.
;
; Input:
;	AH: Scroll direction flag.  If zero, then upwards
;	    scrolling is assumed.  Otherwise, downwards scroll
;	    addresses are generated.
;	AL: Number of lines to scroll (usually 1).
;	CX: Row and column of the upper-left hand window corner.
;	DX: Row/column of the lower-right hand window corner.
;
; Output:
;    ES:SI: Pointing to the 'source' address for screen moves
;    ES:DI: Pointing to the 'dest' address for screen moves
;	CH: Number of lines to be scrolled
;	CL: Number of characters per scroll line
;**********************************************************************
SET_SCROLL_ADDRESSES PROC NEAR
	PUBLIC	SET_SCROLL_ADDRESSES
	PUSHREG	<AX>
	PUSH	AX			;Save the scroll count (ie, distance)
	TEST	AH,AH			;Scrolling upwards?
	JNE	SA1			;No, set addresses for scroll down
	CALL	SET_SCREEN_PTR		;Yes - Point ES:DI to the start addr
	MOV	SI,DI			;Set Source = Dest address
	JMP	SHORT SA2
SA1:	PUSH	CX			;Save source position
	MOV	CH,DH			;Get the starting character position
	CALL	SET_SCREEN_PTR		;Point ES:DI to the source address
	MOV	SI,DI			;Copy to source ptr (already in dest)
	POP	CX			;Restore source position
SA2:	TEST	AL,AL			;Is scroll count zero?
	JZ	SA6			;Yes - set line length and return
SA3:	TEST	AH,AH			;Scrolling upwards?
	JNE	SA4			;No, set address for reverse scroll
	ADD	SI,ROW_INCREMENT	;Point source to next line
	CMP	SI,END_OF_PAGE		;Beyond end of screen?
	JBE	SA5			;No
	SUB	SI,PAGE_SIZE 		;Yes - wrap around
	JMP	SHORT SA5
SA4:	SUB	SI,ROW_INCREMENT	;Point source to previous line
	CMP	SI,START_OF_PAGE	;Below start of page?
	JGE	SA5			;No, continue
	ADD	SI,PAGE_SIZE		;Screen wrapped around - adjust
SA5:	DEC	AL			;Repeated 'Scroll-Count' yet?
	JNE	SA3			;No, repeat until pointers correct
SA6:	SUB	CX,DX			;Subtract dest posn from start
	NEG	CX			;Make it a positive count
	ADD	CX,(1 SHL 8) OR 1	;Adjust counts for zero offset
	POP	AX			;Restore the scroll count
	SUB	CH,AL			;Subtract lines which will be erased
	POPREG	<AX>			;Restore registers
	RET
SET_SCROLL_ADDRESSES ENDP



;**********************************************************************
; SOFTWARE_SCROLL: (DIRECTION, SOURCE, DEST, LINE_COUNT, SCROLL_COUNT)
;
;	Software_Scroll performs a scroll of a specified segment
; of the screen by moving bytes around on the screen.
;
; Input:
;	AH: Scroll direction.  If AH is 0, upwards scrolling is
;	    used - otherwise the screen is scrolled downwards.
;	CH: Number of lines to be scrolled
;	CL: Number of characters per scroll line
;    ES:SI: Pointing to the 'source' address for screen moves
;    ES:DI: Pointing to the 'dest' address for screen moves
;
; Output:
;    ES:DI: Pointing to first line to be cleared
;**********************************************************************
SOFTWARE_SCROLL PROC NEAR
	PUBLIC	SOFTWARE_SCROLL
	PUSHREG	<AX,CX,DX,SI,BP>	;Save registers
SWS1:	MOV	BP,1			;Assume one line of bytes / char
	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	SWS2			;No
	CMP	VIDEO_MODE,7		;In the monochrome text mode?
	JE	SWS2			;Yes - chars only one byte-line tall
	MOV	BP,FONT_HEIGHT		;No, have FONT_HEIGHT lines / char
SWS2:	PUSH	SI			;Save source pointer
	PUSH	DI			;Save dest pointer
	PUSH	CX			;Save line count
	MOV	CH,0			;Extend char count to a word
	CMP	COLUMN_INCREMENT,1	;Are attribute bytes involved?
	JE	SWS3			;No, have correct byte count
	SHL	CX,1			;Yes - include attributes in count
SWS3:	CALL	MOVE_SCREEN_BYTES	;Move a line of bytes (maybe chars)
	DEC	BP			;Moved all of the character yet?
	JE	SWS4			;Yes - go on to next line
	XOR	SI,SCREEN_PARTITION_OFS	;No, point to the next scan line
	XOR	DI,SCREEN_PARTITION_OFS
	TEST	BP,1			;Did we just move an odd scan line?
	JNE	SWS3			;No, already pointing correctly
	PUSH	AX			;Save direction flag
	MOV	AH,0			;Force pointers to next scan line
	MOV	DX,SCAN_LINE_INCREMENT	;Want to inc ptrs to next scan line
	CALL	INCREMENT_POSITION	;Point src & dest to next scan line
	POP	AX			;Restore direction flag
	JMP	SHORT SWS3		;Continue till done w/character
SWS4:	POP	CX			;Restore line count
	POP	DI			;Restore dest pointer
	POP	SI			;Restore source pointer
	MOV	DX,ROW_INCREMENT	;Get offset to next character line
	CALL	INCREMENT_POSITION	;Point to start of next char line
	DEC	CH			;Moved all lines yet?
	JNZ	SWS1			;No, loop until all moved
	POPREG	<BP,SI,DX,CX,AX>	;Yes - restore registers
	RET
SOFTWARE_SCROLL ENDP



;**********************************************************************
; MOVE_SCREEN_BYTES: (SOURCE, DEST, COUNT)
;
;	Move_Screen_Bytes is used to move a byte string from one screen
; location to another.  It checks to see whether or not the move will
; wrap around at the end of the screen - if so, the move is split into
; two phases, so that the information is transferred correctly.
;
; Input:
;	ES: Screen segment
;	SI: Pointer to source string
;	DI: Pointer to destination string
;	CX: Byte count
;**********************************************************************
MOVE_SCREEN_BYTES PROC NEAR
	PUBLIC	MOVE_SCREEN_BYTES
	PUSHREG	<AX,BX,CX,DX,SI,DI,BP>
	MOV	BP,OFFSET END_OF_SCREEN	;Assume alpha mode for end-screen
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JBE	MSB1			;Yes - end-screen addr set correctly
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	MSB1			;Yes - screen end all set
	CMP	SI,OFFSET END_OF_PART	;Graphics - in the first partition?
	JA	MSB1			;No, end-of-screen correct
	MOV	BP,OFFSET END_OF_PART	;Partition 1 - point to end-screen
; Note that it is possible for either the source or dest to wrap around -
; it is impossible for BOTH to wrap around simultaneously!
MSB1:	MOV	DH,0			;Clear the wrap-around flag
	MOV	AX,SI			;Get the source pointer
	ADD	AX,CX			;Add in length of string to move
	CMP	AX,BP			;Going to wrap around on source?
	JBE	MSB2			;No, continue
	MOV	BX,CX			;Yes - save original byte count
	MOV	AX,BP			;Point to the end of screen
	SUB	AX,SI			;Determine how many can be moved
	INC	AX			;Include last byte in count
	MOV	CX,AX			;Save as new pass-one count
	SUB	BX,CX			;Set BX to remaining bytes to move
	MOV	DH,1			;Set the wrap-around flag to src
	JMP	SHORT MSB3
MSB2:	MOV	AX,DI			;Get the dest pointer
	ADD	AX,CX			;Add in number of bytes to move
	CMP	AX,BP			;Going to wrap around on dest?
	JBE	MSB3			;No, continue
	MOV	BX,CX			;Yes - save original byte count
	MOV	AX,BP			;Point to end of screen
	SUB	AX,DI			;See how many can be moved first pass
	INC	AX			;Include last byte in count
	MOV	CX,AX			;Save as pass-one count
	SUB	BX,CX			;Determine pass two count
	MOV	DH,2			;Set the wrapped-around flag for dest
MSB3:	PUSH	DS			;Save the data segment
	PUSH	ES			;Copy screen segment
	POP	DS
	SHR	CX,1			;Make byte count a word count
	JNC	MSB4			;Moving an even number of bytes
	MOVSB				;Moving odd count - move the odd byte
MSB4:	REP	MOVSW			;Copy a passes worth of words
MSB5:	POP	DS			;Restore data segment
	TEST	DH,DH			;Wrapped around?
	JZ	MSB8			;No, done!
	MOV	CX,BX			;Yes - get the pass two byte count
	MOV	AX,OFFSET SCREEN	;Assume start of screen
	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	MSB6			;No, continue
	CMP	VIDEO_MODE,7		;In the monochrome mode?
	JE	MSB6			;Yes - try for second pass
	CMP	BP,OFFSET END_OF_SCREEN	;Graphics - in second partition?
	JB	MSB6			;No, continue
	MOV	AX,OFFSET END_OF_PART+1	;Set base addr of partition 2
MSB6:	CMP	DH,1			;Did the source wrap around?
	JNE	MSB7			;No, it was the dest ptr
	MOV	SI,AX			;Yes - point src to start of screen
	JMP	SHORT MSB1		;Go for next pass
MSB7:	MOV	DI,AX			;Wrap dest around to start of screen
	JMP	SHORT MSB1		;Copy the next chunk of data!
MSB8:	POPREG	<BP,DI,SI,DX,CX,BX,AX>
	RET
MOVE_SCREEN_BYTES ENDP




