	PAGE	,132
	TITLE	MFM_150 Multi-Functional Monitor

;**********************************************************************
;
;                         ---------------------
;------------------------- Z-150 Debug Monitor ------------------------
;                         ---------------------
;
;	       Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME	CS:MONITOR_SEGMENT, DS:ROM_DATA, ES:NOTHING

	INCLUDE	../ROM/ROM.LIT
	INCLUDE ../ROM/IO.LIT
	INCLUDE ../ROM/INTR.LIT
	INCLUDE ../SYS/SYS.EXT
	INCLUDE ../DISK/DISK.LIT
	INCLUDE ../DISK/DISK.EXT
	INCLUDE ../VIDEO/VIDEO.LIT
	INCLUDE ../VIDEO/VIDEO.EXT
	INCLUDE	DEBUG.LIT
	INCLUDE	DEBUG.EXT

	EXTRN	SET_DEFAULT_VIDEO_MODE:NEAR, CONSOLE_OUTPUT:NEAR
	EXTRN	DISP_STRING:NEAR, DISP_HEX_WORD:NEAR, DISP_HEX_BYTE:NEAR
	EXTRN	DISP_HEX_POINTER:NEAR, GET_HEX_POINTER:NEAR, GET_HEX_WORD:NEAR
	EXTRN	GET_HEX_RANGE:NEAR, GET_LINE:NEAR, GET_CHAR:NEAR
	EXTRN	DISP_DEC_WORD:NEAR, SKIP_SPACES:NEAR, BEEP:NEAR
	EXTRN	TAB_POSN:NEAR, SPACE:NEAR, ASCII_TO_HEX:NEAR, MAP_UPPER:NEAR
	EXTRN	CHECK_CR:NEAR, BACK_SPACE:NEAR, CRLF:NEAR, NEXT_PARM:NEAR
	EXTRN	GET_KEY_BUFF:NEAR, GET_STRING:NEAR, STACK_TOP:WORD
	EXTRN	TEST_COMMAND:NEAR, SET_VIDEO_MODE:NEAR, SET_SCROLL_MODE:NEAR
	EXTRN	INIT_INTERRUPTS:NEAR, DISASM:NEAR, SET_INTR_PTR:NEAR
	EXTRN	GET_INTR_PTR:NEAR, DATA_SEGMENT:WORD, CURSOR_FLAG:BYTE
	EXTRN	INTR_FLAG_MOD:BYTE, HELP_MSG:BYTE
	EXTRN	MFM_150_MSG1:BYTE, MFM_150_MSG2:BYTE, REG_NAMES:BYTE
	EXTRN	FLAG_NAMES:WORD, CONTINUE_MSG:BYTE



;**********************************************************************
; MFM_150:
;
;	MFM_150 is the initial entry point to the MFM_150 Multi-
; Functional Monitor.  It initializes the video screen, prints a
; sign-on message, fills the user's CPU registers with 0's, and then 
; it returns control to the main command processor routine 
; 'Command_Loop'.
;
; NOTE: MFM_150 reinitializes the stack pointer to point to the internal
; 	monitor stack buffer - thus the calling program will not regain
; 	control.
;**********************************************************************
MFM_150 PROC NEAR
	PUBLIC	MFM_150
	CLI				;Disable interrupts
	CLD				;Set forward byte moves
	PUSH	CS			;Point SS to monitor's segment
	POP	SS
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
MFM1:	MOV	AL,IN_SERVICE_INTRS	;Get command to read in-service register
	OUT	INTERRUPT_CTRL,AL	;Send command to intr controller
	IN	AL,INTERRUPT_CTRL	;Check for interrupts-in-progress
	TEST	AL,AL			;Are any interrupts currently active?
	JZ	MFM2			;No, interrupt controller cleared
	MOV	AL,END_OF_INTR		;Send an end-of-interrupt command...
	OUT	INTERRUPT_CTRL,AL	;...to service the interrupt
	JMP	SHORT MFM1		;Loop until interrupts cleared
MFM2:	MOV	SP,OFFSET STACK_TOP	;Stack now points to monitor stack
	CALL	SET_DEFAULT_VIDEO_MODE	;Init and clear the video screen
	MOV	SI,OFFSET MFM_150_MSG1	;Print 'MFM_150 Monitor, Version x.y'
	CALL	DISP_STRING		;...   'Memory Size: '
	MOV	AX,MEMORY_SIZE		;Get the current memory size
	CALL	DISP_DEC_WORD		;Display it in decimal
	MOV	SI,OFFSET MFM_150_MSG2	;Print 'K bytes'
	CALL	DISP_STRING		;...   'Enter "?" for help'
	MOV	BREAKPOINT_COUNT,0	;Clear out breakpoint count
	PUSH	CS			;Point ES to the REGISTERS segment
	POP	ES
	MOV	DI,OFFSET REGISTERS	;Point to the REGISTERS structure
	MOV	CX,SIZE REGS		;Set CX to length of REGISTERS
	XOR	AX,AX			;Set AX to fill with 0
	REP	STOSB			;Set all CPU registers to 0
	MOV	WORD PTR DISPLAY_PTR,AX	;Initialize default display ptr to 0
	MOV	WORD PTR DISPLAY_PTR[2],AX
	MOV	WORD PTR UNASSEMBLE_PTR[2],CS   ;Set default disassembly address
	MOV	WORD PTR UNASSEMBLE_PTR,OFFSET MFM_150
	MOV	REGISTERS.RCS,CS	;Set user's CS to point to the
	MOV	REGISTERS.RIP,OFFSET MFM_150 ;MFM_150 routine (ie, monitor)
	JMP	COMMAND_LOOP		;Execute the debugger
MFM_150 ENDP



;**********************************************************************
; COMMAND_LOOP:
;
;	Command_Loop is the command processor for the MFM_150 monitor.
; It prints out a prompt, waits for a user response, and then branches
; to an appropriate command routine to service the command.
;
; NOTE: Command_Loop, like MFM_150, reinitializes the stack pointer -
;	thus the caller will not regain control.
;**********************************************************************
COMMAND_LOOP PROC NEAR
	PUBLIC	COMMAND_LOOP
	STI				;Enable interrupts
	CLD				;Use forward byte moves
	PUSH	CS			;Point SS to monitor segment
	POP	SS
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	MOV	SP,OFFSET STACK_TOP	;Set the monitor's stack pointer
	MOV	TRACE_COUNT,0		;Force abort if trap flag gets set
CL1:	MOV	SI,OFFSET PROMPT_MSG	;Print out the '->' prompt message
	CALL	DISP_STRING
	CALL	GET_LINE		;Get a command from the user
	MOV	SI,0			;Point to the start of the cmd line
	CALL	SKIP_SPACES		;Skip leading spaces in the command
	MOV	AL,COMMAND_LINE[SI]	;Get the command character
	CMP	AL,CR			;Is this a blank line?
	JE	CL1			;Yes - go get another command
	INC	SI			;Point to the next char in the line
	CALL	MAP_UPPER		;Convert char to upper case
	MOV	BX,0			;Clear offset into command table
	MOV	CX,NUMBER_OF_COMMANDS	;Get the number of possible commands
CL2:	CMP	AL,CS:CMD_TABLE[BX].CMD	;Is this the right command?
	JE	CL3			;Yes - execute the command routine
	ADD	BX,TYPE CMD_TABLE	;No, point to the next command entry
	LOOP	CL2			;Continue till command found
	JMP	SHORT CL4		;ERROR - command not recognized!
CL3:	CALL	CS:CMD_TABLE[BX].ROUTINE;Execute the right service routine
	JNC	CL1			;No error - get & do another command
CL4:	MOV	AX,SI			;Get the pointer to bad character
	MOV	AH,AL			;Place it in AH
	ADD	AH,LINE_START		;Add in length of prompt string
	DEC	AH			;Point back to char in error
	CALL	SPACE			;Advance cursor to point to bad cmd
	MOV	SI,OFFSET ERROR_MSG	;Print out '^ Invalid command!'
	CALL	DISP_STRING
	CALL	BEEP			;Beep to identify error
	JMP	CL1			;Go and get another command
COMMAND_LOOP ENDP



;**********************************************************************
; BOOT_COMMAND: (TEXT_PTR)
;
;	Boot_Command is called to boot an operating system from disk.
; The command syntax is:
;		B [{F | W}] [<drive>][:<boot_string>]
; Where: F refers to Floppy (this is the default), W to Winchester.
;	 <drive> is a drive number from 0-3.  The default is 3.
;	 <boot_string> is a sequence of characters used by some
;	 	mass-storage devices to determine which 'parti-
;	 	tion' to boot from.
;
; Input:
;	SI: Pointer to input string
;
; Output:
;	CY: Set to indicate that the boot command was invalid
;	SI: Pointing to first character determined to be in error.
;
; Note: This routine will return only if the boot attempt is not
;	successful, or a command line error is given.
;**********************************************************************
BOOT_COMMAND PROC NEAR
	PUSHREG	<AX,DI>
	CALL	GET_CHAR		;Get the first character
	CALL	MAP_UPPER		;Convert it to upper case
	MOV	AH,BOOT_DRIVE		;Initially, set drive to last booted
	CMP	AL,'F'			;Is this a boot-from-Floppy request?
	JNE	BC1			;No, continue
	AND	AH,7FH			;Yes - mask out winchester-select bit
	JMP	SHORT BC2		;Go get another character
BC1:	CMP	AL,'W'			;Is this a boot-from-Winchester?
	JNE	BC3			;No, try again
	OR	AH,80H			;Yes - set the mass-storage flag
BC2:	CALL	GET_CHAR		;Get the next char from the cmd line
BC3:	CMP	AL,'0'			;Is this a drive select?
	JB	BC4			;No, must be boot string
	CMP	AL,'3'			;Is this a valid drive?
	JA	BC4			;ERROR - invalid drive select
	SUB	AL,'0'			;Turn drive name into drive number
	AND	AH,11111100B		;Mask out old drive number in AH
	OR	AH,AL			;Select new drive number
	CALL	GET_CHAR		;Get the next char from the cmd line
BC4:	MOV	BOOT_DRIVE,AH		;Record boot drive number for boot
	CMP	AL,':'			;Is this a boot string separator?
	JNE	BC6			;No, must be end of line
BC5:	CALL	GET_CHAR		;Get a char from the command line
	MOV	BOOT_STRING,AL		;Record it in the boot string
	CMP	AL,CR			;Is it the end of the line?
	JE	BC7			;Yes - boot the operating system
	CMP	AL,'1'			;Is this a valid partition?
	JC	BC8			;ERROR - invalid partition
	CMP	AL,'4' + 1		;Is this within range?
	CMC				;Set the carry flag if not!
	JC	BC8			;ERROR - out-of-range partition
	CALL	GET_CHAR		;Yes - get the next character
BC6:	CMP	AL,CR			;Is this the end of the line?
	STC				;Set carry if not...
	JNZ	BC8			;ERROR - unrecognized info on line
BC7:	MOV	DL,BOOT_DRIVE		;Get the drive number as a parameter
	MOV	AL,BOOT_STRING		;...and the boot partition, too
	INT	BOOT_INTR		;Boot operating system from disk
BC8:	POPREG	<DI,AX>
	RET
BOOT_COMMAND ENDP



;**********************************************************************
; COLOR_COMMAND:
;
;	Color_Command displays a sixteen color color-bar which shows
; off the colors which can be generated on the Z-150.  This is quite
; useful for aligning a color monitor.
;
;**********************************************************************
COLOR_COMMAND PROC NEAR
	PUBLIC	COLOR_COMMAND
	PUSHREG	<AX,BX,CX,DX,SI>
	PUSH	WORD PTR VIDEO_MODE	;Save current video mode
	MOV	AL,3			;Choose 80x25 color / monochrome
CC0:	MOV	AH,0			;Set video mode function
	INT	VIDEO_IO_INTR		;Set video mode to 80x25 color
	MOV	BX,0			;Page 0, Foregrd & Backgrd = 0
	MOV	DX,0			;Row = 0
	MOV	AL,REV_VIDEO_BLANK	;Character to be displayed
	MOV	CX,5			;Number of chars for each color
CC1:	MOV	AH,2			;Set cursor position function
	INT	VIDEO_IO_INTR		;Update cursor position
	MOV	AH,9			;Write character with attributes func
	INT	VIDEO_IO_INTR		;Display 5 chars of specified color
	ADD	DX,CX			;Update col. pos. (no overflow to DH)
	INC	BL			;Go to next color
	CMP	BL,10H			;Have all colors been displayed?
	JC	CC1			;No - display chars in next color
	MOV	BL,0			;Reset color of character for display
	MOV	DL,BL			;Init column to 0
	INC	DH			;Update row position
	CMP	DH,NUMBER_OF_ROWS-3	;Has the screen been filled?
	JC	CC1			;No - repeat till done
	POP	CX			;Retrieve old video mode
	CALL	CRLF			;Display a CR-LF at the end of line
	CMP	CL,7			;Was the monochrome card the default?
	JE	CC2			;Yes - no video mode change
	CMP	CL,3			;Was the orig. video mode 80x25 color?
	JE	CC2			;Yes - no video mode change
	CMP	CL,2			;In 80x25 black and white?
	JE	CC2			;Yes - don't clear the screen
	MOV	SI,OFFSET CONTINUE_MSG	;Pointer to message to be displayed
	CALL	DISP_STRING		;Display the string
	MOV	AH,0			;Read character from keyboard function
	INT	KEYBOARD_IO_INTR	;Wait till a key is struck
	MOV	AL,CL			;Retrieve old video mode
	MOV	AH,0			;Set video mode function
	INT	VIDEO_IO_INTR		;Reset video mode to original mode
	CMP	CL,6			;Entering high-resolution graphics?
	JNE	CC2			;No, just quit
	MOV	CURSOR_FLAG,CURSOR_ENABLED ;Yes - turn on the graphics cursor
CC2:	CLC				;Clear the error flag
	POPREG	<SI,DX,CX,BX,AX>
	RET
COLOR_COMMAND ENDP



;**********************************************************************
; DISPLAY_COMMAND: (TEXT_PTR)
;
;	Display_Command is called to display the contents of system RAM.
; The syntax of the DISPLAY_MEMORY command ('D') is as follows:
;	    D [<start_address> [ [,<end_address>] | [L<length>] ] ]
; If the start address is not specified, the display will start with
; the byte following that which was last displayed.  If only an offset
; is given for the start address, then the contents of the DS register
; are used as the segment.  If only a start address is given, then a
; total of 80H (128) bytes will be displayed.  
;
; Input:
;       SI: Pointing to the first command line parameter.
;
; Output:
;	CY: Set if an error occurred (illegal command syntax).
;       SI: Pointing to the first character in error, if an error
;	    occurred.
;**********************************************************************
DISPLAY_COMMAND PROC NEAR
	PUBLIC	DISPLAY_COMMAND
	PUSHREG	<AX,CX,DI,ES>
	LES	DI,DISPLAY_PTR		;Get the default display offset
	MOV	CX,DEFAULT_DISPLAY_LENGTH ;Set the default display count
	CALL	CHECK_CR		;Is a default display requested?
	JC	DCMD1			;Yes - show bytes after last display
	MOV	AH,TRUE			;Set accept-default-length flag
	CALL	GET_HEX_RANGE		;Get the start address, range
	JC	DCMD10			;ERROR - invalid command format
	CALL	CHECK_CR		;At end of line?
	CMC				;Set carry if not...
	JC	DCMD10			;ERROR - garbage on line
DCMD1:	MOV	AX,DI			;Get the start display address
	ADD	AX,CX			;Point to next byte after displayed
	MOV	WORD PTR DISPLAY_PTR,AX    ;Write this new address in memory
	MOV	WORD PTR DISPLAY_PTR[2],ES ;Write the default segment, too
	PUSH	DI			;Save pointer to start of line
	PUSH	CX			;Save # of bytes to display, as well
	CALL	DISP_HEX_POINTER	;Display start address of line
	MOV	AH,2			;Display a couple of spaces
	CALL	SPACE
DCMD2:	MOV	AL,BYTE PTR ES: [DI]	;Retrieve a byte of memory
	CALL	DISP_HEX_BYTE		;Display it in ASCII
	INC	DI			;Point to the next byte
	DEC	CX			;Done with hex display for this line?
	JZ	DCMD4			;Yes - display in ASCII now
	MOV	AX,DI			;Get the current line pointer
	AND	AL,0FH			;Mask off horizontal char number
	CMP	AL,08H			;Are we halfway through the line?
	MOV	AL,' '			;[Assume not - get a space]
	JNZ	DCMD3			;No, continue
	MOV	AL,'-'			;Yes - display a dash instead
DCMD3:	CALL	CONSOLE_OUTPUT		;Write separator between bytes
	TEST	DI,0FH			;Done with this line (in hex)?
	JNZ	DCMD2			;No, continue
DCMD4:	MOV	AH,61			;Yes - tab out to ASCII field
	CALL	TAB_POSN
	POP	CX			;Restore character count
	POP	DI			;Restore pointer to start of line
DCMD5:	MOV	AL,BYTE PTR ES: [DI]	;Get a byte of memory
	AND	AL,7FH			;Mask out parity bit
	CMP	AL,' '			;Is this a control character?
	JB	DCMD6			;Yes - display a '.' instead
	CMP	AL,7FH			;Is this a DELETE?
	JNZ	DCMD7			;No, display char as is
DCMD6:	MOV	AL,'.'			;Use a '.' for non-displayable char
DCMD7:	CALL	CONSOLE_OUTPUT		;Display byte as ASCII character
	INC	DI			;Point to the next char to display
	DEC	CX			;Done with line yet?
	JZ	DCMD8			;Yes - print end-of-line characters
	TEST	DI,0FH			;At end of line yet?
	JNZ	DCMD5			;No, continue
DCMD8:	CALL	CRLF			;Display CR-LF for end of line
	JCXZ	DCMD9			;Return if done
	JMP	SHORT DCMD1		;Loop until all displayed
DCMD9:	CLC				;Clear error flag
DCMD10:	POPREG	<ES,DI,CX,AX>		;Restore registers
	RET
DISPLAY_COMMAND ENDP



;**********************************************************************
; EXAMINE_COMMAND: (TEXT_PTR)
;
;	Examine_Command is called to examine and optionally modify
; individual bytes of memory.  The syntax of the command is:
;			E <address>
; Examine_Command will then display an individual byte of memory,
; and prompt for input with a '.'.  If a hex number is entered it
; will be deposited in the byte location;  if a '-' is entered the
; previous location is displayed, or if a space is entered the next
; location will be displayed.  Entry of a carriage return terminates
; the modification process.
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurs.
;       SI: Pointing to text in error, if an error occurs.
;**********************************************************************
EXAMINE_COMMAND PROC NEAR
	PUBLIC	EXAMINE_COMMAND
	PUSHREG	<AX,CX,DI,ES>
	CALL	GET_HEX_POINTER		;Get the address of the first byte
	JC	ECMD1			;ERROR - bad pointer entered
	CALL	CHECK_CR		;Anything else on the line?
	CMC				;[Set the carry flag if so...]
	JNC	ECMD2			;No, display the byte to be modified
ECMD1:	JMP	ECMD12			;ERROR - return w/carry set
ECMD2:	CALL	DISP_HEX_POINTER	;Display address of the byte
	MOV	AH,2			;Display a couple of spaces
	CALL	SPACE
ECMD3:	MOV	AL,BYTE PTR ES: [DI]	;Get a byte from memory
	CALL	DISP_HEX_BYTE		;Display it as 2 ASCII chars
	MOV	AL,'.'			;Display the terminator...
	CALL	CONSOLE_OUTPUT		;...as a request for input
	MOV	CX,0			;Set result, digit count to 0
ECMD4:	CALL	GET_KEY_BUFF		;Get a character from the console
	CMP	AL,CR			;Is the character a carriage-return?
	JNE	ECMD5			;No, continue
	CALL	CRLF			;Yes - echo a newline on the console
	TEST	CL,CL			;Have any characters been entered?
	JZ	ECMD12			;No, return no-error condition
	MOV	BYTE PTR ES: [DI],CH	;Yes - write byte to memory
	JMP	SHORT ECMD12		;Return w/carry cleared
ECMD5:	CMP	AL,' '			;Is the char a 'go-forward'?
	JNE	ECMD7			;No, continue
	TEST	CL,CL			;Was a character entered?
	JZ	ECMD6			;No, don't write to memory!
	MOV	BYTE PTR ES: [DI],CH	;Yes - write the entered byte to mem
ECMD6:	INC	DI			;Increment the address
	TEST	DI,07H			;Just written last byte of a line?
	JZ	ECMD9			;Yes - go to the next line
	MOV	AL,TAB			;No, advance to the next tab stop
	CALL	CONSOLE_OUTPUT
	JMP	SHORT ECMD3		;Go on to the next byte of this line
ECMD7:	CMP	AL,'-'			;Is the char a 'go-back'?
	JNE	ECMD10			;No, continue
	CALL	CONSOLE_OUTPUT		;Yes - echo '-' to console
	TEST	CL,CL			;Was a character entered?
	JZ	ECMD8			;No, continue with the previous byte
	MOV	BYTE PTR ES: [DI],CH	;Yes - write the entered byte to mem
ECMD8:	DEC	DI			;Decrement the address
ECMD9:	CALL	CRLF			;Display carriage-return, line-feed
	JMP	SHORT ECMD2		;Go back to the previous byte
ECMD10:	CMP	AL,BS			;Was this a backspace?
	JNE	ECMD11			;No, continue
	CMP	CL,0			;Have we gotten any characters?
	JZ	ECMD4			;No, can't backspace!
	DEC	CL			;Yes - decrement character count
	CALL	BACK_SPACE		;Back up cursor by one
	SHR	CH,1			;Divide result by 16
	SHR	CH,1
	SHR	CH,1
	SHR	CH,1
	JMP	SHORT ECMD4		;Go on for another character
ECMD11:	CMP	CL,2			;Already entered two chars?
	JGE	ECMD4			;Yes - ignore the entered result
	MOV	AH,AL			;Save the character in AH
	CALL	MAP_UPPER		;Convert the char to upper case
	CALL	ASCII_TO_HEX		;Is the character a hex digit?
	JC	ECMD4			;No, ignore it
	SHL	CH,1			;Yes - multiply result * 16
	SHL	CH,1
	SHL	CH,1
	SHL	CH,1
	OR	CH,AL			;Include new hex digit in result
	INC	CL			;Increment the digits-entered count
	MOV	AL,AH			;Retrieve the character
	CALL	CONSOLE_OUTPUT		;Echo it on the display
	JMP	SHORT ECMD4		;Go get another character
ECMD12:	POPREG	<ES,DI,CX,AX>
	RET
EXAMINE_COMMAND ENDP



;**********************************************************************
; FILL_COMMAND: (TEXT_PTR)
;
;	Fill_Command is called to fill a range of memory with a
; set of constants.  The list of constants may be entered in ASCII
; and/or Hex.  During the fill operation, the list is automatically
; reused for the entire range of memory.  The syntax is:
;	F <start_address>{,<end_address> | L<length>},<data_list>
; where: <data_list> is:
;	{"<text>" | <byte_value>}[,"<text>" | <byte_value>]...	
; where: <text> is:
;	Any sequence of printable ASCII characters.
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurs.
;       SI: Pointing to text in error, if an error occurs.
;**********************************************************************
FILL_COMMAND PROC NEAR
	PUBLIC	FILL_COMMAND
	PUSHREG	<AX,BX,CX,DX,DI,ES>
	MOV	AH,0			;Set no-default-length flag
	CALL	GET_HEX_RANGE		;Set ES:DI to block addr, CX=length
	JC	FC3			;ERROR - invalid memory range
	PUSH	DI			;Save address of memory block
	MOV	DI,OFFSET STRING_BUFF	;Point to the fill string buffer
	CALL	GET_STRING		;Read the string from the cmd line
	POP	DI			;[Restore the memory address]
	JC	FC3			;ERROR - invalid string entered
	MOV	DX,BX			;Copy string length into DX
	MOV	BX,0			;Point to the start of the fill buff
FC1:	MOV	AL,STRING_BUFF[BX]	;Get a byte of the fill string
	INC	BX			;Point to the next fill byte
	STOSB				;Store it in the destination block
	CMP	BX,DX			;Out of fill bytes?
	JNE	FC2			;No, keep using them up
	MOV	BX,0			;Yes - start using them over again
FC2:	LOOP	FC1			;Repeat until memory filled
	CLC				;Clear error flag
FC3:	POPREG	<ES,DI,DX,CX,BX,AX>
	RET
FILL_COMMAND ENDP



;**********************************************************************
; GO_COMMAND: (TEXT_PTR)
;
;	Go_Command is used to execute a user program with optional
; breakpoints.  Up to eight optional breakpoints may be entered - 
; if the processor encounters one of the breakpoints, it will return
; control to the MFM_150 monitor and print out the contents of all
; user registers (see BREAKPOINT_INTERRUPT).  The optional <start_addr>
; is used to begin execution at a different location:  if none is entered,
; execution will resume at the location in the CS and IP registers.
; The syntax of the Go command is:
;		G [=<start_addr>][,<breakpoint_addr>]...
;
; 	Note that the calling program will not receive control if no
; errors are encountered - the user's program will begin instead.
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set (indicating that an error occurred).
;       SI: Pointer to invalid information.
;**********************************************************************
GO_COMMAND PROC NEAR
	PUBLIC	GO_COMMAND
	PUSHREG	<AX,BX,DI,ES>
	CALL	CHECK_CR		;Any commands on line?
	JC	GC1			;No, begin execution at old address
	MOV	ES,REGISTERS.RCS	;Yes - pick up current CS as default
	CMP	COMMAND_LINE[SI],'='	;Re-assigning the processor address?
	JNE	GC1			;No, continue
	INC	SI			;Yes - point past the '='
	CALL	GET_HEX_POINTER		;Pick up the new processor address
	JC	GC4			;ERROR - invalid processor address
	MOV	REGISTERS.RCS,ES	;Place entered segment in user reg
	MOV	REGISTERS.RIP,DI	;Record entered offset as well
GC1:	MOV	BX,0			;Set breakpoint pointer to 0
	MOV	BREAKPOINT_COUNT,0	;Clear number of breakpoints to 0
GC2:	CALL	CHECK_CR		;Any breakpoints to process?
	JC	GC3			;No, begin execution
	CMP	BREAKPOINT_COUNT,8	;Yes - entered too many breakpoints?
	CMC				;Set carry flag if so...
	JC	GC4			;ERROR - too many breakpoints!
	CALL	GET_HEX_POINTER		;Pick up a breakpoint from the cmd
	JC	GC4			;ERROR - invalid breakpoint entered
	MOV	BREAKPOINT[BX].UCS,ES	;Write segment into breakpoint table
	MOV	BREAKPOINT[BX].UIP,DI	;Write offset into breakpoint table
	MOV	AL,BYTE PTR ES: [DI]	;Get the byte at the breakpoint
	MOV	BYTE PTR BREAKPOINT[BX].UDATA,AL   ;Record it in the breakpoint table
	MOV	BYTE PTR ES:[DI],BREAKPOINT_OPCODE ;Place breakpoint in mem
	ADD	BX,SIZE BRKTAB		;Point to the next breakpoint entry
	INC	BREAKPOINT_COUNT	;Increment number of breakpoints
	JMP	SHORT GC2		;No, go pick up more breakpoints
GC3:	PUSH	DS			;Save the current data segment
	MOV	AX,CS			;Get the code segment in DS
	MOV	DS,AX
	MOV	SI,OFFSET BREAKPOINT_INTERRUPT ;Point to the breakpoint routine
	MOV	AL,BREAKPOINT_INTR	;Get the breakpoint intr vector
	CALL	SET_INTR_PTR		;Give control of breakpoints to MFM-150
	POP	DS			;Restore the data segment
	MOV	AH,FALSE		;Set normal-execution command
	JMP	EXECUTE			;Begin executing the user's program
GC4:	POPREG	<ES,DI,BX,AX>
	RET
GO_COMMAND ENDP



;**********************************************************************
; EXECUTE: (SINGLE_STEP_FLAG)
;
;	Execute is called in single-step or normal execution modes to
; begin executing a user's program.  Registers are set to those
; contained in the global REGISTERS structure.
;
;	Execute will not return to the caller.
;
; Input:
;	AH: If cleared (zero), execution will begin normally.  If
;	    AH is non-zero, however, execution will begin in single-
;	    step mode.
;**********************************************************************
EXECUTE PROC NEAR
	PUBLIC	EXECUTE
	CLI				;No interrupts here!
	MOV	INTR_FLAG_MOD,TRUE	;Assume that this will change INTR flag
	AND	REGISTERS.FLAGS,NOT TRAP_FLAG	;Remove any existing trap bit
	TEST	AH,AH			;Is single-step mode requested?
	MOV	AX,REGISTERS.FLAGS	;[Get the user's flags into AX]
	JNZ	EX1			;Yes - single step an instruction
	JMP	EX6			;No, execute program directly
;
; Test to see if instruction will modify interrupt-flag status
;
EX1:	PUSH	AX			;Yes - save the user's flags
	MOV	ES,REGISTERS.RCS	;Get the user's code segment register
	MOV	SI,REGISTERS.RIP	;...and the instruction pointer
	MOV	AL,ES:[SI]		;Get the opcode to be stepped through
	PUSH	CS			;Point ES to the code segment
	POP	ES
	MOV	DI,OFFSET INTR_MOD_TBL	;Point to opcodes which mod intr flag
	MOV	CX,4			;Number of opcodes in table
	REPNE	SCASB			;Does this modify the intr flag?
	JE	EX2			;Yes - let new flags be used
	MOV	INTR_FLAG_MOD,FALSE	;No, force old flags value to be used
;
; Simulate execution of INT nnn instruction
;
EX2:	CMP	AL,INT_CODE		;Is this a software interrupt opcode?
	JNE	EX3			;No, continue
	MOV	SS,REGISTERS.RSS	;Yes - get the user's stack segment
	MOV	SP,REGISTERS.RSP	;...and the stack pointer, too
	ADD	REGISTERS.RIP,2		;Point past the INT instruction
	PUSH	REGISTERS.FLAGS		;Fake the software interrupt
	PUSH	REGISTERS.RCS		;...by pushing the flags, code segment,
	PUSH	REGISTERS.RIP		;...and the instruction pointer
	MOV	ES,REGISTERS.RCS	;Get the user's code segment again
	MOV	AL,ES:[SI+1]		;Get the interrupt number
	CALL	GET_INTR_PTR		;Get pointer to interrupt routine
	MOV	REGISTERS.RCS,DS	;Save the new code segment
	MOV	REGISTERS.RIP,SI	;...and the new instruction pointer
	JMP	SHORT EX4		;Exit to the step routine
;
; Simulate execution of PUSHF instruction
;
EX3:	CMP	AL,PUSHF_CODE		;Is this a push-flags instruction?
	JNE	EX5			;No, can step it normally
	MOV	SS,REGISTERS.RSS	;Yes - get the user's stack segment
	MOV	SP,REGISTERS.RSP	;...and the stack pointer, too
	PUSH	REGISTERS.FLAGS		;Yes - fake execution of PUSHF
	INC	REGISTERS.RIP		;Point past the PUSHF instruction
EX4:	MOV	REGISTERS.RSS,SS	;Save the user's stack segment
	MOV	REGISTERS.RSP,SP	;...and the stack pointer
	JMP	STEP_CONTINUE		;EXIT DIRECTLY TO STEP ROUTINE!
EX5:	POP	AX			;Restore the user's flags image
	OR	AX,TRAP_FLAG		;Set trap flag in local copy of flags
	AND	AX,NOT INTR_FLAG	;Keep interrupts out during step only
;
; Trace/Execute normally
;
EX6:	MOV	BX,REGISTERS.RBX	;Now, get all
	MOV	CX,REGISTERS.RCX	; 	  ...of the
	MOV	DX,REGISTERS.RDX	;	        ...user's
	MOV	SI,REGISTERS.RSI	;		      ...registers
	MOV	DI,REGISTERS.RDI	;...into
	MOV	BP,REGISTERS.RBP	;    ...the
	MOV	SP,REGISTERS.RSP	;	...actual
	MOV	DS,REGISTERS.RDS	;	      ...CPU
	MOV	SS,REGISTERS.RSS	;		 ...registers!
	MOV	ES,REGISTERS.RES
	PUSH	AX			;Save the user's flags on the stack
	PUSH	REGISTERS.RCS		;Place the user's code segment on the stack
	PUSH	REGISTERS.RIP		;Write the offset, as well
	MOV	AX,REGISTERS.RAX	;Get the user's AX register
	IRET				;'Return' to the user's program!
EXECUTE ENDP



;**********************************************************************
; HELP_COMMAND:
;
;	Help_Command displays a quick summary of the commands available
; through the MFM_150 monitor.
;**********************************************************************
HELP_COMMAND PROC NEAR
	PUSHREG	<SI>			;Save SI
	MOV	SI,OFFSET HELP_MSG	;Point to the help message
	CALL	DISP_STRING		;Print it!
	POPREG	<SI>			;Restore SI
	RET
HELP_COMMAND ENDP



;**********************************************************************
; HEX_COMMAND: (TEXT_PTR)
;
;	Hex_Command is used to calculate the sum and difference of
; two hex numbers.  This is frequently used to calculate address of
; an array element, etc.  The syntax of the Hex command is:
;		H <offset>,<offset>
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurred.
;       SI: Pointer to invalid information if an error occurs.
;**********************************************************************
HEX_COMMAND PROC NEAR
	PUBLIC	HEX_COMMAND
	PUSHREG	<AX,CX,DX>
	CALL	GET_HEX_WORD		;Get the first offset
	JC	HC1			;ERROR - invalid first offset
	MOV	CX,AX			;Store the first offset in CX
	CALL	GET_HEX_WORD		;Get the second offset
	JC	HC1			;ERROR - invalid second offset
	CALL	CHECK_CR		;Is this the end of the line?
	CMC				;[Set carry flag if so...]
	JC	HC1			;ERROR - extra data on line
	MOV	DX,AX			;Store second offset in DX
	MOV	SI,OFFSET SUM_MSG	;Print out 'Sum: '
	CALL	DISP_STRING
	ADD	AX,CX			;Calculate the sum of the offsets
	CALL	DISP_HEX_WORD		;Display the sum
	MOV	SI,OFFSET DIFF_MSG	;Print out '  Diff: '
	CALL	DISP_STRING
	MOV	AX,CX			;Get the first offset
	SUB	AX,DX			;Calculate the difference of the offsets
	CALL	DISP_HEX_WORD		;Display difference on line
	CALL	CRLF			;Display trailing newline
	CLC				;Flag no-error condition
HC1:	POPREG	<DX,CX,AX>
	RET
HEX_COMMAND ENDP



;**********************************************************************
; INPUT_COMMAND: (TEXT_PTR)
;
;	Input_Command is used to read I/O ports directly.  The syntax
; is:
;				I <port>
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurred.
;       SI: Pointer to invalid information if an error occurs.
;**********************************************************************
INPUT_COMMAND PROC NEAR
	PUBLIC	INPUT_COMMAND
	PUSHREG	<AX,DX>
	CALL	GET_HEX_WORD		;Get the port number
	JC	ICMD1			;ERROR - invalid port number
	CALL	CHECK_CR		;Check to see if thats all
	CMC				;[Set carry flag if not...]
	JC	ICMD1			;ERROR - extra garbage on line
	MOV	DX,AX			;Get the port number in DX
	IN	AL,DX			;Read the port's contents
	CALL	DISP_HEX_BYTE		;Display it on the console
	CALL	CRLF			;Display a newline
	CLC				;Clear error flag
ICMD1:	POPREG	<DX,AX>
	RET
INPUT_COMMAND ENDP



;**********************************************************************
; MOVE_COMMAND: (TEXT_PTR)
;
;	Move_Command is used to transfer a sequence of bytes from
; one location to another.  In the event of an overlapping move,
; Move_Command will automatically perform the move in the direction
; which will prevent memory from being overwritten.  The syntax of
; the move command is:
;	M <src_start_address>,{L<length>|<src_end_offset>},<dest_ptr>
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurred.
;       SI: Pointer to invalid information if an error occurs.
;**********************************************************************
MOVE_COMMAND PROC NEAR
	PUBLIC	MOVE_COMMAND
	PUSHREG	<AX,BX,CX,SI,DI,DS,ES>
	MOV	AH,FALSE		;Flag to use no default length
	CALL	GET_HEX_RANGE		;Point ES:DI to memory, CX to length
	JC	MC5			;ERROR - invalid command entered
	PUSH	ES			;Save the source segment
	PUSH	DI			;Save the source offset
	CALL	GET_HEX_POINTER		;Point ES:DI to the dest address
	POP	AX			;[Restore the source offset in AX]
	POP	DS			;[Restore the source segment in DS]
	JC	MC5			;ERROR - invalid destination address
	MOV	SI,AX			;Place the source offset in SI
	DEC	CX			;Decrement the move count...
	MOV	AX,DS			;Get the source segment
	MOV	BX,ES			;...and the destination segment
	CMP	AX,BX			;Are source and dest in same segment?
	JNE	MC2			;No, can use normal forward move
	MOV	AX,SI			;Yes - get the start address
	ADD	AX,CX			;Point AX to byte following move
	CMP	SI,DI			;Is the source less than the dest?
	JNB	MC1			;No, source address is greater
	CMP	AX,DI			;Is there a forward overlap?
	JA	MC3			;Yes - must use backward move
	JMP	SHORT MC2		;No, can use normal move
MC1:	CMP	AX,DI			;Is there a backward overlap?
	JBE	MC3			;Yes - use backward move!
MC2:	REP	MOVSB			;Perform the forward byte move
	MOVSB				;Move last byte of string
	JMP	SHORT MC4		;Return
MC3:	ADD	SI,CX			;Point SI to last byte...
	DEC	SI			;...of source block
	ADD	DI,CX			;Point DI to last byte...
	DEC	DI			;...of dest block
	STD				;Set backward-move flag
	REP	MOVSB			;Move the entire block backward
	MOVSB				;Move last byte of block
MC4:	CLD				;Reset the forward-move flag
	CLC				;Clear the carry flag - no errors
MC5:	POPREG	<ES,DS,DI,SI,CX,BX,AX>
	RET
MOVE_COMMAND ENDP



;**********************************************************************
; OUTPUT_COMMAND: (TEXT_PTR)
;
;	Output_Command is called to output information directly to an
; output port.  The syntax is:
;			O <port>,<value>
;
; Input:
;       SI: Pointer to first command line parameter.
;
; Output:
;	CY: Set if an error is detected.
;       SI: Pointer to invalid information if an error does occur.
;**********************************************************************
OUTPUT_COMMAND PROC NEAR
	PUBLIC	OUTPUT_COMMAND
	PUSHREG	<AX,DX>
	CALL	GET_HEX_WORD		;Get the port number
	JC	OCMD1			;ERROR - invalid port number
	MOV	DX,AX			;Get the port number in DX
	CALL	GET_HEX_WORD		;Now read the value to output
	JC	OCMD1			;ERROR - invalid port contents
	CALL	CHECK_CR		;Check to see if thats all
	CMC				;[Set error flag if garbage on line]
	JC	OCMD1			;ERROR - extra data on line
	OUT	DX,AL			;Output value to the port
OCMD1:	POPREG	<DX,AX>
	RET
OUTPUT_COMMAND ENDP



;**********************************************************************
; REGISTER_COMMAND: (TEXT_PTR)
;
;	Register_Command is used to examine and change the CPU
; registers.  The syntax of the Register Command ('R') is:
;			R [<register>]
; If no register name is specified, all register's contents are
; shown.  If a register name is given, the current register
; contents are shown, and the user is prompted for a value to
; be placed into the register (a carriage-return returns without
; changing the register contents).
;
; Input:
;       SI: Pointing to the first command line parameter.
;
; Output:
;	CY: Set if an error occurred.
;       SI: Pointing to character where an error was detected.
;**********************************************************************
REGISTER_COMMAND PROC NEAR
	PUBLIC	REGISTER_COMMAND
	PUSHREG	<AX,BX>			;Save registers
	CALL	CHECK_CR		;Is this a display-registers command?
	JC	RCMD2			;Yes - display registers
	CALL	GET_REGISTER		;No, try to get a register name in BX
	JC	RCMD4			;ERROR - invalid register name
	CALL	CHECK_CR		;Anything past register name?
	CMC				;[Set error flag if so...]
	JC	RCMD4			;ERROR - garbage on line
	MOV	AH,':'			;Use ':' as reg/contents delimiter
	CALL	DISP_REGISTER		;Display register contents
	MOV	AL,'-'			;Display a dash for contents request
	CALL	CONSOLE_OUTPUT
	CALL	GET_LINE		;Read user response to request
	MOV	SI,0			;Point to the start of the line
	CALL	CHECK_CR		;Was a carriage-return alone entered?
	JC	RCMD3			;Yes - return
	CMP	BX,FLAGS		;Is this the flags register?
	JNZ	RCMD1			;No, get command as hex value
	CALL	MODIFY_FLAGS		;Yes - change the flags register
	JMP	SHORT RCMD4		;Return carry/no-carry status
RCMD1:	CALL	GET_HEX_WORD		;Get command line as a hex number
	JC	RCMD4			;ERROR - invalid contents entered
	MOV	REGISTERS[BX],AX	;Place entered value in user reg
	JMP	SHORT RCMD3		;Return
RCMD2:	CALL	DISP_ALL_REGISTERS	;Display all user CPU registers
	MOV	ES,REGISTERS.RCS	;Get the user's code segment
	MOV	DI,REGISTERS.RIP	;...and instruction pointer
	MOV	WORD PTR UNASSEMBLE_PTR[2],ES ;Init the disassembly segment
	MOV	WORD PTR UNASSEMBLE_PTR,DI    ;...and location counter
	MOV	CX,1			;Set disasm count to 1
	CALL	DISASM			;Display the current instruction
RCMD3:	CLC				;Flag no-error condition
RCMD4:	POPREG	<BX,AX>			;Restore registers
	RET
REGISTER_COMMAND ENDP



;**********************************************************************
; GET_REGISTER: (TEXT_PTR)
;
;	Get_Register is called to read a register name from the
; command line.  If the register is valid, the offset of the register
; in the 'REGISTERS' structure will be returned.
;
; Input:
;       SI: Pointer to start of register name.
;
; Output:
;	CY: If the carry flag is set upon return, then an invalid
;	    register name was entered.
;       SI: Pointer to invalid register name if an error occurs.
;	BX: If the register name was valid, then BX will contain
;	    the offset of the register contents in the 'REGISTERS'
;	    structure.
;**********************************************************************
GET_REGISTER PROC NEAR
	PUBLIC	GET_REGISTER
	PUSHREG	<AX,CX>
	MOV	AX,WORD PTR COMMAND_LINE[SI] ;Retrieve the register name
	CALL	MAP_UPPER		;Convert char to upper case
	XCHG	AH,AL			;Place bytes in right order
	CALL	MAP_UPPER		;Make sure case is correct!
	MOV	CX,15			;Get the maximum number of registers
	MOV	BX,0			;Clear register offset to 0
GR1:	CMP	AH,CS: REG_NAMES[BX]	;Does the first char match?
	JNE	GR2			;No, go on to the next
	CMP	AL,CS: REG_NAMES[BX+1]	;Check to see if second char matches
	JE	GR3			;Found entry - return no-error
GR2:	ADD	BX,TYPE REGISTERS	;Point to the next register
	LOOP	GR1			;Loop until found / not-found
	STC				;Set error flag - bad register name
	JMP	SHORT GR5		;Return
GR3:	CMP	BX,FLAGS		;SPECIAL CASE: Is this the PC alias?
	JBE	GR4			;No, return register number
	MOV	BX,RIP			;Yes - set register to IP (ins. ptr.)
GR4:	ADD	SI,2			;Point beyond the register name
	CLC				;Clear the error flag
GR5:	POPREG	<CX,AX>
	RET
GET_REGISTER ENDP



;**********************************************************************
; DISP_ALL_REGISTERS:
;
;	Disp_All_Registers is called to display the contents of all of
; the user's CPU registers.  The routine uses the register values
; which have been recorded in the global 'REGISTERS' structure.
;**********************************************************************
DISP_ALL_REGISTERS PROC NEAR
	PUBLIC	DISP_ALL_REGISTERS
	PUSHREG	<AX,BX,CX>
	MOV	CX,14			;Number of registers to display in CX
	MOV	BX,0			;Point to the first register
DAR1:	MOV	AH,'='			;Use an '=' between name / contents
	CALL	DISP_REGISTER		;Display a register's contents
	MOV	AL,' '			;Display a space
	CALL	CONSOLE_OUTPUT
	CMP	CX,7			;Finished with first line of display?
	JNE	DAR2			;No, continue
	CALL	CRLF			;Yes - go on to next line
DAR2:	ADD	BX,TYPE REGISTERS	;Point to the next register
	LOOP	DAR1			;Loop until all registers displayed
	CALL	CRLF			;Display a final CRLF
	POPREG	<CX,BX,AX>
	RET
DISP_ALL_REGISTERS ENDP



;**********************************************************************
; DISP_REGISTER: (REGISTER, DELIMITER)
;
;	Disp_Register is called to display a register's name, as
; well as its contents, separated by a user-specified delimiter.
;
; Input:
;	BX: Offset into register table.
;	AH: Delimiter to use between the register's name, and its
;	    contents.
;**********************************************************************
DISP_REGISTER PROC NEAR
	PUBLIC	DISP_REGISTER
	PUSHREG	<AX,BX,CX,DX>
	MOV	AL,CS:REG_NAMES[BX]	;Get the first char of the reg name
	CALL	CONSOLE_OUTPUT		;Display it
	MOV	AL,CS:REG_NAMES[BX+1]	;Get second char of register name
	CALL	CONSOLE_OUTPUT		;Display it, as well
	MOV	AL,AH			;Get the user-specified delimiter
	CALL	CONSOLE_OUTPUT		;Display it
	MOV	DX,REGISTERS[BX]	;Now, get the register contents
	CMP	BX,FLAGS		;Is this the flags register?
	JNE	DR4			;No, continue
	MOV	CX,16			;Yes - set the bit count
	MOV	BX,0			;Clear index into flag names
DR1:	PUSH	BX			;Save pointer to start of name
	SHR	DX,1			;Get a flag bit
	JNC	DR2			;Jump if flag not set
	ADD	BX,2			;Flag set - point to 'on' flag name
DR2:	CMP	CS: FLAG_NAMES[BX],0	;Is this a defined flag?
	JZ	DR3			;No, skip this one
	MOV	AL,' '			;Display a space after the flag name
	CALL	CONSOLE_OUTPUT
	MOV	AX,CS: FLAG_NAMES[BX]	;Get the flag name in AX
	XCHG	AL,AH			;Put bytes in right order
	CALL	CONSOLE_OUTPUT		;Display the first char of flag name
	MOV	AL,AH			;Display second char of flag name
	CALL	CONSOLE_OUTPUT
DR3:	POP	BX			;Restore ptr to start of name pair
	ADD	BX,4			;Point to the next flag name pair
	LOOP	DR1			;Loop until all flag names displayed
	JMP	SHORT DR5		;Return
DR4:	MOV	AX,REGISTERS[BX]	;Get the register's contents
	CALL	DISP_HEX_WORD		;Display it on the console
DR5:	POPREG	<DX,CX,BX,AX>
	RET
DISP_REGISTER ENDP



;**********************************************************************
; MODIFY_FLAGS: (TEXT_PTR)
;
;	Modify_Flags is called to permit modification of the processor
; flags register.  It scans a list of supplied flags for valid flag
; names, and then it sets/resets the specified bits.  Note that
; MODIFY_FLAGS will permit conflicting flags to be entered (ie, UP &
; DN) - the last conflicting flag is the one actually changed.
;
; Input:
;       SI: Pointer to text containing flag names.
;
; Output:
;	CY: The carry flag is set upon return if an invalid flag
;	    name is encountered.
;       SI: Will point to the error encountered, if an error occurs.
;**********************************************************************
MODIFY_FLAGS PROC NEAR
	PUBLIC	MODIFY_FLAGS
	PUSHREG	<AX,BX,CX,DX>
MF1:	MOV	CX,1			;Set the current-bit flag
	MOV	DL,FALSE		;Clear the set/reset flag
	MOV	BX,0			;Clear offset into FLAG_NAMES
	MOV	AX,WORD PTR COMMAND_LINE[SI] ;Get a flag name
	CALL	MAP_UPPER		;Convert a char to upper case
	XCHG	AH,AL			;Get the other char of the name
	CALL	MAP_UPPER		;Convert it to upper case, as well
MF2:	CALL	SKIP_SPACES		;Skip any leading spaces
	CMP	AX,CS: FLAG_NAMES[BX]	;Is this a valid flag name?
	JE	MF4			;Yes - found the right flag name
MF3:	ADD	BX,2			;Point to the next flag name
	XOR	DL,1			;Invert the set/reset flag
	JNZ	MF2			;Now going to do 'set' pass
	SHL	CX,1			;Just moved on to next bit - shift mask
	JNC	MF2			;Loop until found / all bits checked
	JMP	SHORT MF7		;ERROR - flag not found
MF4:	TEST	DL,DL			;Is this the reset pass?
	JNZ	MF5			;No, continue
	NOT	CX			;Invert the bit mask
	AND	REGISTERS.FLAGS,CX	;Clear out the specified flag
	JMP	SHORT MF6
MF5:	OR	REGISTERS.FLAGS,CX	;Set the specified flag
MF6:	ADD	SI,2			;Point to the next flag name
	CALL	SKIP_SPACES		;Skip any spaces or tabs
	CALL	CHECK_CR		;Read all flags yet?
	JNC	MF1			;No, read some more
	CLC				;Clear the error flag
MF7:	POPREG	<DX,CX,BX,AX>
	RET
MODIFY_FLAGS ENDP



;**********************************************************************
; SEARCH_COMMAND: (TEXT_PTR)
;
;	Search_Command is called to search a range of memory for a
; set of constants.  The list of constants may be entered in ASCII
; and/or Hex.  The syntax is:
;	S <start_address>{,<end_address> | L<length>},<data_list>
; where: <data_list> is:
;	{"<text>" | <byte_value>}[,"<text>" | <byte_value>]...	
; where: <text> is:
;	Any sequence of printable ASCII characters.
;
; Input:
;       SI: Pointer to command line parameters.
;
; Output:
;	CY: Set if an error occurs.
;       SI: Pointing to text in error, if an error occurs.
;**********************************************************************
SEARCH_COMMAND PROC NEAR
	PUBLIC	SEARCH_COMMAND
	PUSHREG	<AX,BX,CX,DX,DI,ES>
	PUSH	DS			;Save the data segment on the stack
	MOV	AH,0			;Set no-default-length flag
	CALL	GET_HEX_RANGE		;Set ES:DI to block addr, CX=length
	JC	SC4			;ERROR - invalid memory range
	PUSH	DI			;Save pointer to memory block
	MOV	DI,OFFSET STRING_BUFF	;Point to the search string buffer
	CALL	GET_STRING		;Read the string from the cmd line
	POP	DI			;[Restore memory pointer]
	JC	SC4			;ERROR - invalid string entered
SC1:	PUSH	CS			;Copy code segment into DS
	POP	DS
	DEC	CX			;Decrement search count by 1...
	MOV	AL,STRING_BUFF[0]	;Get the first character to find
	REPNE	SCASB			;Find an occurrence of this character
	JE	SC2			;Occurrence found - continue
	SCASB				;Compare last byte of block...
	CLC				;[Clear error flag if no match]
	JNE	SC4			;String not found - return
SC2:	MOV	SI,OFFSET STRING_BUFF[1];Point SI to the string
	PUSH	DI			;Save pointer to memory
	PUSH	CX			;Save length of memory remaining
	MOV	CX,BX			;Get string length in CX
	DEC	CX			;Decrement it - starting with byte 1
	REPE	CMPSB			;Compare the two strings
	POP	CX			;[Restore amount of memory left]
	POP	DI			;[Restore pointer to start of string]
	JNE	SC3			;No match here - continue
	POP	DS			;Restore the data segment
	DEC	DI			;Point back to the data byte found
	CALL	DISP_HEX_POINTER	;Display the found-string's address
	CALL	CRLF			;...with a newline
	PUSH	DS			;Re-save the data segment on stack
SC3:	INC	DI			;Point to the next byte of memory
	TEST	CX,CX			;Loop until all occurrences found
	JNZ	SC1
	CLC				;Clear error flag	
SC4:	POP	DS			;Restore the data segment
	POPREG	<ES,DI,DX,CX,BX,AX>
	RET
SEARCH_COMMAND ENDP



;**********************************************************************
; T_COMMAND: (TEXT_PTR)
;
;	T_Command is an intermediate entry used to determine if a
; command starting with a 'T' is a TRACE command, or a TEST command.
; It will determine which is intended, and then it will vector to the
; appropriate routine.
;
; Output:
;	CY: Set if an error occurs while executing the TRACE/TEST
;	    command.
;	SI: Pointing to invalid character in command line if an error
;	    occurs.
;**********************************************************************
T_COMMAND PROC NEAR
	MOV	AL,COMMAND_LINE[SI]	;Get second char from the cmd line
	CALL	MAP_UPPER		;Convert it to upper case
	CMP	AL,'E'			;Is this an 'E' (from tEst)?
	JNE	TC2			;No, exit
	MOV	AL,COMMAND_LINE[SI+1]	;Get third char from the cmd line
	CALL	MAP_UPPER		;Convert it to upper case
	CMP	AL,'S'			;Is this an 'S' (from teSt)?
	JNE	TC2			;No, must be a trace command
	MOV	AL,COMMAND_LINE[SI+2]	;Get fourth char from the cmd line
	CALL	MAP_UPPER		;Convert it to upper case
	CMP	AL,'T'			;Is this an 'T' (from tesT)?
	JNE	TC2			;No, execute trace command
TC1:
	CALL	TEST_COMMAND		;Execute TEST (menu-diagnostics)
	CLC				;Clear the error flag (can't happen)
	JMP	SHORT TC3		;Return
TC2:	CALL	TRACE_COMMAND		;Execute the single-step command
TC3:	RET				;Return to command processor
T_COMMAND ENDP



;**********************************************************************
; TRACE_COMMAND: (TEXT_PTR)
;
;	Trace_Command is used to single-step through a user's
; program a single instruction at a time.  After each instruction
; is executed, the contents of the CPU registers are displayed.
; Note that multiple instructions may be executed (via <trace_count>
; below) - after the last instruction is executed, control is returned
; to the user.  If no new processor address is entered, execution
; will begin at the address contained in the REGISTERS structure.  If
; no trace count is entered, a default value of 1 is used.  Syntax 
; is:
;		T [=<start_addr>][,<trace_count>]
;
; Input:
;	SI: Pointer into command line at first argument.
;
; Output:
;	CY: Set indicating error occurred.
;	SI: Pointer to invalid characters in command line.
;**********************************************************************
TRACE_COMMAND PROC NEAR
	PUBLIC	TRACE_COMMAND
	PUSHREG	<AX>
	CALL	CHECK_CR		;Any commands on line?
	JC	TRC1			;No, begin execution at old address
	MOV	ES,REGISTERS.RCS	;Yes - pick up current CS as default
	CMP	COMMAND_LINE[SI],'='	;Re-assigning the processor address?
	JNE	TRC1			;No, continue
	INC	SI			;Yes - point past the '='
	CALL	GET_HEX_POINTER		;Pick up the new processor address
	JC	TRC3			;ERROR - invalid processor address
	MOV	REGISTERS.RCS,ES	;Place entered segment in user reg
	MOV	REGISTERS.RIP,DI	;Record entered offset as well
TRC1:	CALL	CHECK_CR		;Was a trace count entered?
	MOV	AX,1			;[Set trace count to 1 if so...]
	JC	TRC2			;No, do one single-step
	CALL	GET_HEX_WORD		;No, pick up the trace count
	JC	TRC3			;ERROR - invalid trace count
	CALL	CHECK_CR		;At end of line yet?
	CMC				;[Set carry if so...]
	JC	TRC3			;ERROR - extra information in command
TRC2:	MOV	TRACE_COUNT,AX		;Record trace count in memory
	PUSH	DS			;Save the data segment
	MOV	AX,CS			;Get the code segment in DS
	MOV	DS,AX
	MOV	SI,OFFSET SINGLE_STEP_INTERRUPT ;Point to the trace routine
	MOV	AL,SINGLE_STEP_INTR	;Get the single-step (trace) vector #
	CALL	SET_INTR_PTR		;Give control of stepping to MFM-150
	POP	DS			;Restore the data segment
	MOV	AH,TRUE			;Set single-step flag
	JMP	EXECUTE			;Single-step the user's program!
TRC3:	POPREG	<AX>
	RET
TRACE_COMMAND ENDP



;**********************************************************************
; UNASSEMBLE_COMMAND: (TEXT_PTR)
;
;	Unassemble_Command is called to disassemble the contents of
; a program into source form.  The syntax of the UNASSEMBLE command 
; ('U') is as follows:
;	    U [<start_address> [ [,<end_address>] | [L<length>] ] ]
; If the start address is not specified, the display will start with
; the byte following that which was last disassembled.  If only an 
; offset is given for the start address, then the existing segment
; will be used.  If only a start address is given, then a total of 20H
; (32) bytes will be disassembled.
;
; Input:
;       SI: Pointing to the first command line parameter.
;
; Output:
;	CY: Set if an error occurred (illegal command syntax).
;       SI: Pointing to the first character in error, if an error
;	    occurred.
;**********************************************************************
UNASSEMBLE_COMMAND PROC NEAR
	PUBLIC	UNASSEMBLE_COMMAND
	PUSHREG	<AX,CX,DI,ES>
	LES	DI,UNASSEMBLE_PTR	;Get the default diassembly pointer
	MOV	CX,DEFAULT_DISASM_LENGTH ;Set the default disassembly count
	CALL	CHECK_CR		;Is a default disassembly requested?
	JC	UCMD1			;Yes - show bytes after last disasm
	MOV	AH,TRUE			;Set accept-default-length flag
	CALL	GET_HEX_RANGE		;Get the start address, range
	JC	UCMD2			;ERROR - invalid command format
	CALL	CHECK_CR		;At end of line?
	CMC				;Set carry if not...
	JC	UCMD2			;ERROR - garbage on line
UCMD1:	MOV	AX,DI			;Get the start display address
	ADD	AX,CX			;Point to next byte after displayed
	CALL	DISASM			;Call the disassembler
	MOV	WORD PTR UNASSEMBLE_PTR,DI    ;Update the location counter
	MOV	WORD PTR UNASSEMBLE_PTR[2],ES ;Write the new default segment
	CLC				;Flag no errors
UCMD2:	POPREG	<ES,DI,CX,AX>
	RET
UNASSEMBLE_COMMAND ENDP



;**********************************************************************
; VIDEO_COMMAND:
;
;	Video_Command is called to set a new video mode or scroll
; mode.  This permits the user to easily use the different video
; capabilities of the Z-150.  The syntax is:
;		V [M<mode>] [S<scroll>]
; Where: <mode> is the desired video mode, as follows:
;		0 - 40 x 25 Monochrome Text
;		1 - 40 x 25 Color Text
;		2 - 80 x 25 Monochrome Text
;		3 - 80 x 25 Color Text
;		4 - Medium Resolution Graphics, Color
;		5 - Medium Resolution Graphics, Monochrome
;		6 - High Resolution Graphics
; ...and <scroll> is the desired scroll mode:
;		0 - Compatible (software) scrolling (all video modes)
;		1 - Hardware (jump) scrolling (not for 40x25)
;		2 - Smooth scrolling (only for high-res)
;**********************************************************************
VIDEO_COMMAND PROC NEAR
	PUBLIC	VIDEO_COMMAND
	PUSHREG	<AX>
VC1:	CALL	GET_CHAR		;Get the first char from the cmd line
	CALL	MAP_UPPER		;Convert it to upper case
	CMP	AL,'M'			;Is the video mode being set?
	JNE	VC3			;No, continue
	CALL	GET_HEX_WORD		;Yes - get the video mode
	JC	VC7			;ERROR - invalid mode entered
	CMP	AX,7 + 1		;Is this a valid video mode?
	CMC				;Set carry flag if not
	JC	VC7			;ERROR - invalid video mode
	AND	IO_CONFIG,NOT VIDEO_MODE_MASK ;Clear out the old video spec
	OR	IO_CONFIG,VIDEO_MODE_80;Assume that color 8025 will be used
	CMP	AL,7			;Specifying monochrome video mode?
	JNE	VC2			;No, enter a color video mode
	OR	IO_CONFIG,VIDEO_MODE_MONO;Yes - force monochrome as current brd
VC2:	CALL	SET_VIDEO_MODE		;Change to the requested video mode
	CMP	AL,6			;Entering high-res graphics mode?
	JNE	VC1			;No, finished with mode set
	MOV	CURSOR_FLAG,CURSOR_ENABLED ;Yes - turn the software cursor on
	JMP	SHORT VC1		;Go and get rest of line
VC3:	CMP	AL,'S'			;Is this a set-scroll-mode command?
	JNE	VC6			;No, check for machine mode select
	CALL	GET_HEX_WORD		;Get the desired scroll mode
	JC	VC7			;ERROR - invalid scroll mode!
	CMP	AX,2 + 1		;Is this a valid scroll mode?
	CMC				;Set the carry flag if not
	JC	VC7			;ERROR - illegal scroll mode
	CALL	SET_SCROLL_MODE		;Change to the requested scroll mode
	JMP	SHORT VC1		;Go and get more options
VC6:	SUB	AL,CR			;Is this the end of the line?
	ADD	AL,0FFH			;Set the carry flag if not...
VC7:	POPREG	<AX>
	RET
VIDEO_COMMAND ENDP



;**********************************************************************
; BREAKPOINT_INTERRUPT:
;
;	Breakpoint_Interrupt is given control when the CPU executes
; a software breakpoint instruction (ie, when a breakpoint has been
; 'hit'.  It will print out a message, display the registers, and
; return control to the MFM_150 monitor.  If entered at 'SOFTWARE_
; BREAKPOINT', it will not assume that a breakpoint instruction
; caused the interrupt.
;**********************************************************************
BREAKPOINT_INTERRUPT PROC FAR
	PUBLIC	BREAKPOINT_INTERRUPT, SOFTWARE_BREAKPOINT
	MOV	REGISTERS.RAX,AX	;Save user's AX register
	POP	AX			;Get user's Instruction Pointer
	DEC	AX			;Point back to breakpoint location
	MOV	REGISTERS.RIP,AX	;Record new program counter
	JMP	SHORT BI1		;Finish executing breakpoint

SOFTWARE_BREAKPOINT:

	MOV	REGISTERS.RAX,AX	;Save user's AX register
	POP	AX			;Get user's Instruction Pointer
	MOV	REGISTERS.RIP,AX	;Record it in the REGISTERS structure
BI1:	POP	AX			;Place user's CS in AX
	MOV	REGISTERS.RCS,AX	;Save...
	MOV	REGISTERS.RBX,BX	; ...contents
	MOV	REGISTERS.RCX,CX	;	  ...of
	MOV	REGISTERS.RDX,DX	;	    ...user's
	MOV	REGISTERS.RSI,SI	;		  ...registers
	MOV	REGISTERS.RDI,DI	;...in the
	MOV	REGISTERS.RBP,BP	;      ...global
	MOV	REGISTERS.RDS,DS	;	     ...REGISTERS
	MOV	REGISTERS.RES,ES	;		      ...structure
	POP	AX			;Get flags register
	MOV	REGISTERS.FLAGS,AX	;Record flags in REGISTERS
	MOV	REGISTERS.RSP,SP	;Record user's stack segment...
	MOV	REGISTERS.RSS,SS	;...and stack pointer
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	PUSH	CS			;Set up the monitor's stack segment
	POP	SS
	MOV	SP,OFFSET STACK_TOP	;Set the monitor's stack pointer
	STI				;Enable keystrokes to occur
	CLD				;Clear the direction flag
	CMP	BREAKPOINT_COUNT,0	;Any breakpoints to remove?
	JE	BI3			;No, skip breakpoint scan
	MOV	SI,0			;Point to start of breakpoint table
BI2:	MOV	ES,BREAKPOINT[SI].UCS	;Pick up a breakpoint's segment
	MOV	DI,BREAKPOINT[SI].UIP	;...so ES:DI points to the breakpoint
	MOV	AL,BYTE PTR BREAKPOINT[SI].UDATA ;Get the instruction at the breakpoint
	MOV	BYTE PTR ES: [DI],AL	;Place it back where it belongs
	ADD	SI,SIZE BRKTAB		;Point to the next breakpoint
	DEC	BREAKPOINT_COUNT	;Restored all user breakpoints yet?
	JNZ	BI2			;No, loop until finished
BI3:	CALL	CRLF			;Yes - display a blank line
	CALL	DISP_ALL_REGISTERS	;Display contents of the CPU registers
	MOV	ES,REGISTERS.RCS	;Pick up the user's code segment
	MOV	WORD PTR UNASSEMBLE_PTR[2],ES ;Record code seg for disassembly
	MOV	DI,REGISTERS.RIP	;Record Instruction Pointer, too
	MOV	WORD PTR UNASSEMBLE_PTR,DI
	MOV	CX,1			;Set count to 1 byte
	CALL	DISASM			;Disassemble code at user's CS:IP
	JMP	COMMAND_LOOP		;Return to MFM_150 monitor
BREAKPOINT_INTERRUPT ENDP



;**********************************************************************
; SINGLE_STEP_INTERRUPT:
; STEP_CONTINUE:
;
;	Single_Step_Interrupt is called when a user program finishes
; executing a single-stepped instruction.  It will display the
; user's registers and decrement the single-step count.  If the
; trace count becomes zero, execution is terminated and control is
; returned to the MFM_150 command processor - otherwise, execution
; continues in single-step mode.
;
;	Step_Continue is called when an instruction has been 'faked'.
; It will display registers, etc, and continue stepping.
;**********************************************************************
SINGLE_STEP_INTERRUPT PROC NEAR
	PUBLIC	SINGLE_STEP_INTERRUPT
	MOV	REGISTERS.RAX,AX	;Save
	MOV	REGISTERS.RBX,BX	; ...contents
	MOV	REGISTERS.RCX,CX	;	  ...of
	MOV	REGISTERS.RDX,DX	;	    ...user's
	MOV	REGISTERS.RSI,SI	;		  ...registers
	MOV	REGISTERS.RDI,DI	;...in the
	MOV	REGISTERS.RBP,BP	;      ...global
	MOV	REGISTERS.RDS,DS	;	     ...REGISTERS
	MOV	REGISTERS.RES,ES	;		      ...structure
	POP	DI			;Get user's Instruction Pointer
	MOV	REGISTERS.RIP,DI	;Record it in the user REGISTERS
	POP	ES			;Place user's code segment in ES
	MOV	REGISTERS.RCS,ES	;Record it in REGISTERS
	MOV	WORD PTR UNASSEMBLE_PTR[2],ES ;Record code segment for disasm
	MOV	WORD PTR UNASSEMBLE_PTR,DI    ;...and the IP, too
	POP	AX			;Get the user's flags register
	CMP	INTR_FLAG_MOD,TRUE	;Was the INTR flag modified?
	JE	SSI1			;Yes - use flags resulting from step
	AND	REGISTERS.FLAGS,INTR_FLAG ;Set old flags to INTR status only
	OR	AX,REGISTERS.FLAGS	;Restore the old interrupt status
SSI1:	MOV	REGISTERS.FLAGS,AX	;Record flags in REGISTERS
	MOV	REGISTERS.RSP,SP	;Record the user's stack segment
	MOV	REGISTERS.RSS,SS	;...and stack pointer
STEP_CONTINUE:
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	PUSH	CS			;Set up the monitor's stack segment
	POP	SS
	MOV	SP,OFFSET STACK_TOP	;Set the monitor's stack pointer
	STI				;Permit keystrokes to occur
	CLD				;Clear the direction flag
	CALL	DISP_ALL_REGISTERS	;Display all of the CPU registers
	MOV	ES,REGISTERS.RCS	;Point to the user's code segment
	MOV	DI,REGISTERS.RIP	;Point to the current instruction
	MOV	CX,1			;Set count to 1 byte
	CALL	DISASM			;Disassemble code at CS:IP
	CMP	TRACE_COUNT,0		;Done single-stepping?
	JZ	SSI2			;Yes - exit to the monitor
	DEC	TRACE_COUNT		;Finished single-stepping yet?
	JNZ	SSI3			;No, keep it up!
SSI2:	JMP	COMMAND_LOOP		;Yes - exit to the MFM_150 monitor
SSI3:	CALL	CRLF			;Display a blank line after registers
	MOV	AH,TRUE			;Set the single-stepping flag
	JMP	EXECUTE			;Do it all over again
SINGLE_STEP_INTERRUPT ENDP



;**********************************************************************
;		    D E B U G G E R   M E S S A G E S
;**********************************************************************
PROMPT_MSG	DB	'->',0

SUM_MSG		DB	'Sum: ',0
DIFF_MSG	DB	'  Diff: ',0

ERROR_MSG	DB	'^ Invalid Command!',CR,LF,0



;**********************************************************************
;	 	  I N T R   F L A G   M O D I F I E R S
;**********************************************************************
		PUBLIC	INTR_MOD_TBL
INTR_MOD_TBL	EQU	THIS BYTE
		STI			;These instructions modify...
		CLI			;...the user's interrupt-
		IRET			;...enable flag.
		POPF


;**********************************************************************
;	       C O M M A N D   D I S P A T C H   T A B L E
;**********************************************************************
		PUBLIC	CMD_TABLE
CMD_TABLE	CMDTBL	<'?', HELP_COMMAND>		;Help
		CMDTBL	<'B', BOOT_COMMAND>		;Boot from disk
		CMDTBL	<'C', COLOR_COMMAND>		;Color bar
		CMDTBL	<'D', DISPLAY_COMMAND>		;Display memory
		CMDTBL	<'E', EXAMINE_COMMAND>		;Examine/change RAM
		CMDTBL	<'F', FILL_COMMAND>		;Fill memory
		CMDTBL	<'G', GO_COMMAND>		;Execute program
		CMDTBL	<'H', HEX_COMMAND>		;Hex arithmetic
		CMDTBL	<'I', INPUT_COMMAND>		;Input from port
		CMDTBL	<'M', MOVE_COMMAND>		;Move memory contents
		CMDTBL	<'O', OUTPUT_COMMAND>		;Output to port
		CMDTBL	<'R', REGISTER_COMMAND>		;Examine/change regs
		CMDTBL	<'S', SEARCH_COMMAND>		;Search memory
		CMDTBL	<'T', T_COMMAND>		;Trace/Test command
		CMDTBL	<'U', UNASSEMBLE_COMMAND>	;Disassembler
		CMDTBL	<'V', VIDEO_COMMAND>		;Set video/scroll/mode
NUMBER_OF_COMMANDS	EQU ($ - CMD_TABLE) / SIZE CMDTBL



MONITOR_SEGMENT ENDS

	END

