Page	60,132
Title	-IBM 3101 Terminal Emulator
;
;	Copyright (c) 1985
;	Morrow Designs, Inc.
;	San Leandro, California
;
;	Last Update 31_Oct_85
;
;
%Out	Terminal.asm
page
;--------------------------------------------------------------------------
; Equates (22_May_85)
;--------------------
;
; Include Files
;--------------
;	Rom.lit
;	Intr.lit
;	Io.lit
;	Icon.lit
;
	.xlist
	Include ..\Rom\Rom.lit
	Include ..\Rom\Intr.lit
	Include ..\IO\Io.lit
	Include Icon.lit
	.list

Last_Row	equ	23		; last row of screen 0-23
Last_Col	equ	79		; last column 0-79
Video_Int	equ	10h		; video interrupt
Serial_Int	equ	14h		; serial interrupt
Kbrd_Int	equ	16h		; keyboard interrupt
Carr_Status	equ	080h		; DCD bit

page
;======================================================================
Screen_Segment Segment At (0B800h)
;=================================
;
	;Define Screen Memory Segment

Screen_Segment	EndS


;======================================================================
Monitor_Segment Segment Word Public
;==================================
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:nothing

	Extrn	Stk_Phone:	Word
	Extrn	Change_State:	Near
	Extrn	System_Setup:	Word
	Extrn	Beep:		Near

	Extrn	Set_Video_Mode:Near
	Extrn	Set_Cursor_Position:Near
	Extrn	Disp_Char_Attr:Near
	Extrn	DS_to_Rom_Data:Near

	Extrn	Get_Key_Status:Near
	Extrn	Get_Key_Buff:Near

	Extrn	Term_Col:	Byte	; current screen col
	Extrn	Term_Row:	Byte	; current screen row
	Extrn	Term_Attr:	Byte	; attribute for screen
	Extrn	Term_Port:	Byte	; com1 / com2 port number
	Extrn	Term_Carr:	Byte	; hangup on carrier loss if true
	Extrn	Term_Esc_Rcvd:	Byte	; true if escape processing
	Extrn	Term_Sbufp:	Word	; next store pointer
	Extrn	Term_Rbufp:	Word	; next remove pointer
	Extrn	Term_Bufc:	Word	; number of chars in buffer
	Extrn	Term_Sbuf:	Byte	; ring buffer for com chars
	Extrn	Term_Sbufe:	Byte	; end of ring buffer

page
;----------------------------------------------------------------------
; Fixed Data Area
;----------------
;
;
Esc_Table Label	Byte
;-------------------
; 	Table of valid escape and control codes.  This table is searched
;	by the escape processor routine.  If a match is found, then the
;	escape processor vectors to the appropriate handler for the code.
;	If no match is found, then the code is ignored.
;
	db	07h	; bell
	db	08h	; Backspace
	db	09h	; Tab
	db	0ah	; LF
	db	0dh	; CR
	db	'H'	; HOME
	db	'C'	; right
	db	'D'	; left
	db	'B'	; down
	db	'A'	; up
	db	'Y'	; set cursor position
	db	'L'	; clrscrn
	db	'J'	; ClrEos
	db	'I'	; ClrEol
	db	'N'	; InsLine
	db	'O'	; DelLine
	db	'Q'	; DelChar
	db	'P'	; InsChar
	db	'5'	; Send Cursor Location
Esc_Size equ	($ - Esc_Table)
	

Esc_Vect label word
;------------------
;	Vector table for escape and control sequences.  After finding the
;	command in the escape table, the escape handler routine vectors
;	to the appropriate routine through this table of pointers.
;
	dw	Beep			;System Bell Function
	dw	left
	dw	Tab_Code
	dw	Lf_Code
	dw	Cr_Code
	dw	home
	dw	right
	dw	left
	dw	down
	dw	up
	dw	Direct_Cur
	dw	Clr_Scr
	dw	Del_EOS
	dw	Del_EOL
	dw	Ins_Line
	dw	Del_Line
	dw	Del_Char
	dw	Ins_Char
	dw	Send_Curs

page
;----------------------------------------------------------------------
Term_Entry Proc Near; (11_Jul_85)
;--------------------------------
;	1) This code initializes the variables used by the 3101 terminal
;	   emulator.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Monitor_Segment
Public	Term_Entry

	push	cs
	push	cs
	pop	ds				; DS:= Monitor Segment
	pop	es				; ES:= Monitor Segment
	mov	word ptr Term_Col, 0		; init row/col
	mov	Term_Attr,7			; init screen write attribute
	mov	Term_Esc_Rcvd,0			; init to no esc sequence
;***	mov	Term_Port,0			; set by Phone Icon code
	mov	Term_Sbufp,offset Term_Sbuf	; init ringbuffer store ptr
	mov	Term_Rbufp,offset Term_Sbuf	; init ringbuffer remove ptr
	mov	Term_Bufc,0			; init ringbuffer char count
	mov	sp,Offset Stk_Phone		; Initialize the Stack Pointer

	mov	al,3				; 80x25 Alpha Numeric
	call	Set_Video_Mode			; Set the Video Mode

	mov	ah,Icon_Phone-2			; AH:= New Task Number
	mov	al,1				; AL:= New State
	call	Change_State			; Change State

	mov	cx,500h				; CX:= Delay for State Change
TLp1:	loop	TLp1				; Wait for State to Change
	jmp	Term_Main			; Start the Terminal Emulator

Term_Entry	EndP

page
;----------------------------------------------------------------------
Term_Main Proc Near; (11_Jul_85)
;-------------------------------
;	This is the main loop.  It checks for input from the
;	serial port and sends it to the character dispatcher to
;	be handled and or displayed.  It also checks the keyboard
;	for input and sends any characters out the serial port.
;	If Half Duplex is enabled, it will aslo echo the characters
;	to the screen.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:nothing

	; get and process COM port -> screen characters
TeLp1:	call	Ser_In			; see if a character exists
	test	al,al			; check status
	jz	TeSk1			; if no character, then check keyboard
	call	Disp_Char		;  else, display character rcvd on screen

	; get and process keyboard -> COM port characters
TeSk1:	call	Get_Key_Status		;If (Keyboard Status eq	Active)
	jz	TeLp1

	call	Get_Key_Buff		;	Read Key (AL=Ascii AH=Scan)

	; test for and process function keys F1 thru F8,
	; map them to ESC, 'a' thru 'h', RETURN
	test	al,al			; function key hit ?
	jnz	TeSk2			; skip if not

	sub	ah,03Bh			; remove F1 value
	js	TeLp1			; skip if < F1

	cmp	ah,7			; only allow up to F8
	ja	TeArr			; Not a function key, check if arrow

	push	ax			; save function key number
	mov	al,Esc
	call	Ser_Out			; map function keys to ESC
	pop	ax			; restore function key number

	mov	al,ah
	add	al,'a'			; map number to 'a' thru 'h'
	call	Ser_Out
	mov	al,Cr
	call	Ser_Out			; end with a RETURN
	jmp	TeLp1

TeArr:	sub	ah,0dh			; Arrows start 0dh above F1
	js	TeLp1			; Not an arrow key
	mov	al,0bh			; 0 = UP = ^K = 0bh
	jz	TeSk2			;   so send ^K if UP

	cmp	ah,8			; see if above range
	ja	TeLp1			; Not an arrow key
	mov	al,0ah			; 8 = DOWN = ^J = 0ah
	jz	TeSk2			;   so send ^J if down

	mov	al,0ch			; al = ^L = RIGHT
	cmp	ah,5
	ja	TeLp1			; Not Right or Left
	jz	TeSk2			; send ^L if RIGHT arrow key

	cmp	ah,3			; last case is Left Arrow 
	jnz	TeLp1			;   if not LEFT then return
	mov	al,08h			;   else, send a ^H
					; fall thru to TeSk2
	
	; send normal characters to COM port
TeSk2:	call	Ser_Out			; send to port
	jmp	TeLp1			; and loop

Term_Main	EndP

page
;----------------------------------------------------------------------
Ser_In Proc Near; (21_May_85)
;----------------------------
;	checks com line input ringbuffer and returns with al= 0 if
;	no characters input, else al = char. received.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:nothing

	call	Ser_Buf			; loadup ringbuffer in case empty
	mov	ax,Term_Bufc		; get current ring buffer count
	test	al,al			; anything io buffer ?
	jz	empty			; exit if not
	mov	si,Term_Rbufp		; get next remove pointer
	cld				; forward moves
	lodsb				; get next char
	cmp	si,offset Term_Sbufe	; at end of buffer ?
	jne	noend			; skip if not
	mov	si,offset Term_Sbuf	; restart at beginning

noend:	mov	Term_Rbufp,si		; set new remove pointer
	dec	Term_Bufc		; remove from count

empty:	ret				; return char or zero in al

Ser_In	EndP

page
;----------------------------------------------------------------------
Ser_Buf Proc Near; (21_May_85)
;-----------------------------
;	checks COM1 or COM2 depending on what is selected and buffers any
;	input chars in ring buffer
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Monitor_Segment

	mov	ah,3			; get status of com port
	mov	dl,Term_Port		; selected by Phone Icon
	mov	dh,0			; clear high byte
	int	Serial_Int		; ah = line control status, al = modem status

	test	Term_Carr,0FFh		; are we checking for carrier ?
	jz	nocck			; skip if not
	and	al,Carr_Status		; isolate conection indication bits
	cmp	al,Carr_Status		; we still connected ?
	je	nocck			; skip if so

	mov	al,0			; AL:= New State Number (state 0)
	mov	ah,Icon_Phone-2		; AH:= 0 Based Task Number (Phone Icon)
	call	Change_State		; invoke change state routine
Hang:	jmp	Hang			; loop if returned

nocck:	mov	al,0			; set to return no char flag
	and	ah,00000001b		; check if char available
	jz	No_Char			; if not, al is 0 and return
	mov	ah,02			; else, set to get character
	int	Serial_Int		; and read it to al
					; ah = sio status, al = char
	and	al,07fh			; mask off 8th bit 
	cmp	Term_Bufc,256		; is buffer full ?
	je	No_Char			; ignore char if ringbuffer overflow
	mov	di,Term_Sbufp		; get next store pointer
	cld				; forward moves

	push	cs
	pop	es			; ES:= Base of Monitor Segment
	stosb				; buffer the char
	cmp	di,offset Term_Sbufe	; at end of ring buffer ?
	jne	nowrap			; skip if not
	mov	di,offset Term_Sbuf	; restart at buffer beginning

nowrap:	mov	Term_Sbufp,di		; set new ring buffer store pointer
	inc	Term_Bufc		; count the chars in ringbuffer

No_Char: ret

Ser_Buf		EndP

page
;----------------------------------------------------------------------
Ser_Out Proc Near; (21_May_85)
;-----------------------------
;	checks if a character can be sent. If a character can be sent,
;	then it is.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:nothing

	mov	bl,al			; save char in bl
Retry:	mov	ah,3			; get status of com port
	mov	dl,Term_Port		; selected by Phone Icon
	mov	dh,0			; clear high byte
	int	Serial_Int
					; ah = line control status
	and	ah,00100000b		; check if char can be sent
	jnz	Xmt_Act			; skip if can send
	push	bx			; save char
	call	Ser_Buf			; catch any input chars from COM port
	pop	bx			; restore char
	jmp	Retry			; try to send again

Xmt_Act: mov	ah,01			; else, set to send character
	mov	al,bl
	int	Serial_Int		; and send it from al
	ret				; ah = xmit status

Ser_Out		EndP

page
;----------------------------------------------------------------------
Disp_Char Proc Near; (11_Jul_85)
;-------------------------------
;	Display or process the character in AL.
;
	test	Term_Esc_Rcvd,0ffh	; check if in escape sequence
	jz	Not_Seq			; if zero, then not in escape sequence
	js	In_DCA			; if bit 7 set then receiving DCA
	jmp	Esc_Hndlr		; if > 0 then in escape so process

Not_Seq:
	cmp	al,' '			; see if control character
	jb	ctrl_Char		; if so, then handle it. Otherwise.....

	; write normal characters to the screen
	mov	cx,1			; only 1 character
	mov	bh,0			; page 0
	mov	bl,Term_Attr		; whatever current attribute is

	push	ds
	call	DS_to_Rom_Data		;DS:= Rom Data
	call	Disp_Char_Attr		;Print Character on the Screen
	pop	ds

	mov	ax,word ptr Term_Col	; adjust column
	inc	al			; and increment it.
	cmp	al,Last_Col+1
	jnz	Same_Line		; if not, then just restore it

	inc	ah
	cmp	ah, Last_Row+1
	jnz	Not_EOS			; if at end of screen,

	mov	al,0			; make col 0
	mov	ah,Last_Row		;   make current row 24

	push	ax
	call	Set_Cur			;   Position Cursor @ Col-0, Row-24
	pop	ax

	jmp	Lf_Code			; make like a line feed
Not_EOS:
	xor	al,al			; and set column 0
Same_Line:
	jmp	Set_Cur			; be sure to update cursor position!

	; process control characters
Ctrl_Char:				; handle control character
	cmp	al,ESC			; see if escape rcvd.
	jnz	Not_Esc			; If char was escape
	inc	Term_Esc_Rcvd		;     set escape flag to a one
No_Good:
	ret				;     return

Not_Esc:
	cmp	al,bell			; see if in range
	jc	No_Good			;  if not in range then no good
	cmp	al,Cr+1
	jnc	No_Good			;  else
	jmp	Esc_Hndlr		;  use thread system to execute code
	
	; process row/column cursor addressing
In_DCA:	test	Term_Esc_Rcvd,01000000b  ; see if this is first  char (ROW)
	jz	Its_Col			; if not then its column
	and	Term_Esc_Rcvd,10000000b  ;  else, mark that row was rcvd.
	sub	al,' '			; adjust base of character
	cmp	al,Last_Row		; modulo screen height
	jbe	Row_OK			; make sure it's in range
	mov	al,last_Row

Row_OK:	mov	Term_Row,al		; set new row
	ret

Its_Col:				; data is column so sequence done
	mov	Term_Esc_Rcvd,0		; clear escape rcvd flag
	sub	al,' '			; adjust column
	cmp	al,Last_Col		; modulo screen width
	jbe	Col_OK			; make sure it's in range
	mov	al,last_Col		; truncate it if out of range

Col_OK:	mov	ah,Term_Row		; get the row and column in ah, al
	jmp	Set_Cur			; and set new cursor position.

Esc_Hndlr: 
	mov	Term_Esc_Rcvd,0		; clear escape flag
	cld				; set for upwards search
	sub	si,si			; clear si for lookup
	mov	cx,Esc_Size		; set up number of entries to be checked

Srch:	cmp	al,byte ptr cs:[si+Esc_Table]	; check an entry
	Je	Take_Vect		; if found, then take the vector
	inc	si			; else
	loop	Srch			;    search rest of table
	ret				; if not in table, ignore code

Take_Vect:
	add	si,si			; compute offset for vector
	mov	ax,word ptr Term_Col	; load up row, col in ax
	call	Word Ptr ds:[si + offset Esc_Vect] ; call the handler
	jmp	Set_Cur

Disp_Char 	EndP

page
;----------------------------------------------------------------------
; ESCAPE and CONTROL Character handlers.  Each of the following routines
; is called to handle a non-printable character.  On entry to the routines,
; AX contains the current row and column of the cursor (0-23, 0-79).
;----------------------

;----------------------------------------------------------------------
Tab_Code Proc Near; (21_May_85)
;------------------------------
;	Perform a tab if not past column 71, else advance char 1 space
;
	cmp	al,Last_Col-8
	jae	right			; if within 8 of EOL, just 1 space
	add	al,08
	and	al,78h			; align col to next tab (tabs every 8)
	ret

Tab_Code	EndP

;----------------------------------------------------------------------
Lf_Code Proc Near; (21_May_85)
;-----------------------------
;	Perform line feed, scroll if necessary
;
	inc	ah			; next row
	cmp	ah,Last_Row
	jbe	return			; return if still on screen
	mov	di,0			; dest is screen buffer start
	mov	si,(Last_Col+1) * 2	; start is second line on screen
	mov	cx,Last_Row		; number of lines to move
	cld				; forward moves
	call	Lin_Lp			; invoke line mover
	mov	ax,Last_row * 256	; point to last line on screen
	jmp	Del_EOL			; use delete to line end to clear last line

Lf_Code		EndP

page
;----------------------------------------------------------------------
Lin_Lp Proc Near; (21_May_85)
;----------------------------
;	Move lines on screen routine, set direction flag before calling this
;	code.
;	Entry Register Values:
;		cx = number of lines to move
;		si = source line
;		di = destination line
;
Assume	cs:Monitor_Segment, ds:Screen_Segment, es:Screen_Segment

	push	cx			; save line loop counter
	push	di
	push	si			; save loop control registers
	pushf				; save direction
	call	Ser_Buf			; buffer any incoming chars
	popf
	pop	si
	pop	di			; restore loop control variables

	push	ds
	push	es
	mov	ax,Screen_Segment	; set DS and ES for screen segment
	mov	ds,ax
	mov	es,ax
	mov	cx,Last_Col+1		; move one line
	rep movsw			; copy it down
	pop	es
	pop	ds			; restore segments
	pop	cx			; restore line loop counter
	loop	Lin_Lp
	ret
Lin_Lp		EndP

;----------------------------------------------------------------------
Cr_Code Proc Near
;----------------
;	Carriage Return
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	xor	al,al			; goto col 0 of current line
	ret

Cr_Code		EndP

page
;----------------------------------------------------------------------
Clr_Scr Proc Near; (11_Jul_85)
;-----------------------------
;	Clr_Scr and Home Cursor   Used to Clear the screen and home the cursor.
;	Also, the entry point Home is used when Home escape code is received.
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Screen_Segment

	mov	di,0			; dest is screen buffer start
	mov	cx,Last_Row+1		; number of lines to clear

Era_Lp:	push	cx			; save line loop counter
	push	es
	mov	ax,Screen_Segment	; set ES for screen segment
	mov	es,ax
	mov	cx,Last_Col+1		; erase one line
	mov	ax,0720h		; clear last on screen
	cld				; forward moves
	rep stosw			; blank the line
	pop	es			; restore segment
	push	di			; save loop control register
	call	Ser_Buf			; buffer any incoming chars
	pop	di			; restore loop control variable
	pop	cx			; restore line loop counter
	loop	Era_Lp

Home:	mov	bh,0			;select page 0
	mov	dx,0			;setup row/col
	call	Set_Cursor_Position	;Set the Cursor Position
	mov	ax,0			; return with rol/col at home
	ret

Clr_Scr		EndP	

;----------------------------------------------------------------------
Right Proc Near
;--------------
;	move cursor right, don't advance if in last column
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	inc	al			; set new column
	cmp	al,Last_Col
	jbe	return			; return if not past line end

	dec	al
Return:	ret

Right		EndP

;----------------------------------------------------------------------
Left Proc Near
;-------------
;	Move cursor left if not already at left end of screen
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	dec	al			; set new column
	jns	LrSk1			; return if not before line start
	inc	al
LrSk1:	ret

Left		EndP

page
;----------------------------------------------------------------------
Down Proc Near
;-------------
;	Move cursor down one line if not already on bottom line
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	inc	ah			; set new row
	cmp	ah,Last_ROW
	jbe	return			; return if not past last row
	dec	ah
	ret

Down		EndP

;----------------------------------------------------------------------
Up Proc Near
;-----------
;	Move cursor up one line if not already at top
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	dec	ah			; set new row
	jns	return			; return if not going off top of screen
	inc	ah
	ret

Up		EndP

page
;----------------------------------------------------------------------
Direct_Cur Proc Near
;-------------------
;	Direct cursor adressing. Just set flags for pending characters
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Nothing

	mov	Term_Esc_Rcvd,11000000b	; set dca flags (row/col coming next)
	jmp	Return_RC

Direct_Cur	EndP

;----------------------------------------------------------------------
Del_EOS Proc Near
;----------------
;	Delete from cursor to end of screen
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

dellp:	push	ax			; save current position
	call	Ser_Buf			; buffer any incoming chars
	pop	ax
	push	ax			; copy screen position
	call	Del_Eol			; clear a line
	pop	ax			; get screen position back
	mov	al,0			; reset col
	inc	ah			; advance to next row
	cmp	ah,Last_Row+1		; at screen end ?
	jne	dellp			; loop if not
	jmp	Return_RC

Del_EOS		EndP

;----------------------------------------------------------------------
Del_EOL Proc Near
;----------------
;	Clear from cursor to end of line
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Screen_Segment

	mov	cl,80
	sub	cl,al			; get chars to end of line
	xor	ch,ch
	mov	di,ax
	and	di,0ffh			; isolate start col
	add	di,di			; byte offset
	mov	al,ah
	cbw				; word value for row number
	mov	bl,(Last_Col+1)*2	; byte size of a line
	mul	bl			; get memory offset to line start
	add	di,ax			; cursor loc in screen memory
	push	es
	mov	ax,Screen_Segment
	mov	es,ax			; refer to screen memory
	mov	ax,0720h		; clear last line on screen
	cld				; forward moves
	rep stosw			; clear it out
	pop	es			; restore our ES
	jmp	Return_RC

Del_EOL		EndP

page
;----------------------------------------------------------------------
Ins_Line Proc Near
;-----------------
;	Insert line at cursor, scroll screen down, and put cursor in col 0.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Nothing

	mov	al,0
	mov	Term_Col,al		; force new column
	mov	bl,Last_Row
	sub	bl,ah			; calc number of lines to move
	jne	notilr			; skip if must move screen lines
	jmp	Del_EOL			; just do delete to line end

notilr:	mov	si,(Last_Row)*(Last_col+1)*2-2  ; source is end of 23rd line
	lea	di,[si+(Last_Col+1) * 2]  ; dest is end of 24th line
	mov	cl,bl			; number of lines to move
	mov	ch,0			; make it a word value
	std				; backward moves
	call	Lin_Lp			; invoke line mover
	mov	ax,word ptr Term_Col	; restore cursor row/col
	jmp	Del_EOL			; exit thru line erase

Ins_Line 	EndP


;----------------------------------------------------------------------
Del_Line Proc Near
;-----------------
;	Delete line that cursor is on, move following lines up 1, and
;	put cursor in col 0.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Nothing

	mov	al,0
	mov	Term_Col,al		; force new column
	mov	bl,Last_Row
	sub	bl,ah			; calc number of lines to move
	jne	notlr			; skip if must move screen lines
	jmp	Del_EOL			; just do delete to line end

notlr:	xchg	al,ah			; get word value of current line
	mov	cl,(Last_Col+1)*2
	mul	cl			; memory offset to line start
	mov	di,ax			; dest is screen line start
	lea	si,[di+(Last_Col+1) * 2] ; start is next line start
	mov	cl,bl			; number of lines to move
	mov	ch,0			; make it a word value
	cld				; forward moves
	call	Lin_Lp			; invoke line mover
	mov	ax,Last_Row * 256	; point to last line on screen
	jmp	Del_EOL			; use delete to line end to clear last line

Del_Line 	EndP

page
;----------------------------------------------------------------------
Del_Char Proc Near
;-----------------
;	Delete character at cursor, move following chars on line over 1.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Screen_Segment
Public	Del_Char

	cbw				; clear row from ah
	mov	di,ax			; di := Column to move to
	inc	ax
	mov	si,ax			; si := column to move from
	mov	cx,Last_Col
	sub	cl,al			; cx := number of chars till end of line
	cld				; set for forward moves

; Fall thru ...
;
; move screen memory routine
; enter with direction flag set for forward or reverse move,
; si and di must have start and end column on current line,
; cx must have number of screen positions to move
;
Blk_Move:
	add	si,si
	add	di,di			; adjust cols for attributes in screen memory
	mov	al,ah			; al := row cursor is on
	cbw				; make it a word value
	mov	bl,160
	mul	bl			; ax := row * 80 (chars/line) * 2 (bytes/char)
	add	si,ax			; add offsets to si and di before moving
	add	di,ax		

	push	ds
	push	es
	mov	ax,Screen_Segment	; set DS and ES for screen segment
	mov	ds,ax
	mov	es,ax
	jcxz	No_Move			; if in last column then just blank the char
	rep movsw			; adjust the screen memory

No_Move:
	mov	ax,0720h		; clear last char on line
	stosw
	pop	es
	pop	ds


; Common Return
;--------------
;
Return_RC:
	mov	ax,word ptr Term_Col	; restore row/col for return
	ret				; general return from control/escape routines

Del_Char 	EndP

;----------------------------------------------------------------------
Ins_Char Proc Near
;-----------------
;	Insert a char at cursor position.
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	mov	cx,Last_Col
	sub	cl,al			; cx := number of chars till end of line
	mov	di,Last_Col
	mov	si,Last_Col-1
	std				; set for reverse memory move
	jmp	Blk_Move

Ins_Char	EndP

;----------------------------------------------------------------------
Send_Curs Proc Near
;------------------
;	Send current cursor location.
;
Assume	cs:Monitor_Segment, ds:Nothing, es:Nothing

	ret

Send_Curs	EndP

;-------------------------------------------------------------------
Set_Cur	Proc Near; (11_Jul_85)
;-----------------------------
;	Sets the cursor position on the screen to the position contained
;	in AH (ROW), and AL (COLUMN).  Also sets new cursor position.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, es:Nothing

	mov	word ptr Term_Col,ax	;save in row, column area
	mov	dx,ax			;set up cursor position in dx
	mov	bh,0			;video page number
	call	Set_Cursor_Position	;Position the Cursor
	ret				;Return

Set_Cur		EndP

Monitor_Segment	EndS
		End
