/***********************************************************************
*
* ob2bin - Convert asm7090 assembler output to scatter load tape binary
*         format.
*
* Changes:
*   01/20/05   DGP   Original.
*	
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <errno.h>

#include "sysdef.h"
#include "objdef.h"
#include "parity.h"

#define SYSCUR 0102
#ifdef WIN32
#define EOMWORD 0500000000000I64
#else
#define EOMWORD 0500000000000LL
#endif

static t_uint64 memory[MEMSIZE];
static uint8 memctl[MEMSIZE];

static uint8 word[12];
static uint8 ctlbits;

/***********************************************************************
* writeword - write a word.
***********************************************************************/

static void
writeword (t_uint64 wrd, FILE *fd)
{
   int i;

   for (i = 0; i < 6; i++)
   {
      word[i] = oddpar[(uint8)((wrd >> 30) & 077)];
      if (i == 0) word[i] |= ctlbits;
      else ctlbits = 0;
      wrd <<= 6;
   }
   fwrite (word, 1, 6, fd);
}

/***********************************************************************
* writeblock - write a block of words with address.
***********************************************************************/

static void
writeblock (int loadaddr, int wrdcnt, FILE *fd)
{
   t_uint64 wrd1;
   int i;

#ifdef DEBUG
   printf ("writeblock: loadaddr = %05o, wrdcnt = %o\n", loadaddr, wrdcnt);
#endif

   /*
   ** Write out loader control word
   */

   wrd1 = ((t_uint64)040 << 30) | ((t_uint64)wrdcnt << DECRSHIFT) |
	 (t_uint64)loadaddr;
   writeword (wrd1, fd);

   /*
   ** Write the corresponding block
   */

   for (i = 0; i < wrdcnt; i++)
   {
      writeword (memory[loadaddr+i], fd);
   }

}

/***********************************************************************
* Main procedure
***********************************************************************/

main (int argc, char *argv[])
{
   FILE *infd;
   FILE *outfd;
   char *bp;
   char *infile;
   char *outfile;
   int loadaddr;
   int curraddr;
   int entaddr;
   int wrdcnt;
   int ibldr;
   int i;
   char inbuf[82];

   /*
   ** Process command line arguments
   */

   infile = NULL;
   outfile = NULL;
   ibldr = 0;
   loadaddr = 0200;
   curraddr = 0200;
   entaddr = -1;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];
      if (*bp == '-')
      {
         bp++;
	 switch (*bp)
	 {
	 case 'i': /* IBSYS loader image */
	    ibldr = 1;
	    break;

	 case 'l': /* Load address, for relocatable images */
	    i++;
	    loadaddr = atoi(argv[i]);
	    break;

	 default:
	    goto usage;
	 }
      }
      else if (infile == NULL)
         infile = argv[i];
      else if (outfile == NULL)
         outfile = argv[i];
      else
      {
      usage:
         fprintf (stderr, "usage: obj2bin [options] infile outfile\n");
	 fprintf (stderr, " options:\n");
         fprintf (stderr, "    -i           - IBSYS loader image\n");
         fprintf (stderr, "    -l addr      - Load address\n");
	 exit (1);
      }
   }

   if (infile == NULL || outfile == NULL) goto usage;

   /*
   ** Open the files.
   */

   if ((infd = fopen (infile, "r")) == NULL)
   {
      fprintf (stderr, "obj2bin: input open failed: %s",
	       strerror(errno));
      fprintf (stderr, "filename: %s", infile);
      exit (1);
   }

   if ((outfd = fopen (outfile, "wb")) == NULL)
   {
      fprintf (stderr, "obj2bin: output open failed: %s",
	       strerror(errno));
      fprintf (stderr, "filename: %s", outfile);
      exit (1);
   }

   /*
   ** Load "memory" with the program
   */

   memset (&memory, '\0', sizeof (memory));
   memset (memctl, '\0', sizeof (memctl));

   while (fgets (inbuf, sizeof(inbuf), infd))
   {
      char *op = inbuf;
      int relo;

      for (i = 0; i < 5; i++)
      {
	 char otag;
	 char item[16];
	 t_uint64 lldata;

	 otag = *op++;
	 if (otag == ' ') break;
	 lldata = 0;
	 strncpy (item, op, 12);
	 item[12] = '\0';
#ifdef WIN32
	 sscanf (item, "%i64o", &lldata);
#else
	 sscanf (item, "%llo", &lldata);
#endif

#ifdef DEBUG
	 printf ("loadaddr = %05o, curraddr = %05o\n",
		  loadaddr, curraddr);
	 printf ("   otag = %c, item = %s\n", otag, item);
#ifdef WIN32
	 printf ("   lldata = %12.12i64o\n", lldata);
#else
	 printf ("   lldata = %12.12llo\n", lldata);
#endif
#endif

         switch (otag)
	 {
	 case ABSORG_TAG:
	    curraddr = loadaddr = (int)(lldata & ADDRMASK);
	    break;

	 case RELORG_TAG:
	    curraddr = (int)(lldata + loadaddr) & ADDRMASK;
	    break;

	 case BSS_TAG:
	    curraddr += (int)(lldata & ADDRMASK);
	    curraddr &= ADDRMASK;
	    break;

	 case ABSDATA_TAG:
	    memory[curraddr] = lldata;
	    memctl[curraddr] = otag;
	    curraddr++;
	    break;

	 case RELADDR_TAG:
	    memory[curraddr] = lldata + loadaddr;
	    memctl[curraddr] = otag;
	    curraddr++;
	    break;

	 case RELBOTH_TAG:
	    lldata += loadaddr;
	 case RELDECR_TAG:
	    relo = ((lldata & DECRMASK) >> DECRSHIFT) + loadaddr;
	    lldata &= ~DECRMASK;
	    lldata |= (relo << DECRSHIFT); 
	    memory[curraddr] = lldata;
	    memctl[curraddr] = otag;
	    curraddr++;
	    break;

	 case ABSXFER_TAG:
	    goto LOADED;
	 case ABSENTRY_TAG:
	    entaddr = (int)(lldata & ADDRMASK);
	    break;

	 case RELXFER_TAG:
	    goto LOADED;
	 case RELENTRY_TAG:
	    entaddr = (int)(lldata & ADDRMASK) + loadaddr;
	    break;

	 default: ;
	 }
	 op += 12;
      }
   }

LOADED:

   /*
   ** Program is loaded dump it in tape binary format.
   */

   wrdcnt = 0;
   loadaddr = -1;
   ctlbits = 0200;

   /*
   ** Dump SYSCUR location first, if defined.
   */

   if (ibldr && memctl[SYSCUR])
   {
      writeblock (SYSCUR, 1, outfd);
      ctlbits = 0;
      memctl[SYSCUR] = 0;
      memory[SYSCUR] = 0;
   }

   /*
   ** Now dump the rest of the program.
   */

   for (i = 0; i < MEMSIZE; i++)
   {
      if (memctl[i])
      {
	 if (loadaddr < 0) loadaddr = i;
#ifdef DEBUG
         printf ("loadaddr = %05o, tag = %c\n", loadaddr, memctl[i]);
#endif
         switch (memctl[i])
	 {
	 case ABSDATA_TAG:
	 case RELADDR_TAG:
	 case RELDECR_TAG:
	 case RELBOTH_TAG:
	    wrdcnt++;
	    break;
	 case RELORG_TAG:
	 case ABSORG_TAG:
	    if (loadaddr > 0)
	    {
	       writeblock (loadaddr, wrdcnt, outfd);
	       ctlbits = 0;
	       wrdcnt = 0;
	    }
	    loadaddr = (int)(memory[i] & ADDRMASK);
	    break;
	 default: ;
	 }
      }
      else
      {
         if (loadaddr >= 0)
	 {
	    writeblock (loadaddr, wrdcnt, outfd);
	    ctlbits = 0;
	    wrdcnt = 0;
	    loadaddr = -1;
	 }
      }
   }

   if (wrdcnt)
   {
      writeblock (loadaddr, wrdcnt, outfd);
   }

   /*
   ** Write End of Module word.
   */

   writeword (EOMWORD+3, outfd);

   fclose (infd);
   fclose (outfd);

   return (0);
}
