/* paper tape editor - compile in compact, large, or huge model only */
#pragma warn -ucp
#pragma warn -sig

#include <io.h>
#include <stdio.h>
#include <ctype.h>
#include <process.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <alloc.h>
#include <dir.h>

#define BUFSZ 65530UL
#define E 0x80
#define X 0x40
#define O 0x20
#define C 0x10

#define SRCH_RM (unsigned char *)"\x1\x80"

#define DIRTY_LEFT 1
#define DIRTY_RIGHT 2
#define DIRTY_UP 4
#define DIRTY_DOWN 8
#define DIRTY_CHANGE 0x10

typedef unsigned char *ptr;
typedef struct { int size; ptr p; } qreg;

FILE *file;
ptr buffer, point, end;
char cmdbuf[256];
char format = 1;	/* 0 = hex, 1 = numeric, 2 = pt codes, 3 = alpha */
qreg q[10];
int wrapMode = 0;
int unsleft = 0;
ptr firstCharWin;
int pointX = 1, pointY = 1;
int wX = 1;
int dirty = DIRTY_CHANGE;
int deltaRec = 0;
int emsg = 0;

void cprintfw(char *p)
{
    window(1,25,80,25);
    gotoxy(wX, 1);
    putch(' ');
    cprintf(p);
    clreol();
    emsg = 1;
    wX = wherex();
    window(1,1,80,25);
    gotoxy(pointX, pointY);
}

void putchw(char c)
{
    window(1,25,80,25);
    gotoxy(wX, 1);
    putch(c);
    wX = wherex();
    window(1,1,80,25);
    gotoxy(pointX, pointY);
}

unsigned char parity(unsigned char c)	/* Takes char, returns C bit */
{
    unsigned char d = 0;
    
    while (c)
    {
	d += c & 1;
	c = c >> 1;
    }
    return d & 1 ? 0 : C;
}

/****************************************************************
* incvt - convert string of N, P, H, or A format data to FBL hex.
* N: [-][O#0-9] | [ @;] ... :		(space is a code point)
* P: { [EXOC8421] ... ' ' } ... :	(space is a separator)
* H: [0-9A-F] ... :			(spaces are ignored)
* A: 1620 alpha + '#' RM, ';' EOL ... :	(space is a code point)
*	Converts data in place, changing null-terminated argument
*	string into FBL string occupying the same space.
*	Returns total length of converted arg or -1 if error.
****************************************************************/
unsigned char tbl1[] =
{ '0', ' ', '.', ')', '+', '$', '*', '-', '/', ',', '(', '=', '@', '#', ';','~',0};
unsigned char tbl2[] =
{0x20,0x10,0x6b,0x7c,0x70,0x5b,0x4c,0x40,0x31,0x3b,0x2c,0x0b,0x1c,0x2a,0x80,0x7F,0};

int incvt(ptr cp)
{
    static unsigned char scratch[80];
    static int len = 0;
    unsigned char *ip = scratch, *op = cp, *rp;
    unsigned char c;
    int pt, n;
    ptr p;

    pt = 0;				/* Initialize accumulator */
    rp = strchr(cp, ':');		/* Sep marks end of text */
    if (rp == NULL)			/* No separator, */
	rp = cp + strlen(cp) + 1;	/*  use entire string */
    else
	*rp++ = '\0';			/* Chop off input string at sep */
    /* rp points to next command (if any) */
    if (rp == (cp+1) && len != 0)	/* Empty argument = repeat */
    {
	/* Move rest of cmdbuf forward to make room for repeated op */
	memmove(cp+len, cp+1, strlen(cp+1)+1);
	rp = cp + len;		/* Repeated op takes this many bytes */
    }
    else
	strcpy(scratch, cp);

    /* We do '*++op' to skip the first byte, which we fill in with length */

    switch(format)
    {
	case 0:		/* Bare hex */
	{
	    while (sscanf(ip, "%2x%n", &pt, &n) == 1)
	    {
		*++op = pt;
		ip += n;
	    }
	    break;
	}

	case 1:		/* 1620 numerics */
	{
	    while ((c = *ip++) != '\0')
	    {	/* Analyze one input char */
		switch (tolower(c))
		{
		    case '-': pt |= X; continue;	/* Flag */
		    case 'o': pt |= O; break;		/* O-type zero */
		    case '#': pt |= O|8|2; break;	/* RM */
		    case ' ': pt = C; break;		/* Space code */
		    case '@': pt = C|8|4; break;	/* NB */
		    case ';': pt = E; break;		/* EOL */
		    case '~': pt = 0x7F; break;		/* Tapefeed */
		    default:
			if (isdigit(c))
			    pt |= (c - '0');
			else
			{
			    cprintfw("Bad digit, need [-][O#0-9] or [ ~@;]");
			    return -1;
			}
		}
		if (pt == 0) pt = O;	/* Turn unescorted zero into O */
		pt ^= parity(pt);
		*++op = pt, pt = 0;	/* Output and reset */
	    }
	    break;
	}

	case 2:		/* Raw PT codes */
	{
	    /* Set up first token.  If none, return error */
	    if ((ip = strtok(ip, " \t")) == NULL) return -1;
	    do
	    {	/* We definitely have a token here */
		while ((c = *ip++) != '\0')
		{	/* Analyze one char */
		    switch (tolower(c))
		    {
			case 'e': pt |= E; break;
			case 'x': pt |= X; break;
			case 'o': pt |= O; break;
			case 'c': pt |= C; break;
			default:
			    if (isdigit(c))
				pt |= (c - '0');
			    else
			    {
				cprintfw("Bad PT code, need [EXOC8421]...");
				return -1;
			    }
		    }
		}
		*++op = pt, pt = 0;		/* Output and reset */
	    } while ((ip = strtok(NULL, " \t")) != NULL);
	    break;
	}
	case 3:		/* 1620 alphamerics */
	{
	    while ((c = *ip++) != '\0')
	    {	/* Analyze one input char */
		c =  (tolower(c));
		if (c >= '1' && c <= '9')
		    pt = c - '0';
		else if (c >= 'a' && c <= 'i')
		    pt = X | O | (c - 'a' + 1);
		else if (c >= 'j' && c <= 'r')
		    pt = X | (c - 'j' + 1);
		else if (c >= 's' && c <= 'z')
		    pt = O | (c - 's' + 2);
		else
		{
		    p = memchr(tbl1, c, sizeof(tbl1));
		    if (p == NULL)
		    {
			cprintfw(
"Bad character, need [-0-9A-Z .)+$*/,(=@#~;]");
			return -1;
		    }
		    else
			pt = tbl2[(int)(p - tbl1)];
		}
		pt ^= parity(pt);
		*++op = pt, pt = 0;	/* Output and reset */
	    }
	    break;
	}

    }
    *cp = op - cp;			/* Set length byte */
    len = op - cp + 1;			/* Save in case repeat next time */
    return (int)(rp - cp);		/* We ate this many chars */
}

/****************************************************************
* Primitive Search
* Input is "start from" pointer, FBL byte array, and
*  number of times to search.
* Negative times means backward search.  0 times is no-op.
* Returns just past S.S., or NULL if fail.
****************************************************************/
ptr psearch(ptr p, ptr cp, int times)
{
    int ssi;
    
    if (times > 0)			/* Forward search */
	while ((unsigned int)p < (unsigned int)end)/* Still places to look */
	{
	    ssi = 1;				/* First SS byte */
	    while (*p++ == cp[ssi] && (unsigned int)p <= (unsigned int)end)
	    {
		if (ssi++ == cp[0])		/* Match */
		{
		    if (--times == 0) return p;	/* Nth match */
		    ssi = 1;			/* Reset search string */
		}
	    }
	    p -= ssi - 1;			/* Partial match, back up */
	}
    else if (times < 0)			/* Backward search */
    {
	while ((unsigned int)p > (unsigned int)buffer)/* Still places */
	{
	    ssi = cp[0];			/* Start with last SS byte */
	    while (*--p == cp[ssi] && (unsigned int)p >= (unsigned int)buffer)
	    {
		if (--ssi == 0)			/* No more SS to go */
		{
		    if (++times == 0) { p += cp[0]; return p; }
		    ssi = cp[0];		/* Reset SS */
		}
	    }
	    p += cp[0] - ssi;			/* Reset partial match */
	}
    }
    return NULL;				/* Search failed */
}

/****************************************************************
 * rec - move around record by record
 * Moves starting at 'start'.
 * + is forward, - is back, 0 goes to the beginning of the record.
 * Returns addr of new record, BOF/EOF if fail.
 ****************************************************************/
ptr rec(ptr start, int num)
{
    ptr p;

    if (num <= 0) num--;		/* 0L => -1, etc */
    p = psearch(start, SRCH_RM, num);
    if (p == NULL)			/* Not found */
	if (num > 0)			/* Forward search failed, */
	    p = end;			/* go to EOF */
	else				/* Backward search failed, */
	    p = buffer;			/* go to BOF */
    return p;
}

/****************************************************************
 * scroll - scroll the screen
 ****************************************************************/
void scroll(int num)
{
    if (wrapMode)
	firstCharWin += num * (format ? 80 : 20);
    else
	firstCharWin = rec(firstCharWin, num);
}

/****************************************************************
 * scrollCheck - scroll if necessary.
 * Side effects: sets point = newP, sets pointX and pointY.
 ****************************************************************/
void scrollCheck(ptr newP)
{
    int lineSize = format ? 80 : 20;
    ptr oldP = point;
    ptr p;
    int deltaY = 0;
    int maxY = (format == 2) ? 2 : 24;

    point = newP;	/* Adjust point */

    if ((unsigned int)oldP < (unsigned int)newP)
    {
	p = oldP;
	while((unsigned int)p < (unsigned int)newP)
	    if (*p++ == E)
	    {
		deltaRec++;	/* Accumulate for stat display */
		deltaY++;
	    }
    }
    else
    {
	p = newP;
	while((unsigned int)p < (unsigned int)oldP)
	    if (*p++ == E)
	    {
		deltaRec--;
		deltaY--;
	    }
    }

    if (format == 0)
	pointX = (pointX - 1) / 3 + 1;

    if (wrapMode)	/* Calc line skips based on number of chars */
    {
	pointX = (unsigned int)(point-buffer)%lineSize + 1;
	deltaY = (unsigned int)(newP-buffer)/lineSize
	       - (unsigned int)(oldP-buffer)/lineSize;
    }
    else		/* Calc line skips based on eol hits */
    {
			/* Also calculate horizontal scroll position */
	pointX = (unsigned int)newP
	       - (unsigned int)rec(newP, 0)
	       - unsleft + 1;
	if (pointX < 1)
	{
	    unsleft += pointX - 1;
	    pointX = 1;
	    dirty |= DIRTY_RIGHT;
	}
	else if (pointX > lineSize)
	{
	    unsleft += pointX - lineSize;
	    pointX = lineSize;
	    dirty |= DIRTY_LEFT;
	}
    }

    if (format == 0)
	pointX = (pointX - 1) * 3 + 1;

    if (format == 2)
    {
	pointY = (pointY + 4) / 10;
	if (pointY == 0)
	    pointY = 1;
    }

    pointY += deltaY;
    if (pointY > maxY)
    {
	scroll(pointY - maxY);
	pointY = maxY;
	dirty |= DIRTY_UP;
    }
    else if (pointY < 1)
    {
	scroll(pointY - 1);
	pointY = 1;
	dirty |= DIRTY_DOWN;
    }
    if (format == 2)
	pointY = pointY * 10 - 4;
}

/****************************************************************
* Search, as called from cmd line.
* Input is ptr into cmd line and number of times to search.
* + is forward, - is reverse, 0 is nop.
* Returns number of cmd line chars we ate.
* Leaves point after S.S., or undisturbed if not found.
****************************************************************/
int searchcmd(ptr cp, int times)		/* cp is cmdline chars */
{
    int advance;
    ptr p;

    advance = incvt(cp);
    if (advance > 0)
	if ((p = psearch(point, cp, times)) != NULL)
	    scrollCheck(p);
	else
	    cprintfw("Not found");
    return advance;
}

/****************************************************************
* A bunch of output formatting routines
****************************************************************/
void aA(char c)			/* Alpha mode */
{
    int i;
    ptr p;

    p = memchr(tbl2, c, sizeof(tbl2));
    if (p != NULL)
	i = tbl1[(int)(p - tbl2)];
    else if ((c & 0xF) > 9 || (c & 0xF) == 0)
	i = '?';
    else
    {
	i = (c & 0xF) - 1;
	switch (c & (X|O))
	{
	    case X|O: i += 'A'; break;
	    case X:  i += 'J'; break;
	    case O: i += 'S' - 1; break;
	    case 0: i += '1'; break;
	    default: i = '!'; break;
	}
    }
    putch(i);
}

void nD(char c)			/* Numeric mode */
{
    char d, t[10];

    if (c == (C|8|4)) d = '@';			/* NB */
    else if (c == (char) E) d = ';';		/* EOL */
    else if (c == (O|8|2)) d = '#';		/* RM */
    else if (c == (X|O|C|8|2)) d = '#';		/* Flagged RM */
    else if (c == (X|O|C)) d = 'O';		/* Alt -0 */
    else if (c == C) d = ' ';			/* Space */
    else if (c == 0x7F) { c = 0; d = '~'; }	/* Tapefeed */
    else if ((c & (O|0x0F)) > O) d = '?';	/* ERROR */
    else { itoa(c & 0x0F, t, 16); d = t[0]; }	/* Digit 0-9 */
    if (c&X)
	textattr(0x01);				/* Underline for flag */
    putch(d);
    if (c&X) textattr(0x07);			/* Back to normal video */
}

void nH(char c)			/* Raw hex */
{
    char s[3];

    sprintf(s, "%2.2X", (unsigned char) c);
    putch(s[0]);
    putch(s[1]);
    putch(' ');
}

/* PT mode, one for each channel */
void p1(char c) { putch((c & 1) ? '1' : ' '); }
void p2(char c) { putch((c & 2) ? '2' : ' '); }
void p4(char c) { putch((c & 4) ? '4' : ' '); }
void p8(char c) { putch((c & 8) ? '8' : ' '); }
void pX(char c) { putch((c & X) ? 'X' : ' '); }
void pO(char c) { putch((c & O) ? 'O' : ' '); }
void pC(char c) { putch((c & C) ? 'C' : ' '); }
void pE(char c) { putch((c & E) ? 'E' : ' '); }
void pSprocket(char c) { putch('.'); }
void pSep(char c) { putch('-'); }

void (*outcvt[4][12])(char) = {
    { nH },
    { nD },
    { pSep, p1, p2, p4, pSprocket, p8, pC, pO, pX, pE },
    { aA }
};
    
/****************************************************************
 * Print a line of record data
 * Calls *printAByte to print one byte of record data
 * Starts at p
 * Stops at EOF
 * Stops at EOL if !wrapMode
 * Prints at most one line of text
 * Returns pointer to next byte
 ****************************************************************/
ptr prline(ptr p, int hScroll, void (*printAByte)(char))
{
    int ccount = format ? 80 : 20;
    unsigned char c = 0;

    if (!wrapMode)
	while (hScroll--)
	    if (*p++ == E)
	    {
		c = E;
		break;
	    }

    while (ccount--
	   && (unsigned int)p < (unsigned int)end
	   && (c != E || wrapMode))
	(*printAByte)(c = *p++);
    clreol();
    return p;
}

void prstat(void)
{
    static int recno, totrec;
    unsigned int recpos, recsiz;
    int unsright;
    ptr startpos, endpos;
    ptr p;
    int lineSize = format ? 80 : 20;

    if (!emsg)
    {
	startpos = rec(point, 0);		/* Where begin? */
	recpos = (unsigned int)(point - startpos + 1);
	endpos = rec(point, 1);			/* Where end? */
	recsiz = (unsigned int)(endpos - startpos);/* How big is rec */
	unsright = wrapMode ? 0 : recsiz - unsleft - lineSize;
	if (unsright < 0) unsright = 0;

	if (dirty & DIRTY_CHANGE)
	{
	    for (recno=0, p=buffer; (unsigned int)p<(unsigned int)point; p++)
		/* Where are we? */
		if (*p == E)
		    recno++;
	    deltaRec = 0;
	    for (totrec=recno;(unsigned int)p<(unsigned int)end;p++)
		if (*p == E)
		    totrec++;
	}
	else
	{
	    recno += deltaRec;
	    deltaRec = 0;
	}

	window(1,25,80,25);
	clreol();
	gotoxy(23,1);
	cprintf("%d(%d)", recno+1, totrec);
	cprintf(" %ld(%ld)", point-buffer, end-buffer);
	cprintf(" %d <%d(%d)> %d",unsleft,recpos,recsiz,unsright);
	wX = 1;
	window(1,1,80,25);
	gotoxy(pointX, pointY);
    }
}

void prscreen(void)
{
    int row;
    ptr p;
    ptr q;
    int channel, firstChannel;

    prstat();

    if (dirty != 0)
    {
	p = firstCharWin;
	channel = firstChannel = (format == 2) ? 9 : 0;
	for (row = 1; row <= 24; row++)
	{
	    gotoxy(1, row);
	    q = prline(p, unsleft, outcvt[format][channel]);
	    if (channel == 0)		/* Advance to next line's data */
	    {
		if (!wrapMode && *(q-1) != E)	/* Get into next record */
		    p = rec(q, 1);
		else
		    p = q;				/* Simply advance */
		channel = firstChannel;
		if (row + channel > 24) break;	/* All or none, not part */
	    }
	    else
		channel--;	/* Print same data for all channels */
	}
	dirty = 0;
	prstat();
    }
    gotoxy(pointX, pointY);
}

int replace(unsigned char *cp)			/* cp is cmdline chars */
{
    int advance, count;
    ptr p = point;

    advance = incvt(cp);
    if (advance > 0)
    {
	count = *cp++;
	while (count-- > 0)
	    *p++ = *cp++;
	if ((unsigned int)p > (unsigned int)end)
	    end = p;
	scrollCheck(p);
    }
    dirty |= DIRTY_CHANGE;
    return advance;
}

int insert(unsigned char *cp)			/* cp is cmdline chars */
{
    int advance, size;

    advance = incvt(cp);
    if (advance > 0)
    {
	size = *cp++;
	memmove(point+size, point, end - point);	/* Open a hole */
	memmove(point, cp, size);			/* Fill it */
	scrollCheck(point+size);
	end += size;
    }
    dirty |= DIRTY_CHANGE;
    return advance;
}

void delchar(long int size)
{
    if (size < 0)
    {	/* Back up by size and del forward from there */
	size = -size;
	if (size > (unsigned int)point - (unsigned int)buffer)
	    size = (unsigned int)point - (unsigned int)buffer;
	scrollCheck(point-size);
    }
    if (size > ((unsigned int)end - (unsigned int)point))
	size = ((unsigned int)end - (unsigned int)point);
    end -= size;
    memmove(point, point+size, (unsigned int)end-(unsigned int)point);
    if ((unsigned int)point > (unsigned int)end)
	point = end;
    dirty |= DIRTY_CHANGE;
}

void delrec(int num)	/* Delete record by record */
{
    delchar(rec(point, num) - point);
}

void loadQ(int num, int i)	/* Load Q-register i */
{
    ptr p = rec(point, num);

    q[i].size = abs(p - point);
    if (q[i].p != NULL) farfree(q[i].p);
    q[i].p = farmalloc(q[i].size);
    if (q[i].p == NULL)
    {
	cprintfw("Malloc failed");
	return;
    }

    memcpy(q[i].p, num > 0 ? point : p, q[i].size);
}

void dumpQ(int i)
{
    if (q[i].p != NULL)
    {
	memmove(point + q[i].size, point, end - point);	/* Open a hole */
	memcpy(point, q[i].p, q[i].size);		/* Fill it */
	scrollCheck(point+q[i].size);
	end += q[i].size;
	dirty |= DIRTY_CHANGE;
    }
    else
	cprintfw("No such Q-register");
}

void uparrow_action(int num)
{
    int oldX;
    ptr p;

    if (wrapMode)
    {
    /*
     * If we uparrow from the first line, p becomes seg:FFetc, which
     * as an unsigned int looks large, hence the num > 0 check.
     * Maybe I should have used huge pointers...
     */
	p = point + num * (format ? 80 : 20);
	if ((unsigned int)p < (unsigned int)buffer)
	    p = buffer;
	if ((unsigned int)p > (unsigned int)end)
	    if (num > 0)
		p = end;
	    else
		p = buffer;
    }
    else
    {
	oldX = point - rec(point, 0);
	p = rec(point, num) + oldX;
	if ((unsigned int)p > (unsigned int)end)
	    p = end;
    }
    scrollCheck(p);
}

void l_action(int num)
{
    ptr p = rec(point, num);
    scrollCheck(p);
}

void c_action(int num)
{
    ptr p = point;
    if ((unsigned int)p + num <= (unsigned int)buffer
	|| (num < 0 && (unsigned int)p + num >= (unsigned int)end))
    {
	cprintfw("Beginning of file");
	p = buffer;
    }
    else if ((unsigned int)p + num >= (unsigned int)end)
    {
	p = end;
	cprintfw("End of file");
    }
    else
	p += num;
    scrollCheck(p);
}

void main(int argc, char *argv[])
{
    char *cp, c;
    char temp[L_tmpnam];
    char path1[80], path2[80], drive[5], name[20], dir[80], ext[10];
    FILE *t;
    int i, n;
    int ennset;
    long int enn;
    int done;
    ptr p;

    for (n=0; n<10; n++) q[n].p = NULL;
    /* Can't use the full 65536, because far pointers don't inc the seg */
    buffer = (ptr) farmalloc(BUFSZ);
    fnsplit(argv[1], drive, dir, name, ext);
    if (strlen(ext) == 0)
	strcpy(ext, ".PT");
    fnmerge(path2, drive, dir, name, ext);
    file = fopen(path2, "r+b");
    if (file)
    {
	enn = filelength(fileno(file));
	if (enn > BUFSZ)
	{
	    fprintf(stderr, "File too large\n");
	    exit(1);
	}
	if (fread(buffer, enn, 1, file) != 1)
	{
	    fprintf(stderr, "Error reading file\n");
	    exit(1);
	}
    }
    else
	enn = 0;
    point = buffer;
    end = buffer + enn;

    firstCharWin = buffer;

    while (1)
    {					/* Main loop - get cmd line */
	prscreen();

	i = 0;
	done = 0;
	while (done == 0)
	{
	    c = getch();
	    if (emsg)
	    {
		window(1,25,80,25);
		gotoxy(1,1);
		clreol();
		wX = 1;
		emsg = 0;
		window(1,1,80,25);
		gotoxy(pointX, pointY);
	    }
	    if (c == 0)
	    {
		switch (c = getch())
		{
		case 'H':
		case 'P':
		case 'K':
		case 'M':
		case 'Q':
		case 'I':
		case 'O':
		case 'G':
			  cmdbuf[i++] = c;
			  done = 1;
			  break;
		default: cprintfw("?"); break;
		}
	    }
	    else if (c == '\b')		/* Destructive backspace */
	    {
		if (i > 0)
		{
		    i--;
		    putchw('\b');
		    putchw(' ');
		    putchw('\b');
		}
	    }
	    else if (c == '\r')
		done = 1;
	    else
	    {
		putchw(c);
		cmdbuf[i++] = tolower(c);
	    }
	}
	for (; i<sizeof(cmdbuf); i++)
	    cmdbuf[i] = 0;
	cp = cmdbuf;

	while (1)			/* Inner loop - process cmds */
	{   /* Look first for 'number'. This catches -n too. */
	    ennset = 0;
	    if (sscanf(cp, "%ld%n", &enn, &n) == 1)
		{ ennset = 1; cp += n; }
	    if (!ennset) enn = 1;	/* Use default */

	    /* Now try for a letter.  If none, must've hit eol. */
tryLetter:
	    if (sscanf(cp, "%1s%n", &c, &n) != 1)
		break;
	    cp += n;
	    switch (c)
	    {
	    case 'z': if (ennset)
			  { cprintfw("Number followed by z"); *cp = 0; }
		      enn = end - buffer + 1; ennset = 1;
		      goto tryLetter;
	    case '-': if (ennset)
		          { cprintfw("Number followed by '-'"); *cp = 0; }
		      enn = -1; ennset = 1; goto tryLetter;
	    case 'H': uparrow_action(-enn); break;
	    case 'P': uparrow_action(enn); break;
	    case 'K': c_action(-enn); break;
	    case 'M': c_action(enn); break;
	    case 'Q': if (format == 2)		/* Page down */
		          c = pointY > 10 ? 2 : 3;
		      else
			  c = 48 - pointY;
		      uparrow_action(c + (enn-1) * ((format == 2) ? 2 : 24));
		      break;
	    case 'I': if (format == 2)		/* Page up */
		          c = pointY < 10 ? 2 : 3;
		      else
			  c = pointY + 23;
		      uparrow_action(-c - (enn-1) * ((format == 2) ? 2 : 24));
		      break;
	    case 'O': if (wrapMode)		/* End */
		      {
			  if (format == 0)
			      i = 19 - (pointX - 1) / 3;
			  else
			      i = 80 - pointX;
			  c_action(i);
		      }
		      else
		      {
			  l_action(1);
			  c_action(-1);
		      }
		      break;
	    case 'G': if (wrapMode)		/* Home */
		      {
			  if (format == 0)
			      i = (pointX - 1) / 3;
			  else
			      i = pointX - 1;
			  c_action(-i);
		      }
		      else
			  l_action(0);
		      break;
	    case 'x': if (sscanf(cp, "%d%n", &i, &n) == 1 && i >= 0 && i < 10)
		      {
			  cp += n;
			  loadQ(enn, i);
		      }
		      else
		      {
			  cprintfw("Missing or wrong Q-register number");
			  *cp = '\0';
		      }
		      break;
	    case 'g': if (sscanf(cp, "%d%n", &i, &n) == 1 && i >= 0 && i < 10)
		      {
			  cp += n;
			  dumpQ(i);
		      }
		      else
		      {
			  cprintfw("Missing or wrong Q-register number");
			  *cp = '\0';
		      }
		      break;
	    case 'j': if (!ennset) enn = 0;
		      p = buffer + enn;
		      if ((unsigned int)p >= (unsigned int)end)
		      {
			  p = end;
			  cprintfw ("End of file");
		      }
		      scrollCheck(p);
		      break;
	    case 'r': c_action(-enn); break;
	    case 'c': c_action(enn); break;
	    case 'l': l_action(enn); break;
	    case 'd': delchar(enn); break;
	    case 'k': delrec(enn); break;
	    case 'i': if ((n = insert(cp)) > 0) cp += n; else *cp = '\0';
		      break;
	    case 'o': if ((n = replace(cp)) > 0) cp += n; else *cp = '\0';
		      break;
	    case 's': if ((n = searchcmd(cp, enn)) > 0)
			  cp += n;
		      else
			  *cp = '\0';		/* Break out of inner loop */
		      break;
	    case 'v': dirty |= DIRTY_CHANGE; prscreen();
	    case 'e': switch(tolower(*cp++))
		      {
		      case 'f': format = enn;
			        clrscr();
				dirty |= DIRTY_CHANGE;
				firstCharWin = buffer;
				pointX = pointY = 1;
				p = point;
				point = buffer;
				scrollCheck(p);		/* For format 2 */
				break;
		      case 'w': wrapMode = enn;
				clrscr();
				dirty |= DIRTY_CHANGE;
				firstCharWin = buffer;
				pointX = pointY = 1;
				p = point;
				point = buffer;
				scrollCheck(p);
				break;
		      case 'q': clrscr(); exit(0);
		      case 'x': clrscr();
			        tmpnam(temp);
				t = fopen(temp, "wb");
				if (t == NULL)
				    fprintf(stderr,
					"Error opening temp file\n");
				else if (fwrite(buffer,end-buffer,1,t) != 1)
				    fprintf(stderr,"Error writing file\n");
				else
				{
				    fclose(t);
				    printf("Wrote %ld bytes to %s\n", end-buffer, path2);
				    fnsplit(argv[1], drive, dir, name, ext);
				    strcpy(ext, ".BAK");
				    fnmerge(path1, drive, dir, name, ext);
				    remove(path1);
				    if (file != NULL)
					rename(path2, path1);
				    rename(temp, path2);
				    exit(0);
				}
				break;
		      default: cprintfw("Unknown 'e' command"); *cp = '\0';
		      }
		      break;
	    default: cprintfw("Unknown command"); *cp = '\0';
	    }
	}
    }
}
