/***********************************************************************
*
* asmoptab.c - Opcode table for the IBM 7090 assembler.
*
* Changes:
*   05/21/03   DGP   Original.
*   12/08/04   DGP   Started adding MAP pseudo ops.
*   02/02/05   DGP   Added some missing opcodes, THREE, SEVEN & ZERO.
*   03/01/05   DGP   Change IIA inst type.
*   03/25/05   DGP   Added 704 model and instructions.
*	
***********************************************************************/

#include <string.h>
#include <memory.h>
#include <stdlib.h>

#include "asmdef.h"
#include "asmdmem.h"

/*
** Standard opcodes
*/

OpCode optable[NUMOPS] =
{
   { "ABS",    ABS_T,    0,     TYPE_P,     704  },
   { "ACL",    00361,    0,     TYPE_B,     704  },
   { "ADD",    00400,    0,     TYPE_B,     704  },
   { "ADM",    00401,    0,     TYPE_B,     704  },
   { "ALS",    00767,    0,     TYPE_B,     704  },
   { "ANA",    04320,    0,     TYPE_B,     704  },
   { "ANS",    00320,    0,     TYPE_B,     704  },
   { "ARS",    00771,    0,     TYPE_B,     704  },
   { "AXC",    04774,    0,     TYPE_B,     709  },
   { "AXT",    00774,    0,     TYPE_B,     709  },
   { "BCD",    BCD_T,    0,     TYPE_P,     704  },
   { "BCI",    BCI_T,    0,     TYPE_P,     704  },
   { "BEGIN",  BEGIN_T,  0,     TYPE_P,     704  },
   { "BES",    BES_T,    0,     TYPE_P,     704  },
   { "BFT",    BFT_T,    0,     TYPE_P,     709  },
   { "BNT",    BNT_T,    0,     TYPE_P,     709  },
   { "BOOL",   BOOL_T,   0,     TYPE_P,     704  },
   { "BRA",    07000,    0,     TYPE_A,     704  },
   { "BRN",    03000,    0,     TYPE_A,     704  },
   { "BSF",    04764,    0,     TYPE_E,     709  },
   { "BSR",    00764,    00220, TYPE_E,     709  },
   { "BSS",    BSS_T,    0,     TYPE_P,     704  },
   { "BST",    00764,    00200, TYPE_E,     704  },
   { "BTT",    00760,    0,     TYPE_E,     709  },
   { "CAD",    04700,    0,     TYPE_B,     704  },
   { "CAL",    04500,    0,     TYPE_B,     704  },
   { "CALL",   CALL_T,   0,     TYPE_P,     704  },
   { "CAQ",    04114,    0,     TYPE_C,     709  },
   { "CAS",    00340,    0,     TYPE_B,     704  },
   { "CFF",    00760,    00030, TYPE_E,     704  },
   { "CHS",    00760,    00002, TYPE_E,     704  },
   { "CLA",    00500,    0,     TYPE_B,     704  },
   { "CLM",    00760,    00000, TYPE_E,     704  },
   { "CLS",    00502,    0,     TYPE_B,     704  },
   { "COM",    00760,    00006, TYPE_E,     704  },
   { "COMMON", COMMON_T, 0,     TYPE_P,     704  },
   { "CONTRL", CONTRL_T, 0,     TYPE_P,     704  },
   { "COUNT",  COUNT_T,  0,     TYPE_P,     704  },
   { "CPY",    00700,    0,     TYPE_B,     704  },
   { "CPYD",   05000,    0,     TYPE_A,     709  },
   { "CPYP",   04000,    0,     TYPE_A,     709  },
   { "CRQ",    04154,    0,     TYPE_C,     709  },
   { "CTL",    02000,    0,     TYPE_CHAN,  709  },
   { "CTLR",   02000,    02,    TYPE_CHAN,  709  },
   { "CTLW",   02400,    00,    TYPE_CHAN,  709  },
   { "CVR",    00114,    0,     TYPE_C,     709  },
   { "DCT",    00760,    00012, TYPE_E,     704  },
   { "DEBM",   01210,    01212, TYPE_DISK,  709  },
   { "DEC",    DEC_T,    0,     TYPE_P,     704  },
   { "DETAIL", DETAIL_T, 0,     TYPE_P,     704  },
   { "DFAD",   00301,    0,     TYPE_B,     7094 },
   { "DFAM",   00305,    0,     TYPE_B,     7094 },
   { "DFDH",   04240,    0,     TYPE_B,     7094 },
   { "DFDP",   04241,    0,     TYPE_B,     7094 },
   { "DFMP",   00261,    0,     TYPE_B,     7094 },
   { "DFSB",   00303,    0,     TYPE_B,     7094 },
   { "DFSM",   00307,    0,     TYPE_B,     7094 },
   { "DLD",    00443,    0,     TYPE_B,     7094 },
   { "DNOP",   01212,    01212, TYPE_DISK,  709  },
   { "DREL",   01204,    01212, TYPE_DISK,  709  },
   { "DSAI",   01007,    01212, TYPE_DISK,  709  },
   { "DSBM",   01211,    01212, TYPE_DISK,  709  },
   { "DSEK",   01012,    01212, TYPE_DISK,  709  },
   { "DST",    04603,    0,     TYPE_B,     7094 },
   { "DUAM",   04305,    0,     TYPE_B,     7094 },
   { "DUFA",   04301,    0,     TYPE_B,     7094 },
   { "DUFM",   04261,    0,     TYPE_B,     7094 },
   { "DUFS",   04303,    0,     TYPE_B,     7094 },
   { "DUP",    DUP_T,    0,     TYPE_P,     704  },
   { "DUSM",   04307,    0,     TYPE_B,     7094 },
   { "DVCY",   01005,    01212, TYPE_DISK,  709  },
   { "DVH",    00220,    0,     TYPE_B,     704  },
   { "DVHA",   01011,    01212, TYPE_DISK,  709  },
   { "DVP",    00221,    0,     TYPE_B,     704  },
   { "DVSR",   01002,    01212, TYPE_DISK,  709  },
   { "DVTA",   01010,    01212, TYPE_DISK,  709  },
   { "DVTN",   01004,    01212, TYPE_DISK,  709  },
   { "DWRC",   01006,    01212, TYPE_DISK,  709  },
   { "DWRF",   01003,    01212, TYPE_DISK,  709  },
   { "EAD",    00671,    0,     TYPE_B,     704  },
   { "EAXM",   00760,    00016, TYPE_E,     709  },
   { "ECA",    00561,    0,     TYPE_B,     704  },
   { "ECQ",    04561,    0,     TYPE_B,     704  },
   { "ECTM",   04760,    00006, TYPE_E,     709  },
   { "EDP",    00672,    0,     TYPE_B,     704  },
   { "EFTM",   04760,    00002, TYPE_E,     709  },
   { "EJECT",  EJE_T,    0,     TYPE_P,     704  },
   { "ELD",    00670,    0,     TYPE_B,     704  },
   { "EMP",    00673,    0,     TYPE_B,     704  },
   { "EMTM",   04760,    00016, TYPE_E,     7094 },
   { "ENB",    00564,    0,     TYPE_B,     709  },
   { "END",    END_T,    0,     TYPE_P,     704  },
   { "ENDM",   ENDM_T,   0,     TYPE_P,     704  },
   { "ENDQ",   ENDQ_T,   0,     TYPE_P,     704  },
   { "ENK",    00760,    00004, TYPE_E,     709  },
   { "ENTRY",  ENT_T,    0,     TYPE_P,     704  },
   { "EQU",    EQU_T,    0,     TYPE_P,     704  },
   { "ERA",    00322,    0,     TYPE_B,     709  },
   { "ESB",    04671,    0,     TYPE_B,     704  },
   { "ESM",    04761,    00040, TYPE_E,     709  },
   { "ESNT",   04021,    0,     TYPE_B,     7094 },
   { "EST",    04673,    0,     TYPE_B,     704  },
   { "ESTM",   04760,    00005, TYPE_E,     709  },
   { "ETC",    ETC_T,    0,     TYPE_P,     704  },
   { "ETM",    00760,    00007, TYPE_E,     704  },
   { "ETT",    04760,    0,     TYPE_E,     704  },
   { "EVEN",   EVEN_T,   0,     TYPE_P,     704  },
   { "EXTERN", EXT_T,    0,     TYPE_P,     704  },
   { "FAD",    00300,    0,     TYPE_B,     704  },
   { "FAM",    00304,    0,     TYPE_B,     709  },
   { "FDH",    00240,    0,     TYPE_B,     704  },
   { "FDP",    00241,    0,     TYPE_B,     704  },
   { "FILE",   FILE_T,   0,     TYPE_P,     704  },
   { "FIVE",   05000,    0,     TYPE_A,     704  },
   { "FMP",    00260,    0,     TYPE_B,     704  },
   { "FOR",    04000,    0,     TYPE_A,     704  },
   { "FOUR",   04000,    0,     TYPE_A,     704  },
   { "FRN",    00760,    00011, TYPE_E,     709  },
   { "FSB",    00302,    0,     TYPE_B,     704  },
   { "FSM",    00306,    0,     TYPE_B,     709  },
   { "FUL",    FUL_T,    0,     TYPE_P,     704  },
   { "FVE",    05000,    0,     TYPE_A,     704  },
   { "GOTO",   GOTO_T,   0,     TYPE_P,     704  },
   { "HEAD",   HEAD_T,   0,     TYPE_P,     704  },
   { "HED",    HEAD_T,   0,     TYPE_P,     704  },
   { "HPR",    00420,    0,     TYPE_B,     704  },
   { "HTR",    00000,    0,     TYPE_B,     704  },
   { "ICC",    07000,    02,    TYPE_CHAN,  709  },
   { "IFF",    IFF_T,    0,     TYPE_P,     704  },
   { "IFT",    IFT_T,    0,     TYPE_P,     704  },
   { "IIA",    00041,    0,     TYPE_B,     709  },
   { "IIB",    IIB_T,    0,     TYPE_P,     709  },
   { "IIL",    04051,    0,     TYPE_D,     709  },
   { "IIR",    00051,    0,     TYPE_D,     709  },
   { "IIS",    00440,    0,     TYPE_B,     709  },
   { "INDEX",  INDEX_T,  0,     TYPE_P,     704  },
   { "IOCD",   00000,    0,     TYPE_A,     709  },
   { "IOCDN",  00000,    02,    TYPE_A,     709  },
   { "IOCP",   04000,    0,     TYPE_A,     709  },
   { "IOCPN",  04000,    02,    TYPE_A,     709  },
   { "IOCT",   05000,    0,     TYPE_A,     709  },
   { "IOCTN",  05000,    02,    TYPE_A,     709  },
   { "IOD",    00766,    00333, TYPE_E,     704  },
   { "IORP",   02000,    0,     TYPE_A,     709  },
   { "IORPN",  02000,    02,    TYPE_A,     709  },
   { "IORT",   03000,    0,     TYPE_A,     709  },
   { "IORTN",  03000,    02,    TYPE_A,     709  },
   { "IOSP",   06000,    0,     TYPE_A,     709  },
   { "IOSPN",  06000,    02,    TYPE_A,     709  },
   { "IOST",   07000,    0,     TYPE_A,     709  },
   { "IOSTN",  07000,    02,    TYPE_A,     709  },
   { "IOT",    00760,    00005, TYPE_E,     709  },
   { "IRP",    IRP_T,    0,     TYPE_P,     704  },
   { "KEEP",   KEEP_T,   0,     TYPE_P,     704  },
   { "LABEL",  LABEL_T,  0,     TYPE_P,     704  },
   { "LAC",    00535,    0,     TYPE_B,     709  },
   { "LAR",    03000,    0,     TYPE_CHAN,  709  },
   { "LAS",    04340,    0,     TYPE_B,     709  },
   { "LAXM",   04760,    00016, TYPE_E,     709  },
   { "LBL",    LBL_T,    0,     TYPE_P,     704  },
   { "LBOOL",  LBOOL_T,  0,     TYPE_P,     704  },
   { "LBT",    00760,    00001, TYPE_E,     704  },
   { "LCC",    06400,    02,    TYPE_CHAN,  709  },
   { "LCHA",   00544,    0,     TYPE_B,     709  },
   { "LCHB",   04544,    0,     TYPE_B,     709  },
   { "LCHC",   00545,    0,     TYPE_B,     709  },
   { "LCHD",   04545,    0,     TYPE_B,     709  },
   { "LCHE",   00546,    0,     TYPE_B,     709  },
   { "LCHF",   04546,    0,     TYPE_B,     709  },
   { "LCHG",   00547,    0,     TYPE_B,     709  },
   { "LCHH",   04547,    0,     TYPE_B,     709  },
   { "LDA",    00460,    0,     TYPE_B,     704  },
   { "LDC",    04535,    0,     TYPE_B,     709  },
   { "LDI",    00441,    0,     TYPE_B,     709  },
   { "LDIR",   LDIR_T,   0,     TYPE_P,     704  },
   { "LDQ",    00560,    0,     TYPE_B,     704  },
   { "LFT",    04054,    0,     TYPE_D,     709  },
   { "LFTM",   04760,    00004, TYPE_E,     709  },
   { "LGL",    04763,    0,     TYPE_B,     704  },
   { "LGR",    04765,    0,     TYPE_B,     709  },
   { "LIP",    06000,    02,    TYPE_CHAN,  709  },
   { "LIPT",   01000,    02,    TYPE_CHAN,  709  },
   { "LIST",   LIST_T,   0,     TYPE_P,     704  },
   { "LIT",    LIT_T,    0,     TYPE_P,     704  },
   { "LITORG", LITORG_T, 0,     TYPE_P,     704  },
   { "LLS",    00763,    0,     TYPE_B,     704  },
   { "LMTM",   00760,    00016, TYPE_E,     7094 },
   { "LNT",    04056,    0,     TYPE_D,     709  },
   { "LOC",    LOC_T,    0,     TYPE_P,     704  },
   { "LORG",   LORG_T,   0,     TYPE_P,     704  },
   { "LRS",    00765,    0,     TYPE_B,     704  },
   { "LSNM",   04760,    00010, TYPE_E,     709  },
   { "LTM",    04760,    00007, TYPE_E,     704  },
   { "LXA",    00534,    0,     TYPE_B,     704  },
   { "LXD",    04534,    0,     TYPE_B,     704  },
   { "MACRO",  MACRO_T,  0,     TYPE_P,     704  },
   { "MAX",    MAX_T,    0,     TYPE_P,     704  },
   { "MIN",    MIN_T,    0,     TYPE_P,     704  },
   { "MON",    05000,    0,     TYPE_A,     704  },
   { "MPR",    04200,    0,     TYPE_B,     704  },
   { "MPY",    00200,    0,     TYPE_B,     704  },
   { "MSE",    04760,    00000, TYPE_E,     704  },
   { "MTH",    07000,    0,     TYPE_A,     704  },
   { "MTW",    06000,    0,     TYPE_A,     704  },
   { "MZE",    04000,    0,     TYPE_A,     704  },
   { "NOCRS",  NOCRS_T,  0,     TYPE_P,     704  },
   { "NOP",    00761,    0,     TYPE_B,     704  },
   { "NTR",    01000,    0,     TYPE_A,     704  },
   { "NULL",   NULL_T,   0,     TYPE_P,     704  },
   { "NZT",    04520,    0,     TYPE_B,     709  },
   { "OAI",    00043,    0,     TYPE_B,     709  },
   { "OCT",    OCT_T,    0,     TYPE_P,     704  },
   { "OFT",    00444,    0,     TYPE_B,     709  },
   { "ONE",    01000,    0,     TYPE_A,     704  },
   { "ONT",    00446,    0,     TYPE_B,     709  },
   { "OPD",    OPD_T,    0,     TYPE_P,     704  },
   { "OPSYN",  OPSYN_T,  0,     TYPE_P,     704  },
   { "OPVFD",  OPVFD_T,  0,     TYPE_P,     704  },
   { "ORA",    04501,    0,     TYPE_B,     704  },
   { "ORG",    ORG_T,    0,     TYPE_P,     704  },
   { "ORGCRS", ORGCRS_T, 0,     TYPE_P,     704  },
   { "ORS",    04602,    0,     TYPE_B,     704  },
   { "OSI",    00442,    0,     TYPE_B,     709  },
   { "PAC",    00737,    0,     TYPE_B,     709  },
   { "PAI",    00044,    0,     TYPE_B,     709  },
   { "PAX",    00734,    0,     TYPE_B,     704  },
   { "PBT",    04760,    00001, TYPE_E,     704  },
   { "PCA",    00756,    0,     TYPE_B,     7094 }, 
   { "PCC",    PCC_T,    0,     TYPE_P,     704  },
   { "PCD",    04756,    0,     TYPE_B,     7094 },
   { "PCG",    PCG_T,    0,     TYPE_P,     704  },
   { "PDC",    04737,    0,     TYPE_B,     709  },
   { "PDX",    04734,    0,     TYPE_B,     704  },
   { "PIA",    04046,    0,     TYPE_B,     709  },
   { "PMC",    PMC_T,    0,     TYPE_P,     704  },
   { "PON",    01000,    0,     TYPE_A,     704  },
   { "PSE",    00760,    00000, TYPE_E,     704  },
   { "PSLA",   00664,    0,     TYPE_B,     709  },
   { "PSLB",   04664,    0,     TYPE_B,     709  },
   { "PSLC",   00665,    0,     TYPE_B,     709  },
   { "PSLD",   04665,    0,     TYPE_B,     709  },
   { "PSLE",   00666,    0,     TYPE_B,     709  },
   { "PSLF",   04666,    0,     TYPE_B,     709  },
   { "PSLG",   00667,    0,     TYPE_B,     709  },
   { "PSLH",   04667,    0,     TYPE_B,     709  },
   { "PTH",    03000,    0,     TYPE_A,     704  },
   { "PTW",    02000,    0,     TYPE_A,     704  },
   { "PUNCH",  PUNCH_T,  0,     TYPE_P,     704  },
   { "PURGE",  PURGE_T,  0,     TYPE_P,     704  },
   { "PXA",    00754,    0,     TYPE_B,     709  },
   { "PXD",    04754,    0,     TYPE_B,     704  },
   { "PZE",    00000,    0,     TYPE_A,     704  },
   { "QUAL",   QUAL_T,   0,     TYPE_P,     704  },
   { "RBOOL",  RBOOL_T,  0,     TYPE_P,     704  },
   { "RCD",    00762,    00321, TYPE_E,     704  },
   { "RCHA",   00540,    0,     TYPE_B,     709  },
   { "RCHB",   04540,    0,     TYPE_B,     709  },
   { "RCHC",   00541,    0,     TYPE_B,     709  },
   { "RCHD",   04541,    0,     TYPE_B,     709  },
   { "RCHE",   00542,    0,     TYPE_B,     709  },
   { "RCHF",   04542,    0,     TYPE_B,     709  },
   { "RCHG",   00543,    0,     TYPE_B,     709  },
   { "RCHH",   04543,    0,     TYPE_B,     709  },
   { "RCT",    00760,    00014, TYPE_E,     709  },
   { "RDC",    00760,    00352, TYPE_E,     709  },
   { "RDR",    00762,    00300, TYPE_E,     704  },
   { "RDS",    00762,    0,     TYPE_E,     704  },
   { "REF",    REF_T,    0,     TYPE_P,     704  },
   { "REM",    REM_T,    0,     TYPE_P,     704  },
   { "RETURN", RETURN_T, 0,     TYPE_P,     704  },
   { "REW",    00772,    0,     TYPE_B,     704  },
   { "RFT",    00054,    0,     TYPE_D,     709  },
   { "RIA",    04042,    0,     TYPE_D,     709  },
   { "RIB",    RIB_T,    0,     TYPE_P,     709  },
   { "RIC",    00760,    00350, TYPE_E,     709  },
   { "RIL",    04057,    0,     TYPE_D,     709  },
   { "RIR",    00057,    0,     TYPE_D,     709  },
   { "RIS",    00445,    0,     TYPE_B,     709  },
   { "RND",    00760,    00010, TYPE_E,     704  },
   { "RNT",    00056,    0,     TYPE_D,     709  },
   { "RPR",    00762,    00361, TYPE_E,     704  },
   { "RQL",    04773,    0,     TYPE_B,     704  },
   { "RSCA",   00540,    0,     TYPE_B,     709  },
   { "RSCB",   04540,    0,     TYPE_B,     709  },
   { "RSCC",   00541,    0,     TYPE_B,     709  },
   { "RSCD",   04541,    0,     TYPE_B,     709  },
   { "RSCE",   00542,    0,     TYPE_B,     709  },
   { "RSCF",   04542,    0,     TYPE_B,     709  },
   { "RSCG",   00543,    0,     TYPE_B,     709  },
   { "RSCH",   04543,    0,     TYPE_B,     709  },
   { "RTB",    00762,    00220, TYPE_E,     704  },
   { "RTD",    00762,    00200, TYPE_E,     704  },
   { "RTT",    04760,    00012, TYPE_E,     704  },
   { "RUN",    04772,    00220, TYPE_E,     709  },
   { "SAR",    03000,    02,    TYPE_CHAN,  709  },
   { "SAVE",   SAVE_T,   0,     TYPE_P,     704  },
   { "SAVEN",  SAVEN_T,  0,     TYPE_P,     704  },
   { "SBM",    04400,    0,     TYPE_B,     704  },
   { "SCA",    00636,    0,     TYPE_B,     7094 },
   { "SCD",    04636,    0,     TYPE_B,     7094 },
   { "SCHA",   00640,    0,     TYPE_B,     709  },
   { "SCHB",   04640,    0,     TYPE_B,     709  },
   { "SCHC",   00641,    0,     TYPE_B,     709  },
   { "SCHD",   04641,    0,     TYPE_B,     709  },
   { "SCHE",   00642,    0,     TYPE_B,     709  },
   { "SCHF",   04642,    0,     TYPE_B,     709  },
   { "SCHG",   00643,    0,     TYPE_B,     709  },
   { "SCHH",   04643,    0,     TYPE_B,     709  },
   { "SDH",    00776,    00220, TYPE_E,     709  },
   { "SDL",    00776,    00200, TYPE_E,     709  },
   { "SDN",    00776,    0,     TYPE_B,     709  },
   { "SET",    SET_T,    0,     TYPE_P,     704  },
   { "SEVEN",  07000,    0,     TYPE_A,     704  },
   { "SIB",    SIB_T,    0,     TYPE_P,     709  },
   { "SIL",    04055,    0,     TYPE_D,     709  },
   { "SIR",    00055,    0,     TYPE_D,     709  },
   { "SIX",    06000,    0,     TYPE_A,     704  },
   { "SLF",    00760,    00140, TYPE_E,     704  },
   { "SLN",    00760,    00140, TYPE_E,     704  },
   { "SLQ",    04620,    0,     TYPE_B,     704  },
   { "SLT",    04760,    00140, TYPE_E,     704  },
   { "SLW",    00602,    0,     TYPE_B,     704  },
   { "SMS",    07000,    0,     TYPE_CHAN,  709  },
   { "SNS",    02400,    02,    TYPE_CHAN,  709  },
   { "SPACE",  SPC_T,    0,     TYPE_P,     704  },
   { "SPR",    00760,    00360, TYPE_E,     704  },
   { "SPT",    00760,    00360, TYPE_E,     704  },
   { "SPU",    00760,    00340, TYPE_E,     704  },
   { "SSM",    04760,    00003, TYPE_E,     704  },
   { "SSP",    00760,    00003, TYPE_E,     704  },
   { "SST",    SST_T,    0,     TYPE_P,     704  },
   { "STA",    00621,    0,     TYPE_B,     704  },
   { "STCA",   00544,    0,     TYPE_E,     709  },
   { "STCB",   04544,    0,     TYPE_E,     709  },
   { "STCC",   00545,    0,     TYPE_E,     709  },
   { "STCD",   04545,    0,     TYPE_E,     709  },
   { "STCE",   00546,    0,     TYPE_E,     709  },
   { "STCF",   04546,    0,     TYPE_E,     709  },
   { "STCG",   00547,    0,     TYPE_E,     709  },
   { "STCH",   04547,    0,     TYPE_E,     709  },
   { "STD",    00622,    0,     TYPE_B,     704  },
   { "STI",    00604,    0,     TYPE_B,     709  },
   { "STL",    04625,    0,     TYPE_B,     709  },
   { "STO",    00601,    0,     TYPE_B,     704  },
   { "STP",    00630,    0,     TYPE_B,     704  },
   { "STQ",    04600,    0,     TYPE_B,     704  },
   { "STR",    05000,    0,     TYPE_A,     709  },
   { "STT",    00625,    0,     TYPE_B,     709  },
   { "STZ",    00600,    0,     TYPE_B,     704  },
   { "SUB",    00402,    0,     TYPE_B,     704  },
   { "SVN",    07000,    0,     TYPE_A,     704  },
   { "SWT",    00760,    00160, TYPE_E,     704  },
   { "SXA",    00634,    0,     TYPE_B,     709  },
   { "SXD",    04634,    0,     TYPE_B,     704  },
   { "SYN",    EQU_T,    0,     TYPE_P,     704  },
   { "TCD",    TCD_T,    0,     TYPE_P,     704  },
   { "TCH",    01000,    0,     TYPE_A,     709  },
   { "TCM",    05000,    02,    TYPE_CHAN,  709  },
   { "TCNA",   04060,    0,     TYPE_B,     709  },
   { "TCNB",   04061,    0,     TYPE_B,     709  },
   { "TCNC",   04062,    0,     TYPE_B,     709  },
   { "TCND",   04063,    0,     TYPE_B,     709  },
   { "TCNE",   04064,    0,     TYPE_B,     709  },
   { "TCNF",   04065,    0,     TYPE_B,     709  },
   { "TCNG",   04066,    0,     TYPE_B,     709  },
   { "TCNH",   04067,    0,     TYPE_B,     709  },
   { "TCOA",   00060,    0,     TYPE_B,     709  },
   { "TCOB",   00061,    0,     TYPE_B,     709  },
   { "TCOC",   00062,    0,     TYPE_B,     709  },
   { "TCOD",   00063,    0,     TYPE_B,     709  },
   { "TCOE",   00064,    0,     TYPE_B,     709  },
   { "TCOF",   00065,    0,     TYPE_B,     709  },
   { "TCOG",   00066,    0,     TYPE_B,     709  },
   { "TCOH",   00067,    0,     TYPE_B,     709  },
   { "TDC",    06400,    0,     TYPE_CHAN,  709  },
   { "TEFA",   00030,    0,     TYPE_B,     709  },
   { "TEFB",   04030,    0,     TYPE_B,     709  },
   { "TEFC",   00031,    0,     TYPE_B,     709  },
   { "TEFD",   04031,    0,     TYPE_B,     709  },
   { "TEFE",   00032,    0,     TYPE_B,     709  },
   { "TEFF",   04032,    0,     TYPE_B,     709  },
   { "TEFG",   00033,    0,     TYPE_B,     709  },
   { "TEFH",   04033,    0,     TYPE_B,     709  },
   { "THREE",  03000,    0,     TYPE_A,     704  },
   { "TIF",    00046,    0,     TYPE_B,     709  },
   { "TIO",    00042,    0,     TYPE_B,     709  },
   { "TITLE",  TITLE_T,  0,     TYPE_P,     704  },
   { "TIX",    02000,    0,     TYPE_A,     704  },
   { "TLQ",    00040,    0,     TYPE_B,     704  },
   { "TMI",    04120,    0,     TYPE_B,     704  },
   { "TNO",    04140,    0,     TYPE_B,     704  },
   { "TNX",    06000,    0,     TYPE_A,     704  },
   { "TNZ",    04100,    0,     TYPE_B,     704  },
   { "TOV",    00140,    0,     TYPE_B,     704  },
   { "TPL",    00120,    0,     TYPE_B,     704  },
   { "TQO",    00161,    0,     TYPE_B,     704  },
   { "TQP",    00162,    0,     TYPE_B,     704  },
   { "TRA",    00020,    0,     TYPE_B,     704  },
   { "TRCA",   00022,    0,     TYPE_B,     709  },
   { "TRCB",   04022,    0,     TYPE_B,     709  },
   { "TRCC",   00024,    0,     TYPE_B,     709  },
   { "TRCD",   04024,    0,     TYPE_B,     709  },
   { "TRCE",   00026,    0,     TYPE_B,     709  },
   { "TRCF",   04026,    0,     TYPE_B,     709  },
   { "TRCG",   00027,    0,     TYPE_B,     709  },
   { "TRCH",   04027,    0,     TYPE_B,     709  },
   { "TSX",    00074,    0,     TYPE_B,     704  },
   { "TTL",    TTL_T,    0,     TYPE_P,     704  },
   { "TTR",    00021,    0,     TYPE_B,     704  },
   { "TWO",    02000,    0,     TYPE_A,     704  },
   { "TWT",    03400,    0,     TYPE_CHAN,  709  },
   { "TXH",    03000,    0,     TYPE_A,     704  },
   { "TXI",    01000,    0,     TYPE_A,     704  },
   { "TXL",    07000,    0,     TYPE_A,     704  },
   { "TZE",    00100,    0,     TYPE_B,     704  },
   { "UAM",    04304,    0,     TYPE_B,     709  },
   { "UFA",    04300,    0,     TYPE_B,     704  },
   { "UFM",    04260,    0,     TYPE_B,     704  },
   { "UFS",    04302,    0,     TYPE_B,     704  },
   { "UNLIST", UNLIST_T, 0,     TYPE_P,     704  },
   { "UNPNCH", UNPNCH_T, 0,     TYPE_P,     704  },
   { "USE",    USE_T,    0,     TYPE_P,     704  },
   { "USM",    04306,    0,     TYPE_B,     709  },
   { "VDH",    00224,    0,     TYPE_C,     709  },
   { "VDP",    00225,    0,     TYPE_C,     709  },
   { "VFD",    VFD_T,    0,     TYPE_P,     704  },
   { "VLM",    00204,    0,     TYPE_C,     709  },
   { "WDR",    00766,    00300, TYPE_E,     704  },
   { "WEF",    00770,    00220, TYPE_E,     704  },
   { "WPB",    00766,    00362, TYPE_E,     704  },
   { "WPD",    00766,    00361, TYPE_E,     704  },
   { "WPR",    00766,    00361, TYPE_E,     704  },
   { "WPU",    00766,    00341, TYPE_E,     704  },
   { "WRS",    00766,    0,     TYPE_E,     704  },
   { "WTB",    00766,    00220, TYPE_E,     704  },
   { "WTD",    00766,    00200, TYPE_E,     704  },
   { "WTR",    00000,    0,     TYPE_CHAN,  709  },
   { "WTV",    00766,    00030, TYPE_E,     704  },
   { "XCA",    00131,    0,     TYPE_B,     709  },
   { "XCL",    04130,    0,     TYPE_B,     709  },
   { "XEC",    00522,    0,     TYPE_B,     709  },
   { "XIT",    00021,    0,     TYPE_B,     704  },
   { "XMT",    00000,    02,    TYPE_A,     709  },
   { "ZAC",    04754,    0,     TYPE_B,     704  },
   { "ZERO",   00000,    0,     TYPE_A,     704  },
   { "ZET",    00520,    0,     TYPE_B,     709  },
   { "ZSA",    00634,    0,     TYPE_B,     709  },
   { "ZSD",    04634,    0,     TYPE_B,     704  },
};

extern int cpumodel;		/* CPU model (704, 709, 7090, 7094) */
extern int fapmode;		/* FAP assembly mode */

/*
** Added opcodes
*/

extern int opdefcount;		/* Number of user defined opcode in opdef */
extern OpDefCode *freeops;	/* Reusable opdef codes */
extern OpDefCode *opdefcode[MAXDEFOPS];/* The user defined opcode table */

static char retopcode[MAXSYMLEN+2]; /* Return opcode symbol buffer */
static OpCode retop;		/* Return opcode buffer */

/***********************************************************************
* freeopd - Link a opcode entry on free chain.
***********************************************************************/

void
freeopd (OpDefCode *or)
{
   or->next = freeops;
   freeops = or;
}

/***********************************************************************
* stdlookup - Lookup standard opcode. Uses binary search.
***********************************************************************/

static OpCode *
stdlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("stdlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = NUMOPS;

   done = FALSE;
   while (!done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (last == mid) break;
      r = strcmp (optable[mid].opcode, op);
      if (r == 0)
      {
	 if (fapmode || optable[mid].model <= cpumodel)
	    ret = &optable[mid];
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* addlookup - Lookup added opcode. Uses binary search.
***********************************************************************/

static OpCode *
addlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("addlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = opdefcount;
   
   while (opdefcount && !done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (opdefcount == 1) done = TRUE;
      else if (last == mid) break;
         
      r = strcmp (opdefcode[mid]->opcode, op);
      if (r == 0)
      {
	 strcpy (retopcode, op);
	 retop.opcode = retopcode;
	 retop.opvalue = opdefcode[mid]->opvalue;
	 retop.opmod = opdefcode[mid]->opmod;
	 retop.optype = opdefcode[mid]->optype;
	 ret = &retop;
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* oplookup - Lookup opcode.
***********************************************************************/

OpCode *
oplookup (char *op)
{
   OpCode *ret = NULL;
   int i;
   int opch;
   char opbf[MAXSYMLEN+2];

#ifdef DEBUGOP
   printf ("oplookup: Entered: op = %s\n", op);
#endif

   /*
   ** Check added opcodes first, incase of override
   */

   if (!(ret = addlookup (op)))
   {
      /*
      ** Check standard opcode table
      */

      if (!(ret = stdlookup (op)))
      {

	 /*
	 ** Check if it's an I/O opcode. Like WPR[A-H]
	 */

	 strcpy (opbf, op);
	 i = strlen(opbf) - 1;
	 opch = opbf[i];
	 opbf[i] = '\0';
	 if (i == 3 && opch >= 'A' && opch <= 'H')
	 {
	    if ((ret = stdlookup (opbf)) != NULL)
	    {
	       strcpy (retopcode, op);
	       retop.opcode = retopcode;
	       retop.opvalue = ret->opvalue;
	       retop.opmod = ret->opmod | (opch - 'A' + 1) << 9;
	       retop.optype = ret->optype;
	       ret = &retop;
	    }
         }
      }
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* opadd - Add opcode.
***********************************************************************/

void
opadd (char *op, unsigned c0, unsigned c1, int type)
{
   OpDefCode *new;
   int lo, up;

#ifdef DEBUGOP
   printf ("opadd: Entered: op = %s, c0 = %4.4o, c1 = %8.8o, type = %d\n",
	    op, c0, c1, type);
#endif

   /*
   ** Allocate storage for opcode and fill it in
   */

   if (opdefcount+1 > MAXDEFOPS)
   {
      fprintf (stderr, "asm7090: Op Code table exceeded\n");
      exit (ABORT);
   }

   if (freeops)
   {
      new = freeops;
      freeops = new->next;
   }
   else if ((new = (OpDefCode *)DBG_MEMGET (sizeof (OpDefCode))) == NULL)
   {
      fprintf (stderr, "asm7090: Unable to allocate memory\n");
      exit (ABORT);
   }

   memset (new, '\0', sizeof (OpDefCode));
   strcpy (new->opcode, op);
   new->opvalue = c0;
   new->opmod = c1;
   new->optype = type;

   if (opdefcount == 0)
   {
      opdefcode[0] = new;
      opdefcount++;
      return;
   }

   /*
   ** Insert pointer in sort order.
   */

   for (lo = 0; lo < opdefcount; lo++)
   {
      if (strcmp (opdefcode[lo]->opcode, op) > 0)
      {
	 for (up = opdefcount + 1; up > lo; up--)
	 {
	    opdefcode[up] = opdefcode[up-1];
	 }
	 opdefcode[lo] = new;
	 opdefcount++;
	 return;
      }
   }
   opdefcode[opdefcount] = new;
   opdefcount++;
}

/***********************************************************************
* opdel - Delete an op code from the table.
***********************************************************************/

void
opdel (char *op)
{
   int i;

#ifdef DEBUGOP
   printf ("opdel: Entered: op = %s\n",
	    op);
#endif
   for (i = 0; i < opdefcount; i++)
   {
      if (!strcmp (opdefcode[i]->opcode, op))
      {
         DBG_MEMREL (opdefcode[i]);
	 for (; i < opdefcount; i++)
	 {
	    opdefcode[i] = opdefcode[i+1];
	 }
	 opdefcount --;
         return;
      }
   }
}
