/***********************************************************************
*
* asmoptab.c - Opcode table for the IBM 7090 assembler.
*
* Changes:
*   05/21/03   DGP   Original.
*   12/08/04   DGP   Started adding MAP pseudo ops.
*   02/02/05   DGP   Added some missing opcodes, THREE, SEVEN & ZERO.
*   03/01/05   DGP   Change IIA inst type.
*	
***********************************************************************/

#include <string.h>
#include <memory.h>
#include <stdlib.h>

#include "asmdef.h"
#include "asmdmem.h"

/*
** Standard opcodes
*/

OpCode optable[NUMOPS] =
{
   { "ABS",    ABS_T,    0,     TYPE_P,     7090 },
   { "ACL",    00361,    0,     TYPE_B,     7090 },
   { "ADD",    00400,    0,     TYPE_B,     7090 },
   { "ADM",    00401,    0,     TYPE_B,     7090 },
   { "ALS",    00767,    0,     TYPE_B,     7090 },
   { "ANA",    04320,    0,     TYPE_B,     7090 },
   { "ANS",    00320,    0,     TYPE_B,     7090 },
   { "ARS",    00771,    0,     TYPE_B,     7090 },
   { "AXC",    04774,    0,     TYPE_B,     7090 },
   { "AXT",    00774,    0,     TYPE_B,     7090 },
   { "BCD",    BCD_T,    0,     TYPE_P,     7090 },
   { "BCI",    BCI_T,    0,     TYPE_P,     7090 },
   { "BEGIN",  BEGIN_T,  0,     TYPE_P,     7090 },
   { "BES",    BES_T,    0,     TYPE_P,     7090 },
   { "BFT",    BFT_T,    0,     TYPE_P,     7090 },
   { "BNT",    BNT_T,    0,     TYPE_P,     7090 },
   { "BOOL",   BOOL_T,   0,     TYPE_P,     7090 },
   { "BRA",    07000,    0,     TYPE_A,     7090 },
   { "BRN",    03000,    0,     TYPE_A,     7090 },
   { "BSF",    04764,    0,     TYPE_E,     7090 },
   { "BSR",    00764,    0,     TYPE_E,     7090 },
   { "BSS",    BSS_T,    0,     TYPE_P,     7090 },
   { "BTT",    00760,    0,     TYPE_E,     7090 },
   { "CAD",    04700,    0,     TYPE_B,     7090 },
   { "CAL",    04500,    0,     TYPE_B,     7090 },
   { "CALL",   CALL_T,   0,     TYPE_P,     7090 },
   { "CAQ",    04114,    0,     TYPE_C,     7090 },
   { "CAS",    00340,    0,     TYPE_B,     7090 },
   { "CFF",    00760,    00030, TYPE_E,     7090 },
   { "CHS",    00760,    00002, TYPE_E,     7090 },
   { "CLA",    00500,    0,     TYPE_B,     7090 },
   { "CLM",    00760,    00000, TYPE_E,     7090 },
   { "CLS",    00502,    0,     TYPE_B,     7090 },
   { "COM",    00760,    00006, TYPE_E,     7090 },
   { "COMMON", COMMON_T, 0,     TYPE_P,     7090 },
   { "CONTRL", CONTRL_T, 0,     TYPE_P,     7090 },
   { "COUNT",  COUNT_T,  0,     TYPE_P,     7090 },
   { "CPY",    00700,    0,     TYPE_B,     7090 },
   { "CPYD",   05000,    0,     TYPE_A,     7090 },
   { "CPYP",   04000,    0,     TYPE_CHAN,  7090 },
   { "CRQ",    04154,    0,     TYPE_C,     7090 },
   { "CTL",    02000,    0,     TYPE_CHAN,  7090 },
   { "CTLR",   02000,    02,    TYPE_CHAN,  7090 },
   { "CTLW",   02400,    00,    TYPE_CHAN,  7090 },
   { "CVR",    00114,    0,     TYPE_C,     7090 },
   { "DCT",    00760,    00012, TYPE_E,     7090 },
   { "DEBM",   01210,    01212, TYPE_DISK,  7090 },
   { "DEC",    DEC_T,    0,     TYPE_P,     7090 },
   { "DETAIL", DETAIL_T, 0,     TYPE_P,     7090 },
   { "DFAD",   00301,    0,     TYPE_B,     7094 },
   { "DFAM",   00305,    0,     TYPE_B,     7094 },
   { "DFDH",   04240,    0,     TYPE_B,     7094 },
   { "DFDP",   04241,    0,     TYPE_B,     7094 },
   { "DFMP",   00261,    0,     TYPE_B,     7094 },
   { "DFSB",   00303,    0,     TYPE_B,     7094 },
   { "DFSM",   00307,    0,     TYPE_B,     7094 },
   { "DLD",    00443,    0,     TYPE_B,     7094 },
   { "DNOP",   01212,    01212, TYPE_DISK,  7090 },
   { "DREL",   01204,    01212, TYPE_DISK,  7090 },
   { "DSAI",   01007,    01212, TYPE_DISK,  7090 },
   { "DSBM",   01211,    01212, TYPE_DISK,  7090 },
   { "DSEK",   01012,    01212, TYPE_DISK,  7090 },
   { "DST",    04603,    0,     TYPE_B,     7094 },
   { "DUAM",   04305,    0,     TYPE_B,     7094 },
   { "DUFA",   04301,    0,     TYPE_B,     7094 },
   { "DUFM",   04261,    0,     TYPE_B,     7094 },
   { "DUFS",   04303,    0,     TYPE_B,     7094 },
   { "DUP",    DUP_T,    0,     TYPE_P,     7090 },
   { "DUSM",   04307,    0,     TYPE_B,     7094 },
   { "DVCY",   01005,    01212, TYPE_DISK,  7090 },
   { "DVH",    00220,    0,     TYPE_B,     7090 },
   { "DVHA",   01011,    01212, TYPE_DISK,  7090 },
   { "DVP",    00221,    0,     TYPE_B,     7090 },
   { "DVSR",   01002,    01212, TYPE_DISK,  7090 },
   { "DVTA",   01010,    01212, TYPE_DISK,  7090 },
   { "DVTN",   01004,    01212, TYPE_DISK,  7090 },
   { "DWRC",   01006,    01212, TYPE_DISK,  7090 },
   { "DWRF",   01003,    01212, TYPE_DISK,  7090 },
   { "ECTM",   04760,    00006, TYPE_E,     7090 },
   { "EFTM",   04760,    00002, TYPE_E,     7090 },
   { "EJECT",  EJE_T,    0,     TYPE_P,     7090 },
   { "EMTM",   04760,    00016, TYPE_E,     7094 },
   { "ENB",    00564,    0,     TYPE_B,     7090 },
   { "END",    END_T,    0,     TYPE_P,     7090 },
   { "ENDM",   ENDM_T,   0,     TYPE_P,     7090 },
   { "ENDQ",   ENDQ_T,   0,     TYPE_P,     7090 },
   { "ENK",    00760,    00004, TYPE_E,     7090 },
   { "ENTRY",  ENT_T,    0,     TYPE_P,     7090 },
   { "EQU",    EQU_T,    0,     TYPE_P,     7090 },
   { "ERA",    00322,    0,     TYPE_B,     7090 },
   { "ESNT",   04021,    0,     TYPE_B,     7094 },
   { "ESTM",   04760,    00005, TYPE_E,     7090 },
   { "ETC",    ETC_T,    0,     TYPE_P,     7090 },
   { "ETM",    00760,    00007, TYPE_E,     7090 },
   { "ETT",    04760,    0,     TYPE_E,     7090 },
   { "EVEN",   EVEN_T,   0,     TYPE_P,     7090 },
   { "EXTERN", EXT_T,    0,     TYPE_P,     7090 },
   { "FAD",    00300,    0,     TYPE_B,     7090 },
   { "FAM",    00304,    0,     TYPE_B,     7090 },
   { "FDH",    00240,    0,     TYPE_B,     7090 },
   { "FDP",    00241,    0,     TYPE_B,     7090 },
   { "FILE",   FILE_T,   0,     TYPE_P,     7090 },
   { "FIVE",   05000,    0,     TYPE_A,     7090 },
   { "FMP",    00260,    0,     TYPE_B,     7090 },
   { "FOR",    04000,    0,     TYPE_A,     7090 },
   { "FOUR",   04000,    0,     TYPE_A,     7090 },
   { "FRN",    00760,    00011, TYPE_E,     7090 },
   { "FSB",    00302,    0,     TYPE_B,     7090 },
   { "FSM",    00306,    0,     TYPE_B,     7090 },
   { "FUL",    FUL_T,    0,     TYPE_P,     7090 },
   { "FVE",    05000,    0,     TYPE_A,     7090 },
   { "GOTO",   GOTO_T,   0,     TYPE_P,     7090 },
   { "HEAD",   HEAD_T,   0,     TYPE_P,     7090 },
   { "HPR",    00420,    0,     TYPE_B,     7090 },
   { "HTR",    00000,    0,     TYPE_B,     7090 },
   { "ICC",    07000,    02,    TYPE_CHAN,  7090 },
   { "IFF",    IFF_T,    0,     TYPE_P,     7090 },
   { "IFT",    IFT_T,    0,     TYPE_P,     7090 },
   { "IIA",    00041,    0,     TYPE_B,     7090 },
   { "IIB",    IIB_T,    0,     TYPE_P,     7090 },
   { "IIL",    04051,    0,     TYPE_D,     7090 },
   { "IIR",    00051,    0,     TYPE_D,     7090 },
   { "IIS",    00440,    0,     TYPE_B,     7090 },
   { "INDEX",  INDEX_T,  0,     TYPE_P,     7090 },
   { "IOCD",   00000,    0,     TYPE_A,     7090 },
   { "IOCDN",  00000,    02,    TYPE_A,     7090 },
   { "IOCP",   04000,    0,     TYPE_A,     7090 },
   { "IOCPN",  04000,    02,    TYPE_A,     7090 },
   { "IOCT",   05000,    0,     TYPE_A,     7090 },
   { "IOCTN",  05000,    02,    TYPE_A,     7090 },
   { "IORP",   02000,    0,     TYPE_A,     7090 },
   { "IORPN",  02000,    02,    TYPE_A,     7090 },
   { "IORT",   03000,    0,     TYPE_A,     7090 },
   { "IORTN",  03000,    02,    TYPE_A,     7090 },
   { "IOSP",   06000,    0,     TYPE_A,     7090 },
   { "IOSPN",  06000,    02,    TYPE_A,     7090 },
   { "IOST",   07000,    0,     TYPE_A,     7090 },
   { "IOSTN",  07000,    02,    TYPE_A,     7090 },
   { "IOT",    00760,    00005, TYPE_E,     7090 },
   { "IRP",    IRP_T,    0,     TYPE_P,     7090 },
   { "KEEP",   KEEP_T,   0,     TYPE_P,     7090 },
   { "LABEL",  LABEL_T,  0,     TYPE_P,     7090 },
   { "LAC",    00535,    0,     TYPE_B,     7090 },
   { "LAR",    03000,    0,     TYPE_CHAN,  7090 },
   { "LAS",    04340,    0,     TYPE_B,     7090 },
   { "LBL",    LBL_T,    0,     TYPE_P,     7090 },
   { "LBOOL",  LBOOL_T,  0,     TYPE_P,     7090 },
   { "LBT",    00760,    00001, TYPE_E,     7090 },
   { "LCC",    06400,    02,    TYPE_CHAN,  7090 },
   { "LCHA",   00544,    0,     TYPE_B,     7090 },
   { "LCHB",   04544,    0,     TYPE_B,     7090 },
   { "LCHC",   00545,    0,     TYPE_B,     7090 },
   { "LCHD",   04545,    0,     TYPE_B,     7090 },
   { "LCHE",   00546,    0,     TYPE_B,     7090 },
   { "LCHF",   04546,    0,     TYPE_B,     7090 },
   { "LCHG",   00547,    0,     TYPE_B,     7090 },
   { "LCHH",   04547,    0,     TYPE_B,     7090 },
   { "LDA",    00460,    0,     TYPE_B,     7090 },
   { "LDC",    04535,    0,     TYPE_B,     7090 },
   { "LDI",    00441,    0,     TYPE_B,     7090 },
   { "LDIR",   LDIR_T,   0,     TYPE_P,     7090 },
   { "LDQ",    00560,    0,     TYPE_B,     7090 },
   { "LFT",    04054,    0,     TYPE_D,     7090 },
   { "LFTM",   04760,    00004, TYPE_E,     7090 },
   { "LGL",    04763,    0,     TYPE_B,     7090 },
   { "LGR",    04765,    0,     TYPE_B,     7090 },
   { "LIP",    06000,    02,    TYPE_CHAN,  7090 },
   { "LIPT",   01000,    02,    TYPE_CHAN,  7090 },
   { "LIST",   LIST_T,   0,     TYPE_P,     7090 },
   { "LIT",    LIT_T,    0,     TYPE_P,     7090 },
   { "LITORG", LITORG_T, 0,     TYPE_P,     7090 },
   { "LLS",    00763,    0,     TYPE_B,     7090 },
   { "LMTM",   00760,    00016, TYPE_E,     7094 },
   { "LNT",    04056,    0,     TYPE_D,     7090 },
   { "LOC",    LOC_T,    0,     TYPE_P,     7090 },
   { "LORG",   LORG_T,   0,     TYPE_P,     7090 },
   { "LRS",    00765,    0,     TYPE_B,     7090 },
   { "LSNM",   04760,    00010, TYPE_E,     7090 },
   { "LTM",    04760,    00007, TYPE_E,     7090 },
   { "LXA",    00534,    0,     TYPE_B,     7090 },
   { "LXD",    04534,    0,     TYPE_B,     7090 },
   { "MACRO",  MACRO_T,  0,     TYPE_P,     7090 },
   { "MAX",    MAX_T,    0,     TYPE_P,     7090 },
   { "MIN",    MIN_T,    0,     TYPE_P,     7090 },
   { "MON",    05000,    0,     TYPE_A,     7090 },
   { "MPR",    04200,    0,     TYPE_B,     7090 },
   { "MPY",    00200,    0,     TYPE_B,     7090 },
   { "MSE",    04760,    00000, TYPE_E,     7090 },
   { "MTH",    07000,    0,     TYPE_A,     7090 },
   { "MTW",    06000,    0,     TYPE_A,     7090 },
   { "MZE",    04000,    0,     TYPE_A,     7090 },
   { "NOCRS",  NOCRS_T,  0,     TYPE_P,     7090 },
   { "NOP",    00761,    0,     TYPE_B,     7090 },
   { "NTR",    01000,    0,     TYPE_A,     7090 },
   { "NULL",   NULL_T,   0,     TYPE_P,     7090 },
   { "NZT",    04520,    0,     TYPE_B,     7090 },
   { "OAI",    00043,    0,     TYPE_B,     7090 },
   { "OCT",    OCT_T,    0,     TYPE_P,     7090 },
   { "OFT",    00444,    0,     TYPE_B,     7090 },
   { "ONE",    01000,    0,     TYPE_A,     7090 },
   { "ONT",    00446,    0,     TYPE_B,     7090 },
   { "OPD",    OPD_T,    0,     TYPE_P,     7090 },
   { "OPSYN",  OPSYN_T,  0,     TYPE_P,     7090 },
   { "OPVFD",  OPVFD_T,  0,     TYPE_P,     7090 },
   { "ORA",    04501,    0,     TYPE_B,     7090 },
   { "ORG",    ORG_T,    0,     TYPE_P,     7090 },
   { "ORGCRS", ORGCRS_T, 0,     TYPE_P,     7090 },
   { "ORS",    04602,    0,     TYPE_B,     7090 },
   { "OSI",    00442,    0,     TYPE_B,     7090 },
   { "PAC",    00737,    0,     TYPE_B,     7090 },
   { "PAI",    00044,    0,     TYPE_B,     7090 },
   { "PAX",    00734,    0,     TYPE_B,     7090 },
   { "PBT",    04760,    00001, TYPE_E,     7090 },
   { "PCA",    00756,    0,     TYPE_B,     7094 }, 
   { "PCC",    PCC_T,    0,     TYPE_P,     7090 },
   { "PCD",    04756,    0,     TYPE_B,     7094 },
   { "PCG",    PCG_T,    0,     TYPE_P,     7090 },
   { "PDC",    04737,    0,     TYPE_B,     7090 },
   { "PDX",    04734,    0,     TYPE_B,     7090 },
   { "PIA",    04046,    0,     TYPE_B,     7090 },
   { "PMC",    PMC_T,    0,     TYPE_P,     7090 },
   { "PON",    01000,    0,     TYPE_A,     7090 },
   { "PSE",    00760,    00000, TYPE_E,     7090 },
   { "PTH",    03000,    0,     TYPE_A,     7090 },
   { "PTW",    02000,    0,     TYPE_A,     7090 },
   { "PUNCH",  PUNCH_T,  0,     TYPE_P,     7090 },
   { "PURGE",  PURGE_T,  0,     TYPE_P,     7090 },
   { "PXA",    00754,    0,     TYPE_B,     7090 },
   { "PXD",    04754,    0,     TYPE_B,     7090 },
   { "PZE",    00000,    0,     TYPE_A,     7090 },
   { "QUAL",   QUAL_T,   0,     TYPE_P,     7090 },
   { "RBOOL",  RBOOL_T,  0,     TYPE_P,     7090 },
   { "RCD",    00762,    00321, TYPE_E,     7090 },
   { "RCHA",   00540,    0,     TYPE_B,     7090 },
   { "RCHB",   04540,    0,     TYPE_B,     7090 },
   { "RCHC",   00541,    0,     TYPE_B,     7090 },
   { "RCHD",   04541,    0,     TYPE_B,     7090 },
   { "RCHE",   00542,    0,     TYPE_B,     7090 },
   { "RCHF",   04542,    0,     TYPE_B,     7090 },
   { "RCHG",   00543,    0,     TYPE_B,     7090 },
   { "RCHH",   04543,    0,     TYPE_B,     7090 },
   { "RCT",    00760,    00014, TYPE_E,     7090 },
   { "RDC",    00760,    00352, TYPE_E,     7090 },
   { "RDR",    00762,    00300, TYPE_E,     7090 },
   { "RDS",    00762,    0,     TYPE_E,     7090 },
   { "REF",    REF_T,    0,     TYPE_P,     7090 },
   { "REM",    REM_T,    0,     TYPE_P,     7090 },
   { "RETURN", RETURN_T, 0,     TYPE_P,     7090 },
   { "REW",    00772,    0,     TYPE_B,     7090 },
   { "RFT",    00054,    0,     TYPE_D,     7090 },
   { "RIA",    04042,    0,     TYPE_D,     7090 },
   { "RIB",    RIB_T,    0,     TYPE_P,     7090 },
   { "RIC",    00760,    00350, TYPE_E,     7090 },
   { "RIL",    04057,    0,     TYPE_D,     7090 },
   { "RIR",    00057,    0,     TYPE_D,     7090 },
   { "RIS",    00445,    0,     TYPE_B,     7090 },
   { "RND",    00760,    00010, TYPE_E,     7090 },
   { "RNT",    00056,    0,     TYPE_D,     7090 },
   { "RPR",    00762,    00361, TYPE_E,     7090 },
   { "RQL",    04773,    0,     TYPE_B,     7090 },
   { "RSCA",   00540,    0,     TYPE_B,     7090 },
   { "RSCB",   04540,    0,     TYPE_B,     7090 },
   { "RSCC",   00541,    0,     TYPE_B,     7090 },
   { "RSCD",   04541,    0,     TYPE_B,     7090 },
   { "RSCE",   00542,    0,     TYPE_B,     7090 },
   { "RSCF",   04542,    0,     TYPE_B,     7090 },
   { "RSCG",   00543,    0,     TYPE_B,     7090 },
   { "RSCH",   04543,    0,     TYPE_B,     7090 },
   { "RTB",    00762,    00220, TYPE_E,     7090 },
   { "RTD",    00762,    00200, TYPE_E,     7090 },
   { "RUN",    04772,    0,     TYPE_E,     7090 },
   { "SAR",    03000,    02,    TYPE_CHAN,  7090 },
   { "SAVE",   SAVE_T,   0,     TYPE_P,     7090 },
   { "SAVEN",  SAVEN_T,  0,     TYPE_P,     7090 },
   { "SBM",    04400,    0,     TYPE_B,     7090 },
   { "SCA",    00636,    0,     TYPE_B,     7094 },
   { "SCD",    04636,    0,     TYPE_B,     7094 },
   { "SCHA",   00640,    0,     TYPE_B,     7090 },
   { "SCHB",   04640,    0,     TYPE_B,     7090 },
   { "SCHC",   00641,    0,     TYPE_B,     7090 },
   { "SCHD",   04641,    0,     TYPE_B,     7090 },
   { "SCHE",   00642,    0,     TYPE_B,     7090 },
   { "SCHF",   04642,    0,     TYPE_B,     7090 },
   { "SCHG",   00643,    0,     TYPE_B,     7090 },
   { "SCHH",   04643,    0,     TYPE_B,     7090 },
   { "SDN",    00776,    0,     TYPE_B,     7090 },
   { "SET",    SET_T,    0,     TYPE_P,     7090 },
   { "SEVEN",  07000,    0,     TYPE_A,     7090 },
   { "SIB",    SIB_T,    0,     TYPE_P,     7090 },
   { "SIL",    04055,    0,     TYPE_D,     7090 },
   { "SIR",    00055,    0,     TYPE_D,     7090 },
   { "SIX",    06000,    0,     TYPE_A,     7090 },
   { "SLF",    00760,    00140, TYPE_E,     7090 },
   { "SLN",    00760,    00140, TYPE_E,     7090 },
   { "SLQ",    04620,    0,     TYPE_B,     7090 },
   { "SLT",    04760,    00140, TYPE_E,     7090 },
   { "SLW",    00602,    0,     TYPE_B,     7090 },
   { "SMS",    07000,    0,     TYPE_CHAN,  7090 },
   { "SNS",    02400,    02,    TYPE_CHAN,  7090 },
   { "SPACE",  SPC_T,    0,     TYPE_P,     7090 },
   { "SPR",    00760,    00360, TYPE_E,     7090 },
   { "SPT",    00760,    00360, TYPE_E,     7090 },
   { "SPU",    00760,    00340, TYPE_E,     7090 },
   { "SSM",    04760,    00003, TYPE_E,     7090 },
   { "SSP",    00760,    00003, TYPE_E,     7090 },
   { "SST",    SST_T,    0,     TYPE_P,     7090 },
   { "STA",    00621,    0,     TYPE_B,     7090 },
   { "STCA",   00544,    0,     TYPE_E,     7090 },
   { "STCB",   04544,    0,     TYPE_E,     7090 },
   { "STCC",   00545,    0,     TYPE_E,     7090 },
   { "STCD",   04545,    0,     TYPE_E,     7090 },
   { "STCE",   00546,    0,     TYPE_E,     7090 },
   { "STCF",   04546,    0,     TYPE_E,     7090 },
   { "STCG",   00547,    0,     TYPE_E,     7090 },
   { "STCH",   04547,    0,     TYPE_E,     7090 },
   { "STD",    00622,    0,     TYPE_B,     7090 },
   { "STI",    00604,    0,     TYPE_B,     7090 },
   { "STL",    04625,    0,     TYPE_B,     7090 },
   { "STO",    00601,    0,     TYPE_B,     7090 },
   { "STP",    00630,    0,     TYPE_B,     7090 },
   { "STQ",    04600,    0,     TYPE_B,     7090 },
   { "STR",    05000,    0,     TYPE_A,     7090 },
   { "STT",    00625,    0,     TYPE_B,     7090 },
   { "STZ",    00600,    0,     TYPE_B,     7090 },
   { "SUB",    00402,    0,     TYPE_B,     7090 },
   { "SVN",    07000,    0,     TYPE_A,     7090 },
   { "SWT",    00760,    00160, TYPE_E,     7090 },
   { "SXA",    00634,    0,     TYPE_B,     7090 },
   { "SXD",    04634,    0,     TYPE_B,     7090 },
   { "SYN",    EQU_T,    0,     TYPE_P,     7090 },
   { "TCD",    TCD_T,    0,     TYPE_P,     7090 },
   { "TCH",    01000,    0,     TYPE_A,     7090 },
   { "TCM",    05000,    02,    TYPE_CHAN,  7090 },
   { "TCNA",   04060,    0,     TYPE_B,     7090 },
   { "TCNB",   04061,    0,     TYPE_B,     7090 },
   { "TCNC",   04062,    0,     TYPE_B,     7090 },
   { "TCND",   04063,    0,     TYPE_B,     7090 },
   { "TCNE",   04064,    0,     TYPE_B,     7090 },
   { "TCNF",   04065,    0,     TYPE_B,     7090 },
   { "TCNG",   04066,    0,     TYPE_B,     7090 },
   { "TCNH",   04067,    0,     TYPE_B,     7090 },
   { "TCOA",   00060,    0,     TYPE_B,     7090 },
   { "TCOB",   00061,    0,     TYPE_B,     7090 },
   { "TCOC",   00062,    0,     TYPE_B,     7090 },
   { "TCOD",   00063,    0,     TYPE_B,     7090 },
   { "TCOE",   00064,    0,     TYPE_B,     7090 },
   { "TCOF",   00065,    0,     TYPE_B,     7090 },
   { "TCOG",   00066,    0,     TYPE_B,     7090 },
   { "TCOH",   00067,    0,     TYPE_B,     7090 },
   { "TDC",    06400,    0,     TYPE_CHAN,  7090 },
   { "TEFA",   00030,    0,     TYPE_B,     7090 },
   { "TEFB",   04030,    0,     TYPE_B,     7090 },
   { "TEFC",   00031,    0,     TYPE_B,     7090 },
   { "TEFD",   04031,    0,     TYPE_B,     7090 },
   { "TEFE",   00032,    0,     TYPE_B,     7090 },
   { "TEFF",   04032,    0,     TYPE_B,     7090 },
   { "TEFG",   00033,    0,     TYPE_B,     7090 },
   { "TEFH",   04033,    0,     TYPE_B,     7090 },
   { "THREE",  03000,    0,     TYPE_A,     7090 },
   { "TIF",    00046,    0,     TYPE_B,     7090 },
   { "TIO",    00042,    0,     TYPE_B,     7090 },
   { "TITLE",  TITLE_T,  0,     TYPE_P,     7090 },
   { "TIX",    02000,    0,     TYPE_A,     7090 },
   { "TLQ",    00040,    0,     TYPE_B,     7090 },
   { "TMI",    04120,    0,     TYPE_B,     7090 },
   { "TNO",    04140,    0,     TYPE_B,     7090 },
   { "TNX",    06000,    0,     TYPE_A,     7090 },
   { "TNZ",    04100,    0,     TYPE_B,     7090 },
   { "TOV",    00140,    0,     TYPE_B,     7090 },
   { "TPL",    00120,    0,     TYPE_B,     7090 },
   { "TQO",    00161,    0,     TYPE_B,     7090 },
   { "TQP",    00162,    0,     TYPE_B,     7090 },
   { "TRA",    00020,    0,     TYPE_B,     7090 },
   { "TRCA",   00022,    0,     TYPE_B,     7090 },
   { "TRCB",   04022,    0,     TYPE_B,     7090 },
   { "TRCC",   00024,    0,     TYPE_B,     7090 },
   { "TRCD",   04024,    0,     TYPE_B,     7090 },
   { "TRCE",   00026,    0,     TYPE_B,     7090 },
   { "TRCF",   04026,    0,     TYPE_B,     7090 },
   { "TRCG",   00027,    0,     TYPE_B,     7090 },
   { "TRCH",   04027,    0,     TYPE_B,     7090 },
   { "TSX",    00074,    0,     TYPE_B,     7090 },
   { "TTL",    TTL_T,    0,     TYPE_P,     7090 },
   { "TTR",    00021,    0,     TYPE_B,     7090 },
   { "TWO",    02000,    0,     TYPE_A,     7090 },
   { "TWT",    03400,    0,     TYPE_CHAN,  7090 },
   { "TXH",    03000,    0,     TYPE_A,     7090 },
   { "TXI",    01000,    0,     TYPE_A,     7090 },
   { "TXL",    07000,    0,     TYPE_A,     7090 },
   { "TZE",    00100,    0,     TYPE_B,     7090 },
   { "UAM",    04304,    0,     TYPE_B,     7090 },
   { "UFA",    04300,    0,     TYPE_B,     7090 },
   { "UFM",    04260,    0,     TYPE_B,     7090 },
   { "UFS",    04302,    0,     TYPE_B,     7090 },
   { "UNLIST", UNLIST_T, 0,     TYPE_P,     7090 },
   { "UNPNCH", UNPNCH_T, 0,     TYPE_P,     7090 },
   { "USE",    USE_T,    0,     TYPE_P,     7090 },
   { "USM",    04306,    0,     TYPE_B,     7090 },
   { "VDH",    00224,    0,     TYPE_C,     7090 },
   { "VDP",    00225,    0,     TYPE_C,     7090 },
   { "VFD",    VFD_T,    0,     TYPE_P,     7090 },
   { "VLM",    00204,    0,     TYPE_C,     7090 },
   { "WDR",    00766,    00300, TYPE_E,     7090 },
   { "WEF",    00770,    0,     TYPE_E,     7090 },
   { "WPB",    00766,    00362, TYPE_E,     7090 },
   { "WPD",    00766,    00361, TYPE_E,     7090 },
   { "WPR",    00766,    00361, TYPE_E,     7090 },
   { "WPU",    00766,    00341, TYPE_E,     7090 },
   { "WRS",    00766,    0,     TYPE_E,     7090 },
   { "WTB",    00766,    00220, TYPE_E,     7090 },
   { "WTD",    00766,    00200, TYPE_E,     7090 },
   { "WTR",    00000,    0,     TYPE_CHAN,  7090 },
   { "WTV",    00766,    00030, TYPE_E,     7090 },
   { "XCA",    00131,    0,     TYPE_B,     7090 },
   { "XCL",    04130,    0,     TYPE_B,     7090 },
   { "XEC",    00522,    0,     TYPE_B,     7090 },
   { "XIT",    00021,    0,     TYPE_B,     7090 },
   { "XMT",    00000,    02,    TYPE_A,     7090 },
   { "ZAC",    04754,    0,     TYPE_B,     7090 },
   { "ZERO",   00000,    0,     TYPE_A,     7090 },
   { "ZET",    00520,    0,     TYPE_B,     7090 },
   { "ZSA",    00634,    0,     TYPE_B,     7090 },
   { "ZSD",    04634,    0,     TYPE_B,     7090 },
};

extern int cpumodel;		/* CPU model (709, 7090 7094) */
extern int fapmode;		/* FAP assembly mode */

/*
** Added opcodes
*/

extern int opdefcount;		/* Number of user defined opcode in opdef */
extern OpDefCode *freeops;	/* Reusable opdef codes */
extern OpDefCode *opdefcode[MAXDEFOPS];/* The user defined opcode table */

static char retopcode[MAXSYMLEN+2]; /* Return opcode symbol buffer */
static OpCode retop;		/* Return opcode buffer */

/***********************************************************************
* freeopd - Link a opcode entry on free chain.
***********************************************************************/

void
freeopd (OpDefCode *or)
{
   or->next = freeops;
   freeops = or;
}

/***********************************************************************
* stdlookup - Lookup standard opcode. Uses binary search.
***********************************************************************/

static OpCode *
stdlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("stdlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = NUMOPS;

   done = FALSE;
   while (!done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (last == mid) break;
      r = strcmp (optable[mid].opcode, op);
      if (r == 0)
      {
	 if (fapmode || optable[mid].model <= cpumodel)
	    ret = &optable[mid];
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* addlookup - Lookup added opcode. Uses binary search.
***********************************************************************/

static OpCode *
addlookup (char *op)
{
   OpCode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGOP
   printf ("addlookup: Entered: op = %s\n", op);
#endif

   lo = 0;
   up = opdefcount;
   
   while (opdefcount && !done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGOP
      printf (" mid = %d, last = %d\n", mid, last);
#endif
      if (opdefcount == 1) done = TRUE;
      else if (last == mid) break;
         
      r = strcmp (opdefcode[mid]->opcode, op);
      if (r == 0)
      {
	 strcpy (retopcode, op);
	 retop.opcode = retopcode;
	 retop.opvalue = opdefcode[mid]->opvalue;
	 retop.opmod = opdefcode[mid]->opmod;
	 retop.optype = opdefcode[mid]->optype;
	 ret = &retop;
	 done = TRUE;
      }
      else if (r < 0)
      {
	 lo = mid;
      }
      else 
      {
	 up = mid;
      }
      last = mid;
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* oplookup - Lookup opcode.
***********************************************************************/

OpCode *
oplookup (char *op)
{
   OpCode *ret = NULL;
   int i;
   int opch;
   char opbf[MAXSYMLEN+2];

#ifdef DEBUGOP
   printf ("oplookup: Entered: op = %s\n", op);
#endif

   /*
   ** Check added opcodes first, incase of override
   */

   if (!(ret = addlookup (op)))
   {
      /*
      ** Check standard opcode table
      */

      if (!(ret = stdlookup (op)))
      {

	 /*
	 ** Check if it's an I/O opcode. Like WPR[A-H]
	 */

	 strcpy (opbf, op);
	 i = strlen(opbf) - 1;
	 opch = opbf[i];
	 opbf[i] = '\0';
	 if (i == 3 && opch >= 'A' && opch <= 'H')
	 {
	    if ((ret = stdlookup (opbf)) != NULL)
	    {
	       strcpy (retopcode, op);
	       retop.opcode = retopcode;
	       retop.opvalue = ret->opvalue;
	       retop.opmod = ret->opmod | (opch - 'A' + 1) << 9;
	       retop.optype = ret->optype;
	       ret = &retop;
	    }
         }
      }
   }

#ifdef DEBUGOP
   printf (" ret = %8x\n", ret);
#endif
   return (ret);
}

/***********************************************************************
* opadd - Add opcode.
***********************************************************************/

void
opadd (char *op, unsigned c0, unsigned c1, int type)
{
   OpDefCode *new;
   int lo, up;

#ifdef DEBUGOP
   printf ("opadd: Entered: op = %s, c0 = %4.4o, c1 = %8.8o, type = %d\n",
	    op, c0, c1, type);
#endif

   /*
   ** Allocate storage for opcode and fill it in
   */

   if (opdefcount+1 > MAXDEFOPS)
   {
      fprintf (stderr, "asm7090: Op Code table exceeded\n");
      exit (ABORT);
   }

   if (freeops)
   {
      new = freeops;
      freeops = new->next;
   }
   else if ((new = (OpDefCode *)DBG_MEMGET (sizeof (OpDefCode))) == NULL)
   {
      fprintf (stderr, "asm7090: Unable to allocate memory\n");
      exit (ABORT);
   }

   memset (new, '\0', sizeof (OpDefCode));
   strcpy (new->opcode, op);
   new->opvalue = c0;
   new->opmod = c1;
   new->optype = type;

   if (opdefcount == 0)
   {
      opdefcode[0] = new;
      opdefcount++;
      return;
   }

   /*
   ** Insert pointer in sort order.
   */

   for (lo = 0; lo < opdefcount; lo++)
   {
      if (strcmp (opdefcode[lo]->opcode, op) > 0)
      {
	 for (up = opdefcount + 1; up > lo; up--)
	 {
	    opdefcode[up] = opdefcode[up-1];
	 }
	 opdefcode[lo] = new;
	 opdefcount++;
	 return;
      }
   }
   opdefcode[opdefcount] = new;
   opdefcount++;
}

/***********************************************************************
* opdel - Delete an op code from the table.
***********************************************************************/

void
opdel (char *op)
{
   int i;

#ifdef DEBUGOP
   printf ("opdel: Entered: op = %s\n",
	    op);
#endif
   for (i = 0; i < opdefcount; i++)
   {
      if (!strcmp (opdefcode[i]->opcode, op))
      {
         DBG_MEMREL (opdefcode[i]);
	 for (; i < opdefcount; i++)
	 {
	    opdefcode[i] = opdefcode[i+1];
	 }
	 opdefcount --;
         return;
      }
   }
}
