#!/usr/bin/perl

# Program to split card decks
# By Eric Smith <eric@brouhaha.com>
# Copyright 2000 The Computer Museum History Center
# http://www.computerhistory.org/
# $Id: cardsep.pl,v 1.3 2000/07/15 22:58:53 eric Exp eric $

use strict;

### Global variables:
my (
  %asc,       # card code to text table, hash of two-element arrays,
              #     [ alphanumeric, numeric ]

  $line,      # current line being processed

  $lines,     # line count, doesn't include non-card metadata (but does include separator cards)
  $cards_in_deck,

  $warn_bad_punches,

  $errors,    
  $warnings,

  $filename,  # name of output file, derived from cabinet/tray/drawer/deck

  $drawer_id, # drawer identifier, "c1t01d1"
  $deck,      # sequence number of deck within drawer, first deck is 1

# from metadata:

  $cabinet,   # identification of drawer
  $tray,
  $drawer,

  $date,      # date, from metadata
  $noc        # number of cards

);


%asc = (
	''       => [ ' ', '0' ],
	'12-3-8' => [ '.', '<RM821>' ],
	'12-4-8' => [ ')', '<NB>' ],
	'12'     => [ '+', '0' ],
        '11-3-8' => [ '$', '-<RM821>' ],
        '11-4-8' => [ '*', '-<NB>' ],
        '11'     => [ '-', '-0' ],
        '0-1'    => [ '/', '1' ],
        '0-3-8'  => [ ',', '<RM821>' ],
        '0-4-8'  => [ '(', '<NB>' ],
        '0-6-8'  => [ '<special>', '<special>' ],
        '3-8'    => [ '=', '<RM821>' ],
        '4-8'    => [ '@', '<NB>' ],
        '12-1'   => [ 'A', '1' ],
        '12-2'   => [ 'B', '2' ],
        '12-3'   => [ 'C', '3' ],
        '12-4'   => [ 'D', '4' ],
        '12-5'   => [ 'E', '5' ],
        '12-6'   => [ 'F', '6' ],
        '12-7'   => [ 'G', '7' ],
        '12-8'   => [ 'H', '8' ],
        '12-9'   => [ 'I', '9' ],
        '11-0'   => [ '-0', '-0' ],
        '11-1'   => [ 'J', '-1' ],
        '11-2'   => [ 'K', '-2' ],
        '11-3'   => [ 'L', '-3' ],
        '11-4'   => [ 'M', '-4' ],
        '11-5'   => [ 'N', '-5' ],
        '11-6'   => [ 'O', '-6' ],
        '11-7'   => [ 'P', '-7' ],
        '11-8'   => [ 'Q', '-8' ],
        '11-9'   => [ 'R', '-9' ],
        '0-2'    => [ 'S', '2' ],
        '0-3'    => [ 'T', '3' ],
        '0-4'    => [ 'U', '4' ],
        '0-5'    => [ 'V', '5' ],
        '0-6'    => [ 'W', '6' ],
        '0-7'    => [ 'X', '7' ],
        '0-8'    => [ 'Y', '8' ],
        '0-9'    => [ 'Z', '9' ],
        '0'      => [ '0', '0' ],
        '12-0'   => [ '0', '0' ],
        '1'      => [ '1', '1' ],
        '2'      => [ '2', '2' ],
        '3'      => [ '3', '3' ],
        '4'      => [ '4', '4' ],
        '5'      => [ '5', '5' ],
        '6'      => [ '6', '6' ],
        '7'      => [ '7', '7' ],
        '8'      => [ '8', '8' ],
        '9'      => [ '9', '9' ],
        '0-2-8'  => [ '<RM>', '<RM>' ],
        '11-2-8' => [ '-<RM>', '-<RM>' ],
        '12-2-8' => [ '-<RM>', '-<RM>' ],
        '0-7-8'  => [ '<GM>', '<GM>' ],
        '11-7-8' => [ '-<GM>', '-<GM>' ],
        '12-7-8' => [ '-<GM>', '-<GM>' ]
	);


$warn_bad_punches = 0;

$errors = 0;
$warnings = 0;

$lines = 0;

while (<>) {
    chomp;
    $line = $_;
    if ($line =~ /^\$/) {
	&process_meta_line ($_);
    } elsif (defined ($cabinet) && defined ($tray) && defined ($drawer)
                                && defined ($deck) && defined ($date)) {
	&process_card_image ($_);
    } else {
        die ('missing header information\n');
    }
}
exit 0;

sub process_meta_line {
    my ($line) = @_;

    if ($line =~ /^\$FILENAME/) {
	if ($line =~ /^\$FILENAME "c([0-9]+)t([0-9]+)d([0-9]+)"$/) {
	    $cabinet = $1;
            $tray = $2;
            $drawer = $3;
	    $deck = 0;
	    $drawer_id = sprintf "c%dt%02dd%d", $cabinet, $tray, $drawer;
	    # print "cabinet $cabinet, tray $tray, drawer $drawer\n";
	} else {
	    die "bad FILENAME directive\n";
	}
    } elsif ($line =~ /^\$DATE/) {
	$date = substr ($line, 6);
	# print "DATE = $date\n";
    } elsif ($line =~ /^\$COLUMN_BINARY/) {
	# don't need to do anything
    } elsif ($line =~ /^\$NUMBER_OF_CARDS/) {
	$noc = substr ($line, 17);
	if ($noc != $lines) {
	    printf "%s: expected %d lines, found %d\n", $drawer_id, $noc, $lines;
	    $warnings++;
	}
    } elsif ($line =~ /^\$END_OF FILE/) {
	if (substr ($line, 13) =~ /"c([0-9]+)t([0-9]+)d([0-9]+)"/) {
	    if (($1 ne $cabinet) || ($2 ne $tray) || ($3 ne $drawer)) {
		print "$drawer_id: END_OF FILE directive doesn\'t match FILENAME directive\n";
		$warnings++;
	    }
        } else {
	    printf "$drawer_id: bad END_OF FILE directive\n";
	    $warnings++;
	}
	undef $cabinet, $tray, $drawer, $date;
    } else {
	die "$drawer_id: unrecognized meta card '%s'\n", $line;
    }
}

sub process_card_image {
    my ($line) = @_;
    my $i;
    my $asc = "";

    $lines++;
    if (length ($line) != 160) {
	printf "$drawer_id: incorrect line length on line %d\n", $lines;
	return;
    }
    for ($i = 0; $i < 160; $i += 2) {
	$asc = $asc . &convert (substr ($line, $i, 2));
    }
    if (substr ($asc, 0, 76) eq ' -00123456789JKLMNOPQR<RM>@                               IBM 1620 SEPARATOR') {
	&close_deck;
	# print "------SEPARATOR RECORD------\n";
	$deck++;
	&open_deck;
    } elsif (substr ($asc, 0, 73) eq ' .)+$*-/,(=@ABCDEFGHI-0JKLMNOPQRSTUVWXYZ00123456789<RM>   IBM 1620 BEGIN ') {
        #  followed by 'C1 T01 D1  '
	if ($deck != 0) {
	    die "$drawer_id: BEGIN record inside deck!\n";
	}
        # print "------BEGIN RECORD------\n";
	$deck = 1;
	&open_deck;
    } elsif (substr ($asc, 0, 73) eq ' .)+$*-/,(=@ABCDEFGHI-0JKLMNOPQRSTUVWXYZ00123456789<RM>   IBM 1620 END   ') {
        #  followed by 'C1 T01 D1  '
	printf "$drawer_id:  %d decks\n", $deck;
	$deck = 0;
	&close_deck;
        # print "------END RECORD------\n";
    } else {
	if ($deck == 0) {
	    die "$drawer_id: data card outside deck!\n";
	} else {
	    print FILE "$line\n";
	    $cards_in_deck++;
	}
    }
}

sub convert {
    my ($in) = @_;
    my $c1 = ord (substr ($in, 0, 1));
    my $c2 = ord (substr ($in, 1, 1));
    my @punches;
    my $punches;

    if (($c1 < 0x40) || ($c1 > 0x7f)) {
	die "$drawer_id: bad character in card data '%s'\n", $in;
    }
    $c1 -= 0x40;
    $c2 -= 0x40;
    if ($c1 & 0x20) { push @punches, "12"; }
    if ($c1 & 0x10) { push @punches, "11"; }
    if ($c1 & 0x08) { push @punches, "0"; }
    if ($c1 & 0x04) { push @punches, "1"; }
    if ($c1 & 0x02) { push @punches, "2"; }
    if ($c1 & 0x01) { push @punches, "3"; }
    if ($c2 & 0x20) { push @punches, "4"; }
    if ($c2 & 0x10) { push @punches, "5"; }
    if ($c2 & 0x08) { push @punches, "6"; }
    if ($c2 & 0x04) { push @punches, "7"; }
    if ($c2 & 0x02) { push @punches, "8"; }
    if ($c2 & 0x01) { push @punches, "9"; }
    $punches = join "-", @punches;
    if (! exists $asc {$punches}) {
	if ($warn_bad_punches) {
	    printf "c%dt%02dd%dk%03d line %d:  unrecognized punch '%s'\n", $cabinet,
	         $tray, $drawer, $deck, $line, $punches;
	    $warnings++;
	}
	return ('?');
    }
    return ($asc {$punches}[0]);
}


sub open_deck {
    $filename = sprintf "c%dt%02dd%dk%03d", $cabinet, $tray, $drawer, $deck;
    open FILE, ">$filename" or die "Can't open output file \"$filename\"\n";
    printf FILE "\$FILENAME \"$filename\"\n", 
    print FILE "\$DATE $date\n";
    print FILE "\$COLUMN_BINARY\n";
    $cards_in_deck = 0;
}

sub close_deck {
    print FILE "\$NUMBER_OF_CARDS $cards_in_deck\n";
    printf FILE "\$END_OF FILE \"$filename\"\n";
    close FILE;
}

