/* sktlib.c */

/*
 *  Includes
 */
#include <stdio.h>
#include <string.h>

#include "debug.h"
#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "tcpdat.h"
#include "sesutl.h"
#include "sktlib.h"
#include "mapskt.h"
#include "bytprc.h"
#include "cticks.h"
#include "suspnd.h"


/*
 * Unused Slot
 */
/* SKTL02.C */

/*
 *  skread
 *
 *  Read data from the queue buffer (specified by sknum)
 *  into the user buffer for up to n bytes.
 *
 *  Returns number of bytes read or <0 if error.
 */
/* SKTL04.C */
int skread(sknum,buffer,n)
int sknum,n;
char *buffer;
{
	int i;
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	/*
	 * read from queue buffer
	 */
	i = dequeue(&skt->in,buffer,n);

	/*
	 * enter event if more data
	 */
	if(i) {
		if(quedata(&skt->in))
			skptuev(CONCLASS,CONDATA,sknum);

		return(i);
	}

	return(sktest(sknum));
}

/*
 *  skgtevent
 *
 *  Retrieves the next event (and clears it) which matches bits in
 *  the given mask.  Returns the event number or 0 on no event present.
 *  Also returns the exact class and the associated integer in reference
 *  parameters.
 *
 *  The way the queue works:
 *	There is always a dummy record pointed to by nnelast.
 *	When data is put into the queue, it goes into nnelast,
 *	then nnelast looks around for another empty one to obtain.
 *	It looks at nnefree first, then bumps one from nnefirst if necessary.
 *	When data is retrieved, it is searched from nnefirst to nnelast.
 *	Any freed record is appended to nnefree.
 */
/* SKTL06.C */
int skgtevent(mask,retclass,retdat)
register int mask;
int *retclass,*retdat;
{
	register int i,j;

	i = nnefirst;
	j = 0;
	while(i!=nnelast) {
		if(mask&nnq[i].eclass) {
			if(i==nnefirst) {
				/*
				 * step nnefirst
				 */
				nnefirst = nnq[nnefirst].next;
			} else {
				/*
				 * bypass record i
				 */
				nnq[j].next = nnq[i].next;
			}
			nnq[i].next = nnefree;
			/*
			 * install in free list
			 */
			nnefree = i;
#ifdef	DEBUGOPTION
if(debug&0x01) {
	printf("event: class=0x%x, event=%d, data=%d, idx=%d\r\n",
		nnq[i].eclass, nnq[i].event, nnq[i].idata, i);
}
#endif
			*retdat = nnq[i].idata;
			*retclass = nnq[i].eclass;
			return(nnq[i].event);
		}
		j = i;
		i = nnq[i].next;
	}
	return(0);
}

/*
 *  skabort
 *
 *  Abort the socket connection for sknum.
 */
/* SKTL08.C */
VOID skabort(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * Verify the socket parameter
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return;

	skptevent(SCKTABORT,-1,sknum);
}

/*
 *  skposterr
 *  place an error into the event q
 *  Takes the error number and puts it into the error structure
 */
/* SKTL10.C */
VOID skposterr(num)
register int num;
{
	if(skptevent(ERRCLASS,num,-1))
		/*
		 * only if we lost an event
		 */
		skptuev(ERRCLASS,501,-1);
}

/*
 *  skptuev
 *
 *  put a unique event into the queue
 *  First searches the queue for like events
 */
/* SKTL12.C */
int skptuev(class,event,dat)
int class,event,dat;
{
	register int i;

	i = nnefirst;
	while(i!=nnelast) {
		if(nnq[i].idata==dat &&
			nnq[i].event==event &&
			nnq[i].eclass==class)
			return(0);
		i = nnq[i].next;
	}
	return(skptevent(class,event,dat));
}

/*
 *  skptevent
 *
 *  add an event to the queue.
 *  Will probably get the memory for the entry from the free list.
 *  Returns 0 if there was room, 1 if an event was lost.
 */
/* SKTL14.C */
int skptevent(class,event,dat)
int class,event,dat;
{
	register int i;

	i = nnelast;
	/*
	 * put data in
	 */
	nnq[i].eclass = class;
	nnq[i].event = event;
	nnq[i].idata = dat;
	/*
	 * there is a spot in free list
	 */
	if(nnefree>=0) {
		nnq[i].next = nnelast = nnefree;
		/*
		 * remove from free list
		 */
		nnefree = nnq[nnefree].next;
		return(0);
	} else {
		nnq[i].next = nnelast = nnefirst;
		/*
		 * lose oldest event
		 */
		nnefirst = nnq[nnefirst].next;
		return(1);
	}
}

/*
 *  skgtip
 *
 *  Sets *st* to the current ip number 
 */
/* SKTL16.C */
VOID skgtip(st)
register char *st;
{
	movebytes(st,nnipnum,4);
}

/*
 *  skgtmask
 *
 *  Get the network mask.
 */
/* SKTL18.C */
VOID skgtmask(st)
register char *st;
{
	movebytes(st,nnmask,4);
}

/*
 *  skempty
 *
 *  Discard data in the queue buffer (specified by sknum)
 */
/* SKTL20.C */
int skempty(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-1);

	skt->in.rdptr = skt->in.wtptr;

	skdeque(sknum);
	return(0);
}

/*
 *  skdeque
 *
 *  Inform TCPIP of any dequeued data.  <0 on error.
 */
/* SKTL22.C */
int skdeque(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-1);

	/*
	 * Dequeue any data used
	 */
	if(skt->in.rdptr == skt->in.wtptr ||
	  (skt->in.size <= HIWATER && quespace(&skt->in) >= HIWATER)) {
		skt->out.lasttime = 0L;
	}
	return(0);
}

/*
 *  skwrite
 *
 *  Write data into the output queue (specified by sknum).
 *  TCPIP will distribute the data onto the wire later.
 *
 *  Returns number of bytes sent or <0 if an error.
 */
/* SKTL24.C */
int skwrite(sknum,buffer,n)
int sknum,n;
char *buffer;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	if(skt->state!=SEST)
		return(-1);

	pushsk++;
	return(enqueue(&skt->out,buffer,n));
}

/*
 *  skwchar
 *
 *  Write data into the output queue (specified by sknum).
 *  TCPIP will distribute the data onto the wire later.
 *
 *  Returns number of bytes sent or <0 if an error.
 */
/* SKTL26.C */
int skwchar(sknum,c)
int sknum,c;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	if(skt->state!=SEST)
		return(-1);

	pushsk++;
	return(enqueue(&skt->out,(char *) &c,1));
}

/*
 *  sksendwait
 *
 *  Invalid socket		- return(-2)
 *  Connection dropped		- return(-1)
 *  Data sent			- return( 0)
 *  Timeout			- return( 1)
 */
/* SKTL28.C */
int sksendwait(sknum,wt)
int sknum;
long wt;
{
	register struct socket *skt;
	long time;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	/*
	 * Wait until connection dropped, data sent, or timeout
	 */
	time = cticks(NULL);
	while(1) {
		ntsleep(0);
		if(!skenque(sknum,1))
			return(0);
		if(sktest(sknum))
			return(-1);
		if(elapsed(time) >= wt)
			return(1);
		suspnd(0);
	}
}

/*
 *  skenque
 *
 *  Conditionally sets the push bit on the specified socket data
 *  and informs TCPIP of the enqueued data.
 *  Returns bytes remaining in queue.  <0 if an error.
 */
/* SKTL30.C */
int skenque(sknum,pflag)
int sknum;
int pflag;
{
	register struct socket *skt;

	pushsk = 0;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	skt->out.push = pflag;

	/*
	 * Trigger TCPIP into action
	 */
	if(outqpsh(skt)) {
		skt->out.lasttime = 0L;
	}

	return(quedata(&skt->out));
}

/*
 *  sktest
 *
 *  Checks to see if a particular session has been established yet.
 *  Returns 0 if the connection is in "established" state.
 */
/* SKTL32.C */
int sktest(sknum)
int sknum;
{
	long time;
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	if(skt->state==SEST) {
		return(0);
	} else {
		if(skt->state==SCWAIT) {
			if(quedata(&skt->in)) {
				return(0);
			}
			ntclose(sknum);
			time = cticks(NULL);
			while(skt->state==SCWAIT &&
			      elapsed(time) < 300L) {
				ntsleep(0);
				suspnd(0);
			}
		}
	}
	return(-1);
}

/*
 *  skrelease
 *
 *  Finish the closing process on socket sknum.
 *	1 - issue a close if in state STWAIT/STLAST
 *	2 - return(-1) if in state SCLOSED
 *	3 - loop until a timeout if flag is set,
 *	    then issue a close and release the socket
 *
 *	return -1 if the socket is closed or released
 *	else return sknum
 */
/* SKTL34.C */
int skrelease(sknum,tflag)
int sknum,tflag;
{
	register struct socket *skt;
	long time;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-1);

	time = cticks(NULL);
	do {
		ntsleep(0);
		switch(skt->state) {
		default:		/* check for timeout */
			if(elapsed(time) < (long) RELEASETIME) {
				break;
			}

		case SLAST:		/* close the socket */
		case STWAIT:
			skclose(sknum);

 		case SCLOSED:		/* socket is released */
			return(-1);
			break;

		}
		if(tflag)
			suspnd(15);
	} while(tflag);
	return(sknum);
}

/*
 *  skstate
 *
 *  Returns the state of the connection.  <0 if an error.
 */
/* SKTL36.C */
int skstate(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	/*
	 * socket state
	 */
	return(skt->state);
}

/*
 *  skclose
 *
 *  Start the closing process on socket sknum.
 */
/* SKTL38.C */
VOID skclose(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * Verify the socket parameter
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return;

	ntclose(sknum);
}

/*
 *  skrurgent
 *
 *  Returns pointer to urgent input data
 */
/* SKTL40.C */
char *skrurgent(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(NULL);

	return(skt->in.urgent);
}

/*
 *  skwurgent
 *
 *  Loads the output urgent data flag
 */
/* SKTL42.C */
int skwurgent(sknum,i)
int sknum,i;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(0);

	skt->out.urgent = i;
	return(1);
}

/*
 *  skgtftp
 *
 *  This routine provides the information needed to open an ftp connection
 *  back to the originator of the command connection.  The other side's IP
 *  number and the port numbers are returned in an INTEGER array ( convenient
 *  for use in PORT commands ).
 */
/* SKTL44.C */
int skgtftp(sknum,a)
int sknum,a[];
{
	register struct socket *skt;
	register unsigned int i;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	a[0] = skt->tcpout.i.ipdest[0] & 0xFF;
	a[1] = skt->tcpout.i.ipdest[1] & 0xFF;
	a[2] = skt->tcpout.i.ipdest[2] & 0xFF;
	a[3] = skt->tcpout.i.ipdest[3] & 0xFF;
	i = intswap(skt->tcpout.t.source);
	a[4] = i>>8 & 0xFF;
	a[5] = i    & 0xFF;
	i = intswap(skt->tcpout.t.dest);
	a[6] = i>>8 & 0xFF;
	a[7] = i    & 0xFF;

	return(0);
}

/*
 *  Force the from port on the connection
 */
/* SKTL46.C */
int skfromport(sknum,fromport)
int sknum,fromport;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	/*
	 * allow the from port to be forced
	 */
	skt->skfport = fromport;

	return(0);
}

/*
 *  skqlen
 *
 *  Returns the number of bytes waiting to be read
 *  from the incoming queue for socket sknum.
 *  <0 if error.
 */
/* SKTL48.C */
int skqlen(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	return(quedata(&skt->in));
}

/*
 *  skroom
 *
 *  Returns the space available in
 *  the outgoing queue for socket sknum.
 *  <0 if error.
 */
/* SKTL50.C */
int skroom(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	return(quespace(&skt->out));
}

/*
 *  skleft
 *
 *  Returns the number of bytes left in
 *  the outgoing queue for socket sknum.
 *  <0 if error.
 */
/* SKTL52.C */
int skleft(sknum)
int sknum;
{
	register struct socket *skt;

	/*
	 * check validity
	 */
	skt = (struct socket *) skvalid(sknum);
	if(skt==-1)
		return(-2);

	return(quedata(&skt->out));
}

/*
 *  skvalid
 *
 *  Verify that the socket number is valid
 *  and map the port for access
 */
/* SKTL90.C */
struct socket *skvalid(sknum)
register int sknum;
{
	if(sknum<0 || sknum>=LSCKTS || !sktlist[sknum].region)
		return(-1);

	return((struct socket *) mapskt(sknum));
}

/*
 *  skerrstring
 *
 *  returns the string associated with a particular error number
 *
 *  error number is formatted %4d at the beginning of the string
 */
/* SKTL95.C */
static char *errs[] = {
	"   0 (Error Code)",
	/* addsess() messages */
	"  10 Require machine name/address",
	"  11 Querying the Domain Name Server",
	"  12 No name server, cannot resolve IP address",
	"  13 Internal Sdomain() allocation error",
	"  14 Trying to open TCP connection",
	"  15 Could not open the connection",
	"  16 Domain lookup failed",
	"  17 Domain lookup successful",
#ifdef	ERRORMESSAGES
	/* ??? */
	" 100 Network jammed, probable break in wire",
	" 101 Could not initialize hardware level network driver",
	" 102 The conflicting machine is using the same IP number",
	" 103 Request failed, an IP number is required",
	" 104 Event queue filled, probably non-fatal",
	/* IP: */
	" 300 Packet with options or wrong version received",
	" 301 Bad checksum",
	" 302 Fragmented packet reassembly error",
	" 303 Unknown higher layer protocol",
	/* TCP: */
	" 400 Bad checksum",
	" 401 Packet received for invalid port -- reset sent",
	" 402 ACK invalid, syn sent",
	" 403 Unknown state",
	" 404 Connection reset by other host",
	" 405 Null port specified for ack/trans",
	/* TCP: */
	" 500 Session limit reached",
	" 501 Socket limit reached",
	" 502 Job failed to start",
	" 503 Socket allocation error",
	" 504 Local host or gateway not responding",
	" 505 Not allowed to connect to broadcast address",
	" 506 Host is refusing connection",
	/* ICMP: */
	" 600 Echo reply",
	" 603 Destination unreachable",
	" 604 Source Quench",
	" 605 Redirect, another gateway is more efficient",
	" 608 Echo requested (ping requested)",
	" 611 Time Exceeded on Packet",
	" 612 Parameter problem in IP",
	" 613 Timestamp request",
	" 614 Timestamp reply",
	" 615 Information request",
	" 616 Information reply",
	" 699 Checksum error",
	/* UDP: */
	" 700 Bad checksum",
	/* Domain: */
	" 800 Name request to server failed",
	" 801 Using default domain",
	" 802 Name does not exist",
	" 803 UDP name server did not resolve the name",
	" 804 Name server failed, unknown reason",
	" 805 Host machine not in configuration file",
	" 806 Missing IP number, requires domain lookup",
	/* Session: */
	" 900 Cannot find or open configuration file",
	" 901 Cannot allocate memory for processing",
	" 902 Invalid keyword in configuration file",
	" 903 Element too long (>200), maybe missing quote",
	" 904 Probable missing quote marks, place field on one line",
	" 905 'name' field required before other machine entries",
	" 906 Invalid IP number",
	" 907 Subnet mask invalid",
	" 908 IP address for this machine is invalid",
	" 909 Invalid services request",
#endif
	""
};

static char errspace[80];	/* room for user-defined errors */

char *skerrstring(errno)
register int errno;
{
	register int i;
	char s[10];

	if(errno<0)
		return(errspace);

	sprintf(s,"%4d",errno);
	i = 1;
	do {
		if(!strncmp(errs[i],s,4))
			/*
			 * pointer to error message
			 */
			return(errs[i]+5);
		i++;
	/*
	 * until NULL found
	 */
	} while(*errs[i]);
	/*
	 * error unknown
	 */
	sprintf(errs[0],"%4d",errno);
	errs[0][4] = ' ';
	return(errs[0]);
}

                                                                                                                                                                              