/* tcp.c */

#define	SCKTMASTER
#define TCP_UDP

/*
 * DROP Packet (1) or RESET Packet (0)
 * on unavailable socket or job count exceeded.
 */

#define	DROP_RESET	1

/*
 *  Includes
 */
#include <stdio.h>

#include "vtcpip.h"
#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "prodat.h"
#include "tcpdat.h"
#include "mapskt.h"
#include "bytprc.h"
#include "hdrchk.h"
#include "detach.h"
#include "cticks.h"
#include "suspnd.h"


/*
 * tcpinterpret
 *
 * Called when a packet comes in and passes the IP checksum and is of
 * TCP protocol type.  Check to see if we have an open connection on
 * the appropriate socket and stuff it in the right buffer
 */
int tcpinterpret(pkt,tlen)
int tlen;
register TCPKT *pkt;
{
	int i,j,ret,sknum,myport,hlen,hisport;
	register struct socket *skt;
	register struct service *sp;

#ifdef	DEBUGOPTION
if(debug&0x104) {
	dmpfil("tcpinp",(DLAYER *)pkt,sizeof(DLAYER)+sizeof(IPLAYER)+tlen);
}
#endif
	/*
	 * First, fill the pseudo header with its fields,
	 * then run our checksum to confirm it.
	 */
	if(pkt->t.check) {
		/*
		 * move both addresses
		 */
		movebytes(tcps.source,pkt->i.ipsource,8);
		tcps.z = 0;
		tcps.proto = pkt->i.protocol;
		tcps.tcplen = intswap(tlen);
		/*
		 * compute checksum
		 */
		if(tcpcheck((char *)&tcps,(char *)&pkt->t,tlen)) {
			ntposterr(400);
			return(2);
	  	}
	}
	/*
	 * bytes offset to data
	 */
	hlen = pkt->t.hlen >> 2;
	/*
	 *  find the port which is associated with the incoming packet
	 *  First try open connections, listeners, and then services
	 */
	myport = intswap(pkt->t.dest);
	hisport = intswap(pkt->t.source);
#ifdef	DEBUGOPTION
if(debug&0x02) {
	printf("\r\nmyport=%u, hisport=%u\r\n", myport,hisport);
}
#endif
	/*
	 * Try open connections
	 */
	for (sknum=0; sknum<GSCKTS && sktlist[sknum].region; sknum++) {
		if(sktlist[sknum].inport == myport &&
		   sktlist[sknum].outport == hisport &&
		   comparen(&sktlist[sknum].remoteip,&pkt->i.ipsource,4)) {
#ifdef	DEBUGOPTION
if(debug&0x02) {
			printf("open:      inport=%u, outport=%u\r\n",
				sktlist[sknum].inport,sktlist[sknum].outport);
}
#endif
			if((ret = tcpdo(sknum,pkt,tlen,hlen)) != -1) {
				return(ret);
			}
		}
	}
	/*
	 * Try listeners
	 */
	for (sknum=0; sknum<GSCKTS && sktlist[sknum].region; sknum++) {
		if(	sktlist[sknum].inport == myport &&
		       !sktlist[sknum].outport &&
			pkt->t.flags & TSYN) {
#ifdef	DEBUGOPTION
if(debug&0x02) {
			printf("listener:  inport=%u, outport=%u\r\n",
				sktlist[sknum].inport,sktlist[sknum].outport);
}
#endif
			return(tcpdo(sknum,pkt,tlen,hlen));
		}
	}
	/*
	 * Try services
	 */
	if(pkt->t.flags & TSYN) {
	    sp = Scon.srvc;
	    while(sp != NULL) {
		if(sp->port == myport) {
			/*
			 * Check session count
			 */
			for (i=0,j=0; i<GSCKTS && sktlist[i].region; i++) {
				if(sktlist[i].jobnum &&
					sktlist[i].inport == myport) {
					skt = (struct socket *) mapskt(i);
					if (skt->state >= SLISTEN)
						j++;
				}
			}
			if(j >= sp->maxses) {
#if	DROP_RESET
				/*
				 * Most systems when reaching the session
				 * limit ignore the incoming request.
				 * This allows the requester to timeout
				 * and retry the connection.
				 */
				ntposterr(500);
				return(1);
#else
				/*
				 * Alternatively, Reject the Packet
				 */
				break;
#endif
			}
			/*
			 * Create a socket
			 */
			sknum = makesocket(0);
			if(sknum < 0) {
#if	DROP_RESET
				/*
				 * Most systems when reaching the socket
				 * limit ignore the incoming request.
				 * This allows the requester to timeout
				 * and retry the connection.
				 */
				ntposterr(501);
				return(1);
#else
				/*
				 * Alternatively, Reject the Packet
				 */
				break;
#endif
			}
			/*
			 * Try to start up service
			 */
			j = strtdj(sp->cmdfil);
			if(j) {
			    /*
			     * Abort sockets with duplicate job numbers
			     */
			    for(i=0; i<GSCKTS && sktlist[i].region; i++) {
				if(abs(sktlist[i].jobnum) == j) {
				    Stmrunset(SRVCABORT,sktlist[i].inport,-j);
				    Stmrset(SCKTABORT,
					sktlist[i].jobnum,
					sktlist[i].sktid,
					LASTTIME/TICKSPERSEC);
				}
			    }
			    skt = (struct socket *) mapskt(sknum);
			    skt->jobnum = -j;
			    sktlist[sknum].jobnum = -j;
			} else {
#if	DROP_RESET
			    /*
			     * Most systems when failing to activate
			     * a task ignore the incoming request.
			     * This allows the requester to timeout
			     * and retry the connection.
			     */
			    ntposterr(502);
			    return(1);
#else
			    /*
			     * Alternatively, Reject the Packet
			     */
			    break;
#endif
			}
			/*
			 * Set socket for listening
			 */
			skt->sktport = myport;
			ntlisten(sknum);
			/*
			 * Set an abort timer event incase of failure
			 */
			Stmrset(SRVCABORT,myport,-j,ATTACHTIMEOUT);
			/*
			 * Now allow connection to be made
			 */
#ifdef	DEBUGOPTION
if(debug&0x02) {
			printf("services:  inport=%u, outport=%u\r\n",
				sktlist[sknum].inport,sktlist[sknum].outport);
}
#endif
			return(tcpdo(sknum,pkt,tlen,hlen));
		}
		sp = sp->next;
	    }
	}
	/*
	 *  no matching port was found to handle this packet, reject it
	 */
	tcpreset(pkt);
	/*
	 * no error message if it is a SYN
	 */
	if(!(pkt->t.flags & TSYN))
		/*
		 * invalid port for incoming packet
		 */
		ntposterr(401);
	return(1);
}

/*
 * tcpdo
 *
 * Deliver the incoming packet.
 *
 * Returns:
 *	 1	on error
 *	 0	on OK
 *	-1	on backout of Invalid STWAIT Sequence
 */
static int tcpdo(sknum,pkt,tlen,hlen)
int sknum,tlen,hlen;
register TCPKT *pkt;
{
	register struct socket *skt;
	register int ret;

	ret = 0;
	skt = (struct socket *) mapskt(sknum);

#ifdef	DEBUGOPTION
if(debug&0x02) {
	printf("sknum=%d, skt->state(in)=%d, ", sknum, skt->state);
}
#endif

	switch(skt->state) {
		case SLISTEN:	/* waiting for remote connection */
			/*
			 * TSYN Required
			 */
			if(!(pkt->t.flags & TSYN))
				break;

			/*
			 * receive SYN
			 * remember anything important
			 * from the incoming TCP header
			 */
			skt->out.size = intswap(pkt->t.window);
			sktlist[sknum].outport =
				skt->out.port =
				intswap(pkt->t.source);
			skt->in.nxt = longswap(pkt->t.seq)+1;
			/*
			 * see if MSS option is there
			 */
			checkmss(skt,pkt,hlen);
			/*
			 * set the necessary fields
			 * in the outgoing TCP packet
			 */
			skt->tcpout.t.dest = pkt->t.source;
			skt->tcpout.t.ack = longswap(skt->in.nxt);
			skt->tcpout.t.flags = TSYN|TACK;
			/*
			 * initialize all of the low-level
			 * transmission stuff(IP and lower)
			 */
			movebytes(sktlist[sknum].remoteip,
					pkt->i.ipsource,4);
			movebytes(skt->tcps.dest,
					pkt->i.ipsource,4);
			movebytes(skt->tcpout.i.ipdest,
					pkt->i.ipsource,4);
			movebytes(skt->tcpout.d.dest,
					pkt->d.me,DADDLEN);
			tcpsend(skt,4);
			/*
			 * syn received
			 */
			skt->state = SSYNR;

			/*
			 * If TACK then fall through to SSYNR
			 */
			if(!(pkt->t.flags & TACK))
				break;

		case SSYNR:
			if(!(pkt->t.flags & TACK)) {
				/*
				 * not the right one
				 */
				tcpreset(pkt);
				ret = 1;
				break;
/*
 * this method sometimes	tcpsend(skt,4);
 * caused net thrashing		break;
 */
			}
			/*
			 * response not needed
			 */
			skt->tcpout.t.hlen = 20<<2;
			skt->out.lasttime = cticks(NULL);
			/*
			 * count SYN as sent
			 */
			skt->out.nxt++;
			/*
			 * starting ACK value
			 */
			skt->out.ack = longswap(pkt->t.ack);
			/*
			 * allowed window
			 */
			skt->out.size = intswap(pkt->t.window);
			/*
			 * starting ACK flag
			 */
			skt->tcpout.t.flags = TACK;
			/*
			 * see if MSS option is there
			 */
			checkmss(skt,pkt,hlen);
			/*
			 * drop through to established
			 */
			skt->state = SEST;
			ntptevent(CONCLASS,CONOPEN,sknum);

		case SEST:	/* normal data transmission */	
			/*
			 * check and accept a possible piggybacked ack
			 * if this is a duplicate ACK then assume
			 * subsequent data not received and resend 
			 */
			if((ackcheck(sknum,pkt) == 2) &&
			   (skt->out.nxt != skt->out.ack)) {
				skt->out.nxt = skt->out.ack;
				skt->out.lasttime = 0L;
			}
			estabcon(sknum,pkt,tlen,hlen);
			break;

		case SSYNS:	/* check to see if it ACKS correctly */
				/* remember that tcpout is pre-set-up */
			if(pkt->t.flags & TACK) {
				if(longswap(pkt->t.ack) != skt->out.nxt) {
					ntposterr(402);
					ret = 1;
					break;
				}
			}
			if(pkt->t.flags & TRESET) {
				ntposterr(506);
				skt->state = SCLOSED;
				ntptuev(CONCLASS,CONCLOSE,sknum);
				ret = 1;
				break;
			}
			if(pkt->t.flags & TSYN) {
				skt->in.nxt = longswap(pkt->t.seq) + 1;
				skt->tcpout.t.ack = longswap(skt->in.nxt);
				skt->out.ack = longswap(pkt->t.ack);
				skt->out.size = intswap(pkt->t.window);
				skt->out.lasttime = 0L;
				if(pkt->t.flags & TACK) {
					/*
					 * need to send ACK
					 */
					skt->tcpout.t.flags = TACK;
					skt->state = SEST;
					checkmss(skt,pkt,hlen);
					ntptevent(CONCLASS,CONOPEN,sknum);
				} else {
					/*
					 * need to send SYN & ACK
					 */
					skt->tcpout.t.flags = TSYN|TACK;
					skt->state = SSYNR;
				}
			}
			break;

		case SCWAIT:
			ackcheck(sknum,pkt);
			if(skt->in.rdptr==skt->in.wtptr) {
				skt->tcpout.t.flags = TFIN;
				skt->state = SLAST;
				skt->out.lasttime = 0L;
				/*
				 * connection closing
				 */
				ntptuev(CONCLASS,CONCLOSE,sknum);
			}
			break;

		case SLAST:	/* check ack of FIN, */
				/* or reset to see if we are done */
#ifdef	AGGRESSIVE
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			} else {
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
				skt->state = SCLOSED;
			}
#else
			if((pkt->t.flags & TRESET) ||
				(longswap(pkt->t.ack) == skt->out.nxt+1)) {
				skt->state = SCLOSED;
			}
#endif
			break;

		case SFW1:	/* waiting for ACK of FIN */
				/* throw away data */
			skt->in.nxt = longswap(pkt->t.seq)+tlen-hlen;
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			} else
			if(longswap(pkt->t.ack) != skt->out.nxt+1) {
				if(pkt->t.flags & TFIN) {
					/*
					 * got FIN,no ACK for mine
					 * account for FIN byte
					 */
					skt->in.nxt++;
					skt->tcpout.t.ack =
						longswap(skt->in.nxt);
					/*
					 * cause last ACK to be sent
					 */
					skt->tcpout.t.flags = TACK;
					skt->out.lasttime = 0L;
					skt->state = SCLOSING;
				}
			} else
			if(pkt->t.flags & TFIN) {
				/*
				 * ACK and FIN
				 *
				 * account for his FIN flag
				 */
				skt->in.nxt++;
				skt->tcpout.t.ack =
					longswap(skt->in.nxt);
				/*
				 * account for my FIN
				 */
				skt->out.nxt++;
				/*
				 * cause last ACK to be sent
				 */
				skt->tcpout.t.flags = TACK;
				tcpsend(skt,0);
				/*
				 * At this point the connection is closed.
				 * To make the socket immediateley
				 * available the STWAIT state
				 * is bypassed.  (This is 'NOT' the
				 * official TCP/IP proceedure, but is
				 * used to allow reuse of the limited
				 * number of sockets.)
				 */
				/*	skt->out.lasttime = 0L;	*/
				/*	skt->state = STWAIT;	*/
#ifdef	AGGRESSIVE
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
#endif
				skt->state = SCLOSED;
			} else {
				/*
				 * got ACK, no FIN
				 * account for my FIN byte
				 */
				skt->out.nxt++;
				skt->state = SFW2;
#ifdef	AGGRESSIVE
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
				skt->state = SCLOSED;
#endif
			}
			break;

		case SFW2:	/* want FIN */
			skt->in.nxt = longswap(pkt->t.seq)+tlen-hlen;
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			} else
			if(pkt->t.flags & TFIN) {
				/*
				 * we got FIN
			 	 * count his FIN byte
				 */
				skt->in.nxt++;
				skt->tcpout.t.ack =
					longswap(skt->in.nxt);
				/*
				 * cause last ACK to be sent
				 */
				skt->tcpout.t.flags = TACK;
				tcpsend(skt,0);
				/*
				 * At this point the connection is closed.
				 * To make the socket immediateley
				 * available the STWAIT state
				 * is bypassed.  (This is 'NOT' the
				 * official TCP/IP proceedure, but is
				 * used to allow reuse of the limited
				 * number of sockets.)
				 */
				/*	skt->out.lasttime = 0L;	*/
				/*	skt->state = STWAIT;	*/
#ifdef	AGGRESSIVE
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
#endif
				skt->state = SCLOSED;
			}
			break;

		case SCLOSING:	/* want ACK of FIN */
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			} else
			if(!ackcheck(sknum,pkt)) {
				/*
				 * account for my FIN byte
				 */
				skt->out.nxt++;
				/*
				 * At this point the connection is closed.
				 * To make the socket immediateley
				 * available the STWAIT state
				 * is bypassed.  (This is 'NOT' the
				 * official TCP/IP proceedure, but is
				 * used to allow reuse of the limited
				 * number of sockets.)
				 */
				/*	skt->state = STWAIT;	*/
#ifdef	AGGRESSIVE
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
#endif
				skt->state = SCLOSED;
			}
			break;

		case STWAIT:	/* ack FIN again? */
			if(pkt->t.flags & TSYN) {
				/*
				 * This is an Invalid sequence.
				 * Most likely this means the opening
				 * of a connection with the same port
				 * number.  Close the socket and
				 * backout to open a new connection.
				 */
				skt->state = SCLOSED;
				ret = -1;
				break;
			}
#ifdef	AGGRESSIVE
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			} else {
				skt->tcpout.t.flags = TRESET;
				tcpsend(skt,0);
				skt->state = SCLOSED;
			}
#else
			if(pkt->t.flags & TRESET) {
				skt->state = SCLOSED;
			}
			/*
			 * only if he wants it
			 */
			if(pkt->t.flags & TFIN)
				skt->out.lasttime = 0L;

			if(skt->out.lasttime != 0L) {
				if(elapsed(skt->out.lasttime) > WAITTIME) {
					skt->state = SCLOSED;
				}
			}
#endif
			break;			

		case SCLOSED:
			break;

		default:	/* unknown tcp state */
			ntposterr(403);
			break;
	}

	if(skt->state == SCLOSED) {
		sktlist[sknum].inport = skt->in.port = 0;
		sktlist[sknum].outport = skt->out.port = 0;
	}

#ifdef	DEBUGOPTION
if(debug&0x02) {
	printf("         skt->state(out)=%d\r\n", skt->state);
}
#endif

	return(ret);
}

/*
 * checkmss
 *
 * Look at incoming SYN,ACK packet and check for the options field
 * containing a TCP Maximum segment size option.  If it has one,
 * then set the port's internal value to make sure that it never
 * exceeds that segment size.
 */
static void checkmss(skt,pkt,hlen)
register struct socket *skt;
register TCPKT *pkt;
int hlen;
{
	unsigned int i;

	/*
	 *  check header for maximum segment size option
	 */
	if(hlen > 20 &&
		pkt->x.options[0] == 2 &&
		pkt->x.options[1] == 4) {
		/*
		 * swapped value of maxseg
		 */
		movebytes((char *)&i,(char *)&pkt->x.options[2],2);
		i = intswap(i);
		/*
		 * we have our own limits too
		 */
		if(i < skt->sendsize)
			skt->sendsize = i;
	}
}

/*
 * tcpreset
 *
 *  Send a reset packet back to sender
 *  Use the packet which just came in as a template to return to
 *  sender.  Fill in all of the fields necessary and dlsend it back.
 */
static int tcpreset(pkt)
register TCPKT *pkt;
{
	register unsigned int tport;
	register int rstlen;
	long seqlen;
	struct pseudotcp xxx;

	/*
	 * don't reset a reset
	 */
	if(pkt->t.flags & TRESET)
		return(1);
	/*
	 *  swap TCP layer portions for sending back
	 */
	if(pkt->t.flags & TACK) {
		/*
		 * ack becomes next seq #
		 */
		pkt->t.seq = pkt->t.ack;
		/*
		 * ack # is 0
		 */
		pkt->t.ack = 0L;
	} else {
		/*
		 * ack becomes the sequence number plus the number of
		 * data bytes (plus one if the SYN bit is set)
		 */
		seqlen = longswap(pkt->t.seq) + pkt->i.tlen - sizeof(IPLAYER);
		if(pkt->t.flags & TSYN) {
			pkt->t.ack = longswap(seqlen + 1);
		} else {
			pkt->t.ack = longswap(seqlen);
		}
		/*
		 * seq # is 0
		 */
		pkt->t.seq = 0L;
	}
	pkt->t.flags = TRESET;
	/*
	 * swap port #'s
	 */
	tport = pkt->t.source;
	pkt->t.source = pkt->t.dest;
	pkt->t.dest = tport;
	pkt->t.hlen = 20<<2;
	pkt->t.window = 0;
	/*
	 *  create pseudo header for checksum
	 */
	xxx.z = 0;
	xxx.proto = pkt->i.protocol;
	xxx.tcplen = intswap(20);
	movebytes(xxx.source,pkt->i.ipsource,4);
	movebytes(xxx.dest,pkt->i.ipdest,4);
	pkt->t.check = 0;	
	pkt->t.check = tcpcheck((char *)&xxx,
				(char *)&pkt->t,sizeof(struct tcph));

	/*
	 *  IP and data link layers
	 */	

	/*
	 * machine it came from
	 */
	movebytes(pkt->i.ipdest,pkt->i.ipsource,4);
	movebytes(pkt->i.ipsource,nnipnum,4);
	pkt->i.tlen = intswap(sizeof(IPLAYER)+sizeof(TCPLAYER));
	pkt->i.identity = nnipident++;
	pkt->i.ttl = TCPTTL;
	pkt->i.check = 0;
	pkt->i.check = ipcheck((char *)&pkt->i,10);
	rstlen = sizeof(DLAYER) + sizeof(IPLAYER) + sizeof(TCPLAYER);

	/*
	 * data link address
	 */
	movebytes(pkt->d.dest,pkt->d.me,DADDLEN);
	/*
	 * my address
	 */
	movebytes(pkt->d.me,dblank.me,DADDLEN);

#ifdef	DEBUGOPTION
if(debug&0x104) {
	dmpfil("tcprst",(DLAYER *)pkt,rstlen);
}
#endif
	return(dlsend((DLAYER *)pkt,rstlen));
}

/*
 *  tcpsend
 *
 *  transmits a TCP packet.
 *
 *   For IP:
 *      sets identity,check,totallen
 *   For TCP:
 *      sets seq and window from port information,
 *	fills in the pseudo header and computes the checksum.
 *      Assumes that all fields not filled in here are filled in by the
 *      calling proc or were filled in by makesocket().
 *      (see all inits in protinit)
 */
int tcpsend(skt,dlen)
register struct socket *skt;
register int dlen;
{
	register int tcplen;

	/*
	 * set current in.size
	 */
	skt->in.size = inroom(skt);

	/*
	 * do IP header first
	 */
	skt->tcpout.i.identity = intswap(nnipident++);
	skt->tcpout.i.tlen =
		intswap(sizeof(struct iph)+sizeof(struct tcph) + dlen);
	/*
	 * install checksum
	 */
	skt->tcpout.i.check = 0;
	skt->tcpout.i.check = ipcheck((char *)&skt->tcpout.i,10);

	/*
	 * do TCP header
	 */
	skt->tcpout.t.seq = longswap(skt->out.nxt);
	/*
	 * if the port has some credit limit, use it instead of large
	 * window buffer.  Generally demanded by hardware limitations.
	 */
	if(skt->credit < skt->in.size)
		skt->tcpout.t.window = intswap(skt->credit);
	else
		skt->tcpout.t.window = intswap(skt->in.size);
	/*
	 * prepare pseudo-header for checksum
	 */
	skt->tcps.tcplen = intswap(dlen+sizeof(TCPLAYER));
	skt->tcpout.t.check = 0;
	skt->tcpout.t.check = tcpcheck((char *)&skt->tcps,
		(char *)&skt->tcpout.t,dlen+sizeof(struct tcph));
	skt->out.lasttime = cticks(NULL);
	tcplen = sizeof(DLAYER) + sizeof(IPLAYER) + sizeof(TCPLAYER) + dlen;

#ifdef	DEBUGOPTION
if(debug&0x104) {
	dmpfil("tcpout",(DLAYER *)&skt->tcpout,tcplen);
}
#endif
	return(dlsend((DLAYER *)&skt->tcpout,tcplen));
}

/*
 *  ackcheck
 *
 *  take an incoming packet and see if there is an ACK for the outgoing
 *  side.  Use that ACK to dequeue outgoing data.
 */
static int ackcheck(sknum,pkt)
int sknum;
register TCPKT *pkt;
{
	register struct socket *skt;
	register int i,j;
	long ak,rttl;

	skt = (struct socket *) mapskt(sknum);

	if((pkt->t.flags & TRESET) && (pkt->t.seq == skt->tcpout.t.ack)) {
		ntposterr(404);
		skt->state = SCLOSED;
		ntptuev(CONCLASS,CONCLOSE,sknum);
		return(1);
	}
	/*
	 * check ACK flag
	 */
	if(!(pkt->t.flags & TACK))
		return(1);
	/*
	 * allowable transmission size
	 */
	skt->out.size = intswap(pkt->t.window);

	/*
	 *  rmqueue any bytes which have been ACKed, update skt->out.ack
	 *  to the new ack number for outgoing.  Update send window.
	 */

	/*
	 * other side's ACK
	 */
	ak = longswap(pkt->t.ack);
	/*
	 * If ak is not increasing (above skt->out.ack) then we should assume
	 * that it is a duplicate packet or one of those stupid keepalive
	 * packets that 4.2 sends out.
	 *
	 * ((a) - (b)) handles sequence space wrap-around.
	 * Overflow/Underflow for the operation makes the result
	 * below correct (-, 0, +) for any a, b in the sequence space.
	 * Results:	result	implies
	 *		  -	 a < b
	 *		  0	 a = b
	 *		  +	 a > b
	 */
	i = ((ak) - (skt->out.ack));
	if(i > 0) {
		/*
		 * take off of queue
		 */
		rmqueue(skt,i);
		skt->out.ack = ak;
	/*
	 * Check to see if this acked our most recent transmission.
	 * If so, adjust the RTO value to reflect the newly measured RTT.
	 * This formula reduces the RTO value so that it gradually
	 * approaches twice the most recent round trip measurement.
	 * This essentially allows a variance equal to the round trip time.
	 * When a packet is retransmitted, this value is doubled
	 * (exponential backoff).
	 */
		if(skt->out.lasttime!=0L) {
			if(skt->out.ack==skt->out.nxt) {
				rttl = elapsed(skt->out.lasttime);
				if(rttl<MAXRTO) {
					/*
					 * smoothing function
					 */
					i = rttl;
					if(i<(MINRTO>>1))
						i = MINRTO>>1;
					i <<= 1;
					j = skt->rto;
					skt->rto = (i + j + j + j)>>2;
				}
			}
			if(outqpsh(skt)) {
				skt->out.lasttime = 0L;
			}
		}
		return(0);
	} else
	/*
	 * ReACKed Sequence
	 */
	if(i == 0) {
		return(2);
	}
	return(1);
}	

/**/
#define	OUTOFORDER
/**/

#ifdef	OUTOFORDER

/*
 *  estabcon
 *
 *  take a packet which has arrived for an established
 *  connection and put it where it belongs.
 *
 *  Support for Out of Order Packet
 *  reassembly has been added.
 */
int estabcon(sknum,pkt,tlen,hlen)
int sknum;
TCPKT *pkt;
int tlen,hlen;
{
	int dlen,room,offset,i,j;
	long psq,nxt;
	register struct socket *skt;
	register SQSPC *seqi,*seqj;
	register int nitems;
	char *ptr,*bufr;

	skt = (struct socket *) mapskt(sknum);
	nitems = skt->seqncs;

	/*
	 *  see if we want this packet, or is it a duplicate?
	 */
	psq = longswap(pkt->t.seq);
	nxt = skt->in.nxt;

	room = inroom(skt);
	dlen = tlen-hlen;

	offset = ((psq) - (nxt));

	if((offset >= 0) && (offset <= room) && ((offset + dlen) <= (room))) {
		/*
		 * compute offset into queue
		 */
		i = skt->in.wtptr - skt->in.where;
		i = QUEUESIZE - i;
		if(i < offset) {
			ptr = skt->in.where + offset - i;
		} else {
			if(i == offset) {
				ptr = skt->in.where;
			} else {
				ptr = skt->in.wtptr + offset;
			}
		}
		/*
		 * move data into queue at offset position
		 */
		bufr = pkt->x.data + hlen - 20;
		i = ptr - skt->in.where;
		i = QUEUESIZE - i;
		if(i < dlen) {
			movebytes(ptr,bufr,i);
			movebytes(skt->in.where,bufr + i,dlen - i);
			ptr = skt->in.where + dlen - i;
		} else {
			movebytes(ptr,bufr,dlen);
			if(i == dlen) {
				ptr = skt->in.where;
			} else {
				ptr += dlen;
			}
		}
		/*
		 * Add entry into sequence array
		 * for newly added data
		 */
		seqi = &skt->seq[nitems];
		seqi->sq	= psq;
		seqi->sqnxt	= psq + dlen;
		seqi->sqflg	= pkt->t.flags;
		seqi->sqptr	= ptr;
		if(nitems < (NITEMS - 1)) {
			nitems++;
		}
		/*
		 * compact the array entries
		 */
	loop:
		for(i=0; i<nitems; i++) {
		    seqi = &skt->seq[i];
		    for(j=i+1; j<nitems; j++) {
			seqj = &skt->seq[j];
			/*
			 * j'th packet preceeds overlaying i'th packet
			 */
			if((seqj->sq <= seqi->sq) &&
			   (seqj->sqnxt >= seqi->sq)) {
				seqi->sq = seqj->sq;
				if(seqj->sqnxt > seqi->sqnxt) {
					seqi->sqnxt = seqj->sqnxt;
					seqi->sqptr = seqj->sqptr;
				}
				seqi->sqflg |= seqj->sqflg;
				nitems = shuffle(seqj,j,nitems);
				goto loop;
			}
			/*
			 * i'th packet preceeds overlaying j'th packet
			 */
			if((seqi->sq <= seqj->sq) &&
			  (seqi->sqnxt >= seqj->sq)) {
				if(seqj->sqnxt > seqi->sqnxt) {
					seqi->sqnxt = seqj->sqnxt;
					seqi->sqptr = seqj->sqptr;
				}
				seqi->sqflg |= seqj->sqflg;
				nitems = shuffle(seqj,j,nitems);
				goto loop;
			}
		    }
		}
		/*
		 * search for next data sequence
		 */
		for(i=0; i<nitems; i++) {
		    seqi = &skt->seq[i];
		    if(seqi->sq == nxt) {
			dlen = seqi->sqnxt - seqi->sq;
			if(dlen > 0) {
				/*
				 * new ACK value
				 */
				skt->in.nxt = seqi->sqnxt;
				skt->tcpout.t.ack = longswap(skt->in.nxt);
				/*
				 * new window size
				 */
				skt->in.size -= dlen;
				skt->in.wtptr = seqi->sqptr;
				/*
				 * urgent flag
				 */
				if(seqi->sqflg & TURG) {
					skt->in.urgent = (int) skt->in.wtptr;
				}
				/*
				 * packet time
				 */
				skt->in.lasttime = cticks(NULL);
				/*
				 * tell user
				 */
				ntptuev(CONCLASS,CONDATA,sknum);
			}
			pkt->t.flags = seqi->sqflg;
			checkfin(sknum,pkt);
			nitems = shuffle(seqi,i,nitems);
			break;
		    }
		}
	} else {
		/*
		 * make the ACK time out
		 */
		skt->out.lasttime = 0L;
	}
	skt->seqncs = nitems;
}

/*
 *  shuffle
 */
int shuffle(seqn,n,nitems)
register SQSPC *seqn;
int n;
register int nitems;
{
	register int i;

	if(nitems <= 0) {
		nitems = 0;
	} else {
		--nitems;
		i = nitems - n;
		if(i > 0) {
			movebytes(seqn,seqn + 1,i * sizeof(SQSPC));
		}
	}
	return(nitems);
}

#else

/*
 *  estabcon
 *
 *  take a packet which has arrived for an established connection
 *  and put it where it belongs.
 */
static int estabcon(sknum,pkt,tlen,hlen)
register int sknum;
register TCPKT *pkt;
int tlen,hlen;
{
	int dlen,i,j;
	long sq,want;
	register struct socket *skt;

	skt = (struct socket *) mapskt(sknum);
	skt->in.size = inroom(skt);

	dlen = tlen-hlen;
	/*
	 *  see if we want this packet, or is it a duplicate?
	 */
	sq = longswap(pkt->t.seq);
	want = skt->in.nxt;
	if(sq != want) {
	 	/*
		 * overlap
		 */
		i = ((sq) - (want));
		j = ((sq + dlen) - (want));
		if(i < 0 && j >= 0) {
			/*
			 * offset desired (i < 0)
			 */
			hlen -= i;
			/*
			 * skip this much (i < 0)
			 */
			dlen += i;
		} else {
			/*
			 * make the ACK time out
			 */
			skt->out.lasttime = 0L;
			return(-1);
		}
	} else {
		if(dlen <= 0) {
			/*
			 * only an ACK packet
			 *
			 * might still have FIN
			 */
			checkfin(sknum,pkt);
			return(0);
		}
	}
	/*
	 *  If we have room in the window, update the ACK field values
	 */
	if(skt->in.size >= dlen) {
		/*
		 * new ack value
		 */
		skt->in.nxt += dlen;
		skt->tcpout.t.ack = longswap(skt->in.nxt);
		/*
		 * new window size
		 */
		skt->in.size -= dlen;
		/*
		 * move data to queue
		 */
		enqueue(skt,pkt->x.data+hlen-20,dlen);
		/*
		 * urgent flag set ?
		 */
		if(pkt->t.flags & TURG)
			skt->in.urgent = (int) skt->in.wtptr;
		/*
		 * packet time
		 */
		skt->in.lasttime = cticks(NULL);
		/*
		 * tell user about it
		 */
		ntptuev(CONCLASS,CONDATA,sknum);
		/*
		 * Check the FIN bit to see if this connection is closing
		 */
		checkfin(sknum,pkt);
	} else {
		/*
		 * no room in input buffer - ignore data
		 */
	}
	return(0);
}

/*
 *  enqueue
 *
 *  Queue up data coming in from the network.
 *  QUEUESIZE is the size limitation of the advertised window.
 */
int enqueue(skt,buffer,nbytes)
register struct socket *skt;
register char *buffer;
int nbytes;
{
	register int i;

	i = inroom(skt);
	if(i<=0 || nbytes==0)
		/*
		 * no room at the inn
		 */
		return(0);
	if(nbytes>i)
		nbytes = i;
	/*
	 * room at end
	 */
	i = skt->in.wtptr - skt->in.where;
	i = QUEUESIZE - i;
	if(i<nbytes) {
		movebytes(skt->in.wtptr,buffer,i);
		movebytes(skt->in.where,(char *)(buffer+i),nbytes-i);
		skt->in.wtptr = skt->in.where + nbytes - i;
	} else {
		movebytes(skt->in.wtptr,buffer,nbytes);
		if(i==nbytes) {
			skt->in.wtptr = skt->in.where;
		} else {
			skt->in.wtptr += nbytes;
		}
	}
	/*
	 * more stuff here
	 */
	return(nbytes);
}

#endif

/*
 *  checkfin
 *
 *  Check the FIN bit of an incoming packet to see if the connection
 *  should be closing, ACK it if we need to.
 *  Half open connections immediately, automatically close.  We do
 *  not support them.  As soon as the incoming data is delivered, the
 *  connection will close.
 */
static void checkfin(sknum,pkt)
register int sknum;
register TCPKT *pkt;
{
	register struct socket *skt;

	skt = (struct socket *) mapskt(sknum);

	if(pkt->t.flags & TFIN) {
		/*
		 * fin bit found
		 *
		 * count the FIN byte
		 */
		skt->in.nxt++;
		/*
		 * close-wait
		 */
		skt->state = SCWAIT;
		/*
		 * set ACK in packet
		 */
		skt->tcpout.t.flags = TACK;
		skt->tcpout.t.ack = longswap(skt->in.nxt);
		skt->credit = 0;
		/*
		 * connection closing
		 */
		ntptuev(CONCLASS,CONCLOSE,sknum);
		/*
		 * At this point, we know that we have received all data
		 * that the other side is allowed to send.  Some of that data
		 * may still be in the incoming queue.  As soon as that queue
		 * empties, finish off the TCP close sequence.  We are not
		 * allowing the user to utilize a half-open connection, but
		 * we cannot close before the user has received all of the
		 * data from the incoming queue.
		 */

		/*
		 * data remaining ?
		 */
		if(skt->in.rdptr==skt->in.wtptr) {
			skt->tcpout.t.flags |= TFIN;
			skt->state = SLAST;
			/*
			 * connection closing
			 */
			ntptuev(CONCLASS,CONCLOSE,sknum);
		}
		tcpsend(skt,0);
	}
}
                                                                                           