/* protin.c */

/*
 *  'protin' initializes packets to make them ready for transmission.
 *  For many purposes, pre-initialized packets are created for use by the
 *  protocol routines, especially to save time creating packets for
 *  transmit.
 *
 *	Important note :  Assumes that the hardware has been initialized
 *	and has set all the useful addresses such as the hardware addresses.
 *
 *  As this is a convenient place for it, this file contains many of the
 *  data declarations for packets which are mostly static (pre-allocated).
 */

#define DEF_PROTIN

/*
 *  Includes
 */
#include <stdio.h>
#include <malloc.h>

#include "vtcpip.h"
#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "prodat.h"
#include "tcpdat.h"
#include "mapskt.h"
#include "bytprc.h"
#include "cticks.h"

/*
 * protinit () 
 *
 * Calls all the other packet initialization keep this order as
 * some packet inits require lower layers already be initialized.
 */
VOID protinit()
{
	/*
	 * dlayer packets
	 */
	etherinit();
	/*
	 * ARP packets
	 */
	arpinit(); 
	/*
	 * ip packets
	 */
	ipinit();
	/*
	 * tcp packets
	 */
	tcpinit();
	/*
	 * udp packets
	 */
	udpinit();
}

/*
 * nteventinit ()
 *
 * Setup all the pointers for the event queue -- makes a circular list
 * which is required for error messages. ( called from Snetinit () )
*/

VOID nteventinit()
{
	register int i;

	for(i=0; i<NEVENTS; i++)
		nnq[i].next = i+1;
	nnq[NEVENTS-1].next = -1;
	nnefirst = 0;
	nnelast = 0;
	nnefree = 1;
}

/*
 * etherinit ()
 *
 * Setup the ethernet headers ( dlayer ) -- this needs to be done
 * first as it is copied for the other headers .
 */
VOID etherinit()
{
	movebytes(broadaddr,bseed,DADDLEN);
	/*
	 * some are broadcast
	 */
	movebytes(dblank.dest,broadaddr,DADDLEN);
	/*
	 * always from me
	 */
	movebytes(dblank.me,nnmyaddr,DADDLEN);
	/*
	 * mostly IP packets
	 */
	dblank.type = EIP;
}

/*
 * arpinit ()
 *
 * Setup an arp packet -- also sets up an arp cache
 */
VOID arpinit()
{
	register int i;

	movebytes(&arp.d,&dblank,sizeof(DLAYER));
	/*
	 * 0x0806 is ARP type
	 */
	arp.d.type = EARP;
	/*
	 * Ether = 1
	 */
	arp.hrd = intswap(HTYPE);
	/*
	 * IP protocol = 0x0800
	 */
	arp.pro = intswap(ARPPRO);
	/*
	 * Ethernet hardware length
	 */
	arp.hln = DADDLEN;
	/*
	 * IP length = 4
	 */
	arp.pln = 4;
	/*
	 * sender's hardware addr
	 */
	movebytes(arp.sha,nnmyaddr,DADDLEN);
	/*
	 * target hardware addr
	 */
	movebytes(arp.tha,broadaddr,DADDLEN);
	/*
	 * sender's IP addr
	 */
	movebytes(arp.spa,nnipnum,4);
	/*
	 *  initialize the ARP cache to 0 time, none are gateways to start
	 */
	for(i=0; i<CACHELEN; i++) {
		arpc[i].tm = 0L;
		arpc[i].gate = 0;
	}
}

/*
 * ipinit ()
 *
 * initialize one packet to use for internet transmission -- most packets
 * will be tcp/udp, so they will be initialized at a different layer,
 * but some require a generic ip packet.
 *
 * Also takes a guess at setting a netmask if it hasn't happened by now.
 *
 */
VOID ipinit()
{
	movebytes(&ipblank.d,&dblank,sizeof(DLAYER));
	/*
	 * smallest header, version 4
	 */
	ipblank.i.versionandhdrlen = 0x45;
	/*
	 * normal service
	 */
	ipblank.i.service = 0;
	/*
	 * no data yet, maximum size 
	*/
	ipblank.i.tlen = 576;
	ipblank.i.identity = 0;
	/*
	 * not a fragment of a packet
	 */
	ipblank.i.frags = 0;
	/*
	 * time to live in seconds
	 */
	ipblank.i.ttl = UDPTTL;
	/*
	 * default to UDP
	 */
	ipblank.i.protocol = PROTUDP;
	/*
	 * disable checksums for now
	 */
	ipblank.i.check = 0;
	/*
	 * my return address
	 */
	movebytes(ipblank.i.ipsource,nnipnum,4);
	/*
	 * to ?
	 */
	movebytes(ipblank.i.ipdest,broadip,4);
	/*
	 *  Make a blank ICMP packet for sending ICMP requests or responses
	 */
	movebytes(&icmpblank,&ipblank,sizeof(DLAYER)+sizeof(IPLAYER));
	icmpblank.i.protocol = PROTICMP;		
	/*
	 * create a mask which can determine whether a machine
	 * is on the same wire or not.  RFC950
	 * Only set the mask if not previously set.
	 * This mask may be replaced by a higher level
	 * request to set the subnet mask.
	 */

	/*
	 * now blank
	 */
	if(comparen(nnmask,"\0\0\0\0",4)) {
		if(!(nnipnum[0]&0x80)) {
			/*
			 * class A
			 */
			ntstmask(nnamask);
		} else
		if((nnipnum[0]&0xC0)==0x80) {
			/*
			 * class B
			 */
			ntstmask(nnbmask);
		} else 
		if((nnipnum[0]&0xC0)==0xC0) {
			/*
			 * class C
			 */
			ntstmask(nncmask);
		}
	}
}

/*
 * udpinit ()
 *
 * Setup ulist for receive of udp packets
 */
VOID udpinit()
{
	ulist.stale = 1;
	ulist.length = 0;

	movebytes(&ulist.udpout,&ipblank,sizeof(DLAYER)+sizeof(IPLAYER));
	ulist.tcps.proto = PROTUDP;
	movebytes(ulist.tcps.source,nnipnum,4);
}

/*
 * tcpinit ()
 *
 * setup for makesocket ()
 */
VOID tcpinit()
{
	register int i,sknum;

	/*
	 * no sockets open yet
	 */
	for(sknum=0; sknum<GSCKTS; sknum++) {
		sktlist[sknum].region =
		sktlist[sknum].jobnum =
		sktlist[sknum].sktid =
		sktlist[sknum].inport =
		sktlist[sknum].outport = 0;
		for(i=0; i<4; i++)
			sktlist[sknum].remoteip[i] = '\0';
	}
}

/*
 * makesocket
 *
 * This is the initialization for TCP based communication.  When a
 * socket needs to be created, this routine is called to do as much
 * pre-initialization as possible to save overhead during operation.
 *
 * This structure is created upon opening of a socket,
 * either listening or wanting to send.
 *
 * A TCP socket, in this implementation, includes all of the state
 * data for the socket connection, a complete packet for the
 * TCP transmission, and two queues, one each for sending and receiving.
 * The data associated with the queues is in struct window.
 */

/*
 * Smallest sequence number - 1
 */
static int seqnum = 0x07FF;

int makesocket(jobnum)
int jobnum;
{
	register int i,j;
	int nn,retval,sknum;
	register struct socket *skt;
	struct socket *q;
	int  sktstate[GSCKTS];
	int  sktjobnum[GSCKTS];
	long lasttime[GSCKTS];

	/*
	 * Check to see if any other connection is done with its socket
	 * buffer space.  The connection states of SCLOSED or STWAIT/SLAST
	 * with a timeout are checked.  The WAITTIME is relatively small,
	 * (just a few seconds) to allow quicker reuse of the limited
	 * number of available sockets.
	 */
	skt = -1;
	/*
	 * Scan for SCLOSED sockets
	 */
	sknum = 0;
	while(skt==-1 && sknum<GSCKTS && sktlist[sknum].region) {
		q = (struct socket *) mapskt(sknum);
		if(q->state==SCLOSED) {
			skt = q;
		} else {
			sktstate[sknum] = q->state;
			sktjobnum[sknum] = q->jobnum;
			lasttime[sknum] = q->out.lasttime;
		}
		/*
		 * socket # to return
		 */
		retval = sknum++;
	}
	/*
	 * Scan for STWAIT sockets with a time out
	 */
	if(skt==-1) {
		sknum = 0;
		while(skt==-1 && sknum<GSCKTS && sktlist[sknum].region) {
			if(sktstate[sknum]==STWAIT &&
			   elapsed(lasttime[sknum]) > WAITTIME) {
				skt = (struct socket *) mapskt(sknum);
			}
			/*
			 * socket # to return
			 */
			retval = sknum++;
		}
	}
	/*
	 * Scan for SLAST sockets with a time out
	 */
	if(skt==-1) {
		sknum = 0;
		while(skt==-1 && sknum<GSCKTS && sktlist[sknum].region) {
			if(sktstate[sknum]==SLAST &&
			   elapsed(lasttime[sknum]) > LASTTIME) {
				skt = (struct socket *) mapskt(sknum);
			}
			/*
			 * socket # to return
			 */
			retval = sknum++;
		}
	}
	/*  
	 * None available, get a new one
	 */
	if(skt==-1) {
		for(sknum=0; sktlist[sknum].region; sknum++)
			if(sknum>=GSCKTS) {
				/*
				 * out of room for sockets
				 */
				ntposterr(501);
				return(-1);
			}
		skt = (struct socket *) mapskt(sknum);
		retval = sknum;
	}
	if(skt==-1) {
		ntposterr(503);
		return(-1);
	}
	/*
	 * clear the whole socket
	 */
	clrskt(skt);
	/*
	 * initialize sktlist parameters
	 */
	sktlist[retval].region = 1;
	sktlist[retval].jobnum = jobnum;
	sktlist[retval].sktid =
	sktlist[retval].inport =
	sktlist[retval].outport = 0;
	for(i=0; i<4; i++)
		sktlist[retval].remoteip[i] = '\0';
	/*
	 * create a unique sequential sktid/port number
	 *
	 * This routine assumes that any user defined
	 * port will never be in the range 0x0800 - 0x3FFF
	 */
	do {
		if(++seqnum > 0x3FFF)
			seqnum = 0x0800;
		for(i=0; i<GSCKTS && seqnum!=sktlist[i].sktid; i++);
	} while(i<GSCKTS);
	sktlist[retval].sktid = sktlist[retval].inport =
	skt->sktid = skt->in.port = seqnum;
	/*
	 * socket allocated to a specific job
	 */
	skt->jobnum = jobnum;
	/*
	 * queuing parameters
	 */
	setupwindow(&skt->in,QUEUESIZE);
	setupwindow(&skt->out,QUEUESIZE);
	/*
	 * static initialization of non-zero socket elements
	 */
	skt->credit = nnwin;
	skt->sendsize = nnmtu;
	skt->rto = MINRTO;

	strncpy(skt->hostname,Scon.me,sizeof(skt->hostname));

	movebytes(&skt->tcpout,&ipblank,sizeof(DLAYER)+sizeof(IPLAYER));
	skt->tcpout.i.tlen = 0;
	skt->tcpout.i.protocol = PROTTCP;
	skt->tcpout.t.source = intswap(seqnum);
	skt->tcpout.t.seq = longswap(skt->out.nxt);

	skt->tcps.proto = PROTTCP;
	movebytes(skt->tcps.source,nnipnum,4);

	/*
	 *  install default maximum segment size which will be
	 *  sent out in the first ACK-SYN packet
	 */
	skt->tcpout.x.options[0] = 2;
	skt->tcpout.x.options[1] = 4;
	nn = intswap(nnseg);
	movebytes((char *)&skt->tcpout.x.options[2],(char *)&nn,2);
	skt->tcpout.t.hlen = 24<<2;

	/*
	 *  This socket is allocated for WAITTIME
	 */
	skt->state = STWAIT;
	return(retval);
}

/*
 * setupwindow
 *
 * Configure information about a window *w*
 */

VOID setupwindow(w,wsize)
register struct window *w;
register unsigned int wsize;
{
	w->rdptr = w->wtptr = w->queue = w->where;
	/*
	 * nothing here yet
	 */
	w->lasttime = cticks(NULL);
	w->ack = w->nxt = ((w->lasttime<<12) & 0x0FFFF000L);
	w->size = wsize - 1;
}
                                                                                                                                                                                                                           