/*
 *   config.c
 *
 *   Reads and stores the appropriate information from
 *   the config file specifying the configuration of
 *	(1) this machine
 *	(2) the available domain name servers
 *	(3) the available gateways
 */

/*
 *	Includes
 */
#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "tcpdat.h"
#include "stask.h"
#include "sesutl.h"
#include "sktlib.h"
#include "bytprc.h"
#include "mapskt.h"
#include "confgs.h"


static	char	*Sspace;

static	int	lineno,		/* line number in hosts file */
		position,	/* position for scanning string */
		constate,	/* state for config file parser */
		inquote;	/* flag, inside quotes now */

static char *Skeyw[] = {
/*==0==*/	"",	
		"name",		/* name of session */
		"host",		/* name of host */
		"hostip",	/* IP number */
		"port",		/* TCP port to go for */
/*==5==*/	"nameserver",	/* name server level */
		"gateway",	/* gateway level */
		"retrans",	/* initial retrans time */
		"contime",	/* timeout for opening connection */
		"mwin",		/* window to allow for this host */
/*==10==*/	"mseg",		/* maximum transfer size(in) */
		"mtu",		/* transfer unit (out) */
		"delete",	/* character for character deletion */
		"crmap",	/* map for Berkeley 4.3 */
		"duplex",	/* half duplex for IBM machines */
/*
 *  above are part of the data structure
 */
/*==15==*/	"copyfrom",	/* copy from another machine */
/*
 *  following are one-time entries
 */
		"netmask",	/* subnetting mask */
		"myip",		/* local machine's IP # */
		"myname",	/* identifying info */
		"domain",	/* default domain for lookup */
/*==20==*/	"nndomto",	/* time-out for DOMAIN */
		"nnretry",	/* # of retries */
		"nnarpto",	/* time-out for ARPs */
		"nndto",	/* time-out for data layer */
		"nnwin",	/* default maximum window */
/*==25==*/	"nnpkt",	/* default maximum packets per transq */
		"nnseg",	/* default maximum transfer size */
		"nnmtu",	/* default maximum transfer unit */
		"service",	/* enable service */
		"task",		/* enable task */
/*==30==*/	"logsession",	/* log TCPIP activity */
		""
	};

/*
 * Convert to Lower Case
 */
static VOID strlwr(st)
register char *st;
{
	while(*st) {
		*st = tolower(*st);
		*st++;
	}
}

/*
 *  Sgetspace
 *
 *  get some working space
 */
char *Sgetspace(siz)
register int siz;
{
	register char *s;

	s = (char *) malloc(siz);
	if(s == NULL) {
		Serrline(901);
	}
	return(s);
}

/*
 *  Sreadhosts
 *
 *  read in the hosts file into our in-memory data structure.
 *  Handle everything by keyword, see docs for specifications about file.
 */
int Sreadhosts()
{
	register FILE *fp;
	register int c,retval;

	Smachlist = Smptr = NULL;
	mno = 0;
	/*
	 * state vars
	 */
	position = constate = inquote = lineno = 0;
	/*
	 * open the configuration file
	 */
	if(NULL==(fp = fopen(Smachfile,"r"))) {
		Serrline(900);
		return(1);
	}
	/*
	 * get room for gathering stuff
	 */
	Sspace = Sgetspace(256);
	if(Sspace==NULL) {
		return(1);
	}
	retval = 0;
	while(!retval) {
		c = getc(fp);
		if(c=='#' && !inquote)
			/*
			 * skip to EOL
			 */
			while(c!=EOF && c!='\n' && c!='\r')
				c = getc(fp);
		if(c=='\n' || c=='\r')
			lineno++;
		/*
		 * add character to token
		 */
 		retval = Scontoken(c);
	 }
	fclose(fp);
	free(Sspace);
	/*
	 * make sure name is in list
	 */
	Smadd(-1,"default");
	/*
	 * EOF is normal end
	 */
	if(retval==EOF)
		return(0);
	else
		return(retval);
}

/*
 *  Serrline
 *
 *  prints the line number of the host file error and posts the event
 *  for the line number error and posts the hosts file error.
 */
VOID Serrline(n)
register int n;
{
	register char *p;

	p = skerrstring(-1);
	sprintf(p,"Config file: error at line %4d",lineno+1);
	skposterr(-1);
	skposterr(n);
}

/*
 *  Scontoken
 *
 *  tokenize the strings which get passed to Sconfile.
 *  Handles quotes and uses separators:  <33, ;:=
 */
int Scontoken(c)
register int c;
{
	register int retval;

	if(c==EOF) {
		Sspace[position++] = '\0';
		Sconfile(Sspace);
		/*
		 * make sure last entry gets copied
		 */
		if(!Sflags[0]) {
			if(ncstrcmp("default",Smptr->sname))
				Scopyfrom("default");
			else
				Scopyfrom("==");
		}
		return(-1);
	}
	/*
	 * skip over junk before token
	 */
	if(!position && Sissep(c))
		return(0);
	if(inquote || !Sissep(c)) {
		if(position>200) {
			Serrline(903);
			return(1);
		}
		/*
		 *  check for quotes,
		 * a little mixed up here, could be reorganized
		 */
		if(c=='"' ) {
			/*
			 * beginning of quotes
			 */
			if(!inquote) {
				inquote = 1;
				return(0);
			} else {
				/*
				 * turn off flag and drop through
				 */
				inquote = 0;
			}
		} else {
			/*
			 * check for EOL inside quotes
			 */
			if(c=='\n') {
				Serrline(904);
				return(1);
			}
			/*
			 * include in current string
			 */
			Sspace[position++] = (char)c;
			return(0);
		}
	}
	Sspace[position++] = '\0';
	/*
	 * pass the token along
	 */
	retval = Sconfile(Sspace);
	position = 0;
	inquote = 0;
	Sspace[0] = '\0';
	return(retval);
}

/*
 *  Sconfile
 *
 *  take the characters read from the file and parse them for keywords
 *  which require configuration action.
 */
int Sconfile(s)
register char *s;
{
	register int i;
	int a,b,c,d;

	switch(constate) {
		case 0:		/* lookup keyword */
			if(!(*s))
				/*
				 * empty token
				 */
				return(0);

			/*
			 * search the list for this keyword
			 */
			for(i=1; *Skeyw[i] && ncstrcmp(Skeyw[i],s); i++);

			if(!(*Skeyw[i])) {
				/*
				 * not in list
				 */
				Serrline(902);
				return(0);
			}
			/*
			 * change to state for keyword
			 */
			constate = 100+i;
			/*
			 *  check if this is a machine specific parm
			 *  without a machine to give it to.
			 *  "name" being the only machine specific
			 *  parm allowed, of course
			 */
			if(Smptr==NULL &&
				constate>CONNAME &&
				constate<=NUMSPECS) {
				Serrline(905);
				return(1);
			}
			break;

		case CONNAME:	/* session name */
			/*
			 *  allocate space for upcoming parameters
			 */
			if(Smachlist==NULL) {
				Smachlist = (struct machinfo *)
					Sgetspace(sizeof(struct machinfo));
				if(Smachlist == NULL)
					return(2);
				Smptr = Smachlist;
			} else {
				if(!Sflags[0]) {
					/*
					 * to make sure 'default' gets set
					 */
					if(ncstrcmp("default",Smptr->sname))
						Scopyfrom("default");
					else
						Scopyfrom("==");
				}
				Smptr->next = (struct machinfo *)
					Sgetspace(sizeof(struct machinfo));
				if(Smptr->next == NULL)
					return(2);
				Smptr = Smptr->next;
			}
			Smptr->next = NULL;
			/*
			 * guarantee to be null
			 */
			Smptr->hname = NULL;
			/*
			 * size of name string
			 */
			Smptr->sname = Sgetspace(position);
			if(Smptr->sname == NULL)
				return(2);
			/*
			 * keep name field
			 */
			strcpy(Smptr->sname,s);
			/*
			 * back to new keyword
			 */
			constate = 0;
			/*
			 * we have no parms
			 */
			for(i=0; i<NUMSPECS-99; i++)
				Sflags[i] = 0;
			/*
			 * new machine number
			 */
			Smptr->mno = ++mno;
			break;

		case CONHOST:	/* also a host name */
			Smptr->hname = Sgetspace(position);
			if(Smptr->hname == NULL)
				return(2);
			strcpy(Smptr->hname,s);
			constate = 0;
			/*
			 * set the flag to indicate hostname is found
			 */
			Sflags[CONHOST-100] = 1;
			break;

		case CONIP:	/* IP number for host */
			if(4!=sscanf(s,"%d.%d.%d.%d",&a,&b,&c,&d)) {
				Serrline(906);
				return(3);
			}
			/*
			 * keep number
			 */
			Smptr->hostip[0] = (char)a;
			Smptr->hostip[1] = (char)b;
			Smptr->hostip[2] = (char)c;
			Smptr->hostip[3] = (char)d;
			Smptr->mstat = HFILE;
			constate = 0;
			/*
			 * set the flag to indicate ip number found
			 */
			Sflags[CONIP-100] = 1;
			break;

		case CONPORT:	/* the port number */
			i = atoi(s);
			if(i<1)
				i = 0;
			Smptr->port = i;
			Sflags[CONPORT-100] = 1;
			constate = 0;
			break;

		case CONNS:	/* nameserver level */
			Smptr->nameserv = (char)atoi(s);
			/*
			 * keep NS
			 */
			if(!Sns || (Sns->nameserv>Smptr->nameserv))
				Sns = Smptr;
			constate = 0;
			Sflags[CONNS-100] = 1;
			break;

		case CONGATE:	/* this machine is a gateway */
			/*
			 * gateway level
			 */
			Smptr->gateway = (char)atoi(s);
			constate = 0;
			/*
			 * set the flag for this name being a gateway
			 */
			Sflags[CONGATE-100] = 1;
			break;

		case CONRETR:	/* how long before retransmitting */
			Smptr->retrans = atoi(s) * TICKSPERSEC;
			constate = 0;
			Sflags[CONRETR-100] = 1;
			break;

		case CONTO:	/* time until time out */
			i = atoi(s);
			if(i>2) {
				Smptr->conto = i;
				Sflags[CONTO-100] = 1;
			}
			constate = 0;
			break;

		case CONWIND:	/* transmission window for this host */
			Smptr->window = atoi(s);
			/*
			 * Limit to Machine Defaults
			 */
			if (Smptr->window >nnwin) {
				Smptr->window = nnwin;
			} else
			if (Smptr->window<MINWNDOSIZE) {
				Smptr->window = MINWNDOSIZE;
			}
			constate = 0;
			Sflags[CONWIND-100] = 1;
			break;

		case CONSEG:	/* segment size */
			Smptr->maxseg = atoi(s);
			/*
			 * Limit to Machine Defaults
			 */
			if (Smptr->maxseg>nnseg) {
				Smptr->maxseg = nnseg;
			} else
			if (Smptr->maxseg<MINSEG) {
				Smptr->maxseg = MINSEG;
			}
			constate = 0;
			Sflags[CONSEG-100] = 1;
			break;

		case CONMTU:	/* maximum transmission unit */
			Smptr->mtu = atoi(s);
			/*
			 * Limit to Machine Defaults
			 */
			if (Smptr->mtu>nnmtu) {
				Smptr->mtu = nnmtu;
			} else
			if (Smptr->mtu<TMINSIZE) {
				Smptr->mtu = TMINSIZE;
			}
			constate = 0;
			Sflags[CONMTU-100] = 1;
			break;

		case CONDELETE:	/* character deletion parameter */
			if(!ncstrcmp(s,"backspace"))
				Smptr->delete = 8;
			else
				Smptr->delete = 127;
			constate = 0;
			Sflags[CONDELETE-100] = 1;
			break;

		case CONCRMAP:	/* carriage return mapping */
			if(!ncstrcmp(s,"NONE"))
				Smptr->crmap = 0;
			else
			if(!ncstrcmp(s,"LINEFEED"))
				Smptr->crmap = 10;
			else
			if(!ncstrcmp(s,"4.3BSDCRNUL"))
				Smptr->crmap = BSDCRNUL;
			else
				Smptr->crmap = 0;
			Sflags[CONCRMAP-100] = 1;
			constate = 0;
			break;

		case CONDUP:	/* duplex */
			if(!ncstrcmp(s,"half")) {
				Smptr->halfdup = 1;
				Sflags[CONDUP-100] = 1;
			}
			constate = 0;
			break;

		case CONFROM:	/* copy the rest from another entry */
			Scopyfrom(s);
			Sflags[0] = 1;
			constate = 0;
			break;

		/*
		 *  now the one-time entries
		 *  Generally this information goes
		 *  into the "Scon" structure for later
		 *  retrieval by other routines.
		 */

		case CONME:	/* what my name is */
			strlwr(s);
			strncpy(Scon.me,s,sizeof(Scon.me));
			Scon.me[sizeof(Scon.me)-1] = '\0';
			constate = 0;
			break;

		case CONMYIP:	/* what my ip number is */
			constate = 0;
			if(!ncstrcmp(s,"rarp")) {
				movebytes(Scon.myipnum,s,4);
				break;
			}
			if(!ncstrcmp(s,"bootp")) {
				movebytes(Scon.myipnum,s,4);
				break;
			}
			if(4!=sscanf(s,"%d.%d.%d.%d",&a,&b,&c,&d)) {
				Serrline(908);
				return(3);
			}
			/*
			 * put number back in s
			 */
			Scon.myipnum[0] = (char)a;
			Scon.myipnum[1] = (char)b;
			Scon.myipnum[2] = (char)c;
			Scon.myipnum[3] = (char)d;
			break;

		case CONMASK:	/* the subnet mask */
			if(4!=sscanf(s,"%d.%d.%d.%d",&a,&b,&c,&d)) {
				Serrline(907);
				return(3);
			}
			Scon.netmask[0] = (char)a;
			Scon.netmask[1] = (char)b;
			Scon.netmask[2] = (char)c;
			Scon.netmask[3] = (char)d;
			Scon.havemask = 1;
			constate = 0;
			break;

		case CONDEF:	/* default domain */
			/*
			 * space for name
			 */
			Scon.defdom = Sgetspace(position);
			if(Scon.defdom == NULL)
				return(2);
			/*
			 * copy it in
			 */
			strcpy(Scon.defdom,s);
			constate = 0;
			break;

		case CONDOMTO:	/* DOMAIN timeout value */
			i = atoi(s);
			if(i>1)
				nndomto = i;
			constate = 0;
			break;

		case CONNDOM:	/* DOMAIN number of retries */
			i = atoi(s);
			if(i>1)
				nnretry = i;
			constate = 0;
			break;

		case CONARPTO:	/* need to lengthen arp time-out (secs) */
			i = atoi(s);
			if(i>0)
				nnarpto = i;
			constate = 0;
			break;

		case CONDTO:	/* Data layer timeout */
			i = atoi(s);
			if(i>0)
				nndto = i;
			constate = 0;
			break;

		case CONNWIND:	/* default maximum window */
			nnwin = atoi(s);
			if (nnwin>MAXWNDOSIZE) {
				nnwin = MAXWNDOSIZE;
			} else
			if (nnwin<MINWNDOSIZE) {
				nnwin = MINWNDOSIZE;
			}
			constate = 0;
			break;

		case CONNPKT:	/* default maximum packet count per transq */
			nnpkt = atoi(s);
			if (nnpkt>MAXPCNT) {
				nnpkt = MAXPCNT;
			} else
			if (nnpkt<MINPCNT) {
				nnpkt = MINPCNT;
			}
			constate = 0;
			break;

		case CONNSEG:	/* default maximum segment size */
			nnseg = atoi(s);
			if (nnseg>MAXSEG) {
				nnseg = MAXSEG;
			} else
			if (nnseg<MINSEG) {
				nnseg = MINSEG;
			}
			constate = 0;
			break;

		case CONNMTU:	/* default maximum transfer unit */
			nnmtu = atoi(s);
			if (nnmtu>TMAXSIZE) {
				nnmtu = TMAXSIZE;
			} else
			if (nnmtu<TMINSIZE) {
				nnmtu = TMINSIZE;
			}
			constate = 0;
			break;

		case CONSRVC:		/* service requests */
		case CONTASK:		/* task requests */
		case CONLOGSESSION:	/* log sessions */
		default:
			constate = 0;
			break;
	}
	return(0);
}

/*
 *  Shostfile
 *
 *  if the user wants to change the host file name from 'tcpip.cfg' to
 *  something else.
 */
VOID Shostfile(ptr)
register char *ptr;
{
	/*
	 *  note that the area with the file name must stay allocated for
	 *  later reference, typically it is in some argv[] parm.
	 */
	Smachfile = ptr;	
}

/*
 *  Sissep
 *
 *  is the character a valid separator for the hosts file?
 *  separators are white space, special chars and =
 */
int Sissep(c)
register int c;
{
	if(c<33 || c=='=')
		return(1);
	return(0);
}
                                