/* prodef.h */

/*
 *  Protocol structures for network communication
 *
 *  This file contains the structure definitions for each type of 
 *  protocol that this program wishes to handle.  A companion file,
 *  'protinit.c' initializes sample versions of each type of header, 
 *  improving the efficiency of sending packets with constants in most
 *  of the fields.
 */

/*
 *  Ethernet frames
 *
 *  All Ethernet transmissions should use this Ethernet header which
 *  denotes what type of packet is being sent.
 *
 *  The header is 14 bytes.  The first 6 bytes are the target's hardware
 *  Ethernet address, the second 6 are the sender's hardware address and
 *  the last two bytes are the packet type.  Some packet type definitions
 *  are included here.
 *
 *  the two-byte packet type is byte-swapped, PDP-11 is lo-hi, Ether is hi-lo
 */

/*
 * swapped codes
 */
#define  EXNS    0x0006
#define  EIP     0x0008
#define  EARP    0x0608
#define  ERARP	 0x3580
#define  ECHAOS  0x0408

/*
 * these don't need swapping
 */
/*
#define  EXNS    0x0600
#define  EIP     0x0800
#define  EARP    0x0806
#define  ERARP	 0x8035
#define  ECHAOS  0x0804
*/


typedef struct ether {
	unsigned
	int	iostat,		/* io device status */
		iolen;		/* io packet length */
	char	dest[DADDLEN],	/* where the packet is going */
		me[DADDLEN];	/* who am i to send this packet */
	unsigned
	int	type;		/* Ethernet packet type  */
}DLAYER;

/*
 *  Dave Plummer's  Address Resolution Protocol (ARP) (RFC-826) and 
 *  Finlayson, Mann, Mogul and Theimer's Reverse ARP packets.
 *
 *  Note that the 2 byte ints are byte-swapped.  The protocols calls for
 *  in-order bytes, and the PDP-11 is lo-hi ordered.
 *   
 *  These codes require swapping
 */
#define RARPR	0x0004	 	/*  RARP reply, from host */
#define RARPQ	0x0003		/*  RARP request */
#define ARPREP  0x0002		/*  reply */
#define ARPREQ  0x0001		/*  request */
#define ARPPRO	0x0800		/*  IP protocol */
#define HTYPE 	0x0001		/*  Ethernet hardware type */

typedef struct plummer {
	DLAYER	d; 	/* data link layer packet header */
	unsigned
	int	hrd,	/* hardware type, Ethernet = 1 */
		pro;	/* protocol type to resolve for */
	char	hln,	/* byte length of hardware addr = 6 for ETNET */
		pln;	/* byte length of protocol = 4 for IP */
	unsigned
	int	op;	/* opcode, request = 1, reply = 2, RARP = 3,4 */
	char	sha[DADDLEN],
		spa[4],
		tha[DADDLEN],
		tpa[4];
		/*
		 *  the final four fields (contained in 'rest') are:
		 *	sender hardware address:   sha	   hln bytes
		 *	sender protocol address:   spa	   pln bytes
		 *	target hardware address:   tha	   hln bytes
		 *	target protocol address:   tpa	   pln bytes
		 */
}ARPKT;

/*
 *  ARP cache
 *  Data structure for saving low-level information until needed
 */
struct acache {
	char	hrd[DADDLEN],	/* hardware address for this IP address */
		ip[4],		/* the IP # in question */
		gate;		/* is this a gateway? */
	long	tm;		/* time information */
};

/*
 *   Internet protocol
 */
typedef struct iph {
	char	versionandhdrlen;	/* I prefer to OR them myself */
					/* each half is four bits */
	char	service;		/* type of service for IP */
	unsigned
	int	tlen,			/* total length of IP packet */
		identity,		/* these are all BYTE-SWAPPED! */
		frags;			/* combination of flags and value */
	char	ttl,			/* time to live */
		protocol;		/* higher level protocol type */
	unsigned
	int	check;			/* header checksum, byte-swapped */
	char	ipsource[4],		/* IP addresses */
		ipdest[4];
}IPLAYER;

/*  
 *  full IP packet, with data and ip header
 */
typedef struct ip {
	DLAYER	d;
	IPLAYER	i;
	union {
		char data[536];		/* largest recommended, */
		char options[40];	/* may include options */
	} x;
}IPKT;

#define PROTUDP		17
#define PROTTCP		6	/* standard protocol types for IP */
#define PROTICMP	1

/*
 *  ICMP packet
 *
 *  all of them are of a similar form, some generic fields are spec'd here.
 */
typedef struct icmph {
	char	type,		/* ICMP type field */
		code;		/* ICMP code field */
	unsigned
	int	check,		/* checksum */
		part1,part2;	/* depends on type and code */
}ICMPLAYER;

typedef struct icmp {
	DLAYER	d;
	IPLAYER i;
	ICMPLAYER c;
	char	data[ICMPMAX];
}ICMPKT;

/*
 *  TCP protocol
 *	define both headers required and create a data type for a typical
 *	outgoing TCP packet (with IP header)
 *   
 *  Note:  So far, there is no way to handle IP options fields
 *	which are associated with a TCP packet.  They are mutually exclusive
 *	for both receiving and sending.  Support may be added later.
 *
 *   The tcph and iph structures can be included in many different types of
 *   arbitrary data structures and will be the basis for generic send and
 *   receive subroutines later.  For now, the packet structures are optimized 
 *   for packets with no options fields.  (seems to be almost all of them from
 *   what I've observed.
 */

typedef struct tcph {
	unsigned
	int	source,dest;	/* TCP port numbers, all byte-swapped */
	long	seq,ack;	/* sequence, ACK numbers */
	char	hlen, 		/* length of TCP header in 4 byte words */
		flags;		/* flag fields */
	unsigned
	int	window,		/* advertised window, byte-swapped */
		check,		/* TCP checksum of whole packet */
		urgent;		/* urgent pointer, when flag is set */
}TCPLAYER;

/*
 *  used for computing checksums in TCP
 */
struct pseudotcp {
	char	source[4],dest[4],	/* IP #'s for source,dest */
		z,proto;		/* zero and protocol number */
	unsigned
	int	tcplen;			/* byte-swapped length field */
};

typedef struct tcp {
	DLAYER	d;
	IPLAYER	i;
	TCPLAYER t;
	union {
		char options[40];	/* not very likely, except on SYN */
		char data[TMAXSIZE];	/* largest TCP data we will use */
	} x;
}TCPKT;

/* 
 *  flag field definitions, first two bits undefined
 */

#define TURG	0x20
#define TACK	0x10
#define TPUSH	0x08
#define TRESET	0x04
#define TSYN	0x02
#define TFIN	0x01

/*
 *  Session structure and definitions to be
 *  included with the socket allocation
 */

/*  
 *  terminal modes, can be changed by external program
 *  VTEK is a VT terminal that can accept graphics
 *  TEK  is a tek graphics mode for a VT
 *  VT   is a VT-only type, cannot be changed to graphics mode 
 */

#define VTEKTYPE 	1
#define DUMBTYPE 	2
#define VTTYPE 		3
#define TEKTYPE 	4
#define RASTYPE 	5


struct twin {
char	linemode[80],	/* line mode buffer for session */
	termtype[16],	/* terminal type */
	capfil[16];	/* capture file name */
FILE	*capfp;		/* capture file io pointer */
int	capon,		/* does this session own a capture file? */
	pnum,		/* port number associated */
	telstate,	/* telnet state for this connection */
	substate,	/* telnet subnegotiation state */
	termsent,	/* has terminal type been sent? */
	termstate,	/* terminal type for this connection */
	width,		/* width of the window */
	rows,		/* Number of rows in the window */
	binary,		/* negotiate for binary traffic */
	igoahead,	/* negotiation for suppress go-ahead */
	ugoahead,	/* neg. for his suppress go-ahead */
	echo,		/* line mode or echo mode? */
	halfdup,	/* half duplex mode overrides line mode */
	crfollow,	/* what is supposed to follow a CR? NUL or LF? */
	bksp,		/* what keycode for backspace? */
	del,		/* for delete? */
	lmflag,		/* Are we in linemode? */
	slc[19];
};

/*
 *   TCP queuing
 *   data types for all TCP queuing operations
 *   Each open port will have one of these structures assigned to it.
 */

struct window {
	long	nxt,		/* sequence number, not byte-swapped */
		ack;		/* what the other machine acked */
	long	lasttime;	/* (signed) used for timeout checking */
	char	where[QUEUESIZE],	/* storage for queue */
		*rdptr,		/* queue read  data position */
		*wtptr,		/* queue write data position */
		*queue,		/* user queue read/write posiiton */
		push;		/* flag for TCP push */
	unsigned
	int	urgent,		/* urgent flag / pointer for TCP */
		size,		/* size of window advertised */
		port;		/* port numbers from one host or another */
};

/*
 * Out of Order Packet Reassembly Sequence Data
 */

typedef struct sequence {
	long		sq;
	long		sqnxt;
	unsigned int	sqflg;
	char	*	sqptr;
}SQSPC;

/*
 * TCP/IP 'Socket'
 */

struct socket {
	int	sktport;	/* telnet=23, ftp=21, other... */
	int	skfport;	/* forced from port */
	char	hostname[70];	/* internet name of host */
	char	request[70];	/* requested machine/address */
	struct	window in,out;	/* TCPIP <--> USER  queues */
	SQSPC	seq[NITEMS];	/* TCPIP Out of Order Packet Reassembly */
	int	seqncs;		/* Out of Order Packet Count */
	TCPKT	tcpout;		/* pre-initialized as much as possible */
	int	state;		/* connection state */
	struct	pseudotcp tcps;	/* pseudo-tcp for checksumming */
	int	credit,		/* choked-down window for fast hosts */
		sendsize,	/* MTU value for this connection */
		rto;		/* retrans timeout */
	struct	twin  session;	/* session associated with this port */
	struct	machinfo *mpp;	/* socket machine info record pointer */
	struct	machinfo mach;	/* socket machine info record */
	char	sname[70];	/* name of session */
	char	hname[70];	/* name of that machine */
	int	jobnum;		/* associated TSX+/RT11 job number */
	int	sktid;		/* socket id */
};

#ifdef	TCPIP_DEF
struct plist {
	int	region,		/* region allocated status */
		jobnum,		/* associated system job */
		sktid,		/* socket id */
		inport,		/* input  port # */
		outport;	/* output port # */
	char	remoteip[4];	/* ip connected to */
};
#else
struct plist {
	int	lpnum;		/* local port number */
	int	sktid;		/* socket id */
};
#endif

/*
 *  TCP states
 *	each connection has an associated state in the connection flow.
 *	the order of the states now matters, those less than a certain
 *	number are the "inactive" states.
 */
#define SCLOSED 	1
#define SLISTEN 	2
#define STWAIT		3
#define SSYNR   	4
#define SSYNS		5
#define SEST		6
#define SFW1		7
#define SFW2		8
#define SCLOSING 	9
#define SCWAIT		10
#define SLAST		11

/*
 *	 services which we will want to use
 */
#define HFTP		 21
#define HTELNET		 23
#define	HSMTP		 25
#define HCNCT		 27
#define	HGOPH		 70
#define HHTTP		 80
#define	HPOP3		110
#define	HLPD		515

/*
 *  UDP
 *   User Datagram Protocol
 *   Each packet is an independent datagram, no sequencing information
 *
 *   UDP uses the identical checksum to TCP
 */

typedef struct udph {
	unsigned
	int	source,
		dest;		/* port numbers, all byte-swapped */
	unsigned
	int	length,		/* length of packet, including hdr */
		check;		/* TCP checksum of whole packet */
}UDPLAYER;

typedef struct udp {
	DLAYER	d;
	IPLAYER	i;
	UDPLAYER u;
	char	data[UMAXLEN];	/* largest UDP data we will use */
}UDPKT;

struct uport {
	UDPKT	udpout;
	struct	pseudotcp tcps;	/* pseudo-tcp for checksumming */
	unsigned
	int	listen,		/* what port should this one listen to? */
		length;		/* how much data arrived in last packet? */
	char	data[UMAXLEN],	/* incoming, last datagram of that type */
		who[4],		/* who sent it to me ? */
		stale;		/* have we read this packet yet? */
};

/*
 *  event queue
 *
 *  records what happens, especially errors, and keeps them for any
 *  routines that poll, looking for what happened.
 *  Sixteen event classes are masked in the event class word.
 *  There can be 65535 event types per class.
 *  The data field normally contains the socket number this event
 *  is associated with.
 */
struct eq {
	int	eclass,		/* class, defined in evtdef.h */
		event,		/* which event */
		idata,		/* socket number / or other data */
		next;		/* ordering for events in queue  */
};

/*  
 *  events which can occur and be placed into the event queue
 */
#define NEVENTS 50

/*
 *   Definition of the TSX+/RT11 interprocess message format
 *   used for communication between TCPIP and all
 *   client processes.
 */
typedef struct	message {
	char	uid[8];		/* message channel id */
	int	jobnum;		/* TSX+/RT11 job number */
	int	class,		/* message class */
		event,		/* message event */
		data;		/* message data */
}MSGPKT;

                                               