/* frags.c */

/*
 *	Includes
 */
#include <stdio.h>

#include "vtcpip.h"
#include "dfault.h"
#include "evtdef.h"
#include "hstdef.h"
#include "prodef.h"
#include "prodat.h"
#include "tcpdat.h"
#include "kbdutl.h"
#include "mapskt.h"
#include "cticks.h"
#include "bytprc.h"
#include "frags.h"


/*
 *	IP Fragment Assembler
 *
 *	Thanks to Svyatoslov Ws. Mironov for the
 *	IP Fragment Assembler coding.
 *
 *	Svyatoslav Ws. Mironov
 *	Institute of Physiology SPbUniversity
 *	papa@svyat.usr.pu.ru
 *
 *****
 *
 *	The coding has been slightly modified to use a
 *	mapped region for the packet reassembly buffers.
 *
 *	1.	maprgn() and clrrgn() routines added to GBLSKT.MAC
 *
 *	2.	Module IP.C modified to call frgasm()
 *
 */

/*
 *	OverView
 *
 *	The IP Fragment Assembler attempts to
 *	reassemble incoming fragmented IP packets.
 *
 *	(1)	At the first instance of a fragmented
 *		packet the fragment reassembly buffer
 *		is created as a mapped region named
 *		MRGN00.  This 8K byte region contains
 *		enough space to simultaneously reassemble
 *		5 fragmented IP packets.
 *
 *	(2)	The arriving fragmented packet is then allocated
 *		a fragmentation reassembly buffer or inserted
 *		into a previously allocated buffer.  If the
 *		reassembled packet is too large the packet is
 *		is dropped and the reassembly buffer cleared.
 *
 *	(3)	When the fragmented IP packet has been
 *		completely reassembled the full IP packet
 *		is copied from the reassembly buffer
 *		(a mapped region) into the normal IP packet
 *		buffer (a non mapped area contained in the
 *		ENDRVR.MAC module).  This overwrites the last
 *		incoming IP fragment with the complete
 *		reassembled IP packet.  The reassembly buffer
 *		is cleared.
 *		
 *	(4)	The mechanism used to handle incomplete or
 *		lost fragments is: (a) if all buffers are not
 *		in use, then ignore those buffers with missing
 *		fragments, (b) if the processing of fragmented
 *		packets requires another buffer, then scan the
 *		fragmentation buffers for the oldest packet
 *		(must be at least 5 seconds old), and drop the
 *		old packet creating space for a new fragmented
 *		packet.
 *
 *	(5)	If space cannot be allocated for a new reassembly
 *		buffer following the proceedure outlined in (4)
 *		above then the packet is dropped.
 */


/*
 * Local Debugging Definitions
 */

/*
#define	pktDBG
#define	asmDBG
#define	copDBG
#define	chkDBG
*/

static	int	initfrag = 1;	/* Initialize Fragmentation Buffer Flag */

/*
 *  frgasm
 *
 *  try to reassemble fragmented IP packets
 *
 *  returned codes:
 *   1 - packet assembled
 *   0 - fragment buffered
 *  -1 - no room for buffering
 *  -2 - packet too large
 */
int frgasm(pkt)
register XPKT *pkt;
{
	register ASMB *buff;
	register FRGN *frgn;

#ifdef pktDBG
    printf("frgasm: pkt TL %04d, IDN %o, FLGS %o, FO %03d, TTL %03d\r\n",
	intswap		(pkt->i.tlen),			/* totat lengh */
	intswap		(pkt->i.identity),		/* idetification */
			(pkt->i.frags & 0xe0) >> 5,	/* flags */
	intswap		(pkt->i.frags) & 0x1FFF,	/* frag offset */
	(unsigned int)	(pkt->i.ttl)			/* time to live */
    );
#endif
	/*
	 * Create and Initialize the
	 * Mapped Fragmentation Buffer
	 * on the first call to frgasm().
	 */
	if (initfrag) {
		frgn = (FRGN *) maprgn(FRAGREGION);
		if (frgn == NULL) {
#ifdef asmDBG
	printf ("frgasm: maprgn() failed\r\n");
#endif
			return(-1);
		}
		clrrgn(frgn);
		initfrag = 0;
	}

	/*
	 * get buffer address for this fragment
	 */
	if ((buff=idnsch(pkt)) == NULL) {
		if ((buff=allocb(pkt)) == NULL) {
#ifdef asmDBG
	printf ("frgasm: allocb() failed\r\n");
#endif
			return (-1);
		}
	}
#ifdef asmDBG
	printf ("frgasm: buff = %o\r\n",buff);
#endif

	/*
	 * Set Time
	 */
	cticks(&buff->time);

	/*
	 * Last fragment processing
	 */
	if ( !(pkt->i.frags & 0x20) ) {
		buff->length	= (8 * (intswap(pkt->i.frags) & 0x1FFF))
				+ intswap(pkt->i.tlen)
				- sizeof(IPLAYER);
#ifdef asmDBG
	printf ("frgasm: buff->length = %d\r\n", buff->length);
#endif
		/*
		 * Check Packet Size
		 */
		if (buff->length > IMAXSIZE) {
			buff->i.versionandhdrlen = 0;
#ifdef asmDBG
	printf("frgasm: packet too large\r\n");
#endif
			return(-2);
		}
	}

	/*
	 * copy data to buffer
	 */
	if (copdat(buff,pkt)) {
	    /* 
	     * last fragment received?
	     */
	    if (buff->length) {
		/*
		 * all fragments assembled?
		 */
		if (asmchk(buff)) {
			/*
			 * rebuild IP header
			 */
			buff->i.frags = 0;
			buff->i.check = 0;
			buff->i.tlen = intswap(sizeof(IPLAYER)+buff->length);
			/*
			 * replace incoming packet with reassembled packet
			 */
			movebytes(pkt, buff,
			    sizeof(DLAYER) + sizeof(IPLAYER) + buff->length);
			/*
			 * Release Buffer
			 */
			buff->i.versionandhdrlen = 0;
#ifdef asmDBG
	printf ("frgasm: fragmented packet assembled\r\n");
#endif
			return (1);
		}
	    }
	} else {
	    /*
	     * Packet too large
	     */
	    buff->i.versionandhdrlen = 0;
#ifdef asmDBG
	printf ("frgasm: copdat() failed\r\n");
#endif
	    return (-2);
	}
#ifdef asmDBG
	printf ("frgasm: waiting for more...\r\n");
#endif
	return (0);
}


/*
 *  copdat
 *
 *  Copy fragment's data to buffer with bitmap filling
 *  Returns 0 (FALSE) if buffer size is insufficient, else
 *  Returns 1 (TRUE).
 */
int copdat(buff,pkt)
register ASMB *buff;
register XPKT *pkt;
{
	unsigned int length,sh,shift;
	unsigned int i,s;
	int lg,d;
	char bit;

	/*
	 * data length in bytes
	 */
	length	= intswap(pkt->i.tlen) - sizeof(IPLAYER);
	/*
	 * data length in offset units (8 byte)
	 */
	lg = length/8;
	if (length - lg*8) {
		lg += 1;
	}
	/*
	 * data shift in offset units (8 byte)
	 */
	sh = intswap(pkt->i.frags) & 0x1FFF;
	/*
	 * data shift in bytes
	 */
	shift = sh << 3;
	/*
	 * Check Buffer Size
	 */
	if ((shift + length) > IMAXSIZE) {
		return (0);
	}

#ifdef copDBG
	printf("copdat: shift = %d, length = %d, shift+length = %d,\r\n",
		shift,length,shift+length);
	printf("copdat: sh = %d, lg = %d\r\n", sh,lg);
#endif

	/*
	 * copy data fragment
	 */
	movebytes((buff->x.data) + shift, pkt->x.data, length);

	/*
	 * bitmap filling: (8 bytes/bit)
	 */
	s = sh >> 3;
	s <<= 3;
	d = sh - s;
	if (d) {

#ifdef copDBG
	printf("copdat: bitmap[%d] BITS:", s >>3);
#endif
		for (bit = (1 << d);
		     lg > 0, 8-d > 0;
		     bit <<= 1, lg--, sh++, d++ ) {
			buff->bitmap[s >> 3] |= bit;
#ifdef copDBG
	printf(" %o",(unsigned int) bit);
#endif
		}
#ifdef copDBG
	printf("\r\n");
#endif
	}

#ifdef copDBG
	printf("copdat: bitmap BYTES:");
#endif
	for (i= (sh >> 3); lg >= 8; i++, lg -= 8) {
		buff->bitmap[i] = 0xFF;
#ifdef copDBG
	printf(" %d",i);
#endif
	}
#ifdef copDBG
	printf("\r\n");
#endif

#ifdef copDBG
	printf("copdat: bitmap[%d] BITS:",i);
#endif
	for (bit = 1; lg > 0; bit <<= 1, lg -= 1) {
		buff->bitmap[i] |= bit;
#ifdef copDBG
	printf(" %o",(unsigned int) bit);
#endif
	}
#ifdef copDBG
	printf("\r\n");
#endif
	return (1);
}

/*
 *  asmchk
 *
 *  Returns 1 (TRUE) if all fragments are received or
 *  Returns 0 (FALSE) if something is missing.
 */
int asmchk(buff)
register ASMB *buff;
{
	register int lg,i;
	char bit;

#ifdef chkDBG
	printf("asmchk: buff->length = %d\r\n", buff->length);
#endif
	if (buff->length == 0) {
		return (0);
	}

	lg = buff->length/8;
	if (buff->length - (lg * 8)) {
		lg += 1;
	}
#ifdef chkDBG
	printf("asmchk: lg = %d\r\n", lg);
#endif

	for (i=0; lg >= 8 ; i++, lg -= 8) {
		if (buff->bitmap[i] != 0xFF) {
#ifdef chkDBG
	printf("asmchk: BYTES: lg = %d, i = %d\r\n", lg,i);
#endif
			return (0);
		}
	}

	for (bit = 1; lg > 0; bit <<= 1, lg -= 1) {
		if ((buff->bitmap[i] & bit) == 0 ) {
#ifdef chkDBG
	printf("asmchk: BITS: lg = %d, bit = %o\r\n",lg,(unsigned int) bit);
#endif
			return (0);
		}
	}
	return (1);
}

/*
 *  idnsch
 *
 *  Search fragmentation buffers for matching fragment identification
 *  Returns address of matching buffer or
 *  Returns NULL if not found.
 */
ASMB *idnsch(pkt)
register XPKT *pkt;
{
	register int i;
	register ASMB *buff;
	FRGN *frgn;

	buff = NULL;
	frgn = (FRGN *) maprgn(FRAGREGION);

	for (i=0; i < MAXBFN; i++) {
	  buff = &frgn->asmbuf[i];
	  if (buff->i.versionandhdrlen != 0) {
	    if (buff->i.identity == pkt->i.identity) {
	      if (buff->i.protocol == pkt->i.protocol) {
		if (comparen(buff->i.ipsource,pkt->i.ipsource,4)) {
		  if (comparen(buff->i.ipdest,pkt->i.ipdest,4)) {
		    return (buff);
		  }
	        }
	      }
	    }
	  }
	}
	return (NULL);
}

/*
 *  allocb
 *
 *  Return Address of New Fragmentation Reassembly Buffer or
 *  Return NULL if none available
 */

ASMB *allocb(pkt)
XPKT *pkt;
{
	register int i,j;
	register ASMB *buff;
	long itime,jtime;
	FRGN *frgn;

	buff = NULL;
	frgn = (FRGN *) maprgn(FRAGREGION);

	/*
	 * Check for a free buffer
	 */
	for (i=0; i < MAXBFN; i++) {
		if (frgn->asmbuf[i].i.versionandhdrlen == 0) {
			buff = &frgn->asmbuf[i];
			break;
		}
	}
	/*
	 * Scan for oldest buffer more than 5 seconds old
	 */
	if (buff == NULL) {
		for (i=0,j=-1; i < MAXBFN; i++) {
			itime = elapsed(frgn->asmbuf[i].time);
			if (itime > TICKSPERSEC*5) {
				if ((j == -1) || (itime > jtime)) {
					j = i;
					jtime = itime;
				}
			}
		}
		if (j != -1) {
			buff = &frgn->asmbuf[j];
		}
	}
	/*
	 * Found a Buffer, Initialize and Copy Fragment
	 */
	if (buff != NULL) {
		/*
		 * Inititialize Length
		 */
		buff->length = 0;
		/*
		 * Clear data bitmap
		 */
		for (j=0; j < sizeof(buff->bitmap); j++) {
			buff->bitmap[j] = 0;
		}
		/*
		 * Copy Ethernet and IP headers
		 */
		movebytes (buff, pkt, sizeof(DLAYER) + sizeof(IPLAYER));
	}
	return (buff);
}

                                                                                                                                                                                                                                                                         