/* Undelete.c V1.0: A program to recover deleted files.
        
   This program will scan a disk for the given filename, and copy the file
to another disk if found.  It is NOT case sensitive, and will find all
copies of the file on the disk with the same name.

   When the file is copied, the number of the first block of the file is
appended to the name to keep multiple copies from overwriting each other.

author: James Cooper Jr.
	2104B Rogers Dr.
	Fayetteville, NC 28303

With thanks to:
	Tom Wilcox
	3047 Cameron Way
	Santa Clara, CA 95051

for his fixdisk.c program.  This program is given to the public domain, but
please keep the credits intact. */

#include "exec/types.h"
#include "exec/nodes.h"
#include "exec/lists.h"
#include "exec/memory.h"
#include "exec/interrupts.h"
#include "exec/ports.h"
#include "exec/libraries.h"
#include "exec/io.h"
#include "exec/tasks.h"
#include "exec/execbase.h"
#include "exec/devices.h"
#include "lattice/ctype.h"
#include "devices/trackdisk.h"
#include "stdio.h"

#define TD_READ CMD_READ
#define BLOCKSIZE TD_SECTOR

struct MsgPort *diskport;
struct IOExtTD *diskreq;

#define NAMELENGTH 408
#define FILENAME   409
#define TypeDATA     8

struct DataBlock
   {
    LONG  type, key, seqnum, size, next, checksum;
    UBYTE data[BLOCKSIZE-24];
   } DataBlock, *diskdata;

FILE *fopen();

int blocks = 0;

extern struct MsgPort *CreatePort();
extern struct IORequest *CreateExtIO();

ULONG diskChangeCount;

#include "devices/extio.h"

char name[80];

Bool ReadBlock (Block, Kind)
   LONG Block;
   char *Kind;
{
   diskreq->iotd_Req.io_Length = BLOCKSIZE;
   diskreq->iotd_Req.io_Data = (APTR) diskdata;
               /* show where to put the data when read */
   diskreq->iotd_Req.io_Command = ETD_READ;
               /* check that disk not changed before reading */
   diskreq->iotd_Count = diskChangeCount;
   diskreq->iotd_Req.io_Offset = BLOCKSIZE*Block;
   DoIO(diskreq);

   if (diskreq->iotd_Req.io_Error != 0)
      { 
	 printf("*** Can't read %s from block %ld; error %ld\n",
		Kind,Block,diskreq->iotd_Req.io_Error); 
	 return (FALSE);
      }
   return (TRUE);
}

MotorOn()
{
    /* TURN ON DISK MOTOR ... old motor state is returned in io_Actual */
    diskreq->iotd_Req.io_Length = 1;  /* 1 => motor is to be turned on */
    diskreq->iotd_Req.io_Command = TD_MOTOR;   /* operate on the motor */
    DoIO(diskreq);
}

MotorOff()
{
    diskreq->iotd_Req.io_Length = 0; /* 0 => motor is to be turned off */
    diskreq->iotd_Req.io_Command = TD_MOTOR;   /* operate on the motor */
    DoIO(diskreq);
}

CopyFile(FirstBlock)
   LONG FirstBlock;
{
   FILE *file;
   LONG block = FirstBlock;

   printf ("\n\n");

   sprintf (name, "DF1:%s.%ld", &DataBlock.data [FILENAME], block);
   printf ("Writing file %s\n", name);

   if ((file = fopen (name, "w")) == NULL)
   {
       printf ("Cannot open %s\n\n", name);
       return (0);
   }

   for (block = FirstBlock; block != 0; block = DataBlock.next)
   { 
     if (!ReadBlock (block, "file data")) break;     

     if (fwrite (DataBlock.data, DataBlock.size, 1, file) != 1)
        printf ("*** Can't write data block %ld from disk block %ld\n",
        DataBlock.seqnum, block);
     blocks = blocks-1;
   }

   if (block==0) { printf ("File %s is complete\n\n", name); }
   else printf ("File %s has been truncated\n\n", name);

   fclose (file);
}

StrComp(str1, str2)
char *str1, *str2;
{
   int index = 0;

   while (1)
   {
      if (str1[index] == 0 && str2[index] == 0) return(0);
      if (toupper(str1[index]) != toupper(str2[index])) return(-1);
      index++;
   }
}

main(argc, argv)
int argc;
char **argv;   
{
   LONG block;
   char *lookedfor;

   if (argc == 1)
   {
      printf ("Usage: Undelete <filename>\n\n");
      exit(0);
   }

   lookedfor = argv[1];

   diskdata = &DataBlock;       /* point to first location in disk buffer */
   if ((diskport = CreatePort(0,0)) == 0) exit(100);    /* error */

   /* make an io request block for communicating with the disk */
   diskreq = (struct IOExtTD *)CreateExtIO(diskport,sizeof(struct IOExtTD));

   if(diskreq == 0) { DeletePort(diskport); exit(200); }

   /* open the device for access, unit 0 is builtin drive */
   if (0 != OpenDevice(TD_NAME,0,diskreq,0)) exit (150);

   printf ("Put disk with deleted file in INTERNAL drive.\n");
   printf ("(Cancel the requester if the disk is unreadable.)\n");
   printf ("Put empty disk in EXTERNAL drive.\n");
   printf ("Press RETURN to begin.\n");

   getchar();

   /* now get the disk change value */
   diskreq->iotd_Req.io_Command = TD_CHANGENUM;
   DoIO(diskreq);
   diskChangeCount = diskreq->iotd_Req.io_Actual;

   printf ("Scanning disk for %s...\n", lookedfor);

   for (block = 0; block < 1760; block++)
   {
      if (ReadBlock (block, "information") && DataBlock.type == TypeDATA)
      {
         blocks += 1;
         if (DataBlock.seqnum == 1)
	 {
	    if (ReadBlock(DataBlock.key, "file header"))
	    {
	       DataBlock.data [FILENAME + DataBlock.data [NAMELENGTH]] = 0;
	       sprintf (name, "%s", &DataBlock.data [FILENAME]);
	    }
	    else
	       name[1] = 0;
	    printf ("Name = %s%cK\r", name, 0x9b);
	    if (StrComp(name, lookedfor) == 0) CopyFile (block);
	 }
      }
   }

   printf ("\n\nAll disk blocks have been scanned.\n");
   MotorOff();
   CloseDevice(diskreq);
   DeleteExtIO(diskreq, sizeof(struct IOExtTD));
   DeletePort(diskport);
   exit(0);
}
