/*
BEGIN DOCUMENTATION

Name: Daniel T. Soldahl				Created: 11/03/83
					    Last Update: 03/20/84
Title: TCMFILE.C: Read/Write Text Window to Disk

Index:

Abstract:


Usage:	rc = tcmget(filespec,fptr,bufptr,sizptr);
	rc = tcmput(filespec,fptr,bufptr,lastline);
	rc = tcmcls(fptr);

Parameters:
    INPUT:
	char	*filespec;	C string of standard RSX filespec
	FILE	**fptr;		Pointer to global (static) file for this window
	char	*bufptr;	Pointer to file I/O buffer (text window)
	int	lastline;	Number of lines to write to file
	int	*sizptr;	Maximum number of characters to input
    OUTPUT:
	int	rc;		Return code for operation
	int	sizptr;		Actual number of characters input

Environment: RSX11M V4.0, DECUS C Compiler

See Also: TCM.C

Description:	Read or write TCM text buffer to RSX file system.  This 
	    implementation uses standard (FCS) sequential files.  
	    Tcmget opens, reads, then closes the file.  Tcmput opens, 
	    writes, then closes the file.  An external file pointer is
	    used so that later tcmget can open without closing and 
	    tcmput can write without opening.  Tcmcls will then
	    be used by TWINDE to close the file.

Example(s):

Uses:

Internal:	Each line of the text buffer is assumed to be a C 
	    string which corresponds to one variable record in the 
	    file.  A null string is used to mark the end of the 
	    buffer.

Update History:

END DOCUMENTATION
*/


#include <clang.c>
#include <stdio.h>
#include <cx.h>
#include "tcmdefs.h"
#include "tcmerr.h"

/* #define debug 0 */

/***	TCMGET: Read text buffer from file    ***/

int
tcmget(filespec,fptr,bufptr,bufsiz)
    charpointer filespec,bufptr;
    FILE **fptr;
    int *bufsiz;
{
    register int i,blen;
    register charpointer s;
    int rc;

	iff (*fptr = fopen(filespec,"r")) == NULL then
	    rc = TE_INF;
	else
	{	/* read file into buffer */
	    blen = *bufsiz;
	    s = bufptr;
	    while((blen > 1) and (fgets(s,blen,*fptr) != NULL))
	    {
		i = strlen(s);
		blen -= i;
#ifdef debug
		printf("%d bytes in buffer,line at %o = '%s'\n",blen,s,s);
#endif
		s += i;
	    }
	    iff ferror(*fptr) == 1 then
		rc = TE_FIO;
	    else
		rc = TS_SUC;
	    fclose(*fptr);
	    *bufsiz -= blen;
	}
	*fptr = NULL;		/* Done with this file */
	return(rc);
} /* end tcmget */


/***	TCMPUT: Write text buffer to file    ***/

int
tcmput(filespec,fptr,bufptr,lastline)
    charpointer filespec,bufptr;
    FILE **fptr;
    int lastline;
{
    register int c,count;
    register charpointer rline;
    int rc;

	iff (*fptr = fopen(filespec,"w")) == NULL then
	    rc = TE_OTF;
	else
	{
	    rc = TS_SUC;
	    for(c = 1,rline = bufptr; c <= lastline; c++) /* and (rline < bufend) */
	    {
		count = strlen(rline);
#ifdef debug
		printf("%d byte line at %o = '%s'\n",count,rline,rline);
#endif
		fput(rline,count,*fptr);
		iff ferror(*fptr) == 1 then
		{
		    rc = TE_FIO;
		    tcmerr(rc);
		    break;
		}
		rline += ++count;
	    }
	    fclose(*fptr);
	    *fptr = NULL;
	}
	return(rc);
} /* end tcmput */



/***	TCMCLS: Close previously opened file (now does nothing)    ***/

tcmcls(fptr)
    FILE **fptr;
{
	iff *fptr != NULL then	/* should alway = NULL */
	    fclose(*fptr);
} /* end tcmcls */
