/*
 * lc [-1dlptv] [dir ...]
 */
#include <stdio.h>
#include <hdr.h>

#define	FMAX	100

/*
 * This is the format of an
 * RSX directory entry.
 */
struct	dirent
{
	int	d_fnum;
	int	d_fseq;
	int	d_gok;
	int	d_name[4];
	int	d_ver;
};

/*
 * File table entry.
 * Saves information from the
 * directory.
 */
struct	file
{
	int	f_fnum;
	int	f_fseq;
	char	f_name[12];
	int	f_ver;
};

int	dflag	0;
int	lflag	0;
int	oflag	0;
int	pflag	0;
int	tflag	0;
int	vflag	0;
struct	hdr	ib;
int	nfile;
int	talloc	0;
int	tused	0;
struct	file	file[FMAX];

main(argc, argv)
char *argv[];
{
	register char *p;
	register int c, i;
	int nf;

	nf = argc-1;
	for(i=1; i<argc; ++i) {
		p = argv[i];
		if(*p == '-') {
			++p;
			while(c = *p++)
				switch(c) {

				case 'l':
				case 'L':
					++lflag;
					break;

				case '1':
					++oflag;
					break;

				case 'v':
				case 'V':
					++vflag;
					break;

				case 'd':
				case 'D':
					++dflag;
					break;

				case 'p':
				case 'P':
					++pflag;
					break;

				case 't':
				case 'T':
					++tflag;
					break;

				default:
					usage();
				}
			argv[i] = 0;
			--nf;
		}
	}
	if(nf == 0)
		list("", 0);
	else
		for(i=1; i<argc; ++i)
			if(p = argv[i])
				list(p, nf!=1);
}

usage()
{
	fprintf(stderr, "Usage: lc [-1dlptv] [dir dir ...]\n");
	exit(1);
}

list(s, b)
char *s;
{
	if(!collect(s) || nfile==0)
		return;
	if(!dflag)
		sort();
	if(b)
		printf("\n%s:\n\n", s);
	output();
}

collect(s)
char *s;
{
	register struct file *p;
	register FILE *dir;
	struct dirent db;
	char dn[40];

	if(!dirname(dn, s)) {
		err(s, "bad format");
		return(0);
	}
	if((dir=fopen(dn, "ru")) == NULL) {
		err(s, "cannot list");
		return(0);
	}
	p = file;
	while(fget(&db, sizeof(db), dir) == sizeof(db)) {
		if(db.d_fnum == 0)
			continue;
		if(p >= &file[FMAX]) {
			err(s, "too many files");
			break;
		}
		p->f_fnum = db.d_fnum;
		p->f_fseq = db.d_fseq;
		r50toa(p->f_name, db.d_name, 4);
		p->f_ver = db.d_ver;
		++p;
	}
	fclose(dir);
	nfile = p-file;
	return(1);
}

dirname(db, s)
char *db;
register char *s;
{
	register char *d;
	register int c;
	int uic, dev, u, g;
	char dn[60];

	dev = 0;
	uic = 0;
	u = getuid();
	g = getgid();
	while(*s) {
		if(*s == '[') {
			if(uic)
				return(0);
			++uic;
			++s;
			g = 0;
			while((c = *s++) && c>='0' && c<='7')
				g = (g<<3) + c - '0';
			if(c != ',')
				return(0);
			u = 0;
			while((c = *s++) && c>='0' && c<='7')
				u = (u<<3) + c - '0';
			if(c != ']')
				return(0);
		} else {
			if(dev)
				return(0);
			++dev;
			d = s;
			while((c = *s++) && c!=':')
				;
			if(c != ':')
				return(0);
		}
	}
	s = db;
	if(dev)
		while((*s++ = *d++) != ':')
			;
	d = "[0,0]";
	while(*s++ = *d++)
		;
	--s;
	sprintf(dn, "%03o%03o.dir", g, u);
	d = dn;
	while(*s++ = *d++)
		;
	return(1);
}

err(s, m)
char *s, *m;
{
	if(*s)
		fprintf(stderr, "%s: %s\n", s, m);
	else
		fprintf(stderr, "%c%s\n", *m&~' ', m+1);
}

sort()
{
	register struct file *b, *m, *p;
	struct file tf;

	for(b=&file[0]; b<&file[nfile-1]; ++b) {
		m = b;
		for(p=b+1; p<&file[nfile]; ++p)
			if(cmp(m->f_name, p->f_name))
				m = p;
		if(m != b) {
			copy(&tf, m, sizeof(tf));
			copy(m,   b, sizeof(tf));
			copy(b, &tf, sizeof(tf));
		}
	}
}

cmp(a, b)
register char *a, *b;
{
	register int n;
	int ca, cb;

	n = 12;
	while(n-- && ((ca=*a++)==(cb=*b++)))
		if(ca == '\0')
			break;
	return(ca > cb);
}

output(nf)
{
	register struct file *p;
	register int c, n;
	int w;

	w = 1;
	if(!oflag && !lflag) {
		printf("Files:\n\n");
		w = 4;
	}
	p = file;
	n = 0;
	while(nfile--) {
		if(n >= w) {
			putchar('\n');
			n = 0;
		}
		c = outname(p);
		++n;
		if(lflag || n<w)
			while(c++ < 18)
				putchar(' ');
		if(lflag)
			longinfo(p);
		++p;
	}
	if(n)
		putchar('\n');
	if(lflag && tflag) {
		if(tused > talloc) /* Bug in TKB */
			tused = talloc;
		printf("\nTotal allocated: %6d\n", talloc);
		printf("Total used:      %6d\n", tused);
	}
}

outname(p)
register struct file *p;
{
	register char *s;
	register int c;
	char vb[8];

	c = 0;
	s = p->f_name;
	while(s<&p->f_name[9] && *s!=' ') {
		putchar(*s++);
		++c;
	}
	s = &p->f_name[9];
	if(*s != ' ') {
		putchar('.');
		++c;
		while(s<&p->f_name[12] && *s!=' ') {
			putchar(*s++);
			++c;
		}
	}
	if(c == 0) {
		putchar('.');
		++c;
	}
	if(vflag) {
		sprintf(vb, ";%d", p->f_ver);
		s = vb;
		while(*s) {
			putchar(*s++);
			++c;
		}
	}
	return(c);
}

longinfo(p)
struct file *p;
{
	register char *d, *t;
	register int n;

	if((n=gethdr(&ib, p->f_fnum, p->f_fseq)) < 0) {
		printf("Error (%d) reading file attributes", n);
		return;
	}
	printf("%4d%4d ", ib.h_ufat.f_efbk[1], ib.h_ufat.f_hibk[1]);
	talloc += ib.h_ufat.f_hibk[1];
	tused += ib.h_ufat.f_efbk[1];
	printf("%c%c", ib.h_ucha&UC_CON?'c':' ', ib.h_ucha&UC_DLK?'l':' ');
	if(pflag) {
		printf(" [");
		for(n=0; n<16; ++n) {
			if(n==4 || n==8 || n==12)
				putchar(',');
			if((ib.h_fpro&01) == 0)
				putchar("rwed"[n&03]);
			else
				putchar('-');
			ib.h_fpro >>= 1;
		}
		putchar(']');
	}
	if(ib.i_rvdt[0]) {
		d = &ib.i_rvdt;
		t = &ib.i_rvti;
	} else {
		d = &ib.i_crdt;
		t = &ib.i_crti;
	}
	printf(" %.2s:%.2s", t, t+2);
	printf(" %.2s-%c%c%c-%.2s", d, d[2], d[3]|' ', d[4]|' ', d+5);
}
