/* A source program: scan.c in the C Language (ANSI).  Tabs set at 8.

	SOFTFONT GENERATOR: ENCODES a BITMAP DRAWING to SOFTFONT
			Version: 3.1  Mar. 1997
			(c) by David S. Lawyer

pat2sf Converts a drawing of a bitmap (a pattern file) into the code
required for downloading (soft-font).  This (scan.c) is the main file
for pat2sf.

Copyright (c) by David S. Lawyer, 1990-7 and may be freely used,
copied, and modified by anyone provided that any significant
modifications or improvements be also made freely available.  If sold,
labels or documentation must state or imply that this is free
software.  The author reserves the right to transfer this copyright to
any organization which will freely distribute it (such as the Free
Software Foundation).  Do not remove this copyright notice.

Thanks to Victor Poon for his Wyse60 font programs which provided some
of the ideas used in this program.

For more details about using this program see the file: pat2sf.doc.
Source code files: scan.c (this, contains main()), encode.c (lang. 
specific), & pat2sf.h   For a guide to this code see pat2sf.des

The word character, (or char, ch, etc.) has two possible distinct meanings.
The normal meaning is a byte and starts with a lower case c.  The other
meaning is that of a bit-map or Character matrix (a Character to be
encoded) and will begin with a capital C.  Thus the * that you type in
a drawing is a ch while the drawing of a Character (using many *'s)
is a Char.

Input is a bitmap "pattern-file" containing rows of Character patterns
with pixels represented by *'s.  Look at files named *.pat for examples.
A Character shape is sometimes called a "glyph" so a pattern is a kind
of glyph.  See encode.how to see how a character is encoded into soft-font
This input file may be used as standard input (use -s flag) or the
input file may be given as an argument. The latter permits some
interactive response to error messages.

Output of the softfont file is via stdout and thus all messages which
appear on the screen get there via stderr.  To run this program (after
you compile it) create a pattern_file named say my_patterns
by using any convenient editor.  Then use my_patterns as input to this
program thereby obtaining the soft-font code in your output file named
say my_softfont.  To download this font you could attempt to copy
my_softfont to a terminal or printer but in most cases you will want
to use a * shell script * (= batch file).  This script may set up the
device correctly, assign the font a name or bank number, etc. and then
download the soft-font.

The coding rules used in this program for Wyse and VT220 terminals came
from the Wyse 99GT Programmer's Guide chapters 7 and 8.

INPUT PATTERN FILE FORMAT

These are the "drawings" of many Characters.  The file begins with
optional  comment lines, each starting with #.  The next two lines are
file-header lines for the entire file.  The first must contain three
integers: cell width, cell height, and Character-matrices per band.
Index origin is 1.  They may be separated by any other characters such
as spaces, words, etc.  The second line should be an important
comment.  Both of these header lines will be displayed on the CRT each
time you generate softfont.  These 2 lines are also put at the end of
the softfont file of code.  

Then comes bands ( =thick-rows) of dot-matrix Characters (= "* patterns"
or Character-matrices).  Each such band must be preceded by a horizontal
"band header line".  These horizontal header lines should contain the
Character numbers but may contain anything (including a blank line).
Each pixel is represented by a star: * but the background (= blank pixels)
may be represented by either dots or blank spaces.  Don't try to mix these
two formats in the same pattern file.  Each dot-matrix Character is
separated from the adjacent one to its right (or left) by one or more 
vertical "columns" of: 
1. If background is dots: spaces and/or tabs
2. If background is spaces: vertical bars: |
*/ 
/*---------------------------preprocessor stuff-------------------------------*/
#include <stdio.h>   /* for some C compliers you should: #include <stdlib.h>  */
#include <getopt.h>
#include "pat2sf.h"    /* in this directory */

/* CELL is a rectangle which contains a Character dot matrix (= bitmap).  
 * Sizes shown for CELLs are in number of pixels (= dots).
 * ERROR (or "error") is a mistake in the format of the input pattern file.
 */
#define CELL_WIDTH_MIN	   5    /* If smaller, assumes typo error & quits     */
#define CELL_HEIGHT_MIN    6	/* If smaller, assumes typo error & quits     */
#define LINE_LENGTH	  80	/* Max. length of pattern file lines.	      */
#define MAX_ERRORS	   2 	/* If using stdin,exits if no. of errors>this.*/
#define DOT '.'
#define SPACE ' '
#define ENCODE   0		/* flag for encodeChP()			      */
#define SETOPTS  1		/* flag for encodeChP()			      */
/*----------------------external vars & functs--------------------------------*/
extern void ListLangs();/* In encode.c. Lists codes for font langs	      */
/*--------------------------type declarations---------------------------------*/
typedef enum { blank_pix, eof, eol, n_pix, scan } err_type;
typedef enum { false = 0, true = 1 } boolean;
/*--------------------forward declarations------------------------------------*/
				/* FUNCTIONS in this FILE  (except main()     */
void get_BB(void); 		/* gets Bounding Box dimensions.	      */
void fill_band(void);		/* Fills band[][][] with * pixels	      */
void eof_err();			/* fscanf ret -1. Unexpected EOF (End Of File)*/
void eol_err(char);		/*\n expected.  Junk at eol?  Line too long?  */
void n_pix_err(int );  		/* wrong Number of PIXels in a scan of Ch row */
void scan_err();  		/* fscanf ret. 0. No legal pixels left to scan*/
void err_show_and_tell(int, err_type);	/* Prints bad lines. Called by err f's*/
void err_ask(err_type);		/* User interaction.      "    "    "     "   */
void err_exit();		/* prints std. error message and exits        */
void usage(int);		/* Shows user how to type cmnd line & options */
/*---------------------global variables---------------------------------------*/
boolean 		/* Bounding Box (BB) vars:			      */
    blank_BB = false,	/* if true, Char is blank (like a space)	      */
    need_BB;		/* do we need to calculate BB vars.? for this lang.?  */
int
    BBx_min = -1,	/* min x dist. to top of Character		      */
    BBx_max,		/* max x dist. to bottom of Character		      */
    BBy_min,		/* min y dist. to left side of Character	      */
    BBy_max,		/* max y dist. to right side of Character	      */

    Cell_height,	/* index-origin = 0.  =Cell_Height--  Calculated      */
    Cell_width,		/* index-origin = 0.  =Cell_Width--   Calculated      */
    Pad_level = 0,	/* # of nulls to pad each line.  Command line option  */
    Chars_Per_Band,	/* # of Character-dot-matrices per "large row" (band) */
    nChars_coded = 0,   /* How many dot-matrices Chars. have we encoded?      */
    Char_no;		/* in 0..Chars_Per_Band - 1; Seq. no. of Char_matrix  */
			/* Used as index into band[Char_no][][] in fill_band()*/
unsigned char		/* Pixels of data for a band[][][] of Chars.	      */
    band[CHARS_PER_BAND_MAX] [CELL_HEIGHT_MAX] [CELL_WIDTH_MAX + 1];
unsigned char const
    (* Char_matrix)[CELL_WIDTH_MAX + 1];	/* See long explanation below */
char    
    blank_pixel,		/* =space ' ' or dot '.' in pattern file.     */
    ch;				/* a test byte-char read.		      */

/* Char_matrix[][] = band [Char_no] is a matrix slice of band[][][]
 * containing the *-pattern for one Character.  Char_matrix is a true pointer
 * and unlike an array, it can be assigned to point to any suitable matrix.
 * The last dimension: [Cell_Width_Max +1], must be given in the pointer decl.
 * since it must know how to do pointer arithmetic with its first index
 * to locate rows.  Char_matrix may be indexed just like an array but it
 * is declared as a pointer to a one dim. array (a row).  The main decl.
 * is in encode.c.  It's used for "calling" various ?encodeCh() functions
 * in encode.c using a global var (Char_matrix) that only needs to be declared
 * once in encode.c.  It's similar to passing a function as a parameter in
 * a function call.  Elementary books on C don't explain how to do the 
 * equivalent using global arrays.  Neither the pointer: Char_matrix nor
 * elements of its matrix are constant, but * Char_matrix is since it does
 * not represent a variable and nothing may be assigned to it. 
 */
/*---------------------static file variables----------------------------------*/
static int
    i = 0, j = 0,	/* General purpose indicies.			      */
    row, col,		/* row or col no. of band[][][] or Char_matrix[][]    */
    Last_Ch_No,		/* Index limit on scanning.  Normally = Chars_Per_Band*/
			/* unless last band short. index_o.=1. fill_band uses.*/
    Cell_Height,	/* = index-origin = 1  Cell_height++ Read from header */
    Cell_Width,		/* = index-origin = 1.  Cell_width++ Read from header */
    band_no = 0,	/* What band is being parsed now?   Error functs need.*/
    line_no = 0,	/* line number now being parsed in pat file. "    "   */
    error_count = 0;	/* How many errors found thus far in pat. file?       */
static boolean
    std_in = false,	/* program uses stdin for pat file?  Used by err_ask  */
    error  = false;	/* Did a call of fscanf() results in a format error?  */
			/* Set false in fill_band() and true in err_ask()     */
static FILE
    *fp;					/* pointer to input file      */
static long int
    foffset[MAX_LINES +1];	/* Offset of a line_no into fp file.  Used    */
static void 			/* in error messages to show bad row.	      */
(* usageP)() = NULL; 	/* ptr to funct which prints usage for a given lang   */
/***************************** start of main **********************************/
main(int argc, char **argv) {
char
	Font_info[160],		/* for storing font lang, options, cell size  */
	header_1[LINE_LENGTH + 3],	    /* for storing 1st header line    */
	header_2[LINE_LENGTH + 3],	    /* for storing 2nd header line    */
	header[LINE_LENGTH + 3],	    /* for storing a band header line */
	*lang = NULL,	/* lang of terminal or printer.  NULL is undefined    */
	*ptr_1,		/* for extracting data (cell size, etc.) from header_1*/
	*lang_opts;	/* Points to opts for a specific lang of font.  Added */
			/* to Font_info later on.			      */
int	
	nscan;		/* # of items of data assigned by fscanf.  Should = 3 */
long    f_offset;	/* offset of pointer from start of pat. file	      */
boolean
	caseH = false,	/* flag to mark --help option			      */
	caseV = false,	/* flag to mark --version option		      */
	info_missing = false;	/* Is data missing from the 1st header line ? */
char * (* encodeChP) (); /* ptr to funct which encodes a Char from band[][][] */
void (* addEndP) ();     /*ptr to funct. Adds a trailer to the entire softfont*/
void (* sizeChkP) ();    /*ptr to funct which checks if cell size sufficient  */
/*-----------------------parse command line-----------------------------------*/
/* Options that are used for only for one lang of font are used as parameters
 * (via argv) to the call of encodeCh() defined in encode.c.  Then getopt
 * is used again there to parse the rest of the command line.
 */
static struct option long_opts[] = {
{"stdin", 0, 0, 's'}, 			{"pad_level", 1, 0, 'p'},
{"lang", 1, 0, 'L'},			{"help", 1, 0, 'h'},
{"version", 0, 0, 'v'},			{0, 0, 0, 0}	/* Zeros req'd per man*/
};
if  ( argc == 1 ) 	usage(EXIT);		/* No args given.  Wants help */
while (( ch = getopt_long (argc, argv, "+L:h:p:sv", long_opts, &j))
								!= EOF){
    i++;					/* for Help count	      */
    switch (ch) {
	case  0 : break;			/* long option w/non-null flag*/
	case 'h': caseH = true;			/* Can't use other opts.      */
		  lang = optarg;		/* font lang or "list"	      */
		  i--;				/* should == 0		      */
		  break;
	case 'p': Pad_level = atoi(optarg);
		  break;
	case 's': fp =  stdin;
		  fprintf(stderr, "Using standard input.\n");
		  std_in = true;
		  break;
	case 'v': printf("pat2sf, VERSION \n"); 
		  caseV = true;
		  i--;
		  break;
	case '?': 		/* getopt will print the illegal option used  */
		  fprintf(stderr, "ERROR: The above option might possibly"
			" be legal but out of place.\n"); 
		  usage(ERR_EXIT);
	case ':':				/* broken in gcc compiler     */
		  fprintf(stderr, "ERROR: You failed to give a parameter to"
		     " an option which requires it.\n");	
		  usage(ERR_EXIT);
	default : fprintf(stderr, "ERROR: Program error which should never"
 			"happen.  getopt_long() returned %c.\n", ch);  
      }
  }
/*-------------------------------caseH (help)---------------------------------*/
if (caseH || caseV)
    if (i != 0  || optind < argc )
	fprintf(stderr, "If you ask for help or version, don't use"
		"other options or try to run the program. \n" );
if (caseH) {
    if ( ! lang) usage(EXIT);			/* No args to help request    */
    if (! strcmp(lang, "list") ) {	/* Show list of valid language names. */
	ListLangs(EXIT);			/* Found in encode.c 	      */
     }
 }				
  else { /* not caseH */
/*------------------------------get & test lang-------------------------------*/
    if (caseV) exit(0);		/* if caseV & caseH then caseH controls exit  */
    lang = argv[optind++]; 	/* lang must exist;  either par. of -h or here*/
   }
if ( lang == NULL ) {		/*  above read 0 at end argv[]		      */
    fputs("You failed to specify a font language. \n", stderr);
    ListLangs(ERR_EXIT);
  }
/*----------------------------set funct ptrs----------------------------------*/
Set_Fptrs(&usageP, &addEndP, &encodeChP, &sizeChkP, lang);
if (caseH)				/* above errors if lang invalid	      */
    usage(EXIT);			/* also prints usageP (just set),     */
/*-----------------------------get/set lang opts------------------------------*/
/* Will set lang-specific options in actual (selected) encode funct as 
 * static vars.  argv[optind] should point to 1st lang option in selected
 * encodeCh funct. in encode.c.  lang_opts will be put into font_info.
 */
lang_opts = (* encodeChP) (argc, argv, SETOPTS); 
/*-----------------------------open pattern file------------------------------*/
if (! std_in) { 
    if ( (fp = fopen( argv[optind], "r")) == (FILE*) NULL) {
	fprintf(stderr, "ERROR: Can't open input pattern_file: %s\n"
	    "Check spelling or maybe your format of your command line is"
	    "wrong. \n", argv[optind]);
	exit (-1);
      }				
  }
/*-----------------------garbage at end of cmd line?--------------------------*/
if (++optind < argc) {			/* didn't ++ above since used for err.*/
    fputs("ERROR: The name of the pattern_file should be the last"
	" item on your command \n line.  The following words/symbols are"
	" either illegal or out-of-place:\n", stderr);
    while ( optind < argc)	/* optind == argc when argv[optind] == (null) */
	fprintf(stderr, "%s ", argv[optind++] );
    fputs("\n", stderr);
    exit (-1);
 }
/*---------------------parse & print header lines-----------------------------*/
/* Seek past comment lines starting with # and get 1st header.		      */
foffset[ ++line_no ]  =  ftell(fp); 
while( ( ch = fgetc(fp) , ch == '#') ) {
    fscanf( fp, "%*[^\n\r]");  		/* skip over comment lines	      */
    fscanf( fp, "%*[\n\r]"); 		/* skip over \n	\r		      */
    foffset[ ++line_no ]  =  ftell(fp); 
 }
ungetc( ch, fp);			/* put back 1st ch of header_1	      */
for(i = 1; i <= 3; i++) {		/* Get 3 parameters out of header_1.  */
    fscanf( fp, "%*[^0-9]");	/* Skip over any leading non-digits.  */
    nscan = fscanf( fp, "%d", &j);	/* put i th parameter into j	      */
    if (nscan == 1)
	switch(i) {
	    case 1: Cell_Width   =   j;	break;
	    case 2: Cell_Height  =   j;	break;
	    case 3: Chars_Per_Band = j;	break;
	 }
    else
	info_missing = true;	/*Will print err msg later after printing info*/
	fscanf( fp, "%*[0-9]");	/* Skip over digits.	      */
 }
Cell_width  = Cell_Width - 1;			/* Index origin = 0	      */
Cell_height = Cell_Height - 1;			/* Index origin = 0	      */
fseek(fp, foffset[ line_no ], 0);	/* Restore pos. of fp to rescan line  */
fgets( header_1, sizeof(header_1), fp);		/* Read first header line.    */
foffset[ ++line_no ]  =  ftell(fp); 
						
/* The first non-comment line of the source file must look something like: 
 * 7x12 cell.  8 Chars/band ....   => Cell_Width = 7    Cell_Height = 12
 * The second header line of the pattern input file is just a brief description
 * of the font.  Both lines: header_1 and header_2 will be printed to the 
 * terminal at the start and end of font generating.  They will also be put
 * at the end of the softfont (output) file.
 */
fputs( header_1, stderr);			/* Echo it to your terminal.  */
fgets( header_2, sizeof(header_2), fp);		/* Read second header line.   */
foffset[ ++line_no ]  =  ftell(fp); 
fputs( header_2, stderr);			/* Echo it to your terminal.  */
/*-----------------------pack & print Font_info-------------------------------*/
sprintf(Font_info, "%s\nPad_level = %d nulls/line.  Cell size: %dx%d (width x\
  height) Chars/band = %d.\n",
  lang_opts, Pad_level, Cell_Width, Cell_Height, Chars_Per_Band);
fputs(Font_info, stderr);
/*------------------------check for bad size data-----------------------------*/
if (info_missing) {
    fputs("\n###ERROR### in last 3 parameters above.  Didn't find all 3\
 parameters in first\nheader line  of pattern file.  The Cell height, width and\
 Chars/band must \n all be there.\n",
	stderr);
    exit (-1);
 }
(* sizeChkP)(Cell_Height, Cell_Width); /* Is size of height/width OK for lang?*/

if( Cell_Height > CELL_HEIGHT_MAX   ||   Cell_Width > CELL_WIDTH_MAX ) {
    fprintf(stderr, "\n###ERROR### Cell size is too large.  Fix typo or modify\
 #define in pat2sf.h. \nThe current #define sets: Max cell height = %d,\
 Max cell width = %d \n", CELL_HEIGHT_MAX, CELL_WIDTH_MAX);
    exit(-1);
 }
if( Cell_Height < CELL_HEIGHT_MIN   ||   Cell_Width < CELL_WIDTH_MIN ) {
    fprintf(stderr, "\n###ERROR### Cell size is too small.  Fix typo or modify\
 #define in pat2sf.h. \nThe current #define sets: Min cell height = %d,\
 Min cell width = %d \n", CELL_HEIGHT_MIN, CELL_WIDTH_MIN );
    exit(-1);
 }
if (Chars_Per_Band > CHARS_PER_BAND_MAX) {
    fprintf(stderr, "\n###ERROR### Chars/band is too large.  Fix typo or modify\
 #define in pat2sf.h. \nThe current #define sets: Max Chars/band = %d \n",
		CHARS_PER_BAND_MAX);
    exit(-1);
 }
/*---------------------------get blank_pixel type-----------------------------*/
f_offset = ftell(fp);			/* Save offset for future restoration.*/
fscanf(fp, "%*[^\n]\n");		/* Skip over band header line.	      */
fscanf(fp, "%*[ \t*^\n]");		/* Skip over legal chs. except | or . */
ch = fgetc(fp);
switch (ch) {
  case '|':	blank_pixel = SPACE;		break;
  case '.':	blank_pixel = DOT;		break;
  default:	err_show_and_tell (7, blank_pix);
 }
fseek(fp, f_offset, 0);			/* Restore pointer to offset loc.     */
fprintf(stderr, "Cell size is for patten file cells.  Actual cells are larger."
    " Blank_pixel = '%c'\n", blank_pixel);
fputs("Is the above correct???  If not, your pattern file or command line are\
 wrong!\n" , stderr);			/*"above" => the last few fput/fprintf*/
/*--------------------------fill & encode band--------------------------------*/
fputs("Start of encoding font. Below are band header lines from your pattern"
    "file:\n", stderr);
while (true) {				/* exits per break below	      */
    if ( fgets(header, sizeof(header), fp) == (char*) NULL) /*Get header line */
	break;				/* break out of while loop. File ended*/
    fputs(header, stderr);		/* print header line of band	      */
    band_no ++ ;			/* for error messages		      */
    Last_Ch_No = Chars_Per_Band;	/* fill_band()may reduce if band short*/
    fill_band();			/* Reads pattern into band[][][] array*/
    foffset[ ++line_no ]  =  ftell(fp); /* Should be at start of next header  */
    if (error_count == 0)       	/* Don't encode Chars if errors	      */
				/* We encode one Ch at a time from this band. */
				/* Index origins: Last_Ch_No:0, Char_no:1     */
				/* Last_Ch_No may change during iteration.    */
	for (Char_no = 0; Char_no < Last_Ch_No; Char_no ++) {
	    Char_matrix = band[ Char_no ];     /* Passed to encode as a global*/
	    if (need_BB)
		get_BB();		    /* get bounding box of Char_matrix*/
	    (* encodeChP)(argc, argv, ENCODE);
	    for (i = 1; i <= Pad_level; i++)	/* Append padding nulls       */
		putchar('\0');
	    nChars_coded ++;		
	 }
 }		
/*------------------------print ending info-----------------------------------*/
if (error_count > 0) {		/* Don't say "finished" if output is garbage. */
    fputs("Found no more errors but more may exist.  Correct the above errors "
          "and try \nagain.  The output softfont file is likely garbage.\n",
	   stderr);
    exit(-1);
 }
(* addEndP)();				/* add ending. Will ensure comments OK*/
fputs(header_1, stdout);		/* put as ASCI comments at end of sf  */
fputs(header_2, stdout);
fprintf(stdout, "%sEncoded %d Characters.\n", Font_info, nChars_coded);
		
fprintf( stderr, "FINISHED!   Encoded shape of %d Characters.\n%s",
	nChars_coded, Font_info);
fputs(header_1, stderr);		/* stderr is CRT when encoding done   */
fputs(header_2, stderr);

/* In many cases, the above info put on CRT at end of encoding is not needed 
 * since the same info was given at the start of encoding and is still visible
 * on the CRT.  But there are a some cases where the initial info lines have
 * scrolled off the CRT.  */
}
/*----------------------------end of main-------------------------------------*/
/*******************************get_BB()***************************************/
/*		get the Bounding Box = min. rectangle enclosing Ch.
* Finds the values required to calculate the height and widths of a character
* inside a cell.  First scans across rows starting at the top row until the
* first * is hit in the top row of the character.  BBx_min is the # of blank
* rows on top (= row# of first row, io=0).  Then it does the same starting
* from the bottom finding BBx_max = the last row, io=0.  Then we repeat as
* above, but scans along columns.
*/
void get_BB(void) {
for (row = 0; row <= Cell_height; row++)	
    for (col = 0; col <= Cell_width; col++) 
	if ( Char_matrix [row][col] == '*') {
	    BBx_min = row;
	    goto EndLoop1;
	  }      EndLoop1:
if ( BBx_min == -1 ) {		/* above iteration never found a * => a blank */
    blank_BB = true;
    return;
  }
/* Since the Char_matrix in not blank, we must eventually find a * which
 * will stop the iteration via a break.  Thus no need for an exit 
 * conditional on the outer "for".
 */
for (row = Cell_height; ; row--)
    for (col = 0; col <= Cell_width; col++) 
	if ( Char_matrix [row][col] == '*') {
	    BBx_max = row;
	    goto EndLoop2;
	  }      EndLoop2:
/* Once we find the top and bottom rows of the Ch. then no need to iterate
 * over bottom and top blank rows.  We only go bet. BBx_min and BBx_max.
 */
for (col = 0; ; col++) 
    for (row = BBx_min; row <= BBx_max; row++)
	if ( Char_matrix [row][col] == '*') {
	    BBx_min = col;
	    goto EndLoop3;
	  }      EndLoop3:
for (col = Cell_width; ; col--)
    for (row = BBx_min; row <= BBx_max; row++)
	if ( Char_matrix [row][col] == '*') {
	    BBx_max = col;
	    goto EndLoop4;
	  }      EndLoop4:
}
/*--------------------------end get_BB()--------------------------------------*/
/**************************** fill_band() *************************************/
/*
* A band of Char-matrices is read from the pattern file pointed to by fp. 
* The pixels read (. or * or <space>) are put into the band array.
* This function obtains all its parameters via global variables.  It  
* copies a "tall row" of Character glyphs from the pattern file into the 
* band[][][]  array.  This is done by copying several ascii rows (= a band)
* from the pattern file into the band[][][] array.  Only (null terminated)
* pixels strings go into the band[][][].  Separator chars. are left out.
* The null terminators are not needed in the band[][][] but fscanf appends
* them and not using the scan functions would be somewhat more complicated.
* Extensive format error checking is done here and error functions are called
* to print error messages as needed.
*
* Note that fscanf(fp, "%*[A]%[B]", ... ) will exit if the A match fails
* and B will never get scanned.  This requires special treatment for
* Char_no==0 where there may be no initial separator characters before the
* first Char. matrix.  Thus for Char_no==0 there are 2@ fscanf()'s so that
* the failure of the first skipping one doesn't cause the 2nd to fail.
* An illegal pixel (such as a #) after a valid pixel will cause fscanf to 
* exit and return 1 (=> AOK).  But the count n_pix2 - n_pix1 should be
* wrong.  Premature end of line also returns 0.
*/
void
fill_band(void) {
int	n_scan,			/* No. of strings copied by fscanf(). Expect=1*/
	n_pix1, n_pix2,		/* No. of pixels copied by fscanf() thus far. */
	n_pixels_read;		/* n_pix2 - n_pix1. no. pixels read by fscanf */
char	* ptr;			/* for pointing to band[][][]		      */
for (row = 0; row <= Cell_height; row ++) {	/* Each row has several Chars.*/
    foffset[ ++line_no] = ftell(fp);	/* End of this line & start of new one*/
    for (Char_no = 0; Char_no < Last_Ch_No; Char_no ++) {
    	error = false;			/* No errors yet on these fscanf()'s  */
	fscanf(fp, "%*[ \r\t]");	/* skip over in-line white space      */
	if (blank_pixel == DOT) {
	    n_scan = fscanf(fp,"%n%[.*]%n",
	    			&n_pix1, ptr = band[Char_no][row], &n_pix2); 
	    for (i = 0; i <= Cell_width; ptr++, i++)
		if (*ptr == '.')		/* Convert dots '.' to spaces */
		    *ptr = ' ';
	  }
	  else if (blank_pixel == SPACE)  {
	    fscanf(fp, "%*[|]");	/* skip over border |'s		      */
	    n_scan = fscanf(fp,"%n%[ *]%n",
					 &n_pix1, band[Char_no][row], &n_pix2); 
	   }
/*----------------------error checks on fscanf--------------------------------*/
/* Have scanned one row of one Char.  Expect: n_scan == 1 (one string read) and
 * n_pix2 - n_pix1 == Cell_Width.  Error messages are generated by error
 * functions named ..._err.  The offending lines are displayed on the
 * screen by calls to err_show_and_tell().  error = true is set by err_ask()
 * which is called by error functions.
 */ 
	n_pixels_read = n_pix2 - n_pix1;	/* no. pixels read by fscanf  */
	switch (n_scan) {		/* n_scan=no. strings got OK by fscanf*/
	case 1:				/* fscanf got a string		      */
	    if(n_pixels_read  == Cell_Width)
		 continue;		/* AOK.Jump out of sw. Do next Char_no*/
	      else 			/* Scanned width in error	      */
		n_pix_err(n_pixels_read);     /* Give n_pix_err string length */
	    break;
	case -1: 			/* Unexpected end of file.	      */
	    eof_err();    		/* Error msg.  Will exit.	      */
	case 0:				/* fscan couldn't read (and assign).  */
	    if (row == 0) {		/* Perhaps band is short;or illegal ch*/
		ch = fgetc(fp);		/* \n if short. Prev. skipped white sp*/
		ungetc( ch, fp);	/* put back ch for scan_err() or eol  */
		if (ch == '\n') {	/* Likely a short band	      */
		    Last_Ch_No = Char_no;	/* Use this for rest of band  */
		    goto EXIT_LOOP;		/* Quit loop on this line     */
	     	 }			/* "break" would only exit switch     */
	     }
	    scan_err(); 		/* not short band and n_scan ==  0    */
	 }				/* end of switch (n_scan)		      */
	if (error) {			/* True if an error in this scan      */
	    fscanf(fp, "%*[^\n]");	/* Skip to end of line \n to continue */
	    break;			/* break out of loop for this line    */
	 }
     }				/* End for on Char_no.  Exit => line scanned  */
     EXIT_LOOP:			/*Come from case 0 inside switch if short band*/
/*------------------------------at end of line--------------------------------*/
    fscanf(fp, "%*[ |\t\r]");	/* Skip past spaces, tabs, cr, | to new_line. */
    fscanf(fp, "%c", &ch);		/* Should be \n. 		      */
    if (ch != '\n') {
	eol_err(ch);			/*error message			      */
	fscanf(fp, "%*[^\n]");		/* Skip to \n to continue scanning.   */
	fscanf(fp, "\n");		/* Skip over \n to continue scanning. */
      }
 }	/* End of for on rows.  Exits loop after filling a band array.	      */
}
/*-----------------------------end fill_band()--------------------------------*/
/****************************** eof_err() *************************************/
void eof_err()		/*End of file reached unexpectedly		      */
{
err_show_and_tell(3, eof);
fputs("From eof_err(): fscanf returned -1.  Unexpected end of pattern"
  "file. \nCheck for short Char-matrices in last rows of file. \n", stderr);
exit(-1);
}
/***************************** n_pix_err() ************************************/
void n_pix_err(int bad_width){		/* Wrong number of pixels in pat. cell*/
err_show_and_tell(3, n_pix);
ch = fgetc(fp);
fprintf(stderr, "From n_pix_err(): ");
if ( bad_width < Cell_Width )		/* Short width due to illegal pixel ? */
    fprintf(stderr, " An illegal pixel is likely: %c (= Hex %x). ",  ch, ch);
  else 
    fprintf(stderr, " A cell width on the middle line is too wide. ");
fprintf(stderr, " The number of \nlegal pixels read (=%d) in a Char. row is"
  " not equal to the Cell_Width of %d. \n" ,bad_width, Cell_Width);
err_ask(n_pix);
}
/******************************* scan_err() ***********************************/
void  scan_err()
{
ch = fgetc(fp);
err_show_and_tell(3, scan);
fprintf( stderr,"From scan_err(): Couldn't accept char '%c'.  It's either"
  " an invalid pixel\n or separator (or perhaps a char before it is"
  " invalid).  If the middle \n line is a header line, type h for help.\n", ch);
err_ask(scan);
}
/******************************** eol_err() ***********************************/
void eol_err(char ch)
{
err_show_and_tell(5, eol);
fprintf(stderr, "From eol_err(): Expected a new_line char. but got: '%c'"
  " (= Hex %x).\n  May be junk near end of line or too many patterns per row"
  " (Only %d allowed) \n", 	ch, ch, Chars_Per_Band);
err_ask(eol);
}
/*************************** err_show_and_tell() ******************************/
/* Really should be "tell and show".  General error info used by all error
 * functions.  Prints general info on location of error and shows (displays)
 * it using foffset to find its location in the pattern file.  Displays
 * it by seeking to start of a line (of input file) and prints nlines.
 * The suspect lines displayed should be the ones which were just scanned.
 */
void err_show_and_tell(
int nlines,				/* # of lines to show (=print)	      */
err_type err) {				/* type of error which called this    */
int start_line;				/* start showing with this line	      */
long orig_offset;
char line[LINE_LENGTH + 3];
error_count ++;
fprintf( stderr, "\n***ERROR*** in your pattern file: "); 
if ( err != blank_pix )
    fprintf(stderr, " in band %d, line %d.  It's in row %d \n of the Character"
      " matrix. \nAn error likely in the middle line shown below near Char."
      " no. %d (1 is first): \n", 	band_no, line_no, row + 1, Char_no + 1);
if (err == blank_pix )
    fprintf(stderr, " You likely have too many header lines \n at the start"
      " of your file.  See documentation. \n	Below are some header lines"
      " from the pattern file:\n");
orig_offset = ftell(fp);		/* Save offset for future restoration */
start_line = line_no - nlines/2;	/* Start half way back from line_no.  */
start_line = (start_line > 0) ? start_line : 1;	   /* start_line can't be neg.*/
fseek(fp, foffset[start_line], 0);	/* Move file ptr to 1st line to show  */
for(i = 1; i <= nlines; i++) {
    if ( fgets(line, sizeof(line), fp) != (void*) 0 )/* Also gets \n at eol*/
    fputs(line, stderr);
 }
if ( err == blank_pix)
    exit (-1);				/* Can't continue since line count ng */
fseek(fp, orig_offset, 0);		/* Restore file pointer to orig. loc. */
}
/******************************* err_ask() ************************************/
/* Called by every error function except eof_err().  Ask user: quit?   help?  */
void err_ask(err_type err)
{
char ans;
error = true;			/* so we can skip to a new line in fill_band()*/
if (std_in) {			/* Using stdin for pattern file doesn't allow */
    if ( error_count > MAX_ERRORS )		/* for any interactive input. */
	exit(-1);			
    return;
 }
while (true) {
    fputs("Look for More errors: m   Quit: q   Help: h  Then hit <return> or "
	  "<enter>", stderr);
    scanf("%*[\n\r\t ]");		/* Skip over possible white space     */
    ans = getchar();
    switch (ans) {
      case 'm': return;
      case 'q': exit (-1);
      case 'h': fputs("NOTE: A single error may cause some additional \n"
	" misleading error messages.  Continuing to look for more errors"
	" after finding\n an error may not find all the errors in your pattern" 
	" file.\n", 			stderr);

	if (err == scan)
	fputs("\nIf the \"offending\" line is a header line don't continue"
	" looking for more\n errors since the row count is out of kilter:"
	" Quit and check the height of\n your Character-matrix patterns"
	" prior to the first such header seen.\n",	stderr);
		break;
      default: fprintf(stderr, "	You responded with a '%c'  Please type"
	" only m, q, or h. \n", ans);
     }		/* end switch 	   */
 }		/* end while(true) */
}
/****************************** err_exit() ************************************/
void err_exit() {
fputs( "FATAL ERROR: No soft-font was created.  Try again.\n", stderr );
exit(-1);
}
/******************************** usage() *************************************/
void usage( 
int flag) 		 /* Should usage() exit, return, or err_exit()?	      */
{
fprintf(stderr,  
  "--------------------------------To get soft-font:---------------------------"
									"----\n"
  "pat2sf  [OPTIONS]  lang  [LANG_OPTIONS]  pattern_file  >  softfont_file\n"
  "     lang:  the soft-font language.  E.g. vt for vt220 terminals.  Consult"
								 " help.\n"
  "     pattern_file:  the name of the file in which you drew your Characters."
									"\n"
  "     softfont_file: the name of the file which this program will create.\n"
  "OPTIONS:   	 	(Don't use = for arguments to letter options.) \n"
  "  -p   --pad_level=80   Append 80 etc. (default=0) padding nulls to each \n"
  "			   line of output.\n"
  "  -s   --stdin      use standard input.\n" 		);
if (usageP == NULL)				/* usageP not set to a lang   */
    fprintf(stderr, 
	"LANG_OPTIONS:  Depends on language.  See help below to show them.\n");
  else
    (*usageP)();
fprintf(stderr, 
  "-----------------------------------To get help:-----------------------------"
									"----\n"
  " pat2sf -v   --version    shows version number.\n"
  " pat2sf		  	Display this message \n"
  " pat2sf --help list    Show list of supported Languages (LANG). \n"
  " pat2sf --help LANG    Show LANG_OPTIONS for Language LANG in this message."
									"\n"
  "NOTE: You must read the manual to find out format of pattern_file, etc.\n"); 
if (flag == EXIT)     exit (0);
if (flag == ERR_EXIT) err_exit();   /* else if flag == RETURN return normally.*/
}
