program HazeltineEmulation;
uses
  Crt,                                    { TURBO PASCAL CRT unit }
  Unit_A0,                                { Include Level Zero Code Module }
  Unit_A1,                                { Level One functions }
  Unit_A2;                                { Level Two  - error codes only }

  const
    DFLTHZFA = $18;
    EXITCHAR = 3;
    ETX = $03;
    STATUSLINEATTR = $0f;
    MAXVIDEO=10;

  type
    ByteArray = array[0..1920] of byte;
    BytePtr = ^ByteArray;



  type
   ScreenMatrix = record
     case integer of
       0: (rc : array [1..24, 1..80] of byte);
       1: (ndx: array [1..1920] of byte);
     end;

  type
   FieldDefRecord = record
     col : byte;
     row : byte;
     ndx : word;
     len : word;
   end;

  var
    PortNo,
    BaudRate,
    StopBits,
    IrqLevel,
    PortAddress,
    VideoMode : word;


  var
    OrigMode : integer;

  var
    ScrData   : ScreenMatrix;           { the ascii data }
    ScrAttr   : ScreenMatrix;           { the PC test attr's }
    ScrHzFA   : ScreenMatrix;           { the HZLTN F/A's sent }
    ScrFTab   : ScreenMatrix;           { Frwd Tabbing indexing }
    ScrBTab   : ScreenMatrix;           { Bkwd Tabbing indexing }

  const
    MAXFIELD  = 512;
    FLDCOL    =   1;
    FLDROW    =   2;
    FLDLEN    =   3;
  var
    FldDef    : array [0..MAXFIELD] of FieldDefRecord;
    ErrFldDef : FieldDefRecord;
    XmtFldDef : FieldDefRecord;
    MxFld     : integer;

  const
    FkeyArray  : array [0..7] of byte = ($11,$12,$13,$14,$1a,$1c,$1e,$1f);
    PortArray  : array [1..4] of word = ($3f8,$2f8,$3e8,$2e8);

  const                               {   L   Lb  G   Gb  H   Hb  W   Wb   }
    ColorArray : array [1..MAXVIDEO,0..7] of byte =
                  {   L   Lb  G   Gb  H   Hb  W   Wb   }
         {  1  }   (($07,$87,$78,$f8,$0f,$8f,$70,$f0),
         {  2  }    ($07,$87,$60,$e0,$0f,$8f,$70,$f0),
         {  3  }    ($12,$92,$5b,$db,$1a,$9a,$3b,$bb),
         {  4  }    ($09,$89,$1f,$9f,$0b,$8b,$3f,$bf),
         {  5  }    ($03,$83,$3f,$Bf,$02,$82,$2f,$af),
         {  6  }    ($02,$82,$2f,$Af,$03,$83,$3f,$Bf),
         {  7  }    ($03,$83,$3f,$Bf,$07,$87,$7f,$Ff),
         {  8  }    ($76,$f6,$6f,$ef,$70,$f0,$0f,$8f),
         {  9  }    ($73,$f3,$3f,$bf,$71,$f1,$17,$97),
         {  0  }    ($0d,$8d,$5c,$dc,$0e,$8e,$7e,$fe));

  var
    ClrScrAttr,
    Next980    : byte;

    NextOp     : integer;

    InsertActive,
    ErrorLineActive,
    XmitCharActive,
    ExitRequest,
    F7On,
    F8On,
    KeyBoardLocked :  boolean;

    Blanks    : string[255];

    Cur980Index, Max980Index : word;
    Cur980String : string[255];


const
    { Level Zero requires an area of memory to be allocated for an	}
    { Input/Output buffer. The size of the buffer must include 4 extra	}
    { bytes needed by Level Zero for internal processing.  Note that	}
    { there is only one buffer which is used for both an input and	}
    { output buffer according to the values passed to Level Zero when	}
    { the program is opened.						}

    INQSIZE  = 6000;			{ The input queue size		}
    OUTQSIZE = 2000;			{ The output queue size 	}
    BUFSIZE  = 8004;

var
  IOBuffer      : array[1..BUFSIZE] of byte;   { Asynch I/O Buffer }


procedure UpdateStatusLine(msgcol : integer; msgstr : string);
  var
    row, col :integer;
    savTextAttr : byte;
  begin
    row := WhereY;
    col := WhereX;
    savTextAttr := TextAttr;
    gotoXY(msgcol,25);
    TextAttr := STATUSLINEATTR;
    write(msgstr);
    gotoXY(col,row);
    TextAttr := savTextAttr;
  end;

procedure ReportError(msg : string);
  var
    len : integer;
    temp : string[59];

  begin
    temp := copy(blanks,1,59);
    len := length(msg);
    if len > 59 then len := 59;
    move(msg[1],temp[1],len);
    UpdateStatusLine(1,temp);
  end;



  {*********************************************************************}
  { ShowResult	Show the result of the last function and error message	}
  {*********************************************************************}

  procedure ShowResult(ErrorCode : word);

    var
      ErrorCodeStr : string[9];
      Message,
      Fix	: string;

  begin { ShowResult }

    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
    { __ErrMsgA2  Return extended error message.			}
    {									}
    { procedure __ErrMsgA2(ErrorCode : word;				}
    {		       var Message   : string;				}
    {		       var PosFix    : string); 			}
    {									}
    { Although it is not part of Unit_A1, __ErrMsgA2 has text messages	}
    { for errors returned by all the units.  It can be useful for any	}
    { ASYNCH program, especially when it is being debugged.		}
    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

    __ErrMsgA2(ErrorCode,Message,Fix);

    str(ErrorCode:3, ErrorCodeStr);
    ReportError(concat(ErrorCodeStr,'  ',Message));

{   Halt;  }

  end;	{ ShowResult }


  {*********************************************************************}
  { OpenPort  Open using a Unit_A1 function call.			}
  {*********************************************************************}

  procedure OpenPort;
    var
      ErrorCode : word;

  begin { OpenPort }

    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
    { __OpenA1	 Open a COM port for interrupt service. 		}
    {									}
    { function __OpenA1(PortNo	  : word;				}
    {			InQSize   : word;				}
    {			OutQSize  : word;				}
    {			IntLevel  : word;				}
    {			PortAds   : word;				}
    {			BufferAds : pointer) : word;			}
    {									}
    { In the example below, the port is opened with the default 	}
    { interrupt line and port address.	The 'at' statement (@) is used  }
    { to pass the address of the transmission buffer. The port address	}
    { (PortAds) is really the starting address of the first of several	}
    { UART registers that have been mapped into the PC's I/O address    }
    { space.  The IntLevel is one of the 8259 Programable Interrupt	}
    { Controller channels. Some of those lines are used for keyboard	}
    { and disk I/O, so be careful if you don't use the defaults.        }
    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

    ErrorCode := __OpenA1(PortNo,INQSIZE,OUTQSIZE,
                  IrqLevel,PortArray[PortAddress],@IOBuffer);

      if (ErrorCode <> 0) then
	 ShowResult(ErrorCode)

  end;	{ OpenPort }


  {*********************************************************************}
  { ClosePort  Close an open COM port					}
  {*********************************************************************}

  procedure ClosePort;

    var
      QSize,
      ErrorCode : word;
      Message,
      Fix	: string;

  begin {ClosePort}

    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
    { __CloseA1  Close a COM port.					}
    {									}
    { function __CloseA1(PortNo : word);				}
    {									}
    { You will get an error if you try to close a port that has not	}
    { been opened, or if the number of the port is greater than the	}
    { constant _MAX_PORT.  One common error is to close a port without	}
    { checking the input or output queues to insure that there are no	}
    { characters waiting.  Here, the output queue is checked to insure	}
    { that all characters have been sent.  In case there is a		}
    { transmision problem and the queue is not draining, the error is	}
    { reported, and the port is simply closed.				}
    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

      ErrorCode := __OQSizeA1(PortNo,QSize);
      if (ErrorCode <> 0) then
	 ShowResult(ErrorCode)
      else if (QSize <> 0) then
        begin
           Writeln('There were ',QSize,' characters in the ouptut ',
			'queue when it was closed.');
	   Writeln;
	end;

      ErrorCode := __CloseA1(PortNo);

      if (ErrorCode <> 0) then
	 ShowResult(ErrorCode)

  end;	{ClosePort}


  {*********************************************************************}
  { SetOptions	Set some of the transmission options			}
  {*********************************************************************}

  procedure SetOptions;

    var
      OptionNo,
      ReturnNo,
      Value,
      ErrorCode : word;

  begin { SetOptions }

	      {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
	      { __SetOpA1  Set a transmission option.			}
	      { 							}
	      { function __SetOpA1(PortNo    : word;			}
	      { 		   Option    : word;			}
	      { 		   Parameter : word) : word;		}
	      { 							}
	      { There are 10 options, and only the most necessary were	}
	      { included here. Be sure to check the error code every	}
	      { time to see if the value was set correctly. Also the	}
	      { values that you are setting are really registers in the }
	      { UART. They will remain there even after your program	}
	      { terminates.  The "power up" defaults are listed in the  }
	      { manual. 						}
	      {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

        {Baud Rates: 0=110, 1=150, 2=300, 3=600, 4=1200,   }
        {            5=2400,6=4800,7=9600,8=19200          }
     ErrorCode := __SetOpA1(PortNo,1,BaudRate);
     if ErrorCode <> 0 then
       ShowResult(ErrorCode);

	{Parity Value: 0=none, 1=odd, 2=even  }
     ErrorCode := __SetOpA1(PortNo,2,1);
     if ErrorCode <> 0 then
       ShowResult(ErrorCode);

        {Data Bits: 0=5, 1=6, 2=7, 3=8  }
     ErrorCode := __SetOpA1(PortNo,3,2);
     if ErrorCode <> 0 then
       ShowResult(ErrorCode);

        {Stop Bits: 0=1, 1=2  }
     ErrorCode := __SetOpA1(PortNo,4,StopBits);
     if ErrorCode <> 0 then
       ShowResult(ErrorCode);

  end;	{ SetOptions }

  {*********************************************************************}
  { ReadString	Read a string from a COM port.				}
  {*********************************************************************}

  procedure ReadString;

    var
      ErrorCode,
      PStatus,
      NumRead,
      IQSize	 : word;
      Strng      : string;

  begin {ReadString}

    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
    { __RdStA1	 Read a string from a COM port. 			}
    {									}
    { function __RdStA1(PortNo	   : word;				}
    {			UpBound    : word;				}
    {			BufferAds  : pointer;				}
    {		    var NumRead    : word;				}
    {		    var InQSize    : word;				}
    {		    var PortStatus : word) : word;			}
    {									}
    { Many of the ASYNCH functions are more suited for arrays, rather	}
    { than Turbo Pascal strings.  This example shows how to adapt the	}
    { __RdStA1 function so that it can be used with a string variable.	}
    { In the example, BufferAds is the first position of the string,	}
    { String[1], and the length of the string, String[0] is set with	}
    { NumRead. The port status is returned with the string. If it is	}
    { non-zero, then some transmission error occurred (versus an ASYNCH }
    { error returned as the function value).  The constant _STATUS_ERR	}
    { is a bit mask that can be used to filter transmission errors	}
    { (versus flow control conditions). 				}
    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

    ErrorCode :=
    __RdStA1(PortNo,255,@Cur980String[1],Max980Index,IQSize,PStatus);

    Cur980String[0] := Chr(Max980Index);
    if ((PStatus and _STATUS_ERR) <> 0) then
       begin
	 Strng := 'Tx err:';
	 if ((PStatus and _INPUT_LOST) <> 0) then
           Strng := concat(Strng,'  Input Lost');
	 if ((PStatus and _ASYNCH_ERR) <> 0) then
           Strng := concat(Strng,'  Async Err');
	 if ((PStatus and _PARITY_ERR) <> 0) then
           Strng := concat(Strng,'  Parity');
	 if ((PStatus and _OVERRUN) <> 0) then
           Strng := concat(Strng,'  Overrun');
	 if ((PStatus and _FRAMING) <> 0) then
           Strng := concat(Strng,'  Framing');
	 if ((PStatus and _BREAK) <> 0) then
           Strng := concat(Strng,'  BREAK');
         ReportError(Strng);
       end;
    if (ErrorCode <> 0) and (ErrorCode <> _IN_Q_EMPTY) then
      ShowResult(ErrorCode);

  end;	{ReadString}


  {*********************************************************************}
  { WriteString  Send a string over a COM port. 			}
  {*********************************************************************}

  procedure WriteString;

    var
      ErrorCode,
      PStatus,
      NumWrit,
      IQSize	 : word;
      Strng	 : string;

  begin {WriteString}

    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}
    { __WrtStA1  Write a string to a COM port.				}
    {									}
    { function __WrtStA1(PortNo    : word;				}
    {			 UpBound   : word;				}
    {			 BufferAds : pointer;				}
    {		     var NumWrit   : word) : word;			}
    {									}
    { Since ASYNCH functions use memory addresses, some adjustments	}
    { must be made to pass Turbo Pascal strings to the "write string"   }
    { function. The 'at' operator (@) is used to pass the address of    }
    { the first character in the string, since String[0] is the length	}
    { byte.								}
    {@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@}

    ErrorCode := __WrtStA1(PortNo,Length(Strng),@Strng[1],NumWrit);
    If ErrorCode <> 0 then
      ShowResult(ErrorCode);

  end; {WriteString}

  procedure WriteBytes(Obuf : pointer; OCount : word);

    var
      ErrorCode,
      NumWrit    : word;

  begin {WriteBytes}

    ErrorCode := __WrtStA1(PortNo,OCount,OBuf,NumWrit);
    If ErrorCode <> 0 then
      ShowResult(ErrorCode);

  end; {WriteBytes}


  procedure WriteSingleByte(ByteData : char);

    var
      ErrorCode : word;

  begin {WriteSingleByte}

    ErrorCode := __WrtChA1(PortNo,ByteData);
    If ErrorCode <> 0 then
      ShowResult(ErrorCode);

  end; {WriteSingleByte}



  procedure ReportUnexpected(unexp : byte);
    var
      unexpStr : string[3];
    begin
      str(unexp:3, unexpStr);
      ReportError(concat('Unexpected Transmission Character: ',unexpStr));
    end;


  function ChkNextOp : boolean;
    begin
      if KeyPressed then begin
        NextOp := Ord(ReadKey);
        if NextOp = 0 then
          NextOp := Ord(ReadKey) + $100;
        if NextOp = EXITCHAR then begin
          ChkNextOp := true;
          ExitRequest := true;
          Next980 := $ff;
          end
        else
          if KeyboardLocked then
            ChkNextOp := false
          else
            ChkNextOp := true;
        end
      else begin
        ChkNextOp := false;
        NextOp := 0;
        end;
     end;


  function ChkNext980 : boolean;
    begin
      if Cur980Index < Max980Index then
       begin
        Inc(Cur980Index);
        Next980 := ord(Cur980String[Cur980Index]);
        ChkNext980 := true;
       end
      else {read another string}
       begin
        ReadString;
        if Max980Index >= 1 then
         begin
          Cur980Index := 1;
          Next980 := Ord(Cur980String[1]);
          ChkNext980 := true;
         end
        else
         begin
          Cur980Index := 0;
          ChkNext980 := false;
          Next980 := 0;
         end;
       end;
     end; {ChkNext980}


  procedure GetNext980;
    begin;
      while not ChkNext980 do
       begin
        if ChkNextOp then
          if Next980 = $ff then exit;
       end;
    end;


  procedure ClearScreenMatrices;
    begin
      FillChar(ScrData, SizeOf(ScrData), ' ');
      FillChar(ScrAttr, SizeOf(ScrAttr),
        ColorArray[VideoMode,DFLTHZFA and $07]);
      FillChar(ScrHzFA, SizeOf(ScrHzFA), 0);
      ScrHzFA.rc[1,1] := DFLTHZFA;
      MxFld := 0;
      FldDef[0].col := 1;
      FldDef[0].row := 1;
      FldDef[0].ndx := 1;
      FldDef[0].len := 1;
      ErrFldDef.col := 1;
      ErrFldDef.row := 1;
      ErrFldDef.ndx := 1;
      ErrFldDef.len := 0;
      ErrorLineActive := false;
      XmitCharActive := false;
   end;

  procedure ClearUnprotected;
    var
      fld, col, row, len : integer;

    begin
      for fld := 1 to MxFld do begin
        col := FldDef[fld].col;
        row := FldDef[fld].row;
        len := FldDef[fld].len;

        FillChar(ScrData.rc[row,col], len, ' ');
        TextAttr := ScrAttr.rc[row,col];
        GotoXY(col,row);
        Write(Copy(Blanks,1,len));
        end;
    end;

  procedure OpClearUnprotected;

    begin
      if ErrorLineActive then begin
        FillChar(ScrData.rc[ErrFldDef.row,ErrFldDef.col],
          ErrFldDef.len, ' ');
        TextAttr := ScrAttr.rc[ErrFldDef.row,ErrFldDef.col];
        GotoXY(ErrFldDef.col,ErrFldDef.row);
        Write(Copy(Blanks,1,ErrFldDef.len));
        end
      else
        ClearUnprotected;
    end;


procedure Beep;

  begin
    Sound(880);
    Delay(150);
    NoSound;
  end;


procedure BuildScreen;

    { BUILD SCREEN DATA STRUCTURES AND PAINT THE SCREEN AT COMPLETION OF  }
    { NEW SCREEN TRANSMISSION.                                            }
    { set the first "start of field" to the first screen position         }
    { LOOP                                                                }
    { find next "end of field" - this will be either the position before  }
    {   the next "start of field", or the end of the screen.              }
    { update the text attrib array (ScrAttr), using FillChar              }
    { if the field is unprotected,                                        }
    {   add it to the field definition array                              }
    {   ???update the FTab and BTab arrays???                             }
    { write the field data to the screen                                  }
    {   if the field attribute is the same as that with which the screen  }
    {     was cleared, then only bother with writing the non-blanks       }
    {   else build a temp string item from the ScrData which goes in the  }
    {     field and use a string write to paint it onto the screen.  if   }
    {     the field spans lines, however, then make a string write for    }
    {     each line                                                       }
    { if it is not the end of the screen then redo the LOOP               }
    { Now the New Screen should be complete: the data structures built    }
    {   and the protected part of the display and unprotected fields      }
    {   painted on the screen                                             }


  procedure BuildTabArrays;
    var
      sndx, endx, len, fld, lfld : integer;

    begin
      if MxFld = 0 then
       begin
        FillChar(ScrFTab, SizeOf(ScrFTab), chr(0));
        FillChar(ScrBTab, SizeOf(ScrBTab), chr(0));
       end
      else
       begin
        ScrBTab.ndx[1] := 0;
        sndx := 1;
        lfld := 0;
        for fld := 1 to MxFld do
         begin
          endx := FldDef[fld].ndx;
          len := endx - sndx;
          if len > 0 then
           begin
            FillChar(ScrFTab.ndx[sndx], len, chr(fld));
            FillChar(ScrBTab.ndx[sndx+1], len, chr(lfld));
           end;
          sndx := endx;
          lfld := fld;
         end; {for}
        len := 1920 - sndx;
        FillChar(ScrFTab.ndx[sndx], len+1, chr(lfld));
        FillChar(ScrBTab.ndx[sndx+1], len, chr(lfld));
       end;
    end;


  procedure ProcessField(fieldstrt, fieldend : integer);
    var
      fieldFA, fieldcol, fieldrow : byte;
      fieldlen, ndx, colndx, rowndx : integer;
      temp : string[255];

    begin
      fieldlen := fieldend - fieldstrt + 1;
      fieldcol := (fieldstrt-1) mod 80 + 1;
      fieldrow := (fieldstrt-1) div 80 + 1;
      fieldFA := ScrHzFA.ndx[fieldstrt];
      if fieldlen > 1 then
        FillChar(ScrHzFA.ndx[fieldstrt+1], fieldlen-1, fieldFA and $7f);
      TextAttr := ColorArray[VideoMode,fieldFA and $07];
      if (fieldFA and $20) <> 0 then
        TextAttr := ((TextAttr and $f0) or ((TextAttr shr 4) and $07));
      FillChar(ScrAttr.ndx[fieldstrt], fieldlen, TextAttr);
      if (fieldFA and $40) = 0 then
       begin                                 { process unprotected field }
         Inc(MxFld);
         FldDef[MxFld].col := (fieldstrt) mod 80 + 1;
         FldDef[MxFld].row := (fieldstrt) div 80 + 1;
         FldDef[MxFld].ndx := fieldstrt + 1;
         FldDef[MxFld].len := fieldlen - 1;
       end;
      if (TextAttr and $70) <> (ClrScrAttr and $70) then
       begin
        gotoXY(fieldcol, fieldrow);
        ndx := fieldstrt;
        while fieldlen > 255 do begin
          temp[0] := chr(255);
          move(ScrData.ndx[ndx],temp[1],255);
          write(temp);
          Inc(ndx,255);
          Dec(fieldlen,255);
          end;
        if fieldlen > 0 then begin
          temp[0] := chr(fieldlen);
          move(ScrData.ndx[ndx],temp[1],fieldlen);
          write(temp);
          end;
        end
      else
       begin
        rowndx := fieldrow;
        colndx := fieldcol;
        for ndx := fieldstrt to fieldend do
         begin
          if ScrData.ndx[ndx] <> 32 then
           begin
            gotoXY(colndx,rowndx);
            write(chr(ScrData.ndx[ndx]));
           end;
          if colndx < 80 then
            Inc(colndx)
          else begin
            colndx := 1;
            Inc(rowndx);
            end;
         end;
       end;
     end; {ProcessField}


  var
    fieldstrt, fieldend : integer;
    i : integer;


  begin
    TextAttr := ColorArray[VideoMode,ScrHzFA.ndx[1] and $07];
    Window(1,1,80,24);
    ClrScr;
    Window(1,1,80,25);
    ClrScrAttr := TextAttr;
    fieldstrt := 1;
    for i := 2 to 1920 do
     begin
      if ScrHzFA.ndx[i] <> 0 then
       begin                                   { process previous field }
        ProcessField(fieldstrt,i-1);
        fieldstrt := i;
       end; {if}
     end; {for}
    if fieldstrt < 1920 then
     begin
      ProcessField(fieldstrt, 1919);
     end;

    BuildTabArrays;

    GotoXY(1,1);

  end; {BuildScreen}


procedure ProcessF7On;            {F7ON}
  begin
    F7On := True;
    UpdateStatusLine(62,'AVAIL');
  end;


procedure ProcessF7Off;           {F7OFF}
  begin
    F7On := false;
    UpdateStatusLine(62,'WAIT ');
  end;


procedure ProcessF8On;            {F8ON}
  begin
    F8On := True;
    UpdateStatusLine(69,'T/O');
  end;


procedure ProcessF8Off;           {F8OFF}
  begin
    F8On := false;
    UpdateStatusLine(69,'   ');
  end;


procedure ProcessKeyboardLock;    {KBDLK}
  begin
    KeyBoardLocked := true;
  end;


procedure ProcessKeyboardUnlock;  {KBUNL}
  var
    x : boolean;
  begin
    repeat x := ChkNextOP
     until (NextOp = 0) or ExitRequest;
    KeyBoardLocked := false;
  end;




procedure NewScreen;

  function GetCursorIndex : integer;
    var
      csrcol, csrrow : byte;

    begin
      GetNext980;
      if Next980 >= $60 then
        csrcol := Next980 - $60
      else
        csrcol := Next980;
      if csrcol > 79 then csrcol := 0;
      GetNext980;
      csrrow := Next980 - $60;
      if csrrow > 23 then csrrow := 0;
      GetCursorIndex := (csrrow) * 80 + csrcol + 1;
    end; {GetCursorIndex}

  var
    cont : boolean;
    ndx : integer;


  begin
    ClearScreenMatrices;
    ndx := 1;
    cont := false;
    repeat
      GetNext980;
      case Next980 of
        $20..$7d : begin
                    ScrData.ndx[ndx] := Next980;
                    Inc(ndx);
                   end;
        $7f : ;
        $7e :
         begin
          GetNext980;
          case Next980 of
            $11: begin {POSCUR}
                  ndx := GetCursorIndex;
                 end;
            $20: begin {FADef}
                  GetNext980;
                  ScrHzFA.ndx[ndx] := (Next980 or $80);
                  Inc(ndx,1);
                 end;
            $12: begin {HOME}
                  BuildScreen;
                  cont := true;
                 end;
            $1c: begin {CLSCRN}
                  if ndx <> 1 then
                   begin
                    ClearScreenMatrices;
                    ndx := 1;
                   end;
                 end;
            $06: begin {KBUNL}
                  BuildScreen;
                  ProcessKeyboardUnlock;
                  cont := true;
                 end;
            $ff: begin {ABORT}
                  cont:= true;
                 end;
          else
            ReportUnexpected(Next980);
          end; {case}
        end; {case $7e}

      else
        ReportUnexpected(Next980);
      end; {case}

    until (cont);




  end; {NewScreen}

  procedure XmitCursorPosition;
    var
      col : integer;

    begin
      col := WhereX;
      if col <= $20 then Inc(col,$5f)
      else Dec(col,1);
      WriteSingleByte(chr(col));
      WriteSingleByte(chr(WhereY - 1));
      WriteSingleByte(chr(ETX));
    end;


  procedure XmitErrorLine;
    var
      ndx, endx : integer;

    begin
      if XmitCharActive then
	ndx := XmtFldDef.ndx + 1
      else ndx := 80;

      endx := (WhereY - 1) * 80 + WhereX - 1;
      for ndx := ndx to endx do begin
	if (ScrHzFA.ndx[ndx] and $c0) = 0 then
	  WriteSingleByte(chr(ScrData.ndx[ndx]));
	end;
      WriteSingleByte(chr(ETX));
      XmitCharActive :=false;
      ErrorLineActive := false;
    end;


  procedure XmitScreenData;
    var
      fld, row, col, len : integer;

    begin
      for fld := 1 to MxFld do
       begin
        col := FldDef[fld].col;
        row := FldDef[fld].row;
        len := FldDef[fld].len;

        GotoXY(col,row);
        WriteBytes(@ScrData.rc[row,col],len);
       end;
      GotoXY(80,24);
      WriteSingleByte(chr(ETX));
    end;

  procedure ProcessPositionCursor;
    var
      csrcol, csrrow : byte;

    begin
      GetNext980;
      if Next980 >= $60 then
        csrcol := Next980 - $5F
      else
        csrcol := Next980 + 1;
      if csrcol > 80 then csrcol := 1;
      GetNext980;
      csrrow := Next980 - $5F;
      if csrrow > 24 then csrrow := 1;
      GotoXY(csrcol,csrrow);
      TextAttr := ScrAttr.rc[csrrow,csrcol];
    end;


procedure ProcessFADef;           {F/A DEF}
  var
    row, col, ndx, endx, len : integer;
    temp : string [255];

  begin
    GetNext980;
    TextAttr := ColorArray[VideoMode,Next980 and $07];
    ScrData.rc[WhereY,WhereX] := $20;
    write(' ');
    row := WhereY;
    col := WhereX;
    ndx := (row - 1) * 80 + col;
    if (Next980 and $40) = 0 then begin
      ErrFldDef.ndx := ndx;
      ErrFldDef.col := col;
      ErrFldDef.row := row;
      end;
    ScrHzFA.ndx[ndx-1] := Next980 or $80;
    endx := ndx;
    while (endx <= 1920) and (ScrHzFA.ndx[endx] < $80) do
      Inc(endx);
    len := endx - ndx;
    if len > 0 then
      FillChar(ScrHzFA.ndx[ndx], len, Next980);
    FillChar(ScrAttr.ndx[ndx-1], len+1, TextAttr);
    while len > 255 do begin
      temp[0] := chr(255);
      move(ScrData.ndx[ndx],temp[1],255);
      write(temp);
      Inc(ndx,255);
      Dec(len,255);
      end;
    if len > 0 then begin
      temp[0] := chr(len);
      move(ScrData.ndx[ndx],temp[1],len);
      write(temp);
      end;
    gotoXY(col,row);
  end;


procedure ProcessFAErase;         {F/A ERASE}
  var
    fieldFA : byte;
    row, col, ndx, endx, len : integer;
    temp : string [255];

  begin
    ndx := (WhereY - 1) * 80 + WhereX;
    if ndx > 1 then begin
      fieldFA := ScrHzFA.ndx[ndx - 1] and $7f;
      TextAttr := ScrAttr.ndx[ndx - 1];
      end
    else begin
      fieldFA := DFLTHZFA;
      TextAttr := ColorArray[VideoMode,DFLTHZFA and $07];
      end;
    ScrHzFa.ndx[ndx] := fieldFa;
    ScrAttr.ndx[ndx] := TextAttr;
    ScrData.ndx[ndx] := Next980;
    write(chr(Next980));
    row := WhereY;
    col := WhereX;
    Inc(ndx);
    endx := ndx;
    while (endx <= 1920) and (ScrHzFA.ndx[endx] < $80) do
      Inc(endx);
    len := endx - ndx;
    if len > 0 then begin
      FillChar(ScrHzFA.ndx[ndx], len, fieldFA);
      FillChar(ScrAttr.ndx[ndx], len, TextAttr);
      end;
    while len > 255 do begin
      temp[0] := chr(255);
      move(ScrData.ndx[ndx],temp[1],255);
      write(temp);
      Inc(ndx,255);
      Dec(len,255);
      end;
    if len > 0 then begin
      temp[0] := chr(len);
      move(ScrData.ndx[ndx],temp[1],len);
      write(temp);
      end;
    gotoXY(col,row);
  end;


procedure Process980Tab;          {TAB}
  var
    row, col, ndx : integer;

  begin
    ndx := ScrFTab.rc[WhereY,WhereX];
    col := FldDef[ndx].col;
    row := FldDef[ndx].row;
    gotoXY(col, row);
    TextAttr := ScrAttr.rc[row,col];
  end;


procedure Process980BakTab;       {BAKTAB}
  var
    row, col, ndx : integer;

  begin
    ErrorLineActive := true;
    row := ErrFldDef.row;
    col := ErrFldDef.col;
    gotoXY(col, row);
    TextAttr := ScrAttr.rc[row,col];
  end;



procedure Process980Leadin;       {LEAD IN}
  begin
          GetNext980;
          case Next980 of
            $01:                             {F7ON}
                 ProcessF7On;

            $02:                             {F7OFF}
                 ProcessF7Off;

            $05:                             {LOCCUR}
                 XmitCursorPosition;

            $06:                             {KBUNL}
                 ProcessKeyBoardUnlock;

            $0E: begin {XMIT}
                   if (WhereX <> 80) or (WhereY <> 24) then
                     XmitErrorLine
                   else
                     XmitScreenData;
                 end;
            $0f:                             {F8ON}
                 ProcessF8On;

            $11:                             {POSCUR}
                 ProcessPositionCursor;

            $12:                             {HOME}
                 GotoXY(1,1);

	    $14:                             {BAKTAB}
		 Process980BakTab;

            $15:                             {KBDLK}
                 ProcessKeyBoardLock;

            $18:                             {F8OFF}
                 ProcessF8Off;

            $1c:                             {CLSCRN}
                 NewScreen;

            $1d:                             {CLRUPD}
                 ClearUnprotected;

            $20:                             {F/A DEF}
                 ProcessFADef;

            $ff: begin {ABORT}
                 end;
          else
            ReportUnexpected(Next980);
          end; {case}
  end;

procedure ProcessXmitChar;     {XMITCHAR}
  begin
    XmtFldDef.row := WhereY;
    XmtFldDef.col := WhereX;
    XmtFldDef.ndx := (XmtFldDef.row - 1) * 80 + XmtFldDef.col;
    XmitCharActive := true;
  end;


procedure ProcessOpBakTab;          {BAKTAB}
  var
    row, col, ndx : integer;

  begin
    if ErrorLineActive then begin
      row := ErrFldDef.row;
      col := ErrFldDef.col;
      end
    else begin
      ndx := ScrBTab.rc[WhereY,WhereX];
      col := FldDef[ndx].col;
      row := FldDef[ndx].row;
      end;
    gotoXY(col, row);
    TextAttr := ScrAttr.rc[row,col];
  end;


procedure ProcessOpTab;          {TAB}
  var
    row, col, ndx : integer;

  begin
    if ErrorLineActive then begin
      row := ErrFldDef.row;
      col := ErrFldDef.col;
      end
    else begin
      ndx := ScrFTab.rc[WhereY,WhereX];
      col := FldDef[ndx].col;
      row := FldDef[ndx].row;
      end;
    gotoXY(col, row);
    TextAttr := ScrAttr.rc[row,col];
  end;

function ValidOpPrintable(OpChar, FieldFA : byte) : boolean;
  begin
    ValidOpPrintable := true;
    fieldFA := fieldFA and $18;
    case (fieldFA and $18) of
      $18 : ;                                    {ALPHANUMERIC}
      $08 :                                      {NUMERIC}
              case chr(OpChar) of
                '0'..'9' : ;
                 '-', ' ', '+', '.', ',' : ;
              else begin
                Beep;
                ValidOpPrintable := false;
                end;
              end;

      $10 : begin
              case chr(OpChar) of
                'A'..'Z' : ;
                'a'..'z' : ;
                ' ', '.', ',': ;

              else begin
                Beep;
                ValidOpPrintable := false;
                end;
              end;
            end;

      $00 : ValidOpPrintable := false;
    end;
  end;


procedure ProcessOpPrintable;    {PRINTABLE}
  var
    col, row, ecol : integer;
    temp : string [81];

  begin
    col := WhereX;
    row := WhereY;
    if (ScrHzFA.rc[row,col] and $c0) <> 0 then ProcessOpTab
    else if not ValidOpPrintable(NextOp, ScrHzFA.rc[row,col]) then begin
      end
    else begin
      TextAttr := ScrAttr.rc[row,col];
      if InsertActive then begin
        ecol := col + 1;
	while (ecol <= 80) and ((ScrHzFA.rc[row,ecol] and $c0) = 0) do
          Inc(ecol);
        Dec(ecol);
        if ecol > col then begin
          temp[0] := chr(ecol - col);
          for ecol := ecol - 1 downto col do
            ScrData.rc[row,ecol+1] := ScrData.rc[row,ecol];
          move(ScrData.rc[row,col+1], temp[1], ord(temp[0]));
          gotoXY(col+1,row);
          write(temp);
          gotoXY(col,row);
          end;
        end;
      write(chr(NextOP));
      ScrData.rc[row,col] := NextOP;
      col := WhereX;
      row := WhereY;
      if (ScrHzFA.rc[row,col] and $c0) <> 0 then ProcessOpTab
      end;
  end;


procedure ProcessInsert;
  var
    row, col :integer;
    savTextAttr : byte;
    temp : string[6];
  begin
    InsertActive := not InsertActive;
    if InsertActive then
      temp := 'INSERT'
    else
      temp := '      ';
    UpdateStatusLine(74,temp);
  end;


procedure ProcessDelete;
  var
    col, row, ndx, ecol : integer;
    temp : string[81];

  begin
    row := WhereY;
    col := WhereX;
    ndx := (row - 1) * 80 + col;
    if (ScrHzFA.ndx[ndx] and $c0) = 0 then begin
      ecol := col + 1;
      while (ecol <= 80) and ((ScrHzFA.rc[row,ecol] and $c0) = 0) do begin
        ScrData.rc[row,ecol-1] := ScrData.rc[row,ecol];
        Inc(ecol);
        end;
      Dec(ecol);
      ScrData.rc[row,ecol] := $20;
      temp[0] := chr(ecol - col + 1);
      move(ScrData.ndx[ndx], temp[1], ecol-col+1);
      TextAttr := ScrAttr.ndx[ndx];
      write(temp);
      gotoXY(col,row);
     end;
  end;


procedure ProcessBackSpace;        {BACKSPACE}
  begin
    gotoXY(WhereX-1,WhereY);
    ProcessDelete;
  end;


procedure ProcessDownArrow;        {DOWN ARROW}
  begin
    if WhereY <> 24 then
      gotoXY(WhereX,WhereY+1);
  end;


procedure LeftScreen;              {CTL LEFT ARROW}
  begin
    gotoXY(1,WhereY);
  end;

procedure RightScreen;             {CTL RIGHT ARROW}
  begin
    gotoXY(80,WhereY);
  end;


procedure ClearEndOfLine;          {CTL END}
  var
    savcol, col, row : integer;

  begin
    row := WhereY;
    savcol := WhereX;
    for col := savcol to 80 do
      if (ScrHzFA.rc[row,col] and $40) = 0 then
        if ScrData.rc[row,col] <> $20 then begin
          TextAttr := ScrAttr.rc[row,col];
          ScrData.rc[row,col] := $20;
          gotoXY(col,row);
          write(' ');
          end;
    gotoXY(savcol,row);
  end;


procedure DeleteLine;            {CTL PAGE UP}
  var
    row, col : integer;

  begin
    if (MxFld = 1) and (FldDef[1].len > 1900) then begin
      col := WhereX;
      row := WhereY;
      window(1,1,80,24);
      gotoXY(col,row);
      DelLine;
      window(1,1,80,25);
      gotoXY(col,row);
      if (row < 24) then begin
        move(ScrData.rc[row+1,1],ScrData.rc[row,1],80*(24-row));
        end;
      FillChar(ScrData.rc[24,1],80,' ');
      end;
  end;


procedure InsertLine;            {CTL PAGE DOWN}
  var
    row, col, i : integer;

  begin
    if (MxFld = 1) and (FldDef[1].len > 1900) then begin
      col := WhereX;
      row := WhereY;
      window(1,1,80,24);
      gotoXY(col,row);
      InsLine;
      window(1,1,80,25);
      gotoXY(col,row);
      if (row < 24) then begin
        for i:= 23 downto row do
          move(ScrData.rc[i,1],ScrData.rc[i+1,1],80);
        end;
      FillChar(ScrData.rc[row,1],80,' ');
      end;
  end;


procedure ProcessFunctionKey;    {FUNCTION KEYS}
  var
    tbuf : array [1..2] of byte;
  begin
    tbuf[1] := FkeyArray[Nextop-($3b+$100)];
    tbuf[2] := ETX;
    WriteBytes(@tbuf[1],2);
  end;

procedure ProcessVideoChange;
  var
    ndx : integer;
    savcol, savrow : integer;
    fieldFA : byte;

  begin
    VideoMode := NextOp - ($100+119);
    savcol := WhereX;
    savrow := WhereY;
    GotoXY(1,1);

    for ndx := 1 to 1920 do begin
      fieldFA := ScrHzFA.ndx[ndx];
      TextAttr := ColorArray[VideoMode,fieldFA and $07];
      if (fieldFA and $20) <> 0 then
        TextAttr := ((TextAttr and $f0) or ((TextAttr shr 4) and $07));
      ScrAttr.ndx[ndx] := TextAttr;
      write(chr(ScrData.ndx[ndx]));
      end;
    gotoXY(savcol,savrow);
  end;

procedure ReportSyntax;

    begin
      Writeln;
      Writeln(
'HAZELTINE MOD 1 EMULATION  Version 1.0');
      writeln(
' Syntax is HZLTN10 <COMPORT BAUDRATE STOPBITS IRQLEVEL PORTADDRESS VIDEO>');
      writeln(
'  e.g.  HZLTN 3 7 0 3 4 1');
      writeln(
'        => COM3:, 9600 baud, 1 stopbit, IRQ 3, Ports 3e8-3ef, Mono');
      writeln('');
      writeln(
'    Available Values are:');
      writeln(
'      COMPORT:     1, 2, 3, 4');
      writeln(
'      BAUDRATE:    0=110, 1=150, 2=300, 3=600, 4=1200,');
      writeln(
'                   5=2400,6=4800,7=9600,8=19200');
      writeln(
'      STOPBITS:    0=1 StopBit, 1=2 StopBits');
      writeln(
'      IRQLEVEL:    4 (usually COM1:),  3 (usually COM2:/COM3:/COM4:)');
      Writeln(
'                   Sometimes available: 2 (PC-Rsvd), 5 (PC-Fixed Disk)');
      writeln(
'      PORTADDRESS: 1=$3F8-$3FF, 2=$2F8-$2FF, 3=$3E8-$3EF, 4=$2E8-$2EF');
      writeln(
'      VIDEO:       1=B/W, 2-',MAXVIDEO,'=Color');
      Halt;
    end;


  procedure Initialize;
    var
       errcode : integer;
       I : integer;

    begin {Initialize}

      if ParamCount <> 6 then
        ReportSyntax;

      Val(ParamStr(1), Portno, errcode);
      if (errcode > 0) or (PortNo < 1) or (PortNo > 4) then
        ReportSyntax;

      Val(ParamStr(2), BaudRate, errcode);
      if (errcode > 0) or (BaudRate < 0) or (BaudRate > 8) then
        ReportSyntax;

      Val(ParamStr(3), StopBits, errcode);
      if (errcode > 0) or (StopBits < 0) or (StopBits > 1) then
        ReportSyntax;

      Val(ParamStr(4), IrqLevel, errcode);
      if (errcode > 0) or (IrqLevel < 2) or (IrqLevel > 5) then
        ReportSyntax;

      Val(Paramstr(5), PortAddress, errcode);
      if (errcode > 0) or (PortAddress < 1) or (PortAddress > 4) then
        ReportSyntax;

      Val(Paramstr(6), VideoMode, errcode);
      if (errcode > 0) or (VideoMode < 0) or (VideoMode > MAXVIDEO) then
        ReportSyntax;

      CheckBreak := false;
      OrigMode := LastMode;
      TextAttr := ColorArray[VideoMode,DFLTHZFA AND $07];
      if VideoMode = 1 then
        TextMode(BW80)
      else
        TextMode(CO80);
   {  CheckSnow := false;  }

      FillChar(Blanks, SizeOf(Blanks), ' ');
      Blanks[0]:=Chr(255);

      ReportError('HAZELTINE MOD 1 EMULATION  Version 1.1');
  {    ReadLn;   }


      OpenPort;
      SetOptions;

      ExitRequest := false;
      ErrorLineActive := false;

      ClearScreenMatrices;
      Cur980Index := 0;
      Max980Index := 0;

      InsertActive := false;
      F7On := false;
      F8On := false;
      KeyBoardLocked := false;



    end;  {Initialize}





var
  i, j, k : integer;


begin

    Initialize;

 {   Write('load the buffer and then press a key to continue');
    Readln;  }

    repeat
      if (ChkNext980) then
        case Next980 of
          $20..$5F,
          $61..$7d :                         {PRINTABLE}
                     if ScrHzFA.rc[WhereY,WhereX] >= $80 then
                       processFAErase
                     else begin
                       ScrData.rc[WhereY,WhereX] := Next980;
                       write(chr(Next980));
                     end;

          $7e :      Process980Leadin;       {LEAD IN}

          $07:       Beep;                   {BELL}

	  $09:       Process980Tab;          {TAB}

          $60:                               {XMITCHAR}
                     ProcessXmitChar;

          $10:                               {DLE - Right Cursor}
                     ;

        else
          ReportUnexpected(Next980);
        end; {case}

     if (ChkNextOp) then
      begin
        case NextOp of
          $20..$7d :                       {PRINTABLE}
                    ProcessOpPrintable;

          $100+$0f :                       {BAKTAB}
		    ProcessOpBakTab;

          $100+$3b..$100+$42:              {FUNCTION KEYS 1..8}
                    ProcessFunctionKey;

          $08:                             {BACKSPACE}
          	    ProcessBackSpace;

          $09,$0d:                         {TAB}
		    ProcessOpTab;

          $1b:                             {ESC - Clear Error Line}
		    ReportError(' ');

          $100+$43..$100+$44:              {FUNCTION KEYS 9..10}
                    ;

          $100+71 :                        {HOME}
                    gotoXY(1,1);

          $100+72 :                        {UP ARROW}
                    gotoXY(WhereX,WhereY-1);

          $100+75 :                        {LEFT ARROW}
                    gotoXY(WhereX-1,WhereY);

          $100+77 :                        {RIGHT ARROW}
                    gotoXY(WhereX+1,WhereY);

          $100+80 :                        {DOWN ARROW}
                    ProcessDownArrow;

          $100+82 :                        {INSERT}
                    ProcessInsert;

          $100+83 :                        {DELETE}
                    ProcessDelete;

          $100+115 :                       {CTL LEFT ARROW}
                    LeftScreen;

          $100+116 :                       {CTL RIGHT ARROW}
                    RightScreen;

          $100+117 :                       {CTL END}
                    ClearEndOfLine;

          $100+118 :                       {CTL PAGE DOWN}
                    InsertLine;

          $100+119 :                       {CTL HOME}
                    OpClearUnprotected;

          $100+120..$100+119+MAXVIDEO :    {Alt 1..}
                    ProcessVideoChange;

          $100+132 :                       {CTL PAGE UP}
                    DeleteLine;

          EXITCHAR:                        {EXIT REQUEST}
                    ;

        else
          ;
        end; {case}
      end;

    until (ExitRequest);

    ClosePort;

    TextMode(OrigMode);

end.