
/* A source program in the C Language.
*
*			SOFTFONT to BITMAP DRAWING (=sf2pat)
*			Version: 2.1  Mar. 1993
*			(c) by David S. Lawyer
*	Converts softfont for Wyse and VT220 terminals to a drawing
*	of a bitmap (a pattern file).  The pattern file may then be
*	edited and converted back into softfont using my fontgen
*	program.  This permits one to edit existng softfonts.  sf2pat
*	means SoftFont to PATtern.
*
*	Copyright (c) by David S. Lawyer, 1993 and may be freely used, 
*	copied, and modified by anyone provided that any significant
*	modifications or improvements be also made freely available.
*	If sold, the purchaser must be informed that this is free software.
*	The author reserves the right to transfer this copyright to 
*	any organization which will freely distribute it (such as the
*	Free Software Foundation).  Do not remove this copyright notice.
*
* Some video display terminals permit users to design their own characters
* or alphabets and download them.  I have written another program "pat2sf"
* to help people do this.  The program you are now reading is named sf2pat.
* It is only useful if you have my "pat2sf" program.  The pat2sf program
* takes bitmap drawings of characters which you (as a font designer) 
* "draw" in a pattern file and converts it to softfont code ready to 
* download to a terminal.  This program does the opposite: It converts
* softfont code a a pattern file.  In order to use and understand it, read
* the documentation: patsf.doc.
*
* The decoding rules used in this program for Wyse and VT220 terminals came
* from the Wyse 99GT Programmer's Guide chapters 7 and 8.
*
* Variable names:  If 2 integers have the same name except one starts with a 
* capital letter, the Capital one has a value 1 greater than its mate.  The
* words character, char, ch, etc. refer to the large characters of a 
* dot-matrix  pattern and not to a byte-character.  A "byte-char" is called
* a byte, letter, etc. although it's declared as a char.
*/ 
/*---------------------------preprocessor stuff-------------------------------*/
#include <stdio.h>   /* for some C compliers you should: #include <stdlib.h>  */
/* CELL is a rectangle which contains a character dot matrix (= bitmap).  
 * Sizes shown for CELLs are in number of pixels (= dots).
 */
#define CELL_HEIGHT_MAX   18	/* Used to allocate array size.		      */
#define CHARS_PER_BAND_MAX 8	/* Ditto.				      */
#define CELL_WIDTH_MAX	   8	/* Ditto.  Note: Wyse widths can't exceed 8.  */
#define MAX_CODE_LENGTH	 100	/* Max length of softfond code for a char.    */
#define FALSE 0
#define TRUE  1
#define BOOL int
/*----------------------external variables------------------------------------*/
extern char *optarg;
extern int optind;
/*--------------------forward declarations------------------------------------*/
			/* FUNCTIONS in this FILE:  besides main()	      */
void draw_band();	/* Draws a band of bitmaps in a pattern file.	      */
void print_band();	/* Prints a band of char-matrix patterns to stdout    */
void not_sf_err();	/* Error message if input file not valid softfont.    */
void V_width_err();	/* Error message for VT220 if wider than max width.   */
void usage();		/* Shows user how to type the command line (& options)*/
/*---------------------------global vars--------------------------------------*/
unsigned char
    band[CHARS_PER_BAND_MAX] [CELL_HEIGHT_MAX] [CELL_WIDTH_MAX + 1];

				/* Several pattern_chars.  
				 * band [char_no] [row] [col]   Each element
				 * is either a * or <space>  "pixel".   +1
				 * for null string terminator.
				 */ 
char 
    letter;		/* a byte-char read from input.			      */
int
    Cell_Height = 0,	/* for cells in pattern file.			      */
    Cell_Width = 8,	/* Default = 8.					      */
    Chars_Per_Band = 8,	/* # of character dot matrices per "large row" (band) */
    Chars_per_band = 7,	/* Chars_Per_Band - 1, due to index_orig. = 0	      */
    band_no = 0,	/* What band is being processed now?		      */
    nchars_read = 0,	/* How many softfontchars. have we read?	      */
    i;			/* General purpose scratch interger. 		      */
FILE
    *fp;		/* pointer to input file			      */
BOOL
    isVt220 = FALSE,	/* Is this softfont for a Vt220 terminal ?	      */
    isWyse  = FALSE,	/* Is this softfont for a Wyse terminal ?	      */
    eof     = FALSE;    /* Did we hit End of File (or junk at end of file) ?  */
/***************************start of main**************************************/
void main(argc, argv)
int argc; 
char **argv;  
{    
BOOL
    preview = FALSE,	/* preview => Don't send many messages to terminal    */
			/* since it's  being used for previewing pattern file */
			/* output.					      */
    std_in = FALSE;	/* Does this program use stdin for pattern file?      */
/*--------------------parse command line--------------------------------------*/
if 	  ( argc == 1 ) usage(argv[0]);
while (( letter = getopt(argc, argv, "VWsph:w:b:") ) != EOF) {
     switch (letter) {
	case 'V': isVt220 = TRUE;
		  break;
	case 'W': isWyse = TRUE;
		  break;
	case 's': fp =  stdin;
		  fprintf(stderr, "Using standard input.\n");
		  std_in = TRUE;
		  break;
	case 'p': preview = TRUE;
		  break;
	case 'h': Cell_Height = atoi(optarg);
		  break;
	case 'w': Cell_Width = atoi(optarg);
		  break;
	case 'b': Chars_Per_Band = atoi(optarg);
		  Chars_per_band = Chars_Per_Band - 1;
		  break;
	case '?': usage(argv[0]);
      }
 }
if (! std_in) {
    if ( (fp = fopen( argv[optind], "r")) == (FILE*) NULL) {
	fprintf(stderr, "Can't open input file: %s\n", argv[optind]);
	usage(argv[0]);
     }  
 }
/*--------------------------check for bad data--------------------------------*/
if ( isWyse || isVt220 ) ;		/* All OK.  Null statement */
  else {				/* You failed to select terminal type */
    fputs("You must either use the -W option for Wyse or the -V option for\
 VT220. \n", stderr);
    exit(-1);
   }
if ( isWyse ) {
    if( Cell_Width > 8 ) {
	fputs("Program can't handle characters for Wyse over width = 8. \n",
	   stderr);
	exit(-1);
     }
    if( Cell_Width > CELL_WIDTH_MAX ) {	   /* For Wyse, cell width is unknown.*/
    fprintf(stderr, "\n###ERROR### Cell width is too large. Try again or\
 modify #define in C program.\nThe current #define sets: Max cell width = %d \n"
	, CELL_WIDTH_MAX);
    exit(-1);
     }
 } 				/* end if isWyse */
if (isVt220) {			/* For VT220, cell height is unknown.	      */
    if( Cell_Height > CELL_HEIGHT_MAX )  {
    fprintf(stderr, "\n###ERROR### Cell height is too large.  Try again or\
 modify #define in C program.\nThe current #define sets: Max cell height = %d\
 \n", CELL_HEIGHT_MAX);
    exit(-1);
     }
 } 				/* end if isVt220 */
if (Chars_Per_Band > CHARS_PER_BAND_MAX) {
    fprintf(stderr, "\n###ERROR### Chars/band is too large.  Fix typo or modify\
 #define in C program. \nThe current #define sets: Max Chars/band = %d \n",
		CHARS_PER_BAND_MAX);
    exit(-1);
}
/*---------------------------call draw_band()---------------------------------*/
if (! preview)		/* If preview, no messages are sent to CRT	      */
    fputs( "Pattern file of character matricies being drawn.  Now doing band\
 (=tall-row):\n", stderr);
while( ! eof ) {		/* eof detected by draw_band().  global var.  */
    band_no ++;
    if (! preview) {
	fprintf( stderr, "%d ", band_no);
	if (  (band_no % 25) == 0  )
	    fputc('\n', stderr);
     }
    draw_band();		/* Draws a band of * dot-matirces to stdout   */
 }
if (! preview) 
    fputs("\n", stderr);
fputs("Finished!  Look at pattern file to see if it came out OK.\n", stderr); 
}
/*----------------------------end of main-------------------------------------*/
/* draw_band() reads the softfont code for several chars, decodes the softfont
 * code to pixels (*'s) and prints a pattern file band on stdout.  It reads
 * the softfont code one line at a time.  One char is expected per line except
 * that comments (or garbage) are allowed at the end of the softfont file.  The
 * header at the start of a VT220 softfont file will be skipped over OK.     
 */
/******************************draw_band()*************************************/
void draw_band() {
register unsigned char
    byte;			/* Byte encoded from a row or partial-col of  */
				/* pixels of a character matrix.	      */
char    
    letter,			/* a test char byte read.		      */
    sixel,			/* a six bit byte (00??????) for VT220's      */
    char_code[MAX_CODE_LENGTH];	/* Store softfont for a character here.	      */
char * 
    ptr;			/* General purpose pointer.		      */
register int
    row, col; 			/* Loop index counter.			      */
int
    char_no,			/* Loop index counter.			      */
    start_row, stop_row,	/* VT220: row range for loops	     	      */
    level,			/* VT220: current level of sixels    	      */
    nscan,			/* No. of strings copied by fscanf.	      */
    bad_count,			/* VT220: no. of non-char intitial lines in sf*/
    width_count,		/* VT220: Width of cell.		      */
    line_length;		/* of a line in softfont file = code of a char*/
static int
    first_line_length;		/* of a line in softfont file = code of a char*/
/*
 * First, the softfont file is scanned and the bytes are decoded, with
 * the resulting * and <space> pixels  put into the  band[][][] array.  
 * For example: 01101101 = 6D (hex.).
 * becomes " ** ** * " in the band[][][] array.  For the VT220 scanning 
 * is down columns rather than along level rows from left to right (Wyse
 * format).  See the documentation for the font generator program.
 * The 01101101 as softfont may either be the two ASCII bytes: 6 and D 
 * (Wyse method) or it may be one ASCII byte: m (= ASCII 6D ) (VT220 method)
 * For VT220 3F must be subtacted from the byte before decoding it to 
 * printed pixels.
 *
 * When pixel characters are put into a band[][][] we say they are "banded".
 * Then the band[][][] is written to the standard output after placing |'s 
 * between characters.  A row of + symobls are substituted for band headings
 * in the generated pattern file.
 */
/*----------------------------for loop on char--------------------------------*/
for (char_no = 0 ; char_no <= Chars_per_band ; char_no ++) {
/*--------------------------test next line of sf------------------------------*/
/* Checks first character on the next line of the softfont file and quits if
 * this char is illegal.  Unless we are at the start of a VT200 softfont, an
 * illegal char is assumed to mean the end of the file and we set eof = TRUE
 * (even though no EOF was read).
 * comments (garbage) at the end of the file.
 */
NEXT_LINE:
    letter = getc(fp);		/* Test first byte ch of next line.	      */
    if ( letter == EOF) {
	eof = TRUE;
	print_band(char_no);
	return;
     }
    if (isVt220) 
	if ( letter < 0x3F ) { 	/* Garbage or header or end of softfont)      */
	    if (nchars_read == 0   &&	 ++ bad_count <= 2) {		
						/* could be header on 1st line*/
		fscanf(fp,"%*[^\n]");		/* skip to next line	      */
		fscanf(fp,"\n");
		goto NEXT_LINE;      /* 12 lines up. Try next line of sf file.*/
	     }
	    eof = TRUE;		/* But some non-sf files will not reach here  */
	 }
    if (isWyse    &&    letter != '\033' )/* \033 = ESC at start of Wyse sf ch*/
	if ( nchars_read == 0 )  		/* sf file is bad	      */
	    not_sf_err();
	  else
	    eof = TRUE;
    if ( eof) {
	print_band(char_no);
	return;
     }
    ungetc(letter, fp);		/* byte-char was OK.  Put it back.	      */
    nchars_read ++;		/* OK to read char code on this line	      */
/*-----------------------read a line of sf------------------------------------*/
    fgets(char_code, MAX_CODE_LENGTH, fp);
    line_length = strlen(char_code);
    if ( line_length == 0) {			/* Likeley end of file.	      */
	eof = TRUE;
	print_band(char_no);
	return;
     }
    if (nchars_read == 1) 
	first_line_length = line_length;
      else if( line_length != first_line_length) {
	    fprintf(stderr, "###ERROR### in softfont file.   After\
 reading %d characters, line \nlength of the encoded softfont char was %d\
 instead of %d \n", nchars_read, line_length, first_line_length);
	    exit(-1);
	 }
/*-------------------------decode softfont to pixels--------------------------*/
/* Still in loop: for (char_no = 0 ; char_no <= Chars_per_band ; char_no ++)  */
/*############################## VT220 #######################################*/
if (isVt220) {
/* If no character height given, we make it multiples of 6.		      */

    ptr = char_code;
    level = 0;				/* Initial top level of sixels	      */
NEXT_LEVEL:
    start_row = 6 * level;
    stop_row = start_row + 5; 
    for (col = 0 ; TRUE ; col ++) {
	switch (byte = *ptr++) { 		/*Any case jumps out of switch*/
	    case '/' : level ++;		/* / marks end of a level     */
		       goto NEXT_LEVEL;		/* repeat for loop above      */
	    case ';' : goto END_CH;		/* ; marks end of a char      */
	    case '\n': not_sf_err();
	    case '\0': not_sf_err();
	 }
	sixel = byte - 0x3F; 		/* If not a "case" should be a sixel  */
	for (row = start_row;  row <= stop_row;  row++) {
	    if ( sixel & 0x01)				/* if bit set	      */
		 band [char_no] [row] [col] = '*';
	      else
		 band [char_no] [row] [col] = ' ';
	    sixel >>= 1;				/* double bit_value   */
	 }	   	/* end for on row.  Got a pixel.  Have decoded a sixel*/

     }	/* end for on col. Got pixel col of a sixel.  Have banded a level.    */
	/* due to goto looping on NEXT_LEVEL Have banded a char.	      */
END_CH:
/*------------------------------calc cell size--------------------------------*/
    if (nchars_read == 1) {
	if (col > CELL_WIDTH_MAX) 
	    V_width_err();
	Cell_Width  = col;		/* last col ++ in for test failure    */
	if( Cell_Height == 0 )		/* user failed to specify it	      */
            Cell_Height = stop_row + 1;	/* + 1 since index origin = 0.	      */
     }
/*------------------------------ add \0 term. --------------------------------*/
    for (row = 0; row <= stop_row;  row++)
	band [char_no] [row] [col] = '\0'; /* last col ++ in for test failure*/
} 	/* endif for VT220 *
/*------------------------------- end VT220 ----------------------------------*/
/* Still in loop: for (char_no = 0 ; char_no <= Chars_per_band ; char_no ++)  */
/*################################## WYSE ####################################*/
if (isWyse) { 
/* Will discard all bank & character
 * number info.  Correct this deficiency later?  getline is presumed to skip
 * over any null padding and read the \n at eol.  It is presumed not to count
 * the nulls returned to line_length.
 */
    ptr = char_code + 6;			/* skip over 6-byte ch header */
    Cell_Height = (line_length -7) / 2;		/* 7 = \n at eol & ch header  */
    for( row = 0;  row < Cell_Height;  row ++) { 
        sscanf(ptr, "%2x", &i);			/* must use int i to read byte*/
	byte = (char)i;
	ptr += 2;
	for (col = 0 ; col < Cell_Width ; col ++) {
	    if ( byte & 0x80)				/* if first bit set   */
		 band [char_no] [row] [col] = '*';
	      else
		 band [char_no] [row] [col] = ' ';
	    byte <<= 1;					/* shift to next bit  */
	 }		  /* end for on cols.  Banded a short row of pixels   */
	 band [char_no] [row] [col] = '\0';		/*Add null string term*/
	     				/* col ++ in last for has put us there*/
 }					/* end for on row.  Have encoded a ch */
}					/* endif for WYSE terminals	      */
/*######################### end WYSE #########################################*/
} 	     /* end for on char_no.  Encode a char. Have encoded a band of chs*/
print_band(char_no);
}
/*-------------------------------end draw_band()------------------------------*/
/***********************************print_band*********************************/
void print_band(nchars) 
int nchars;
{
int row, char_no;
if (nchars == 0)				/* nothing to print	      */
    return;
puts("+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\
++++++");			/* substitute band header.		      */
for (row = 0; row < Cell_Height; row ++) {
    for (char_no = 0; char_no < nchars; char_no ++) {
	  printf("|%s|", band[char_no] [row]);
     }	
     putchar('\n');
 }		/* End for on rows.  Have printed a band array.		      */
}
/*---------------------------end of draw_band---------------------------------*/
/******************************not_sf_err()************************************/
void not_sf_err() {
fprintf(stderr, "###ERROR### Line %d of your softfont file is invalid.  May not\
 be softfont \n or options -W or -V may be wrong.\n", nchars_read);
exit (-1);
}
/*****************************V_width_err()************************************/
void V_width_err() {
fprintf(stderr, "Width of a softfont char too wide.  Max char cell length = %d.\
 Inspect char \n(or line) %d of \nyour softfont file.  Edit souce code #define\
 if you need it \nwider.\n",
	nchars_read, CELL_WIDTH_MAX);
exit(-1);
}
/*********************************usage()**************************************/
/* Prints to term. & exits if no or incorrect options given.		      */
void usage(argv0)
char *argv0;
{
    fprintf(stderr, "Usage: %s [options]  [softfont_file]  > pattern_file\
\n Options:\
\n-V  font is for a VT220 type terminal.\
\n-W  font is for a Wyse type terminal.\
\n-s  use Standard input.\
\n-p  I'm sending output to screen by not typing >.  Suppress any messages."\
						, argv0);
    fprintf(stderr, "\
\n-h 16  Only for VT220.  Char-cell Height=16.  Default is up to 5 too high.\
\n-w  7  Only for Wyse.  Char-cell Width is 7.  Default = 8.\
\n-b  6  chars/Band = 6. (Put 6 dot-matrix chars per line).  Default = 8.\n");
exit (-1);
}
