/*################################ comment ###################################*/
/*		encode.c (A C source code file for BitFontEdit) v. 3.1
*		Copyright 1993-1997 by David S. Lawyer
*		See the manual for conditions of free use 
* The functions in this file are specific to various languages for
* terminals & printers.  This file is used by the main file: scan.c.
* The major function is ?encodeCh() which is passed Char_matrix[][]
* (as a global var.) containing * for "on" pixels and space (' ') for
* blank pixels.  Extensive error checking is done in scan.c so that
* there is no need here to check if Char_matrix[][] contains any invalid
* chars.  It may only have '*', ' ', or nulls as string terminators.
* Presently, Char_matrix is merely a pointer to a slice of band[][][].
*
* Set_Fptrs(): Sets the pointers-to-functions to point to the correct function
*	       for the selected device language.
* lang_err( ): Used in above.  Prints error msg. if lang. doesn't exist.
* ListLangs(): Prints (to stdout) a list of the languages which may 
*	       be encoded and the name to use as an option.
* NOTE: ? is a string representing the language. ? may be VT, Wyse, etc.
* ?EncodeCh(): Encodes a Ch: *'s from Char_matrix[][] to soft-font
* ?SizeChkF(): Checks that the size (height and witdth) of the Chars. in ? 
*	       Lang. is small enough to fit into the array: band[][][].
* ?UsageF():   Prints (to stderr) the options one may use for ? language
* ?EndFontF(): Adds a trailer (Ending) to entire softfont for ? language.
*
* All of the ? functions above are called by pointers which point to them.
* All such function of the same ? language are grouped together in this file.
* To add a new language (different encoding format, etc.) one must write
* the above 4@ ? functions above for the font language and modify the first 2
* non-? functions to include the language.  If a ? function is not need for a
* particular language, one may substitute a "null()" funtion for it in
* Set_Fptrs().  Null function definitions are grouped together in this file.

* ?encodeCh() function:
* This scans the Char_matrix (= band[Char_no][][]) and encodes the pixels 
* (* or <space>) into softfont.  * represents 1 and the background
* pixels <space> represents 0.  Then the resulting bytes of
* softfont (in the correct language) are sent to the standard output.  
*/

#include <stdio.h>	/* for some C compliers use <stdlib.h>		      */
#include <getopt.h>	/* optind and char * optarg defines here	      */
#include "pat2sf.h"
/****************************global vars***************************************/
unsigned char const 			     /* See comment at start of main()*/
	(* Char_matrix)[CELL_WIDTH_MAX + 1]; /* matrix of * pixels for a Ch.  */
extern int	
	Cell_height, 		/* index origin = 0		      */
	Cell_width,		/* index origin = 0		      */
	nChars_coded,	/* How many dot-matrix chars. have we encoded so far? */
	Char_no;	/* in 0..\hars_Per_Band - 1; seq. no. of pattern-char */
/* Functions exported directly (not via setting pointers) from this file:
 * void Set_Fptrs()   Sets values of function pointers.
 * void ListLangs(int);  Lists all font languages supported & name par. to use.
 */
/*************************** file (static) vars *******************************/
static int 
	row, col, 			/* Loop index counter.		      */
	i,				/* general local use int	      */
	Last_Ch_No;			/* If last band is short, index_o. =1 */
static char	
	SpecificFontInfo[80],	/* info on dev lang., incl. pars. user gives  */
	ch,			/* char returned by getopt_long()	      */
	pixel;			/* char read from pattern == * or <sp> or .   */
static unsigned char 
	byte,			/* Byte encoded from a row or partial-col of  */
				/* pixels of a character matrix.	      */
	bit_value;		/* Increment byte with this value	      */
/*-----------------------------forward declarations---------------------------*/
void lang_err( char * lang,  char * valid_lang);	/* err in lang name   */
void ListLangs(int);		/*Lists symbols for all font languages*/
/*############################### VT220 ######################################*/
/*-------------------------------VtEndFont()----------------------------------*/
void VtEndFont(char * Font_info) {
    fputs("\033\\\n", stdout);	/* ESC\RET:  ESC=\033  \=\\    RET=\n	      */
    printf("%s", Font_info);
}
/*-------------------------------VtEncode()-----------------------------------*/
char * VtEncode(int argc, char **argv, int flag)
{
int
    start_row, stop_row,		/* VT220: row range for loops	      */
    Last_level,				/* VT220: # of levels of sixels -1    */
    level;				/* VT200: current level of sixels     */
/*-------------------------------getopt part----------------------------------*/
if (flag) {		/* flag == 1 for getting options but none to get      */
    return "Font for VT220 terminal:\n";  
 }
/*----------------------------encode part-------------------------------------*/
/* A character-matrix is split by horizonal lines into levels of
* height 6.  A vertical 6-bit "byte" is a "sixel".  The top bit is low
* order.  Scanning starts from left to right across the top 6-high band of
* sixels (getting 6 rows per scan).  Then the next band of 6 rows is
* scanned etc.  3F is added to each sixel which converts it into a
* lower-ASCII (printable) 7-bit byte.  '/' separates the bitmap
* encoding of one level from the next. ';' ends the encoding of a
* character.  To load VT220 softfont a long header for the entire font
* is also required (see terminal.doc).  This program doesn't generate
* such a header.
*/
Last_level = Cell_height/6;				/* Index origin = 0   */
for (level = 0;  level <= Last_level ; level ++) {  	/* level of sixels    */
    start_row = 6 * level;
    if (level != Last_level)  stop_row = start_row + 5; 
      else  		  stop_row = Cell_height;    	/* at last level      */
    for (col = 0 ; col <= Cell_width ; col ++) {
	bit_value = 1;
	byte = 0;						/* initialize */
	for (row = start_row;  row <= stop_row;  row++) {
	    pixel = Char_matrix [row] [col];
	    if (pixel == '*')
		byte += bit_value;
	    bit_value <<= 1 ;			/* double bit_value */
	 }	   	/* end for on row. Parsed a pixel.Have encoded a sixel*/
	putchar(byte + 0X3f);		/* +3F makes byte printable   */
      }
	 	/* End for on col to encode a sixel. Have encoded a level     */
	        /* Next check final column which should contain only nulls.   */
	        /* col ++ in last for statement has left us in this final col.*/
/*-----------------------output /; separators---------------------------------*/
	if (level != Last_level)  putchar('/');
	  else			  putchar(';');		/*after last iteration*/
     }	/* end for on this level.  Encoded one level. Have encoded one char.*/
    putchar('\n');			/* End of large_char encoding (EOL)   */
} 	     /* end for on Char_no.  Encode a char. End of function VtEncode()*/
/*----------------------------------VtUsage()---------------------------------*/
void VtUsage(void) {
fputs("\nLANG_OPTIONS for VT terminals:  No special options.  \n", stderr); }
/*############################### end VT220 ##################################*/

/*################################## WYSE ####################################*/
/*-------------------------------WyseEndFont()--------------------------------*/
void WyseEndFont(char * Font_info) {
    printf("%s", Font_info);
}
/*------------------------------WyseSizeChk()---------------------------------*/
void WyseSizeChk(int Cell_Height, int Cell_Width)
{
 if (Cell_Width > 8 ) {
    fputs("Program can't handle characters for Wyse with  cell-width > 8. \n",
	   stderr);
    exit(-1);	      }
}
/*-------------------------------WyseEncode()---------------------------------*/
char * WyseEncode(int argc, char **argv, int flag)
{
static int 
    First_char = 0x20,  /* Wyse: Hex no. of first character. Cmd line opt.    */
    Bank = 1;		/* Wyse: Bank to load font into.  Command line opt.   */
void WyseUsage(void);	/* Forward declaration				      */
/*------------------------------getopts part----------------------------------*/
static struct option long_opts[] = {
{"bank", 1, 0, 'b'}, 			{"first_ch", 1, 0, 'f'},
{0, 0, 0, 0}	/* Zeros req'd per man*/
};
if (flag) {				/* flag == 1 for getting options      */
    while (( ch = getopt_long (argc, argv, "+b:f:", long_opts, &i)) != EOF)
      switch (ch) {
	case 'b': Bank = atoi(optarg);
		  break;
	case 'f': sscanf (optarg, "%x", &First_char );
		  break;
	default : WyseUsage();
		  exit(-1);
       }
    sprintf(SpecificFontInfo, "Font for WYSE terminal: Bank = %d"
	 	"  First_char = %x hex.", Bank, First_char);
    return SpecificFontInfo;		/* Bank and First_char static after reutrn.   */
 }
/*-----------------------------encode part------------------------------------*/
/* A header ESCcA37B means: Put this character shape in cell 7B of bank 3.
*  If a row in a character matrix is .**.**.* it represents the byte
*  01101101 = 6D (hex.).  It's encoded by two ASCII bytes: 6 and D 
*/
     printf("\033cA%1d%02X", Bank, nChars_coded + First_char);  /* char header*/
     for(row = 0; row <= Cell_height; row++) {
	bit_value = 128;					/* = 10000000 */
	byte = 0;						/* initialize */
	for (col = 0 ; col <= Cell_width ; col ++) {
	    pixel = Char_matrix [row] [col];
	    if (pixel == '*')
		byte += bit_value;
	    bit_value >>= 1;				/* halve bit_value    */
	 }		    	/* end for on cols.  Have a byte for a row.   */
	printf("%02X", byte);   /* the bitmap code.  2 ascii bytes.	      */
				/* Next check last col. for null terminator.  */
				/* col ++ in last for has put us there.	      */
     }				/* end for on row.  Have encoded a character  */
    putchar('\031');				/* ^Y trailer		      */
						/* Not needed on all Wyses?   */
    putchar('\n');			/* End of large_char encoding (EOL)   */
} 	     /* Encoded a char. End of Wencode()			      */
/*----------------------------------WyseUsage()-------------------------------*/
void WyseUsage(void) {
fputs("LANG_OPTIONS for WYSE:\n"
  " -b 2 --bank 2     => Bank 2 will get font (default = 1).\n"
  " -f 7 --first_ch 7 => first character is 7 hex (in ASCII code chart."
  " Default \n        is 20 hex (=40 octal, =32 decimal, = the <space>"
  " character in ASCII).\n", 	stderr);
}
/*############################ end WYSE ######################################*/
/*############################ PCL5 (HP) #####################################*/
char * HpEncode(int argc, char **argv, int flag)
{
/*-------------------------------getopt part----------------------------------*/
if (flag) {		/* flag == 1 for getting options but none to get      */
    strcpy(SpecificFontInfo, "Font for HP PCL5 printer :\n");  
    return SpecificFontInfo;
 }
/*-----------------------------encode part------------------------------------*/
/* Each char must be prefixed by an esc seq. including a 14+ character 
 * descriptor but the following code only does the easy part of encoding the
 * data.   
 */
     for(row = 0; row <= Cell_height; row++) {
	byte = 0; 			/* initialize 	 	      */
	bit_value = 128;					/* = 10000000 */
	for (col = 0 ; col <= Cell_width ; col ++) {
	    if (bit_value == 0) {		/* next byte in row	      */
		bit_value = 128;				
		putchar(byte);			/* output binary data byte    */
		byte = 0; 			/* initialize 	 	      */
	     }
	    pixel = Char_matrix [row] [col];
	    if (pixel == '*')
		byte += bit_value;
	    bit_value >>= 1;				/* halve bit_value    */
	 }		    	/* end for on cols.  Have a byte for a row.   */
	putchar(byte);		/* output final binary data byte for row.     */
	pixel = Char_matrix [row] [col];	/* col++ in "for" puts us on 0*/
     }				/* end for on row.  Have encoded a character  */
}
/*----------------------------------HpUsage()---------------------------------*/
void HpUsage(void) {
fputs("LANG_OPTIONS for HP PCL Printer Language:\n"
      "Not yet implemented.   \n", stderr); }
/*############################## end HP PCL5 #################################*/
/*############################# Null funct ##################################*/
void nullsizeChk(int x, int y) {}
/*############################# Set_Fptrs ####################################*/
/* Sets actual pars (pointers to functs) to point to the correct lang of funct.
 * In order to set a pointer, one must know the pointer's address.  So the 
 * actual par. of the function call is an address (&ptr).  The formal par.
 * sets its own value to this address.  The formal par. thus points to an 
 * address of a pointer in the calling program.  It is thus a pointer to a
 * pointer.  This explains the double *'s (**) below.
 * If only C allowed arrays of functions, then encode[Vt] (= VtEncode) could
 * be: 1. used inside a loop as encode[x] with x taking on all values of
 * an enumeration type (not allowed in C since arrays of functions not
 * allowed)  2. encode[Vt] would then be used as the name of a function
 * definition (also not allowed in C).  
 * The switch() below is based only on the first letter of the language.  If
 * many languages are supported and 2 languages have the same first letter,
 * then string compare: strcasecmp() will be utilized for more than just a
 * warning. 
 */
void Set_Fptrs(
void (**usageF)	(void),			  	   void (**endFontF) (char*),
char *(**encodeCh) (int, char**, int),		   void (**sizeChkF) (int, int),
char* lang    )
{
	     (*sizeChkF)  = nullsizeChk;
switch( * lang ){				/* first letter of lang name  */
   case 'V': case 'v':				/* VT (VT220) language (term)*/
	     if ( strcasecmp( lang, "vt") )
		lang_err(lang, "vt");
	     (*encodeCh)  = VtEncode;           
	     (*usageF)    = VtUsage;
	     (*endFontF)  = VtEndFont;
	     break;
   case 'W': case 'w':				/* wyse language (terminal)   */
	     if ( strcasecmp( lang, "wyse") )
		lang_err(lang, "wyse");
	     (*encodeCh)  = WyseEncode;            
	     (*sizeChkF)  = WyseSizeChk;
	     (*usageF)    = WyseUsage;
	     (*endFontF)  = WyseEndFont;
	     break;
   case 'H': case'h':				/* HP PCL5 language (printer) */
	     if ( strcasecmp( lang, "hp") )
		lang_err(lang, "hp");
	     (*encodeCh)  = HpEncode;           
	     (*usageF)    = HpUsage;
	     break;
   case 'l': if ( strcmp( lang, "list" ) )
		printf( "You input \"%s\".  If you meant \"list\" here's the"
							"list:\n", lang); 
	     ListLangs(EXIT);			/* See below.  usage... exits.*/
    default: fprintf(stderr, "\n***ERROR*** Expected font language name but"
						" got:  %s.\n", lang);
	     ListLangs(ERR_EXIT);			/* See below.  usage... exits.*/
 }
}
/******************************* lang_err() ***********************************/
void lang_err( char * lang,  char * valid_lang) {
fprintf( stderr, "WARNING: You input \"%s\" as a device language.  If you"
	" meant %s you're OK.\n", lang, valid_lang);
}
/******************************** ListLangs() *********************************/
void ListLangs( int flag) {		/* flag is ERR_EXIT, EXIT or RETURN   */
fprintf(stderr, "The following are short names for the device LANGuage:\n"
  "list  (display this list)\n"
  "vt     VT220 terminal font.\n"
  "wyse   Wyse terminal font.\n"
  "hp     NOT YET WORKING: HP's  (Hewlett Packard's) PCL5.\n");    	
if (flag == EXIT)      exit(0);
if (flag == ERR_EXIT)  err_exit();
}
/****************************** end of encode.c *******************************/
