/* gifview.c
			********
       GIFtoPS Converter - May 16, 1988 by Scott Hemphill
I wrote this program, and hereby place it in the public domain, i.e.
    there are no copying restrictions of any kind.
			********
    GIFView Utility - 6 December, 1989 by Gregory Reid
Modified program to use vt320 SIXEL graphics rather than output ps.
			********
GIFView 2.0 - 30 October 1991 by doctorgonzo <sboyle@maths.tcd.ie>
Now optimises, allows RGB & output shades alteration, interactive
	    etc. Lotsa new stuff, basically.
			********
	GIFView 2.1 - 28 November 1991 by doctorgonzo
Debugging, deglitching, mucky bits cleaned up, optimised the optimiser.
			********
	GIFView 2.2 - 5 December 1991 by doctorgonzo
   Optimisation estimator added. More debugging. Pipe ability added.
			********
  NOTE: #define TEST will use non-terminal specific output
		for debugging purposes.
			********/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "vtgraph.h"

#define min(x,y) ((x)<(y) ? (x):(y))
#define XY_RATIO (Ratio*(((float)scrwidth)/((float)scrheight)))
#define FALSE 0
#define TRUE 1

#ifdef TEST
#define CLS "\n\n\n"
#define CLR2EOL "             "
#else
#define CLS "\033[H\033[J"
#define CLR2EOL "\033[K"
#endif
#define CR "\015"

typedef char bool;
typedef struct codestruct {
    struct codestruct *prefix;
    unsigned char first,suffix;
} codetype;

unsigned char colormap[256][3], *raster, currchar, BLAKchar, WHITchar;
unsigned int screenwidth, screenheight, top, left, width, height;
char *FName, Arg0[31], Arg1[31], Arg2[31], Arg3[31];
short colortable[256];
codetype *codetable;
FILE *infile;
float Ratio;
bool interleaved, verbose, PipeMode=0, global;
int ImgX, ImgY, NColors, colorbits, lastX, lastY;
int datasize,codesize,codemask, clear, eoi, colorbits;
int RGB_r=30, RGB_g=40, RGB_b=10, FullBrite=66, HalfBrite=33;

void checksignature(), develop_info(), domenu(), fatal(), initcolors(),
    optimise(), outcode(), process(), readextension(), readfile(), readimage(),
    readraster(), readscreen(), show_header(), usage();
int pack(), parse(), develop(), estim_area();
char getyn(), *globtilde(), *malloc();
float msqrt();

void usage() {
    fprintf(stderr,"usage: gifview [-q] input-file\n    -q to switch off verbosity\n");
    exit(-1);
}

void fatal(s)
char *s; {
    fprintf(stderr,"gifview: %s\n",s);
    exit(-1);
}

void checksignature() {
    char buf[6];

    fread(buf,1,6,infile);
    if (strncmp(buf,"GIF",3)) fatal("file is not a GIF file");
}

/* Get information which is global to all the images stored in the file. */

void readscreen() {
    unsigned char buf[7];

    fread(buf,1,7,infile);
    screenwidth = buf[0] + (buf[1] << 8);
    screenheight = buf[2] + (buf[3] << 8);
    global = buf[4] & 0x80;
    if (global) {
	colorbits = (1<<(buf[4] & 0x07)+1);
	fread(colormap,3,colorbits,infile);
    }
}

/* Convert a color map (local or global) to an array of short integers
 * stored in colortable.  RGB is converted to 8-bit grayscale using
 * integer arithmetic.
 */

void initcolors(ncolors)
int ncolors; {
    register unsigned color;
    register i, lr, lg, lb;

    lr=RGB_r*(255/100);
    lg=RGB_g*(255/100);
    lb=RGB_b*(255/100);
    NColors=ncolors;
    for (i=0; i<ncolors; i++) {
	color = lr*colormap[i][0] + lg*colormap[i][1] + lb*colormap[i][2];
	color >>= 8;
	colortable[i] = color;
    }
}

/* Output the bytes associated with a code to the raster array. */

void outcode(p,fill)
register codetype *p;
register unsigned char **fill; {
    if (p->prefix) outcode(p->prefix,fill);
    *(*fill)++ = p->suffix;
}

/* Process a compression code.  "clear" resets the code table.  Otherwise
 * make a new code table entry, and output the bytes associated with the
 * code.
 */

void process(code,fill)
register code;
unsigned char **fill; {
    static avail,oldcode;
    register codetype *p;

    if (code == clear) {
	codesize = datasize + 1;
	codemask = (1 << codesize) - 1;
	avail = clear + 2;
	oldcode = -1;
    } 
    else if (code < avail) {
	outcode(&codetable[code],fill);
	if (oldcode != -1) {
	    p = &codetable[avail++];
	    p->prefix = &codetable[oldcode];
	    p->first = p->prefix->first;
	    p->suffix = codetable[code].first;
	    if (((avail&codemask)==0) && (avail<4096)) {
		codesize++;
		codemask += avail;
	    }
	}
	oldcode = code;
    } 
    else if (code == avail && oldcode != -1) {
	p = &codetable[avail++];
	p->prefix = &codetable[oldcode];
	p->first = p->prefix->first;
	p->suffix = p->first;
	outcode(p,fill);
	if (((avail&codemask)==0) && (avail<4096)) {
	    codesize++;
	    codemask += avail;
	}
	oldcode = code;
    } 
    else {
	fatal("illegal code in raster data");
    }
}

/* Decode a raster image. [doc's note: I don't understand any of this GIF
 * business, but it works.] ;-)
 */

void readraster(width,height)
unsigned width,height; {
    unsigned char *fill = raster;
    unsigned char buf[255];
    register bits=0;
    register unsigned count,datum=0;
    register unsigned char *ch;
    register int code;

    datasize = getc(infile);
    clear = 1 << datasize;
    eoi = clear+1;
    codesize = datasize + 1;
    codemask = (1 << codesize) - 1;
    codetable = (codetype *)malloc(4096*sizeof(codetype));
    if (!codetable) fatal("not enough memory for code table");
    for (code=0; code<clear; code++) {
	codetable[code].prefix = (codetype*)0;
	codetable[code].first = code;
	codetable[code].suffix = code;
    }
    for (count = getc(infile); count > 0; count = getc(infile)) {
	fread(buf,1,count,infile);
	for (ch=buf; count-->0; ch++) {
	    datum += *ch << bits;
	    bits += 8;
	    while (bits >= codesize) {
		code = datum & codemask;
		datum >>= codesize;
		bits -= codesize;
		if (code == eoi) goto exitloop;
		process(code,&fill);
	    }
	}
    }
exitloop:
    if (fill != raster + width*height) fatal("raster has the wrong size");
    free(codetable);
}

/* Read image information (position, size, local color map, etc.) and pass
 * them to the menu.
 */

void readimage() {
    unsigned char buf[9];
    bool local;
    int *interleavetable;
    register i,row;

    if (verbose)
	printf("Building image from GIF data.\n");
    fread(buf,1,9,infile);
    left = buf[0] + (buf[1] << 8);
    top = buf[2] + (buf[3] << 8);
    width = buf[4] + (buf[5] << 8);
    height = buf[6] + (buf[7] << 8);
    ImgX=left+width;
    ImgY=top+height;
#ifdef REAL_RATIO
    Ratio=((float)ImgX)/((float)ImgY);
#else
    Ratio=1.0;
#endif
    local = buf[8] & 0x80;
    interleaved = buf[8] & 0x40;
    if (local)
	colorbits = (1<<(buf[8] & 0x7)+1);
    else if (!global)
	fatal("no colormap present for image");
    initcolors(colorbits);
    raster = (unsigned char*)malloc(width*height);
    if (!raster) fatal("not enough memory for image");
    readraster(width,height);
    if (interleaved) {
	interleavetable = (int *)malloc(height*sizeof(int));
	if (!interleavetable)
	    fatal("not enough memory for interleave table");
	row=0;
	for (i=top; i<top+height; i+=8) interleavetable[i] = row++;
	for (i=top+4; i<top+height; i+=8) interleavetable[i] = row++;
	for (i=top+2; i<top+height; i+=4) interleavetable[i] = row++;
	for (i=top+1; i<top+height; i+=2) interleavetable[i] = row++;
	if (PipeMode) optimise(interleavetable);
	else domenu(interleavetable);
	free(interleavetable);
    } else {
	if (PipeMode) optimise((int *)NULL);
	else domenu((int *)NULL);
    }
    free(raster);
}

/* Draw border and text description, then draw box for 16x6 bitmap. */

void show_header () {
    int ypos;

#ifdef TEST
    return;
#else
    printf ("\033[H\033[J"); /* clear screen */
    printf ("\033[2;59f%s","GIF Viewer 2.2");
    printf ("\033[4;52f%s","By Greg Reid & doctorgonzo");
    printf("\033[1;1f");
    printf("Filename: %s\n",FName);
    printf("Image size:  %dx%dx%d\n",ImgX,ImgY,NColors);
    printf("Display ratio: %3.2f\n",Ratio);
    printf("RGB balance: %d/%d/%d\n",RGB_r,RGB_g,RGB_b);
    printf("Thresholds:  f:%d, h:%d\n",FullBrite,HalfBrite);
    printf("%snterleaved\n",(interleaved) ? "I" : "Non-i");
    printf("Chars used: %d%%\n",(currchar*100)/cells);
    printf ("\033(0");

    printf ("\033[%d;%dflqqqqqqqqqqqqqqqqk", 1, 31);
    for (ypos = 2; ypos < 8; ypos ++) {
	printf ("\033[%d;%dfx", ypos, 31);
	printf ("\033[%d;%dfx", ypos, 48);
    }
    printf ("\033[%d;%dfmqqqqqqqqqqqqqqqqj", 8, 31);
    printf ("\033(B");
#endif
}

/* Develop a image into grwidth*grheight bitmap using anti-aliasing and the
 * image optimisation stuff. Return early if character supply runs out.
 */

int develop(grwidth,grheight,interleavetable)
int grheight,grwidth;
int *interleavetable; {
    register area,sum,col,row,grrow,grcol;
    register float tx_ratio, ty_ratio;
    int mgrrow,right,bot,halfwidth,pcused,pcdone;
    int localFull, localHalf;

    localFull=FullBrite*(255/100);
    localHalf=HalfBrite*(255/100);
    bot=top+height;
    right=left+width;
    halfwidth=grwidth/2; /*  halftone dither */
    if ((halfwidth>right)||(grheight>bot)) {
	if (verbose) {
	    printf("Development %dx%d: higher than original resolution.%s",grwidth/charwidth,grheight/charheight,CR);
	    fflush(stdout);
	}
	return(1);
    }
    currchar=0;
    BLAKchar=WHITchar=255;
    ty_ratio=((float)bot)/((float)grheight);
    tx_ratio=((float)right)/((float)halfwidth);
    if (interleaved) {
	for (mgrrow=0; mgrrow<grheight; mgrrow+=charheight) {
	    if (verbose) {
		pcdone=(mgrrow*100)/grheight;
		pcused=(currchar*100)/cells;
		printf("Development %dx%d: %d%% complete, %d%% chars used (%d)%s",grwidth/charwidth,grheight/charheight,pcdone,pcused,pcused-pcdone,CR);
		fflush(stdout);
	    }
	    for (grrow = mgrrow; grrow < mgrrow+charheight; grrow++) {
		for (grcol = 0; grcol < halfwidth; grcol++) {
		    sum=area=0;
		    for (row=grrow*ty_ratio+top; row<(grrow+1)*ty_ratio+top; row++)
			for (col=grcol*tx_ratio+left; col<(grcol+1)*tx_ratio+left; col++) {
			    area++;
			    sum+=colortable [*(raster+interleavetable[row]*width+col)];
			}
		    sum /= area; /* This area kludge ensures clean shading. */
		    if (sum > localFull) plot (grcol*2+(grrow%2), grrow);
		    if (sum > localHalf) plot (grcol*2+((grrow+1)%2), grrow);
		}
	    }
	    if (pack(mgrrow,0,mgrrow+charheight,grwidth)) {
		if (verbose) {
		    pcdone=(mgrrow*100)/grheight;
		    printf("Development %dx%d: %d%% complete, 100%% chars used (%d)%s",grwidth/charwidth,grheight/charheight,pcdone,100-pcdone,CR);
		    fflush(stdout);
		}
		return(1);
	    }
	}
    } else {
	for (mgrrow=0; mgrrow<grheight; mgrrow+=charheight) {
	    if (verbose) {
		pcdone=(mgrrow*100)/grheight;
		pcused=(currchar*100)/cells;
		printf("Development %dx%d: %d%% complete, %d%% chars used (%d)%s",grwidth/charwidth,grheight/charheight,pcdone,pcused,pcused-pcdone,CR);
		fflush(stdout);
	    }
	    for (grrow = mgrrow; grrow < mgrrow+charheight; grrow++) {
		for (grcol = 0; grcol < halfwidth; grcol++) {
		    area=0;
		    sum=0;
		    for (row=grrow*ty_ratio+top; row<(grrow+1)*ty_ratio+top; row++)
			for (col=grcol*tx_ratio+left; col<(grcol+1)*tx_ratio+left; col++) {
			    area++;
			    sum+=colortable [*(raster+row*width+col)];
			}
		    sum /= area;
		    if (sum > localFull) plot (grcol*2+(grrow%2), grrow);
		    if (sum > localHalf) plot (grcol*2+((grrow+1)%2), grrow);
		}
	    }
	    if (pack(mgrrow,0,mgrrow+charheight,grwidth)) {
		if (verbose) {
		    pcdone=(mgrrow*100)/grheight;
		    printf("Development %dx%d: %d%% complete, 100%% chars used (%d)%s",grwidth/charwidth,grheight/charheight,pcdone,100-pcdone,CR);
		    fflush(stdout);
		}
		return(1);
	    }
	}
    }
    if (verbose) {
	pcused=(currchar*100)/cells;
	printf("Development %dx%d: 100%% complete, %d%% chars used (%d)%s",grwidth/charwidth,grheight/charheight,pcused,pcused-100,CR);
	fflush(stdout);
    }
    return(0);
}

/* Nuts'n'bolts of the optimisation process. Check for duplicates of earlier
 * chars, and don't bother defining them in the dispind*[] arrays if they're
 * already there. Checks plain black and plain white first.
 */

int pack(ptop,pleft,pbot,pright)
int ptop,pleft,pbot,pright; {
    register char a,b,success,l,k;
    register int j,i;

    for (i=ptop/charheight;i<pbot/charheight;i++) {
	for (j=pleft/charwidth;j<(pright+charwidth-1)/charwidth;j++) {
	    a=display[j][i][0][0];
	    b=display[j][i][1][0];
	    success=1;
	    if ((a==0)&&(b==0)) {
		for (l=charwidth-1;l;l--) {
		    if ((display[j][i][0][l]==0)
			&& (display[j][i][1][l]==0)) { success++; }
		    else { l=1; }
		}
		if (success==charwidth) {
		    if (BLAKchar==255) {
			BLAKchar=currchar; success=0;
		    } else
			scrmap[j][i]=BLAKchar;
		    goto SKIP;
		}
	    } else if ((a==63)&&(b==63)) {
		for (l=charwidth-1;l;l--) {
		    if ((display[j][i][0][l]==63)
			&& (display[j][i][1][l]==63)) { success++; }
		    else { l=1; }
		}
		if (success==charwidth) {
		    if (WHITchar==255) {
			WHITchar=currchar; success=0;
		    } else
			scrmap[j][i]=WHITchar;
		    goto SKIP;
		}
	    }
	    for (k=0;k<currchar;k++) {
		if ((display[dispindx[k]][dispindy[k]][0][0]==a)
 		 && (display[dispindx[k]][dispindy[k]][1][0]==b)) {
		    success=1;
		    for (l=charwidth-1;l;l--) {
			if ((display[dispindx[k]][dispindy[k]][0][l]==display[j][i][0][l])
			    && (display[dispindx[k]][dispindy[k]][1][l]==display[j][i][1][l]))
			    { success++; }
			else { l=1; }
		    }
		    if (success==charwidth) {
			scrmap[j][i]=k;
			break;
		    }
		}
	    }
SKIP:	    ;
	    if (success!=charwidth) {
		scrmap[j][i]=currchar;
		dispindx[currchar]=j;
		dispindy[currchar]=i;
		if ((++currchar)>=cells) return(1);
	    }
	}
    }
    return(0);
}

/* Just a little function to glob tildes in file names - handy for
 * UNIX systems as ~ refers to the user's home directory.
 */

char *globtilde(name)
char *name; {
    char fname[255];

    if ((name==NULL)||(*name=='\0')) return(NULL);
    if (*name=='~') {
	name++;
	strcpy(fname,getenv("HOME"));
	(void) strcat(fname,name);
    } else {
	strcpy(fname,name);
    }
    return(fname);
}

/* Thanks to Marc-Michael Brandis <brandis@inf.ethz.ch> for this sqrt()
 * routine... saves having to include the maths library!
 */

float msqrt(x)
float x; {
        float y, q, lastq;

        y = 2.0;
	q = -1.0;
        do {
          lastq = q;
          q = x/y;
          y = (q+y)/2.0;
        } while (q != lastq);
        return(q);
}

/* Optimise output by binary search until redundant chars -> 0. */

void optimise(interleave)
int *interleave; {
    float minX, minY, diffX, diffY, rat, chrs;
    bool lastworked;

    rat=XY_RATIO;
    chrs=(double)cells;
    do { /* ensure min. size'll fit into 96 chars */
	minY=msqrt(chrs/rat);
	minX=rat*minY;
	chrs-=.1;
    } while (minY*minX>(double)cells);
    diffX=(24.0*rat)-minX;
    if (diffX>((float)scrwidth)-minX) {
	diffY=(((float)scrwidth)/rat)-minY;
	diffX=((float)scrwidth)-minX;
    } else {
	diffY=((float)scrheight)-minY;
    }
    clear_image();
    if (develop(((int)(minX+diffX))*charwidth,((int)(minY+diffY))*charheight,interleave)) {
        diffX/=2.0; diffY/=2.0;
	printf("[%dx%d < optimum < %dx%d]%s\n",(int)minX,(int)minY,(int)(minX+2*diffX),(int)(minY+2*diffY),CLR2EOL);
	do {
	    clear_image();
	    if (!(develop(((int)(minX+diffX))*charwidth,((int)(minY+diffY))*charheight,interleave))) {
		lastworked=TRUE;
		minX+=diffX; minY+=diffY;
	    } else {
		lastworked=FALSE;
	    }
	    diffX/=2.0; diffY/=2.0;
	    printf("[%dx%d < optimum < %dx%d]%s\n",(int)minX,(int)minY,(int)(minX+2*diffX),(int)(minY+2*diffY),CLR2EOL);
	} while (diffX>0.25);
	if (!lastworked) {
	    diffX=diffY=0;
	    clear_image();
	    (void) develop(((int)(minX+diffX))*charwidth,((int)(minY+diffY))*charheight,interleave);
	}
    }
    printf("%s",CLS);
    lastX=(int)(minX+diffX); lastY=(int)(minY+diffY);
    show_image(stdout,lastX,lastY,currchar);
}

/* estimate the area gained through optimisation. */

int estim_area(used)
unsigned char used; {
    float szX, denom;

    if (used==0) /* div by zero */
	szX=999;
    else
	szX=((double)cells)*msqrt(XY_RATIO/used);
    return(szX);
}

/* Read a GIF extension block (and do nothing with it). */

void readextension() {
    int code, count;
    char buf[255];

    code = getc(infile);
    while (count = getc(infile)) fread(buf,1,count,infile);
}

void develop_info() {
    printf("Enter:\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n\n",
	"[t]hreshold full half - alter shading thresholds (0 light, 255 dark)",
	"[b]alance red green blue - alter the RGB balance: integers, must add up to 255",
	"[o]ptimise screen size (once shading is acceptable)",
	"[r]atio number - alter image x:y ratio",
	"[return] redraw the image",
	"[s]ave file - output last image to file",
	"[q]uit");
}

/* parse a command line into arguments separated by spaces. */

int parse(ArgLine)
char *ArgLine; {
    register char *A0,*A1,*A2,*A3;

    Arg0[0]='\0';
    Arg1[0]='\0';
    Arg2[0]='\0';
    Arg3[0]='\0';
    A0=ArgLine;
    A1=A2=A3=(char *)NULL;
    if (!(A1=strchr(A0,' '))) {
	(void) strncpy(Arg0,A0,30); Arg0[31]='\0';
	return (1);
    }
    *A1='\0';
    while (*(++A1)==' ') ;
    (void) strncpy(Arg0,A0,30); Arg0[31]='\0';
    if (!(A2=strchr(A1,' '))) {
	(void) strncpy(Arg1,A1,30); Arg1[31]='\0';
	return (2);
    }
    *A2='\0';
    while (*(++A2)==' ') ;
    (void) strncpy(Arg1,A1,30); Arg1[31]='\0';
    if (!(A3=strchr(A2,' '))) {
	(void) strncpy(Arg2,A2,30); Arg2[31]='\0';
	return (3);
    }
    *A3='\0';
    while (*(++A3)==' ') ;
    (void) strncpy(Arg2,A2,30); Arg2[31]='\0';
    (void) strncpy(Arg3,A3,30); Arg3[31]='\0';
    return (4);
}

char getyn() {
    char ArgLine[255];

    if (gets(ArgLine)==NULL) {
	setup_vt(stdout);
	printf("Bye.\n");
	exit(0);
    }
}

void domenu(interleavetable)
int *interleavetable; {
    int nargs;
    bool changed;
    int i1, i2, i3;
    FILE *outFile;
    float d1, projX, projY;
    char ArgLine[255], *outName;

    printf("Developing picture:\n");
    (void) develop(240,72,interleavetable);
    show_header();
    show_image(stdout,16,6,currchar);
    lastX=16;
    lastY=6;
    develop_info();
    changed=FALSE;
    while (1) {
	printf("GIFview> ");
	if (gets(ArgLine)==NULL) {
	    setup_vt(stdout);
	    printf("Bye.\n");
	    exit(0);
	}
	nargs = parse(ArgLine);
	switch (*Arg0) {
	case 'b':
	    if ((nargs!=4)
		  || (sscanf(Arg1,"%d",&i1)+sscanf(Arg2,"%d",&i2)+sscanf(Arg3,"%d",&i3) != 3)
		  || (i1<0)||(i2<0)||(i3<0)||(i1>100)||(i2>100)||(i3>100)) {
		printf("The 3 RGB balance values must be in the range 0<=val<=100 and total 100.\n");
		printf("RGB balance: %d/%d/%d\n",RGB_r,RGB_g,RGB_b);
	    } else if (i1+i2+i3!=100) {
		i3=100-(i1+i2);
		if (i3<0) {
		    printf("The 3 RGB balance values must be in the range 0<=val<=100 and total 100.\n");
		    printf("RGB balance: %d/%d/%d\n",RGB_r,RGB_g,RGB_b);
		} else {
		    printf("Changing B to %d so RGB totals 100.\n",i3);
		    RGB_r = i1; RGB_g = i2; RGB_b = i3; changed=1;
		    initcolors(colorbits);
		}
	    } else {
		RGB_r = i1; RGB_g = i2; RGB_b = i3; changed=1;
		initcolors(colorbits);
	    }
	    break;
	case 'r':
	    if ((nargs==2) && (sscanf(Arg1,"%f",&d1)) && (d1>0)) {
		Ratio=d1;
		changed=1;
	    } else {
		printf("The display ratio controls the x:y ratio of the image displayed. EG: a ratio of\n");
		printf("2.0 gives a display twice as wide as it is high.\n");
		printf("Current ratio setting: %3.2f\n",Ratio);
	    }
	    break;
	case 't':
	    if ((nargs==3)
		  && (sscanf(Arg1,"%d",&i1)+sscanf(Arg2,"%d",&i2)==2)
		  && (i1>=0)&&(i2>=0)&&(i1<=100)&&(i2<=100)&&(i1>=i2)) {
		FullBrite=i1;
		HalfBrite=i2;
		changed=TRUE;
	    } else {
		printf("The 2 threshold values must be in the range 0<=half<=full<=100.\n");
		printf("Shading thresholds - full: %d, half: %d\n",FullBrite,HalfBrite);
	    }
	    break;
	case 'q':
	    if (changed) {
		printf("Are you sure you want to quit? (y/n) [n] ");
		if (getyn()=='y') {
		    setup_vt(stdout);
		    printf("Bye.\n");
		    exit(0);
		}
	    } else {
		setup_vt(stdout);
		printf("Bye.\n");
		exit(0);
	    }
	    break;
	case 'o':
	    if ((lastX==16)&&(lastY==6)) {
		projX = estim_area(currchar)*1.15;
		printf("Approximate optimised image size: %1.0fx%1.0f.\n",projX,projX/XY_RATIO);
		if (currchar>65) {
		    printf("Most of the characters are used in the standard image. As this is the\n");
		    printf("case, optimisation is probably not going to improve the image size greatly.\n");
		    printf("Do you still want to optimise? (y/n) [y] ");
		    if (getyn()=='n')
			break;
		}
	    } else if (!changed) {
		printf("Are you sure you want to re-optimise? (y/n) [y] ");
		if (getyn()=='n')
		    break;
	    }
	    optimise(interleavetable);
	    printf("Chars used: %d%% Size: %dx%d\n",(currchar*100)/cells,lastX,lastY);
	    changed=FALSE;
	    break;
	case 's':
	    if (nargs!=2) {
		printf("You must provide a file name to output!\n");
	    } else {
		if (((outName=globtilde(Arg1))==NULL)||((outFile=fopen(outName,"w"))==NULL)) {
		    printf("Unable to output to %s\n",outName);
		} else {
		    if (changed) {
			printf("Redeveloping picture:\n");
			clear_image();
			(void) develop(240,72,interleavetable);
			show_header();
			show_image(stdout,16,6,currchar);
			lastX=16;
			lastY=6;
			develop_info();
			changed=FALSE;
		    }
		    setup_vt(outFile);
		    show_image(outFile,lastX,lastY,currchar);
		    fclose(outFile);
		    printf("Output saved in %s\n",outName);
		}
	    }
	    break;
	case '\0': /* newline */
	    if (((lastX!=16)||(lastY!=6)) && (!changed)) {
		printf("Are you sure you want to redevelop? (y/n) [n] ");
		if (getyn()=='n')
		    goto NOFANX;
	    }
	    if (((lastX!=16)||(lastY!=6)) || (changed)) {
		printf("Redeveloping picture:\n");
		clear_image();
		(void) develop(240,72,interleavetable);
		lastX=16;
		lastY=6;
		changed=FALSE;
	    }
	    show_header();
	    show_image(stdout,lastX,lastY,currchar);
	    develop_info();
NOFANX:	    ;
	    break;
	default:
	    develop_info();
	    break;
	}
    }
}

void readfile() {
    bool quit = FALSE;
    int ch;

    do {
	ch = getc(infile);
	switch (ch) {
	case '\0':  
	    break;  /* this kludge for non-standard files */
	case ',':
	    setup_vt (stdout);
	    clear_image();
	    readimage();
	    break;
	case ';':
	    quit = TRUE;
	    break;
	case '!':
	    readextension();
	    break;
	default:
	    fatal("illegal GIF block type");
	    break;
	}
    } while (!quit);
}

main(argc,argv)
int argc;
char *argv[]; {
    int ch;
    extern optind;
    extern char *optarg;

    verbose=1;
    while ((ch=getopt(argc,argv,"q"))>0) {
	switch(ch) {
	case '?': usage();
	case 'q': verbose=0;
	}
    }
    if (optind!=argc-1) {
	PipeMode=1;
	infile=fdopen(dup(fileno(stdin)),"r"); /* get GIF file from stdin */
	checksignature();
	readscreen();
	readfile();
	exit(0);
    }
    infile = fopen(argv[optind],"r");
    if (infile==NULL) {
	perror(strcat("gifview: ",argv[optind]));
	exit(-1);
    }
    if ((FName=strrchr(argv[optind],'/'))==0)
	FName=argv[optind];
    else
	FName++;
    setup_vt(stdout);
    checksignature();
    readscreen();
    readfile();
    exit(0);
}
