#
/*
 * PROPRIETARY INFORMATION.  Not to be reproduced, transmitted, or disclosed
 * in any way without permission.
 *
 * UNIX RT-11 SJ Emulator.
 *
 * Produced by Human Computing Resources Corp.,
 * 10 St. Mary Street, Toronto, Ont. Canada
 *
 * Version:	2		Date:	January 1979
 * Author:	Mike Tilson
 * Description:
 *	Various traps are handled here.  The EMT trap is the RT-11 system
 *	call.  It will be interpreted and passed on to various routines
 *	to be handled.
 * Revisions:
 */
#include "defns.h"
#include "decls.h"
#include "stdaddrs.h"
#include "errors.h"

/* utrap is set to non-zero to indicate that it should be called when
 * trap routine exits to simulate trap to a user's trap handler.
 * utrapps sets the ps to call the handler at.
 */
extern word	utrap, utrapps;
boolean trpflg, sfpaflg;
word sfpahandler, traphandler;	/* user routines to handle various traps */

#define BUS_VEC	04
#define ILL_VEC	010
#define IOT_VEC	020
#define TRAP_VEC 034
#define FLT_VEC	0244
#define BP_VEC	014


/*
 * some defines to allow accessing of a few things by name
 */
#define r5	((&r4)[-2])
#define r0	oldsp[0]
#define oldpc	oldsp[1]
#define oldps	oldsp[2]

/*
 * RT-11 System Call
 */
rt_syscall(r4, r3, r2, r1, oldsp)
	word *oldsp;
{
	register int code;
	register word *ip;
	register int channel;
	extern byte lglintrnl;
	extern word retps;

	signal(EMT_TRAP, rt_syscall);

	/* fetch the EMT code */
	ip = oldpc;
	code = ip[-1];
	/* if the "trapcatch" feature is in operation, this instruction
	 * may be a TRAP rather than an EMT (trapcatch will have requested
	 * an EMT signal when a TRAP is detected in the user code).  If
	 * we have a TRAP, execute the user's TRAP handling code, if he has
	 * filled in the trap vector.
	 */
	if((code&0177400)==0104400) {
		if(chkvec(TRAP_VEC))
			return;
		error(TRAPTRAP);
	}

	/* initialize */
	rt_error = NO;
	retval = NO;
	code =& 0377;
	switch(code) {
	case 0377:
		error(RESERVED);
		break;

	case 0376:
		if(ip == &lglintrnl) {
			/* if emt came from the right spot, then we
			 * are doing a mfps routine.  Save the PS and
			 * return.
			 */
			retps = oldps;
			return;
		}
		error(INTRNL_EMT);
		break;

	case 0375:
		arglist(r0);
		break;

	case 0374:
		onearg(r0.hibyte&0377, r0.lobyte&0377);
		break;

	default:
		if(code>=0360 && code<=0373)
			error(INTRNL_EMT);
		else if(code>=0340 && code<=0357) {
			stackarg(code, r0, oldsp+3);
			/*
			 * WARNING: "stackarg" calls twiddle the stack,
			 * so "oldsp" is no longer valid.
			 */
		}
		else {
			/*
			 * As of RT11 Version 3B, those turkeys at DEC
			 * were still distributing major items of system
			 * software (e.g. MACRO, LINK) which used V1
			 * system calls.  This is the minimum set
			 * needed to run some of those programs.
			 */
			channel = code&017;
			code =& ~017;
			switch(code) {
			case 0200:
			case 0220:
				pushsp(r0);
				/* at this point the user stack looks like the
				 * "area" arg of a normal read, except that it
				 * is missing the first word.  That word will
				 * not be accessed, so rtread won't notice, so
				 * long as we pass a properly fudged address.
				 *
				 * many of these system calls are fudged.
				 */
				if(code==0200)
					rtread(channel, oldsp+2);
				else
					rtwrite(channel, oldsp+2);
				stackadjust(-4);
				break;

			case 040:
				pushsp(r0);
				enter(channel, oldsp+2);
				stackadjust(-2);
				break;

			case 0240:
				rtwait(channel);
				break;

			case 020:
				lookup(channel, &r0-1);
				break;

			case 0:
				delete(channel, &r0-1);
				break;

			case 0120:
				savestatus(channel, &r0-1);
				break;

			case 0140:
				reopen(channel, &r0-1);
				break;

			case 0100:
				rename(channel, &r0-1);
				break;

			case 0160:
				rtclose(channel);
				break;

			default:
				error(V1EMT);
			}
		}
		break;
	}
	if(retval) {
		r0 = rvalue;
	}
	if(rt_error) {
		oldps =| CARRY;
		rt_error--;
		ERRBYT->charac = rt_error;
		if(reporterrors)
			message("*", RETERR, rt_error);
	}
	else {
		oldps =& ~CARRY;
		ERRBYT->charac = 0;
	}
}

/*
 * Illegal instruction
 */
ill_instruc(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	/* signal catching does not need to be reset for this one */
	if(trpflg) {
		trpflg = NO;
		utrap = traphandler;
		utrapps = CARRY;
		return;
	}
	if(chkvec(ILL_VEC))
		return;
	error(ILLTRAP);
}

/*
 * Trace trap
 */
trace_trap(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	/****** Debugging code for tracking down certain
	 ****** horrible RT-11 programs:
	register int *ip;
	ip = oldpc;
	if(((*ip)&0177000)==0104000) {
		printf("TRAP about to be executed at pc=%o\n", uloc);
		core();
	}
	*********
	*********/
	/* signal catching does not need to be reset for this one */
	if(chkvec(BP_VEC))
		return;
	error(BPTRAP);
}

/*
 * IOT trap
 */
iot_trap(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	if(chkvec(IOT_VEC)) {
		signal(IOT_TRAP, iot_trap);
		return;
	}
	error(IOTRAP);
}

/*
 * Floating point exception
 */
flt_excp(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	signal(FLT_EXCP, flt_excp);
	if(sfpaflg) {
		sfpaflg = NO;
		utrap = sfpahandler;
		utrapps = 0;
		return;
	}
	if(chkvec(FLT_VEC))
		return;
	error(FLT_TRAP);
}

/*
 * "Bus error" (trap thru 4)
 */
bus_err(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	signal(BUS_ERR, bus_err);
	if(trpflg) {
		trpflg = NO;
		utrap = traphandler;
		utrapps = 0;
		return;
	}
	if(chkvec(BUS_VEC))
		return;
	error(TRAP4);
}

/*
 * Memory management violation (nxm memory, usually due to a reference
 * to the I/O page.)
 */
seg_err(r4, r3, r2, r1, oldsp)
	int *oldsp;
{
	error(IOPG_REF);
}


/*
 * RT-11 user intercepts for several traps
 */

/*
 * catch floating point exception
 */
sfpa(p)
	word *p;
{
	sfpaflg = sfpahandler = p[1];
}

/*
 * intercept bus error and illegal instruction traps
 */
trpset(p)
	word *p;
{
	trpflg = traphandler = p[1];
}



/*
 * when we get a trap, see if the user has filled in a vector
 * for the trap.
 */
struct iovec {
	word	vpc;
	word	vps;
};
chkvec(vec)
	struct iovec *vec;
{
	if(vec->vpc) {
		utrap = vec->vpc;
		utrapps = vec->vps;
		return(YES);
	}
	return(NO);
}
