/* $PDP11: dnbridge_bridge.c,v 1.12 2013/06/29 18:06:55 form Exp $ */

/*
 * Copyright (c) 2009, 2013 Oleg Safiullin <form@pdp-11.org.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <sys/queue.h>
#include <sys/tree.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <net/if.h>
#ifdef __linux__
#include <linux/if_tun.h>
#endif
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#ifdef VDE
#include <libvdeplug.h>
#endif
#include <netdb.h>
#include <pcap.h>
#ifdef __linux__
#include <signal.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "dnbridge.h"
#include "missing.h"


struct bridge_nodes_list bridge_pool = LIST_HEAD_INITIALIZER(&bridge_pool);
struct bridge_nodes_tree bridge_nodes = SPLAY_INITIALIZER(&bridge_nodes);
struct bridge_ports_tree bridge_ports = RB_INITIALIZER(&bridge_ports);
struct bridge_hosts_tree bridge_hosts = SPLAY_INITIALIZER(&bridge_hosts);
static struct bridge_node *nodes;
static char error[BRIDGE_ERROR_STRLEN];
static int bridge_debug;
static int sock = -1;

static struct itimerval itv = {
	{ BRIDGE_EXPIRE_TIME / 4, 0 },
	{ BRIDGE_EXPIRE_TIME / 4, 0 }
};

static struct event ev_udp;
static struct event ev_sighup;
static struct event ev_sigint;
static struct event ev_sigterm;
static struct event ev_sigalrm;


static __inline int
port_compare(const struct bridge_port *a, const struct bridge_port *b)
{
	return (strcmp(a->bp_name, b->bp_name));
}

static __inline int
node_compare(const struct bridge_node *a, const struct bridge_node *b)
{
	return (memcmp(a->bn_addr, b->bn_addr, sizeof(a->bn_addr)));
}

static __inline int
host_compare(const struct bridge_host *a, const struct bridge_host *b)
{
	int diff;

	if ((diff = memcmp(&a->bh_addr, &b->bh_addr, sizeof(a->bh_addr))) != 0)
		return (diff);
	return (memcmp(&a->bh_port, &b->bh_port, sizeof(b->bh_port)));
}

RB_PROTOTYPE(bridge_ports_tree, bridge_port, bp_entry, port_compare)
RB_GENERATE(bridge_ports_tree, bridge_port, bp_entry, port_compare)

SPLAY_PROTOTYPE(bridge_nodes_tree, bridge_node, bn_entry, node_compare)
SPLAY_GENERATE(bridge_nodes_tree, bridge_node, bn_entry, node_compare)

SPLAY_PROTOTYPE(bridge_hosts_tree, bridge_host, bh_entry, host_compare)
SPLAY_GENERATE(bridge_hosts_tree, bridge_host, bh_entry, host_compare)

static const char *
node_address(const u_int8_t addr[BRIDGE_NODE_ADDRLEN])
{
	static char addrstr[32];

	if (addr[0] == 0xaa && addr[1] == 0 && addr[2] == 4 && addr[3] == 0) {
		u_int area = addr[5] >> 2;
		u_int node = (u_int)addr[4] | ((addr[5] & 3) << 8);

		(void)snprintf(addrstr, sizeof(addrstr),
		    "%02x:%02x:%02x:%02x:%02x:%02x (%u.%u)", addr[0], addr[1],
		    addr[2], addr[3], addr[4], addr[5], area, node);
	} else
		(void)snprintf(addrstr, sizeof(addrstr),
		    "%02x:%02x:%02x:%02x:%02x:%02x", addr[0], addr[1],
		    addr[2], addr[3], addr[4], addr[5]);

	return (addrstr);
}

static ssize_t
bridge_send(const struct bridge_port *bp, time_t ts, const void *buf,
    size_t len)
{
	const struct packet_header *ph = buf;
	struct bridge_host *bh;
	struct sockaddr_in sin;

	switch (ntohs(ph->ph_type)) {
	case BRIDGE_PKT_MOPDL:
	case BRIDGE_PKT_MOPRC:
		if (!(bp->bp_flags & BPF_MOP))
			goto nosnd;
		break;
	case BRIDGE_PKT_DECNET:
		if (!(bp->bp_flags & BPF_DECNET))
			goto nosnd;
		break;
	case BRIDGE_PKT_LAT:
		if (!(bp->bp_flags & BPF_LAT))
			goto nosnd;
		break;
	case BRIDGE_PKT_LOOPBACK:
		break;
	}

	if ((bp->bp_flags & BPF_PASSIVE) &&
	    ts - bp->bp_time >= BRIDGE_EXPIRE_TIME) {
	nosnd:	if (bridge_debug > 2)
			(void)fprintf(stderr,
			    "Not sending packet to %s, port %s\n",
			    node_address(ph->ph_dst), bp->bp_name);
		return (len);
	}

	if (bridge_debug > 2)
		(void)fprintf(stderr, "Sending packet to %s, port %s\n",
		    node_address(ph->ph_dst), bp->bp_name);

	switch (bp->bp_type) {
	case BPT_UDP:
		bh = bp->bp_desc;

		/* XXX: move struct to host */
		bzero(&sin, sizeof(sin));
		sin.sin_family = AF_INET;
#ifndef MISSING_SA_LEN
		sin.sin_len = sizeof(sin);
#endif
		sin.sin_port = bh->bh_port;
		sin.sin_addr = bh->bh_addr;

		return (sendto(sock, buf, len, 0, (struct sockaddr *)&sin,
		    sizeof(sin)));
	case BPT_PCAP:
		return (pcap_inject(bp->bp_desc, buf, len));
#ifdef VDE
	case BPT_VDE:
		return (vde_send(bp->bp_desc, buf, len, 0));
#endif
	default:
		return (write(bp->bp_fd, buf, len));
	}
}

/*ARGSUSED*/
static void
bridge_io(int fd, short event __attribute__((__unused__)), void *arg)
{
	struct bridge_port *b, *bp = arg;
	u_int8_t pkt[BRIDGE_MAX_PKTLEN + 2];
	struct bridge_node *sn, *dn, bn;
	const struct packet_header *ph;
	struct pcap_pkthdr h;
	ssize_t len = 0;

	if (bp == NULL) {
		struct sockaddr_in sin;
		socklen_t slen = sizeof(sin);
		struct bridge_host *bh, hh;

		if ((len = recvfrom(fd, pkt, sizeof(pkt), 0,
		    (struct sockaddr *)&sin, &slen)) < 0) {
			syslog(LOG_ERR, "recvfrom: %m");
			return;
		}

		hh.bh_addr = sin.sin_addr;
		hh.bh_port = sin.sin_port;
		if ((bh = SPLAY_FIND(bridge_hosts_tree, &bridge_hosts,
		    &hh)) == NULL) {
			if (bridge_debug > 1)
				(void)fprintf(stderr,
				    "Unexpected packet from %s:%u\n",
				    inet_ntoa(sin.sin_addr),
				    ntohs(sin.sin_port));
			return;
		}

		bp = bh->bh_bport;
	}

	switch (bp->bp_type) {
	case BPT_PCAP:
		if ((ph = (const struct packet_header *)pcap_next(bp->bp_desc,
		    &h)) == NULL)
			return;
		len = h.caplen;
		break;
	case BPT_TAP:
		if ((len = read(fd, pkt, sizeof(pkt))) < 0) {
			syslog(LOG_ERR, "(%s) read: %m", bp->bp_name);
			return;
		}
		/*FALLTHROUGH*/
	case BPT_UDP:
		ph = (const struct packet_header *)pkt;
		break;
#ifdef VDE
	case BPT_VDE:
		if ((len = vde_recv(bp->bp_desc, pkt, sizeof(pkt), 0)) < 0) {
			syslog(LOG_ERR, "(%s) read: %m", bp->bp_name);
			return;
		}

		ph = (const struct packet_header *)pkt;
		break;
#endif
	default:
		return;
	}

	if (len < BRIDGE_MIN_PKTLEN || len > BRIDGE_MAX_PKTLEN) {
		if (bridge_debug)
			(void)fprintf(stderr,
			    "(%s) read: Invalid packet size %lu\n",
			    bp->bp_name, len);
		return;
	}

	switch (ntohs(ph->ph_type)) {
	case BRIDGE_PKT_MOPDL:
	case BRIDGE_PKT_MOPRC:
		if (!(bp->bp_flags & BPF_MOP))
			return;
		break;
	case BRIDGE_PKT_DECNET:
		if (!(bp->bp_flags & BPF_DECNET))
			return;
		break;
	case BRIDGE_PKT_LAT:
		if (!(bp->bp_flags & BPF_LAT))
			return;
		break;
	case BRIDGE_PKT_LOOPBACK:
		break;
	default:
		if (bridge_debug > 1)
			(void)fprintf(stderr, "(%s) Unsupported protocol %x\n",
			    bp->bp_name, ntohs(ph->ph_type));
		return;
	}

	/* multicast/broadcast source */
	if (ph->ph_src[0] & 1) {
		if (bridge_debug)
			(void)fprintf(stderr,
			    "(%s) Multicast/broadcast source %s\n",
			    bp->bp_name, node_address(ph->ph_src));
		return;
	}

	(void)time(&bp->bp_time);

	bcopy(ph->ph_src, bn.bn_addr, sizeof(bn.bn_addr));
	if ((sn = SPLAY_FIND(bridge_nodes_tree, &bridge_nodes, &bn)) != NULL) {
		if (sn->bn_port != bp) {
			if (bridge_debug)
				(void)fprintf(stderr,
				    "(%s) Moved node %s from port %s\n",
				    bp->bp_name, node_address(ph->ph_src),
				    sn->bn_port->bp_name);
			LIST_REMOVE(sn, bn_list);
			LIST_INSERT_HEAD(&bp->bp_nodes, sn, bn_list);
			sn->bn_port = bp;
			/* XXX: possible loop */
		}

		sn->bn_time = bp->bp_time;
	} else if ((sn = LIST_FIRST(&bridge_pool)) != NULL) {
		bcopy(ph->ph_src, sn->bn_addr, sizeof(sn->bn_addr));
		sn->bn_time = bp->bp_time;
		sn->bn_port = bp;
		LIST_REMOVE(sn, bn_list);
		LIST_INSERT_HEAD(&bp->bp_nodes, sn, bn_list);
		(void)SPLAY_INSERT(bridge_nodes_tree, &bridge_nodes, sn);
		if (bridge_debug)
			(void)fprintf(stderr, "(%s) Added node %s\n",
			    bp->bp_name, node_address(ph->ph_src));
	} else if (bridge_debug > 1)
		(void)fprintf(stderr, "(%s) No room for node %s\n",
		    bp->bp_name, node_address(ph->ph_src));

	if (!(ph->ph_dst[0] & 1)) {
		bcopy(ph->ph_dst, bn.bn_addr, sizeof(bn.bn_addr));
		dn = SPLAY_FIND(bridge_nodes_tree, &bridge_nodes, &bn);
	} else
		dn = NULL;

	if (dn != NULL) {
		/* XXX: check for errors */
		if (dn->bn_port != bp)
			(void)bridge_send(dn->bn_port, bp->bp_time, ph, len);
	} else {
		if (bridge_debug > 2)
			(void)fprintf(stderr,
			    "(%s) Broadcasting packet to %s\n", bp->bp_name,
			    node_address(ph->ph_dst));

		RB_FOREACH(b, bridge_ports_tree, &bridge_ports)
			/* XXX: log errors */
			if (bp != b)
				(void)bridge_send(b, bp->bp_time, ph, len);
	}
}

static __dead void
bridge_shutdown(void)
{
	struct bridge_port *bp;

	(void)setitimer(ITIMER_REAL, NULL, NULL);

	while ((bp = RB_MIN(bridge_ports_tree, &bridge_ports)) != NULL)
		bridge_del(bp);

	_exit(0);
}

/*ARGSUSED*/
static void
bridge_sig_handler(int sig, short event __attribute__((__unused__)),
    void *arg __attribute__((__unused__)))
{
	struct bridge_node *bn, *nxt;
	time_t tm;

	switch (sig) {
	case SIGHUP:
	case SIGINT:
	case SIGTERM:
		bridge_shutdown();
		/*NOTREACHED*/
	case SIGALRM:
		(void)time(&tm);

		for (bn = SPLAY_MIN(bridge_nodes_tree, &bridge_nodes);
		    bn != NULL; bn = nxt) {
			nxt = SPLAY_NEXT(bridge_nodes_tree, &bridge_nodes, bn);

			if (tm - bn->bn_time >= BRIDGE_EXPIRE_TIME) {
				(void)SPLAY_REMOVE(bridge_nodes_tree,
				    &bridge_nodes, bn);
				LIST_REMOVE(bn, bn_list);
				LIST_INSERT_HEAD(&bridge_pool, bn, bn_list);

				if (bridge_debug)
					(void)fprintf(stderr,
					    "(%s) Dropped node %s\n",
					    bn->bn_port->bp_name,
					    node_address(bn->bn_addr));
			}
		}
		break;
	}
}

static int
get_sockaddr(struct sockaddr_in *sin, const char *host, const char *port)
{
	struct hostent *he;
	struct servent *se;

	bzero(sin, sizeof(*sin));

	if (port == NULL)
		port = DEF_UDP_PORT;

	if (inet_pton(AF_INET, host, &sin->sin_addr) != 1) {
		if ((he = gethostbyname(host)) == NULL) {
			(void)snprintf(error, sizeof(error), "%s: %s", host,
			    hstrerror(h_errno));
			return (-1);
		}

		bcopy(he->h_addr, &sin->sin_addr, sizeof(sin->sin_addr));
	}

	if ((se = getservbyname(port, "udp")) == NULL) {
		const char *errstr;

		sin->sin_port = htons(strtonum(port, 1, 65535, &errstr));
		if (errstr != NULL) {
			(void)snprintf(error, sizeof(error), "%s: %s", port,
			    errstr);
			return (-1);
		}
	} else
		sin->sin_port = se->s_port;

	sin->sin_family = AF_INET;
#ifndef MISSING_SA_LEN
	sin->sin_len = sizeof(*sin);
#endif

	return (0);
}

const char *
bridge_error(void)
{
	return (error);
}

int
bridge_init(int debug, int lowdelay, size_t maxnodes, const char *host,
    const char *port)
{
	struct sockaddr_in sin;
	size_t i;

	if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
		(void)snprintf(error, sizeof(error), "socket: %s",
		    strerror(errno));
		return (-1);
	}

	if (lowdelay) {
		int tos = IPTOS_LOWDELAY;

		if (setsockopt(sock, IPPROTO_IP, IP_TOS, &tos,
		    sizeof(tos)) < 0) {
			(void)snprintf(error, sizeof(error),
			    "IPTOS_LOWDELAY: %s", strerror(errno));
			(void)close(sock);
			return (-1);
		}
	}

	if (get_sockaddr(&sin, host, port) < 0)
		return (-1);

	if (bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		(void)snprintf(error, sizeof(error), "bind: %s",
		    strerror(errno));
		(void)close(sock);
		return (-1);
	}

	if ((nodes = calloc(maxnodes, sizeof(*nodes))) == NULL) {
		(void)close(sock);
		(void)snprintf(error, sizeof(error), "%s", strerror(errno));
		return (-1);
	}

	for (i = 0; i < maxnodes; i++)
		LIST_INSERT_HEAD(&bridge_pool, &nodes[i], bn_list);

	bridge_debug = debug;
	(void)event_init();

	signal_set(&ev_sighup, SIGHUP, bridge_sig_handler, NULL);
	signal_add(&ev_sighup, NULL);

	signal_set(&ev_sigint, SIGINT, bridge_sig_handler, NULL);
	signal_add(&ev_sigint, NULL);

	signal_set(&ev_sigterm, SIGTERM, bridge_sig_handler, NULL);
	signal_add(&ev_sigterm, NULL);

	signal_set(&ev_sigalrm, SIGALRM, bridge_sig_handler, NULL);
	signal_add(&ev_sigalrm, NULL);

	event_set(&ev_udp, sock, EV_READ | EV_PERSIST, bridge_io, NULL);
	if (event_add(&ev_udp, NULL) < 0) {
		(void)snprintf(error, sizeof(error), "event_add: %s",
		    strerror(errno));
		(void)close(sock);
		signal_del(&ev_sighup);
		signal_del(&ev_sigint);
		signal_del(&ev_sigterm);
		signal_del(&ev_sigalrm);
		return (-1);
	}

	if (debug)
		(void)fprintf(stderr,
		    "Bridge initialized, socket: %s:%u, node cache: %lu\n",
		    inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), maxnodes);

	return (0);
}

struct bridge_port *
bridge_add_tap(const char *interface, u_int32_t flags)
{
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	char device[MAXPATHLEN];
#endif
	struct bridge_port *bp;
	struct ifreq ifr;
#ifdef __linux__
	int on = 1;
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__)
	if (strncmp(interface, "tap", 3) != 0) {
#endif
#ifdef __OpenBSD__
	if (strncmp(interface, "tun", 3) != 0) {
#endif
#ifdef __linux__
	if (strncmp(interface, "tap", 3) != 0 &&
	    strncmp(interface, "tun", 3) != 0) {
#endif
		errno = EINVAL;
		(void)snprintf(error, sizeof(error), "%s: %s", interface,
		    strerror(errno));
			return (NULL);
	}

	if ((bp = calloc(1, sizeof(*bp))) == NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(errno));
		return (NULL);
	}

 	(void)strlcpy(bp->bp_name, interface, sizeof(bp->bp_name));
	if (RB_FIND(bridge_ports_tree, &bridge_ports, bp) != NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(EEXIST));
	error:	free(bp);
		return (NULL);
	}

	bzero(&ifr, sizeof(ifr));
	if (strlcpy(ifr.ifr_name, interface,
	    sizeof(ifr.ifr_name)) >= sizeof(ifr.ifr_name)) {
		(void)snprintf(error, sizeof(error), "%s: %s", interface,
		    strerror(errno));
		goto error;
	}

#ifdef __NetBSD__
	if (ioctl(sock, SIOCIFCREATE, &ifr) < 0) {
		if (errno != EEXIST) {
			(void)snprintf(error, sizeof(error),
			    "SIOCIFCREATE: %s: %s", interface, strerror(errno));
			goto error;
		}
	} else
		bp->bp_flags |= BPF_DESTROY;
#endif
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	(void)snprintf(device, sizeof(device), "/dev/%s", interface);
	if ((bp->bp_fd = open(device, O_RDWR | O_NONBLOCK)) < 0) {
		(void)snprintf(error, sizeof(error), "open: %s: %s", device,
		    strerror(errno));
		goto error;
	}

#ifdef __OpenBSD__
	if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
		(void)snprintf(error, sizeof(error), "SIOCGIFFLAGS: %s: %s",
		    interface, strerror(errno));
		goto error;
	}

	ifr.ifr_flags |= IFF_LINK0;

	if (ioctl(sock, SIOCSIFFLAGS, &ifr) < 0) {
		(void)snprintf(error, sizeof(error), "IFF_LINK0: %s: %s",
		    interface, strerror(errno));
		goto error;
	}
#endif	/* __OpenBSD__ */
#endif	/* __FreeBSD__ || __NetBSD__ || __OpenBSD__ */
#ifdef __linux__
	if ((bp->bp_fd = open("/dev/net/tun", O_RDWR | O_NONBLOCK)) < 0) {
		(void)snprintf(error, sizeof(error), "open: /dev/net/tun %s",
		    strerror(errno));
		goto error;
	}

	ifr.ifr_flags |= IFF_TAP | IFF_NO_PI;
	if (ioctl(bp->bp_fd, TUNSETIFF, &ifr) < 0) {
		(void)snprintf(error, sizeof(error), "TUSETIFF: %s: %s",
		    interface, strerror(errno));
		goto error;
	}

	if (ioctl(bp->bp_fd, TUNSETPERSIST, &on) < 0) {
		(void)snprintf(error, sizeof(error), "TUSETPERSIST: %s: %s",
		    interface, strerror(errno));
		goto error;
	}
#endif	/* __linux__ */

	if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
		(void)snprintf(error, sizeof(error), "SIOCGIFFLAGS: %s: %s",
		    interface, strerror(errno));
		goto error;
	}

	ifr.ifr_flags |= IFF_UP;
	if (ioctl(sock, SIOCSIFFLAGS, &ifr) < 0) {
		(void)snprintf(error, sizeof(error), "IFF_UP: %s: %s",
		    interface, strerror(errno));
		goto error;
	}

	bp->bp_type = BPT_TAP;
	bp->bp_flags |= flags;
	LIST_INIT(&bp->bp_nodes);

	event_set(&bp->bp_event, bp->bp_fd, EV_READ | EV_PERSIST,
	    bridge_io, bp);
	if (event_add(&bp->bp_event, NULL) < 0) {
		(void)snprintf(error, sizeof(error), "event_add: %s",
		    strerror(errno));
		(void)close(bp->bp_fd);
		goto error;
	}

	(void)RB_INSERT(bridge_ports_tree, &bridge_ports, bp);

	if (bridge_debug)
		(void)fprintf(stderr, "Added TAP port %s\n", bp->bp_name);

	return (bp);
}

struct bridge_port *
bridge_add_pcap(const char *interface, u_int32_t flags)
{
	static struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, BRIDGE_PKT_LOOPBACK, 5, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, BRIDGE_PKT_MOPRC, 4, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, BRIDGE_PKT_MOPDL, 3, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, BRIDGE_PKT_LAT, 2, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, BRIDGE_PKT_DECNET, 1, 0),
		BPF_STMT(BPF_RET+BPF_K, 0),
		BPF_STMT(BPF_RET+BPF_K, BRIDGE_MAX_PKTLEN),
	};
	static struct bpf_program pgm = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};
	char errbuf[PCAP_ERRBUF_SIZE];
	struct bridge_port *bp;
#if defined(__NetBSD__) || defined(__OpenBSD__) || defined(__FreeBSD__)
	u_int flag = 1;
#endif

	if ((bp = calloc(1, sizeof(*bp))) == NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(errno));
		return (bp);
	}

 	(void)strlcpy(bp->bp_name, interface, sizeof(bp->bp_name));
	if (RB_FIND(bridge_ports_tree, &bridge_ports, bp) != NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(EEXIST));
	error:	free(bp);
		return (NULL);
	}

	if ((bp->bp_desc = pcap_open_live(interface, BRIDGE_MAX_PKTLEN, 1,
	    0, errbuf)) == NULL) {
		(void)snprintf(error, sizeof(error), "%s", errbuf);
		goto error;
 	}

	if (pcap_setfilter(bp->bp_desc, &pgm) < 0) {
		(void)snprintf(error, sizeof(error), "%s: %s", interface,
		    pcap_geterr(bp->bp_desc));
	ioerr:	pcap_close(bp->bp_desc);
		goto error;
	}

	bp->bp_fd = pcap_fileno(bp->bp_desc);

#if defined(__NetBSD__) || defined(__OpenBSD__) || defined(__FreeBSD__)
	if (ioctl(bp->bp_fd, BIOCIMMEDIATE, &flag) < 0) {
		(void)snprintf(error, sizeof(error), "%s: BIOCIMMEDIATE: %s",
		    interface, strerror(errno));
		goto ioerr;
	}

	if (ioctl(bp->bp_fd, BIOCSHDRCMPLT, &flag) < 0) {
		(void)snprintf(error, sizeof(error), "%s: BIOCSHDRCMPLT: %s",
		    interface, strerror(errno));
		goto ioerr;
	}

#ifdef __OpenBSD__
	flag = BPF_DIRECTION_OUT;
	if (ioctl(bp->bp_fd, BIOCSDIRFILT, &flag) < 0) {
		(void)snprintf(error, sizeof(error), "%s: BIOCSDIRFILT: %s",
		    interface, strerror(errno));
		goto ioerr;
	}
#endif	/* __OpenBSD__ */
#ifdef __FreeBSD__
	flag = BPF_D_OUT;
	if (ioctl(bp->bp_fd, BIOCSDIRECTION, &flag) < 0) {
		(void)snprintf(error, sizeof(error), "%s: BIOCSDIRECTION: %s",
		    interface, strerror(errno));
		goto ioerr;
	}
#endif	/* __FreeBSD__ */
#endif	/* __NetBSD__ || __OpenBSD__ || __FreeBSD__ */

	bp->bp_type = BPT_PCAP;
	bp->bp_flags |= flags;
	LIST_INIT(&bp->bp_nodes);

	event_set(&bp->bp_event, bp->bp_fd, EV_READ | EV_PERSIST,
	    bridge_io, bp);
	if (event_add(&bp->bp_event, NULL) < 0) {
		(void)snprintf(error, sizeof(error), "event_add: %s",
		    strerror(errno));
		goto ioerr;
	}

	(void)RB_INSERT(bridge_ports_tree, &bridge_ports, bp);

	if (bridge_debug)
		(void)fprintf(stderr, "Added PCAP port %s\n", bp->bp_name);

	return (bp);
}

struct bridge_port *
bridge_add_udp(const char *host, const char *port, u_int32_t flags)
{
	struct bridge_port *bp;
	struct bridge_host *bh;
	struct sockaddr_in sin;

	if ((bp = calloc(1, sizeof(*bp) + sizeof(*bh))) == NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(errno));
		return (bp);
	}

	if (get_sockaddr(&sin, host, port) < 0) {
	error:	free(bp);
		return (NULL);
	}

	bh = (struct bridge_host *)((caddr_t)bp + sizeof(*bp));
	bh->bh_addr = sin.sin_addr;
	bh->bh_port = sin.sin_port;
	bh->bh_bport = bp;

	(void)snprintf(bp->bp_name, sizeof(bp->bp_name), "%s:%u",
	    inet_ntoa(bh->bh_addr), ntohs(bh->bh_port));
	if (RB_FIND(bridge_ports_tree, &bridge_ports, bp) != NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(EEXIST));
		goto error;
	}

	bp->bp_type = BPT_UDP;
	bp->bp_fd = -1;
	bp->bp_desc = bh;
	bp->bp_flags |= flags;
	LIST_INIT(&bp->bp_nodes);
	(void)RB_INSERT(bridge_ports_tree, &bridge_ports, bp);
	(void)SPLAY_INSERT(bridge_hosts_tree, &bridge_hosts, bh);

	if (bridge_debug)
		(void)fprintf(stderr, "Added UDP port %s\n", bp->bp_name);

	return (bp);
}

#ifdef VDE
struct bridge_port *
bridge_add_vde(char *socket, u_int32_t flags)
{
	struct vde_open_args voa;
	struct bridge_port *bp;

	if ((bp = calloc(1, sizeof(*bp))) == NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(errno));
		return (bp);
	}

	bzero(&voa, sizeof(voa));
	if ((bp->bp_desc = vde_open(socket, "dnbridge", &voa)) == NULL) {
		(void)snprintf(error, sizeof(error), "vde_open: %s",
		    strerror(errno));
		goto error;
	}

 	(void)strlcpy(bp->bp_name, socket, sizeof(bp->bp_name));
	if (RB_FIND(bridge_ports_tree, &bridge_ports, bp) != NULL) {
		(void)snprintf(error, sizeof(error), "%s", strerror(EEXIST));
	ioerr:	(void)vde_close(bp->bp_desc);
	error:	free(bp);
		return (NULL);
	}

	bp->bp_type = BPT_VDE;
	bp->bp_fd = vde_datafd(bp->bp_desc);
	bp->bp_flags |= flags;
	LIST_INIT(&bp->bp_nodes);

	event_set(&bp->bp_event, bp->bp_fd, EV_READ | EV_PERSIST,
	    bridge_io, bp);
	if (event_add(&bp->bp_event, NULL) < 0) {
		(void)snprintf(error, sizeof(error), "event_add: %s",
		    strerror(errno));
		goto ioerr;
	}

	(void)RB_INSERT(bridge_ports_tree, &bridge_ports, bp);

	if (bridge_debug)
		(void)fprintf(stderr, "Added VDE port %s\n", bp->bp_name);

	return (bp);
}
#endif	/* VDE */

void
bridge_del(struct bridge_port *bp)
{
	struct bridge_node *bn;
	struct ifreq ifr;

	switch (bp->bp_type) {
	case BPT_PCAP:
		pcap_close(bp->bp_desc);
		break;
	case BPT_TAP:
		bzero(&ifr, sizeof(ifr));
		(void)strlcpy(ifr.ifr_name, bp->bp_name, sizeof(ifr.ifr_name));
		if (ioctl(sock, SIOCGIFFLAGS, &ifr) == 0) {
			ifr.ifr_flags &= ~IFF_UP;
			(void)ioctl(sock, SIOCSIFFLAGS, &ifr);
		}

		(void)close(bp->bp_fd);
#ifdef __NetBSD__
		if (bp->bp_flags & BPF_DESTROY)
			(void)ioctl(sock, SIOCIFDESTROY, &ifr);
#endif
		break;
	case BPT_UDP:
		(void)SPLAY_REMOVE(bridge_hosts_tree, &bridge_hosts,
		    (struct bridge_host *)bp->bp_desc);
		break;
#ifdef VDE
	case BPT_VDE:
		(void)vde_close(bp->bp_desc);
		break;
#endif
	}

	while ((bn = LIST_FIRST(&bp->bp_nodes)) != NULL) {
		LIST_REMOVE(bn, bn_list);
		LIST_INSERT_HEAD(&bridge_pool, bn, bn_list);
	}

	if (bridge_debug)
		(void)fprintf(stderr, "Deleted port %s\n", bp->bp_name);

	RB_REMOVE(bridge_ports_tree, &bridge_ports, bp);
	free(bp);
}

void
bridge_main(void)
{
	/* XXX: check for error */
	(void)setitimer(ITIMER_REAL, &itv, NULL);

	(void)event_dispatch();
}
