/* $PDP11: dnbridge.c,v 1.12 2015/05/05 07:30:50 form Exp $ */

/*
 * Copyright (c) 2009, 2013 Oleg Safiullin <form@pdp-11.org.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/tree.h>
#include <netinet/in.h>
#include <err.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "dnbridge.h"
#include "missing.h"


static int debug;
static int lowdelay;
static size_t maxnodes = BRIDGE_DEF_NODES;
static char *bind_host = DEF_BIND_HOST;
static char *bind_port = DEF_BIND_PORT;
static char *conf_file = DNBRIDGE_CONF_FILE;


static void
parse_config(FILE *fp)
{
	char buf[256];
	int line = 0;

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		char *s, *argv[8];
		size_t len;
		int i, argc, type;
		u_int32_t flags;

		line++;
		if ((s = strchr(buf, '#')) != NULL)
			*s = '\0';

		len = strlen(buf);
		while (len > 0 &&
		    (buf[len - 1] == '\r' || buf[len - 1] == '\n'))
			buf[--len] = '\0';

		if (len == 0)
			continue;

		for (s = buf, argc = 0; argc < 8 &&
		    (argv[argc] = strsep(&s, " \t")) != NULL;)
			if (argv[argc][0] != '\0')
				argc++;

		if (argc < 3) {
		synerr:	syslog(LOG_ERR, "Syntax error at %s(%d)", conf_file,
			    line);
			exit(1);
		}

		if (s != NULL) {
			syslog(LOG_ERR, "Argument list too long at %s(%d)",
			    conf_file, line);
			exit(1);
		}

		flags = 0;

		if (strcmp(argv[0], "pcap") == 0)
			type = BPT_PCAP;
		else if (strcmp(argv[0], "udp") == 0)
			type = BPT_UDP;
		else if (strcmp(argv[0], "tap") == 0)
			type = BPT_TAP;
#ifdef VDE
		else if (strcmp(argv[0], "vde") == 0)
			type = BPT_VDE;
#endif
		else
			goto synerr;

		for (i = 2; i < argc; i++) {
			if (strcmp(argv[i], "decnet") == 0)
				flags |= BPF_DECNET;
			else if (strcmp(argv[i], "lat") == 0)
				flags |= BPF_LAT;
			else if (strcmp(argv[i], "mop") == 0)
				flags |= BPF_MOP;
			else if (strcmp(argv[i], "passive") == 0)
				flags |= BPF_PASSIVE;
			else
				goto synerr;
		}

		switch (type) {
		case BPT_PCAP:
			if (bridge_add_pcap(argv[1], flags) == NULL) {
			error:	syslog(LOG_ERR, "%s", bridge_error());
				exit(1);
			}
			break;
		case BPT_UDP:
			if ((s = strchr(argv[1], ':')) != NULL)
				*s++ = '\0';

			if (bridge_add_udp(argv[1], s, flags) == NULL)
				goto error;
			break;
		case BPT_TAP:
			if (bridge_add_tap(argv[1], flags) == NULL)
				goto error;
			break;
#ifdef VDE
		case BPT_VDE:
			if (bridge_add_vde(argv[1], flags) == NULL)
				goto error;
			break;
#endif
		}
	}

	if (ferror(fp)) {
		syslog(LOG_ERR, "fgets: %s", conf_file);
		exit(1);
	}

	(void)fclose(fp);
}

static void
print_version(void)
{
	(void)printf("DNBridge v%u.%u, PCAP v%u.%u\n", DNBRIDGE_VERSION_MAJOR,
	    DNBRIDGE_VERSION_MINOR, PCAP_VERSION_MAJOR, PCAP_VERSION_MINOR);
}

static __dead void
usage(void)
{
	extern char *__progname;

	(void)fprintf(stderr, "usage: %s [-dLV] [-f conf_file] [-h bind_host] "
	    "[-n cachesize] [-p bind_port]\n", __progname);
	exit(1);
}

int
main(int argc, char *const argv[])
{
	const char *errstr;
	FILE *fp;
	int ch;

	while ((ch = getopt(argc, argv, "df:h:Ln:p:V")) != -1)
		switch (ch) {
		case 'd':
			debug++;
			break;
		case 'f':
			conf_file = optarg;
			break;
		case 'h':
			bind_host = optarg;
			break;
		case 'L':
			lowdelay++;
			break;
		case 'n':
			maxnodes = (size_t)strtonum(optarg, BRIDGE_MIN_NODES,
			    BRIDGE_MAX_NODES, &errstr);
			if (errstr != NULL)
				errx(1, "%s: %s", optarg, errstr);
			break;
		case 'p':
			bind_port = optarg;
			break;
		case 'V':
			print_version();
			return (0);
		default:
			usage();
			/* NOTREACHED */
		}
	argc -= optind;
	argv += optind;

	if ((fp = fopen(conf_file, "r")) == NULL) {
		err(1, "fopen: %s", conf_file);
		exit(1);
	}

	if (!debug) {
		openlog("dnbridge", LOG_NDELAY | LOG_PID, LOG_DAEMON);

		if (daemon(0, 0) < 0)
			err(1, NULL);
	} else
		openlog("dnbridge", LOG_NDELAY | LOG_PERROR | LOG_PID,
		    LOG_DAEMON);

	if (bridge_init(debug, lowdelay, maxnodes, bind_host, bind_port) < 0) {
		syslog(LOG_ERR, "%s", bridge_error());
		return (1);
	}

	parse_config(fp);
	bridge_main();

	return (0);
}
