/* stream.h - definitions stuff for stream IO */

#define	QPCTL	0100			/* priority control message */
#define	QBSIZE	64			/* "standard" size of queue block */
#define	QBBSIZE	1024			/* size of big queue block */


/*
 * Module consists of a pair of queues.  Each queue manages flow control
 * in a single direction.  Two are required to be a duplex connection.
 */

struct	queue	{
	struct	qinit	*qinfo;
	struct	block	*first;		/* first data block on queue */
	struct	block	*last;		/* last data block on queue */
	struct	queue	*next;		/* next queue downstream */
	struct	queue	*link;		/* next queue */
	caddr_t	ptr;			/* to private data structure */
	short	count;			/* bytes now on queue */
	u_short	flag;			/* Q status flags */
};

/* Queue flags */
#define	QENAB	001		/* Queue is already enabled to run */
#define	QWANTR	002		/* Someone wants to read Q */
#define	QWANTW	004		/* Someone wants to write Q */
#define	QFULL	010		/* Q is considered full */
#define	QREADR	020		/* This is the reader (first) Q */
#define	QUSE	040		/* This queue in use (allocation) */
#define	QNOENB	0100		/* Don't enable Q via putq */
#define	QDELIM	0200		/* This queue generates delimeters */
#define	QBIGB	0400		/* This queue likes big blocks */





/*
 * queue information structure
 */

struct	qinit {
	int	(*putp)();		/* put procedure */
	int	(*srvp)();		/* service procedure */
	long	(*qopen)();		/* called on startup */
	int	(*qclose)();		/* called on finish */
	short	limit;			/* high water mark */
	short	lolimit;		/* low water mark */
};

#define	OTHERQ(q)	((q)->flag & QREADR ? (q)+1 : (q) - 1)
#define	WR(q)		(q+1)
#define	RD(q)		(q-1)

/*
 * Queue data block
 */

struct	block	{
	struct	block	*next;	/* next block in queue */
	u_char		*rptr;	/* next character in block to read */
	u_char		*wptr;	/* next available spot in block */
	u_char		*lim;	/* end of block */
	u_char		*base;	/* data buffer */
	u_char		type;	/* type of block contents */
	u_char		class;	/* size class (+ S_DELIM flag) */
	u_char		bufix;	/* buffer cache index (for BDP) */
	u_char		rbufix;	/* real index for anon block */
	u_char		data[4];/* data contents for tiny blocks */
};

#define	NOBUFIX	255
#define	S_DELIM	0200		/* this block is the last of a record */
#define	CL_MASK	0177

/*
 * Header for a stream: interface to rest of system
 */

struct	stdata {
	struct	queue	*wrq;		/* write queue */
	struct	block	*iocblk;	/* return block for ioctl */
	struct	inode	*inode;		/* backptr, for hangups */
	struct	proc	*wsel;		/* process write-selecting */
	struct	proc	*rsel;		/* process read-selecting */
	short		pgrp;		/* process group, for signals */
	short		flag;	
	char		count;		/* # of processes in stream routines */
};

#define	IOCWAIT		0001		/* Someone wants to do ioctl */
#define	RSLEEP		0002		/* Someone wants to read */
#define	WSLEEP		0004		/* Someone wants to write */
#define	HUNGUP		0010		/* Device has vanished */
#define	RSEL		0020		/* read-select collision */
#define	WSEL		0040		/* write-select collision */
#define	EXECL		0100		/* exclusive-use (no opens) */
#define	STWOPEN		0200		/* waiting for 1st open */

/* classes are:	0 - 4B
 *		1 - 16B
 *		2 - 64B
 *		3 - 256B
 *		4 - 1024B
 *		5 - 4096B
 */

static	struct	class	{
	struct	block	*free;
	int		nblks;		/* current block count */
	int		min;		/* smallest number of blocks */
	int		max;		/* largest number of blocks */
};


#define M_DATA		0000	/* regular data */
#define M_BREAK		0001	/* line break */
#define M_HANGUP	0002	/* line disconnect */
#define M_DELIM		0003	/* data delimiter */
#define	M_ECHO		0004	/* request ACK (1 param) */
#define	M_ACK		0005	/* response to ECHO (1 param) */
#define M_IOCTL		0006	/* ioctl; set/get params */
#define M_DELAY		0007	/* real-time xmit delay (1 param) */
#define	M_CTL		0010	/* device-specific control message */
#define	M_PASS		0011	/* pass file */
#define	M_YDEL		0012	/* stream has started generating delims */
#define	M_NDEL		0013	/* stream has stopped generating delims */

/*
 * control messages (high priority; go to head of queue )
 */

#define	M_SIGNAL	0101	/* generate proces signal */
#define	M_FLUSH		0102	/* flush your queues */
#define	M_STOP		0103	/* stop transmission immediately */
#define	M_START		0104	/* restart transmission after stop */
#define M_IOCACK	0105	/* acknowledge ioctl */
#define M_IOCNAK	0106	/* negative ioctl acknowledge */
#define	M_PRICTL	0107	/* high priority device-specific ctl */
#define	M_IOCWAIT	0110	/* stop ioctl timeout, ack/nak follows later */

#define	setqsched()	(pcc_set_si1())

/*
 * ioctl message packet
 */

#define	STIOCSIZE	16
#define	STIOCHDR	4

struct	stioctl	{
	unsigned char	com[STIOCHDR];	/* four-byte command, lsb first */
	char		data[STIOCSIZE]; /* depends on command */
};

#define	stiocom(bp)	(((struct stioctl *)bp->rptr)->com[0] | \
			(((struct stioctl *)bp->rptr)->com[1]<< 8))
#define	stiodata(bp)	(((struct stioctl *)bp->rptr)->data)

/*
 * for passing files across streams
 */

struct	kpassfd {
	union	{
		struct	file	*p;
		int		fd;
	} f;
	short	uid;
	short	gid;
	short	nice;
	char	logname[8];
};

/*
 * header for messages, see msg.c
 */

struct	stmesg {
	char 	type;
	u_char	magic;
	u_char	losize, hisize;
};

#define	MSGMAGIC	0345
#define	MSGLEN		4	/* true length of struct mesg in bytes */

int	srvp(), putp();
struct	block	*getq();
struct	block	*allocb();
struct	block 	*dupb();
