/* wd3393.c:  SCSI driver for Western Digital 3393 as initiator */
/*		Western Digital Corporation W33c93A SCSI Bus
 *		interface controller manual (Document #79-000199)
 *		as well as the mvme147s mpu vmemodule users manual
 *		(mvme-147s/d1) 68nw9209g90a,
 *		is required to understand this device driver.
 */

#include <sys/types.h>
#include "scsi.h"

#define	SCSIADDR	((u_char *) 0xfffe4000)
#define	SCSIDATA	((u_char *) 0xfffe4001)

#define	TIMER		((struct timer *) 0xfffe1010)

struct	timer	{
	u_short	preload1;
	u_short	count1;
	u_short	preload2;
	u_short count2;
	u_char	icr1;
	u_char	cr1;
	u_char	icr2;
	u_char	cr2;
};

#define	set(a, d)	(*SCSIADDR = (a), *SCSIDATA = (d))
#define	get(a)		(*SCSIADDR = (a), *SCSIDATA)

#define	FC		(0x05000000)

/*
 * The device.  Note that it has to be shoved into a little hole.
 * One thing nice about the part is that it auto-increments
 * the address register.
 * Note that the xfer_cnt is really a 3byte thing with the high byte
 * being used as the sync register.
 * The ownid byte is included so that the long aligns.
 * The struct will break if a compiler that aligns differently
 * is used. (must be on 16-bit boundaries)
 */

static
struct	device	{
	u_char	control;		/* enable/disable certain functions */
	u_char	timeout;		/* = (Tper * Ficlk) / 80 */
	u_char	cdb[12];		/* the command to send */
	u_char	target_lun;		/* set during reconnect */
	u_char	cmd_phase;		/* where the chip is in things */
	long	xfer_cnt;		/* bytes to move */
	u_char	dest_id;		/* scsi destination register */
	u_char	src_id;			/* who am i */
	u_char	status;			/* chip status */
	u_char	command;		/* chip command */
} device;



scsi(wh, rh, dp, n)
	struct scsi_rh *rh;	/* read header */
	struct scsi_wh *wh;	/* where to put the response */
	caddr_t dp;		/* data */
	int n;			/* number of bytes in data */
{
	static int inited = 0;
	register i;
	register u_char *tp, *fp;
	register char *p;
	register reading, status;


	if (! inited) {
		scsi_reset(1);
		delay(100);
		scsi_reset(0);
		delay(100);
		get(0x17);		/* clear int */
		while ((*SCSIADDR & 0x80) != 0)
			;
		set(0, 0x7);		/* scsi id 7, no par, 10MHz */
		delay(2);		/* since I touched the auxstat, wait */
		set(0x18, 0);		/* reset the chip */
		while ((*SCSIADDR & 0x80) == 0)
			;
		get(0x17);		/* this will clear the int bit */
		while ((*SCSIADDR & 0x80) != 0)
			;
		inited = 1;
	}

	/* set up device structure */

	device.control = 0x04;		/*  int on disconn */
	device.timeout = 62;		/* 500 ms */
	tp = device.cdb;
	fp = wh->cmd;
	for (i = 0; i < 10; i++)
		*tp++ = *fp++;
	device.target_lun = 0;
	device.cmd_phase = 0;
	device.xfer_cnt = n & 0xffffff;
	device.dest_id = wh->target;
	device.src_id = 7;		/* really gets set on reconnect */
	device.status = 0;		/* place holder */
	device.command = 9;		/* select and transfer command */

	delay(2);			/* assure that 7us have passed */

	/* feed this to the device */

	*SCSIADDR = 1;			/* start with control reg */
	fp = (u_char *)&device;
	for (i = 0; i < ((char *)&device.command - (char *)&device)+1; i++) {
		*SCSIDATA = *fp++;
	}

	/* Watch for the completion of the command */

	*SCSIADDR = 0x19;			/* point to the data reg */
	reading = wh->flags & SCSI_RD;
	p = dp;
	for (;;) {
		int st;

		status = *SCSIADDR;
		if (status & 0x01 && n > 0) {
			if (reading)
				*p++ = *SCSIDATA;
			else
				*SCSIDATA = *p++;
			n--;
		}
		if (status & 0x40)
			printf("scsi: lci!\n");
		if (status & 0x80) {
			if ((st = get(0x17)) == 0x85)
				break;
			else if (st == 0x16)
				/* do nothing */;
			else
				printf("scsi: status 0x%x\n", st);
		}
	}
}

static
delay(n)		/* use timer2 to delay 'n' ticks */
{
	TIMER->cr2 = 0;			/* turn off timer */
	TIMER->preload2 = -n;
	TIMER->cr2 = 7;			/* start counting */
	while ((TIMER->cr2 >> 4) == 0)
		;			/* wait for timer to roll */
	TIMER->cr2 = 0;			/* turn off */
}

#define	RESET	((char *) 0xfffe102a)

scsi_reset(n)
{
	if (n)
		*RESET = 0x10;
	else
		*RESET = 0;
}
