/*robotfindskitten v1600000.rb1 source code.
 *
 *Copyright (C) 1997,1999 Leonard Richardson leonardr@segfault.org
 *
 *Modifications for the Rainbow 100
 *Copyright (C) 2008 Jeff Armstrong jeff@rainbow-100.com
 *
 */

#include <dos.h>
#include <time.h>
#include <stdlib.h>
#include "phrases.h"

#define VERSION "v1600000.rb1"

#define EMPTY -1
#define ROBOT 0
#define KITTEN 1

/*#define NUM_BOGUS 10 */

/* Rainbow-specific level 1 console */
#define FUNCTION	0x100	/* pertinent bit in 16-bit key return */
#define UP  	(FUNCTION +  0x27)
#define LEFT	(FUNCTION +  0x2D)
#define DOWN	(FUNCTION +  0x29)
#define RGHT	(FUNCTION +  0x2B)
#define EXIT	(FUNCTION +  0x0F)

/* The Rainbow can have different screen widths */
#define SCREEN_WIDTH    80
#define SCREEN_HEIGHT   24
#define PLAY_HEIGHT     20
#define MESSAGE_ROW     3

void finale();
void instructions();
void process_input(int input);
void shutdown();

struct object
{
	int x;
	int y;
	char character;
};

struct object robot;
struct object kitten;
struct object empty;
struct object bogus[MESSAGES];
int bogus_messages[MESSAGES];

int used_messages[MESSAGES];
int NUM_BOGUS;
int screen[SCREEN_WIDTH+1][SCREEN_HEIGHT+1];

/* Function that allows printing of double-height double-width
 * Rainbow characters using VT52 escape codes.
 */
void large_print(char *string) {
        printf("%c%c%c%s\n",033,043,063,string);
        printf("%c%c%c%s\n",033,043,064,string);
}

/* Function that allows printing of double-width
 * Rainbow characters using VT52 escape codes.
 */
void wide_print(char *string)  {
        printf("%c%c%c%s\n",033,043,066,string);
}

/* VT52 escape code to provide inverted text */
void invert() {
        printf("%c[7m",033);
}

/* VT52 escape code to provide normal text */
void normal() {
        printf("%c[0m",033);
}

/* Retrieves Level 1 console input (16bit).
 * Note that this function blocks until input
 * is received.
 */
int inkey()
{
union REGS r;

        do {
                r.x.di = (unsigned int)0x06;
                int86(0x18,&r,&r);
        } while(r.h.cl != 0xFF);
        
        return (int)r.x.ax;
}

/* Returns a random position on the play screen */
void randomposition(int *row, int *col)
{
        *row = (rand() % PLAY_HEIGHT) + SCREEN_HEIGHT-PLAY_HEIGHT;
        *col = (rand() % SCREEN_WIDTH) + 1;
}

/* Disables the cursor during the game */
void rdisablecursor()
{
union REGS r;
        r.x.di = (unsigned int)0x08;
        int86(0x18,&r,&r);
}

/* Enables the cursor when the game is over */
void renablecursor()
{
union REGS r;
        r.x.di = (unsigned int)0x0A;
        int86(0x18,&r,&r);
}

/* Places a sequence of characters at a given screen position
 * using Rainbow fast video via the 0x18 interrupt.  Note that
 * this interrupt requires that bp = ds, meaning that the
 * Turbo C intr() method is necessary to override normal
 * handling of the bp register.  This call makes this version
 * of rfk incompatible with MS compilers. 
 */
void rput(void *data, int len, int row, int col) 
{
struct SREGS segregs;

struct REGPACK rp;

        segread(&segregs);
        
        rp.r_ax = (unsigned int)2;
        rp.r_di = (unsigned int)0x14;
        rp.r_bx = (unsigned int)256*(unsigned int)col + (unsigned int)row;
        rp.r_cx = (unsigned int)len;
        rp.r_si = data;
        rp.r_bp = segregs.ds;
        intr(0x18,&rp);
}

/* Puts a single character at a given position */
void rputchar(char c, int row, int col)
{
        rput(&c,1,row,col);
}

/* Puts a string at the give position.  No width calculations
 * are made, so beware exceeding 132 columns.
 */
void rputstr(char *string, int row, int col)
{
        rput(string,strlen(string),row,col);
}

/* Clears a single row on the screen by writing spaces using
 * fast video routines.
 */
void rclearrow(int row)
{
int i;
        for(i=1;i<=SCREEN_WIDTH;i++)
                rputchar(' ',row,i);
}

/* A fun clear screen routine that actually erases the screen
 * from left to right by placing spaces in all columns.  A 
 * horizontal clearing would be faster by passing entire rows
 * of spaces in, reducing interrupt calls, but this solution
 * is far more whimsical.
 */
void clrscr()
{
int row,col;
        for(col=1;col<=SCREEN_WIDTH;col++) {
                for(row=1;row<=SCREEN_HEIGHT;row++) 
                        rputchar(' ',row,col);
        }
        /* Send cursor home */
        printf("%c[000;000H",033);
}

/* A safe and conistent way of getting a random character.
 * The original code made some wild assumptions about what ASCII
 * codes were available.  This routine limits it to mostly
 * letters and some puctuation.
 */
char randchar()
{
        return 'A' + rand() % 52;
}

/* Shows a message to the user */
void showmessage(char *str) {
        rclearrow(MESSAGE_ROW);
        rputstr(str,MESSAGE_ROW,1);
}

int main(int argc, char *argv[])
{
int supplied_number;
int counter, counter2;
int index;
int input;
int old_y, old_x;
char title[41];
	
	srand(time(NULL)); rand();

	empty.x = -1;
	empty.y = -1;
	empty.character = ' ';

	if (argc == 1) {
	        NUM_BOGUS = 20;
	} else {
	        supplied_number = atoi(argv[1]);
	        if (supplied_number < 0 || supplied_number > MESSAGES) {
	                printf("Error: run-time parameter must be between 0 and %d.\n",MESSAGES);
	                exit(0);
	        } else {
	                NUM_BOGUS = supplied_number;
	        }
	}

	/* Now we have the filling in of the various arrays. */

	/* Create an array to represent the screen so that we can make sure all
	 * the objects go on different squares. */
	for (counter = 0; counter <= SCREEN_WIDTH; counter++)
	{
		for (counter2 = 0; counter2 <= SCREEN_HEIGHT; counter2++)
		{
			screen[counter][counter2] = EMPTY;
		}
	}
	/* Create an array to ensure we don't get duplicate messages. */
	for (counter = 0; counter < MESSAGES; counter++)
	{
		used_messages[counter] = 0;
		bogus_messages[counter] = 0;
		bogus[counter] = empty;
	}

	/* Now we initialize the various game objects. */
	/* Assign a position to the player. */
        randomposition(&robot.y, &robot.x);
	robot.character = '#';
	screen[robot.x][robot.y] = ROBOT;
       
	/* Assign the kitten a character and a color. */
	do
	{
		kitten.character = randchar();
	} while (kitten.character == '#' || kitten.character == ' ' || kitten.character == (char)7);

        do
        {
                randomposition(&kitten.y,&kitten.x);
        } while(screen[kitten.x][kitten.y] != EMPTY && kitten.x+kitten.y == 0);
	screen[kitten.x][kitten.y] = KITTEN;

	/* Now, initialize non-kitten objects. */
	for (counter = 0; counter < NUM_BOGUS; counter++)
	{
		/* Assign a unique position. */
		do
		{
                        randomposition(&(bogus[counter].y),&(bogus[counter].x));
		} while (screen[bogus[counter].x][bogus[counter].y] != EMPTY);
		screen[bogus[counter].x][bogus[counter].y] = counter+2;

		/* Assign a character. */
		do
		{
			bogus[counter].character = randchar(); /* rand() % 254 + 1; */
		} while (bogus[counter].character == '#' || bogus[counter].character == ' ' || bogus[counter].character == (char)7);
	
		/* Assign a unique message. */
		index = 0;
		do
		{
			index = rand() % MESSAGES;
		} while (used_messages[index] != 0);
		bogus_messages[counter] = index;
		used_messages[index] = 1;
	}

	/* Print instructions. */
	clrscr();
        rdisablecursor();
        instructions();
        
	clrscr();
	
        invert();
        /* We're assuming a width of the VERSION value here... */
        sprintf(title,"robotfindskitten %s            ",VERSION);
        large_print(title);
        normal();

	for (counter = 0; counter < NUM_BOGUS; counter++)
	{
		rputchar(bogus[counter].character,bogus[counter].y,bogus[counter].x);
	}

        rputchar(kitten.character,kitten.y,kitten.x);
	rputchar(robot.character,robot.y,robot.x);

	old_x = robot.x;
	old_y = robot.y;

	/* Now the fun begins. */
	
	input = inkey(); 
	while (input != EXIT)
	{
		process_input(input);
		/* Redraw robot, where avaliable */
		if (!(old_x == robot.x && old_y == robot.y))
		{
			rputchar(' ',old_y,old_x);
			rputchar(robot.character,robot.y,robot.x);
			
                        screen[old_x][old_y] = EMPTY;
			screen[robot.x][robot.y] = ROBOT;
		        old_x = robot.x;
			old_y = robot.y;
		}
		input = inkey(); 
	}
        shutdown();	
}

/* Given the keyboard input, interprets it. */
void process_input(int input)
{
int counter;
int check_x = robot.x;
int check_y = robot.y;

	switch (input)
	{
                case UP: /* up */
			check_y--;
			break;
                case DOWN: /* down */
			check_y++;
			break;
                case LEFT: /* left */
			check_x--;
			break;
                case RGHT: /* right */
			check_x++;
			break;
		case EXIT:
			break;
		default: /* invalid command */
			showmessage("Invalid command: Use direction keys or [EXIT].");
			return;
	}

	/* Check for going off the edge of the screen. */
	if (check_y < SCREEN_HEIGHT-PLAY_HEIGHT || 
            check_y > SCREEN_HEIGHT || 
            check_x < 1 || 
            check_x > SCREEN_WIDTH)
	{
		return;
	}

	/* Check for collision */
	if (screen[check_x][check_y] != EMPTY)
	{
		switch (screen[check_x][check_y])
		{
			case ROBOT:
				/* We didn't move. */
				break;
			case KITTEN: /* Found it! */
                                finale();
				shutdown();
				break;
			default:
				showmessage(messages[bogus_messages[screen[check_x][check_y]-2]]);
				break;
		}
                return;
	}
	/* Otherwise, move the robot. */
	robot.x = check_x;
	robot.y = check_y;
}

void instructions()
{
	printf("robotfindskitten %s\n",VERSION);
        printf("Rainbow 100 Edition!\n");
        printf("\nRainbow modifications by Jeff Armstrong in 2008\n\n");
        
        printf("Based on the code by the illustrious Leonard Richardson \xa9 1997\n");
	printf("Written originally for the Nerth Pork robotfindskitten contest\n\n");
        
        printf("\n   In this game, you are robot (%c",robot.character);
	printf("). Your job is to find kitten. This task\n");
        printf("is complicated by the existance of various things which are not kitten.\n");
        printf("Robot must touch items to determine if they are kitten or not. The game\n");
	printf("ends when robotfindskitten. Alternatively, you may end the game by hitting\n");
	printf("the [EXIT] key. See the documentation for more information.\n\n");
	printf("\n   Press any key to start.\n");
	inkey(); 
}

/* Awesome Rainbow-only finale using the Rainbow's double-height
 * and double-width character generation.
 */
void finale() 
{
int i,col,counter;
char line[41];

        for(col=0;col<40;col++)
                line[col] = ' ';
        line[41] = '\0';

        for(i=10;i<18;i++)
                rclearrow(i);
        invert();
        for(counter=0;counter<8;counter++) {
                printf("%c[010;000H",033);	
                line[13+counter] = robot.character;
                line[28-counter] = kitten.character;
                large_print(line);
                delay(2000);
                line[13+counter] = ' ';
                line[28-counter] = ' ';
                                
        }
        normal();

        printf("%c[012;009H",033);
        wide_print("Congratulations, Robot!");	
        printf("%c[013;012H",033);
        wide_print("You found Kitten!");	        
}

/* Shuts down the game by first enabling the cursor and then
 * returning it to the bottom of the screen via VT52 escape
 * sequence.
 */
void shutdown()
{
        renablecursor();
        printf("%c[023;000H",033);
        exit(0);
}