/*
 * rtfile.cpp
 *
 * Implementation module for RTfile class.
 *
 * John Dudeck   SIM International   25-Aug-92
 */
#include <ctype.h>
#include <string.h>
#include <iostream.h>
#include "rtdefs.h"
#include "rtfile.h"

RTfile::RTfile(char * fspec) {
   // Filespec syntax:
   // fspec     ::= drive name extension
   // drive     ::= letter ':'
   //           | 
   // name      ::= letters(0 to 6)
   //           |   letters(0 to 5) '*'
   // extension ::= '.' letters(0 to 3)
   //           |   '.' letters(0 to 2) '*'
   //           |
   // letters   ::= 'A' thru 'Z', '0' thru '9', '$'
   // letter    ::= 'A' thru 'Z'
   //
   // Note:
   // A missing drive letter implies the default drive from rtdefs.h
   // A missing name part implies a name of '*'.
   // A missing dot implies an extension of '*'.
   
   char * cp = fspec;
   while (*cp) *cp++ = toupper(*cp);
   
   char * drv = strchr(fspec, ':');
   char * dot = strchr(fspec, '.');
   
   // Parse drive letter.
   drive[0] = DEFDRIVE;
   if (drv != NULL) {
      drv--;
      if (isalpha(*drv)) drive[0] = *drv;
   }
   drive[1] = '\0';
   
   // Parse name part.
   for (int x = 0; x < RNLEN+1; x++) name[x] = '\0';
   char * np = (drv != NULL) ? drv + 2 : fspec;
   int nln = (dot != NULL) ? dot - np : strlen(np);
   if (nln < 0) nln = 0;
   if (nln > RNLEN) nln = RNLEN;
   strncpy(name, np, nln);
   // Pad with nulls
   for (x = nln;
        x < RNLEN + 1;
        x++) {
      name[x] = '\0';
   }
   // Check for all spaces
   boolean duh = true;
   for (x = 0; x < nln; x++) {
      if (name[x] == '\0') break;
      if (name[x] != ' ') {
         duh = false;
         break;
      }
   }
   if (duh) name[0] = '*';
   
   // Parse extension part.
   for (x = 0; x < 4; x++) ext[x] = '\0';
   if (dot != NULL) strncpy(ext, dot + 1, 3);
   else ext[0] = '*';
}

RTfile::RTfile() {
   // Construct empty RTfile object.
   drive[0] = DEFDRIVE;
   drive[1] = '\0';
   for (int x = 0; x < RNLEN + 1; x++) name[x] = '\0';
   for (x = 0; x < 4; x++) ext[x] = '\0';
}

boolean RTfile::testRTspec(RTfile & infile) {
   // Effects: Compares the input argument against the object's name and
   //          extension fields.  The object's fields may have wildcards.
   // Returns: 'true' if the arguments fit the object's
   //          specification according to wildcard matching.
   // Note:    Wildcard matching follows MS-DOS rules and not RT-11!
   
   if (name[0] == '*' && ext[0] == '*') return true; // Exit on *.*
   
   boolean nameok = false, extok = false;
   
   // Make temp copies of the object's strings.
   char t_name[RNLEN+1], t_ext[RELEN+1];
   strncpy(t_name, name, RNLEN);
   strncpy(t_ext, ext, RELEN);
   
   // Look for '*'s.
   char * namestar = strchr(t_name, '*');
   char * extstar = strchr(t_ext, '*');
   char * tmp;
   
   // Expand '*' to multiple '?'s.
   if (namestar) {
      for (tmp = namestar;
         tmp < (t_name + RNLEN);
         tmp++) {
         * tmp = '?';
      }
   }
   t_name[RNLEN] = '\0';
   if (extstar) {
      for (tmp = extstar;
         tmp < (t_ext + RELEN);
         tmp++) {
         * tmp = '?';
      }
   }
   t_ext[RELEN] = '\0';
   // Compare one char at a time.
   // Name part.
   char * ttmp = t_name, * itmp = infile.name;
   while (ttmp < t_name + RNLEN
      && ((* ttmp) == '?'
       || (* ttmp) == (* itmp))) {
      ttmp++;
      itmp++;
   }
   // If both are at end of string, we have matched.
   if (!(* ttmp) && !(* itmp)) {
      nameok = true;
   }
   // Ext part     
   ttmp = t_ext;
   itmp = infile.ext;
   while (ttmp < t_ext + RELEN
      && ((* ttmp) == '?'
       || (* ttmp) == (* itmp))) {
      ttmp++;
      itmp++;
   }
   // If both are at end of string, we have matched.
   if (!(* ttmp) && !(* itmp)) {
      extok = true;
   }
 
   return nameok && extok;
}

void RTfile::print() {
   // effects: Prints file name to cout.
   cout << drive << ':' << name << '.' << ext;
}