/*
 * rtdisk.cpp
 *
 * Implementation module for RTdisk class.
 *
 * John Dudeck   SIM International   25-Aug-92
 */

#include <string.h>
#include <dir.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <process.h>
#include <iostream.h>
#include <iomanip.h>
#include <dos.h>
#include "rtdisk.h"
#include "rtdefs.h"
#include "decdates.h"


void RTdisk::RTGetHomeBlock(void) {
   // effects: Reads Home Block from diskette and updates the
   // volid, volown, and volsys RTdisk member variables.
   // Must be the initial operation on an RTdisk.
   // Used internally by the object's member functions.
   
   // Get source drive
   int srcdriv = srcfile.drive[0] - 'A';
   
   homeblok hblok;
   int sector = HOMEBLOK_NO;
   // absread(driveno, nosects, startsect, buffer)
   if (absread(srcdriv, HOMEBLKSIZ, sector, &hblok) != 0) {
      perror("Disk problem");
      exit(1);
   }
   strncpy(volid, hblok.rtvol_id, 12);
   volid[12] = '\0';
   strncpy(volown, hblok.rtvol_owner, 12);
   volown[12] = '\0';
   strncpy(volsysid, hblok.rtvol_sys_id, 12);
   volsysid[12] = '\0';
}

RTdisk::RTdisk() {
   // effects: Constructs a RTdisk object with default source and destination.
   char filespec[16];
   
   strcpy(filespec, "D:*.*");
   srcfile = RTfile(filespec);
   destfile = RTfile(filespec);
}

void RTdisk::print_directory() {
   // effects: Prints the directory for the diskette,
   // constrained by the source, to cout.
   // If the source file and extension are *.* a full
   // directory is printed with <Unused> areas included.
   //
   // DEC's format: (RT-11 defaults to two columns, which we won't for now.
   //
   //  01-Nov-91  (today's date)
   //  Volume ID: XXXXXXXXXXXX
   //  Owner    : XXXXXXXXXXXX
   // FILENM.EXT   SSS  01-Nov-91   BBB
   // <Unused>     SSS              BBB
   // FILENM.EXT   SSSP 31-Oct-91   BBB
   // Etc...
   // <Unused>     SSS              BBB
   //  NN Files, LLL Blocks
   //  FFF Free blocks
   //
   // Where BBB is starting block number for the file. 
   // SSS is file size in blocks.
   // P indicates protected file.
   // If entry has no date, date is left blank.

   // Get today's date as DD-MMM-YY.
   char date_string[10];
   getdecdate(date_string);
   
   filecnt = 0;
   bloks_used = 0;
   bloks_free = 0;
   
   // Read home block.
   // Note: Avoid an extra call to RTGetHomeBlock()
   // by doing the call to RTFindFirst here instead of later:
   ercode finderr = RTFindFirst();
   boolean it_fits = !finderr.is_err();

   // Print directory header
   cout << "\nDirectory of RT-11 diskette"
        << "\n\n " << date_string
        << "\n Volume ID: " << volid
        << "\n Owner    : " << volown
        << "\n System ID: " << volsysid
        << '\n';

   // Are we doing a "Full" directory (including <Unused> areas?
   boolean fulldir = (srcfile.name[0] == '*' 
	       && srcfile.ext[0] == '*');
   do {
      if (fulldir || it_fits) {
         if (status & PERM) {  
	    // Print the file name.
	    // NOTE: The "undocumented" Turbo C++ functions
	    // of setw(), setiosflags(), and resetiosflags()
	    // are in the iomanip.h file. Flag 0x0002 is for left-
	    // justification. It was found in the ioflags.h file.
	    cout << setw(6) << setiosflags(0x0002)
		 << curfile.name << "."
		 << setw(3) << curfile.ext << " "
		 << resetiosflags(0x0002);
            // Tally statistics.
            filecnt++;
            bloks_used += leng;
         } else {
            cout << "<Unused>   ";
         }
         // Print the file size.
         cout << dec << setw(6) << leng;
         // Print the protection status.
         if (status & PROT) {
            cout << "P ";
         } else {
            cout << "  ";
         }
         // Print the creation date.
         if (date) {
            rtdatetodec(date_string, date);
            cout << date_string;
         } else {
            cout << "         ";
         }
         // Print the starting block number for the file.
         cout << dec << setw(6) << start_blok;
         cout << '\n';
      }
   } while (finderr = RTFindNext(), it_fits = !finderr.is_err());
         
   // Print statistics.
   cout << setw(3) << dec << filecnt << " Files, "
        << setw(1) << bloks_used << " Blocks\n "
        << bloks_free << " Free blocks\n";
}

void RTdisk::setsrc(char * fspec) {
   // effects: Sets the source file specification for this diskette.
   srcfile = RTfile(fspec);
}

void RTdisk::setdest(char * fspec) {
   // effects: Sets the destination file specification for this diskette.
   destfile = RTfile(fspec);
      
}

char RTdisk::getdestdrive() {
   // effects: Gets the destination drive.
   // returns: The letter 'A'-'Z' of the destination drive.
   return destfile.drive[0];
}

void RTdisk::extract_RTinfo() {
   // effects: Gets the file info from the current RTdisk directory entry
   // and puts into respective member variables.
   // Used by RTFindFirst and RTFindNext only (common code).
   
   // extract the name from rad-50 format.
   char fname[RNLEN+1], fext[RELEN+1];
   r50toa(fname, dblok.dir_entry[entryno].name[0]);
   r50toa(&fname[3], dblok.dir_entry[entryno].name[1]);
   r50toa(fext, dblok.dir_entry[entryno].name[2]);
   // Copy the strings into the fields of the current file name.
   strcpy(curfile.drive, srcfile.drive);
   strcpy(curfile.name, fname);
   strcpy(curfile.ext, fext);
   // Change blanks to nulls.
   char * tmp;

   for (tmp = curfile.name;
        tmp < curfile.name + RNLEN;
        tmp++) {
      if ((* tmp) == ' ') * tmp = '\0';
   }
   for (tmp = curfile.ext;
        tmp < curfile.ext + RELEN;
        tmp++) {
      if ((* tmp) == ' ') * tmp = '\0';
   }
   // Extract status, date, and length fields.
   status = dblok.dir_entry[entryno].status;
   date = dblok.dir_entry[entryno].date;
   leng = dblok.dir_entry[entryno].leng;
}

ercode RTdisk::RTFindFirst() {
   // effects: Sets the current file to the first file name on this
   // diskette which matches the source file specification.
   // The status of the returned file may by any.
   // returns: Error code NOERROR or FILENOTFOUND.
   
   // Read home block
   (void) RTGetHomeBlock();
   
   // Get source drive.
   int srcdriv = srcfile.drive[0] - 'A';

   // Start with first directory segment.
   segment = 0;
   
   // Read through the directory segments.
   do {
      int sector = FIRST_DIRBLOK + segment * 2;
      if (absread(srcdriv, DIRSEGSIZ, sector, &dblok) != 0) {
         perror("Disk problem");
         exit(1);
      }
      start_blok = dblok.strt;
      for (entryno = 0;
           entryno < 72 && ! (dblok.dir_entry[entryno].status & ENDBLK);
           entryno++) {
         
         extract_RTinfo();
         
         // Tally free space.
         if (!(status & PERM)) {
            bloks_free += leng;
         }
         
         // See if name fits filespec constraint.
         if (srcfile.testRTspec(curfile)) return NOERROR;
         
         // The name doesn't fit, so advance the starting block 
         // number by length of file.
         start_blok += leng;
      }
   // Step to the next directory segment if any.
   } while (segment = dblok.next);
   
   return ercode(FILENOTFOUND);
}

ercode RTdisk::RTFindNext() {
   // assumes: Current file has been set by previous call(s) to
   // RTFindFirst() or RTFindNext().
   // effects: Sets the current file to the next file name on this
   // diskette which matches the source file specification.
   // The status of the returned file may by any.
   // returns: Error code NOERROR or FILENOTFOUND.
   
   // Read through the directory segments.
   while (segment || entryno > -1) {
      while (++entryno, entryno < 72 && ! (dblok.dir_entry[entryno].status & ENDBLK)) {
         
         // Advance the starting block number by length of (previous) file.
         start_blok += leng;
         
         extract_RTinfo();
         
         // Tally free space.
         if (!(status & PERM)) {
            bloks_free += leng;
         }
         
         // See if name fits filespec constraint.
         if (srcfile.testRTspec(curfile)) return NOERROR;
      }
      // Step to the next directory segment if any.
      entryno = -1;
      segment = dblok.next;
      if (segment) {
         // Get source drive.
         int srcdriv = srcfile.drive[0] - 'A';
	 int sector = FIRST_DIRBLOK + segment * 2;
	 if (absread(srcdriv, DIRSEGSIZ, sector, &dblok) != 0) {
	    perror("Disk problem");
	    exit(1);
	 }
	 start_blok = dblok.strt;
      }
   }
   return ercode(FILENOTFOUND);
}

ercode RTdisk::RTopen(int filmode) {
   // effects: Opens the current file with the file mode given.
   // Establishes a buffer.
   // Sets cur_blok to the first block of the file.
   // returns: Error code.
   
   if (filmode == open_for_read) {
      if (status & PERM) {
         if (open_stat == closed) {
            // Don't allocate another buffer if already open,
            // othewise just ignore the fact that it was open and continue.
	    blok_buffer = (char *) malloc(DATBUFSIZ);
            if (blok_buffer == NULL) {
               cout << "Not enough memory for file buffer.\n";
               exit(1);
            }
         }
         cur_blok = start_blok;
         eof_stat = false;
         open_stat = open_for_read;
         return ercode(NOERROR);
         } else {
            return ercode(FILENOTFOUND);
         }
   } else {
      return ercode(UNIMPLEMENTED);
   }
}
   
void RTdisk::RTclose() {
   // effects: Closes the current open file.
   // Discards the buffer.
   
   if (blok_buffer) free(blok_buffer);
   blok_buffer = NULL;
   cur_blok = 0;
   eof_stat = false;
   open_stat = closed;
}

ercode RTdisk::RTreads(char ** bufptr) {
   // effects: Checks to see if cur_blok is beyond the end of the file.
   // If so, returns eof error code.
   // Otherwise. Reads the current block from the disk into the buffer
   // and advances cur_blok.
   // returns: Error code, pointer to the buffer.
   if (!RTeof()) {
      int srcdriv = srcfile.drive[0] - 'A';
      if (absread(srcdriv, 1, cur_blok, blok_buffer) != 0) {
         return ercode(READERROR);
      }
      cur_blok++;
      * bufptr = blok_buffer;
      return ercode(NOERROR);
   } else {
      return ercode(EOFILE);
   }
}

boolean RTdisk::RTeof() {
   // effects: Returns whether not cur_blok is beyond the end of the file.
   return (cur_blok == 0 || cur_blok > start_blok + leng - 1);
}

// ercode RTdisk::copy(RTdisk srcdisk) {
      // effects: Copies the RT11 file(s) from the argument to this diskette.
      // returns: Error code.
//    return ercode(UNIMPLEMENTED);
// }

// ercode RTdisk::copy(DOSdisk srcdisk) {
      // effects: Copies the DOS file(s) from the argument to this diskette.
      // returns: Error code.
//    return ercode(UNIMPLEMENTED);
// }
