/*  pokerd - a classical draw poker server
 *  Copyright (C) 2001,2003 Hans P. Rosenfeld <rosenfeld@grumpf.hope-2000.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 *  todo: implement a per user buffer for reading from the sockets
 *        write a version supporting multiple tables and chats
 */

#include "main.h"

char *commands[16][10] = { 
     { "nick",
       "/nick <name>",
       "changes the name of the player",
       "This command changes your nickname to its first argument.\n",
       "If this nickname is already in use, your nickname is not changed.\n",
       "Note that whitespaces are in nicknames are not allowed.\n",
       NULL
     },
     { "whois",
       "/whois <name>",
       "prints information about a player",
       "This command prints some information about a player.\n",
       NULL
     },
     { "quit",
       "/quit",
       "exits the game",
       "Use this command to quit the game and disconnect from the server.\n",
       "Your cards will be put in the discard, the money you own will not be put back in the game.\n",
       NULL
     },
     { "list",
       "/list",
       "prints a list of all players",
       "This command prints a list of all players together with the IP address of their hosts and\n",
       "their current state in the game.\n",
       "The name of the dealer is prefixed with a '*'.\n",
       NULL
     },
     { "info",
       "/info",
       "prints information about the table",
       "This command prints some information about the table.\n",
       NULL
     },
     { "start",
       "/start [limit]",
       "starts the game, optionally specifying a betting limit",
       "This command starts the game, dealing cards and collecting the ante.\n",
       "It may only be used by the dealer.\n",
       "If a limit is specified, no bet or raise can be made that is higher than that.\n",
       NULL
     },
     { "cards",
       "/cards",
       "shows the players's cards",
       "This command shows you your cards and analyzes your hand.\n",
       "You should use it before every /discard.\n",
       NULL
     },
     { "give",
       "/give <name> <value>",
       "gives a player some $$$",
       "This command allows you to give a certain amount of money to another player.\n",
       "Of course you can't give more than you have.\n",
       NULL
     },
     { "fold",
       "/fold",
       "fold cards, stop playing for this round",
       "Use this command to fold your cards and stop playing for this round.\n",
       "You can only use this command if it is your turn.\n",
       NULL
     },
     { "pass",
       "/pass",
       "pass a bet",
       "Use this command to make no bet but pass the bet to the next player.\n",
       "You can only use this command if it is your turn and if the player\n",
       "right to you is the dealer or if he just passed, too.\n",
       NULL
     },
     { "bet",
       "/bet <value>",
       "bet value",
       "Use this command to bet a certain value.\n",
       "Only the player left to the dealer can bet and he can do so only at the start of each betting round.\n",
       "Of course you can't bet more than you have.\n",
       NULL
     },
     { "call",
       "/call",
       "call to a bet",
       "Use this command to call to an outstanding bet.\n",
       "Of course you can't call if you don't own enough money.\n",
       NULL
     },
     { "raise",
       "/raise <value>",
       "call to a bet and raise by value",
       "Use this command to call to an outstanding bet, raising by the specified value.\n",
       "Note that you can only raise once in each betting round.\n",
       "Of course you can't call and raise if you don't own enough money.\n",
       NULL
     },
     { "discard",
       "/discard <cards>",
       "discard some cards",
       "Use this command to discard none, some or all of your cards.\n"
       "If the argument is 'none', no cards are discarded. To specify some cards,\n",
       "give their numbers (use /cards to see them) as argument, not separated by any\n",
       "whitespaces or such. That means, '/discard 134' will discard the first, third\n",
       "and fourth card of your deck.\n"
       "Note that you can discard only once between the two betting rounds.\n",
       NULL
     },
     { "help",
       "/help [topic]",
       "help or help on specific topic",
       "Use this command to get a list of all commands or help on a specific command.\n",
       NULL
     },
     { NULL 
     }
};

int main(int argc, char **argv)
{
     int port=DEF_PORT;
     int sock;
     if(argc==2) port=atoi(argv[1]);
     signal(SIGPIPE,SIG_IGN);
     sock=init(port);
     loop(sock);
     exit(0);
}

int init(int p)
{
     static int port;
     int sock, v=1;
     struct sockaddr_in addr;

     if(p!=0)
	  port=p;
     memset(&addr, 0, sizeof(addr));
     addr.sin_family=AF_INET;
     addr.sin_addr.s_addr=htonl(INADDR_ANY);
     addr.sin_port=htons(port);
     if((sock=socket(PF_INET,SOCK_STREAM,0))<0) {
	  perror("socket()");
	  exit(1);
     }
     if(setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &v, sizeof(v))<0)
	  perror("warning: setsockopt() failed for SO_REUSEADDR");
     if(bind(sock, (struct sockaddr*) &addr, sizeof(addr))<0) {
	  perror("bind()");
	  exit(1);
     }
     if(listen(sock, 5)<0) {
	  perror("listen()");
	  exit(1);
     }
     fcntl(sock, F_SETFL, O_NONBLOCK);
     return(sock);
}

void loop(int sock)
{
     int maxfd=sock, plsock=0;
     fd_set readfds, tmpfds;
     struct player_t *player;
     struct table_t *table=calloc(1, sizeof(struct table_t));

     table->card_deck=shuffle_card_deck(shuffle_card_deck(shuffle_card_deck(initialize_card_deck())));
     table->discard=initialize_empty_card_deck();
start:
     table->stage=S_WAIT;
     FD_ZERO(&tmpfds);
     FD_SET(sock, &tmpfds);
     for(;;) {
	  readfds=tmpfds;
	  if(select(maxfd+1, &readfds, NULL, NULL, NULL)<0) {
	       perror("select()");
	       exit(1);
	  }
	  if(FD_ISSET(sock, &readfds)) {
	       plsock=accept(sock, NULL, 0);
	       if(plsock>maxfd)
		    maxfd=plsock;
	       fcntl(plsock, F_SETFL, O_NONBLOCK);
	       insert_player(new_player(plsock), table);
	       FD_SET(plsock, &tmpfds);
	       if(table->count==5)
		    removefd(sock, &maxfd, table, &tmpfds);
	  }
	  player=table->dealer;
	  do {
	       player=player->next;
	       if(player->broken) {
		    struct player_t *tmpplayer=player->next;
		    char *tmpname=strdup(player->name);
		    
		    if(player->state==ACTIVE)
			 table->act_cnt--;
		    player->state=INACTIVE;
		    free(player->name);
		    free(player->addr);
		    free(player->hand);
		    dis_card_deck(player->cards);
		    removefd(fileno(player->socket), &maxfd, table, &tmpfds);
		    remove_player(player, table);
		    free(player);
		    player=tmpplayer;
		    if(table->act_cnt==1) {
			 send_to_all_players(table, "only one player remaining, ending round\n");
			 next_turn(table);
			 end_round_win(table);
		    }
		    if(table->count==0) {
			 table->dealer=table->active=table->raiser=NULL;
			 table->count=table->act_cnt=table->dis_cnt=table->raise=table->pot=table->limit=0;
			 table->flags=F_NONE;
			 table->stage=S_REINIT;
			 goto start;
		    }
		    send_to_all_players(table, tmpname);
		    send_to_all_players(table, " has quit\n");
		    free(tmpname);
	       }
	       if(FD_ISSET(fileno(player->socket), &readfds)) {
		    char buf[BUFSIZE];
		    
		    if(read_from_player(player, buf, BUFSIZE-1)==NULL)
			 continue;
		    if(buf[0]=='/') { 
			 command(table, &player, buf, &tmpfds, &maxfd);
			 if(table->stage!=S_WAIT)
			      removefd(sock, &maxfd, table, &tmpfds);
			 if(table->stage==S_REINIT) {
			      sock=init(0);
			      goto start;
			 }
		    } else {
			 send_to_all_players(table, "<");
			 send_to_all_players(table, player->name);
			 send_to_all_players(table, "> ");
			 send_to_all_players(table, buf);
			 send_to_all_players(table, "\n");
		    }
	       }
	  } while(player->dealer==0 && table->count!=0);
     }
}

void removefd(int fd, int *maxfd, struct table_t *table, fd_set *fds)
{
     if(fd==*maxfd) {
	  struct player_t *player;
	  player=table->dealer;
	  *maxfd=fileno(player->socket);
	  do {
	       player=player->next;
	       if(fileno(player->socket)>*maxfd)
		    *maxfd=fileno(player->socket);
	  } while(player->dealer==0);
     }
     FD_CLR(fd, fds);
     close(fd);
}

void command(struct table_t *table, struct player_t **pplayer, char *buf, fd_set *fds, int *maxfd)
{
     struct player_t *player=*pplayer;
     char c[]=" \n";
     char *cmd=NULL, *arg1=NULL, *arg2=NULL, *tmp=NULL;
     
     buf++;
     cmd=strtok(buf, c);
     if(cmd==NULL)
	  return;
     arg1=strtok(NULL, c);
     if(arg1)
	  arg2=strtok(NULL, c);
     
     if(strcasecmp(cmd, "help")==0) {
	  if(arg1==NULL) {
	       int i;
	       for(i=0; commands[i][0]!=NULL; i++) {
		    send_to_player(player, commands[i][0]);
		    send_to_player(player, "\t- ");
		    send_to_player(player, commands[i][2]);
		    send_to_player(player, "\n");
	       }
	  } else {
	       int r, i=0;
	       do {
		    r=strcasecmp(arg1, commands[i][0]);
	       } while(r && commands[++i]);
	       send_to_player(player, "usage: ");
	       send_to_player(player, commands[i][1]);
	       send_to_player(player, "\n");
	       for(r=3; commands[i][r]!=NULL; r++)
		    send_to_player(player, commands[i][r]);
	  }

     } else if(strcasecmp(cmd, "nick")==0) {
	  if(arg1==NULL)
	       send_to_player(player, "need argument\n");
	  else if(check_name(table, arg1))
	       send_to_player(player, "This name is already in use, please choose a different one.\n");
	  else {
	       tmp=player->name;
	       player->name=strdup(arg1);
	       send_to_all_players(table, tmp);
	       send_to_all_players(table, " is now known as ");
	       send_to_all_players(table, player->name);
	       send_to_all_players(table, "\n");
	       free(tmp);
	  }
	  
     } else if(strcasecmp(cmd,"whois")==0) {
	  struct player_t *whois=table->dealer;
	  char help[15];
	  
	  if(arg1==NULL)
	       send_to_player(player, "need argument\n");
	  else do {
	       whois=whois->next;
	       if(strcasecmp(arg1, whois->name)==0) {
		    send_to_player(player, whois->name);
		    send_to_player(player, " is on host ");
		    send_to_player(player, whois->addr);
		    send_to_player(player, "\n");
		    send_to_player(player, whois->name);
		    send_to_player(player, " got $");
		    snprintf(help, 14, "%d\n", whois->money);
		    send_to_player(player, help);
		    if(whois->dealer) {
			 send_to_player(player, whois->name);
			 send_to_player(player, " is the dealer\n");
		    }
		    break;
	       }
	       if(whois->dealer)
		    send_to_player(player, "There is no such player.\n");
	  } while(whois->dealer==0);
	  
     } else if(strcasecmp(cmd, "list")==0) {
	  struct player_t *list=table->dealer;
	  do {
	       list=list->next;
	       if(list->dealer)
		    send_to_player(player, "*");
	       send_to_player(player, list->name);
	       send_to_player(player, " on host ");
	       send_to_player(player, list->addr);
	       if(table->stage!=S_WAIT) {
		    send_to_player(player, ", ");
		    if(list->state==ACTIVE)
			 send_to_player(player, "active");
		    else
			 send_to_player(player, "inactive");
	       }
	       send_to_player(player, "\n");
	  } while(list->dealer==0);
	  
     } else if(strcasecmp(cmd, "quit")==0) {
	  send_to_all_players(table,player->name);
	  send_to_all_players(table, " has quit\n");
	  send_to_player(player, "Bye\n");
	  while(player->cards->count)
	       push_first_card(pop_first_card(player->cards), table->discard);
	  if(player->state==ACTIVE)
	       table->act_cnt--;
	  player->state=INACTIVE;
	  *pplayer=player->next;
	  free(player->name);
	  free(player->addr);
	  free(player->hand);
	  dis_card_deck(player->cards);
	  removefd(fileno(player->socket), maxfd, table, fds);
	  remove_player(player, table);
	  free(player);
	  if(table->act_cnt==1) {
	       send_to_all_players(table, "only one player remaining, ending round\n");
	       next_turn(table);
	       end_round_win(table);
	  }
	  if(table->count==0) {
	       table->dealer=table->active=table->raiser=NULL;
	       table->count=table->act_cnt=table->dis_cnt=table->raise=table->pot=table->limit=0;
	       table->flags=F_NONE;
	       table->stage=S_REINIT;
	  }
	  
     } else if(strcasecmp(cmd, "start")==0) {
	  if(table->stage!=S_WAIT)
	       send_to_player(player, "the game is already running\n");
	  else if(table->count<2)
	       send_to_player(player, "you need at least two players\n");
	  else if(player->dealer) {
	       send_to_all_players(table, player->name);
	       send_to_all_players(table, " has started the game.\n");
	       send_to_all_players(table, "locking the door...\n");
	       if(arg1!=NULL) {
		    int limit=atoi(arg1);
		    if(limit>MIN_BET) {
			 table->limit=limit;
			 send_to_all_players(table, "the raise limit is $");
			 send_to_all_players(table, arg1);
			 send_to_all_players(table, "\n");
		    } else {
			 send_to_player(player, "invalid limit, playing without limit\n");
			 table->limit=0;
		    }
	       }
	       send_to_all_players(table, "collecting the ante...\n");
	       ante(table);
	       send_to_all_players(table, "shuffling cards...\n");
	       table->card_deck=shuffle_card_deck(lift_card_deck(table->card_deck));
	       send_to_all_players(table, "dealing cards to all active players...\n");
	       deal(table);
	       table->active=player;
	       clear_raises(table);
	       next_turn(table);
	       table->flags=F_PASS;
	  } else send_to_player(player, "only the dealer can start the game\n");
	  
     } else if(strcasecmp(cmd, "info")==0) {
	  char help[80];
	  send_to_player(player, "Game status: ");
	  send_to_player(player, get_stage(table));
	  send_to_player(player, "\n");
	  send_to_player(player, "The pot is $");
	  snprintf(help, 79, "%d\n", table->pot);
	  send_to_player(player, help);
	  send_to_player(player, table->dealer->name);
	  send_to_player(player," is the dealer\n");
	  if(table->active) {
	       send_to_player(player, table->active->name);
	       send_to_player(player, " is the active player\n");
	  }
	  if(table->limit) {
	       send_to_player(player, "the raise limit is $");
	       snprintf(help, 79, "%d\n", table->limit);
	       send_to_player(player, help);
	  }
	  if(table->raiser) {
	       send_to_player(player, table->raiser->name);
	       send_to_player(player, " did the last raise\n");
	       send_to_player(player, "the last raise was $");
	       snprintf(help, 79, "%d\n", table->raise);
	       send_to_player(player, help);
	  }
	  snprintf(help, 79, "the card deck contains %d cards\n", table->card_deck->count);
	  send_to_player(player, help);
	  snprintf(help, 79, "the discard contains %d cards\n", table->discard->count);
	  send_to_player(player, help);
	  snprintf(help, 79, "there are %d total players\n", table->count);
	  send_to_player(player, help);
	  snprintf(help, 79, "there are %d active players\n", table->count);
	  send_to_player(player, help);
	  
     } else if(strcasecmp(cmd, "cards")==0) {
	  if(player->cards==NULL || player->cards->count==0) {
	       send_to_player(player, "you got no cards\n");
	  } else {
	       int i;
	       char deck[100];
	       player->cards=sort_card_deck_by_number(player->cards);
	       snprint_card_deck(deck, 99, player->cards);
	       send_to_player(player, deck);
	       player->hand=check(player->cards);
	       send_to_player(player, "hand: ");
	       send_to_player(player, get_combination(player->hand));
	       send_to_player(player, "\n");
	       if(player->hand->hcards) {
		    send_to_player(player, "hand cards: ");
		    for(i=0; i!=player->hand->hcards; i++) {
			 struct card_t tmp;
			 tmp.number=player->hand->cards[i];
			 send_to_player(player, get_number(&tmp));
			 send_to_player(player," ");
		    }
		    send_to_player(player, "\n");
	       }
	       if(player->hand->kcards) {
		    send_to_player(player, "kickers: ");
		    for(i=0; i!=player->hand->kcards; i++) {
			 struct card_t tmp;
			 tmp.number=player->hand->cards[i+player->hand->hcards];
			 send_to_player(player, get_number(&tmp));
			 send_to_player(player, " ");
		    }
		    send_to_player(player, "\n");
	       }
	  }
	  
     } else if(strcasecmp(cmd, "give")==0) {
	  struct player_t *give=table->dealer;
	  int i;
	  if(arg1==NULL || arg2==NULL)
	       send_to_player(player, "need argument\n");
	  else do {
	       give=give->next;
	       if(strcasecmp(arg1, give->name)==0) {
		    i=atoi(arg2);
		    if(i==0)
			 send_to_player(player, "giving nothing?\n");
		    else if(i>player->money) 
			 send_to_player(player, "you dont have enough money\n");
		    else {
			 player->money-=i;
			 give->money+=i;
			 send_to_player(player, "gave to ");
			 send_to_player(player, give->name);
			 send_to_player(player, " $");
			 send_to_player(player, arg2);
			 send_to_player(player, "\n");
			 send_to_player(give, player->name);
			 send_to_player(give, " gave you $");
			 send_to_player(give, arg2);
			 send_to_player(give, "\n");
		    }
	       }
	       if(give->dealer)
		    send_to_player(player, "There is no such player.\n");
	  } while(give->dealer==0);
	  
     } else if(strcasecmp(cmd, "fold")==0) {
	  if((table->stage!=S_ANTE) && (table->stage!=S_CARDS))
	       send_to_player(player, "the game hasn't started yet\n");
	  else if(table->active!=player) 
	       send_to_player(player, "it's not your turn, please wait\n");
	  else {
	       send_to_all_players(table, player->name);
	       send_to_all_players(table, " has folded\n");
	       if(player->dealer)
		    table->flags&=~F_RAISE;
	       player->state=INACTIVE;
	       player->hand=NULL;
	       player->lastraise=0;
	       while(player->cards->count)
		    push_first_card(pop_first_card(player->cards), table->discard);
	       table->act_cnt--;
	       if(table->act_cnt==1) {
		    send_to_all_players(table, "only one player remaining, ending round\n");
		    next_turn(table);
		    end_round_win(table);
	       }
	       check_round(table);
	       next_turn(table);
	  }
	  
     } else if(strcasecmp(cmd, "pass")==0) {
	  if((table->stage!=S_ANTE) && (table->stage!=S_CARDS)) 
	       send_to_player(player, "the game hasn't (re)started yet\n");
	  else if(player!=table->active)
	       send_to_player(player, "it's not your turn, please wait\n");
	  else if((table->flags&F_PASS)!=F_PASS)
	       send_to_player(player, "passing not allowed anymore\n");
	  else if(player->dealer) {
	       send_to_all_players(table, "no bet was made, ending round\n");
	       end_round(table);
	  } else {
	       send_to_all_players(table, player->name);
	       send_to_all_players(table, " has passed\n");
	       next_turn(table);
	  }
	  
     } else if(strcasecmp(cmd, "bet")==0) {
	  int bet;
	  if((table->stage!=S_ANTE) && (table->stage!=S_CARDS))
	       send_to_player(player, "the game hasn't (re)started yet\n");
	  else if(player!=table->active)
	       send_to_player(player, "it's not your turn, please wait\n");
	  else if(arg1==NULL)
	       send_to_player(player, "argument needed\n");
	  else if((table->flags&F_PASS)!=F_PASS)
	       send_to_player(player, "betting not allowed anymore\n");
	  else {
	       bet=atoi(arg1);
	       if(bet<MIN_BET)
		    send_to_player(player, "your bet is too low, try again\n");
	       else if(table->limit && bet>table->limit)
		    send_to_player(player, "your bet is higher than the raise limit, try again\n");
	       else if(bet>player->money)
		    send_to_player(player, "you don't have enough money\n");
	       else {
		    player->money-=bet;
		    table->pot+=bet;
		    table->raise=bet;
		    player->lastraise=bet;
		    table->flags&=~F_PASS;
		    table->flags|=F_RAISE|F_RAISED;
		    if(player->dealer)
			 table->flags&=~F_RAISE;
		    table->raiser=player;
		    send_to_all_players(table, player->name);
		    send_to_all_players(table, " has bet $");
		    send_to_all_players(table, arg1);
		    send_to_all_players(table, "\n");
		    next_turn(table);
	       }
	  }
	  
     } else if(strcasecmp(cmd, "call")==0) {
	  if((table->stage!=S_ANTE) && (table->stage!=S_CARDS))
	       send_to_player(player, "the game hasn't (re)started yet\n");
	  else if(table->active!=player)
	       send_to_player(player, "it's not your turn, please wait\n");
	  else if((table->flags&F_RAISED)!=F_RAISED)
	       send_to_player(player, "there are no outstanding bets to be called\n");
	  else if((table->raise-player->lastraise)>player->money)
	       send_to_player(player, "you don't have enough money\n");
	  else {
	       table->pot+=(table->raise-player->lastraise);
	       player->money-=(table->raise-player->lastraise);
	       if(get_next_active_player(player)==table->raiser) {
		    table->raise=0;
		    table->flags&=~F_RAISED;
	       }
	       if(player->dealer)
		    table->flags&=~F_RAISE;
	       send_to_all_players(table, player->name);
	       send_to_all_players(table, " has called\n");
	       check_round(table);
	       next_turn(table);
	  }
	  
     } else if(strcasecmp(cmd, "raise")==0) {
	  if((table->stage!=S_ANTE) && (table->stage!=S_CARDS))
	       send_to_player(player, "the game hasn't (re)started yet\n");
	  else if(table->active!=player)
	       send_to_player(player, "it's not your turn, please wait\n");
	  else if(arg1==NULL)
	       send_to_player(player, "argument needed\n");
	  else if((table->flags&F_RAISE)!=F_RAISE) {
	       send_to_player(player, "raising not allowed anymore\n");
	  } else {
	       int raise=atoi(arg1);
	       if(raise<MIN_BET)
		    send_to_player(player, "your raise is too low\n");
	       else if(table->limit && raise>table->limit)
		    send_to_player(player, "your raise is higher than the raise limit, try again\n");
	       else if((raise+table->raise)>player->money)
		    send_to_player(player, "you don't have enough money\n");
	       else {
		    table->pot+=table->raise;
		    player->money-=table->raise;
		    table->pot+=raise;
		    player->money-=raise;
		    player->lastraise=raise+table->raise;
		    if(get_next_active_player(player)==table->raiser)
			 table->raise=raise;
		    else
			 table->raise+=raise;
		    table->raiser=player;
		    send_to_all_players(table, player->name);
		    send_to_all_players(table, " has called and raised by $");
		    send_to_all_players(table, arg1);
		    send_to_all_players(table, "\n");
		    table->flags|=F_RAISED;
		    if(player->dealer)
			 table->flags&=~F_RAISE;
		    next_turn(table);
	       }
	  }
	  
     } else if(strcasecmp(cmd, "discard")==0) {
	  if(table->stage!=S_NEXTROUND)
	       send_to_player(player, "you can't discard now\n");
	  else if(player->state==INACTIVE)
	       send_to_player(player, "you are inactive\n");
	  else if(arg1==NULL)
	       send_to_player(player, "argument needed\n");
	  else {
	       int arg[5], i, j=0;
	       char help[15];
	       if(strcasecmp(arg1, "none")==0) {
		    send_to_all_players(table, player->name);
		    send_to_all_players(table, " discarded no cards\n");
	       } else if(strcasecmp(arg1, "all")==0) {
		    while(player->cards->count)
			 push_first_card(pop_first_card(player->cards), table->discard);
		    for(i=0; i!=5; i++)
			 push_first_card(pop_first_card(table->card_deck), player->cards);
		    send_to_all_players(table, player->name);
		    send_to_all_players(table, " discarded all his cards\n");
	       } else {
		    for(i=0; i!=5; i++) {
			 if(arg1[i]==0) {
			      arg[i]=0;
			      break;
			 }
			 arg[i]=arg1[i]-48;
		    }
		    for(i=1; i!=6; i++) {
			 if(arg[j]==0) break;
			 if(arg[j]==i) {
			      push_first_card(pop_first_card(player->cards), table->discard);
			      j++;
			 } else {
			      push_last_card(pop_first_card(player->cards), player->cards);
			 }
		    }
		    send_to_all_players(table, player->name);
		    send_to_all_players(table, " discarded ");
		    snprintf(help, 14, "%d", j);
		    send_to_all_players(table, help);
		    send_to_all_players(table, " cards\n");
		    while(j--)
			 push_first_card(pop_first_card(table->card_deck), player->cards);
	       }
	       table->dis_cnt++;
	       if(table->dis_cnt==table->act_cnt) {
		    send_to_all_players(table, "all players discarded, going on to next betting round\n");
		    table->stage=S_CARDS;
		    table->flags=F_PASS;
		    table->active=table->dealer;
		    clear_raises(table);
		    next_turn(table);
	       }
	  }
	  
     } else {
	  send_to_player(player, "unknown command: ");
	  send_to_player(player, cmd);
	  send_to_player(player, "\n");
     }
}
