/*  pokerd - a classical draw poker server
 *  Copyright (C) 2001,2003 Hans P. Rosenfeld <rosenfeld@grumpf.hope-2000.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 *  check() - returns the hand a user has. inspired by kpoker.
 */

#include <stdlib.h>
#include "check.h"

char *combinations[]={ "High Card",      /* 0 */
		       "One Pair",       /* 1 */
		       "Two Pairs",      /* 2 */
		       "Trips",          /* 3 */
		       "Straight",       /* 4 */
		       "Flush",          /* 5 */
		       "Full House",     /* 6 */
		       "Quads",          /* 7 */
		       "Straight Flush", /* 8 */
		       "Royal Flush"     /* 9 */
};

char *get_combination(struct comb_t *comb)
{
     return(combinations[comb->comb]);
}

int check_flush(struct card_deck_t *card_deck)
{
     struct card_t *card=get_first_card(card_deck);
     int color=card->color;

     do {
	  card=card->next;
	  if(card->color!=color)
	       return(0);
     } while(card->next);
     return(1);
}

int check_straight(struct card_deck_t *card_deck)
{
     int cards[5],i;
     struct card_t *card;

     card_deck=sort_card_deck_by_number(card_deck);
     for(i=0; i!=5; i++) {
	  card=pop_first_card(card_deck);
	  cards[i]=card->number;
	  push_last_card(card,card_deck);
     }
     for(i=0; i!=3; i++)
	  if(cards[i]!=cards[i+1]-1)
	       return(0);   
     if(cards[0]==0 && cards[5]==12) /* lowest card is Two, highest is Ace -> Ace counts as One */
	  return(1);
     if(cards[3]==cards[4]-1)
	  return(1);
     else
	  return(0);
     if(cards[4]==12)                /* the highest card is Ace -> might be a Royal Flush */
	  return(HIGH_STRAIGHT);
}

int cmp(const void *c1, const void *c2)
{
     return(*(int*)c2-*(int*)c1);
}

struct comb_t *check(struct card_deck_t *card_deck)
{
     int royal=0;
     struct comb_t *comb=calloc(1, sizeof(struct comb_t));

     if(card_deck->count!=5)
	  return(NULL);
     comb->comb=HIGH_CARD;
     comb->hcards=5;
     comb->kcards=0;
     if(check_flush(card_deck))
	  if((royal=check_straight(card_deck)))
	       if(royal==HIGH_STRAIGHT) {
		    comb->comb=ROYAL_FLUSH;
		    memset(comb->cards, 0, 5);
		    comb->hcards=0;
		    comb->kcards=0;
	       } else {
		    comb->comb=STRAIGHT_FLUSH; 
		    sort_card_deck_by_number(card_deck);              
		    comb->cards[0]=card_deck->first->number;
		    memset(comb->cards+1, 0, 4);              
		    comb->hcards=1;
		    comb->kcards=0;
	       }
	  else {
	       int i=0;
	       struct card_t *card;
	       comb->comb=FLUSH;
	       sort_card_deck_by_number(card_deck);
	       for(card=get_first_card(card_deck); card->next!=NULL; card=card->next)
		    comb->cards[i++]=card->number;
	       comb->hcards=5;
	       comb->kcards=0;
	  }
     else
	  if(check_straight(card_deck)) {
	       comb->comb=STRAIGHT;
	       sort_card_deck_by_number(card_deck);              
	       comb->cards[0]=card_deck->first->number;
	       memset(comb->cards+1, 0, 4);
	       comb->hcards=1;
	       comb->kcards=0;
	  } else { 
	       struct card_deck_t *tmp_deck=calloc(1, sizeof(struct card_deck_t));
	       struct card_deck_t *kickers=calloc(1, sizeof(struct card_deck_t));
	       int match=0, pair=0;
	       int i=0, j=0;
	       while(card_deck->count) {
		    struct card_t *card=pop_first_card(card_deck);
		    struct card_deck_t *swap_deck=calloc(1, sizeof(struct card_deck_t));
		    while(card_deck->count)
			 if(card_deck->first->number==card->number) {
			      match++;
			      push_first_card(pop_first_card(card_deck),tmp_deck);
			 } else
			      push_first_card(pop_first_card(card_deck),swap_deck);
		    while(swap_deck->count)
			 push_first_card(pop_first_card(swap_deck),card_deck);
		    switch(match) {
		    case 1:               /* pair found */
			 if(comb->comb!=TRIPS) {
			      if(!pair) { /* first pair */
				   comb->comb=ONE_PAIR;
				   comb->cards[0]=tmp_deck->first->number;
				   pair=1;
				   comb->hcards=1;
				   comb->kcards=3;
			      } else {   /* second pair */
				   comb->comb=TWO_PAIRS;
				   if(comb->cards[0]>tmp_deck->first->number)
					comb->cards[1]=tmp_deck->first->number;
				   else {
					comb->cards[1]=comb->cards[0];
					comb->cards[0]=tmp_deck->first->number;
				   }
				   comb->hcards=2;
				   comb->kcards=1;
			      }
			 } else {        /* pair after trips: Full House */
			      comb->comb=FULL_HOUSE;
			      comb->cards[1]=tmp_deck->first->number;
			      comb->hcards=2;
			      comb->kcards=0;
			 }
			 break;
		    case 2:             /* trips found */
			 if(!pair) {    /* real trips */
			      comb->comb=TRIPS;
			      comb->cards[0]=tmp_deck->first->number;
			      comb->hcards=1;
			      comb->kcards=2;
			 } else {       /* trips after pair: Full House */
			      comb->comb=FULL_HOUSE;
			      comb->cards[1]=comb->cards[0];
			      comb->cards[0]=tmp_deck->first->number;
			      comb->hcards=2;
			      comb->kcards=0;
			 }
			 break;
		    case 3:             /* quads found */
			 comb->comb=QUADS;
			 comb->cards[0]=tmp_deck->first->number;
			 comb->hcards=1;
			 comb->kcards=1;
			 break;
		    }
		    if(match) push_first_card(card,tmp_deck);
		    else push_first_card(card,kickers);
		    match=0;
	       }
	       if(kickers->count) {
		    sort_card_deck_by_number(kickers);
		    switch(comb->comb) {
		    case HIGH_CARD:
			 i=0;
			 break;
		    case ONE_PAIR:
		    case TRIPS:
		    case QUADS:
			 i=1;
			 break;
		    case TWO_PAIRS:
		    case FULL_HOUSE:
			 i=2;
			 break;
		    }
		    j=i;
		    while(kickers->count) {
			 struct card_t *card=pop_first_card(kickers);
			 comb->cards[j++]=card->number;
			 push_first_card(card, tmp_deck);
		    }
		    qsort(comb->cards+i, 5-i, sizeof(int), &cmp);
	       }
	       while(tmp_deck->count)
		    push_first_card(pop_first_card(tmp_deck), card_deck);
	       free(kickers);
	       free(tmp_deck);
	  }
     sort_card_deck_by_number(card_deck);
     return(comb);
}

int compare_combs(struct comb_t *comb1, struct comb_t *comb2)
{
     int i;
     if(comb1->comb>comb2->comb) return(1);
     if(comb1->comb<comb2->comb) return(-1);
     for(i=0; i!=5; i++)
	  if(comb1->cards[i]>comb2->cards[i]) return(1);
	  else if(comb1->cards[i]<comb2->cards[i]) return(-1);
     return(0);
}

#ifdef CHECKDEBUG

int main(int argc, char **argv)
{
     for(;;) {
	  int i=5;
	  char c;
	  struct card_deck_t *card_deck, *hand=calloc(1, sizeof(struct card_deck_t));
	  struct comb_t *comb;
	  card_deck=initialize_card_deck();
	  while(i--)
	       card_deck=shuffle_card_deck(card_deck);
	  for(i=0; i!=5; i++)
	       push_first_card(pop_first_card(card_deck),hand);
	  print_card_deck(stdout, hand);
	  comb=check(hand);
	  printf("hand: %s\n", get_combination(comb));
	  if(comb->hcards) {
	       printf("hand cards: ");
	       for(i=0; i!=comb->hcards; i++)
		    printf("%d ", comb->cards[i]+2);
	       printf("\n");
	  }
	  if(comb->kcards) {
	       printf("kickers: ");
	       for(i=0; i!=comb->kcards; i++)
		    printf("%d ", comb->cards[i+comb->hcards]+2);
	       printf("\n");
	  }
	  printf("press enter.\n");
	  scanf("%c",&c);
     }
}

#endif

