;***********************************************************************
; SET_ILACE_MODE: (MODE)
;
;	Set_Interlace_Mode is called to provide program control over the
; current interlace mode.  This routine will call the appropriate
; low-level routine to actually perform the I/O.
;
; Input:
;	AL: Interlace Mode, as follows:
;		AL = 0:	Standard 25 line display.
;		AL = 1:	High-density 25 line display.
;		AL = 2: Interlaced 50 line display (in text mode only).
;***********************************************************************
SET_ILACE_MODE PROC NEAR
	PUBLIC	SET_ILACE_MODE
	PUSHREG	<AX,BP>
	CMP	AL,3			;Is this a valid scroll mode?
	JAE	SIM1			;No, ignore the request
	MOV	VIDEO_ACTIVE,FALSE	;Yes - let the request run
	MOV	AH,AL			;Put the request in AH
	CALL	CASE			;Execute the desired routine
	JMP	NEAR PTR SIM1		;All routines return here
	DW	DISABLE_INTERLACE	;If AL=0, Disable interlace mode
	DW	ENABLE_FATC		;If AL=1, Enable Fat-Character mode
	DW	ENABLE_INTERLACE	;If AL=2, Enable interlace mode
SIM1:	POPREG	<BP,AX>
	RET
SET_ILACE_MODE ENDP



;**********************************************************************
; ENABLE_INTERLACE:
;
;	Enable_Interlace is called to turn on interlace mode.  This
; call will only take effect if the current video display board is in
; a text mode.  Calling this routine acts to set the number of displayed
; lines to 50.
;**********************************************************************
ENABLE_INTERLACE PROC NEAR
	PUBLIC	ENABLE_INTERLACE
	PUSHREG	<AX,BX,CX,DX>
	CMP	VIDEO_ACTIVE,TRUE	;Are we scrolling now?
	JE	EINT3			;Yes - ignore the request!
	CMP	VIDEO_MODE,3		;Are we in a color text mode?
	JBE	EINT1			;Yes - can enter interlace
	CMP	VIDEO_MODE,7		;In a monochrome text mode?
	JNE	EINT3			;No, ignore this call
EINT1:	CMP	LAST_ROW,NUMBER_OF_ROWS*2-1 ;Is interlace already on?
	JE	EINT3			;Yes - no point in doing it again!
	MOV	AL,0			;No, turn on page number 0
	CALL	SET_DISPLAY_PAGE
	MOV	DX,VIDEO_BASE		;Point to the video hardware port
EINT2:	MOV	AL,INTERLACE		;Get the interlace control reg #
	OUT	DX,AL			;Select the interlace register
	INC	DX			;Point to the data port
	MOV	AL,EN_INTERLACE		;Get command to enable interlace
	OUT	DX,AL			;Turn interlace on
	MOV	LAST_ROW,NUMBER_OF_ROWS*2-1 ;Set number of lines
	MOV	AX,SCREEN_BYTES		;Use all of memory for video buffer
	MOV	PAGE_SIZE,AX		;Set interlace page size to all of mem
	DEC	AX			;Make this an addressing mask
	MOV	TEXT_SCREEN_MASK,AX	;Set the global addressing mask
	MOV	BH,0			;Set the page number to 0
	MOV	AH,8			;Get code to read a character
	INT	VIDEO_IO_INTR		;Read the current screen attribute
	MOV	BH,AH			;Place the attribute in BH
	MOV	CH,NUMBER_OF_ROWS	;Set coordinates of new lines in CX
	MOV	CL,0
	MOV	DH,LAST_ROW		;Point DX to end of new lines
	MOV	DL,BYTE PTR NUMBER_OF_COLUMNS ;Get the last char posn on the screen
	DEC	DL
	MOV	AX,(6 SHL 8) OR 0	;Set command to clear this window
	INT	VIDEO_IO_INTR		;Clear the lower 25 lines!
	MOV	AL,0			;Get page number 0
	CALL	SET_DISPLAY_PAGE	;Can only interlace on page 0!
EINT3:	POPREG	<DX,CX,BX,AX>
	RET
ENABLE_INTERLACE ENDP



;**********************************************************************
; DISABLE_INTERLACE:
;
;	Disable_Interlace will turn interlace off, if it is already 
; turned on.  This call will automatically adjust the screen RAM so
; that the last 25 lines are displayed.
;**********************************************************************
DISABLE_INTERLACE PROC NEAR
	PUBLIC	DISABLE_INTERLACE
	PUSHREG	<AX,CX,DX,SI,DI,ES>
	CMP	VIDEO_ACTIVE,TRUE	;Is the screen being scrolled?
	JE	DINT3			;Yes - don't interfere!
	MOV	DX,VIDEO_BASE		;No, point to the video hardware
DINT1:	MOV	AL,INTERLACE		;Get the interlace control reg.
	OUT	DX,AL			;Select the interlace port
	INC	DX			;Point to the data port
	MOV	AL,DIS_INTERLACE	;Get command to disable interlace
	OUT	DX,AL			;Turn interlace off
	CMP	LAST_ROW,NUMBER_OF_ROWS*2-1 ;Is interlace on?
	JNE	DINT3			;No, don't turn it off!
	MOV	CX,CURSOR_POSITION[0]	;Get the current cursor position
	SUB	CH,NUMBER_OF_ROWS	;Find out where last 25 lines are
	JB	DINT2			;On first 25 lines already!
	INC	CH			;Point to the first line to move
	MOV	CL,0			;Start the move at beginning of line
	CALL	SET_SCREEN_PTR		;Point to start of the 25 lines
	MOV	SI,DI			;Use this as a source pointer
	MOV	DI,PAGE_BASE[0]		;Point DI to the start of the screen
	MOV	AL,NUMBER_OF_ROWS	;Get the number of rows to move
	MUL	BYTE PTR ROW_INCREMENT	;Find the number of bytes to move
	MOV	CX,AX			;Place the byte count in CX
	CALL	MOVE_SCREEN_BYTES	;Copy last 24 lines to top of screen
	MOV	DX,CURSOR_POSITION[0]	;Get the old cursor position
	MOV	DH,NUMBER_OF_ROWS-1	;Must now be on the last line
	MOV	BH,0			;...for page number 0
	CALL	SET_CURSOR_POSITION	;Set the new cursor position
DINT2:	MOV	LAST_ROW,NUMBER_OF_ROWS-1;Set the last row number
	MOV	AX,SCREEN_BYTES / 4	;Set normal number of screen bytes
	MOV	PAGE_SIZE,AX		;Store as the new page size
	DEC	AX			;Make it a page mask
	MOV	TEXT_SCREEN_MASK,AX	;Store the mask in memory
	MOV	AL,0			;Set the display page...
	CALL	SET_DISPLAY_PAGE	;...to set the scrolling hardware
DINT3:	POPREG	<ES,DI,SI,DX,CX,AX>
	RET
DISABLE_INTERLACE ENDP



;**********************************************************************
; ENABLE_FATC:
;
;	Enable_FATC is called to enable the fat-character mode.
; This mode generates an interlaced display where alternate scan
; lines are repeated.
;**********************************************************************
ENABLE_FATC PROC NEAR
	PUBLIC	ENABLE_FATC
	PUSHREG	<AX,DX>
	CALL	DISABLE_INTERLACE	;Turn off interlace mode if on
	MOV	DX,VIDEO_BASE		;Point to the video hardware
	MOV	AL,INTERLACE		;Get the interlace control reg.
	OUT	DX,AL			;Select the interlace port
	INC	DX			;Point to the data port
	MOV	AL,EN_FATC		;Get command to use fat characters
	OUT	DX,AL			;Turn fat char mode on
	POPREG	<DX,AX>
	RET
ENABLE_FATC ENDP



