Page	60,132
Title	-Keyboard Drivers
;
;
;	Copyright (c) 1983
;	Zenith Data Systems
;	St. Joseph, Michigan
;
;	Copyright (c) 1985
;	Morrow Designs, Inc.
;	San Leandro, California
;
;	Last Update 9_Jul_85
;
;
%Out	Keyboard.asm
page
;======================================================================
Monitor_Segment Segment Word Public
;==================================
;
Assume	cs:Monitor_Segment, ds:Rom_Data, ss:Nothing, es:Nothing


; Include Files
;--------------
;	Rom.lit
;	Intr.lit
;	Sys.ext
;	Io.lit
;	Keyboard.lit
;	Keyboard.ext

	.xlist
	include	../Rom/Rom.lit
	include	../Rom/Intr.lit
	include ../Rom/Sys.ext
	include	../Io/Io.lit
	include	Keyboard.lit
	include	Keyboard.ext
	.list

	extrn	DS_to_Rom_Data:Near
	extrn	Out_Morrow_6845:Near
	extrn	Case:Near
	extrn	Beep:Near
	extrn	Delay:Near
	extrn	Set_Monitor_Stack:Near
	extrn	Set_User_Stack:Near

	extrn	Start_Up:Far
	extrn	Start_Up_Warm:Far

	extrn	Key_Table:Byte
	extrn	Shift_Table:Byte
	extrn	Ctrl_Table:Byte
	extrn	Alt_Table:Byte
	extrn	Key_Type:Byte
	extrn	Old_Shift_Status:Byte

page
;**********************************************************************
; KEYBOARD_IO_INTERRUPT: (FUNCTION, PARAMETERS)	     Interrupt 16H (22)
;
;	Keyboard_IO_Interrupt is the ROM entry point to communicate with
; the keyboard processor.  The following routines are executed by first 
; loading parameters into registers as specified below, and then 
; placing an operation code number in AH.  Finally, an INT 16H instruc-
; tion is executed to perform the function.  The following function 
; codes are recognized by the Keyboard_IO_Interrupt routine:
;
; Function Code 0 - Read Character from Keyboard
;   Output:
;	AL: Character value in ASCII
;	AH: Keyboard processor scan code
;
; Function Code 1 - Check Keyboard Status
;   Output:
;	ZF: The Zero flag is set upon return if no characters are
;	    waiting in the buffer to be processed.  If the Zero
;	    flag is not set, then at least one character is in the
;	    buffer.
;	AX: If characters are waiting in the keyboard buffer, then
;	    AX will contain the first pair of character codes (as
;	    above).
;
; Function Code 2 - Determine Current Shift Status
;   Output:
;	AL: Bit-mapped, as follows:
;		Bit 7 - Insert mode active
;		Bit 6 - Caps_Lock active
;		Bit 5 - Num_Lock active
;		Bit 4 - Scroll_Lock active
;		Bit 3 - Alt key currently pressed
;		Bit 2 - Ctrl key currently pressed
;		Bit 1 - Left Shift key currently pressed
;		Bit 0 - Right shift key currently pressed
;
;**********************************************************************
KEYBOARD_IO_INTERRUPT PROC FAR
	PUBLIC	KEYBOARD_IO_INTERRUPT

	PUSHREG	<BX,BP,DS>		;Save registers
	call	DS_to_Rom_Data		;DS:= Rom Data
	STI				;Let keys into buffer!
	CMP	AH,KMAX			;Is this a valid functon code?
	JAE	KBI1			;No, ignore it
	CALL	CASE			;Yes - execute the proper routine
	JMP	NEAR PTR KBI1		;All routines return here...

KCMD:	DW	GET_KEY_BUFF		;If AH=0, Read Key from Buffer
	DW	GET_KEY_STATUS		;If AH=1, Return Keyboard Status
	DW	GET_Shift_Status	;If AH=2, Return Shift Status
KMAX	EQU	($-KCMD) / 2

KBI1:	POPREG	<DS,BP,BX>		;Restore registers
	JC	KBI2			;Routine wants to save flags
	RET	2			;Return and dump old flags
KBI2:	IRET				;Return with flags saved

KEYBOARD_IO_INTERRUPT ENDP


page
;----------------------------------------------------------------------
Get_Key_Buff Proc Near; (9_Jul_85)
;---------------------------------
;	1) Get_Key_Buff retrieves the next key from the keyboard buffer.
;	2) If at least one key isn't already in the buffer, Get_Key_Buff will
;	   wait for one to be entered.
;	3) Output Register Values:
;		AL -> ASCII code for the entered key.
;		AH -> Keyboard processor scan code for this key.
;
Public	Get_Key_Buff

	PushReg	<si,ds,es>
	call	DS_to_Rom_Data		;DS:= Rom Data
	pushf				;Save current interrupt-enable status
Gkb1:	cli				;Loop	Prevent new keys for a bit
	call	ES_to_Key_Buff		;	ES:= Rom Data
	mov	si,Key_Buff.Head	;	SI:= Buffer Pointer
	cmp	si,Key_Buff.Tail	;	Check for chars in the buffer

	pushf				;	(Save key present status)
	mov	ax,Word Ptr es:[si]	;	AX:= Char from buffer
	inc	si
	inc	si			;	SI:= Next Character
	popf				;	If (there was a key typed)
	jnz	Gkb2			;		Break

	sti				;	let waiting interrupts occur
	jmp	Short Gkb1

Gkb2:	call	Adjust_Key_Ptr		;Wrap pointer around if necessary
	mov	Key_Buff.Head,si	;Update pointer for next pass
	popf				;Restore old interrupt enable status
	clc				;Set new-flags status for return
	PopReg	<es,ds,si>
	ret				;Return

Get_Key_Buff	EndP

page
;----------------------------------------------------------------------
Get_Key_Status Proc Near; (9_Jul_85)
;-----------------------------------
;	1) Get_Key_Status checks to see if any keys are waiting in
;	   the keyboard buffer.
;	2) Register Output Values:
;		ZF -> If the Zero Flag is set upon return, then no keys exist
;	    	      in the keyboard buffer.  If the Zero Flag is reset,
;		      however, then at least one key is in the keyboard buffer.
;		AX -> If any keys are in the keyboard buffer, then AX will
;		      contain the first key in the buffer
;			AL = ASCII
;			AH = Scan code
;
Public	Get_Key_Status

	PushReg	<bx,si,ds,es>
	call	DS_to_Rom_Data		;DS:= Rom Data
	pushf				;Save current interrupt enable status
	cli				;Prevent new keys for a bit
	call	ES_to_Key_Buff		;ES:= Rom Data
	mov	si,Key_Buff.Head	;Retrieve the buffer pointer
	cmp	si,Key_Buff.Tail	;Are there any chars in the buffer?
	mov	ax,Word Ptr es:[si]	;[Get char from buffer, in case]
	mov	bx,ax			;Save key value in BX
	lahf				;Get status of zero flag in AH
	popf				;Restore old intr. enable status

	sahf				;Restore zero flag status from AH
	mov	ax,bx			;Restore key code value
	clc				;Set new-flags return parameter
	PopReg	<es,ds,si,bx>
	ret

Get_Key_Status	EndP

page
;**********************************************************************
; GET_Shift_Status:
;
;	Get_Shift_Status retrieves the current status of the keyboard
; shift/lock/mode flags.
;
; Output:
;	AL: Bit-mapped, as follows:
;		Bit 7 - Insert mode active
;		Bit 6 - Caps_Lock active
;		Bit 5 - Num_Lock active
;		Bit 4 - Scroll_Lock active
;		Bit 3 - Alt key currently pressed
;		Bit 2 - Ctrl key currently pressed
;		Bit 1 - Left Shift key currently pressed
;		Bit 0 - Right shift key currently pressed
;**********************************************************************
GET_Shift_Status PROC NEAR

	MOV	AL,Shift_Status		;Retrieve shift mode status
	XOR	AH,AH			;Set AH to 0, set ZF for compatibility
	STC				;Set save-flags return parameter
	RET
GET_Shift_Status ENDP

page
;**********************************************************************
; KEYBOARD_INTERRUPT:
;
;	Keyboard_Interrupt is the actual interrupt routine first executed
; when a key has been pressed.  It reads in the key value, attempts
; to process it, and then places it in the keyboard buffer if it is
; a valid code-generating key.
;
;**********************************************************************
KEYBOARD_INTERRUPT PROC FAR
	PUBLIC	KEYBOARD_INTERRUPT
	PUSHREG	<AX,BX,CX,DX,SI,DI,DS,ES> ;Save processor state
	call	DS_to_Rom_Data		;DS:= Rom Data
	MOV	SI,False		;Clear the end-of-interrupt-sent flag
	CMP	Feed_Key,False		;while key is being fed from software)
	JNE	KI0			;   ignore keyboard keys
	IN	AL,Keyboard_Data_Port	;get the scan code from kbrd
	MOV	AH,AL			;save it in AH
	IN	AL,Keyboard_Ctrl_Port	;Now read the kbrd control port
	OR	AL,Keyboard_Reset	;Set the clear-keyboard bit
	OUT	Keyboard_Ctrl_Port,AL	;clear the keyboard
	AND	AL,NOT Keyboard_Reset	;clear the reset bit
	OUT	Keyboard_Ctrl_Port,AL	;Enable the keyboard
KI0:	MOV	Feed_Key,False		;Set the key-feed flag to normal-key
	MOV	AL,AH			;get scan code back
	CALL	Set_Monitor_Stack	;Use the local stack while in routines
KI1:	CALL	Control_Process		;check, and process if 'control' key
	MOV	CH,AL			;save key code
					;If (Key Processed by Control_Process)
	JC	KI11			;    then skip to end of interrupt
					;else
	TEST	AL,80H			;if 'normal' key break code
	JNZ	KI11			;    then ignore it and skip to end of interrupt

	TEST	Shift_Status,Alt_Flag	;if (ALT key active)
	JZ	KI3			;  then
	CALL	Alt_Mode		;     process as ALT mode key
	JMP	short KI11		;     then skip to end of interrupt

KI3:	TEST	Shift_Status,Ctrl_Flag	;if (CTRL key active)
	JZ	KI4			;  then
	CALL	Ctrl_Mode		;     process as control key
	JMP	short KI11		;     then skip to end of interrupt

KI4:	MOV	CL,False		;assume unshifted key table to be used
	TEST	AH,Pad_Type		;If (key NOT From numeric pad)
	JZ	KI5			;  then don't check Num_Lock
					;else
	TEST	Shift_Status,Num_Lock	;If (Num_Lock currently active)
	JNZ	KI6			; then  ?????????
	MOV	BX,offset Key_Table	; else select normal keyboard table
	JMP	short KI10		;   and process scan code

KI5:	TEST	AH,Alpha_Type		;If (alpha key)
	JZ	KI7			;  then

	TEST	Shift_Status,Caps_Lock	;   If (Caps_Lock active)
	JZ	KI7			;      then

KI6:	MOV	CL,True			;       select shifted key  table
KI7:	TEST	Shift_Status,Right_Flag OR Left_Flag	;if (a shift key pressed)
	JZ	KI8			;                  then

	XOR	CL,True			;                     over-ride Caps Lock with Shift key
KI8:	MOV	BX,offset Shift_Table	;                     select Shifted Key table
	TEST	CL,CL			;		      if (NOT Shifted key)
	JNZ	KI9			;                       then

	MOV	BX,offset Key_Table	;			  select normal key table
	JMP	short KI10		;Doing lower case, so print screen not possible
KI9:

KI10:	CALL	Put_Table_Key		;Convert key using [bx] table and put in buffer

KI11:	MOV	Last_Key,CH		;Update the 'last-key-typed' storage
	CMP	SI,True			;if (End-of-interrupt NOT sent yet)
	JE	KI12			;  then

	MOV	AL,End_Of_Intr		;     
	OUT	Interrupt_Ctrl,AL	;      Output end-of-interrupt command

KI12:
	mov	al,Shift_Status		; set new shift status
	mov	cs:Old_Shift_Status,al	;  in Old_Shift_Status
	
	CALL	Set_User_Stack		;Restore the user's stack
	POPREG	<ES,DS,DI,SI,DX,CX,BX,AX> ;[Restore user's registers]
	IRET
KEYBOARD_INTERRUPT ENDP



;**********************************************************************
; CONTROL_PROCESS: (KEY_CODE)
;
;	Control_Process is called to determine if a key is a control
; key, and to process it if it is.  'Control' keys include the
; SHIFT/CTRL/ALT, INS and LOCK keys.  It also handles keyboard buffer
; overflow.
;
; Input:
;	AL: Key code (scan code from keyboard)
;
; Output:
;	AL: Key scan code (unmodified)
;	CY: Control key flag.  Set if the key passed to CONTROL_KEY
;	    was treated as one of the control keys.  If the carry
;	    flag is reset, then the key was not a control key, and was
;	    ignored.
;	AH: If the control flag was not set (indicating that further
; 	    processing is required for this key), then AH will
;	    contain the key type (from the KEY_TYPE array, below).
;**********************************************************************
CONTROL_PROCESS PROC NEAR
	PUBLIC	CONTROL_PROCESS
	PUSHREG	<BX,CX>
	MOV	CH,AL			;Save scan code in CH
	AND	AL,7FH			;Mask off break bit
	MOV	BX,offset Key_Type	;Point BX to the Key_Type table
	XLAT	CS: Key_Type		;Convert Scan code to key type
	MOV	AH,AL			;save key type in AH
	MOV	AL,CH			;Get back scan code 
	CMP	AL,Buffer_Full		;if (keyboard processor overrun code)
	JNE	CP1			;  then
	CMP	Last_Key,Buffer_Full	;    if last scan code Not an overrun
	JE	CP0			;      then
	CALL	Beep			;        BEEP!!!!!
CP0:	JMP	CP8			;        Exit keyboard handler
					; else
CP1:	TEST	Shift_Status,Ctrl_Flag	;  if(CTRL key active)
	JZ	CP2			;    then

	TEST	AH,Ctrl_Override	;    Is (key special in CTRL mode)
	JZ	CP2			;       then

	JMP	CP9			;         Pass it off to CTRL_MODE handler

CP2:	MOV	CL,Shift_Status		;Pick up the current shift status
	MOV	BL,CL			;Copy it into BL
	AND	CL,Right_Flag OR Left_Flag ;if (shift mode active) 
	ADD	CL,0FFH			;     then Set carry 
	SBB	CL,CL			;          Set CL to true if so...
	AND	BL,Num_Lock		;   if (num_lock active)
	ADD	BL,0FFH			;     then Set carry
	SBB	BL,BL			;          Set BL to true if so...
	XOR	CL,BL			;If(Num_Lock NOT cancelled by Shift)
	JZ	CP3			;   then

	TEST	AH,Shift_Override	;     if ( key modified by shift)
	JZ	CP3			;        then
	jmp	cp9

CP3:	TEST	AH,Ctrl_Type		;If (key is  Shift, Ctrl, or Alt)
	JZ	CP6			;  then 

	CMP	AL,Last_Key		;    if (Auto repeat of this control key)
	JE	CP8			;      then throw away this key
					;    else
	CMP	AL,Alt_Key OR 80H	;      if(ALT key being released)
	JNE	CP4			;        then

	CMP	Input_Value,0		;      if (numeric value entered)
	JE	CP4			;         then
					;	   (display the ascii character)
	MOV	AL,Input_Value		;           Get entered character code
	MOV	AH,0			;           Set null scan code for compatibility
	CALL	Put_Key			;           Place character in keyboard buffer
	MOV	Input_Value,0		;           Clear value for next try
	MOV	AL,Alt_Key OR 80H	;           Get code for released ALT key again

CP4:	TEST	AL,80H			;Set Flags for release status of shift key being released
	PUSHF				;   and save them
	AND	AL,7FH			;get bare scan code
	MOV	BX,offset Key_Table	;Point to unshifted key table
	XLAT	CS: Key_Table		;and get return code from table
	POPF				;if (key NOT being released)
	JNZ	CP5			;  then
	OR	Shift_Status,AL		;      set the correct shift flag
	JMP	short CP8		;      and Return indicating key processed
					;  else
CP5:	NOT	AL			;      convert AL into a bit mask
	AND	Shift_Status,AL		;      Clear the appropriate shift bit
	JMP	short CP8		;      and Return indicating key processed

CP6:	TEST	AH,Lock_Type		;if (Key NOT locking key)
	JZ	CP9			;  then exit

	TEST	Shift_Status,Alt_Flag	;if (ALT key Depressed)
	JZ	CP7			;  then

	CMP	AL,Ins_Key		;    SPECIAL CASE: if (Insert Key)
	JE	CP9			;      then let ALT_MODE handle it

	CMP	AL,Scroll_Key		;    SPECIAL CASE: If (Scroll Lock Key)
	JE	CP9			;      then treat ALT-INS as flush-key-buffer

CP7:	cmp	al,Num_Key		;    Special Case: Keeps keyboard in sync
	jnz	cp7a			;      with 8039 processor.
					;      8039 DOESN"T REPEAT NUM LOCK!!!

	mov	bx,offset Key_Table 	;Point to unshifted key table
	and	al,7FH			;Mask off the key-release bit
	xlat	cs:Key_Table		;Get code for this locking key
	jmp	CP10			; and only update Shift_Status

CP7a:	CMP	Last_Key,AL		;If (Lock Key Repeating)
	JE	CP8			;  then ignore repeating lock key

	MOV	BX,offset Key_Table 	;Point to unshifted key table
	AND	AL,7FH			;Mask off the key-release bit
	XLAT	CS: Key_Table		;Get code for this locking key
	XOR	Lock_Status,AL		;toggle the status of this locking key
	TEST	CH,80H			;If (locking key being released)
	JNZ	CP8			;  then ignore release entirely
					;  else
CP10:	XOR	Shift_Status,AL		;     Set/Reset the appropriate lock bit
	CMP	CH,Ins_Key		;     SPECIAL CASE:  If (NOT Insert key)
	JNE	CP8			;     then exit 
					;     else
	MOV	AX,Ins_Code		;        get code for the insert key
	CALL	Put_Key			;        Place keycode in the keyboard buffer

CP8:	STC				;Set control-key processed flag

CP9:	MOV	AL,CH			;Return the scan code into AL
	POPREG	<CX,BX>
	RET
CONTROL_PROCESS ENDP

page
;----------------------------------------------------------------------
Alt_Mode Proc Near; (30_May_85)
;------------------------------
;	1) ALT_MODE: (KEY, KEY_TYPE)
;	2) Alt_Mode is called when ALT mode is in effect and a key has been
;	   pressed.  It determines if the ALT mode affects the key:  if so the
;	   appropriately modified key code is processed.  Note that Alt_Mode
;	   also processes several 'special' keys (ie, keyboard reset, etc).
;	3) Entry Register Values:
;		AL: Key code (scan code)
;		AH: Key type, from KEY_TYPE array.
;		    If AH = 0FFH, this was a software interrupt key.
;	4) ALT_Mode will, in the case of a software reset, directly 
;	   branch to the software reset initialization code, rather
;	   than returning to the calling program!
;
Public	Alt_Mode

	PushReg	<bx,cx>
	test	Shift_Status,Ctrl_Flag	;If ( (Control Code eq True) And
	jz	AM5
	cmp	al,Del_Key		;     (Delete Key Is Active) )
	jne	AM11

	mov	ax,Rom_Data
	mov	ds,ax			;	DS:= Initial Data Segment
	mov	Reset_Flag,1234h	;	Set the keyboard-reset flag
	jmp	Start_Up_Warm		;	Goto software reset routine

AM5:	cmp	al,Scroll_Key		;Else If (Alt_Break)
	jne	AM7

	call	Flush_Key_Buff		;	Flush the keyboard buffer
	jmp	Short AM11

AM7:	test	ah,Digit_Type		;Else If (Character a pad number)
	jz	AM9

	mov	bx,offset SHIFT_TABLE	;	point to the shifted case tbl
	xlat	cs:Shift_Table		;	Get the ASCII digit
	sub	al,'0'			;	Make it from '0'..'9' into 0..9
	mov	cl,al			;	Save new digit
	mov	al,Input_Value		;	Get currently entered number
	mov	ch,10			;	Multiply current input by 10
	mul	ch
	add	al,cl			;	Add in just-entered digit
	mov	Input_Value,al		;	Now store new input value
	jmp	Short AM11		;	Done

AM9:	mov	bx,offset Alt_Table	;Else	point to the ALT mode xlat tbl
	call	Put_Table_Key		;	Retrieve key from table, put in buff

AM11:	PopReg	<cx,bx>
	ret				;Return

Alt_Mode	EndP

page
;**********************************************************************
; CTRL_MODE: (KEY, KEY_TYPE)
;
;	Ctrl_Mode is called to process keys pressed when the CTRL key
; is held down.  It determines if the key is affected by CTRL mode,
; and processes the key.
;
; Input:
;	AL: Key code (scan code from keyboard)
;	AH: Key type (from the KEY_TYPE array)
;
; Output:
;	SI: Set to true if an END-OF-INTERRUPT was sent
;**********************************************************************
CTRL_MODE PROC NEAR
PUBLIC	CTRL_MODE

	PUSHREG	<BX>
	CMP	AL,Num_Key		;Is this a 'hold'?
	JNE	CM2			;No, continue

	OR	Lock_Status,Pause_Flag	;Yes - show that pause is active
	IN	AL,Interrupt_Mask	;Read the interrupt mask register
	OR	AL,NOT Keyboard_Intr_Mask ;DISABLE keyboard interrupts
	OUT	Interrupt_Mask,AL
	MOV	AL,End_Of_Intr		;Enable other interrupts to occur
	OUT	Interrupt_Ctrl,AL
	STI				;Let other interrupts in

CM1:	MOV	Last_Key,AL		;Record last-keystroke-entered
	MOV	AL,True			;Set the wait-for-keyboard flag
	CALL	Get_Key			;Read a char from the keyboard
	CALL	Control_Process		;If 'control' key, process it
	JC	CM1			;Key was a control key - keep waiting

	TEST	AL,80H			;Is this 'normal' key being released?
	JNZ	CM1			;Yes - ignore key's release

	CMP	AL,Num_Key		;Was the key another 'hold' key?
	JE	CM1			;Yes - continue waiting

	MOV	SI,True			;No, set the EOI-already-sent flag
	AND	Lock_Status,NOT Pause_Flag ;Clear the pause-active flag
	IN	AL,Interrupt_Mask	;Read the interrupt mask register
	AND	AL,Keyboard_Intr_Mask	;Enable further keyboard interrupts
	OUT	Interrupt_Mask,AL
	JMP	short CM5		;Ignore last key's value and return

CM2:	CMP	AL,Scroll_Key		;Is this a 'break' key?
	JNE	CM3			;No, continue

	call	Flush_Key_Buff		;No, flush keyboard buffer
	MOV	AX,Break_Code		;Now, get a special 'break' keycode
	CALL	Put_Key			;Place special code in the key buffer
	MOV	Break_Flag,80H		;Turn on the 'break-in-progress' flag
	INT	Break_Intr		;Execute the user's break routine
	JMP	CM5			;Return after processing 'break' code

CM3:	CMP	AL,Pgup_Key		;Is this a CTRL-Page-Up?
	JNE	CM4			;No, process key as 'normal' CTRL-XXX

	MOV	AX,Pgup_Code		;SPECIAL CASE - get code: CTRL-PGUP
	CALL	Put_Key			;Place special code in input buffer
	JMP	short CM5		;Finished with this key

CM4:	CMP	AL,Print_Key		;SPECIAL CASE: Is this print-screen?
	JNE	CM41			;No, place key in buffer

	MOV	AL,End_Of_Intr		;No, get the end-of-interrupt cmd
	OUT	Interrupt_Ctrl,AL	;Issue EOI to interrupt controller
	MOV	SI,True			;Set the end-of-intr-issued flag
	INT	Print_Screen_Intr	;Execute print-screen routine
	JMP	short CM5		;Return w/o sending a key code

CM41:	MOV	BX,offset Ctrl_Table	;Point to the CTRL mode table
	CALL	Put_Table_Key		;If key valid, convert & put in buff

CM5:	POPREG	<BX>
	RET

CTRL_MODE ENDP

page
;**********************************************************************
; PUT_TABLE_KEY: (KEY, TABLE_PTR)
;
;	Put_Table_Key is called to retrieve a key code from one of
; the key value tables, and place it in the buffer.  It checks the
; particular key's KEY_TYPE to properly process the key.
;
; Input:
;	AL: Key code (scan code from keyboard processor)
;	BX: Pointer to the current key value table
;**********************************************************************
PUT_TABLE_KEY PROC NEAR
PUBLIC	PUT_TABLE_KEY

	PUSHREG	<AX,BX,CX>
	MOV	CL,AL			;Save the key to be processed
	MOV	AH,AL			;Place it in AH, too
	XLAT	BYTE PTR CS: [BX]	;Get the key's value
	cmp	al,80h			; if (Insert Key)
	jnz	PTK0			; then
	mov	al,0D2h			;   Jam appropriate code
PTK0:	CMP	AL,nul			;Is this an invalid key?
	JE	PTK2			;Yes - ignore it and return

	TEST	AL,80H			;Is this an extended character?
	JZ	PTK1			;No, put char in buffer as is

	MOV	AH,AL			;Yes - place value in AH
	MOV	AL,CL			;Get the key value again
	MOV	BX,offset KEY_TYPE	;Point to the KEY_TYPE array
	XLAT	CS: KEY_TYPE		;Get the key's type in AL
	TEST	AL,EXT_FLAG		;Is this keycode absolute?
	MOV	AL,0			;[Set primary value to 0 if so]
	JNZ	PTK1			;Yes - place the code in the buffer

	AND	AH,7FH			;No, mask out extended flag
PTK1:	CALL	PUT_KEY			;Place the codes in the keyboard buff

PTK2:	POPREG	<CX,BX,AX>
	RET
PUT_TABLE_KEY ENDP

page
;----------------------------------------------------------------------
Put_Key Proc Near
;----------------
;	1) Put_Key is called to record a key in the keyboard buffer.
;	2) Input Register Values:
;		AL -> ASCII value for character
;		AH -> Keyboard processor scan code for key
;
Public	Put_Key
	
	call	Far Ptr Put_Key_Buff	;Place keycode into buffer
	ret				;Return

Put_Key	EndP

page
;----------------------------------------------------------------------
Put_Key_Buff Proc Far; (2_Jul_85)
;--------------------------------
;	1) Put_Key_Buff actually places a key-code pair into the keyboard 
;	   buffer.  If the key will not fit, then Put_Key_Buff will 'beep' to 
;	   inform the user.
;	2) NOTE: This is a FAR routine.
;	3) Input Register Values:
;		AL -> ASCII value for character
;		AH -> Keyboard processor scan code for key
;
Public	Put_Key_Buff

	PushReg	<ax,si,di,ds,es>
	call	DS_to_Rom_Data		;DS:= Rom Data
	mov	si,Key_Buff.Tail	;Point to the first available space
	mov	di,si			;Save pointer in DI
	inc	si
	inc	si			;Advance key ptr to the next word
PKB1:	call	Adjust_Key_Ptr		;Check for buffer wrap-around
	cmp	si,Key_Buff.Head	;Is the buffer full?
	je	Pkb2			;Yes - buffer is full!

	mov	Key_Buff.Tail,si	;Store pointer for next pass
	call	ES_to_Key_Buff		;ES:= Rom Data
	mov	Byte Ptr es:[di],al	;Place ASCII character in buffer
	mov	Byte Ptr es:[di+1],ah	;No, write the scan code in the buffer
	jmp	Short Pkb3		;Return

Pkb2:	call	Beep			;Beep to indicate error
Pkb3:	PopReg	<es,ds,di,si,ax>	;Restore registers
	ret				;Return

Put_Key_Buff	EndP

page
;----------------------------------------------------------------------
Adjust_Key_Ptr Proc Near; (3_Jul_85)
;-----------------------------------
;	1) Adjust_Key_Ptr checks to see that a key pointer is valid - if it
;	   is not, it is set to the first position in the keyboard buffer.
;	2) Input Register Values:
;		SI -> Pointer into keyboard buffer.
;	3) Output Register Values:
;		SI -> Pointer into keyboard buffer, adjusted properly
;
Public	Adjust_Key_Ptr

	cmp	si,offset Key_Buff.Buff+32-2
	jbe	Akp1			;If (SI ge End of Buffer Pointer)

	mov	si,offset Key_Buff.Buff	;	SI:= Start of Buffer
Akp1:	ret				;Return


Adjust_Key_Ptr	EndP

page
;**********************************************************************
; GET_KEY:
;
;	Get_Key gets a single scan code from the keyboard.  Note that
; the routine may either wait for a key, or may be invoked with the
; assumption that a key has already been typed.
;
; Input:
;	AL: Wait flag.  If cleared, the key is assumed to be waiting
;	    at the hardware port.  If set, the routine will wait for
;	    a key to be typed by scanning the interrupt controller for
;	    pending keyboard interrupts - under these circumstances,
;	    the routine assumes that the keyboard interrupt has been
;	    disabled.
;
; Output:
;	AL: Scan code for character read.
;	AH: Scan code for character read.
;**********************************************************************
GET_Key PROC NEAR
	PUBLIC	GET_Key
	TEST	AL,AL			;If (character to be read from port)
	JZ	GK2			;   then go get the character

					; else (we have to wait for a character)

GK1:	MOV	AL,Poll_Pending_IntrS	;repeat
	OUT	Interrupt_Ctrl,AL	;     Request pending ints from intr ctrl port
	IN	AL,Interrupt_Ctrl	;     Read the pending interrupts
	TEST	AL,NOT Keyboard_Intr_Mask ;   test for keyboard int pending
	JZ	GK1			;until  ( Pending Kbrd Interrupt)

GK2:	IN	AL,Keyboard_Data_Port	;Read the character typed
	MOV	AH,AL			;Place it in AH
	IN	AL,Keyboard_Ctrl_Port	;Now read the control port
	OR	AL,Keyboard_Reset	;Set the reset-keyboard bit
	OUT	Keyboard_Ctrl_Port,AL	;Reset the keyboard!
	AND	AL,NOT Keyboard_Reset	;Now, clear the reset bit
	OUT	Keyboard_Ctrl_Port,AL	;Enable the keyboard
	MOV	AL,AH			;Copy the scan code into AL
	RET
GET_Key ENDP


;----------------------------------------------------------------------
Flush_Key_Buff Proc Near; (3_Jul_85)
;-----------------------------------
;	1) Flush_Key_Buff initializes the keyboard buffer variables, so
;	   that the keyboard buffer is empty.
;	2) All data present in the buffer is lost.
;
Public	Flush_Key_Buff

	push	ax
	pushf				;Save interrupt enable status
	cli				;Prevent interrupts

	mov	ax,offset Key_Buff.Buff	;AX:= Start of Buffer
	mov	Key_Buff.Head,ax	;Initialize the head-of-buffer ptr
	mov	Key_Buff.Tail,ax	;Init the buffer-tail pointer, too
	popf				;Restore intr. enable status
	Pop	ax
	ret

Flush_Key_Buff	EndP

page
;----------------------------------------------------------------------
Init_Key_Buff Proc Near; (8_Jul_85)
;----------------------------------
;	1) Init_Key_Buff is called to initialize the keyboard buffer
;	   location at start-up.  It also flushes the keyboard buffer, and
;	   clears the keyboard shift variables.
;
Public	Init_Key_Buff

;*********12-13-85****
	mov	Key_Buff_Segment,DS	; initialize Key buffer segment
	mov	ax,offset Key_Buff.Buff	; point to start of buffer
	mov	Key_Buff_Start,ax	; set start of buffer location
	add	ax,32-2			; calculate last word of buffer
	mov	Key_Buff_End,ax		; and set it
;*************	


	call	Flush_Key_Buff		;Flush out the keyboard buffer
	mov	Shift_Status,0		;Initially, no shifts set
	mov	cs:Old_Shift_Status,0	;  Same with copy in SRam
	mov	Lock_Status,0		;Clear out the lock-key-depressed values
	mov	Input_Value,0		;Clear entered key code to null
	mov	Last_Key,0		;Clear last-key-entered flag
	mov	Feed_Key,False		;Flag that normal key entry is in use

	mov	ax,0 Shl 8 + Reset_Keyboard_Register
	call	Out_Morrow_6845		;Reset the keyboard

	mov	cx,50h
MnLp1:	loop	MnLp1			;Delay for keyboard reset

	mov	al,Read_Keyboard	;Get code to read DIP switch L.S. nibble
	out	Sys_Ctrl_Port,al	;Leave it turned on for compatibility
	ret				;Return

Init_Key_Buff	EndP

;----------------------------------------------------------------------
ES_to_Key_Buff Proc Near; (2_Jul_85)
;-----------------------------------
;	1) This routine forces ES to Rom Data.
;	2) The ES used to be loaded from an SRam variable Key_Buff_Segment.
;
	push	dx
;*****
	mov	dx,Key_Buff_Segment	; get keyboard buffer segment
	cmp	dx,0			; if it's no good, then jam it
	jz	ES_1
	cmp	dx,0ffffh
	jnz	ES_2
;*****
ES_1:	mov	dx,Rom_Data
ES_2:	mov	es,dx			;ES:= Valid Key_Buf_Segment 
	pop	dx
	ret				;Return

ES_to_Key_Buff	EndP

Monitor_Segment	EndS
		End
