	PAGE	,132
	TITLE	Z-150 Keyboard Driver

;**********************************************************************
;
;                       -----------------------
;----------------------- Z-150 Keyboard Driver ------------------------
;                       -----------------------
;
;	       Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES:NOTHING

	INCLUDE	../ROM/ROM.LIT
	INCLUDE	../ROM/IO.LIT
	INCLUDE	../ROM/INTR.LIT
	INCLUDE ../SYS/SYS.EXT
	INCLUDE	KEYBOARD.LIT
	INCLUDE	KEYBOARD.EXT

	EXTRN	CASE:NEAR, BEEP:NEAR, DELAY:NEAR, START_UP:FAR
	EXTRN	DATA_SEGMENT:WORD
	EXTRN	MFM_150:NEAR, SOFTWARE_BREAKPOINT:FAR
	EXTRN	KEY_TABLE:BYTE, SHIFT_TABLE:BYTE, CTRL_TABLE:BYTE
	EXTRN	ALT_TABLE:BYTE, KEY_TYPE:BYTE, NO_BOOT_FLAG:BYTE
	EXTRN	ENABLE_INTERLACE:NEAR, DISABLE_INTERLACE:NEAR, ENABLE_FATC:NEAR
	EXTRN	SET_MONITOR_STACK:NEAR, SET_USER_STACK:NEAR


;**********************************************************************
; KEYBOARD_IO_INTERRUPT: (FUNCTION, PARAMETERS)	     Interrupt 16H (22)
;
;	Keyboard_IO_Interrupt is the ROM entry point to communicate with
; the keyboard processor.  The following routines are executed by first 
; loading parameters into registers as specified below, and then 
; placing an operation code number in AH.  Finally, an INT 16H instruc-
; tion is executed to perform the function.  The following function 
; codes are recognized by the Keyboard_IO_Interrupt routine:
;
; Function Code 0 - Read Character from Keyboard
;   Output:
;	AL: Character value in ASCII
;	AH: Keyboard processor scan code
;
; Function Code 1 - Check Keyboard Status
;   Output:
;	ZF: The Zero flag is set upon return if no characters are
;	    waiting in the buffer to be processed.  If the Zero
;	    flag is not set, then at least one character is in the
;	    buffer.
;	AX: If characters are waiting in the keyboard buffer, then
;	    AX will contain the first pair of character codes (as
;	    above).
;
; Function Code 2 - Determine Current Shift Status
;   Output:
;	AL: Bit-mapped, as follows:
;		Bit 7 - Insert mode active
;		Bit 6 - Caps_Lock active
;		Bit 5 - Num_Lock active
;		Bit 4 - Scroll_Lock active
;		Bit 3 - Alt key currently pressed
;		Bit 2 - Ctrl key currently pressed
;		Bit 1 - Left Shift key currently pressed
;		Bit 0 - Right shift key currently pressed
;
;**********************************************************************
KEYBOARD_IO_INTERRUPT PROC FAR
	PUBLIC	KEYBOARD_IO_INTERRUPT
	PUSHREG	<BX,BP,DS>		;Save registers
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	STI				;Let keys into buffer!
	CMP	AH,KMAX			;Is this a valid functon code?
	JAE	KBI1			;No, ignore it
	CALL	CASE			;Yes - execute the proper routine
	JMP	NEAR PTR KBI1		;All routines return here...
KCMD:	DW	GET_KEY_BUFF		;If AH=0, Read Key from Buffer
	DW	GET_KEY_STATUS		;If AH=1, Return Keyboard Status
	DW	GET_SHIFT_STATUS	;If AH=2, Return Shift Status
KMAX	EQU	($-KCMD) / 2
KBI1:	POPREG	<DS,BP,BX>		;Restore registers
	JC	KBI2			;Routine wants to save flags
	RET	2			;Return and dump old flags
KBI2:	IRET				;Return with flags saved
KEYBOARD_IO_INTERRUPT ENDP



;**********************************************************************
; GET_KEY_BUFF:
;
;	Get_Key_Buff retrieves the next key from the keyboard buffer.
; If at least one key isn't already in the buffer, Get_Key_Buff will
; wait for one to be entered.
;
; Output:
;	AL: ASCII code for the entered key.
;	AH: Keyboard processor scan code for this key.
;**********************************************************************
GET_KEY_BUFF PROC NEAR
	PUBLIC	GET_KEY_BUFF
	PUSHREG	<SI,ES>
	PUSHF				;Save current interrupt-enable status
GKB1:	CLI				;Prevent new keys for a bit
	MOV	ES,KEY_BUFF_SEGMENT	;Point to the keyboard buffer segment
	MOV	SI,KEY_BUFF.HEAD	;Retrieve the buffer pointer
	CMP	SI,KEY_BUFF.TAIL	;Are there any chars in the buffer?
	PUSHF				;Save key present status on stack
	MOV	AX,WORD PTR ES: [SI]	;Get char from buffer, in case
	INC	SI			;Point to the next byte
	INC	SI			;No, advance the key ptr for words
	POPF				;Was a key typed?
	JNZ	GKB2			;Yes - return key code
	STI				;No, let waiting interrupts occur
	JMP	SHORT GKB1		;Loop until key entered
GKB2:	CALL	ADJUST_KEY_PTR		;Wrap pointer around if necessary
	MOV	KEY_BUFF.HEAD,SI	;Update pointer for next pass
	POPF				;Restore old interrupt enable status
	CLC				;Set new-flags status for return
	POPREG	<ES,SI>
	RET
GET_KEY_BUFF ENDP



;**********************************************************************
; GET_KEY_STATUS:
;
;	Get_Key_Status checks to see if any keys are waiting in
; the keyboard buffer.
;
; Output:
;	ZF: If the Zero Flag is set upon return, then no keys exist
;	    in the keyboard buffer.  If the Zero Flag is reset, however,
;	    then at least one key is in the keyboard buffer.
;	AX: If any keys are in the keyboard buffer, then AX will contain
;	    the first key in the buffer (AL = ASCII, AH = Scan code).
;**********************************************************************
GET_KEY_STATUS PROC NEAR
	PUBLIC	GET_KEY_STATUS
	PUSHREG	<BX,SI,ES>
	PUSHF				;Save current interrupt enable status
	CLI				;Prevent new keys for a bit
	MOV	ES,KEY_BUFF_SEGMENT	;Point to the keyboard buffer's seg
	MOV	SI,KEY_BUFF.HEAD	;Retrieve the buffer pointer
	CMP	SI,KEY_BUFF.TAIL	;Are there any chars in the buffer?
	MOV	AX,WORD PTR ES: [SI]	;[Get char from buffer, in case]
	MOV	BX,AX			;Save key value in BX
	LAHF				;Get status of zero flag in AH
	POPF				;Restore old intr. enable status
	SAHF				;Restore zero flag status from AH
	MOV	AX,BX			;Restore key code value
	CLC				;Set new-flags return parameter
	POPREG	<ES,SI,BX>
	RET
GET_KEY_STATUS ENDP



;**********************************************************************
; GET_SHIFT_STATUS:
;
;	Get_Shift_Status retrieves the current status of the keyboard
; shift/lock/mode flags.
;
; Output:
;	AL: Bit-mapped, as follows:
;		Bit 7 - Insert mode active
;		Bit 6 - Caps_Lock active
;		Bit 5 - Num_Lock active
;		Bit 4 - Scroll_Lock active
;		Bit 3 - Alt key currently pressed
;		Bit 2 - Ctrl key currently pressed
;		Bit 1 - Left Shift key currently pressed
;		Bit 0 - Right shift key currently pressed
;**********************************************************************
GET_SHIFT_STATUS PROC NEAR
	PUBLIC	GET_SHIFT_STATUS
	MOV	AL,SHIFT_STATUS		;Retrieve shift mode status
	XOR	AH,AH			;Set AH to 0, set ZF for compatibility
	STC				;Set save-flags return parameter
	RET
GET_SHIFT_STATUS ENDP



;**********************************************************************
; KEYBOARD_INTERRUPT:
;
;	Keyboard_Interrupt is the actual interrupt routine first executed
; when a key has been pressed.  It reads in the key value, attempts
; to process it, and then places it in the keyboard buffer if it is
; a valid code-generating key.
;
; NOTE: If the key being processed is a software interrupt, this
;	routine will exit to the MFM-150 breakpoint routine, rather
;	than returning to the user's program.
;**********************************************************************
KEYBOARD_INTERRUPT PROC FAR
	PUBLIC	KEYBOARD_INTERRUPT
	PUSHREG	<AX,BX,CX,DX,SI,DI,DS,ES> ;Save processor state
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	MOV	SI,FALSE		;Clear the end-of-interrupt-sent flag
	CMP	FEED_KEY,FALSE		;Are we feeding keys in?
	JNE	KI0			;Yes - don't read a new key!
	IN	AL,KEYBOARD_DATA_PORT	;Read the character typed
	MOV	AH,AL			;Place it in AH
	IN	AL,KEYBOARD_CTRL_PORT	;Now read the control port
	OR	AL,KEYBOARD_RESET	;Set the reset-keyboard bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Reset the keyboard!
	AND	AL,NOT KEYBOARD_RESET	;Now, clear the reset bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Enable the keyboard
KI0:	MOV	FEED_KEY,FALSE		;Set the key-feed flag to normal-key
	MOV	AL,AH			;Copy the scan code into AL
	CALL	SET_MONITOR_STACK	;Use the local stack while in routines
KI1:	CALL	CONTROL_PROCESS		;If 'control' key, process it
	MOV	CH,AL			;[Store key code into CH]
	JC	KI11			;Key was a control key - done
	TEST	AL,80H			;Is this 'normal' key being released?
	JNZ	KI11			;Yes - ignore release key code
	TEST	SHIFT_STATUS,ALT_FLAG	;In ALT mode?
	JZ	KI3			;No, continue
	CALL	ALT_MODE		;Yes - process ALT mode keys
	JMP	SHORT KI11		;Return
KI3:	TEST	SHIFT_STATUS,CTRL_FLAG	;In CTRL mode?
	JZ	KI4			;No, try again
	CALL	CTRL_MODE		;Yes - process using control mode
	JMP	SHORT KI11		;Return
KI4:	MOV	CL,FALSE		;Set Shift-Flag to false
	TEST	AH,PAD_TYPE		;Is the entered key from numeric pad?
	JZ	KI5			;No, continue
	TEST	SHIFT_STATUS,NUM_LOCK	;Is Num_Lock currently active?
	JE	KI7			;No, use shift keys for shift status
	JMP	SHORT KI6		;Yes - set special shift state
KI5:	TEST	AH,ALPHA_TYPE		;Is the entered key a letter?
	JZ	KI7			;No, continue
	TEST	SHIFT_STATUS,CAPS_LOCK	;Is Caps_Lock active?
	JZ	KI7			;No, check shift
KI6:	MOV	CL,TRUE			;Set shift-toggle mode
KI7:	TEST	SHIFT_STATUS,RIGHT_FLAG OR LEFT_FLAG	;Is shift pressed?
	JZ	KI8			;No, use same shift status
	XOR	CL,TRUE			;Yes - flip shift state to opposite
KI8:	MOV	BX,OFFSET SHIFT_TABLE	;Point BX to the shifted-case table
	TEST	CL,CL			;Is this key to be shifted?
	JNZ	KI9			;Yes - pointing to right table
	MOV	BX,OFFSET KEY_TABLE	;No, point to Base-Case table
	JMP	SHORT KI10		;Doing lower case, so print screen not possible
KI9:	CMP	AL,PRINT_KEY		;SPECIAL CASE: Is this print-screen?
	JNE	KI10			;No, place key in buffer
	MOV	AL,END_OF_INTR		;No, get the end-of-interrupt cmd
	OUT	INTERRUPT_CTRL,AL	;Issue EOI to interrupt controller
	MOV	SI,TRUE			;Set the end-of-intr-issued flag
	INT	PRINT_SCREEN_INTR	;Execute print-screen routine
	JMP	SHORT KI11		;Return w/o sending a key code
KI10:	CALL	PUT_TABLE_KEY		;Convert key and put in buffer
KI11:	MOV	LAST_KEY,CH		;Update the 'last-key-typed' storage
	CMP	SI,TRUE			;End-of-interrupt already sent?
	JE	KI12			;Yes - don't send another one!
	MOV	AL,END_OF_INTR		;No, get an end-of-interrupt command
	OUT	INTERRUPT_CTRL,AL	;Output it to the interrupt ctrl port
KI12:	CALL	SET_USER_STACK		;Restore the user's stack
	CMP	AH,0FFH			;Was this a software interrupt?
	POPREG	<ES,DS,DI,SI,DX,CX,BX,AX> ;[Restore user's registers]
	JNE	KI13			;Yes - exit normally
	JMP	SOFTWARE_BREAKPOINT	;Perform a software interrupt
KI13:	IRET
KEYBOARD_INTERRUPT ENDP



;**********************************************************************
; CONTROL_PROCESS: (KEY_CODE)
;
;	Control_Process is called to determine if a key is a control
; key, and to process it if it is.  'Control' keys include the
; SHIFT/CTRL/ALT, INS and LOCK keys.  It also handles keyboard buffer
; overflow.
;
; Input:
;	AL: Key code (scan code from keyboard)
;
; Output:
;	AL: Key scan code (unmodified)
;	CY: Control key flag.  Set if the key passed to CONTROL_KEY
;	    was treated as one of the control keys.  If the carry
;	    flag is reset, then the key was not a control key, and was
;	    ignored.
;	AH: If the control flag was not set (indicating that further
; 	    processing is required for this key), then AH will
;	    contain the key type (from the KEY_TYPE array, below).
;**********************************************************************
CONTROL_PROCESS PROC NEAR
	PUBLIC	CONTROL_PROCESS
	PUSHREG	<BX,CX>
	MOV	CH,AL			;Save the key in CH
	AND	AL,7FH			;Mask off 'release' bit
	MOV	BX,OFFSET KEY_TYPE	;Point BX to the key type table
	XLAT	CS: KEY_TYPE		;Place KEY_TYPE in AL
	MOV	AH,AL			;Place the key type in AH
	MOV	AL,CH			;Place scan code in AL again
	CMP	AL,BUFFER_FULL		;Was the keyboard processor overrun?
	JNE	CP1			;No, continue
	CMP	LAST_KEY,BUFFER_FULL	;Was the last keycode an overrun?
	JE	CP0			;Yes - can skip beeping again
	CALL	BEEP			;No, beep to indicate first overrun
CP0:	JMP	CP8			;Return
CP1:	TEST	SHIFT_STATUS,CTRL_FLAG	;Is the CTRL key being held down?
	JZ	CP2			;No, continue
	TEST	AH,CTRL_OVERRIDE	;Is this a special key in CTRL mode?
	JZ	CP2			;No, process it as a control key
	JMP	CP9			;Yes - let CTRL_MODE handle the key
CP2:	MOV	CL,SHIFT_STATUS		;Pick up the current shift status
	MOV	BL,CL			;Copy it into BL
	AND	CL,RIGHT_FLAG OR LEFT_FLAG ;Is shift mode active?
	ADD	CL,0FFH			;Set carry if a shift key is pressed
	SBB	CL,CL			;Set CL to true if so...
	AND	BL,NUM_LOCK		;Is the numeric lock active?
	ADD	BL,0FFH			;Set carry flag if so
	SBB	BL,BL			;Set BL to true if so...
	XOR	CL,BL			;Is the Shift actually set?
	JZ	CP3			;No, continue
	TEST	AH,SHIFT_OVERRIDE	;Yes - is this still a control char?
	JNZ	CP9			;No, use normal key encoding tables
CP3:	TEST	AH,CTRL_TYPE		;Is this Shift, Ctrl, or Alt?
	JZ	CP6			;No, check for locking keys
	CMP	AL,LAST_KEY		;Is this control key repeating?
	JE	CP8			;Yes - throw away this keycode
	CMP	AL,ALT_KEY OR 80H	;Is the ALT key being released?
	JNE	CP4			;No, process key as usual
	CMP	INPUT_VALUE,0		;Was a numeric value entered?
	JE	CP4			;No, don't try to send a key code
	MOV	AL,INPUT_VALUE		;Get entered character code
	MOV	AH,0			;Set null scan code for compatibility
	CALL	PUT_KEY			;Place character in keyboard buffer
	MOV	INPUT_VALUE,0		;Clear value for next try
	MOV	AL,ALT_KEY OR 80H	;Get code for released ALT key again
CP4:	TEST	AL,80H			;Is this shift key being released?
	PUSHF				;Save shift/release status on stack
	AND	AL,7FH			;Remove any release bit
	MOV	BX,OFFSET KEY_TABLE	;Point to the base case key table
	XLAT	CS: KEY_TABLE		;Get the code matching this shift key
	POPF				;Restore status - is key being released?
	JNZ	CP5			;Yes - clear appropriate shift flag
	OR	SHIFT_STATUS,AL		;No, set the correct shift flag
	JMP	SHORT CP8		;Return indicating key processed
CP5:	NOT	AL			;Make AL into a bit mask
	AND	SHIFT_STATUS,AL		;Clear the appropriate shift bit
	JMP	SHORT CP8		;Return indicating key processed
CP6:	TEST	AH,LOCK_TYPE		;Is this a locking key?
	JZ	CP9			;No, not a control key
	TEST	SHIFT_STATUS,ALT_FLAG	;Is the ALT key depressed?
	JZ	CP7			;No, treat control key normally
	CMP	AL,INS_KEY		;SPECIAL CASE: Is this the Insert key?
	JE	CP9			;Yes - treat let ALT_MODE handle it
	CMP	AL,SCROLL_KEY		;SPECIAL CASE: Is this the scroll key?
	JE	CP9			;Yes - treat ALT-INS as flush-key-buffer
CP7:	CMP	LAST_KEY,AL		;Is this lock key repeating?
	JE	CP8			;Yes - ignore repeating key
	MOV	BX,OFFSET KEY_TABLE	;Point to the base case table
	AND	AL,7FH			;Mask off the key-release bit
	XLAT	CS: KEY_TABLE		;Get code for this locking key
	XOR	LOCK_STATUS,AL		;Change the status of this locking key
	TEST	CH,80H			;Is this locking key being released?
	JNZ	CP8			;Yes - ignore release entirely
	XOR	SHIFT_STATUS,AL		;No, Set/Reset the appropriate lock bit
	CMP	CH,INS_KEY		;SPECIAL CASE:  Is this the Insert key?
	JNE	CP8			;No, finished with this locking key
	MOV	AX,INS_CODE		;Yes - get code for the insert key
	CALL	PUT_KEY			;Place keycode in the keyboard buffer
CP8:	STC				;Set control-key processed flag
CP9:	MOV	AL,CH			;Return the scan code into AL
	POPREG	<CX,BX>
	RET
CONTROL_PROCESS ENDP



;**********************************************************************
; ALT_MODE: (KEY, KEY_TYPE)
;
;	Alt_Mode is called when ALT mode is in effect and a key has
; been pressed.  It determines if the ALT mode affects the key:  if so
; the appropriately modified key code is processed.  Note that Alt_Mode
; also processes several 'special' keys (ie, keyboard reset, etc).
;
; Input:
;	AL: Key code (scan code)
;	AH: Key type, from KEY_TYPE array.
;	    If AH = 0FFH, this was a software interrupt key.
;
; NOTE: ALT_Mode will, in the case of a software reset, directly 
;	branch to the software reset initialization code, rather
;	than returning to the calling program!
;**********************************************************************
ALT_MODE PROC NEAR
	PUBLIC	ALT_MODE
	PUSHREG	<BX,CX>
;
; Process CTRL-ALT-DEL, CTRL-ALT-INS, and CTRL-ALT-RETURN
;
	TEST	SHIFT_STATUS,CTRL_FLAG	;In ctrl mode, as well?
	JZ	AM5			;No, continue
	CMP	AL,DEL_KEY		;Is this a normal keyboard reset?
	JNE	AM1			;No, check for exit-to-monitor
	MOV	AX,ROM_DATA		;Point to the initial data segment
	MOV	DS,AX
	MOV	RESET_FLAG,1234H	;Set the keyboard-reset flag
	JMP	START_UP		;EXIT to software reset routine
AM1:	CMP	AL,INS_KEY		;Is this a reset-to-monitor key sequence?
	JNE	AM2			;No, check for software interrupt key
	MOV	AX,ROM_DATA		;Point to the initial data segment
	MOV	DS,AX
	MOV	RESET_FLAG,1234H	;Set the soft-reset flag
	MOV	NO_BOOT_FLAG,TRUE	;Set the do-not-boot flag
	JMP	START_UP		;Re-initialize with no auto-boot
AM2:	CMP	AL,NORMVID_KEY		;Is this a 'N' for normal video?
	JNE	AM3			;No, check for interlace key
	CALL	DISABLE_INTERLACE	;Yes - turn off interlace
	JMP	AM11			;Throw out the key
AM3:	CMP	AL,INTERVID_KEY		;Is this a set-interlace-mode key?
	JNE	AM35			;No, check for 'fat-char' video
	CALL	ENABLE_INTERLACE	;Yes - turn interlace video on
	JMP	SHORT AM11		;Throw out the key
AM35:	CMP	AL,FATCVID_KEY		;Is this a 'fat-char video' key?
	JNE	AM4			;No, check for software breakpoint
	CALL	ENABLE_FATC		;Yes - turn on 'fat characters'
	JMP	SHORT AM11		;Exit and ignore key
AM4:	CMP	AL,RETURN_KEY		;Is this a software interrupt?
	JNE	AM5			;No, check for ALT-only keys
	MOV	AH,0FFH			;Yes - set code for software interrupt
	JMP	SHORT AM11		;Return to Keyboard_Interrupt
;
; Process ALT-BREAK (Flush keyboard buffer)
;
AM5:	CMP	AL,SCROLL_KEY		;Is this a flush-key-buff?
	JNE	AM7			;No, continue
	CALL	FLUSH_KEY_BUFF		;No, flush the keyboard buffer
	JMP	SHORT AM11		;Return
;
; Process ALT-nnn sequences
;
AM7:	TEST	AH,DIGIT_TYPE		;Was the character a pad number?
	JZ	AM9			;No, continue
	MOV	BX,OFFSET SHIFT_TABLE	;Yes - point to the shifted case tbl
	XLAT	CS: SHIFT_TABLE		;Get the ASCII digit
	SUB	AL,'0'			;Make it from '0'..'9' into 0..9
	MOV	CL,AL			;Save new digit
	MOV	AL,INPUT_VALUE		;Get the currently entered number
	MOV	CH,10			;Multiply current input by 10
	MUL	CH
	ADD	AL,CL			;Add in just-entered digit
	MOV	INPUT_VALUE,AL		;Now store new input value
	JMP	SHORT AM11		;Done
AM9:	MOV	BX,OFFSET ALT_TABLE	;Yes - point to the ALT mode xlat tbl
	CALL	PUT_TABLE_KEY		;Retrieve key from table, put in buff
AM11:	POPREG	<CX,BX>
	RET
ALT_MODE ENDP



;**********************************************************************
; CTRL_MODE: (KEY, KEY_TYPE)
;
;	Ctrl_Mode is called to process keys pressed when the CTRL key
; is held down.  It determines if the key is affected by CTRL mode,
; and processes the key.
;
; Input:
;	AL: Key code (scan code from keyboard)
;	AH: Key type (from the KEY_TYPE array)
;
; Output:
;	SI: Set to true if an END-OF-INTERRUPT was sent
;**********************************************************************
CTRL_MODE PROC NEAR
	PUBLIC	CTRL_MODE
	PUSHREG	<BX>
	CMP	AL,NUM_KEY		;Is this a 'hold'?
	JNE	CM2			;No, continue
	OR	LOCK_STATUS,PAUSE_FLAG	;Yes - show that pause is active
	IN	AL,INTERRUPT_MASK	;Read the interrupt mask register
	OR	AL,NOT KEYBOARD_INTR_MASK ;DISABLE keyboard interrupts
	OUT	INTERRUPT_MASK,AL
	MOV	AL,END_OF_INTR		;Enable other interrupts to occur
	OUT	INTERRUPT_CTRL,AL
	STI				;Let other interrupts in
CM1:	MOV	LAST_KEY,AL		;Record last-keystroke-entered
	MOV	AL,TRUE			;Set the wait-for-keyboard flag
	CALL	GET_KEY			;Read a char from the keyboard
	CALL	CONTROL_PROCESS		;If 'control' key, process it
	JC	CM1			;Key was a control key - keep waiting
	TEST	AL,80H			;Is this 'normal' key being released?
	JNZ	CM1			;Yes - ignore key's release
	CMP	AL,NUM_KEY		;Was the key another 'hold' key?
	JE	CM1			;Yes - continue waiting
	MOV	SI,TRUE			;No, set the EOI-already-sent flag
	AND	LOCK_STATUS,NOT PAUSE_FLAG ;Clear the pause-active flag
	IN	AL,INTERRUPT_MASK	;Read the interrupt mask register
	AND	AL,KEYBOARD_INTR_MASK	;Enable further keyboard interrupts
	OUT	INTERRUPT_MASK,AL
	JMP	SHORT CM5		;Ignore last key's value and return
CM2:	CMP	AL,SCROLL_KEY		;Is this a 'break' key?
	JNE	CM3			;No, continue
	CALL	FLUSH_KEY_BUFF		;No, flush keyboard buffer
	MOV	AX,BREAK_CODE		;Now, get a special 'break' keycode
	CALL	PUT_KEY			;Place special code in the key buffer
	MOV	BREAK_FLAG,80H		;Turn on the 'break-in-progress' flag
	INT	BREAK_INTR		;Execute the user's break routine
	JMP	CM5			;Return after processing 'break' code
CM3:	CMP	AL,PGUP_KEY		;Is this a CTRL-Page-Up?
	JNE	CM4			;No, process key as 'normal' CTRL-XXX
	MOV	AX,PGUP_CODE		;SPECIAL CASE - get code: CTRL-PGUP
	CALL	PUT_KEY			;Place special code in input buffer
	JMP	SHORT CM5		;Finished with this key
CM4:	MOV	BX,OFFSET CTRL_TABLE	;Point to the CTRL mode table
	CALL	PUT_TABLE_KEY		;If key valid, convert & put in buff
CM5:	POPREG	<BX>
	RET
CTRL_MODE ENDP



;**********************************************************************
; PUT_TABLE_KEY: (KEY, TABLE_PTR)
;
;	Put_Table_Key is called to retrieve a key code from one of
; the key value tables, and place it in the buffer.  It checks the
; particular key's KEY_TYPE to properly process the key.
;
; Input:
;	AL: Key code (scan code from keyboard processor)
;	BX: Pointer to the current key value table
;**********************************************************************
PUT_TABLE_KEY PROC NEAR
	PUBLIC	PUT_TABLE_KEY
	PUSHREG	<AX,BX,CX>
	MOV	CL,AL			;Save the key to be processed
	MOV	AH,AL			;Place it in AH, too
	XLAT	BYTE PTR CS: [BX]	;Get the key's value
	CMP	AL,nul			;Is this an invalid key?
	JE	PTK2			;Yes - ignore it and return
	TEST	AL,80H			;Is this an extended character?
	JZ	PTK1			;No, put char in buffer as is
	MOV	AH,AL			;Yes - place value in AH
	MOV	AL,CL			;Get the key value again
	MOV	BX,OFFSET KEY_TYPE	;Point to the KEY_TYPE array
	XLAT	CS: KEY_TYPE		;Get the key's type in AL
	TEST	AL,EXT_FLAG		;Is this keycode absolute?
	MOV	AL,0			;[Set primary value to 0 if so]
	JNZ	PTK1			;Yes - place the code in the buffer
	AND	AH,7FH			;No, mask out extended flag
PTK1:	CALL	PUT_KEY			;Place the codes in the keyboard buff
PTK2:	POPREG	<CX,BX,AX>
	RET
PUT_TABLE_KEY ENDP



;**********************************************************************
; PUT_KEY: (KEY, SCAN_CODE)
;
;	Put_Key is called to record a key in the keyboard buffer.
;
; Input:
;	AL: ASCII value for character
;	AH: Keyboard processor scan code for key
;**********************************************************************
PUT_KEY	 PROC NEAR
	PUBLIC	PUT_KEY	
	CALL	FAR PTR PUT_KEY_BUFF	;Place Z-150 keycode into buffer
	RET
PUT_KEY	 ENDP



;**********************************************************************
; PUT_KEY_BUFF: (KEY, SCAN_CODE)
;
; 	Put_Key_Buff actually places a key-code pair into the keyboard 
; buffer.  If the key will not fit, then Put_Key_Buff will 'beep' to 
; inform the user.
;
; Input:
;	AL: ASCII value for character
;	AH: Keyboard processor scan code for key
;
; NOTE: This is a FAR routine.
;
;**********************************************************************
PUT_KEY_BUFF PROC FAR
	PUBLIC	PUT_KEY_BUFF
	PUSHREG	<AX,SI,DI,DS,ES>
	MOV	DS,DATA_SEGMENT		;Point to the monitor data segment
	MOV	SI,KEY_BUFF.TAIL	;Point to the first available space
	MOV	DI,SI			;Save pointer in DI
	INC	SI			;Point SI to the next byte code
	INC	SI			;Advance key ptr to the next word
PKB1:	CALL	ADJUST_KEY_PTR		;Check for buffer wrap-around
	CMP	SI,KEY_BUFF.HEAD	;Is the buffer full?
	JE	PKB2			;Yes - buffer is full!
	MOV	KEY_BUFF.TAIL,SI	;Store pointer for next pass
	MOV	ES,KEY_BUFF_SEGMENT	;Pick up the keyboard buffer segment
	MOV	BYTE PTR ES: [DI],AL	;Place ASCII character in buffer
	MOV	BYTE PTR ES: [DI+1],AH	;No, write the scan code in the buffer
	JMP	SHORT PKB3		;Return
PKB2:	CALL	BEEP			;Beep to indicate error
PKB3:	POPREG	<ES,DS,DI,SI,AX>	;Restore registers
	RET
PUT_KEY_BUFF ENDP



;**********************************************************************
; ADJUST_KEY_PTR: (BUFF_PTR)
;
;	Adjust_Key_Ptr checks to see that a key pointer is valid - if it
; is not, it is set to the first position in the keyboard buffer.
;
; Input:
;	SI: Pointer into keyboard buffer.
;
; Output:
;	SI: Pointer into keyboard buffer, adjusted properly
;**********************************************************************
ADJUST_KEY_PTR PROC NEAR
	PUBLIC	ADJUST_KEY_PTR
	CMP	SI,KEY_BUFF_END		;Gone beyond end of buffer?
	JBE	AKP1			;No, return
	MOV	SI,KEY_BUFF_START	;Yes - point back to start of buffer
AKP1:	RET
ADJUST_KEY_PTR ENDP



;**********************************************************************
; GET_KEY:
;
;	Get_Key gets a single scan code from the keyboard.  Note that
; the routine may either wait for a key, or may be invoked with the
; assumption that a key has already been typed.
;
; Input:
;	AL: Wait flag.  If cleared, the key is assumed to be waiting
;	    at the hardware port.  If set, the routine will wait for
;	    a key to be typed by scanning the interrupt controller for
;	    pending keyboard interrupts - under these circumstances,
;	    the routine assumes that the keyboard interrupt has been
;	    disabled.
;
; Output:
;	AL: Scan code for character read.
;	AH: Scan code for character read.
;**********************************************************************
GET_KEY PROC NEAR
	PUBLIC	GET_KEY
	TEST	AL,AL			;Are we to wait for a character?
	JZ	GK2			;No, pick up the char now
GK1:	MOV	AL,POLL_PENDING_INTRS	;Get command to poll intr requests
	OUT	INTERRUPT_CTRL,AL	;Request pending from intr ctrl port
	IN	AL,INTERRUPT_CTRL	;Read the pending interrupts
	TEST	AL,NOT KEYBOARD_INTR_MASK ;Is a keyboard char waiting?
	JZ	GK1			;No, wait some more
GK2:	IN	AL,KEYBOARD_DATA_PORT	;Read the character typed
	MOV	AH,AL			;Place it in AH
	IN	AL,KEYBOARD_CTRL_PORT	;Now read the control port
	OR	AL,KEYBOARD_RESET	;Set the reset-keyboard bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Reset the keyboard!
	AND	AL,NOT KEYBOARD_RESET	;Now, clear the reset bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Enable the keyboard
	MOV	AL,AH			;Copy the scan code into AL
	RET
GET_KEY ENDP



;**********************************************************************
; DOWNLOAD_BYTE:
;
;	Download_Byte is called to wait for a single character from the
; keyboard port, and to read the character.  Called during power-up
; and diagnostics modes, a timeout is included so that faulty or dis-
; connected keyboard processors will be detected.
;
; Output:
;	CY: Error flag - cleared if character read within a reasonable
;	    amount of time.  The carry flag is set when no character
;	    is received.
;	AL: Scan code for character read.
;	AH: Scan code for character read.
;**********************************************************************
DOWNLOAD_BYTE PROC NEAR
	PUBLIC	DOWNLOAD_BYTE
	PUSHREG	<CX>
	MOV	CX,1000			;Set maximum delay to one second
DB1:	MOV	AL,POLL_PENDING_INTRS	;Get command to poll intr requests
	OUT	INTERRUPT_CTRL,AL	;Request pending from intr ctrl port
	IN	AL,INTERRUPT_CTRL	;Read the pending interrupts
	TEST	AL,NOT KEYBOARD_INTR_MASK ;Is a keyboard char waiting?
	JNZ	DB2			;Yes - get the key
	PUSH	CX			;No, save the delay count
	MOV	CX,1			;Set delay to 1 millisecond
	CALL	DELAY			;Wait for a millisecond
	POP	CX			;Restore time-to-wait count
	LOOP	DB1			;Wait until key received or timeout
	STC				;ERROR - timeout occurred
	JMP	SHORT DB3		;Return error status
DB2:	IN	AL,KEYBOARD_DATA_PORT	;Read the character typed
	MOV	AH,AL			;Place it in AH
	IN	AL,KEYBOARD_CTRL_PORT	;Now read the control port
	OR	AL,KEYBOARD_RESET	;Set the reset-keyboard bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Reset the keyboard!
	AND	AL,NOT KEYBOARD_RESET	;Now, clear the reset bit
	OUT	KEYBOARD_CTRL_PORT,AL	;Enable the keyboard
	MOV	AL,AH			;Copy the scan code into AL
	CLC				;Clear the error flag
DB3:	POPREG	<CX>
	RET
DOWNLOAD_BYTE ENDP



;**********************************************************************
; FLUSH_KEY_BUFF:
;
;	Flush_Key_Buff initializes the keyboard buffer variables, so
; that the keyboard buffer is empty.  All data present in the buffer 
; is lost.
;**********************************************************************
FLUSH_KEY_BUFF PROC NEAR
	PUBLIC	FLUSH_KEY_BUFF
	PUSHREG	<AX>
	PUSHF				;Save interrupt enable status
	CLI				;Prevent interrupts
	MOV	AX,KEY_BUFF_START	;Point to the start of the keyboard buff
	MOV	KEY_BUFF.HEAD,AX	;Initialize the head-of-buffer ptr
	MOV	KEY_BUFF.TAIL,AX	;Init the buffer-tail pointer, too
	POPF				;Restore intr. enable status
	POPREG	<AX>
	RET
FLUSH_KEY_BUFF ENDP



;**********************************************************************
; INIT_KEY_BUFF:
;
;	Init_Key_Buff is called to initialize the keyboard buffer
; location at start-up.  It also flushes the keyboard buffer, and
; clears the keyboard shift variables.
;**********************************************************************
INIT_KEY_BUFF PROC NEAR
	PUBLIC	INIT_KEY_BUFF
	PUSHREG	<AX>
	MOV	KEY_BUFF_SEGMENT,DS	;Init buffer segment in low memory
	MOV	AX,OFFSET KEY_BUFF.BUFF	;Point to the start of the buffer
	MOV	KEY_BUFF_START,AX	;Record start-of-buffer location
	ADD	AX,32 - 2		;Point to last word of buffer
	MOV	KEY_BUFF_END,AX		;Record end-of-buffer location
	CALL	FLUSH_KEY_BUFF		;Flush out the keyboard buffer
	MOV	SHIFT_STATUS,0		;Initially, no shifts set
	MOV	LOCK_STATUS,0		;Clear out the lock-key-depressed values
	MOV	INPUT_VALUE,0		;Clear entered key code to null
	MOV	LAST_KEY,0		;Clear last-key-entered flag
	MOV	FEED_KEY,FALSE		;Flag that normal key entry is in use
	POPREG	<AX>
	RET
INIT_KEY_BUFF ENDP



MONITOR_SEGMENT ENDS

	END

