	PAGE	,132
	TITLE	Z-150 Disk I/O Routines

;**********************************************************************
;
;                       -------------------------
;----------------------- Z-150 Disk I/O Routines ----------------------
;                       -------------------------
;
;		Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES:NOTHING

	INCLUDE	../ROM/ROM.LIT
	INCLUDE ../ROM/IO.LIT
	INCLUDE	../ROM/INTR.LIT
	INCLUDE	DISK.LIT
	INCLUDE DISK.EXT

	EXTRN	DATA_SEGMENT:WORD, CASE:NEAR, GET_INTR_PTR:NEAR, DELAY:NEAR

ROM_DATA SEGMENT
	EXTRN	RESET_FLAG:WORD
ROM_DATA ENDS



;**********************************************************************
; DISK_IO_INTERRUPT: (FUNCTION, PARAMETERS)	    Interrupt #13H (19)
;
;	Disk_IO_Interrupt is the system entry point for performing I/O 
; to the disk drives.  It supports a variety of function calls, selected
; by a value in the AH register.  Once this value has been set
; appropriately, the caller performs an interrupt # 13H (19 decimal) - 
; the selected routine is then executed.
;
;	The functions supported by the Disk_IO routine include:
;
; Function Code 0 - Reset Disk System
;	This call forces all drives to be recalibrated.  Also, the
;	disk controller chip is reset and programmed for the correct
;	drive characteristics.
;
; Function Code 1 - Read Disk Status
;   Output:
;	AL: Status of last disk I/O operation, as follows:
;		00H - No error.  Last operation completed successfully.
;		01H - Illegal function number to disk I/O routine.
;		02H - Address mark (sector header) not found.
;		03H - Disk was write protected.
;		04H - Could not find requested sector.
;		08H - DMA overrun (information lost).
;	       	09H - Indicated transfer would not fit within the
;		      specified segment.
;		10H - CRC read from disk did not match calculated value.
;		20H - Disk controller chip responded with invalid
;		      information.
;		40H - Seek failed.  Could not seek to the requested
;		      track.
;		80H - Disk controller did not respond to I/O request.
;
; Function Code 2 - Read Sectors from Disk
;    Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;    ES:BX: Pointer to buffer to contain data read
;	AL: Number of sectors to read
;   Output:
;	CY: The carry flag will be reset if the read was successful.
;	AH: Disk status (see Read Disk Status call, above).
;
; Function Code 3 - Write Sectors to Disk
;    Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;    ES:BX: Pointer to data to write
;	AL: Number of sectors to write
;   Output:
;	CY: The carry flag will be reset if no error occurred - if
;	    an error did occur, it will be set.
;	AH: Disk status (see Read Disk Status call, above).
;
; Function Code 4 - Verify Sectors (check for readability)
;   Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;	AL: Number of sectors to be verified
;   Output:
;	CY: If an error occurred, the carry flag will be set - otherwise,
;	    the sectors were verified correctly.
;	AH: Disk status (see Read Disk Status call, above).
;
; Function Code 5 - Format Track
;    Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;    ES:BX: Pointer to list of sector headers, one per sector to be
;	    formatted.  The layout of each four byte sector header is:
;			Byte 0 - Track number
;			Byte 1 - Side select
;			Byte 2 - Sector number
;			Byte 3 - Number of bytes per sector:
;				0 = 128 byte sectors
;				1 = 256 byte sectors
;				2 = 512 byte sectors
;				3 = 1024 byte sectors
;	AL: Number of sectors per track.
;   Output:
;	AH: Disk status (see Read Disk Status call, above).
;
;
; NOTE: This code performs only on-track retries when I/O errors occur.
;	If I/O errors are returned to the calling program, the proper 
; 	procedure is to issue a Disk Reset function, and then the 
;	operation should be retried (at least 3 times).
;**********************************************************************
DISK_IO_INTERRUPT PROC FAR
	PUBLIC	DISK_IO_INTERRUPT
	PUSHREG	<BP,DS>
	STI				;Enable interrupts
	CLD				;Use forward byte moves
	MOV	DS,DATA_SEGMENT		;Point to the ROM's data segment
	PUSH	AX			;Save register AX
	MOV	AL,DISK_RETRIES		;Set maximum retry count
	MOV	RETRY_COUNT,AL		;Set the number of retries to perform
	POP	AX			;Restore register AX
DIO1:	PUSH	AX			;Save the operation on the stack
	CMP	AH,DMAX			;Is this a valid function request?
	JAE	DIO2			;No, return error condition
	CALL	CASE			;Yes - perform proper disk function
	JMP	NEAR PTR DIO3		;Routines will return here

DCTBL:	DW	DISK_RESET		;If AH = 0, Reset Disk System
	DW	GET_DISK_STATUS		;If AH = 1, return Drive Status
	DW	DISK_READ		;If AH = 2, Read Disk Sector(s)
	DW	DISK_WRITE		;If AH = 3, Write Disk Sector(s)
	DW	DISK_VERIFY		;If AH = 4, Verify Disk Sector(s)
	DW	DISK_FORMAT		;If AH = 5, Format Track
DMAX	EQU 	($ - DCTBL) / 2		;Number of disk functions

DIO2:	STC				;ERROR - invalid function call
	MOV	AH,COMMAND_ERROR	;Set error code for return
DIO3:	JNC	DIO4			;No error occurred - return 'Ok'
	DEC	RETRY_COUNT		;Dec retry count - permanent error?
	STC				;[Set error flag, just in case]
	JZ	DIO5			;Yes - return with error code in AH
	POP	AX			;No, restore function to be performed
	CMP	AH,1			;Was this a Get-Status?
	CLC				;[Clear carry flag in case it was]
	JZ	DIO6			;Yes - return with disk status in AH
	JMP	SHORT DIO1		;Try the operation again
DIO4:	MOV	AH,0			;No error - set AH to 0
DIO5:	POP	BP			;Throw away function # saved on stack
	MOV	DISK_STATUS,AH		;Record status of this operation
DIO6:	MOV	AL,0			;Force record-count to 0 for compatibility
	POPREG	<DS,BP>
	RET	2
DISK_IO_INTERRUPT ENDP



;**********************************************************************
; DISK_RESET:
;
;	Disk_Reset is called to initialize the disk sub-system.  It
; resets the disk controller, programs it for the requested format
; using the DISK_PARMS_INTR vector, and forces all drives used to
; be recalibrated.
;
; Output:
;	CY: Carry flag indicates whether or not an error occurred
;	    during the format (cleared if no error).
;	AH: If an error occurred during the format (Carry set), then
;	    AH will contain the error code.
;**********************************************************************
DISK_RESET PROC NEAR
	PUBLIC	DISK_RESET
	PUSHREG	<BX,CX,DX,SI>
	PUSH	DS			;Save the ROM's data segment
	CLI				;Protect the following critical code
	MOV	CL,01H			;Get bit mask for drive-on
	MOV	CH,0			;Clear drive number to 0
DRS1:	TEST	DRIVE_STATUS,CL		;Is this drive on?
	JNZ	DRS2			;Yes - have the drive number in CH
	INC	CH			;No, go on to the next drive number
	SHL	CL,1			;Move on to the next drive motor
	CMP	CH,4			;Tried all drives yet?
	JNE	DRS1			;No, check some more
	MOV	CH,0			;Yes - act as though drive 0 were on
DRS2:	MOV	AL,DRIVE_STATUS		;Get the motor control flag
	MOV	CL,4			;Shift the motor control bits...
	SHL	AL,CL			;...into position for I/O
	OR	AL,CH			;Include the drive select
	OR	AL,DISK_INTR_ENABLE	;Set the interrupts-enabled bit
	MOV	DX,DISK_CONTROL_PORT	;Point to the disk card ctrl port
	OUT	DX,AL			;Reset the disk controller
	MOV	RECAL_STATUS,0		;Force all drives to recalibrate
	OR	AL,DISK_CTRL_ENABLE	;Now, permit disk board to work
	PUSH	AX			;Delay a little while longer...
	POP	AX			;...to ensure a solid reset
	OUT	DX,AL			;Enable the disk controller card
	STI				;Permit valid interrupts
	CALL	DISK_CMD_WAIT		;Wait for reset to complete
	MOV	AL,DISK.STAT1		;Ignore any errors, get disk status
	CMP	AL,DISK_OP		;Did a drive change state?
	JNZ	DRS3			;No, return error code
	MOV	AH,DISK_SPECIFY_FUNC	;Yes - get a disk parm specify code
	CALL	SEND_DISK_PARM		;Send specify command to disk ctrlr
	JC	DRS4			;ERROR - return error code
	MOV	AL,DISK_PARMS_INTR	;Point to the disk parameter table
	CALL	GET_INTR_PTR
	MOV	AH,DS: [SI].SPECIFY1	;Get the first specification byte
	CALL	SEND_DISK_PARM		;Send it to the controller
	JC	DRS3			;ERROR - return error code
	MOV	AH,DS: [SI].SPECIFY2	;Now get the second specify byte
	CALL	SEND_DISK_PARM		;Send second parm to controller
	JMP	SHORT DRS4		;Return error / no-error status
DRS3:	MOV	AH,BAD_CTRL_ERROR	;ERROR - bad disk controller
	STC				;Set error flag
DRS4:	MOV	AL,DS: [SI].MOTOR_DELAY	;Get the motor-on time after I/O
	POP	DS			;Restore ROM's data segment
	MOV	DRIVE_MOTOR_COUNT,AL	;Keep motor running after reset
	POPREG	<SI,DX,CX,BX>
	RET
DISK_RESET ENDP



;**********************************************************************
; GET_DISK_STATUS:
;
;	Get_Disk_Status is called to return the I/O status of a disk
; operation.  It returns the same status as that returned in AH for
; the original I/O call.
;
; Output:
;	AL: Status of last disk I/O operation, as follows:
;		00H - No error.  Last operation completed successfully.
;		01H - Illegal function number to disk I/O routine.
;		02H - Address mark (sector header) not found.
;		03H - Disk was write protected.
;		04H - Could not find requested sector.
;		08H - DMA overrun (information lost).
;	       	09H - Indicated transfer would not fit within the
;		      specified segment.
;		10H - CRC read from disk did not match calculated value.
;		20H - Disk controller chip responded with invalid
;		      information.
;		40H - Seek failed.  Could not seek to the requested
;		      track.
;		80H - Disk controller did not respond to I/O request.
;**********************************************************************
GET_DISK_STATUS PROC NEAR
	PUBLIC	GET_DISK_STATUS
	MOV	AL,DISK_STATUS		;Get the status of the last disk opn
	STC				;Indicate that error code is in AH
	RET
GET_DISK_STATUS ENDP



;**********************************************************************
; DISK_READ: (DRIVE, TRACK, SIDE, SECTOR, BUFFER, SECTORS)
;
;	Disk_Read is called to transfer information from the disk
; drive to user memory.  No retries are performed - this is important,
; since the read routine does not wait for the motor to speed up prior
; to attempting to read the disk.  This is done, by the way, to get
; the highest possible performance from the disk drives.
;
; Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;    ES:BX: Pointer to buffer to contain data read
;	AL: Number of sectors to read
;
; Output:
;	CY: The carry flag will be reset if the read was successful.
;	AH: Disk status (see Read Disk Status call, above).
;**********************************************************************
DISK_READ PROC NEAR
	PUBLIC	DISK_READ
	MOV	AH,DISK_DMA_READ	;Get command to do a DMA read
	CALL	SET_DISK_IO_PTR		;Setup DMA controller for transfer
	JC	DR1			;ERROR - return error code
	MOV	AH,DISK_READ_FUNC	;Get the opcode for a disk read
	CALL	DO_DISK_IO		;Read the sectors into memory
DR1:	RET
DISK_READ ENDP



;**********************************************************************
; DISK_WRITE: (DRIVE, TRACK, SIDE, SECTOR, BUFFER, SECTORS)
;
;	Disk_Write is called to transfer information from user memory
; to a disk drive.  No retries are performed.
;
; Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;    ES:BX: Pointer to buffer containing data to be written
;	AL: Number of sectors to write
;
; Output:
;	CY: The carry flag will be reset if the write was successful.
;	AH: Disk status (see Read Disk Status call, above).
;**********************************************************************
DISK_WRITE PROC NEAR
	PUBLIC	DISK_WRITE
	MOV	AH,DISK_DMA_WRITE	;Get command to do a DMA write
	CALL	SET_DISK_IO_PTR		;Setup DMA controller for transfer
	JC	DW1			;ERROR - return error code
	MOV	AH,DISK_WRITE_FUNC	;Get the opcode for a disk write
	CALL	DO_DISK_IO		;Write the user buffer to disk
DW1:	RET
DISK_WRITE ENDP



;**********************************************************************
; DISK_VERIFY: (DRIVE, TRACK, SIDE, SECTOR, SECTORS)
;
;	Disk_Verify is called to read one or more sectors from the disk
; WITHOUT TRANSFERRING INFORMATION.  This permits an applications
; program to ensure that data previously written is readable.
;
; Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;	AL: Number of sectors to read
;
; Output:
;	CY: The carry flag will be reset if the read was successful.
;	AH: Disk status (see Read Disk Status call, above).
;**********************************************************************
DISK_VERIFY PROC NEAR
	PUBLIC	DISK_VERIFY
	MOV	AH,DISK_DMA_VERIFY	;Get DMA opcode to verify disk
	CALL	SET_DISK_IO_PTR		;Set dummy transfer address
					;Needed so that DMA chip will
					;terminate disk read operation.
	MOV	AH,DISK_VERIFY_FUNC	;Get a disk-verify opcode
	CALL	DO_DISK_IO		;Verify the specified sectors
DV1:	RET
DISK_VERIFY ENDP



;**********************************************************************
; DISK_FORMAT: (DRIVE, SIDE, BUFFER, SECTORS)
;
;	Disk_Format is called to format an entire track.  This operation
; will destroy all information on the selected track, and it will
; prepare the disk for future I/O.
;
; Input:
;	DL: Drive (unit) number (0-5)
;	DH: Side select (head) number (0-1)
;    ES:BX: Pointer to list of sector headers, one per sector to be
;	    formatted.  The layout of each four byte sector header is:
;			Byte 0 - Track number
;			Byte 1 - Side select
;			Byte 2 - Sector number
;			Byte 3 - Number of bytes per sector:
;				0 = 128 byte sectors
;				1 = 256 byte sectors
;				2 = 512 byte sectors
;				3 = 1024 byte sectors
;	AL: Number of sectors per track.
;
; Output:
;	AH: Disk status (see Read Disk Status call, above).
;**********************************************************************
DISK_FORMAT PROC NEAR
	PUBLIC	DISK_FORMAT
	MOV	AH,DISK_DMA_WRITE	;Get command to do a DMA write
	CALL	SET_DISK_IO_PTR		;Setup DMA controller for format
	JC	DF1			;ERROR - return error code
	MOV	AH,DISK_FORMAT_FUNC	;Get the opcode for a disk format
	CALL	DO_DISK_IO		;Format the track!
DF1:	RET
DISK_FORMAT ENDP



;**********************************************************************
; DO_DISK_IO: (OPERATION, DRIVE, TRACK, SIDE, SECTOR, NUMBER_OF_SECTORS)
;
;	Do_Disk_IO is called to read, write, verify or format a disk.
; It sends commands to the disk controller, waits for the operation
; to be completed, and checks the status when the operation has
; been completed.
;
; Input:
;	AH: Operation code
;	DL: Drive number (from 0-3)
;	DH: Side (head) number (0-1)
;	CH: Track (cylinder) number
;	CL: Sector (record) number
;	AL: Number of sectors to transfer
;
; Output:
;	CY: If the carry flag is reset upon return, then the operation
;	    was completed successfully.  Otherwise, AH will contain an
;	    error code.
;	AH: Error code (non-zero if an error occurred).  See DISK_STATUS
;	    for more details.
;**********************************************************************
DO_DISK_IO PROC NEAR
	PUBLIC	DO_DISK_IO
	PUSHREG	<BX,CX,SI,ES>
	MOV	BX,AX			;Place opcode in BH, # sectors in BL
	PUSH	DS			;Save data segment
	MOV	AL,DISK_PARMS_INTR	;Get the disk parameter vector number
	CALL	GET_INTR_PTR		;Point to the disk parameter table
	PUSH	DS			;Get new data segment
	POP	ES			;Place in ES register
	POP	DS			;Restore original data segment
	CALL	SELECT_DRIVE		;Setup hardware for I/O to this disk
	CALL	DISK_SEEK		;Move head to the correct track
	JC	DD5			;ERROR - bad seek
	MOV	AH,BH			;Place the disk opcode in AH
	CALL	SEND_DISK_PARM		;Send disk command to controller
	JC	DD5			;ERROR - timeout on I/O command
	MOV	AH,DH			;Get the side select
	AND	AH,01B			;Only 0-1 allowed!
	SHL	AH,1			;Place it in the right bit position
	SHL	AH,1
	OR	AH,DL			;OR in the drive number
	CALL	SEND_DISK_PARM		;Send drive/head select to controller
	JC	DD5			;ERROR - time out
	CMP	BH,DISK_FORMAT_FUNC	;Is this a format command?
	JNE	DD1			;No, continue
	CALL	SEND_FORMAT_PARMS	;Yes - send format parameters
	JMP	SHORT DD2
DD1:	CALL	SEND_RW_PARMS		;Send read/write/verify parameters
DD2:	CALL	DISK_IO_WAIT		;Wait for disk to complete I/O
	JC	DD5			;ERROR - time out
	CALL	READ_DISK_STATUS	;Read disk controller status
	JC	DD6			;ERROR - time out (return directly)
	AND	DISK.STAT1,DISK_OP	;Mask out all but operation code
	JZ	DD6			;Operation successful - return!
	CMP	DISK.STAT1,DISK_OP_BAD	;Was it an 'abnormal termination'?
	JNE	DD4			;ERROR - bad disk controller
	MOV	SI,OFFSET DISK_ERRS	;Point to the disk error table
	MOV	BL,8			;Get the number of bits to check
DD3:	LODS	WORD PTR CS: [SI]	;Get a disk error code / mask
	TEST	DISK.STAT2,AL		;Have we found the error?
	STC				;[Set error flag, just in case]
	JNZ	DD6			;Yes - return correct error code
	DEC	BL			;Gone through all legal errors?
	JNZ	DD3			;No, find another
DD4:	MOV	AH,BAD_CTRL_ERROR	;ERROR - bad disk controller!
DD5:	PUSH	AX			;ERROR - save error code on the stack
	CALL	READ_DISK_STATUS	;Flush status bytes from disk ctrlr
	POP	AX			;Restore the error code
	STC				;Set error flag
DD6:	MOV	AL,ES: [SI].MOTOR_DELAY	;Get the motor-on time after I/O
	MOV	DRIVE_MOTOR_COUNT,AL	;Keep motor running once done
	POPREG	<ES,SI,CX,BX>
	RET
DO_DISK_IO ENDP



;**********************************************************************
; DISK_SEEK: (DRIVE, TRACK, DISK_PARMS_PTR)
;
;	Disk_Seek moves a drive's Read/Write head to the specified
; track, prior to performing I/O operations.
;
; Input:
;	CH: Track (cylinder) to seek to.
;	DL: Drive to be seeked.
;    ES:SI: Disk parameter block pointer
;
; Output:
;	CY: If the carry flag is cleared, then the seek operation
;	    was completed correctly - otherwise, an error occurred,
;	    and AH will contain an error code.
;	AH: Error code if an error occurs.
;**********************************************************************
DISK_SEEK PROC NEAR
	PUBLIC	DISK_SEEK
	PUSHREG	<BX,CX>
	MOV	BL,DL			;Get the drive number as a word
	MOV	BH,0
	MOV	AL,CS:BIT[BX]		;Get a drive mask for this drive
	TEST	RECAL_STATUS,AL		;Has this drive been recalibrated?
	JNZ	DS1			;Yes - skip recalibrate
	OR	RECAL_STATUS,AL		;No, set flag for next pass
	MOV	AH,DISK_RECAL_FUNC	;Get a recalibrate code
	CALL	SEND_DISK_PARM		;Send it to the controller
	JC	DS3			;ERROR - timeout on opcode
	MOV	AH,DL			;Get the drive number to be seeked
	CALL	SEND_DISK_PARM		;Send the drive # to controller
	JC	DS3			;ERROR - timeout on drive #
	CALL	DISK_CMD_WAIT		;Wait for the recal to be processed
	MOV	AH,SEEK_ERROR		;[Possible seek error]
	JC	DS3			;ERROR - failed recalibrate
DS1:	MOV	AH,DISK_SEEK_FUNC	;Send seek command to the disk
	CALL	SEND_DISK_PARM
	JC	DS3			;ERROR - timeout on seek opcode
	MOV	AH,DL			;Send drive number to the controller
	CALL	SEND_DISK_PARM
	JC	DS3			;ERROR - timeout on drive
	MOV	AH,CH			;Send track to the disk
	CALL	SEND_DISK_PARM
	JC	DS3			;ERROR - timeout on track
	CALL	DISK_CMD_WAIT		;Wait for seek to be completed
	MOV	AH,SEEK_ERROR		;[Get error code in case it failed]
	JC	DS3			;ERROR - bad seek
	MOV	CL,ES: [SI].HEAD_SETTLE	;Get the head settling time as a word
	MOV	CH,0
	JCXZ	DS2			;No need to wait for head settling
	SHL	CX,1			;Multiply (2 Milliseconds/count)
	CALL	DELAY			;Wait for the head to settle
DS2:	CLC				;Indicate that no errors occurred
DS3:	POPREG	<CX,BX>
	RET
DISK_SEEK ENDP



;**********************************************************************
; SEND_RW_PARMS: (TRACK, SIDE, SECTOR, DISK_PARMS_PTR)
;
;	Send_RW_Parms sends the parameters required for a disk read,
; write or verify to the disk controller.
;
; Input:
;	CH: Track
;	DH: Side select
;	CL: Sector
;    ES:SI: Pointer to disk parameter block
;
; Output:
;	CY: The carry flag will be set if a timeout occurs.  Also,
;	    register AH will contain the error code for a timeout.
;	AH: Error code, in the event of an error.
;**********************************************************************
SEND_RW_PARMS PROC NEAR
	PUBLIC	SEND_RW_PARMS
	MOV	AH,CH			;Get the track number
	CALL	SEND_DISK_PARM		;Send track number to disk controller
	JC	SRW1			;ERROR - timeout on track
	MOV	AH,DH			;Send side select to disk controller
	CALL	SEND_DISK_PARM
	JC	SRW1			;ERROR - timeout on side select
	MOV	AH,CL			;Send sector to disk controller
	CALL	SEND_DISK_PARM
	JC	SRW1			;ERROR - timeout on sector number
	MOV	AH,ES: [SI].BYTES_PER_SECTOR
	CALL	SEND_DISK_PARM		;Send bytes / sector to disk ctrlr
	JC	SRW1			;ERROR - timeout occurred on BPS
	MOV	AH,ES: [SI].SECTORS_PER_TRACK
	CALL	SEND_DISK_PARM		;Send last sector to controller
	JC	SRW1			;ERROR - timeout on end-of-track
	MOV	AH,ES: [SI].GAP_LENGTH
	CALL	SEND_DISK_PARM		;Send gap length to controller
	JC	SRW1			;ERROR - timeout on gap length
	MOV	AH,ES: [SI].DATA_LENGTH
	CALL	SEND_DISK_PARM		;Send data length to controller
SRW1:	RET				;Return error code to caller
SEND_RW_PARMS ENDP



;**********************************************************************
; SEND_FORMAT_PARMS: (DISK_PARMS_PTR)
;
;	Send_Format_Parms sends the parameters required for a disk
; format.
;
; Input:
;    ES:SI: Pointer to disk parameter table
;
; Output:
;	CY: If a timeout error occurs, the carry flag will be set, and
;	    AH will contain a timeout error code.
;	AH: Error code (if an error occurs).
;**********************************************************************
SEND_FORMAT_PARMS PROC NEAR
	PUBLIC	SEND_FORMAT_PARMS
	MOV	AH,ES: [SI].BYTES_PER_SECTOR
	CALL	SEND_DISK_PARM		;Send bytes-per-sector to controller
	JC	SFP1			;ERROR - timeout on bytes-per-sector
	MOV	AH,ES: [SI].SECTORS_PER_TRACK
	CALL	SEND_DISK_PARM		;Send sectors-per-track to controller
	JC	SFP1			;ERROR - timeout on sectors-per-track
	MOV	AH,ES: [SI].FORMAT_GAP
	CALL	SEND_DISK_PARM		;Send gap length to disk controller
	JC	SFP1			;ERROR - timeout sending gap length
	MOV	AH,ES: [SI].FORMAT_DATA
	CALL	SEND_DISK_PARM		;Send sector data byte to controller
SFP1:	RET				;Return error/no-error status
SEND_FORMAT_PARMS ENDP



;**********************************************************************
; SET_DISK_IO_PTR: (MODE, BUFFER_POINTER, SECTOR_COUNT)
;
;	Set_Disk_IO_Ptr is called to set up the DMA address for
; the next disk I/O transfer.  It also checks to determine if the
; transfer is a legal one (ie, will fit within the specified
; segment without wrapping around) and will return an error if it
; is not.
;
; Input:
;    ES:BX: Pointing to the user's I/O buffer.
;	AH: Mode value to control DMA channel (refer to 8237 data
;	    sheet).
;	AL: Number of sectors to be transferred
;
; Output:
;	CY: Indicates if the transfer is valid (no carry) or
;	    invalid (carry set).
;	AH: If an error occurred, then AH will contain an error code.
;**********************************************************************
SET_DISK_IO_PTR PROC NEAR
	PUBLIC	SET_DISK_IO_PTR
	PUSHREG	<BX,CX,DX,SI,DS>
	CLI				;Keep interrupts out during setup
	MOV	CH,AL			;Place sector count in CH
	MOV	AL,AH			;Get the mode value in AL
	OUT	DMA_RESET,AL		;Prepare chip for new parameter
	OUT	DMA_CONTROL,AL		;Set mode value for DMA controller
	MOV	AX,ES			;Get the segment to transfer to
	CMP	AX,MONITOR_SEGMENT	;Is this program trying to kill the machine?
	JNE	SDP1			;No, let it fly as is
	OR	BX,8000H		;Yes - relocate request into ROM
SDP1:	MOV	CL,4			;Isolate 'offset' part of segment
	ROL	AX,CL
	MOV	CL,AL			;Put high-order segment nibble in CL
	AND	CL,0FH			;Mask out offset from segment number
	AND	AX,0FFF0H		;Mask out to get L.S. 12 bits
	ADD	AX,BX			;Add start address offset
	ADC	CL,0			;Any carry goes into M.S. 4 bits
	PUSH	AX			;Save DMA address
	OUT	DMA_ADDR_2,AL		;Output L.S. byte of I/O address
	MOV	AL,AH			;Now get the M.S. byte
	OUT	DMA_ADDR_2,AL		;Output M.S. byte of I/O address
	MOV	AL,CL			;Get the segment number
	OUT	DMA_SEGMENT_1,AL	;Output segment number to latch
	MOV	AL,DISK_PARMS_INTR	;Point DS:SI to the disk parms table
	CALL	GET_INTR_PTR
	MOV	AH,CH			;Get the sector count
	MOV	AL,0			;AX has sectors * 256
	SHR	AX,1			;Now have sectors * 128
	MOV	CL,DS: [SI].BYTES_PER_SECTOR ;Get shift count for sector size
	SHL	AX,CL			;Generate actual transfer length
	DEC	AX			;Subtract one for DMA controller
	PUSH	AX			;Save transfer length
	OUT	DMA_COUNT_2,AL		;Output L.S. byte of transfer count
	MOV	AL,AH			;Get the M.S. byte
	OUT	DMA_COUNT_2,AL		;Output M.S. byte of transfer count
	MOV	AL,DISK_DMA_ENABLE	;Get the correct mask for the disk
	OUT	DMA_ENABLE,AL		;Enable the DMA channel
	POP	DX			;Restore transfer length
	POP	AX			;Restore transfer start address
	ADD	AX,DX			;Will the transfer wrap-around?
	JNC	SDP2			;No, return
	MOV	AH,DISK_DMA_ERROR	;ERROR - invalid transfer address
SDP2:	MOV	AL,CH			;Restore sector count into AL
	STI				;Permit interrupts to occur
	POPREG	<DS,SI,DX,CX,BX>
	RET
SET_DISK_IO_PTR ENDP



;**********************************************************************
; DISK_CMD_WAIT:
;
;	Disk_Cmd_Wait waits for a disk command to execute (reset,
; seek, or recalibrate).  Once the operation has been completed,
; it will determine the status of the operation and return it.
;
; Output:
;	CY: The carry flag will be set upon return if an error
;	    occurred.
;	AH: If an error occurred, AH will contain the error code
;	    for the problem.
;**********************************************************************
DISK_CMD_WAIT PROC NEAR
	PUBLIC	DISK_CMD_WAIT
	CALL	DISK_IO_WAIT		;Wait for completion of the operation
	JC	DCW1			;ERROR - return error code
	MOV	AH,DISK_SENSE_INTR_FUNC	;Get the sense interrupt opcode
	CALL	SEND_DISK_PARM		;Send command to controller
	JC	DCW1			;ERROR - return the error code
	CALL	READ_DISK_STATUS	;Now read the status bytes returned
	MOV	AL,DISK.STAT1		;Read the disk status
	AND	AL,DISK_SEEK_FLAGS	;Mask out seek completion bits
	CMP	AL,DISK_SEEK_FLAGS	;Did the operation work properly?
	CLC				;[Assume so]
	JNZ	DCW2			;Yes - return no-error flag
DCW1:	MOV	AH,SEEK_ERROR		;ERROR - command failed
	STC
DCW2:	RET
DISK_CMD_WAIT ENDP



;**********************************************************************
; DISK_IO_WAIT:
;
;	Disk_IO_Wait is called to wait for the interrupt which follows
; completion of a disk command.  It has a built-in timeout to prevent
; the system from hanging up in the event that the disk controller
; fails.
;
; Output:
;	CY: Error flag.  Cleared if no error.
;	AH: If an error occurred, then AH will contain the error code
;	    for the problem.
;**********************************************************************
DISK_IO_WAIT PROC NEAR
	PUBLIC	DISK_IO_WAIT
	PUSHREG	<BX,CX>
	MOV	BX,DISK_IO_DELAY	;Set BX to maximum disk I/O delay
DIOW1:	MOV	CX,1			;Set delay count to a millisecond
	CALL	DELAY			;Wait for a millisecond
	TEST	RECAL_STATUS,DISK_INTR_FLAG	;Did the interrupt occur yet?
	JNZ	DIOW2			;Yes - return
	DEC	BX			;Have we waited too long yet?
	JNZ	DIOW1			;No, wait some more...
	MOV	AH,TIMEOUT_ERROR	;Yes - set timeout error code
	STC				;Set error flag for return
DIOW2:	PUSHF				;Save error / no-error status
	AND	RECAL_STATUS,NOT DISK_INTR_FLAG	;Remove interrupt flag
	POPF				;Restore error / no-error status
	POPREG	<CX,BX>
	RET
DISK_IO_WAIT ENDP



;**********************************************************************
; READ_DISK_STATUS:
;
;	Read_Disk_Status is called to read the status of an I/O
; operation from the disk controller.  It is called when an interrupt
; has been received from the disk controller chip, and it will read
; the status into the global 'DISK' structure.
;
; Output:
;	CY: If an error occurred, CY will be set.
;	AH: If CY is set, then AH will contain an appropriate error
;	    code.
;	    Sets global structure DISK.
;**********************************************************************
READ_DISK_STATUS PROC NEAR
	PUBLIC	READ_DISK_STATUS
	PUSHREG	<CX,DX,DI,ES>
	MOV	DI,OFFSET DISK		;Point to the DISK structure
	PUSH	DS			;Point to correct segment
	POP	ES
	MOV	CX,SIZE DISK_RESULTS	;Get the max length of the structure
RDS1:	MOV	AH,DISK_DIRECTION_FLAG	;Set wait-for-input flag
	CALL	DISK_XFER_WAIT		;Wait until a byte is ready
	JC	RDS4			;ERROR - return error code
	INC	DX			;Point to the disk's data port
	IN	AL,DX			;Read byte from the controller
	STOS	BYTE PTR ES: [DI]	;Store in the disk structure
	DEC	DX			;Point DX back to status port
	PUSH	CX			;Save byte count
	MOV	CX,DELAY_ONE_MS / 20	;Get constant for about 50 uSEC delay
RDS2:	LOOP	RDS2			;Wait for about 50 uSEC
	POP	CX			;Restore byte count
	IN	AL,DX			;Now, see what the chip is doing
	TEST	AL,DISK_BUSY_FLAG	;Still sending information?
	JZ	RDS3			;No, return
	LOOP	RDS1			;Loop until complete
	STC				;ERROR - too many status bytes!
	MOV	AH,BAD_CTRL_ERROR	;Set bad-controller status
	JMP	SHORT RDS4		;Return
RDS3:	CLC				;Set no-error flag
RDS4:	POPREG	<ES,DI,DX,CX>
	RET
READ_DISK_STATUS ENDP



;**********************************************************************
; SEND_DISK_PARM: (PARAMETER)
;
;	Send_Disk_Parm sends a byte parameter to the disk controller
; chip (NEC uPD-765 / Intel 8272).  The routine checks for proper
; handshaking from the disk controller chip before transferring the
; parameter - if it is not received within a reasonable amount of time,
; a timeout error will occur.
;
; Input:
;	AH: Byte parameter to be output.
;
; Output:
;	CY: Set if an error occurs, else cleared.  If an error does occur,
;	    then AH will contain an appropriate error code.
;	AH: Error code if an error occurs.
;**********************************************************************
SEND_DISK_PARM PROC NEAR
	PUBLIC	SEND_DISK_PARM
	PUSHREG	<CX,DX>
	MOV	CL,AH			;Save byte to be output
	MOV	AH,0			;Set flag for wait-for-output
	CALL	DISK_XFER_WAIT		;Wait until ready for output
	JC	SDPM2			;ERROR - return error code
	INC	DX			;Point to the disk's data port
	MOV	AL,CL			;Get the disk parm in AL
	OUT	DX,AL			;Send parameter to disk
	MOV	CX,DELAY_ONE_MS/50	;Get a 20 uSEC delay constant
SDPM1:	LOOP	SDPM1			;Delay to let the chip catch up
	CLC				;Set no-error status flag
SDPM2:	POPREG	<DX,CX>
	RET
SEND_DISK_PARM ENDP



;**********************************************************************
; DISK_XFER_WAIT: (DIRECTION_FLAG)
;
;	Disk_Xfer_Wait is called prior to receiving or sending a byte
; parameter to the disk controller chip.  It waits first for the disk
; controller to assert the READY handshake line, and then it waits
; for the direction flag in the controller to match the state of 
; DIRECTION_FLAG.
;
; Input:
;	AH: Direction flag
;
; Output:
;	CY: Indicates whether or not an error has occurred.
;	AH: If an error has occurred, AH will contain an error
;	    code.
;	DX: Pointing to the disk controller's base port.
;**********************************************************************
DISK_XFER_WAIT PROC NEAR
	PUBLIC	DISK_XFER_WAIT
	PUSHREG	<CX>
	OR	AH,DISK_READY_FLAG	;Set the ready flag in the status byte
	MOV	DX,DISK_CONTROLLER_PORT	;Point to the disk controller port
	MOV	CX,DISK_XFER_DELAY	;Maximum retry count
DXW1:	IN	AL,DX			;Read the disk port again
	AND	AL,DISK_READY_FLAG OR DISK_DIRECTION_FLAG ;Mask out no-care bits
	CMP	AL,AH			;Gotten correct status yet?
	JE	DXW2			;Yes - return with status 'OK'
	LOOP	DXW1			;No, loop until correct handshake
	STC				;ERROR - bad disk controller
	MOV	AH,TIMEOUT_ERROR	;Set error return code
DXW2:	POPREG	<CX>
	RET
DISK_XFER_WAIT ENDP



;**********************************************************************
; DISK_INTERRUPT:
;
;	Disk_Interrupt is activated when the disk system has completed
; its I/O.   This routine simply sets a flag in the ROM's data area
; to inform the disk drivers that the interrupt has occurred.
;**********************************************************************
DISK_INTERRUPT PROC FAR
	PUBLIC	DISK_INTERRUPT
	PUSHREG	<AX,DS>
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	OR	RECAL_STATUS,DISK_INTR_FLAG	;Set interrupt flag
	MOV	AL,END_OF_INTR		;Now, issue an EOI to the intr ctrlr
	OUT	INTERRUPT_CTRL,AL
	POPREG	<DS,AX>
	IRET
DISK_INTERRUPT ENDP



;**********************************************************************
; SELECT_DRIVE: (OPERATION, DRIVE, PARMS_PTR)
;
;	Select_Drive is called to prepare a disk drive for I/O.  It
; turns on the correct drive, waits for the disk to come up to speed 
; (if a write or format is requested), and sets up the hardware for
; the appropriate read or write mode.
;
; Input:
;	BH: Operation to be performed (from the Intel 8272/NEC uPD-765
;	    data sheets).
;	DL: Drive to be selected.
;    ES:SI: Pointer to disk parameter block.
;**********************************************************************
SELECT_DRIVE PROC NEAR
	PUBLIC	SELECT_DRIVE
	PUSHREG	<BX,CX,DX>
	MOV	DH,BH			;Save the operation code in DH
	CLI				;Keep interrupts out!
	MOV	CURRENT_DRIVE,DL	;Record current drive number
	MOV	DRIVE_MOTOR_COUNT,0FFH	;Keep those disks spinning!
	MOV	BL,DL			;Make the drive number a word
	MOV	BH,0
	MOV	AL,DRIVE_STATUS		;Get the current drive status
	TEST	AL,CS:BIT[BX]		;Is this drive motor already on?
	PUSHF				;Save the drive-on status on stack
	OR	AL,CS:BIT[BX]		;Flag that we are turning it on
	MOV	DRIVE_STATUS,AL		;Update the global variable
	MOV	CL,4			;Get shift to move motor control bits
	SHL	AL,CL			;Get the actual motor control enables
	OR	AL,DL			;Include actual drive number to select
	OR	AL,DISK_IO_ENABLE	;Set I/O command bits
	MOV	DX,DISK_CONTROL_PORT	;Point to disk hardware control port
	OUT	DX,AL			;Select the right drive
	POPF				;Was this drive already on?
	STI				;[Interrupts are OK, now]
	JNZ	SD1			;Yes - don't have to wait for it
	CMP	DH,DISK_READ_FUNC	;Is this a read or verify?
	JZ	SD1			;Yes - don't wait for motor
	MOV	AL,ES:[SI].MOTOR_START	;No, get the motor start time
	MOV	AH,0			;Extend it to a word
	TEST	AX,AX			;Are we to delay at all?
	JZ	SD1			;No, zero delay time - skip wait
	MOV	CL,200			;Get the multiplier for mSEC
	MUL	CL			;Get milliseconds to wait
	MOV	CX,AX			;Place result in CX
	CALL	DELAY			;Wait for the drive to speed up
SD1:	POPREG	<DX,CX,BX>
	RET
SELECT_DRIVE ENDP



;**********************************************************************
; INIT_DISK:
;
;	Init_Disk is called to initialize the floppy disk subsystems
; variables, and to determine if the disks are working correctly.  It
; does this by performing a disk reset, and then by seeking drive 0
; to tracks 1 and then 30.
;
; Output:
;	CY: The carry flag will be set upon return if an error occurred.
;	AH: In the event of an error, register AH will contain a
;	    disk error code (see the Get_Disk_Status routine, above).
;**********************************************************************
INIT_DISK PROC NEAR
	PUBLIC	INIT_DISK
	PUSHREG	<CX,DX,SI,DI,ES>	;Save registers
	MOV	CURRENT_DRIVE,0		;Set default drive to drive A
	MOV	DRIVE_STATUS,01H	;Flag that drive 1 should run
	MOV	DISK_RETRIES,DEFAULT_RETRIES ;Set the default disk-retry count
	MOV	AX,(0 SHL 8) OR 0	;Set init code for drive 0
	MOV	DX,0			;Set drive number to 0
	INT	DISK_IO_INTR		;Initialize the floppy disk system
	MOV	AX,(0 SHL 8) OR 0	;Set init code again
	INT	DISK_IO_INTR		;Reset the disk a second time
	JC	ID4			;ERROR - the disk reset failed!
	PUSH	DS			;Save the ROM data segment
	MOV	AL,DISK_PARMS_INTR	;Point to the disk parameter block
	CALL	GET_INTR_PTR		;Point DS:SI to it
	PUSH	DS			;Copy segment of DPB into ES
	POP	ES
	POP	DS			;Restore ROM data segment into DS
ID1:	MOV	BH,0			;Set null operation
	CALL	SELECT_DRIVE		;Select the appropriate drive
	MOV	CH,1			;Seek to track 1 first
	CALL	DISK_SEEK		;Perform seek to track 1
	IF	CHEAP			;If cost-reduced machine
	PUSHF				;Save error status on stack
	INC	DL			;Go on to the next drive
	POPF				;Did the seek work?
	JC	ID2			;No, don't count this drive!
	MOV	DRIVE_COUNT,DL		;Yes - update number of drives
ID2:	CMP	RESET_FLAG,1234H	;Is this a warm boot?
	JZ	ID3			;Yes - don't check all drives
	CMP	DL,4			;Tried all drives yet?
	JNE	ID1			;No, continue seeking
ID3:	CMP	DRIVE_COUNT,0		;Are there any valid disk drives?
	STC				;[Set the carry flag, just in case]
	JZ	ID4			;No, must have at least one drive!
	CLC				;Yes - clear the error flag
	ELSE				;If normal Z-150, seek to track 30
	JC	ID4			;ERROR - seek to track 1 failed
	MOV	CH,30			;Set track to track # 30
	CALL	DISK_SEEK		;Perform seek to track 30
	ENDIF
ID4:	MOV	AL,ES: [SI].MOTOR_DELAY	;Get the motor-on time after I/O
	MOV	DRIVE_MOTOR_COUNT,AL	;Keep motor running once done
	POPREG	<ES,DI,SI,DX,CX>	;Restore registers
	RET				;Return error code in AH, flag in CY
INIT_DISK ENDP



;**********************************************************************
;		     D I S K   E R R O R   T A B L E
;**********************************************************************
		PUBLIC	DISK_ERRS
DISK_ERRS	DB	10000000B,RECORD_ERROR
		DB	00000000B,BAD_CTRL_ERROR
		DB	00100000B,CRC_ERROR
		DB	00010000B,OVERRUN_ERROR
		DB	00000000B,BAD_CTRL_ERROR
		DB	00000100B,RECORD_ERROR
		DB	00000010B,WRITE_ERROR
		DB	00000001B,ADDRESS_ERROR


;**********************************************************************
;		           B I T   A R R A Y
;**********************************************************************
		PUBLIC	BIT
BIT		DB	01H,02H,04H,08H,10H,20H,40H,80H



MONITOR_SEGMENT ENDS

	END
