/*****************************************************************************
 * PT - Record and play 1621 and 1624 data.
 *
 * Hardware interface: AT-style LPT.  Data=3BC, Status=3BD, Control=3BE
 *
 * 	STATUS	CTL BIT2=0	CTL BIT2=1
 *
 *	Bit7	Rdr	   I	Data[4]		Busy
 *	Bit6	Punch		Data[3]		Ack
 *	Bit5	Data[7]		Data[2]		PE
 *	Bit4	Data[6]		Data[1]		Selected
 *	Bit3	Data[5]		Data[0]		Error
 *
 *	CONTROL
 *
 *	Bit3	Select	   I L	1 = D75/RdyPc, 0 = D40
 *	Bit2	Init	     H	0 = DDEn/RcvPc, 1 = Rcv21
 *	Bit1	Autofeed   I H	1 = Drive Pcb1 to -S
 *	Bit0	Strobe	   I H	1 = SyncPc and Pcb3, 0 = Enable Mag +5
 *
 * Notes:
 *	PT can record tapes being read by the
 *      1621 reader (including during NonProcess Runout),
 *      and data being punched by the 1624.
 *
 *	If the punch switch is set to OFF, PT suppresses the 1624 punch
 *      and records punch data while faking punch sync signals to the 1620.
 *	If the switch is ON, PT passively records punch data.     
 *
 *	If PT is invoked with a filename, it suppresses the 1621, and
 *      sends file data to the 1620 whenever Rdr goes high.  When PT
 *      runs out of data, it re-enables the 1621 and passively records
 *      any data the 1621 sends.
 *	On exit, PT appends all recorded data to 'rdr.pt' and 'punch.pt'.
 *
 *	To record reader data, we monitor the Rdr line (which is
 *	driven by Sync21) and Data, and
 *	passively record as the 1620 does its normal tape-read
 *	operations or responds to the NonProcess Runout key.
 *	We read on the trailing edge of Rdr, buffering the data
 *	until exit.
 *
 *	If we start with a file, we assert INIT, which removes -12V
 *	from the 1621 photodiodes, routes Clutch20 to Rdr instead of
 *	to Clutch21, disables Rdy21, enables RdyPc, and enables the
 *	data drivers.  We monitor Clutch20, and drive Sync, Rdy, and Data.
 *	Since SyncPc drives Rdr, we only monitor Rdr when SyncPc is
 *	low, which is all the time except for the pulse.
 *	INIT only enables RdyPc.  To assert it we assert SELECT,
 *	which also controls the tape receiver mux.  Since we
 *	aren't receiving, the mux's state doesn't matter.
 *
 *	When we run out of file data we negate INIT and SELECT,
 *	which enables the 1621, and become passive.
 *
 *	When recording we read data via the LPT Status port.
 *	Since there are only 5 status bits, we multiplex the 8 data bits
 *	and the Rdr and Punch bits.
 *
 *	In addition to recording
 *	the tape reader's output, we record Device 2 (punch) data.
 *	The punch receivers are enabled (and the reader receivers
 *	disabled) during the time Pcb1 is low.
 *
 *	We monitor the punch on Status bit 6 (Punch).  Punch
 *	is low whenever the Feed SCR anode is high or Pcb1 is low.
 *	We monitor the anode to get the first byte.
 *	The 1624 is running if Punch goes high and then low in
 *	(T = 38 deg * 0.18 mS/deg + clutch engage delay).
 *	The magnets go on and off with Pcb2.  The SCR gates are driven
 *	from Pcb1 (or Feed on the first cycle) until Pcb3.
 *	We read the data after the trailing (falling) edge of Punch,
 *	i.e., the falling edge of Pcb1.  We do not save the final byte
 *	read, because the final Punch falling edge occurs not because
 *	of Pcb1 dropping but because of Pcb2 picking without Feed.
 *	The 1624 stops in this position.
 *
 *	To suppress the punch, we switch the 1624's magnets from +48 to a
 *	+5 supply which is too weak to pick the magnets.  We also drive
 *	Pcb1 and Pcb3, overriding the idle 1624's integrator outputs.
 *	When idle, Pcb1 is high and Pcb3 is low.  Fortunately, it is
 *	possible to drive integrator outputs either high or low.
 *	Our drive to Pcb3 also controls the +5 magnet supply.
 *	It goes on during the time we are not driving Pcb3 high.
 *	If Punch goes high and stays high, the 1624 is suppressed.  
 *	We drive Pcb1 low and read the data.
 *	Then we drive Pcb3 high and low again.  This drops the SCRs.
 *	Then we drive Pcb1 high and wait 100 uS for Punch.
 *	If it goes high, we immediately drive Pcb1 low and repeat the cycle.
 *	If it stays low, the IO is done.
 *
 *	I picked the bit assignments on the Control port so that
 *	a freshly-booted PC wouldn't disturb the 1620 system.
 *
 */

#include <alloc.h>
#include <conio.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <io.h>
#include <process.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* D75 is also RdyPc.  Absence of D75 is D40. */
/* SYNC is also PCB3.  Absence of SYNC is +5Mag. */
#define D75		0x08
#define RCV21		0x04
#define RCVPC		0x00
#define PCB1		0x02
#define SYNC		0x01

#define POLARITY_FIX	0x80
#define D75MASK		0x38
#define RDRMASK		0x80
#define RDRHIGH		0x00
#define RDRLOW		0x80
#define PUNCHMASK	0x40
#define PUNCHHIGH	0x40
#define PUNCHLOW	0x00

char path[MAXPATH], drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];

unsigned int data = 0x3BC;
unsigned int stat = 0x3BD;
unsigned int ctl = 0x3BE;
unsigned char *buf, *p;
jmp_buf jmpb;
unsigned int dotCount = 0;

int breakFlag;
FILE *f = NULL;
FILE *t;
unsigned int pos;

void usage(void)
{
    printf("usage: pt [file] [-@pos]\n");
}
    
void wait(int n)		/* (0 to 15 uS) + (n * 15 uS) */
{
    int last = inportb(0x61);
    
    n++;
    while (n--)
    {
	while (inportb(0x61) == last)
	    ;
	last = inportb(0x61);
    }
}

void end(int n)
{
    outportb(ctl, RCV21);
    flushall();
    exit(n);
}

void cBreak1(void)			/* Used for playing */
{
    outportb(ctl, RCVPC);			/* Go NOT READY */
    if (!feof(f))
    {
	fseek(f, -1, SEEK_CUR);			/* Cancel prefetch */
	printf("\nUSER BREAK WHILE PLAYING %s AT POSITION %lu (%lu REMAINING)",
	    path, ftell(f), filelength(fileno(f)) - ftell(f));
	printf(", ABORT?[Ny]");
	if (tolower(getche()) == 'y')
	    f = NULL;
    }
    longjmp(jmpb, 1);			/* Restart READY */
}

/*
 * When we are in the data output loop we must break only after
 * dropping sync.  This keeps the 1620 happy and allows us
 * to restart without dropping or doubling a byte.
 */
int cBreak2(void)
{
    breakFlag = 1;		/* We'll catch this at the right time */
    return 1;			/* Continue execution */
}

void writeFile(void)
{
    size_t i;
    
    i = (size_t) (p - buf);
    i = fwrite(buf, 1, i, t);
    if (i != p - buf)
	printf("\nWRITE FAILED! ");
    else
	printf("\nDONE, ");
    printf("%lu bytes written to rdr.pt\n", (long)i);
}

int cBreak3(void)		/* Used for tape recording */
{
    printf("\nUSER BREAK.");
    if (f && !feof(f))
	printf("  PT FILE %s WAS AT POSITION %lu (%lu REMAINING)",
	    path, ftell(f), filelength(fileno(f)) - ftell(f));
    if (p != buf)
    {
	printf("\nWRITE CAPTURED DATA? [Yn]");
	if (tolower(getche()) == 'y')
	    writeFile();
    }
    end(0);			/* Abort program */
    return 0;			/* Just to shut up the compiler */
}

void playFile(void)		/* Play until EOF or -Clutch20 */
{
    int c;
    
    breakFlag = 0;			/* No Ctl-C yet */
    /*
     * Note that c is an int not a char.  This is so it can hold EOF.
     * We fetch one char ahead so feof() will go true after we output
     * the last char on the tape (and fetch the next, which will be EOF).
     */
    c = fgetc(f);			/* Prime the pump */

    putchar('+');			/* Clutch was engaged */
    dotCount = 0;			/* 1st char of rec */
    
    while ((wait(0), (inportb(stat) & RDRMASK) == RDRHIGH && c != EOF)
	|| (wait(0), (inportb(stat) & RDRMASK) == RDRHIGH && c != EOF)
	|| (wait(0), (inportb(stat) & RDRMASK) == RDRHIGH && c != EOF))
    {
	outportb(data, c ^ 0xFF);	/* Output data */
	c = fgetc(f);			/* Prefetch next */
	outportb(ctl, RCVPC|D75|SYNC);	/* Raise SyncPc */
	wait(1);			/* t_PulseWidth */
	outportb(ctl, RCVPC|D75);	/* Lower SyncPc */
	/*
	 * ClutchPc goes low about 70 uS after trailing
	 * edge of SyncPc on final char of record (EOL).
	 * Wait long enough so we see it go low.
	 * Must not wait more than 270 uS, because by that
	 * time it may have gone high again.
	 */
	if (dotCount++ == 100)
	{
	    dotCount = 0;		/* Show activity */
	    putchar('.');		/*  on long records */
	    wait(5);
	}
	else
	    wait(6);			/* Wait for -Clutch */

	if (breakFlag)			/* Ctl-C */
	    cBreak1();			/* Abort/Resume */
    }
    
    if (c != EOF)			/* We lost Clutch20 */
	fseek(f, -1, SEEK_CUR);		/* Cancel prefetch */
    else				/* Ran out of data */
    {
	f = NULL;			/* Don't come back */
	outportb(ctl, RCV21|D75);	/* Back to passive record mode */
    }
}

void recordTape(void)			/* Record one byte */
{
    int c, d;

    /*
     * Sync has gone low.  READ THE DATA!
     */
    c = inportb(stat);			/* Get D[7:5] */
    outportb(ctl, RCV21);		/* Set mux to read D[4:0] */
    wait(1);				/* Waste time while mux settles */
    d = inportb(stat);			/* Get D[4:0] */
    outportb(ctl, RCV21|D75);		/* Back to reading status + D[7:5] */
    wait(1);				/* Let mux settle back to D75 */
    *p++ = ((c & D75MASK) << 2) | ((d ^ POLARITY_FIX) >> 3); /* Store */

    if (dotCount++ == 100)
    {
	dotCount = 0;
	putchar('.');			/* Show activity */
    }
}
    
int main(int argc, char *argv[])
{
    unsigned int c, d;
    unsigned int tick;
    
    outportb(ctl, RCV21|D75);	/* Put us in sane state */
    
    argc--;			/* Consume argv[0] (progname) */
    if (argc > 2)		/* Filename, pos */
    {
	usage();
	end(0);
    }
    
    if (argc)			/* Advance to argv[1] (filename) */
    {
	argv++; argc--;
	if (**argv == '-' || **argv == '/')
	{
	    usage();
	    end(0);
	}
	/* File extension defaults to ".pt" */
	strcpy(path, *argv);
	fnsplit(path, drive, dir, file, ext);
	if (strlen(ext) == 0)
	    strcpy(ext, ".pt");
	fnmerge(path, drive, dir, file, ext);
    
	f = fopen(path, "rb");
	if (f == NULL)
	{
	    perror(*argv);
	    end(1);
	}
    }

    /* Set up default values for options */
    pos = 0;			/* Start playing at beginning of file */

    while (argv++, argc--)	/* Advance to next argv, if any */
    {
	sscanf(*argv, "-@%u", &pos);
    }

    t = fopen("rdr.pt", "ab");
    p = buf = farmalloc(65536L);
    
    if (t == NULL)
    {
	perror("rdr.pt");
	end(1);
    }
    
    if (buf == NULL)
    {
	printf("\nmalloc failed");
	end(1);
    }

    /*
     * MAINTENANCE NOTE: If the program prints enough to cause a
     * text scroll, tape data will be lost on a 12 MHz AT.  We try to
     * avoid that by starting with a blank screen.
     */
    clrscr();		/* Start with nice fresh empty screen */
    
    /*
     * Test for powered-down.
     * STROBE has a 4.7k pull-up resistor at the LPT port.
     * If the 1621 is not powered, that is the only source of +5V.
     * If all the Control lines are high, Status will read 0b01111xxx.
     * A low Control line will, through the transistors in the interface,
     * pull down the +5 and with it the Status lines.
     * They'll read 0b10000xxx.
     */
    outportb(data, 0xFF);			/* Idle the drivers */
    outportb(ctl, RCV21);			/* All Controls high */
    wait(1);
    c = inportb(stat);				/* High if unpowered */
    outportb(ctl, RCV21|SYNC);			/* Take STROBE low */
    wait(10);
    d = inportb(stat);				/* Low if unpowered */
    if ((c & 0xF8) == 0x78 && (d & 0xF8) == 0x80)
    {
	    printf("\n1621 IS POWERED OFF.  POWER ON AND RESTART PT.");
	    end(0);
    }
    /*
     * Test for cable not connected.  If we assert INIT and write
     * FF to our inverting data drivers, we should see LOW on all
     * the status lines, which looks like 0x80 because bit 7 inverts.
     * If the cable is not connected, we'll see HIGH on all status
     * lines, which looks like 0x78.
     * SPECIAL NOTE: To prevent a 1620 Response from causing a
     * Read Check, we must drive tapefeed code onto the data bus
     * before we drop SYNC.
     */
    /* Read D[7:5] with 1621 disabled. */
    outportb(data, 0);			/* Looks like tapefeed */
    outportb(ctl, RCVPC|D75);		/* Tapefeed so safe to drop SYNC */
    wait(1);				/* Keep tapefeed a moment */
    outportb(data, 0xFF);		/* Tape bus floats to -S */
    wait(1);				/* Allow mux to settle */
    c = inportb(stat);			/* Get D[7:5] */
    outportb(ctl, RCVPC);
    wait(1);
    d = inportb(stat);			/* Get D[4:0] */
    if ((c & 0xF8) == 0x78 && (d & 0xF8) == 0x78)
    {
	printf("\nCABLE TO 1621 IS NOT CONNECTED.  CONNECT AND RESTART.");
	end(0);
    }

    if (f)
	fseek(f, pos, SEEK_SET);

    setjmp(jmpb);			/* cBreak1 may return here */

    printf("\nREADY");
    ctrlbrk(cBreak3);
    if (f)
	outportb(ctl, RCVPC|D75);
    else
	outportb(ctl, RCV21|D75);
    wait(1);

    while (1)
    {
    	/* Wait for something to happen */
	/* ??? got to add PUNCHMASK */
	while ((inportb(stat) & RDRMASK) == RDRLOW)
	    kbhit();				/* Trap Ctl-C */

	/*
	 * Rdr has gone high.  If we're playing a file (RCVPC), Rdr
	 * reflects Clutch20, and we start playing immediately.
	 * It's possible for the 1621 to stop with Sync21 high,
	 * but since RCVPC disables Sync21, we won't notice unless
	 * we're passively recording, in which case we don't care,
	 * except that we have to allow Ctl-C to get through.
	 * Kbhit() takes a long time, and if we ran it continuously
	 * we might not notice the falling edge until too late.
	 * Therefore we reduce (but not eliminate!) the risk of
	 * this by running it only once per second while we wait
	 * for Sync21 to go low.  When it does, we record a byte.
	 * We read up to three times, waiting 15 uS between
	 * reads to filter glitches.  The wait(0) on the first
	 * syncs us to the 15 uS tick.  The second and third
	 * wait(0)'s each delay 15 uS from the previous tick.
	 */
	tick = 0;				/* 65536 * 15 uS ~= 1 Sec */
	while ((wait(0), (inportb(stat) & RDRMASK) == RDRHIGH)
	    || (wait(0), (inportb(stat) & RDRMASK) == RDRHIGH)
	    || (wait(0), (inportb(stat) & RDRMASK) == RDRHIGH))
	{
	    if (f)			/* Playing a file? */
		break;			/* Yes, start immediately */
	    if (--tick == 0)		/* Sync21 stuck high? */
		kbhit();		/* Yes, trap Ctl-C, tick wraps */
	}

	if (f)				/* Playing a file */
	{
	    ctrlbrk(cBreak2);		/* Delayed cBreak handler */
	    playFile();
	    ctrlbrk(cBreak3);		/* Standard cBreak handler */
	}
	else				/* Recording 1621 data */
	    recordTape();		/* Record one byte */
    }
    return 0;				/* Just to shut up the compiler */
}
