/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Philips Semiconductor PCF8574 I2C controller driver
 *  $Id: 
 */

#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "specifics.h"
#include "northbridge.h"
#include "platform.h"
#include "pcf8584.h"



/*------------------------------------------------------------------------*/
/* Primitive IO */

#define I2C_DELAY_MSEC	2			/* Arbitrarily chosen */

#define DUMMY_ADDRESS_HACK	0x0		/* Soohoon's hack */

static uint8 pcf_read( const unsigned reg )
{
    uint8 result = inmemb( PCF_ADDR( reg ) ); 
    msleep( I2C_DELAY_MSEC );

    //TRACE( "Read from 0x%X was 0x%02X\n", PCF_ADDR( reg ), result );

    return result;
}

static void pcf_write( const unsigned reg, const uint8 val )
{
    outmemb( PCF_ADDR( reg ), val );
    msleep( I2C_DELAY_MSEC );

    //TRACE( "Write to 0x%X of 0x%02X\n", PCF_ADDR( reg ), val );
    inmemb( DUMMY_ADDRESS_HACK );
}


/* Some higher-level functionality */

#define PCF_START	( PCF_S1_PIN | PCF_S1_ESO | PCF_S1_STA | PCF_S1_ACK )
#define PCF_STOP	( PCF_S1_PIN | PCF_S1_ESO | PCF_S1_STO | PCF_S1_ACK )




/*------------------------------------------------------------------------*/
/* Helpful debug routines */


#ifdef TRACE_ENABLE
static void pcf_reg_dump( const uint8 s1orig )
{
    uint8 s0, s0prime, s1, s1status, s2, s3;

    /* Switch off the serial interface, clear all address bits */
    pcf_write( PCF_CTL, PCF_S1_PIN | PCF_S1_ACK );


    /* Read it back again */
    s1 = pcf_read( PCF_CTL );
    s0prime = pcf_read( PCF_DATA );

    pcf_write( PCF_CTL, s1 | PCF_S1_ES2 );
    s3 = pcf_read( PCF_DATA );

    pcf_write( PCF_CTL, s1 | PCF_S1_ES1 );
    s2 = pcf_read( PCF_DATA );


    /* Enable the serial interface to access the remaining registers */
    s1 |= PCF_S1_ESO;
    pcf_write( PCF_CTL, s1 );

    s1status = pcf_read( PCF_CTL );
    s0 = pcf_read( PCF_DATA );			/* NB check for IACK = 1 */

    pcf_write( PCF_CTL, s1 | PCF_S1_ES2 );
    s3 = pcf_read( PCF_DATA );


    mobo_logf(
	LOG_INFO "PCF8584 Registers:\n"
	LOG_INFO "s0:  0x%02X (data)\n"
	LOG_INFO "s0': 0x%02X (own address)\n"
	LOG_INFO "s1:  0x%02X (control)\n"
	LOG_INFO "s1:  0x%02X (status)\n"
	LOG_INFO "s2:  0x%02X (clock)\n"
	LOG_INFO "s3:  0x%02X (interrupt vector)\n",
	s0, s0prime, s1, s1status, s2, s3 );

    /* restore desired control register value */
    pcf_write( PCF_CTL, s1orig );
}
#endif			/* TRACE_ENABLE */


/*------------------------------------------------------------------------*/

#define TIMEOUT 500

/* The Pending Interrupt Not bit is used in most I2C transactions */
/* Here's how it is useful:
 * - Sending data (by setting STA bit) sets PIN->1
 * - Transmitter mode: On completion of a byte transmission PIN->0
 *	(when S0 is written again PIN->1)
 * - Receiver mode: On completion of a byte read PIN->0
 *	(when S0 is read PIN->1)
 * - On bus error PIN->0 and BER->1
 */

static DBM_STATUS wait_pcf_ready( uint8 *status )
{
    int timeout;

    TRACE( "Running...\n" );

    for ( timeout=0; timeout < TIMEOUT; timeout++ )
    {
	*status = pcf_read( PCF_CTL );
	if ( (*status & PCF_S1_PIN) == 0 )
	{
	    if ( *status & PCF_S1_BER )
	    {
		mobo_logf( LOG_WARN "PCF8584 I2C bus error detected\n" );
		return STATUS_WARNING; 
	    }
	    return STATUS_SUCCESS;
	}
	TRACE( "0x%02X\n", *status );
	msleep( 10 );
    }

    /* Timeout condition */
    mobo_logf( LOG_WARN "PCF8584 timed out on waiting for PIN\n" );
#ifdef TRACE_ENABLE
    pcf_reg_dump( PCF_S1_PIN | PCF_S1_ACK);
#endif /* TRACE_ENABLE */
    return STATUS_FAILURE;
}


static DBM_STATUS wait_pcf_free( void )
{
    int timeout;
    uint8 status;

    TRACE( "Running...\n" );

    for ( timeout=0; timeout < TIMEOUT; timeout++ )
    {
	status = pcf_read( PCF_CTL );
	if ( status & PCF_S1_BB )
	    return STATUS_SUCCESS;

	TRACE( "0x%02X\n", status );
	msleep( 100 );
    }

    /* Timeout condition */
    mobo_logf( LOG_WARN "PCF8584 timed out - bus busy\n" );
#ifdef TRACE_ENABLE
    pcf_reg_dump( PCF_S1_PIN | PCF_S1_ACK );
#endif /* TRACE_ENABLE */
    return STATUS_FAILURE;
}



/* Write one byte and test for remote ack */
static DBM_STATUS push_byte( uint8 byte )
{
    uint8 status;
    DBM_STATUS timeout;

    /* Is the PCF ready to receive a byte? */
    timeout = wait_pcf_ready( &status );
    if ( timeout != STATUS_SUCCESS )
    {
	TRACE( "timeout in waiting for bus to become free before transmit\n" );
	pcf_write( PCF_CTL, PCF_STOP );
	return STATUS_FAILURE;
    }

    /* Write data byte */
    pcf_write( PCF_DATA, byte );

    /* Did the PCF send the byte out and become free? */
    timeout = wait_pcf_ready( &status );
    if ( timeout != STATUS_SUCCESS )
    {
	TRACE( "timeout in waiting for bus to become free after transmit\n" );
	pcf_write( PCF_CTL, PCF_STOP );
	return STATUS_FAILURE;
    }

    /* Did the target acknowledge receipt of the byte? */
    if ( ( status & PCF_S1_LRB ) != 0 )
    {
	TRACE( "target device did not acknowledge\n" );
	pcf_write( PCF_CTL, PCF_STOP );
	return STATUS_FAILURE;
    }

    return STATUS_SUCCESS;
}

static DBM_STATUS pull_byte( uint8 *byte, int n )
{
    DBM_STATUS timeout;
    uint8 status;

    TRACE( "pulling...\n" );

    /* Is the bus ready? */
    timeout = wait_pcf_ready( &status );
    if ( timeout != STATUS_SUCCESS )
    {
	TRACE( "timeout in waiting for bus to become free before read\n" );
	pcf_write( PCF_CTL, PCF_STOP );
	return STATUS_FAILURE;
    }


    /* Did the target acknowledge? */
    if ( n > 1 && ( status & PCF_S1_LRB ) != 0 )
    {
        TRACE( "target device did not acknowledge\n" );
	pcf_write( PCF_CTL, PCF_STOP );
	return STATUS_FAILURE;
    }


    /* How many bytes are there left?  Any special action required? */
    if ( n == 2 )
	pcf_write( PCF_CTL, PCF_S1_ESO );		/* Almost there */

    if ( n == 1 )
	pcf_write( PCF_CTL, PCF_STOP );			/* Stop! */


    /* Read data byte (Having borne in mind the first read returns address) */
    *byte = pcf_read( PCF_DATA );

    TRACE( "pulled 0x%02X\n", *byte );
    return STATUS_SUCCESS;
}


/*------------------------------------------------------------------------*/
/* Public interface functions */

DBM_STATUS i2cdev_init( void )
{
    uint8 status;

    TRACE( "Configuring PCF8584...\n" );

    /* Serial I/F disable during setup */
    pcf_write( PCF_CTL, PCF_S1_PIN );

    /* Program own-address with default value */
    pcf_write( PCF_DATA, 0x55 );

    /* Configure clock: serial clk of 90 kHz, internal clock of 12 MHz */
    pcf_write( PCF_CTL, PCF_S1_PIN | PCF_S1_ES1 );
    pcf_write( PCF_DATA, PCF_S2_CLK12 | PCF_S2_SCL90 );

    /* setup for operation: Enable serial output */
    pcf_write( PCF_CTL, PCF_S1_PIN | PCF_S1_ESO | PCF_S1_ACK );

    status = pcf_read( PCF_CTL );
    if( (status & PCF_S1_INIT) != 0 )
    {
	mobo_logf( LOG_CRIT "PCF8584: Initialisation failed, status=0x%02X\n",
	    status );
	return STATUS_FAILURE;
    }

    TRACE("PCF8584 Initialized, status = 0x%02X\n", status );
    return STATUS_SUCCESS;
}

DBM_STATUS i2cdev_write( const uint8 addr, const int cmd,
			 const size_t count, const uint8 *buf )
{
    int n;
    DBM_STATUS timeout;
    DBM_STATUS sval;

    /* Is bus busy? */
    timeout = wait_pcf_free( );
    if ( timeout != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "I2C: bus found jammed whilst attempting write\n");
	return -1;
    }

    /* Send byte slave address and set start condition */
    pcf_write( PCF_DATA, PCF_WR(addr));
    pcf_write( PCF_CTL, PCF_START );

    /* Do we send a command byte first? */
    if ( cmd != -1 )
    {
	TRACE( "sending command byte 0x%02X\n", cmd );
	sval = push_byte( cmd );
        if ( sval != STATUS_SUCCESS )
        {
            mobo_logf( LOG_CRIT "Failure on write of command byte to 0x%02X\n",
                        addr );
            return STATUS_FAILURE;
        }
    }

    /* Loop over all data bytes to be transmitted */
    for ( n = 0; n < count; n++ )
    {
	sval = push_byte( buf[n] );
	if ( sval != STATUS_SUCCESS )
	{
	    mobo_logf( LOG_CRIT "Failure on write of byte %d of %d to 0x%02X\n",
			n, count, addr );
	    return STATUS_FAILURE;
	}
    }

    /* Send stop condition */
    pcf_write( PCF_CTL, PCF_STOP );

    TRACE( "%d bytes written successfully\n", n );
    return STATUS_SUCCESS;
}


DBM_STATUS i2cdev_read( const uint8 addr, const int cmd,
			const size_t count, uint8 *buf )
{
    int i;
    uint8 dummy_read;
    DBM_STATUS timeout, sval;

    TRACE( "addr 0x%02X, cmd 0x%02X, count %d\n", addr, cmd, count );

    /* Paranoia: if the read fails, we don't want to return misleading data */
    memset( buf, 0xFF, count );

    if ( cmd != -1 )			/* Send out the command byte */
    {
	TRACE( "sending command byte 0x%02X\n", cmd );

	sval = i2cdev_write( addr, -1, 1, (uint8 *)&cmd );
	if ( sval != STATUS_SUCCESS )
	{
	    TRACE( "That didn't work...\n" );
	    return STATUS_FAILURE;
	}
    }

    /* Put slave address out on the bus */
    TRACE( "Putting address onto bus\n" );
    pcf_write( PCF_DATA, PCF_RD(addr) );

    /* Is the bus free? */
    TRACE( "Waiting for BB\n" );
    timeout = wait_pcf_free( );
    if ( timeout != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "I2C: bus was jammed upon attempting read\n");
	return STATUS_FAILURE;
    }

    /* Generate start condition */
    TRACE( "Generating START\n" );
    pcf_write( PCF_CTL, PCF_START );


    /* Perform a dummy read to get the address byte back, which has the side
     * effect of bringing the first read byte in off the bus */

    sval = pull_byte( &dummy_read, count + 1 );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "Error on setup for I2C read from 0x%02X\n", addr );
	return STATUS_FAILURE;
    }


    /* Loop over bytes expected (NB last bytes are special case) */
    for( i=count; i > 0; --i )
    {
	sval = pull_byte( buf++, i );
	if ( sval != STATUS_SUCCESS )
	{
	    mobo_logf( LOG_CRIT "Failure on read of byte %d of %d to 0x%02X\n",
			count-i, count, addr );
	    return STATUS_FAILURE;
	}

    }

    TRACE( "read %d bytes from addr 0x%02X\n", count, addr );
    return STATUS_SUCCESS;
}

