/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/


#include "lib.h"
#include "uilib.h"
#include "northbridge.h"
#include "pci.h"
#include "beepcodes.h"

#include "time.h"

#include "vga.h"
#include "vgatest.h"

#define VGABASE		0xB8000
#define VGABUFSIZ	(32 * 1024)		/* 32K of VGA text pages */



static DBM_STATUS romtest( void );


/*----------------------------------------------------------------------*/
/* Initialize the VGA (via a hotch-potch combination of approaches) and 
 * print a simple 'test card' */


DBM_STATUS vga( int argc, char *argv[] )
{
    int i,j;
    DBM_STATUS status;


    diags_subsys_stat( SYS_VGA, DEV_PROBING );

    /* Check the integrity of each VGA card in the system */
    status = romtest();
    if ( status != STATUS_SUCCESS )
    {
	BeepCode( beep_k_vgawarn );
	mobo_logf( LOG_WARN "VGA: ROM integrity test not happy, VGA "
		            "initialisation may fail\n" );
    }

    status = InitVGA();
    if ( status != STATUS_SUCCESS ) {
	diags_subsys_stat( SYS_VGA, DEV_FAILED );
	return status;
    }
    diags_subsys_stat( SYS_VGA, DEV_SETUP );

    /* runs for a fixed number of iterations and then stops */
    for ( j = 0; j < 8; j++ )
    {
	for ( i = 0; i < 80*25; i++ ) {
	    outmemw( VGABASE + i*2, 0x1F00 | (j % 26 + 'a') );
	}
	usleep( 200000 );
    }

    /* now clear the screen */
    for ( i = 0; i < 80*25; i++ ) {
	outmemw( VGABASE + i*2, 0x1F00 | ' ' );
    }

    vgaerase();			/* tidy the screen up */

    return status;
}


/*----------------------------------------------------------------------*/
/* VGA buffer stress test, uses the 0xB8000-0xC0000 range (ie VGA text) 
 * for a limited "memory test" */

static void logerr( int iter, int addr, int wval, int rval )
{
    printf_dbm("ERROR!  Pass %d, address 0x%08X, wrote 0x%08X read 0x%08X\n",
		iter, VGABASE + addr, wval, rval );
}


DBM_STATUS vgastr(void)
{
    unsigned seed;
    unsigned wval, rval;
    unsigned i,j;
    unsigned long stime, etime, ticks, bwidth;



    /* Initialise the VGA via our hotch-potch combination */
    vgainit();
#if 0			/* this throws up the screen, see if we can bypass */
    vgasetup();
    testvga_init();
#endif

    printf_dbm( "32K VGA buffer random data test...\n");

    for (i=0; ; i++) {			/* event loop */
	seed = rpcc();
	srandom( seed );


	/* Write of data, 32 bits at a time */
	printf_dbm( "Pass %d writing...", i);

	stime = rpcc();
#if 0
	for ( j = 0; j < VGABUFSIZ / sizeof(wval) ; j++) {
	    /* wval = random(); */
	    wval = j;
	    outmeml( VGABASE + j*sizeof(wval), wval );		/* PCI mem wr */
	}
#else
	vgafill();
#endif
	etime = rpcc();
	if ( stime >= etime )		etime += 1UL << 32;	/* wraparound */
	ticks = etime - stime;
	bwidth = ( VGABUFSIZ >> 10 ) * 1000000000000UL;
	bwidth /= primary_impure->CYCLE_CNT * ticks;
	printf_dbm( "%d kbytes/sec\n", bwidth );


	srandom( seed );
	

	/* read of data */
	/* printf_dbm( "Pass %d reading...", i ); */

        stime = rpcc();
        for ( j = 0; j < VGABUFSIZ / sizeof(wval) ; j++) {
            /* wval = random(); */
	    wval = VGABASE + j*sizeof(wval);
	    rval = inmeml( VGABASE + j*sizeof(wval) );          /* PCI mem rd */
	    if ( rval != wval )		logerr( i, j, wval, rval );
        }
        etime = rpcc();
        if ( stime >= etime )           etime += 1UL << 32;     /* wraparound */
        ticks = etime - stime;
        bwidth = ( VGABUFSIZ >> 10 ) * 1000000000000UL;
        bwidth /= primary_impure->CYCLE_CNT * ticks;
        /* printf_dbm( "%d kbytes/sec\n", bwidth ); */
    }

    return STATUS_SUCCESS;
}



/*--------------------------------------------------------------------*/
/* code to scan for all VGA devices present and checksum their option ROMs */

#define VGA_MAGIC0	0x55		/* Magic to mark start of option ROM */
#define VGA_MAGIC1	0xAA

typedef union {
    struct {
	    unsigned char magic0;
	    unsigned char magic1;
	    unsigned char npages;
	    unsigned char code;
    } rom;
    unsigned char raw[4];
} optHdr_t;


DBM_STATUS romtest( void )
{
    unsigned i;
    unsigned rom_base, rom_size;
    unsigned sum;
    optHdr_t optrom;
    PCIDevice_t *T;
    unsigned busno, hoseno;
    String slotname;
    DBM_STATUS rval = STATUS_SUCCESS;

    if ( !VgaDeviceDetected )			/* Did PCI init find one? */
    {
	mobo_logf( LOG_CRIT "VGA: no device was found during PCI bus scan\n" );
	return STATUS_FAILURE;
    } 

    /* Find each VGA device in turn and validate their ROMs */
    for ( T=devices; T!=NULL; T=T->sibling ) {

	if ( ! IS_VGA_DEV( T ) )	continue;
	
	/* got one! */
	busno = T->parent->number;
	hoseno = T->parent->hose;
	slotname = PCIDev2Str( hoseno, T->slot, 0 );
	mobo_logf( LOG_INFO "VGA: Probing for option ROM in hose %d bus %d (%s)\n",
		     hoseno, busno, slotname );

	rom_base = T->PCI_Bar[ MAX_BASE_ADDRESS_REGISTERS ].Base;
	if ( rom_base == 0 ) {
	    mobo_logf( LOG_WARN "VGA: no option ROM on this card\n" );
	    continue;
	}

	/* fill in the details */
        for( i=0; i<sizeof(optHdr_t); i++ )
            optrom.raw[i] = inmemb( rom_base + i );

	if ( (optrom.rom.magic0!=VGA_MAGIC0) ||
	     (optrom.rom.magic1!=VGA_MAGIC1) )
	{
	    mobo_logf( LOG_CRIT "VGA: Hose %d Bus %d, (%s) - ROM has bad header!\n",
			hoseno, busno, slotname );
	    rval = STATUS_WARNING;
	    continue;				/* on to next PCI device */
	}

	rom_size = optrom.rom.npages * 512;	/* npages in 512 byte units */

	/* Option ROM checksumming procedure: sum all bytes in the ROM region,
	 * the least significant byte of the result should be zero
 	 */
	for ( i=0,sum=0; i<rom_size; i++ )
	    sum += inmemb( rom_base + i );

	if ( (sum & 0xFF) == 0 )
	{
	    mobo_logf( LOG_INFO "VGA: Hose %d Bus %d (%s) - %dKB Option ROM "
			"at PCI addr 0x%08X is valid\n",
			hoseno, busno, slotname, rom_size >> 10, rom_base );
	} else {
	    mobo_logf( LOG_CRIT "VGA: Hose %d Bus %d (%s) - %dKB Option ROM "
			"at PCI addr 0x%08X has bad checksum 0x%08X!\n",
			hoseno, busno, slotname, rom_size >> 10, rom_base, sum );
	    rval = STATUS_WARNING;
	}

    }						/* on to next PCI device */

    return rval;
}



/*--------------------------------------------------------------------*/
/* #define VGABUF	0xFFFF8800000B8000UL */
/* #define VGABUF	0xFFFF8800000B8000UL */
#define VGABUF	0x800000B8000UL
#define IOPORT	0x801FC000070UL

DBM_STATUS memiostr( void )
{
    unsigned long start, finish, i, j;

    /* 64K quadwords->mem, 32K longwords->vga */
    const unsigned block = 64 << 10;

    printf_dbm( "\n\nCombined IO and memory test...\n");
    vgainit();

    /* ig_dump(); */

    start = 8UL << 20;			/* start at 8M */
    finish = start + (8UL << 20);	/* finish at 16M */

    for ( i=0;; i++ ) 
    {
	printf_dbm( "Pass %d: PCI but not memory...\n", i );
	for (j=start; j < finish; j+= block)
	{
	    combinedfill2( j, VGABUF );
	}

	printf_dbm( "Pass %d: Memory but not PCI...\n", i );
	for (j=start; j < finish; j+= block)
	{
	    combinedfill3( j, VGABUF );
	}

	printf_dbm( "Pass %d: Memory and 16-bit ISA IO...\n", i );
	for (j=start; j < finish; j+= block)
	{
	    combinedfill4( j, IOPORT );
	}

	printf_dbm( "Pass %d: Memory and 8-bit PCI...\n", i );
	for (j=start; j < finish; j+= block)
	{
	    combinedfill5( j, VGABUF );
	}

	printf_dbm( "Pass %d: Memory and PCI combined...\n", i );
	for (j=start; j < finish; j+= block)
	{
	    combinedfill( j, VGABUF );
	}
    }

    return STATUS_SUCCESS;
}
