/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

#undef TRACE_ENABLE		/* Enable/disable debugging in this module */

#include "lib.h"
#include "platform.h"
#include "uilib.h"
#include "console.h"
#include "nvram.h"
#include "cmos_rtc.h"

/*--------------------------------------------------------------------*/
/* Environment Variables read and write routines */


DBM_STATUS envread( void )
{
    nvenv_t *P;
    int i;
    int val;

    mobo_cls();
    printf_dbm( "Initialising NVRAM environment..." );
    nvenv_init();

    printf_dbm( "done\n"
    		"Reading NVRAM environment..." );

    P = nvenv_nextentry( NULL );
    printf_dbm( "done\n" );

    while( P != NULL )
    {
	printf_dbm( "%20s=%s\n", P->key, P->val );
	P = nvenv_nextentry( P );
    }

    printf_dbm( "\nReading CMOS environment...\n" );

    for (i=0; i<RTC_ENV_NVARS; i++ )
    {
	val = rtc_envrd( i );

	if ( val == -1 )	/* no value stored for this variable */
	    continue;

	printf_dbm( "%20s=%d\n", rtc_env_desc[ i ], val );
    }
    printf_dbm( "done\n" );

    mobo_key(0);
    return STATUS_SUCCESS;
}


static const String argtitle = "Bad arguments";
static const String argmsg = "Format: %s <key> <value>";

DBM_STATUS envwrite( int argc, char *argv[] )
{
    nvenv_t T;
    int i;
    int vallen;
    DBM_STATUS sval;
    int cmos_key, cmos_val;

    if ( argc < 3 )
    {
	mobo_alertf( argtitle, argmsg, argv[0] );
	return STATUS_FAILURE;
    }

    /* Check here to see if the user wishes to write a CMOS-stored variable */
    cmos_key = rtc_envmatch( argv[ 1 ] );
    if ( cmos_key != RTC_ENV_NOKEY )
    {
	/* divert here to handle the command as a case for CMOS */
	/* All CMOS variables (so far) are numbers so we can assume one arg */
	cmos_val = atoi( argv[2] );
	rtc_envwr( cmos_key, cmos_val );
	return STATUS_SUCCESS;
    }

    sval = nvenv_init();
    if ( sval == STATUS_FAILURE )
    {
	mobo_alertf( "No Environment Support",
		"This platform either does not support for NVRAM environment\r"
		"Or it is not functioning correctly" );
	return STATUS_FAILURE;
    }


    /* Parse arguments, accounting for environment values containing spaces */
    T.key = argv[1];

    /* note - our argument has been tokenised, we have no idea of the size or
     * form of white space.  Should I add support for quotes in args? */

    for ( i=2, vallen=0; i<argc; i++ )
	vallen += strlen( argv[i] ) + 1;	/* +1 for a space (or null) */

    TRACE( "value computed to be %d bytes long\n", vallen );
    T.val = malloc( vallen );
    if ( T.val == NULL )
    {
	mobo_alertf("No heap space!",
		"Diags has run out of memory processing your command" );
	return STATUS_FAILURE;
    }
    memset( T.val, 0, vallen );

    for ( i=2; i<argc; i++ )
    {
	TRACE( "Adding '%s' to val string '%s'...\n", argv[i], T.val );
	strcat( T.val, argv[i] );

	if( i < argc-1 )
		strcat( T.val, " " );		/* don't add space if last */
    }

    TRACE( "Writing %s=%s to NVRAM env...\n", T.key, T.val );
    nvenv_write( T.key, T.val );
    return STATUS_SUCCESS;
}


/*--------------------------------------------------------------------*/
/* Fuller implementation to manipulate NV Environment system settings */

#define RETSTR	" (press RETURN to quit):"

static const Point envlist = { 3, 3 };

typedef struct {
    String key;
    String val;
    const String *opt;		/* only applies for option-list variables */
    unsigned short nopts;
    unsigned short flags;
} sysenv_t;

#define SYSENV_OPTLIST	(1<<0)
#define SYSENV_FREE	(1<<1)
#define SYSENV_DECIMAL	(1<<2)
#define SYSENV_CMOS	(1<<3)



static const String autoopts[] = { AA_BOOT, AA_HALT, AA_RESTART };

static const String ttyopts[] = { TT_LOC, TT_COM1, TT_COM2, 
	TT_COM3, TT_COM4, TT_SROM, TT_NONE };

static const String onoffopts[] = { "OFF", "ON" };

static sysenv_t sysenv[] = {
    { KEY_DIAGS_ACTION, NULL, NULL, 0, SYSENV_FREE },
    { KEY_AUTO_ACTION, NULL, autoopts, ARRAYLEN(autoopts), SYSENV_OPTLIST },
    { KEY_BOOT_DEV, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOTDEF_DEV, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOTED_DEV, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOT_FILE, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOTED_FILE, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOT_OSFLAGS, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOTED_OSFLAGS, NULL, NULL, 0, SYSENV_FREE },
    { KEY_BOOT_RESET, NULL, onoffopts, ARRAYLEN(onoffopts), SYSENV_OPTLIST },
    { KEY_DBLX_INITRD, NULL, onoffopts, ARRAYLEN(onoffopts), SYSENV_OPTLIST },
    { KEY_ENABLE_BOOTMEM, NULL, onoffopts, ARRAYLEN(onoffopts), SYSENV_OPTLIST },
    { KEY_BOOTMEM_ALLOC, NULL, NULL, 0, SYSENV_DECIMAL },

#if 0		/* these things are deemed too dull by the implementer... */
    { KEY_DUMP_DEV, NULL, NULL, 0, SYSENV_FREE },
    { KEY_ENABLE_AUDIT, NULL, onoffopts, ARRAYLEN(onoffopts), SYSENV_OPTLIST },
    { KEY_LICENSE, NULL, NULL, 0, SYSENV_FREE },
    { KEY_LANGUAGE, NULL, NULL, 0, SYSENV_FREE },
    { KEY_CHAR_SET, NULL, NULL, 0, SYSENV_FREE },
#endif

    { KEY_TTY_DEV, NULL, ttyopts, ARRAYLEN(ttyopts), SYSENV_OPTLIST },
    { KEY_LOG_DEV, NULL, ttyopts, ARRAYLEN(ttyopts), SYSENV_OPTLIST },
    { KEY_COM1_BAUD, NULL, NULL, 0, SYSENV_CMOS },
    { KEY_THERM_OS, NULL, NULL, 0, SYSENV_DECIMAL },
    { KEY_THERM_HYST, NULL, NULL, 0, SYSENV_DECIMAL },
};



#define MAXDIGITS 16		/* the largest number of digits in a number */

static DBM_STATUS build_sysenv( void )
{
    DBM_STATUS sval;
    int i;
    String cmos_str;
    int cmos_key, cmos_num;

    /* paranoia: don't use any existing read values */
    sval = nvenv_init( );
    if ( sval != STATUS_SUCCESS )	return STATUS_FAILURE;

    for ( i=0; i<ARRAYLEN(sysenv); i++ )
    {
	switch ( sysenv[i].flags )
	{
	    case SYSENV_CMOS:
		cmos_key = rtc_envmatch( sysenv[i].key );
		if ( cmos_key == -1 )
		{
		    sysenv[i].val = NULL;
		    break;
		}

		cmos_num = rtc_envrd( cmos_key );
		if ( cmos_num == -1 )
		{
		    sysenv[i].val = NULL;
		    break;
		}

		cmos_str = malloc( MAXDIGITS );
		if ( cmos_str == NULL )		/* catastrophe: no heap */
		    return STATUS_FAILURE;

		sprintf_dbm( cmos_str, "%d", cmos_num );
		sysenv[i].val = cmos_str;
		break;

	    default:				/* everything else */
		sysenv[i].val = nvenv_lookup( sysenv[i].key );
		break;
	}
    }

    return STATUS_SUCCESS;
}


static int prompt_user( String buf, unsigned bufsz, String msg )
{
    int rval;
    mobo_goto( p_help );
    printf_dbm( "%-64s\r"
                "NVRAM> ", msg );
    rval = mobo_input( buf, bufsz );
    mobo_goto( p_prompt );
    printf_dbm( "%-64s", "" );		/* clear screen by overstriking */
    return rval;
}


static void draw_UI( void )
{
    int i;

    mobo_box( r_scrn, "System Firmware Environment Settings" );

    mobo_goto( envlist );
    for( i=0; i<ARRAYLEN(sysenv); i++ )
	printf_dbm( "%2d: %-16s = %-40s\r", i, sysenv[i].key,
	    (sysenv[i].val==NULL) ? "[No value set in NVRAM]" : sysenv[i].val );
}


static void handle_optlist( sysenv_t *S )
{
    int i;
    int rval;
    DBM_STATUS sval;
    char keybuf[ NVENV_MAXKEY ];
    char optbuf[ 4 ];

    /* List the available options */
    sprintf_dbm( keybuf, "Options available for %s", S->key );
    mobo_box( r_lrgapp, keybuf );
    mobo_goto( p_app );

    for( i=0; i<S->nopts; i++ )
	printf_dbm( "  %d: %s\r", i, S->opt[i] );

    do {
	rval = prompt_user( optbuf, 4, "Enter number for new setting" RETSTR );
	if ( rval == 0 )
		return;			/* no keys pressed -> return hit */

	sscanf( optbuf, "%d", &rval );	/* convert keypress to chosen option */
    } while ( (rval < 0) || (rval >= S->nopts) );

    /* Write the new value into place (nvenv_write will make a private copy) */
    sval = nvenv_write( S->key, S->opt[ rval ] );
    if ( sval != STATUS_SUCCESS )
	mobo_alertf( "NVRAM update error",
		     "The new value was not successfully written to NVRAM" );
}


static void handle_free( sysenv_t *S )
{
    DBM_STATUS sval;
    int rval;
    char valbuf[ NVENV_MAXVAL ];
    char msgbuf[ 80 ];

    sprintf_dbm( msgbuf, "Please enter a new value for %s" RETSTR, S->key );
    rval = prompt_user( valbuf, NVENV_MAXVAL, msgbuf );
    if ( rval == 0 )			return; 	/* user hit return */

    /* Write the new value (nvenv_write will make private copies) */
    sval = nvenv_write( S->key, valbuf );
    if ( sval != STATUS_SUCCESS )
        mobo_alertf( "NVRAM update error",
                     "The new value was not successfully written to NVRAM" );
}


static void handle_decimal( sysenv_t *S )
{
    DBM_STATUS sval;
    int rval;
    char valbuf[ NVENV_MAXVAL ];
    char msgbuf[ 80 ];
    int val;
    int cmos_key;

    sprintf_dbm( msgbuf,
	"Please enter a new decimal value for %s" RETSTR, S->key );
    rval = prompt_user( valbuf, NVENV_MAXVAL, msgbuf );
    if ( rval == 0 )			return; 	/* user hit return */

    if ( sscanf( valbuf, "%d", &val ) == 0 )
    {
	mobo_alertf( "Argument validation",
		"I couldn't parse your input as a decimal number\r"
		"This variable requires a decimal value" );
	return;
    }

    switch ( S->flags )
    {
	case SYSENV_CMOS:			/* ooky-kooky CMOS storage */
	    cmos_key = rtc_envmatch( S->key );
	    if ( cmos_key != -1 )
	    {
		rtc_envwr( cmos_key, val );
		sval = STATUS_SUCCESS;
	    } else {
		sval = STATUS_FAILURE;
	    }
	    break;

	default:
	case SYSENV_DECIMAL:			/* normal (NVRAM) storage */
	    sval = nvenv_write( S->key, valbuf );
	    break;

    }
    if ( sval != STATUS_SUCCESS )
        mobo_alertf( "Environment update error",
                     "The new value was not successfully saved." );
}


DBM_STATUS nvram( void ) 
{
    DBM_STATUS sval;
    int kval;
    char optbuf[4];

    while( TRUE ) {			/* main command loop */

	/* Paranoia: every iteration we re-build the NVRAM contents from ROM */
	sval = build_sysenv();
	if ( sval != STATUS_SUCCESS )		
	{
	    mobo_alertf( "NVRAM failure",
		    "Read of NVRAM environment variables failed" );
	    return STATUS_FAILURE;
	}

	do {
	    draw_UI();
	    kval = prompt_user( optbuf, 4,
			      "Enter number of the setting to change" RETSTR );
	    if ( kval == 0 )
			return STATUS_SUCCESS;	/* -> user hit return */

	    sscanf( optbuf, "%d", &kval );	/* convert ascii to value */
	} while( (kval < 0) || (kval >= ARRAYLEN(sysenv)) );

	switch( sysenv[kval].flags )
	{
	    case SYSENV_OPTLIST:
		handle_optlist( sysenv + kval );
		break;

	    case SYSENV_DECIMAL:
	    case SYSENV_CMOS:			/* signifies CMOS storage */
		handle_decimal( sysenv + kval );
		break;

	    case SYSENV_FREE:
	    default:				/* in case we're lost */
		handle_free( sysenv + kval );
		break;
	}
    }

    return STATUS_SUCCESS;			/* actually never reached */
}

