/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* IRQ test */


/* Theory of operation:
 * 1) 8259 IRQ lines 0-15 are enabled in PC-IO space 
 * 2) For all the interrupting devices I can raise an interrupt from, 
 *    I attempt to raise one interrupt 
 * 3) I first inform the interrupt handler of this test by setting flag
 *    IRQdivert.  The interrupt handler jumps into this module if set 
 * 4) There is provision for a timeout, if no interrupt is raised
 */

#undef TRACE_ENABLE			/* define to enable debugging output */

#include "lib.h"		/* Digital Debug Monitor */
#include "uilib.h"
#include "northbridge.h"
#include "platform.h"

#include "pci.h"
#include "pcidefs.h"
#include "kbd.h"
#include "uart.h"
#include "osf.h"

#include "nttypes.h"		/* for the floppy driver */
#include "floppy.h"
#include "cmos_rtc.h"
#include "lpt.h"


volatile int IntExpected, IRQTaken, IntTaken;

typedef struct {
	char name[8];
	int IRQ;			/* IRQ that interrupt arrives on */
	DBM_STATUS (*raise)( void );	/* Function to raise an interrupt */
	DBM_STATUS (*clear)( void );	/* Function to clear an interrupt */
} IRQ_t;

static IRQ_t *I;

static int irq_ndevs;
static int dev_testing;



/*----------------------------------------------------------------------*/
/* Function prototypes for IRQ-raising driver routines */

static DBM_STATUS rtc_raise( void );
static DBM_STATUS rtc_clear( void );

static DBM_STATUS kbd_raise( void );
static DBM_STATUS kbd_clear( void );

static DBM_STATUS mouse_raise( void );
static DBM_STATUS mouse_clear( void );

static DBM_STATUS timer_raise( void );
static DBM_STATUS timer_clear( void );

static DBM_STATUS com1_raise( void );
static DBM_STATUS com1_clear( void );

static DBM_STATUS com2_raise( void );
static DBM_STATUS com2_clear( void );

static DBM_STATUS fdc_raise( void );
static DBM_STATUS fdc_clear( void );

static DBM_STATUS lpt_raise( void );
static DBM_STATUS lpt_clear( void );

static DBM_STATUS pcidev_raise( void );
static DBM_STATUS pcidev_clear( void );




static const String str_title = "Interrupt System Test";

static const String str_unknown    = "Unknown";
static const String str_raising    = "Raising...";
static const String str_passed     = "Passed.";
static const String str_nodevice   = "No device";
static const String str_noresponse = "No response";
static const String str_spurious   = "Spurious int";
static const String str_unexpected = "Unexpecteds"; 
static const String str_multiple   = "Multiple ints";

static DBM_STATUS result;


/* Table of all ISA devices that can be used to provoke an interrupt */

static IRQ_t isadev[] = {

	{ "LPT1",	7,	lpt_raise,	lpt_clear },
#ifndef CONFIG_SHARK		/* not present in shark systems */
	{ "floppy",	6,	fdc_raise,	fdc_clear },
#endif
	{ "COM1",	4,	com1_raise,	com1_clear },
	{ "COM2",	3,	com2_raise,	com2_clear },
#ifndef CONFIG_SHARK		/* not present in shark systems */
	{ "mouse",	12,	mouse_raise,	mouse_clear },
#endif
	{ "RTC",	8,	rtc_raise,	rtc_clear },
#ifndef CONFIG_SHARK		/* not present in shark systems */
	{ "keybrd",	1,	kbd_raise,	kbd_clear },
#endif
	{ "timer",	0,	timer_raise,	timer_clear },
};


#define IRQDEV_NSTATIC ( sizeof( isadev ) / sizeof( isadev[0] ) )



/* All recognised PCI devices that I know how to interrupt */
/* This table is compared with the PCI bus tree for known, present devices */

typedef struct { 
	unsigned vendor;
	unsigned device;
} PCIKnown_t;

/* note: these two data structure arrays must correspond */

static PCIKnown_t PCIKnown[] = {
	{ PCI_VENDOR_ID_SIIG, PCI_DEVICE_ID_SIIG_CYBER },
};

static IRQ_t PCIPossibleIRQ[] = {
	{ "SIIG", 11, pcidev_raise, pcidev_clear },
};

#define PCIDEVS_NKNOWN ( sizeof( PCIKnown ) / sizeof( PCIKnown_t ) )


/* The data of the current PCI device under test is placed here */

typedef struct {
	IRQ_t *I;
	PCIDevice_t *P;
} PCIFound_t;

/* A list of all slots occupied by devices which I can test */

static PCIFound_t PCIFound[PCI_MAXSLOTS];


/*----------------------------------------------------------------------*/
/* Interface control functions.  All UI updates are done via here */

static void setstate( int i, const String S )
{
    /* Everything is done relative to two global values, r_app and p_app */
    Point p = p_app;
    int maxrow = r_app.h - 2;
    
    IRQ_t *I;
    if ( i >= IRQDEV_NSTATIC )		/* if it is a slot device */
	 I = PCIFound[i - IRQDEV_NSTATIC].I;
    else I = &isadev[i];


    /* find column offset */

    p.x += (i / maxrow) * (r_app.w / 3);
    p.y += i % maxrow;
    
    mobo_goto( p );
    printf_dbm( "%6s[%2d]: %-14s", I->name, I->IRQ, S );
}


static void initstate( void )
{   
    int i;

    mobo_logf( LOG_INFO "IRQ: test underway...\n");
    
    /* Draw the user interface */
    mobo_box( r_app, str_title );
    for ( i=0; i < irq_ndevs; i++ )
        setstate( i, str_unknown );
}




/*----------------------------------------------------------------------*/
/* Interrupt state control/handling mechanisms */


/* this routine is called by the interrupt handler if there is a device IRQ
 * and IRQdivert=TRUE.  Since we're called via the interrupt handler, 
 * interrupts will be disabled during the execution of this function */

#define MAX_MULTIPLES	10

void IRQHandler( int IRQTaken )
{

    /* Acknowledge the interrupt and get the vector in response */
    IntTaken++;


    /* Am I expecting an interrupt or is it spurious? */
    if ( !IntExpected )	{
	setstate( dev_testing, str_unexpected );
	result = STATUS_FAILURE;
	mobo_logf( LOG_CRIT "%s: Expected IRQ %d, got unexpected IRQ%d!\n",
		  I->name, I->IRQ, IRQTaken );
	goto EndOfInt;
    }


    /* is this a multiple interrupt? */
    if ( IntTaken > 1 ) {
	if ( IntTaken < MAX_MULTIPLES )
	{
	    setstate( dev_testing, str_multiple );
	    result = STATUS_FAILURE;
	    mobo_logf( LOG_CRIT "%s: Multiple interrupts, latest on IRQ%d\n", 
		      I->name, IRQTaken );
	} else if ( IntTaken == MAX_MULTIPLES ) {
	    mobo_logf( LOG_WARN "%s: further interrupts handled silently\n",
		I->name );
	}
	goto EndOfInt;

    }


    /* is it the same vector as I am expecting? */
    if ( IRQTaken != I->IRQ ) {
	setstate( dev_testing, str_spurious );
	result = STATUS_FAILURE;
	mobo_logf( LOG_CRIT "%s: Wrong interrupt!  Expected IRQ%d, got IRQ%d\n",
		 I->name, I->IRQ, IRQTaken );
	goto EndOfInt;
    } else {
	
	/* This is our boy.  We deassert here to prevent repercussions */

	I->clear();			/* service the interrupt */
	plat_intclr();			/* clear anything else */
	setstate( dev_testing, str_passed );
	goto EndOfInt;
    }


/* Regardless of how we arrived here, an interrupt has been taken and it's
 * completion must be signalled */

EndOfInt:
    /* depending on which interrupt we came in on, signal one EOI */
    if ( IRQTaken >= 0 && IRQTaken <= 7 ) 
	outportb( 0x20, 0x20 );
    else if ( IRQTaken >= 8 && IRQTaken <= 15)
	outportb( 0xA0, 0x20 );
    else
	mobo_logf( LOG_WARN "%s: don't know what to do about bizarre IRQ %d\n",
		  I->name, IRQTaken);
}


/*----------------------------------------------------------------------*/
/* IRQ device driver routines */

/* Real-time clock drivers.  It can be difficult to trace these interrupts,
 * since PALcode separates out the timer interrupts from other devices.
 * The best thing to do is be aware that this int needs handling a little 
 * differently. */

static DBM_STATUS rtc_raise(void)
{
    /* Implementation note: with the advent of Shark pass 1, we can't actually
     * prevent timer interrupts from happening.  So, instead we introduce a
     * flag here to tell the interrupt handler to ignore timer interrupts.
     */

    irq_include_timer = TRUE;
    return InitRTC();		/* if RTC sets up OK, it will interrupt */
}

static DBM_STATUS rtc_clear(void)
{
    /* Implementation note: with the advent of Shark pass 1, we can't actually
     * prevent timer interrupts from happening.  So, instead we introduce a
     * flag here to tell the interrupt handler to ignore timer interrupts.
     */

    irq_include_timer = FALSE;
    return STATUS_SUCCESS;
}





/* Keyboard interrupts (IRQ1) - when getting a byte from the keyboard,
 * we generate an interrupt. */

static DBM_STATUS kbd_raise( void )
{
    /* first up, enable the keyboard and mouse */
    kbd_ctl_cmd( KBD_CTL_WRMODE );
    kbd_ctl_output( 0x47 );		/* enables keyb, mouse and ints */

    kbd_ctl_cmd(KBD_CTL_WRKOBUF);
    kbd_ctl_output(0x55);               /* a random value for keyboard output */

    return STATUS_SUCCESS;
}


static DBM_STATUS kbd_clear( void )
{
    unsigned mode;

    /* flush the interrupt by reading from 0x60 */
    inportb( KBD_DATA );

    /* disable interrupts from the keyboard */
    kbd_ctl_cmd( KBD_CTL_RDMODE );
    mode = kbd_input();

    mode &= ~KBD_EKI;

    kbd_ctl_cmd( KBD_CTL_WRMODE );
    kbd_ctl_output( mode );

    return STATUS_SUCCESS;
}




/* Mouse interrupts (IRQ12) - when getting a byte from the mouse,
 * we generate an interrupt. */

static DBM_STATUS mouse_raise( void )
{
    /* first up, enable the keyboard and mouse */
    kbd_ctl_cmd( KBD_CTL_WRMODE );
    kbd_ctl_output( 0x47 );             /* enables keyb, mouse and ints */

    kbd_ctl_cmd(KBD_CTL_WRMOBUF);
    kbd_ctl_output(0x55);               /* a random value for keyboard output */

    return STATUS_SUCCESS;
}


static DBM_STATUS mouse_clear( void )
{
    unsigned char mode;

    inportb( KBD_DATA );

    /* disable interrupts from the mouse */
    kbd_ctl_cmd( KBD_CTL_RDMODE );
    mode = kbd_input();

    mode &= ~KBD_EMI;

    kbd_ctl_cmd( KBD_CTL_WRMODE );
    kbd_ctl_output( mode );

    return STATUS_SUCCESS;
}





/* System Timer tests - the problem with these timers is that they interrupt
 * faster relatively than we can print out the message, so we get multiple
 * spurious interrupts */

static DBM_STATUS timer_raise( void )
{
    /* Set the timer 0 mode to be mode 0 (single triggerable pulse) */
    /* we can't control the timing of the pulse - gate is hardwired to 1 */
    outportb( 0x43, 0x30 );		/* timer 0 into mode 0 */
    outportb( 0x40, 0x00 );		/* send LSB of count */
    outportb( 0x40, 0x00 );		/* send MSB of count */
    

    /* the timer is now ticking, wait 55 ms for interrupt */
    return STATUS_SUCCESS;
}

static DBM_STATUS timer_clear( void ) 
{
    /* nothing to be done, its a one-shot thing */
    return STATUS_SUCCESS;
}





/* Serial ports IRQ raising.  We know that COM1 and COM2 are present, and
 * there may be others.  Currently, we only assume the installed ports */

enum serial_regs { RBUF=0, XBUF=0, IER=1, IID=2, FIFO=2, LCTRL=3, MCTRL=4, 
		   LSTAT=5, MSTAT=6, SCRATCH=7 };

static DBM_STATUS serial_raise( unsigned p )
{
    unsigned dlab, ier;

    /* enable interrupts at this serial port */
    dlab = inportb( p + LCTRL );
    dlab &= ~0x80;				/* set DLAB = 0 */
    outportb( p + LCTRL, dlab);

    ier = inportb( p + IER );
    ier |= 0x0F;				/* all kinds of ints */
    outportb( p + IER, ier );
    outportb( p + MCTRL, 0x09 );		/* enable IRQs, DTR */

    return STATUS_SUCCESS;
}

static DBM_STATUS serial_clear( unsigned p )
{
    unsigned dlab, ier ;


    /* prevent further interrupts from the serial port */
    dlab = inportb( p + LCTRL );
    dlab &= ~0x80;                              /* set DLAB = 0 */
    outportb( p + LCTRL, dlab);

    ier = inportb( p + IER );
    ier &= 0xF0;  				/* disable all serial ints */
    outportb( p + IER, ier );

    return STATUS_SUCCESS;
}

#define COM1 0x3F8
#define COM2 0x2F8

static DBM_STATUS com1_raise( void )
{
    if ( io_dev_init( DEV_COM1 ) != STATUS_SUCCESS ) 
    {
	mobo_logf( LOG_WARN "Couldn't setup COM1 for IRQ test\n" );
	return STATUS_WARNING;
    }
    else return serial_raise( COM1 );
}
static DBM_STATUS com1_clear( void ) { return serial_clear( COM1 ); } 

static DBM_STATUS com2_raise( void )
{
    if ( io_dev_init( DEV_COM2 ) != STATUS_SUCCESS ) 
    {
        mobo_logf( LOG_WARN "Couldn't setup COM2 for IRQ test\n" );
        return STATUS_WARNING;
    }
    else return serial_raise( COM2 );
}
static DBM_STATUS com2_clear( void ) { return serial_clear( COM2 ); }



static FLOPPY_DRIVE_INFO d = {  HD_DRIVE_TYPE,
                                HD_DRIVE_TYPE,
                                0,                      /* Current track */
                                0,                      /* =1 for non-DMA */
                                0 };                    /* drive number */


static DBM_STATUS fdc_raise( void )
{
    unsigned char val;

    if (InitializeFloppyDrive( &d ) != STATUS_SUCCESS )	 return STATUS_WARNING;

    /* Enable interrupts in the controller */
    /* port 0x3F2 is the floppy controller digital output register */

    val |= 1 << 3;					/* enable ints */
    outportb( FDC_DIGITAL_OUTPUT_REGISTER, (1<<2));


    /* raise an interrupt from the controller.  This is achieved by 
     * a recalibration operation (seeks track 0 on disk) */
    outportb( 0x3F5, FDC_RECALIBRATE_CMD );

    return STATUS_SUCCESS;
}



static DBM_STATUS fdc_clear( void )
{
    d.DoNotUseDMAAccess = 1;

    /* reset by sensing interrupt status */
    if (InitializeFloppyDrive( &d ) != STATUS_SUCCESS )
		mobo_logf( LOG_WARN "couldn't re-init floppy\n");

    d.DoNotUseDMAAccess = 0;

    return STATUS_SUCCESS;
}



/*
 * Parallel port LPT1
 */

static DBM_STATUS lpt_raise( void )
{
    int status;
    int count;

    /* verify that a parallel port is present */
    if ( io_dev_init( DEV_LPT1 ) != STATUS_SUCCESS ) 
    {
	mobo_logf( LOG_WARN "LPT1 port absent or not functioning\n");
	return STATUS_WARNING;			/* a non-fatal failure */
    }	/* nb also detects presence of connected device */

    /* to enable parport interrupts, set a bit in the LPT control port */
    lptwb( DEV_LPT1, LPT_CTRL, LPCR_INTEN | LPCR_SELECT | LPCR_INIT );

    /* to raise interrupts, put a character out */
    /* taken from devices/lpt.c */

    for (count=0; ; count++) {                  /* timeout loop */

        /* probably no reason to do this, but hey */
        lptwb( DEV_LPT1, LPT_CTRL,
		LPCR_SELECT | LPCR_FEED | LPCR_INIT | LPCR_INTEN );

        status = lptrb( DEV_LPT1, LPT_STAT );
        if ( LP_NO_ERROR(status) && LP_READY(status) )   break;

        if (count >= 1000)
	{
	    mobo_logf( LOG_WARN "LPT: not online - device present?\n");
	    return STATUS_WARNING;
	}
        usleep(1);                              /* timeout delay */
    }

    /* output the char to the data port, wait > 0.5uS */
    lptwb( DEV_LPT1, LPT_DATA, ' ' );

    usleep(1);                          /* says 0.5, but this is safe... */

    /* active STROBE, sleep > 0.5uS, set STROBE high again */
    lptwb( DEV_LPT1, LPT_CTRL, 
	LPCR_INIT | LPCR_SELECT | LPCR_FEED | LPCR_STROBE | LPCR_INTEN );

    usleep(1);

    /* make sure the printer is acknowledging */
    for (count = 0; (lptrb( DEV_LPT1, LPT_STAT ) & LPST_BUSY) != 0; count++ )
    {
        if ( count >= 1000 )   {
                mobo_logf( LOG_WARN "LPT: not responding to strobe - device present?\n");
                return STATUS_WARNING;		/* a non-fatal failure */
        }
        usleep(1);
    }


    /* de-assert strobe */
    lptwb( DEV_LPT1, LPT_CTRL, LPCR_INIT | LPCR_SELECT | LPCR_FEED | LPCR_INTEN );

    return STATUS_SUCCESS;
}

static DBM_STATUS lpt_clear( void )
{
    putcLpt( DEV_LPT1, 0 );		/* do it properly, clear protocol */

    return STATUS_SUCCESS;
}



/*
 * Raising switch mechanism for any PCI slot devices
 * We're complex here because we have to deduce more about which device,
 * at what IO address, we're talking about.
 *
 */

static PCIDevice_t	*curP;

static DBM_STATUS pcidev_raise( void ) 
{
    int i;
    int addr;

    curP = PCIFound[ dev_testing - IRQDEV_NSTATIC ].P;

    switch (curP->vendor_id) {
	case PCI_VENDOR_ID_SIIG:	/* SIIG CyperSerial PCI Serial card */

	    /* find the IO registers base address */
	    for( i=0; curP->PCI_Bar[i].Type != TRUE; i++)		;

	    addr = curP->PCI_Bar[i].Base;
	    serial_raise( curP->PCI_Bar[i].Base );

	    /* a fixup: the SIIG card doesn't interrupt until you read this */
	    inportb( addr + IID );
	    break;
    }
    return STATUS_SUCCESS;
}


static DBM_STATUS pcidev_clear( void )
{
    int i;
    unsigned addr;

    switch (curP->vendor_id) {
	case PCI_VENDOR_ID_SIIG:        /* SIIG CyperSerial PCI Serial card */

	    /* find the IO registers base address */
            for( i=0; curP->PCI_Bar[i].Type != TRUE; i++)               ;

	    addr = curP->PCI_Bar[i].Base;
            serial_clear( addr );

            break;
    }
    return STATUS_SUCCESS;
}

/*----------------------------------------------------------------------*/
/* IRQ test entry point */

DBM_STATUS irq( int argc, char *argv[] )
{
    int i;
    unsigned long raisetime, takentime, etime;
    DBM_STATUS sval;
    const unsigned hi_ipl=7, lo_ipl=3;
    String interp;
    PCIKnown_t *K;
    PCIDevice_t *P;
    int ipl = swpipl( 7 );			/* find IPL to return with */


    /*
     * Set up the user interface, which requires
     * scanning the PCI bus for any interruptible devices
     * to get a full picture of which IRQ tests we can run
     */

    result = STATUS_SUCCESS;		/* innocent until proven guilty */

    diags_subsys_stat( SYS_IRQ, DEV_PROBING );
    irq_ndevs = IRQDEV_NSTATIC;			/* PCI devs add on to this */

    /* match with each device I recognise */
    for ( i=0, K=PCIKnown, I=PCIPossibleIRQ; i < PCIDEVS_NKNOWN; i++, K++, I++ )
    {
	P = PCIDeviceFind( K->vendor, K->device );
	if ( P == NULL )	continue;

	/* we have a first match */
	do {
	    /* Enter the data for this device */
	    PCIFound[ irq_ndevs - IRQDEV_NSTATIC ].I = I;
	    PCIFound[ irq_ndevs - IRQDEV_NSTATIC ].P = P;
	    irq_ndevs++;

	    /* look for a further instance */
	    P = PCIDeviceFindNext( P );
	} while ( P != NULL );
    }

    TRACE( "Calling initstate...\n" );
    initstate( );				/* user interface */



    /*
     * Setup the machine's state so we're ready to start testing ints
     */

    plat_intsetup(1);				/* wire up all ints */

    /* Set the timer 0 mode to be mode 0 (single triggerable pulse) */
    outportb( 0x43, 0x30 );             /* timer 0 into mode 0 */
    outportb( 0x40, 0x01 );             /* send LSB of count */
    outportb( 0x40, 0x00 );             /* send MSB of count */

    usleep( 100000 );			/* let it go, clear int below */


    /* sniff out any spurious interrupts that are going crazy */
    rtc_clear();
    wrmces(MCES_M_MIP | MCES_M_SCE | MCES_M_PCE);	/* clear outstandings */
    plat_intclr();			/* might precede enabling */
    plat_nmiclr();


    swpipl(lo_ipl);		/* all EV6 interrupts on */
    usleep(500000);		/* pause while spurious ints can rise */
    swpipl(hi_ipl);		/* disable ints again while I setup data */

    IRQTestHandler = IRQHandler;
    IRQdivert = TRUE;			/* IRQ tests starting */



    /*
     * Now raise each device in turn, and see what we get 
     */

    /* loop over all known devices attempting to raise interrupts */
    IRQTaken = 0;
    for (dev_testing=0, I=isadev; dev_testing < irq_ndevs; dev_testing++, I++)
    {	
	/* The data depends on the source of IRQ, which depends on test # */
	if ( dev_testing >= irq_ndevs )		/* not a static device */
	{
	    /* a dev we found by probe */
	    I = PCIFound[dev_testing - IRQDEV_NSTATIC].I;
	}
	IntExpected = 1;
	IntTaken = 0;

	swpipl(hi_ipl);			/* disable ints during raise */
	setstate( dev_testing, str_raising );
	mobo_logf( LOG_INFO "Raising IRQ %d via %s...\n", I->IRQ, I->name );
	raisetime = rpcc();
	sval = I->raise();
	swpipl(lo_ipl);			/* any ints should happen now */

	switch( sval ) {

	    case STATUS_FAILURE:
		setstate( dev_testing, str_nodevice );
		result = STATUS_FAILURE;
		mobo_logf( LOG_CRIT "fatal failure from device, skipping\n");
		continue;

	    case STATUS_WARNING:
	        setstate( dev_testing, str_nodevice );
		if ( result == STATUS_SUCCESS )	   result = STATUS_WARNING;
		mobo_logf( LOG_WARN "no response from device, skipping\n");
		continue;

	    case STATUS_SUCCESS:		/* we carry on with the test */
	    default:
		break;
	}

	do {
	    takentime = rpcc();
	    if (takentime <= raisetime)		takentime += 1UL << 32;
	    etime = takentime - raisetime;

	} while ( etime < (1UL << 29) );	/* an amount of time */

	if ( IntTaken == 0 )
	{
	    /* timeout: no interrupt received */
	    setstate( dev_testing, str_noresponse );
	    result = STATUS_FAILURE;
	    mobo_logf( LOG_CRIT "no interrupt!\n");
	    continue;				/* next device */

	    /* Even though no int was raised, I clear the device here */
	    swpipl(hi_ipl);
	    I->clear();
	    swpipl(lo_ipl);

	} else {
	    mobo_logf( LOG_INFO "taken\n");	/* log success */
	}
    }						/* Try next device */


    /* Cleanup the device states */

    IRQdivert = FALSE;			/* tests finished */
    plat_intsetup( 0 );			/* disable system ints */

    kbd_init();				/* fixup keyboard state */
    InitRTC();				/* fixup RTC state */
    swpipl( ipl );


    switch ( result ) {
	case STATUS_SUCCESS:
	    interp = "All tests PASSED";
	    diags_subsys_stat( SYS_IRQ, DEV_PASSED );
	    break;

	case STATUS_FAILURE:
	    interp = "Errors were found, test FAILED";
	    diags_subsys_stat( SYS_IRQ, DEV_FAILED );
	    break;

	case STATUS_WARNING:
	    interp = "Tests PASSED, with minor WARNINGS";
	    diags_subsys_stat( SYS_IRQ, DEV_PASSED );
	    result = STATUS_SUCCESS;		/* doesn't fail everything */
	    break;

	default:
	    interp = "Weird, I don't know whats going on";
	    result = STATUS_FAILURE;
    }
    mobo_alertf( str_title, interp );
    mobo_zap( r_app );

    return result;
}

